#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkprivate.h>

#include "../include/string.h"

#include "guiutils.h"
#include "guirgbimg.h"
#include "guifullscreen.h"
#include "tlist.h"
#include "cdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_directory.h"
#include "edv_date_format.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_obj_info_match.h"
#include "edv_image_io.h"
#include "edv_cursor.h"
#include "edv_utils_gtk.h"
#include "edv_list_cb.h"
#include "image_browser.h"
#include "presentation_mode.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_playerbtn_close_14x14.xpm"
#include "images/icon_playerbtn_eject_14x14.xpm"
#include "images/icon_playerbtn_slideshow_14x14.xpm"   
#include "images/icon_playerbtn_help_14x14.xpm"
#include "images/icon_playerbtn_info_14x14.xpm"
#include "images/icon_playerbtn_play_14x14.xpm"
#include "images/icon_playerbtn_playbackwards_14x14.xpm"
#include "images/icon_playerbtn_rewind_14x14.xpm"
#include "images/icon_playerbtn_rotateccw90_14x14.xpm"
#include "images/icon_playerbtn_rotatecw180_14x14.xpm"
#include "images/icon_playerbtn_rotatecw90_14x14.xpm"
#include "images/icon_playerbtn_seekend_14x14.xpm"


typedef struct _EDVFSImgFrame		EDVFSImgFrame;
#define FSIMG_FRAME(p)			((EDVFSImgFrame *)(p))

typedef struct _EDVFSImg		EDVFSImg;
#define EDV_FSIMG(p)			((EDVFSImg *)(p))
#define EDV_FSIMG_KEY			"EDV/PresentationMode/FSIMG"


/*
 *	Transision Operations:
 */
typedef enum {
	EDV_PRESENTATION_MODE_TRANSISION_OP_NONE,
	EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_BLACK_IN,
	EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_BLACK_OUT,
	EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_WHITE_IN,
	EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_WHITE_OUT,
	EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_BG_IN,
	EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_BG_OUT,
	EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_HIN,
	EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_HOUT,
	EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_VIN,
	EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_VOUT,
	EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_DIN,
	EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_DOUT
} EDVPresentationModeTransisionOP;


/*
 *	Rotate Positions:
 */
typedef enum {
	EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE,
	EDV_PRESENTATION_MODE_ROTATE_POSITION_CW90,
	EDV_PRESENTATION_MODE_ROTATE_POSITION_CCW90,
	EDV_PRESENTATION_MODE_ROTATE_POSITION_CW180
} EDVPresentationModeRotatePosition;


/* Callbacks */
static gint FSImgEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void FSImgDrawCB(
	GtkWidget *widget, GdkRectangle *rect, gpointer data
);
static gint FSImgListEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint FSImgNextFrameTOCB(gpointer data);
static gint FSImgHidePointerTOCB(gpointer data);
static gint FSImgPointerMotionCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint FSImgSlideShowTOCB(gpointer data);

static void FSImgCloseCB(GtkWidget *widget, gpointer data);
static void FSImgSlideshowCB(GtkWidget *widget, gpointer data);
static void FSImgNextCB(GtkWidget *widget, gpointer data);
static void FSImgPrevCB(GtkWidget *widget, gpointer data);
static void FSImgFirstCB(GtkWidget *widget, gpointer data);
static void FSImgLastCB(GtkWidget *widget, gpointer data);
static void FSImgListCB(GtkWidget *widget, gpointer data);
static void FSImgInfoCB(GtkWidget *widget, gpointer data);
static void FSImgRotateCW90CB(GtkWidget *widget, gpointer data);
static void FSImgRotateCCW90CB(GtkWidget *widget, gpointer data);
static void FSImgRotateCW180CB(GtkWidget *widget, gpointer data);
static void FSImgHelpCB(GtkWidget *widget, gpointer data);

/* Utilities */
static EDVPresentationModeTransisionOP FSImgGetTransisionOpFromTransision(
	const edv_presentation_mode_transision transision,
	const gboolean in
);
static EDVVFSObject *FSImgCurrentObject(EDVFSImg *fsimg);
static gboolean FSImgGetBGColorRGBA(
	EDVFSImg *fsimg,
	guint8 *bg_color_rtn
);
static GdkRgbDither FSImgGetDither(EDVFSImg *fsimg);
static void FSImgCurrentViewportSize(
	EDVFSImg *fsimg, gint *width_rtn, gint *height_rtn
);
static void FSImgWarpPointer(GdkWindow *window, gint x, gint y);
static void FSImgShowPointer(EDVFSImg *fsimg, gboolean show);
static void FSImgGrab(EDVFSImg *fsimg);

/* Draw */
static void FSImgDrawPixmapFromFrame(
	EDVFSImg *fsimg, EDVFSImgFrame *frame
);
static void FSImgDraw(EDVFSImg *fsimg);
static void FSImgQueueDraw(EDVFSImg *fsimg);

/* Set Image Data and Clear */
static void FSImgTransision(
	const EDVPresentationModeTransisionOP trans_op,
	GdkWindow *window,
	GdkRgbDither dither,
	GtkStyle *style,
	const guint8 *rgba,
	const gint width, const gint height, const gint bpl,
	const gulong duration_ms,
	const guint8 *bg_color			/* 4 bytes RGBA */
);
static void FSImgClearImageData(
	guint8 *rgba,
	const gint width, const gint height, const gint bpl,
	const guint8 *bg_color
);
static guint8 *FSImgCreateWinImageData(
	const gint win_width, const gint win_height,
	const guint8 *rgba,
	const gint width, const gint height, const gint bpl,
	const guint8 *bg_color			/* 4 bytes RGBA */
);
static gint FSImgSetData(
	EDVFSImg *fsimg,
	const gint width, const gint height, const gint bpl,
	GList *rgba_list,			/* guint8 * */
	GList *delay_list,			/* gulong */
	const guint8 *bg_color			/* 4 bytes RGBA */
);
static gint FSImgClear(
	EDVFSImg *fsimg,
	const gint width, const gint height
);

/* Operations */
static gint FSImgOpen(
	EDVFSImg *fsimg,
	const gchar *path
);
static void FSImgNext(EDVFSImg *fsimg);
static void FSImgPrev(EDVFSImg *fsimg);
static void FSImgFirst(EDVFSImg *fsimg);
static void FSImgLast(EDVFSImg *fsimg);
static void FSImgRefresh(EDVFSImg *fsimg);
static void FSImgRotateCW90(EDVFSImg *fsimg);
static void FSImgRotateCCW90(EDVFSImg *fsimg);
static void FSImgRotateCW180(EDVFSImg *fsimg);

/* Help */
static gboolean FSImgHelpShown(EDVFSImg *fsimg);
static void FSImgHelpShow(EDVFSImg *fsimg, const gboolean show);

/* Message */
static void FSImgMesg(EDVFSImg *fsimg, const gchar *mesg);

/* Info */
static gint FSImgInfoShown(EDVFSImg *fsimg);
static void FSImgInfoShow(EDVFSImg *fsimg, const gint info_level);

/* List */
static gboolean FSImgListShown(EDVFSImg *fsimg);
static void FSImgListShow(EDVFSImg *fsimg, const gboolean show);
static void FSImgListGet(EDVFSImg *fsimg, const gchar *path);

/* Slideshow */
static gboolean FSImgSlideshowActive(EDVFSImg *fsimg);

/* Navigator */
static gboolean FSImgNavigatorShown(EDVFSImg *fsimg);
static void FSImgNavigatorShow(EDVFSImg *fsimg, const gboolean show);
static void FSImgNavigatorUpdate(EDVFSImg *fsimg);

/* Frames */
static EDVFSImgFrame *EDVFSImgFrameNew(void);
static void EDVFSImgFrameDelete(EDVFSImgFrame *frame);

/* FSImg */
static void FSImgRestore(EDVFSImg *fsimg);
static EDVFSImg *FSImgNew(EDVCore *core);
static void FSImgSetBusy(EDVFSImg *fsimg, const gboolean busy);
static void FSImgDeleteRestore(EDVFSImg *fsimg);
static void FSImgDelete(EDVFSImg *fsimg);

/* Presentation Mode */
void EDVPresentationModeEnterFromImbr(
	EDVCore *core,
	EDVImageBrowser *imbr
);
void EDVPresentationModeEnter(EDVCore *core);


/*
 *	Frame:
 */
struct _EDVFSImgFrame {
	guint8		*rgba;
	gulong		delay;
};

/*
 *	Fullscreen Image Data:
 */
struct _EDVFSImg {

	GtkWidget	*toplevel;		/* GTK_WINDOW_POPUP GtkWindow */
	gint		freeze_count,
			busy_count,
			main_loop_level;
	gboolean	need_transision_in;
	GdkVisual	*visual;		/* Visual of toplevel */
	GdkColormap	*colormap;		/* Colormap of toplevel */
	GdkColor	color_fg,
			color_bg,
			color_shadow;
	gboolean	invert_fg;
	GdkFont		*font;
	GdkPixmap	*pixmap;		/* Pixmap for toplevel */
	GdkGC		*gc;
	EDVCore	*core;

	/* Image Information */
	gint		info_level;
	gchar		*info_text1,
			*info_text2,
			*info_text3;
	GdkBitmap	*icon_play_mask,
			*icon_pause_mask;
	gint		img_width,		/* Original size of the
						 * image on file */
			img_height;
	gint		img_shown_width,	/* Size of the image shown */
			img_shown_height;

	/* Image Data Frames (GList of EDVFSImgFrame * */
	gint		frame_width,		/* Allocated size of each
						 * frame's image data */
			frame_height;
	GList		*frames_list,
			*cur_frame;
	guint		frame_next_toid;

	/* Video Modes */
	GList		*vidmodes_list;

	const GdkVideoModeInfo	*orig_vidmode,
				*prev_vidmode,
				*cur_vidmode;

	gint		orig_viewport_x,
			orig_viewport_y;

	gint		orig_pointer_x,
			orig_pointer_y;

	/* Hide Pointer */
	gboolean	hide_pointer;
	guint		hide_pointer_toid;

	/* Slide Show */
	gulong		slideshow_delay;
	guint		slideshow_toid;

	/* Display Effects & Rotation */
	EDVPresentationModeRotatePosition	rotate_position;

	/* Messages */
	gboolean	show_help;
	gchar		*mesg;

	/* Goto List */
	GtkWidget	*clist_toplevel,
			*clist;		/* GtkCList */

	/* Navigator Bar */
	GtkWidget	*navigator_toplevel,
			*close_btn,
			*list_btn,
			*slideshow_btn,
			*next_btns_toplevel,
			*first_btn,
			*prev_btn,
			*next_btn,
			*last_btn,
			*info_btn,
			*rot_btns_toplevel,
			*rot_cw90_btn,
			*rot_ccw90_btn,
			*rot_cw180_btn,
			*help_btn;

};


#define FSIMG_BTN_WIDTH				60
#define FSIMG_BTN_HEIGHT			20

#define FSIMG_NAVIGATOR_HEIGHT			26

#define FSIMG_HELP_MESSAGE	{			\
 "F1",		"Help",					\
 "F2",		"List",					\
 " L",		"",					\
 "F3",		"Slideshow",				\
 "F4,",		"Cancel/Exit/Stop",			\
 " ESC,",	"",					\
 " Q,",		"",					\
 " W",		"",					\
 "F5",		"Refresh",				\
 "F6",		"Rotate Counter-Clockwise 90",		\
 "F7",		"Rotate Clockwise 90",			\
 "F8",		"Rotate Clockwise 180",			\
 "I",		"Display Image Information",		\
 "UP,",		"Previous Image",			\
 " LEFT",	"",					\
 "DOWN,",	"Next Image",				\
 " RIGHT,",	"",					\
 " ENTER,",	"",					\
 " SPACE",	"",					\
 "HOME",	"First Image",				\
 "END",		"Last Image",				\
 NULL,		NULL					\
}


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *      FSImg toplevel GtkWindow event signal callback.
 */
static gint FSImgEventCB( 
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	gboolean keypress = FALSE;
	guint state;
	GdkEventConfigure *configure;
	GdkEventFocus *focus;
	GdkEventKey *key;
	GdkEventButton *button;
	GdkEventMotion *motion;
	GdkEventCrossing *crossing;
	GtkWidget *event_widget;
	EDVCore *core;
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if((widget == NULL) || (event == NULL) || (fsimg == NULL))
	    return(status);

	if(fsimg->freeze_count > 0)
	    return(status);

	event_widget = gtk_get_event_widget(event);
	core = fsimg->core;

	switch((gint)event->type)
	{
	  case GDK_DELETE:
	    if(event_widget != widget)
		return(gtk_widget_event(event_widget, event));
	    /* Switch to tbe original video mode and delete the fsimg */
	    FSImgDeleteRestore(fsimg);
	    fsimg = NULL;
	    status = TRUE;
	    break;

	  case GDK_CONFIGURE:
	    configure = (GdkEventConfigure *)event;
	    if(event_widget != widget)
		return(gtk_widget_event(event_widget, event));
	    /* Move the viewport to align with the movement of the
	     * window
	     */
	    gdk_video_mode_set_viewport(
		configure->x, configure->y
	    );
	    status = TRUE;
	    break;

	  case GDK_EXPOSE:
	    if(event_widget != widget)
		return(gtk_widget_event(event_widget, event));
	    status = TRUE;
	    break;

	  case GDK_FOCUS_CHANGE:
	    focus = (GdkEventFocus *)event;
	    if(event_widget != widget)
		return(gtk_widget_event(event_widget, event));
	    if(focus->in && !GTK_WIDGET_HAS_FOCUS(widget))
	    {
		GTK_WIDGET_SET_FLAGS(widget, GTK_HAS_FOCUS);
		FSImgQueueDraw(fsimg);
	    }
	    else if(!focus->in && GTK_WIDGET_HAS_FOCUS(widget))
	    {
		GTK_WIDGET_UNSET_FLAGS(widget, GTK_HAS_FOCUS);
		FSImgQueueDraw(fsimg);
	    }
	    break;

	  case GDK_KEY_PRESS:
	    keypress = TRUE;
	  case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    state = key->state;
	    switch(key->keyval)
	    {
	      /* Display help */
	      case GDK_F1:
		if(!FSImgListShown(fsimg) && !FSImgSlideshowActive(fsimg))
		{
		    if(keypress)
			FSImgHelpCB(fsimg->help_btn, fsimg);
		    status = TRUE;
		}
		break;

	      /* List */
	      case GDK_F2:
	      case GDK_l:
		if(!FSImgSlideshowActive(fsimg))
		{
		    if(keypress)
			FSImgListShow(
			    fsimg,
			    !FSImgListShown(fsimg)
			);
		    status = TRUE;
		}
		break;

	      /* Slideshow */
	      case GDK_F3:
		if(!FSImgListShown(fsimg))
		{
		    if(keypress)
		    {
			if(fsimg->slideshow_toid > 0)
			{
			    fsimg->slideshow_toid = GTK_TIMEOUT_REMOVE(fsimg->slideshow_toid);
			    FSImgNavigatorUpdate(fsimg);
			}
			else
			{
			    FSImgSlideshowCB(fsimg->slideshow_btn, fsimg);
			}
		    }
		    status = TRUE;
		}
		break;

	      /* Exit */
	      case GDK_F4:
	      case GDK_Escape:
	      case GDK_Cancel:
	      case GDK_q:
	      case GDK_w:
	      case GDK_x:
		if(keypress)
		{
		    FSImgCloseCB(fsimg->close_btn, fsimg);
		}
		status = TRUE;
		break;

	      /* Refresh */
	      case GDK_F5:
		if(!FSImgListShown(fsimg) && !FSImgSlideshowActive(fsimg))
		{
		    if(keypress)
			FSImgRefresh(fsimg);
		    status = TRUE;
		}
		break;

	      /* Rotate Counter-Clockwise 90 */
	      case GDK_F6:
		if(!FSImgListShown(fsimg) && !FSImgSlideshowActive(fsimg))
		{
		    if(keypress)
			FSImgRotateCCW90(fsimg);
		    status = TRUE;
		}
		break;

	      /* Rotate Clockwise 90 */
	      case GDK_F7:
		if(!FSImgListShown(fsimg) && !FSImgSlideshowActive(fsimg))
		{
		    if(keypress)
			FSImgRotateCW90(fsimg);
		    status = TRUE;
		}
		break;

	      /* Rotate Clockwise 180 */
	      case GDK_F8:
		if(!FSImgListShown(fsimg) && !FSImgSlideshowActive(fsimg))
		{
		    if(keypress) 
			FSImgRotateCW180(fsimg);
		    status = TRUE;
		}
		break;

	      /* Previous */
	      case GDK_Left:
	      case GDK_Up:
	      case GDK_Page_Up:
	      case GDK_KP_Left:
	      case GDK_KP_Up:
	      case GDK_KP_Page_Up:
		if(!FSImgListShown(fsimg) && !FSImgSlideshowActive(fsimg))
		{
		    if(keypress)
		        FSImgPrev(fsimg);
		    status = TRUE;
		}
		break;

	      /* Next */
	      case GDK_Right:
	      case GDK_Down:
	      case GDK_Page_Down:
	      case GDK_KP_Right:
	      case GDK_KP_Down:
	      case GDK_KP_Page_Down:
	      case GDK_space:
	      case GDK_Return:
	      case GDK_KP_Enter:
		if(!FSImgListShown(fsimg) && !FSImgSlideshowActive(fsimg))
		{
		    if(keypress)
		        FSImgNext(fsimg);
		    status = TRUE;
		}
		break;

	      /* First */
	      case GDK_Home:
	      case GDK_KP_Home:
		if(!FSImgListShown(fsimg) && !FSImgSlideshowActive(fsimg))
		{
		    if(keypress)
		        FSImgFirst(fsimg);
		    status = TRUE;
		}
		break;

	      /* Last */
	      case GDK_End:
	      case GDK_KP_End:
		if(!FSImgListShown(fsimg) && !FSImgSlideshowActive(fsimg))
		{
		    if(keypress)
			FSImgLast(fsimg);
		    status = TRUE;
		}
		break;

	      /* Information */
	      case GDK_i:
	      case GDK_d:
		if(!FSImgListShown(fsimg))
		{
		    if(keypress)
			FSImgInfoShow(
			    fsimg,
			    FSImgInfoShown(fsimg) + 1
			);
		    status = TRUE;
		}
		break;
	    }
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    if(event_widget != widget)
		return(gtk_widget_event(event_widget, event));
	    if(!GTK_WIDGET_HAS_FOCUS(widget))
		gtk_widget_grab_focus(widget);
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;
	    if(event_widget != widget)
		return(gtk_widget_event(event_widget, event));
	    switch(button->button)
	    {
	      case GDK_BUTTON1:
		if(!FSImgListShown(fsimg) && !FSImgSlideshowActive(fsimg))
		    FSImgNext(fsimg);
		break;

	      case GDK_BUTTON2:
		if(!FSImgListShown(fsimg) && !FSImgSlideshowActive(fsimg))
		    FSImgPrev(fsimg);
		break;

	      case GDK_BUTTON3:
		if(!FSImgListShown(fsimg))
		{
		    if(button->state & GDK_SHIFT_MASK)
		    {
			if(!FSImgSlideshowActive(fsimg))
			    FSImgListShow(fsimg, TRUE);
		    }
		    else
		    {
			FSImgNavigatorShow(
			    fsimg, !FSImgNavigatorShown(fsimg)
			);
		    }
		}
		break;
	    }
	    break;

	  case GDK_MOTION_NOTIFY:
	    motion = (GdkEventMotion *)event;
	    if(event_widget != widget)
		return(gtk_widget_event(event_widget, event));
	    break;

	  case GDK_ENTER_NOTIFY:
	    crossing = (GdkEventCrossing *)event;
	    if(event_widget != widget)
		return(gtk_widget_event(event_widget, event));
	    break;

	  case GDK_LEAVE_NOTIFY:
	    crossing = (GdkEventCrossing *)event;
	    if(event_widget != widget)
		return(gtk_widget_event(event_widget, event));
	    break;
	}

	return(status);
}

/*
 *	FSImg toplevel GtkWindow "draw" signal callback.
 */
static void FSImgDrawCB(
	GtkWidget *widget, GdkRectangle *rect, gpointer data
)
{
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if((widget == NULL) || (fsimg == NULL))
	    return;

	FSImgDraw(fsimg);
}

/*
 *	GtkCList event signal callback.
 */
static gint FSImgListEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	GdkEventButton *button;
	GtkCList *clist;
	EDVCore *core;
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if((widget == NULL) || (event == NULL) || (fsimg == NULL))
	    return(status);

	if(fsimg->freeze_count > 0)
	    return(status);

	if(!GTK_IS_CLIST(widget))
	    return(status);

	clist = GTK_CLIST(widget);
	core = fsimg->core;

	switch((gint)event->type)
	{
	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    if(TRUE)
	    {
		/* Need to grab the pointer and confine it to the
		 * FSImg's toplevel window since the GtkCList will grab
		 * the pointer from it whenever a "button_press_event"
		 * event signal occures
		 */
		const GdkEventMask mask = (
		    (1 << (4 + button->button)) |
		    GDK_POINTER_MOTION_HINT_MASK |
		    GDK_BUTTON_RELEASE_MASK
		);
		GtkWidget *toplevel = fsimg->toplevel;
		gdk_pointer_grab(
		    button->window, FALSE, mask,
		    toplevel->window, NULL, button->time
		);
	    }
	    status = TRUE;
	    break;

	  case GDK_BUTTON_RELEASE:
	    button = (GdkEventButton *)event;
	    if(button->window == clist->clist_window)
	    {
		GList *glist;

		switch(button->button)
		{
		  case GDK_BUTTON1:
		    /* Display selected image */
		    glist = clist->selection_end;
		    if(glist != NULL)
		    {
		        /* Get object from the selected row */
			EDVVFSObject *obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(
			    clist, (gint)glist->data
			));
			if(obj != NULL)
			{
			    FSImgSetBusy(fsimg, TRUE);
			    FSImgNavigatorUpdate(fsimg);

			    /* Open the image */
			    (void)FSImgOpen(
				fsimg,
				obj->path
			    );

			    /* Update the information as needed */
			    FSImgInfoShow(fsimg, FSImgInfoShown(fsimg));

			    /* Hide the list */
			    FSImgListShow(fsimg, FALSE);

			    FSImgSetBusy(fsimg, FALSE);
			    FSImgNavigatorUpdate(fsimg);
			}
		    }
		    break;

		  case GDK_BUTTON3:
		    /* Hide the list */
		    FSImgListShow(fsimg, FALSE);
		    break;
		}
	    }
	    status = TRUE;
	    break;
	}

	return(status);
}

/*
 *	Next frame timeout callback.
 */
static gint FSImgNextFrameTOCB(gpointer data)
{
	GList *glist;
	EDVFSImgFrame *frame;
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return(FALSE);

	if(fsimg->freeze_count > 0)
	    return(TRUE);

	if(fsimg->frames_list == NULL)
	{
	    fsimg->frame_next_toid = 0;
	    return(FALSE);
	}

	/* Go to the next frame or cycle back to the first */
	fsimg->cur_frame = g_list_next(fsimg->cur_frame);
	if(fsimg->cur_frame == NULL)
	    fsimg->cur_frame = fsimg->frames_list;

	/* No more frames? */
	glist = fsimg->cur_frame;
	if(glist == NULL)
	{
	    fsimg->frame_next_toid = 0;
	    return(FALSE);
	}

	/* Get this frame */
	frame = FSIMG_FRAME(glist->data);
	if(frame == NULL)
	{
	    fsimg->frame_next_toid = 0;
	    return(FALSE);
	}

	/* Put this frame's image data on to the toplevel's GdkPixmap */
	FSImgDrawPixmapFromFrame(fsimg, frame);

	/* Queue the drawing since the frame has changed */
	FSImgQueueDraw(fsimg);

	/* Schedual the next frame increment */
	fsimg->frame_next_toid = gtk_timeout_add(
	    MAX(frame->delay, 1l),
	    FSImgNextFrameTOCB, fsimg
	);

	return(FALSE);
}

/*
 *	Hide pointer timeout callback.
 */
static gint FSImgHidePointerTOCB(gpointer data)
{
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return(FALSE);

	if(fsimg->freeze_count > 0)
	    return(TRUE);

	/* Do not hide while busy or if the list is shown */
	if((fsimg->busy_count > 0) || FSImgListShown(fsimg))
	    return(TRUE);

	/* Hide pointer */
	FSImgShowPointer(fsimg, FALSE);

	fsimg->hide_pointer_toid = 0;

	return(FALSE);
}

/*
 *	Pointer motion signal callback.
 *
 *	Called on "button_press_event", "button_release_event", and
 *	"motion_notify_event" signals to schedual the hide pointer
 *	timeout callback.
 */
static gint FSImgPointerMotionCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	gint status = FALSE;
	gulong delay;
	GdkEventMotion *motion;
	CfgList *cfg_list;
	EDVCore *core;
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if((widget == NULL) || (event == NULL) || (fsimg == NULL))
	    return(status);

	if(fsimg->freeze_count > 0)
	    return(status);

	core = fsimg->core;
	cfg_list = core->cfg_list;

	switch((gint)event->type)
	{
	  case GDK_MOTION_NOTIFY:
	    motion = (GdkEventMotion *)event;
	    if(motion->is_hint)
	    {
		gint x, y;
		GdkModifierType mask;
		gdk_window_get_pointer(
		    motion->window, &x, &y, &mask
		);
	    }
	  case GDK_BUTTON_PRESS:
	  case GDK_BUTTON_RELEASE:
	    delay = EDV_GET_L(
		EDV_CFG_PARM_PRESENTATION_HIDE_POINTER_DELAY
	    );
	    if(delay > 0l)
	    {
		/* Show pointer if it is hidden */
		if(fsimg->hide_pointer)
		    FSImgShowPointer(fsimg, TRUE);

		/* Reschedual hide pointer timeout due to recent
		 * pointer motion   
		 */
		fsimg->hide_pointer_toid = GTK_TIMEOUT_REMOVE(fsimg->hide_pointer_toid);
		fsimg->hide_pointer_toid = gtk_timeout_add(
		    (glong)(delay * 1000),
		    FSImgHidePointerTOCB, fsimg
		);
	    }
	    break;
	}

	return(status);
}

/*
 *	Slideshow timeout callback.
 */
static gint FSImgSlideShowTOCB(gpointer data)
{
	GtkCList *clist;
	CfgList *cfg_list;
	EDVCore *core;
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return(FALSE);

	if(fsimg->freeze_count > 0)
	    return(TRUE);

	core = fsimg->core;
	cfg_list = core->cfg_list;

	/* Display the next image */
	FSImgNext(fsimg);

	/* At the last image? */
	clist = GTK_CLIST(fsimg->clist);
	if(clist != NULL)
	{
	    GList *glist = clist->selection_end;
	    if((glist != NULL) ?
		((gint)glist->data >= (clist->rows - 1)) : TRUE
	    )
	    {
		/* Stop the slideshow */
		edv_play_sound_completed(fsimg->core);
		fsimg->slideshow_toid = 0;
		FSImgQueueDraw(fsimg);
		FSImgNavigatorUpdate(fsimg);
		return(FALSE);
	    }
	}

	/* Schedual next call to this function, do not just return
	 * TRUE since there may have been considerable delay while
	 * displaying thenext image
	 */
	fsimg->slideshow_toid = gtk_timeout_add(
	    MAX(fsimg->slideshow_delay, 1000l),
	    FSImgSlideShowTOCB, fsimg
	);

	return(FALSE);
}


/*
 *	Close callback.
 */
static void FSImgCloseCB(GtkWidget *widget, gpointer data)
{
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return;

	if(fsimg->freeze_count > 0)
	    return;

	/* If displaying help then hide help */
	if(FSImgHelpShown(fsimg))
	{
	    FSImgHelpShow(fsimg, FALSE);
	}
	/* If in slideshow then stop slideshow */
	else if(FSImgSlideshowActive(fsimg))
	{
	    fsimg->slideshow_toid = GTK_TIMEOUT_REMOVE(fsimg->slideshow_toid);
	    FSImgQueueDraw(fsimg);
	    FSImgNavigatorUpdate(fsimg);
	}
	/* If the list is mapped then unmap it */
	else if(FSImgListShown(fsimg))
	{
	    FSImgListShow(fsimg, FALSE);
	}
	/* All else exit presentation mode */
	else
	{
	    FSImgDeleteRestore(fsimg);
	}
}

/*
 *	Slideshow callback.
 */
static void FSImgSlideshowCB(GtkWidget *widget, gpointer data)
{
	GtkCList *clist;
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return;

	if(fsimg->freeze_count > 0)
	    return;

	clist = GTK_CLIST(fsimg->clist);

	/* Stop slideshow as needed */
	fsimg->slideshow_toid = GTK_TIMEOUT_REMOVE(fsimg->slideshow_toid);

	/* If there are enough items in the list to have a slideshow
	 * then start slideshow
	 */
	if(clist->rows >= 2)
	    fsimg->slideshow_toid = gtk_timeout_add(
		MAX(fsimg->slideshow_delay, 1000l),
		FSImgSlideShowTOCB, fsimg
	    );

	FSImgQueueDraw(fsimg);
	FSImgNavigatorUpdate(fsimg);
}

/*
 *	Next callback.
 */
static void FSImgNextCB(GtkWidget *widget, gpointer data)
{
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return;

	if(fsimg->freeze_count > 0)
	    return;

	FSImgNext(fsimg);
}

/*
 *	Previous callback.
 */
static void FSImgPrevCB(GtkWidget *widget, gpointer data)
{
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return;

	if(fsimg->freeze_count > 0)
	    return;

	FSImgPrev(fsimg);
}

/*
 *	First callback.
 */
static void FSImgFirstCB(GtkWidget *widget, gpointer data)
{
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return;

	if(fsimg->freeze_count > 0)
	    return;

	FSImgFirst(fsimg);
}

/*
 *	Last callback.
 */
static void FSImgLastCB(GtkWidget *widget, gpointer data)
{
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return;

	if(fsimg->freeze_count > 0)
	    return;

	FSImgLast(fsimg);
}

/*
 *	List callback.
 */
static void FSImgListCB(GtkWidget *widget, gpointer data)
{
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return;

	if(fsimg->freeze_count > 0)
	    return;

	FSImgListShow(
	    fsimg,
	    !FSImgListShown(fsimg)
	);
}

/*
 *	Information callback.
 */
static void FSImgInfoCB(GtkWidget *widget, gpointer data)
{
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return;

	if(fsimg->freeze_count > 0)
	    return;

	FSImgInfoShow(
	    fsimg,
	    FSImgInfoShown(fsimg) + 1
	);
}

/*
 *	Rotate clockwise 90 callback.
 */
static void FSImgRotateCW90CB(GtkWidget *widget, gpointer data)
{
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return;

	if(fsimg->freeze_count > 0)
	    return;

	FSImgRotateCW90(fsimg);
}

/*
 *	Rotate counter-clockwise 90 callback.
 */
static void FSImgRotateCCW90CB(GtkWidget *widget, gpointer data)
{
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return;

	if(fsimg->freeze_count > 0)
	    return;

	FSImgRotateCCW90(fsimg);
}

/*
 *	Rotate clockwise 180 callback.
 */
static void FSImgRotateCW180CB(GtkWidget *widget, gpointer data)
{
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return;

	if(fsimg->freeze_count > 0)
	    return;

	FSImgRotateCW180(fsimg);
}

/*
 *	Help callback.
 */
static void FSImgHelpCB(GtkWidget *widget, gpointer data)
{
	EDVFSImg *fsimg = EDV_FSIMG(data);
	if(fsimg == NULL)
	    return;

	if(fsimg->freeze_count > 0)
	    return;

	FSImgHelpShow(fsimg, !FSImgHelpShown(fsimg));
}


/*
 *	Returns the corresponding transision operation from the
 *	transision.
 */
static EDVPresentationModeTransisionOP FSImgGetTransisionOpFromTransision(
	const edv_presentation_mode_transision transition,
	const gboolean in
)
{
	if(in)
	{
	    switch(transition)
	    {
	      case EDV_PRESENTATION_MODE_TRANSISION_NONE:
		return(EDV_PRESENTATION_MODE_TRANSISION_OP_NONE);
		break;
	      case EDV_PRESENTATION_MODE_TRANSISION_FADE_BLACK:
		return(EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_BLACK_IN);
		break;
	      case EDV_PRESENTATION_MODE_TRANSISION_FADE_WHITE:
		return(EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_WHITE_IN);
		break;
	      case EDV_PRESENTATION_MODE_TRANSISION_FADE_BG:
		return(EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_BG_IN);
		break;
	      case EDV_PRESENTATION_MODE_TRANSISION_MOVE_HORIZONTAL:
		return(EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_HIN);
		break;
	      case EDV_PRESENTATION_MODE_TRANSISION_MOVE_VERTICAL:
		return(EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_VIN);
		break;
	      case EDV_PRESENTATION_MODE_TRANSISION_MOVE_DIAGONAL:
		return(EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_DIN);
		break;
	    }
	}
	else
	{
	    switch(transition)
	    {
	      case EDV_PRESENTATION_MODE_TRANSISION_NONE:
		return(EDV_PRESENTATION_MODE_TRANSISION_OP_NONE);
		break;
	      case EDV_PRESENTATION_MODE_TRANSISION_FADE_BLACK:
		return(EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_BLACK_OUT);
		break;
	      case EDV_PRESENTATION_MODE_TRANSISION_FADE_WHITE:
		return(EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_WHITE_OUT);
		break;
	      case EDV_PRESENTATION_MODE_TRANSISION_FADE_BG:
		return(EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_BG_OUT);
		break;
	      case EDV_PRESENTATION_MODE_TRANSISION_MOVE_HORIZONTAL:
		return(EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_HOUT);
		break;
	      case EDV_PRESENTATION_MODE_TRANSISION_MOVE_VERTICAL:
		return(EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_VOUT);
		break;
	      case EDV_PRESENTATION_MODE_TRANSISION_MOVE_DIAGONAL:
		return(EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_DOUT);
		break;
	    }
	}

	return(EDV_PRESENTATION_MODE_TRANSISION_OP_NONE);
}

/*
 *	Returns the currently selected object or the first object in
 *	the list.
 *
 *	Can return NULL on error.
 */
static EDVVFSObject *FSImgCurrentObject(EDVFSImg *fsimg)
{
	GList *glist;
	GtkCList *clist;

	if(fsimg == NULL)
	    return(NULL);

	clist = GTK_CLIST(fsimg->clist);
	glist = clist->selection_end;
	return(EDV_VFS_OBJECT(gtk_clist_get_row_data(
	    clist,
	    (glist != NULL) ? (gint)glist->data : 0
	)));
}

/*
 *	Gets the current background RGBA color from the configuration.
 */
static gboolean FSImgGetBGColorRGBA(
	EDVFSImg *fsimg,
	guint8 *bg_color_rtn			/* 4 bytes RGBA */
)
{
	CfgList *cfg_list;
	CfgColor *c;
	EDVCore *core;

	if((fsimg == NULL) || (bg_color_rtn == NULL))
	    return(FALSE);

	core = fsimg->core;
	cfg_list = core->cfg_list;

	/* Get the background color */
	c = EDV_GET_COLOR(EDV_CFG_PARM_PRESENTATION_COLOR_BG);
	if(c != NULL)
	{
	    bg_color_rtn[0] = c->r * 0xff;
	    bg_color_rtn[1] = c->g * 0xff;
	    bg_color_rtn[2] = c->b * 0xff;
	    bg_color_rtn[3] = c->a * 0xff;
	}
	else
	{
	    bg_color_rtn[0] = 0xff;
	    bg_color_rtn[1] = 0xff;
	    bg_color_rtn[2] = 0xff;
	    bg_color_rtn[3] = 0xff;
	}

	return(TRUE);
}

/*
 *	Gets the current dither from the configuration.
 */
static GdkRgbDither FSImgGetDither(EDVFSImg *fsimg)
{
	CfgList *cfg_list;
	EDVCore *core;

	if(fsimg == NULL)
	    return(GDK_RGB_DITHER_NONE);

	core = fsimg->core;
	cfg_list = core->cfg_list;

	/* Get the quality */
	switch(EDV_GET_I(EDV_CFG_PARM_IMAGE_QUALITY))
	{
	  case 2:
	    return(GDK_RGB_DITHER_MAX);
	    break;
	  case 1:
	    return(GDK_RGB_DITHER_NORMAL);
	    break;
	  default:
	    return(GDK_RGB_DITHER_NONE);
	    break;
	}
}

/*
 *	Returns the current size of the viewport.
 */
static void FSImgCurrentViewportSize(
	EDVFSImg *fsimg, gint *width_rtn, gint *height_rtn
)
{
	const GdkVideoModeInfo *m;

	if(width_rtn != NULL)
	    *width_rtn = 0;
	if(height_rtn !=  NULL)
	    *height_rtn = 0;

	if(fsimg == NULL)
	    return;

	m = fsimg->cur_vidmode;
	if(m == NULL)
	    return;

	if(width_rtn != NULL)
	    *width_rtn = m->viewport_width;
	if(height_rtn !=  NULL)
	    *height_rtn = m->viewport_height;
}

/*
 *	Warps the pointer relative to the desktop.
 */
static void FSImgWarpPointer(GdkWindow *window, gint x, gint y)
{
#if defined(HAVE_X)
	GdkWindowPrivate *private = (GdkWindowPrivate *)window;
	if((private != NULL) ? (private->xdisplay == NULL) : TRUE)
	    return;

	XWarpPointer(
	    private->xdisplay,
	    None,		/* No source window */
	    private->xwindow,	/* Move relative to destination window */
	    0, 0, 0, 0,		/* No source rectangle */
	    (int)x, (int)y
	);
#else

#endif
}

/*
 *	Shows or hides the pointer.
 */
static void FSImgShowPointer(EDVFSImg *fsimg, gboolean show)
{
	GtkWidget *w;
	EDVCore *core;

	if(fsimg == NULL)
	    return;

	core = fsimg->core;

	/* Already shown or hidden? */
	if(fsimg->hide_pointer != show)
	    return;

	/* Update show pointer marker */
	fsimg->hide_pointer = !show;

	w = fsimg->toplevel;
	if(w != NULL)
	{
	    GdkWindow *window = w->window;
	    if(window != NULL)
	    {
		GdkCursor *cursor;
		if(show)
		{
		    if(fsimg->busy_count > 0)
			cursor = edv_get_cursor(
			    core, EDV_CURSOR_CODE_BUSY
			);
		    else
			cursor = NULL;
		}
		else
		{
		    cursor = edv_get_cursor(
			core, EDV_CURSOR_CODE_INVISIBLE
		    );
		}
		gdk_window_set_cursor(window, cursor);
		gdk_flush();
	    }
	}
}

/*
 *	Grabs the FSImg's window.
 */
static void FSImgGrab(EDVFSImg *fsimg)
{
	GdkWindow *window;
	GtkWidget *w;

	if(fsimg == NULL)
	    return;

	w = fsimg->toplevel;
	window = w->window;
	if(window == NULL)
	    return;

	/* Grab the widget to receive key events */
	gtk_grab_add(w);
	gtk_widget_grab_focus(w);

	/* Grab the pointer and confine it to the window's area */
	gdk_pointer_grab(
	    window,
	    TRUE,		/* Report event relative to event's window */
	    GDK_BUTTON_PRESS_MASK |
	    GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK,
	    window,		/* Confine pointer to this window */
	    NULL,		/* Cursor */
	    GDK_CURRENT_TIME
	);
}


/*
 *	Draws the frame's image data on to the toplevel's GdkPixmap.
 *
 *	The frame specifies the image data to put on to the toplevel's
 *	GdkPixmap.
 *
 *	Does not put the toplevel's GdkPixmap on to the toplevel's
 *	GdkWindow nor does it queue a draw.
 */
static void FSImgDrawPixmapFromFrame(
	EDVFSImg *fsimg, EDVFSImgFrame *frame
)
{
	gint width, height;
	GdkPixmap *pixmap;
	GtkStyle *style;

	if((fsimg == NULL) || (frame == NULL))
	    return;

	width = fsimg->frame_width;
	height = fsimg->frame_height;
	if((width <= 0) || (height <= 0))
	    return;

	/* Get the toplevel's GdkPixmap */
	pixmap = fsimg->pixmap;
	style = gtk_widget_get_style(fsimg->toplevel);
	if((pixmap == NULL) || (style == NULL) || (frame->rgba == NULL))
	    return;

	/* Put the frame's image data on to the toplevel's GdkPixmap */
	gdk_draw_rgb_32_image(
	    pixmap, style->black_gc,
	    0, 0, width, height,
	    FSImgGetDither(fsimg),
	    (guchar *)frame->rgba,
	    width * 4 * sizeof(guint8)
	);
}

/*
 *	Draws the FSImg.
 *
 *	Puts the toplevel's GdkPixmap on to the toplevel's GdkWindow.
 *
 *	Then draws the information, messages, and help on to the
 *	toplevel's GdkWindow as needed.
 */
static void FSImgDraw(EDVFSImg *fsimg)
{
	gboolean navigator_shown;
	gint	win_width = 0, win_height = 0,
		img_shown_width, img_shown_height;
	GdkFont *font;
	GdkColor	*c_fg,
			*c_bg,
			*c_shadow;
	GdkWindow *window;
	GdkPixmap *pixmap;
	GdkGC *gc;
	GtkStyle *style;
	GtkWidget *w;
	CfgList *cfg_list;
	EDVCore *core;

	if(fsimg == NULL)
	    return;

	w = fsimg->toplevel;
	window = w->window;
	style = gtk_widget_get_style(w);
	core = fsimg->core;
	cfg_list = core->cfg_list;
	if((window == NULL) || (style == NULL))
	    return;

	/* Need to perform fade in first? */
	if(fsimg->need_transision_in)
	{
	    guint8 bg_color[4];

	    /* Get the background color */
	    (void)FSImgGetBGColorRGBA(fsimg, bg_color);

	    /* Fade in */
	    if(fsimg->cur_frame != NULL)
	    {
		GList *glist = fsimg->cur_frame;
		EDVFSImgFrame *frame = FSIMG_FRAME(glist->data);
		if(frame != NULL)
		{
		    const edv_presentation_mode_transision trans =
			EDV_GET_I(EDV_CFG_PARM_PRESENTATION_TRANSISION);
		    FSImgTransision(
			FSImgGetTransisionOpFromTransision(
			    trans,
			    TRUE		/* In */
			),
			window,
			FSImgGetDither(fsimg),
			style,
			frame->rgba,
			fsimg->frame_width, fsimg->frame_height, -1,
			EDV_GET_L(EDV_CFG_PARM_PRESENTATION_TRANSISION_INT) / 2l,
			bg_color		/* 4 bytes RGBA */
		    );
		}
	    }

	    fsimg->need_transision_in = FALSE;
	}

	navigator_shown = FSImgNavigatorShown(fsimg);
	img_shown_width = fsimg->img_shown_width;
	img_shown_height = fsimg->img_shown_height;

	/* Draw the background? */
	pixmap = fsimg->pixmap;
	if(pixmap != NULL)
	{
	    gdk_window_get_size(pixmap, &win_width, &win_height);
	    gdk_window_copy_area(
		window, style->black_gc,
		0, 0,
		pixmap,
		0, 0,
		win_width, win_height
	    );
	}

	font = fsimg->font;
	gc = fsimg->gc;
	if((font == NULL) || (gc == NULL))
	    return;

	c_fg = &fsimg->color_fg;
	c_bg = &fsimg->color_bg;
	c_shadow = &fsimg->color_shadow;

	/* Set up GC for text drawing */
	gdk_gc_set_foreground(gc, c_fg);
	gdk_gc_set_background(gc, c_bg);
	gdk_gc_set_font(gc, font);
	gdk_gc_set_function(gc, GDK_COPY);
	gdk_gc_set_fill(gc, GDK_SOLID);
	gdk_gc_set_clip_mask(gc, NULL);
	gdk_gc_set_clip_origin(gc, 0, 0);

#define DRAW_TEXT(_d_,_f_,_b_,_x_,_y_,_s_,_l_) {	\
 gdk_gc_set_foreground(gc, c_fg);		\
 gdk_draw_text(					\
  (_d_), (_f_), gc,				\
  ((_x_) + 1) - (_b_)->lbearing,		\
  ((_y_) + 1) + (_f_)->ascent,			\
  (_s_), (_l_)					\
 );						\
}
#define DRAW_TEXT_DROP_SHADE(_d_,_f_,_b_,_x_,_y_,_s_,_l_) {	\
 gdk_gc_set_foreground(gc, c_bg);		\
 gdk_draw_text(					\
  (_d_), (_f_), gc,				\
  ((_x_) + 1) - (_b_)->lbearing,		\
  ((_y_) + 1) + (_f_)->ascent,			\
  (_s_), (_l_)					\
 );						\
 gdk_gc_set_foreground(gc, c_fg);		\
 gdk_draw_text(					\
  (_d_), (_f_), gc,				\
  (_x_) - (_b_)->lbearing,			\
  (_y_) + (_f_)->ascent,			\
  (_s_), (_l_)					\
 );						\
}

	/* Draw information text? */
	if(fsimg->info_text1 != NULL)
	{
	    const gint	border_major = 5,
			font_height = font->ascent + font->descent;
	    const gchar *s = fsimg->info_text1, *s2;
	    gint x = border_major, y = border_major, len;

	    gdk_gc_set_function(
		gc,
		(fsimg->invert_fg) ? GDK_INVERT : GDK_COPY
	    );

	    while(y < win_height)
	    {
		/* Get pointer to end of line */
		s2 = strchr(s, '\n');
		/* Get length of line */
		len = (s2 != NULL) ? (gint)(s2 - s) : STRLEN(s);

		/* Draw this line? */
		if(len > 0)
		{
		    GdkTextBounds b;
		    gdk_text_bounds(font, s, len, &b);
		    DRAW_TEXT_DROP_SHADE(
			window, font, &b,
			x, y,
			s, len
		    );
		}

		/* Move the drawing position to the next line */
		y += font_height;

		/* Is there a next line? */
		if(s2 != NULL)
		    s = s2 + 1;		/* Seek s to next line */
		else
		    break;
	    }
	}
	if(fsimg->info_text2 != NULL)
	{
	    const gint  border_major = 5,
			font_height = font->ascent + font->descent;
	    const gchar *s = fsimg->info_text2, *s2;
	    gint x, y = border_major, len;

	    gdk_gc_set_function(
		gc,
		(fsimg->invert_fg) ? GDK_INVERT : GDK_COPY
	    );

	    while(y < win_height)
	    {
		/* Get pointer to end of line */
		s2 = strchr(s, '\n');
		/* Get length of line */
		len = (s2 != NULL) ? (gint)(s2 - s) : STRLEN(s);

		/* Draw this line? */
		if(len > 0)
		{
		    GdkTextBounds b;
		    gdk_text_bounds(font, s, len, &b);
		    x = win_width - b.width - border_major;
		    DRAW_TEXT_DROP_SHADE(
			window, font, &b,
			x, y,
			s, len
		    );
		}

		/* Move the drawing position to the next line */
		y += font_height;

		/* Is there a next line? */
		if(s2 != NULL)
		    s = s2 + 1;		/* Seek s to next line */
		else
		    break;
	    }
	}
	if(fsimg->info_text3 != NULL)
	{
	    const gint  border_major = 5,
			font_height = font->ascent + font->descent;
	    const gchar *s = fsimg->info_text3, *s2;
	    gint	x = win_width - border_major,
			y = win_height -
			    (navigator_shown ? FSIMG_NAVIGATOR_HEIGHT : 0) -
			    font_height - border_major - 1,
			len;

	    gdk_gc_set_function(
		gc,
		(fsimg->invert_fg) ? GDK_INVERT : GDK_COPY
	    );

	    while(y >= 0)
	    {
		/* Get pointer to end of line */
		s2 = strchr(s, '\n');
		/* Get length of line */
		len = (s2 != NULL) ? (gint)(s2 - s) : STRLEN(s);

		/* Draw this line? */
		if(len > 0)
		{
		    GdkTextBounds b;
		    gdk_text_bounds(font, s, len, &b);
		    x = win_width - b.width - border_major;
		    DRAW_TEXT_DROP_SHADE(
			window, font, &b,
			x, y,
			s, len
		    );
		}

		/* Move the drawing position to the next line */
		y -= font_height;

		/* Is there a next line? */
		if(s2 != NULL)
		    s = s2 + 1;		/* Seek s to next line */
		else
		    break;
	    }
	}
	if(fsimg->info_level >= 4)
	{
	    const gboolean slideshow_active = FSImgSlideshowActive(fsimg);
	    GdkBitmap *icon_mask = slideshow_active ?
		fsimg->icon_play_mask : fsimg->icon_pause_mask;

	    if(icon_mask != NULL)
	    {
		const gint border_major = 5;
		gint x, y, icon_width, icon_height;

		gdk_window_get_size(icon_mask, &icon_width, &icon_height);

		x = border_major;
		y = win_height -
		    (navigator_shown ? FSIMG_NAVIGATOR_HEIGHT : 0) -
			icon_height - border_major - 1;

		gdk_gc_set_function(
		    gc,
		    (fsimg->invert_fg) ? GDK_INVERT : GDK_COPY
		);
		gdk_gc_set_clip_mask(gc, icon_mask);
		gdk_gc_set_clip_origin(gc, x + 1, y + 1);
		gdk_gc_set_foreground(gc, c_bg);
		gdk_draw_rectangle(
		    window,
		    gc,
		    TRUE,			/* Fill */
		    x + 1, y + 1,
		    icon_width, icon_height
		);
		gdk_gc_set_clip_origin(gc, x, y);
		gdk_gc_set_foreground(gc, c_fg);
		gdk_draw_rectangle(
		    window,
		    gc,
		    TRUE,			/* Fill */
		    x, y,
		    icon_width, icon_height
		);
		gdk_gc_set_clip_mask(gc, NULL);
		gdk_gc_set_clip_origin(gc, 0, 0);
	    }
	}

	/* Draw message */
	if((fsimg->mesg != NULL) && !FSImgHelpShown(fsimg))
	{
	    const gint	border_major = 5,
			font_height = font->ascent + font->descent;
	    gint x, y, len, lines = 0, longest_line_width = 0;
	    const gchar *s = fsimg->mesg, *s2;
	    GdkTextBounds b;

	    /* Count the number of lines and the longest line */
	    while(TRUE)
	    {
		s2 = (const gchar *)strchr((const char *)s, '\n');
		len = (s2 != NULL) ? (gint)(s2 - s) : STRLEN(s);
		gdk_text_bounds(font, s, len, &b);

		lines++;
		if(b.width > longest_line_width)
		    longest_line_width = b.width;

		if(s2 != NULL)
		    s = s2 + 1;
		else
		    break;
	    }

	    /* Set the upper-left starting coordinates for drawing */
	    x = MAX((win_width - longest_line_width) / 2, 0);
	    y = MAX((win_height -
		(navigator_shown ? FSIMG_NAVIGATOR_HEIGHT : 0) -
		(lines * font_height)) / 2, 0);

	    /* Draw the subtractive "shadow" background */
	    gdk_gc_set_function(gc, GDK_AND);
	    gdk_gc_set_foreground(gc, c_shadow);
	    gdk_draw_rectangle(
		window,
		gc,
		TRUE,				/* Fill */
		x - border_major,
		y - border_major,
		longest_line_width + (2 * border_major),
		(lines * font_height) + (2 * border_major)
	    );

	    /* Draw the text */
	    gdk_gc_set_function(gc, GDK_COPY);
	    gdk_gc_set_foreground(gc, c_fg);
	    s = fsimg->mesg;
	    while(y < win_height)
	    {
		/* Get pointer to end of line */
		s2 = strchr(s, '\n');
		/* Get length of line */
		len = (s2 != NULL) ? (gint)(s2 - s) : STRLEN(s);

		/* Draw this line? */
		if(len > 0)
		{
		    gdk_text_bounds(font, s, len, &b);
		    DRAW_TEXT(
			window, font, &b,
			x, y,
			s, len
		    );
		}

		/* Move the drawing position to the next line */
		y += font_height;

		/* Is there a next line? */
		if(s2 != NULL)
		    s = s2 + 1;		/* Seek s to next line */
		else
		    break;
	    }
	}

	/* Draw help? */
	if(FSImgHelpShown(fsimg))
	{
	    const gint	border_major = 5,
			font_height = font->ascent + font->descent,
			column1_spacing = 10;
	    gint	i, x, y, len,
			column1_width = 0,
			lines = 0, longest_line_width = 0;
	    const gchar *s, *mesg[] = FSIMG_HELP_MESSAGE;
	    GdkTextBounds b;

	    /* Count the number of lines and the longest line */
	    for(i = 0; mesg[i] != NULL; i += 2)
	    {
		lines++;

		gdk_string_bounds(font, mesg[i], &b);
		x = b.width + column1_spacing;
		if(x > column1_width)
		    column1_width = x;
	    }
	    for(i = 0; mesg[i] != NULL; i += 2)
	    {
		gdk_string_bounds(font, mesg[i + 1], &b);
		x = column1_width + b.width;
		if(x > longest_line_width)
		    longest_line_width = x;
	    }

	    /* Set the upper-left starting coordinates for drawing */
	    x = MAX(((win_width - longest_line_width) / 2), 0);
	    y = MAX(((win_height -
		(navigator_shown ? FSIMG_NAVIGATOR_HEIGHT : 0) -
		(lines * font_height)) / 2), 0);

	    /* Draw the subtractive "shadow" background */
	    gdk_gc_set_function(gc, GDK_AND);
	    gdk_gc_set_foreground(gc, c_shadow);
	    gdk_draw_rectangle(
		window,
		gc,
		TRUE,				/* Fill */
		x - border_major,
		y - border_major,
		longest_line_width + (2 * border_major),
		(lines * font_height) + (2 * border_major)
	    );

	    /* Draw the text */
	    gdk_gc_set_function(gc, GDK_COPY);
	    gdk_gc_set_foreground(gc, c_fg);
	    for(i = 0; mesg[i] != NULL; i += 2)
	    {
		s = mesg[i];
		len = STRLEN(s);
		if(len > 0)
		{
		    gdk_string_bounds(font, s, &b);
		    DRAW_TEXT(
			window, font, &b,
			x, y,
			s, len
		    );
		}
		s = mesg[i + 1];
		len = STRLEN(s);
		if(len > 0)
		{
		    gdk_string_bounds(font, s, &b);
		    DRAW_TEXT(
			window, font, &b,
			x + column1_width, y,
			s, len
		    );
		}

		y += font_height;
	    }
	}

#undef DRAW_TEXT_DROP_SHADE
#undef DRAW_TEXT
}

/*
 *	Queues a draw for the FSImg.
 */
static void FSImgQueueDraw(EDVFSImg *fsimg)
{
	GtkWidget *w = (fsimg != NULL) ? fsimg->toplevel : NULL;
	if(w != NULL)
	    gtk_widget_queue_draw(w);
}


/*
 *	Performs the transision from one image to the next.
 *
 *	The trans_op specifies the transision operation.
 *
 *	The window specifies the GdkWindow to draw to.
 *
 *	The dither specifies the amount of Gdk RGB Buffer dithering.
 *
 *	The style specifies the GtkStyle to use to draw to the window.
 *
 *	The rgba, width, height, and bpl specifies the image data.
 *
 *	The duration_ms specifies the duration of the transision in
 *	milliseconds.
 *
 *	The bg_color specifies the 4 byte RGBA background color.
 */
static void FSImgTransision(
	const EDVPresentationModeTransisionOP trans_op,
	GdkWindow *window,
	GdkRgbDither dither,
	GtkStyle *style,
	const guint8 *rgba,
	const gint width, const gint height, const gint bpl,
	const gulong duration_ms,
	const guint8 *bg_color			/* 4 bytes RGBA */
)
{
	const gint	bpp = 4,		/* RGBA */
			_bpl = (bpl > 0) ? bpl : (width * bpp);
	gulong start_time_ms;
	guint8 *work_rgba;

	if((trans_op == EDV_PRESENTATION_MODE_TRANSISION_OP_NONE) ||
	   (window == NULL) || (bg_color == NULL) || (rgba == NULL) ||
           (width <= 0) || (height <= 0) || (_bpl <= 0) ||
	   (duration_ms == 0l)
	)
	    return;

	/* Allocate the working image buffer */
	work_rgba = (guint8 *)g_malloc(_bpl * height * sizeof(guint8));
	if(work_rgba == NULL)
	    return;

	/* Flush any pending events so that we can get a more accurate
	 * start time and produce complete transisions
	 */
	gdk_flush();
	start_time_ms = edv_time_ms();

	/* Fade Black In */
	if(trans_op == EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_BLACK_IN)
	{
	    const guint8 black_rgba[] = { 0x00, 0x00, 0x00, 0xff };
	    gint y;
	    gulong cur_time_ms, dt_ms;
	    gfloat src_coeff, tar_coeff;
	    guint8 *tar;
	    const guint8 *src, *src_end;

	    dt_ms = 0l;
	    while(dt_ms < duration_ms)
	    {
		src_coeff = (gfloat)dt_ms / (gfloat)duration_ms;
		tar_coeff = 1.0f - src_coeff;

		for(y = 0; y < height; y++)
		{
		    src = rgba + (y * _bpl);
		    tar = work_rgba + (y * _bpl);
		    src_end = src + (width * bpp);
		    while(src < src_end)
		    {
			tar[0] = (guint8)(
			    (black_rgba[0] * tar_coeff) + (src[0] * src_coeff)
			);
			tar[1] = (guint8)(
			    (black_rgba[1] * tar_coeff) + (src[1] * src_coeff)
			);
			tar[2] = (guint8)(
			    (black_rgba[2] * tar_coeff) + (src[2] * src_coeff)
			);
			tar[3] = black_rgba[3];

			src += bpp;
			tar += bpp;
		    }
		}

		/* Put the image to the window */
		gdk_draw_rgb_32_image(
		    window, style->black_gc,
		    0, 0, width, height,
		    dither,
		    (guchar *)work_rgba,
		    _bpl
		);

		cur_time_ms = edv_time_ms();
		if(cur_time_ms < start_time_ms)
		    break;

		dt_ms = cur_time_ms - start_time_ms;
	    }
	}
	/* Fade Black Out */
	else if(trans_op == EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_BLACK_OUT)
	{
	    const guint8 black_rgba[] = { 0x00, 0x00, 0x00, 0xff };
	    gint y;
	    gulong cur_time_ms, dt_ms;
	    gfloat src_coeff, tar_coeff;
	    guint8 *tar;
	    const guint8 *src, *src_end;

	    dt_ms = 0l;
	    while(dt_ms < duration_ms)
	    {
		tar_coeff = (gfloat)dt_ms / (gfloat)duration_ms;
		src_coeff = 1.0f - tar_coeff;

		for(y = 0; y < height; y++)
		{
		    src = rgba + (y * _bpl);
		    tar = work_rgba + (y * _bpl);
		    src_end = src + (width * bpp);
		    while(src < src_end)
		    {
			tar[0] = (guint8)(
			    (black_rgba[0] * tar_coeff) + (src[0] * src_coeff)
			);
			tar[1] = (guint8)(
			    (black_rgba[1] * tar_coeff) + (src[1] * src_coeff)
			);
			tar[2] = (guint8)(
			    (black_rgba[2] * tar_coeff) + (src[2] * src_coeff)
			);
			tar[3] = black_rgba[3];

			src += bpp;
			tar += bpp;
		    }
		}

		/* Put the image to the window */
		gdk_draw_rgb_32_image(
		    window, style->black_gc,
		    0, 0, width, height,
		    dither,
		    (guchar *)work_rgba,
		    _bpl
		);

		cur_time_ms = edv_time_ms();
		if(cur_time_ms < start_time_ms)
		    break;

		dt_ms = cur_time_ms - start_time_ms;
	    }
	}
	/* Fade White In */
	else if(trans_op == EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_WHITE_IN)
	{
	    const guint8 white_rgba[] = { 0xff, 0xff, 0xff, 0xff };
	    gint y;
	    gulong cur_time_ms, dt_ms;
	    gfloat src_coeff, tar_coeff;
	    guint8 *tar;
	    const guint8 *src, *src_end;

	    dt_ms = 0l;
	    while(dt_ms < duration_ms)
	    {
		src_coeff = (gfloat)dt_ms / (gfloat)duration_ms;
		tar_coeff = 1.0f - src_coeff;

		for(y = 0; y < height; y++)
		{
		    src = rgba + (y * _bpl);
		    tar = work_rgba + (y * _bpl);
		    src_end = src + (width * bpp);
		    while(src < src_end)
		    {
			tar[0] = (guint8)(
			    (white_rgba[0] * tar_coeff) + (src[0] * src_coeff)
			);
			tar[1] = (guint8)(
			    (white_rgba[1] * tar_coeff) + (src[1] * src_coeff)
			);
			tar[2] = (guint8)(
			    (white_rgba[2] * tar_coeff) + (src[2] * src_coeff)
			);
			tar[3] = white_rgba[3];

			src += bpp;
			tar += bpp;
		    }
		}

		/* Put the image to the window */
		gdk_draw_rgb_32_image(
		    window, style->black_gc,
		    0, 0, width, height,
		    dither,
		    (guchar *)work_rgba,
		    _bpl
		);

		cur_time_ms = edv_time_ms();
		if(cur_time_ms < start_time_ms)
		    break;

		dt_ms = cur_time_ms - start_time_ms;
	    }
	}
	/* Fade White Out */
	else if(trans_op == EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_WHITE_OUT)
	{
	    const guint8 white_rgba[] = { 0xff, 0xff, 0xff, 0xff };
	    gint y;
	    gulong cur_time_ms, dt_ms;
	    gfloat src_coeff, tar_coeff;
	    guint8 *tar;
	    const guint8 *src, *src_end;

	    dt_ms = 0l;
	    while(dt_ms < duration_ms)
	    {
		tar_coeff = (gfloat)dt_ms / (gfloat)duration_ms;
		src_coeff = 1.0f - tar_coeff;

		for(y = 0; y < height; y++)
		{
		    src = rgba + (y * _bpl);
		    tar = work_rgba + (y * _bpl);
		    src_end = src + (width * bpp);
		    while(src < src_end)
		    {
			tar[0] = (guint8)(
			    (white_rgba[0] * tar_coeff) + (src[0] * src_coeff)
			);
			tar[1] = (guint8)(
			    (white_rgba[1] * tar_coeff) + (src[1] * src_coeff)
			);
			tar[2] = (guint8)(
			    (white_rgba[2] * tar_coeff) + (src[2] * src_coeff)
			);
			tar[3] = white_rgba[3];

			src += bpp;
			tar += bpp;
		    }
		}

		/* Put the image to the window */
		gdk_draw_rgb_32_image(
		    window, style->black_gc,
		    0, 0, width, height,
		    dither,
		    (guchar *)work_rgba,
		    _bpl
		);

		cur_time_ms = edv_time_ms();
		if(cur_time_ms < start_time_ms)
		    break;

		dt_ms = cur_time_ms - start_time_ms;
	    }
	}
	/* Fade Background In */
	else if(trans_op == EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_BG_IN)
	{
	    gint y;
	    gulong cur_time_ms, dt_ms;
	    gfloat src_coeff, tar_coeff;
	    guint8 *tar;
	    const guint8 *src, *src_end;

	    dt_ms = 0l;
	    while(dt_ms < duration_ms)
	    {
		src_coeff = (gfloat)dt_ms / (gfloat)duration_ms;
		tar_coeff = 1.0f - src_coeff;

		for(y = 0; y < height; y++)
		{
		    src = rgba + (y * _bpl);
		    tar = work_rgba + (y * _bpl);
		    src_end = src + (width * bpp);
		    while(src < src_end)
		    {
			tar[0] = (guint8)(
			    (bg_color[0] * tar_coeff) + (src[0] * src_coeff)
			);
			tar[1] = (guint8)(
			    (bg_color[1] * tar_coeff) + (src[1] * src_coeff)
			);
			tar[2] = (guint8)(
			    (bg_color[2] * tar_coeff) + (src[2] * src_coeff)
			);
			tar[3] = bg_color[3];

			src += bpp;
			tar += bpp;
		    }
		}

		/* Put the image to the window */
		gdk_draw_rgb_32_image(
		    window, style->black_gc,
		    0, 0, width, height,
		    dither,
		    (guchar *)work_rgba,
		    _bpl
		);

		cur_time_ms = edv_time_ms();
		if(cur_time_ms < start_time_ms)
		    break;

		dt_ms = cur_time_ms - start_time_ms;
	    }
	}
	/* Fade Background Out */
	else if(trans_op == EDV_PRESENTATION_MODE_TRANSISION_OP_FADE_BG_OUT)
	{
	    gint y;
	    gulong cur_time_ms, dt_ms;
	    gfloat src_coeff, tar_coeff;
	    guint8 *tar;
	    const guint8 *src, *src_end;

	    dt_ms = 0l;
	    while(dt_ms < duration_ms)
	    {
		tar_coeff = (gfloat)dt_ms / (gfloat)duration_ms;
		src_coeff = 1.0f - tar_coeff;

		for(y = 0; y < height; y++)
		{
		    src = rgba + (y * _bpl);
		    tar = work_rgba + (y * _bpl);
		    src_end = src + (width * bpp);
		    while(src < src_end)
		    {
			tar[0] = (guint8)(
			    (bg_color[0] * tar_coeff) + (src[0] * src_coeff)
			);
			tar[1] = (guint8)(
			    (bg_color[1] * tar_coeff) + (src[1] * src_coeff)
			);
			tar[2] = (guint8)(
			    (bg_color[2] * tar_coeff) + (src[2] * src_coeff)
			);
			tar[3] = bg_color[3];

			src += bpp;
			tar += bpp;
		    }
		}

		/* Put the image to the window */
		gdk_draw_rgb_32_image(
		    window, style->black_gc,
		    0, 0, width, height,
		    dither,
		    (guchar *)work_rgba,
		    _bpl
		);

		cur_time_ms = edv_time_ms();
		if(cur_time_ms < start_time_ms)
		    break;

		dt_ms = cur_time_ms - start_time_ms;
	    }
	}
	/* Move Horizontal In */
	else if(trans_op == EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_HIN)
	{
	    gint x, y;
	    gulong cur_time_ms, dt_ms;
	    gfloat coeff;

	    y = 0;
	    dt_ms = 0l;
	    while(dt_ms < duration_ms)
	    {
		coeff = (gfloat)dt_ms / (gfloat)duration_ms;
		x = (gint)(width * (1.0f - coeff));

		FSImgClearImageData(
		    work_rgba,
		    width, height, _bpl,
		    bg_color
		);

		GUIImageBufferCopyArea(
		    4 * sizeof(guint8),
		    rgba,
		    width, height, _bpl,
		    work_rgba,
		    width, height, _bpl,
		    x, y,
		    FALSE,			/* Do not blend */
		    NULL, NULL
		);

		/* Put the image to the window */
		gdk_draw_rgb_32_image(
		    window, style->black_gc,
		    0, 0, width, height,
		    dither,
		    (guchar *)work_rgba,
		    _bpl
		);

		cur_time_ms = edv_time_ms();
		if(cur_time_ms < start_time_ms)
		    break;

		dt_ms = cur_time_ms - start_time_ms;
	    }
	}
	/* Move Horizontal Out */
	else if(trans_op == EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_HOUT)
	{
	    gint x, y;
	    gulong cur_time_ms, dt_ms;
	    gfloat coeff;

	    y = 0;
	    dt_ms = 0l;
	    while(dt_ms < duration_ms)
	    {
		coeff = (gfloat)dt_ms / (gfloat)duration_ms;
		x = (gint)(-width * coeff);

		FSImgClearImageData(
		    work_rgba,
		    width, height, _bpl,
		    bg_color
		);

		GUIImageBufferCopyArea(
		    4 * sizeof(guint8),
		    rgba,
		    width, height, _bpl,
		    work_rgba,
		    width, height, _bpl,
		    x, y,
		    FALSE,			/* Do not blend */
		    NULL, NULL
		);

		/* Put the image to the window */
		gdk_draw_rgb_32_image(
		    window, style->black_gc,
		    0, 0, width, height,
		    dither,
		    (guchar *)work_rgba,
		    _bpl
		);

		cur_time_ms = edv_time_ms();
		if(cur_time_ms < start_time_ms)
		    break;

		dt_ms = cur_time_ms - start_time_ms;
	    }
	}
	/* Move Vertical In */
	else if(trans_op == EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_VIN)
	{
	    gint x, y;
	    gulong cur_time_ms, dt_ms;
	    gfloat coeff;

	    x = 0;
	    dt_ms = 0l;
	    while(dt_ms < duration_ms)
	    {
		coeff = (gfloat)dt_ms / (gfloat)duration_ms;
		y = (gint)(-height * (1.0f - coeff));

		FSImgClearImageData(
		    work_rgba,
		    width, height, _bpl,
		    bg_color
		);

		GUIImageBufferCopyArea(
		    4 * sizeof(guint8),
		    rgba,
		    width, height, _bpl,
		    work_rgba,
		    width, height, _bpl,
		    x, y,
		    FALSE,			/* Do not blend */
		    NULL, NULL
		);

		/* Put the image to the window */
		gdk_draw_rgb_32_image(
		    window, style->black_gc,
		    0, 0, width, height,
		    dither,
		    (guchar *)work_rgba,
		    _bpl
		);

		cur_time_ms = edv_time_ms();
		if(cur_time_ms < start_time_ms)
		    break;

		dt_ms = cur_time_ms - start_time_ms;
	    }
	}
	/* Move Vertical Out */
	else if(trans_op == EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_VOUT)
	{
	    gint x, y;
	    gulong cur_time_ms, dt_ms;
	    gfloat coeff;

	    x = 0;
	    dt_ms = 0l;
	    while(dt_ms < duration_ms)
	    {
		coeff = (gfloat)dt_ms / (gfloat)duration_ms;
		y = (gint)(height * coeff);

		FSImgClearImageData(
		    work_rgba,
		    width, height, _bpl,
		    bg_color
		);

		GUIImageBufferCopyArea(
		    4 * sizeof(guint8),
		    rgba,
		    width, height, _bpl,
		    work_rgba,
		    width, height, _bpl,
		    x, y,
		    FALSE,			/* Do not blend */
		    NULL, NULL
		);

		/* Put the image to the window */
		gdk_draw_rgb_32_image(
		    window, style->black_gc,
		    0, 0, width, height,
		    dither,
		    (guchar *)work_rgba,
		    _bpl
		);

		cur_time_ms = edv_time_ms();
		if(cur_time_ms < start_time_ms)
		    break;

		dt_ms = cur_time_ms - start_time_ms;
	    }
	}
	/* Move Diagonal In */
	else if(trans_op == EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_DIN)
	{
	    gint x, y;
	    gulong cur_time_ms, dt_ms;
	    gfloat coeff;

	    dt_ms = 0l;
	    while(dt_ms < duration_ms)
	    {
		coeff = (gfloat)dt_ms / (gfloat)duration_ms;
		x = (gint)(width * (1.0f - coeff));
		y = (gint)(-height * (1.0f - coeff));

		FSImgClearImageData(
		    work_rgba,
		    width, height, _bpl,
		    bg_color
		);

		GUIImageBufferCopyArea(
		    4 * sizeof(guint8),
		    rgba,
		    width, height, _bpl,
		    work_rgba,
		    width, height, _bpl,
		    x, y,
		    FALSE,			/* Do not blend */
		    NULL, NULL
		);

		/* Put the image to the window */
		gdk_draw_rgb_32_image(
		    window, style->black_gc,
		    0, 0, width, height,
		    dither,
		    (guchar *)work_rgba,
		    _bpl
		);

		cur_time_ms = edv_time_ms();
		if(cur_time_ms < start_time_ms)
		    break;

		dt_ms = cur_time_ms - start_time_ms;
	    }
	}
	/* Move Diagonal Out */
	else if(trans_op == EDV_PRESENTATION_MODE_TRANSISION_OP_MOVE_DOUT)
	{
	    gint x, y;
	    gulong cur_time_ms, dt_ms;
	    gfloat coeff;

	    dt_ms = 0l;
	    while(dt_ms < duration_ms)
	    {
		coeff = (gfloat)dt_ms / (gfloat)duration_ms;
		x = (gint)(-width * coeff);
		y = (gint)(height * coeff);

		FSImgClearImageData(
		    work_rgba,
		    width, height, _bpl,
		    bg_color
		);

		GUIImageBufferCopyArea(
		    4 * sizeof(guint8),
		    rgba,
		    width, height, _bpl,
		    work_rgba,
		    width, height, _bpl,
		    x, y,
		    FALSE,			/* Do not blend */
		    NULL, NULL
		);

		/* Put the image to the window */
		gdk_draw_rgb_32_image(
		    window, style->black_gc,
		    0, 0, width, height,
		    dither,
		    (guchar *)work_rgba,
		    _bpl
		);

		cur_time_ms = edv_time_ms();
		if(cur_time_ms < start_time_ms)
		    break;

		dt_ms = cur_time_ms - start_time_ms;
	    }
	}

	g_free(work_rgba);
}

/*
 *	Clears the image data with a color.
 *
 *	The rgba, width, height, and bpl specifies the image data.
 *
 *	The bg_color specifies the 4 byte RGBA value to clear the
 *	image data with.
 */
static void FSImgClearImageData(
	guint8 *rgba,
	const gint width, const gint height, const gint bpl,
	const guint8 *bg_color
)
{
	const gint	bpp = 4;		/* RGBA */
	gint		y,
			_bpl = bpl;
	guint32		*ptr32, *end32,
			bg_color32;

	if((rgba == NULL) || (bg_color == NULL))
	    return;

	if(_bpl <= 0)
	    _bpl = width * bpp;

	bg_color32 = *(const guint32 *)bg_color;

	for(y = 0; y < height; y++)
	{
	    ptr32 = (guint32 *)(rgba + (y * bpl));
	    end32 = ptr32 + width;
	    while(ptr32 < end32)
		*ptr32++ = bg_color32;
	}
}

/*
 *	Creates an RGBA image data for final display on a window.
 *
 *	The win_width and win_height specifies the size of the window
 *	that the image is intended to be displayed on.
 *
 *	The bg_color specifies the 4 bytes RGBA background color. If
 *	bg_color is NULL then the returned image data's background
 *	pixels are undefined.
 *
 *	The rgba, width, height, and bpl specifies the image data.
 *
 *	Inputs assumed valid.
 *
 *	Returns the RGBA image data of size win_width and win_height
 *	for final display or NULL on error.
 */
static guint8 *FSImgCreateWinImageData(
	const gint win_width, const gint win_height,
	const guint8 *rgba,
	const gint width, const gint height, const gint bpl,
	const guint8 *bg_color			/* 4 bytes RGBA */
)
{
	const gint	bpp = 4 * sizeof(guint8),
			win_bpl = win_width * bpp;
	guint8 *win_rgba;

	if((win_width <= 0) || (win_height <= 0))
	    return(NULL);

	/* Allocate the image data for the final display */
	win_rgba = (guint8 *)g_malloc(win_bpl * win_height);
	if(win_rgba == NULL)
	    return(NULL);

	/* Clear the target image data with the background color */
	FSImgClearImageData(
	    win_rgba,
	    win_width, win_height, win_bpl,
	    bg_color
	);

	/* Copy the source image to center of the target window image */
	GUIImageBufferCopyArea(
	    bpp,
	    rgba,
	    width, height, bpl,
	    win_rgba,
	    win_width, win_height, win_bpl,
	    (win_width / 2) - (width / 2),
	    (win_height / 2) - (height / 2),
	    TRUE,				/* Blend */
	    NULL, NULL
	);

	return(win_rgba);
}

/*
 *	Displays the specified RGBA image data.
 *
 *	Resizes the fsimg to display the specified RGBA image data using
 *	a suitable video mode size.
 *
 *	Recreates the pixmap and puts the specified RGBA image data on
 *	to it.
 *
 *	Switches video modes as needed.
 *
 *	Grabs the pointer.
 *
 *	Returns values:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error.
 *	-4	User aborted.
 *	-5	No video mode available.
 */
static gint FSImgSetData(
	EDVFSImg *fsimg,
	const gint width, const gint height, const gint bpl,
	GList *rgba_list,			/* guint8 * */
	GList *delay_list,			/* gulong */
	const guint8 *bg_color			/* 4 bytes RGBA */
)
{
	gboolean need_change_vidmode = FALSE;
	gint win_width, win_height;
	GList *glist;
	GdkVideoModeInfo	*m_sel,
				*m;
	GdkRgbDither dither;
	GdkWindow *window;
	GtkStyle *style;
	GtkWidget *w;
	CfgList *cfg_list;
	EDVCore *core;

	if((fsimg == NULL) || (rgba_list == NULL))
	    return(-2);

	core = fsimg->core;
	cfg_list = core->cfg_list;

	dither = FSImgGetDither(fsimg);
	w = fsimg->toplevel;
	window = w->window;
	style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
	    return(-1);

	/* Fade out */
	if(fsimg->cur_frame != NULL)
	{
	    GList *glist = fsimg->cur_frame;
	    EDVFSImgFrame *frame = FSIMG_FRAME(glist->data);
	    if(frame != NULL)
	    {
		const edv_presentation_mode_transision trans =
		    EDV_GET_I(EDV_CFG_PARM_PRESENTATION_TRANSISION);
		FSImgTransision(
		    FSImgGetTransisionOpFromTransision(
			trans,
			FALSE			/* Out */
		    ),
		    window,
		    dither,
		    style,
		    frame->rgba,
		    fsimg->frame_width, fsimg->frame_height, -1,
		    EDV_GET_L(EDV_CFG_PARM_PRESENTATION_TRANSISION_INT) / 2l,
		    bg_color			/* 4 bytes RGBA */
		);
	    }
	}

	/* Update the size of the image on file */
	fsimg->img_width = width;
	fsimg->img_height = height;

	/* Find and nominate a new video mode with a suitable size to
	 * display the image
	 */
	m_sel = NULL;
	for(glist = fsimg->vidmodes_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    m = GDK_VIDEO_MODE_INFO(glist->data);
	    if(m == NULL)
		continue;

	    /* Size too small to fit image? */
	    if((m->viewport_width < width) ||
	       (m->viewport_height < height)
	    )
		continue;

	    if(m_sel != NULL)
	    {
		/* This suitable video mode smaller than the last
		 * nominated one
		 */
		if((m->viewport_width < m_sel->viewport_width) ||
		   (m->viewport_height < m_sel->viewport_height)
		)
		    m_sel = m;
	    }
	    else
	    {
		/* No video mode nominated yet, so nominate this one */
		m_sel = m;
	    }
	}
	/* If no video mode was nominated then nominatee the largest
	 * video mode that is available
	 */
	if(m_sel == NULL)
	{
	    /* Nominate largest video mode that is available */
	    for(glist = fsimg->vidmodes_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		m = GDK_VIDEO_MODE_INFO(glist->data);
		if(m == NULL)
		    continue;

		if(m_sel != NULL)
		{
		    /* This video mode larger than the last nominated
		     * video mode?
		     */
		    if((m->viewport_width > m_sel->viewport_width) ||
		       (m->viewport_height > m_sel->viewport_height)
		    )
			m_sel = m;
		}
		else
		{
		    /* No video mode nominated yet, so nominate this one */
		    m_sel = m;
		}
	    }
	}

	/* If there is still no video mode nominated then just use the
	 * current video mode, otherwise determine if we need to change
	 * video modes
	 */
	if(m_sel == NULL)
	    m_sel = gdk_video_mode_get_current(fsimg->vidmodes_list);
	else
	    need_change_vidmode = (m_sel != gdk_video_mode_get_current(fsimg->vidmodes_list)) ?
		TRUE : FALSE;
	if(m_sel == NULL)
	    return(-5);

	/* Get values of the new video mode and record the previous
	 * video mode
	 */
	fsimg->prev_vidmode = fsimg->cur_vidmode;
	fsimg->cur_vidmode = m_sel;
	win_width = m_sel->viewport_width;
	win_height = m_sel->viewport_height;
	if((win_width <= 0) || (win_height <= 0))
	    return(-2);

	/* Stop the frames draw timeout */
	fsimg->frame_next_toid = GTK_TIMEOUT_REMOVE(fsimg->frame_next_toid);

	/* Clear the current frames list */
	if(fsimg->frames_list != NULL)
	{
	    g_list_foreach(
		fsimg->frames_list, (GFunc)EDVFSImgFrameDelete, NULL
	    );
	    g_list_free(fsimg->frames_list);
	    fsimg->frames_list = NULL;
	    fsimg->cur_frame = fsimg->frames_list;
	    fsimg->frame_width = 0;
	    fsimg->frame_height = 0;
	}

	/* Need to resize the image smaller to fit? */
	if((width > win_width) || (height > win_height))
	{
	    const gint bpp = 4;
	    const gfloat aspect = (gfloat)width / (gfloat)height;
	    gint	rs_width = win_height * aspect,
			rs_height = win_height,
			rs_bpl;
	    gulong delay;
	    const guint8 *rgba;
	    guint8 *rs_rgba;
	    GList *glist_rgba, *glist_delay;
	    EDVFSImgFrame *frame;

	    /* Calculate the resized image size */
	    if((rs_width > win_width) && (aspect > 0.0f))
	    {
		rs_width = win_width;
		rs_height = win_width / aspect;
	    }
	    if((rs_width <= 0) || (rs_height <= 0))
		return(-2);

	    rs_bpl = rs_width * bpp;

	    /* Set the shown image's size as the resized image size */
	    fsimg->img_shown_width = rs_width;
	    fsimg->img_shown_height = rs_height;

	    /* Set the frames' allocated image data size */
	    fsimg->frame_width = win_width;
	    fsimg->frame_height = win_height;

	    /* The size of smaller image that will fit within the
	     * window has now been calculated
	     */

	    /* Iterate through the image data list */
	    for(glist_rgba = rgba_list,
		glist_delay = delay_list;
		glist_rgba != NULL;
	        glist_rgba = g_list_next(glist_rgba),
		glist_delay = g_list_next(glist_delay)
	    )
	    {
		rgba = (guint8 *)glist_rgba->data;
		if(rgba == NULL)
		    continue;

		delay = (glist_delay != NULL) ?
		    (gulong)glist_delay->data : 0l;

		/* Create a new frame */
		frame = EDVFSImgFrameNew();
		if(frame == NULL)
		    break;

		/* Append the new frame to the frames list */
		fsimg->frames_list = g_list_append(
		    fsimg->frames_list,
		    frame
		);

		/* Allocate the resized RGBA image data */
		rs_rgba = (guint8 *)g_malloc(rs_bpl * rs_height);
		if(rs_rgba == NULL)
		    break;

		/* Copy/resize the specified image data to the new
		 * resized image data
		 */
		GUIImageBufferResize(
		    bpp,
		    rgba, width, height, bpl,
		    rs_rgba, rs_width, rs_height, rs_bpl,
		    NULL, NULL
		);

		/* Create the window's RGBA image data and put the
		 * new resized RGBA image data into it
		 */
		frame->rgba = FSImgCreateWinImageData(
		    win_width, win_height,
		    rs_rgba,
		    rs_width, rs_height, rs_bpl,
		    bg_color
		);

		/* Delete the resized RGBA image data */
		g_free(rs_rgba);

		/* Set the delay */
		frame->delay = delay;
	    }
	}
	else
	{
	    /* No need to resize the image to fit */
	    gulong delay;
	    const guint8 *rgba;
	    GList *glist_rgba, *glist_delay;
	    EDVFSImgFrame *frame;

	    /* Set the shown image's size */
	    fsimg->img_shown_width = width;
	    fsimg->img_shown_height = height;

	    /* Set the frames' allocated image data size */
	    fsimg->frame_width = win_width;
	    fsimg->frame_height = win_height;

	    /* Iterate through the image data list */
	    for(glist_rgba = rgba_list,
		glist_delay = delay_list;
		glist_rgba != NULL;
	        glist_rgba = g_list_next(glist_rgba),
		glist_delay = g_list_next(glist_delay)
	    )
	    {
		rgba = (guint8 *)glist_rgba->data;
		if(rgba == NULL)
		    continue;

		delay = (glist_delay != NULL) ?
		    (gulong)glist_delay->data : 0l;

		/* Create a new frame */
		frame = EDVFSImgFrameNew();
		if(frame == NULL)
		    break;

		/* Append the new frame to the frames list */
		fsimg->frames_list = g_list_append(
		    fsimg->frames_list,
		    frame
		);

		/* Create the image data suitable for displaying
		 * on the window
		 */
		frame->rgba = FSImgCreateWinImageData(
		    win_width, win_height,
		    rgba,
		    width, height, bpl,
		    bg_color
		);

		/* Set the delay */
		frame->delay = delay;
	    }
	}

	/* Set the current frame */
	fsimg->cur_frame = fsimg->frames_list;

	/* Recreate the toplevel's GdkPixmap */
	GDK_PIXMAP_UNREF(fsimg->pixmap);
	fsimg->pixmap = gdk_pixmap_new(
	    window,
	    win_width, win_height,
	    -1
	);

	/* Draw the first frame on to the toplevel's GdkPixmap */
	FSImgDrawPixmapFromFrame(
	    fsimg,
	    FSIMG_FRAME(g_list_nth_data(fsimg->frames_list, 0))
	);

	/* Move and resize the FSImg window to display the image
	 *
	 * The widget needs to have its user position and size set for
	 * it's child widgets to configure correctly and the widget's
	 * window has to be moved and resized for the actual affect to
	 * take place for the window
	 */
	gtk_widget_set_usize(w, win_width, win_height);
	gtk_widget_set_uposition(w, 0, 0);
	gdk_window_move_resize(
	    window,
	    0, 0,
	    win_width, win_height
	);

	/* Map and raise the FSImg window */
	gtk_widget_show_raise(w);

	/* Grab the FSImg toplevel window to confine pointer and
	 * receive key events
	 */
	FSImgGrab(fsimg);

	/* Need to change video modes? */
	if(need_change_vidmode)
	{
	    /* Move pointer to center of window */
	    FSImgWarpPointer(
		window,
		win_width / 2, win_height / 2
	    );
	    /* Switch to the nominated video mode */
	    gdk_video_mode_switch(m_sel);
	}

	/* Viewport will be set when the "configure_event" signal is
	 * received
	 */

	/* Redraw since image changed and fade in */
	fsimg->need_transision_in = TRUE;
	FSImgQueueDraw(fsimg);

	FSImgNavigatorUpdate(fsimg);

	/* Schedual the next frame draw */
	if(g_list_length(fsimg->frames_list) > 1)
	{
	    EDVFSImgFrame *frame = FSIMG_FRAME(g_list_nth_data(
		fsimg->frames_list, 0
	    ));
	    if(frame != NULL)
		fsimg->frame_next_toid = gtk_timeout_add(
		    MAX(frame->delay, 1l),
		    FSImgNextFrameTOCB, fsimg
		);
	}

	return(0);
}

/*
 *	Clears the fsimg.
 *
 *	Resizes the fsimg with the specified size using a suitable video
 *	mode size.
 *
 *	Recreates the pixmap and clears it with just the background.
 *
 *	Switches video modes as needed.
 *
 *	Grabs the pointer.
 *
 *	Returns values: 
 *
 *	0	Success
 *	-1	General error
 *	-2	Invalid value.   
 *	-3	Systems error
 *	-4	User aborted
 *	-5	No video mode available
 */
static gint FSImgClear(
	EDVFSImg *fsimg,
	const gint width, const gint height
) 
{
	gboolean need_change_vidmode = FALSE;
	gint	_width = width,
		_height = height,
		win_width, win_height;
	guint8 bg_color[4];
	GList *glist;
	const GdkVideoModeInfo *m_cur;
	GdkVideoModeInfo	*m_sel,
				*m;
	GdkRgbDither dither;
	GdkWindow *window;
	GtkStyle *style;
	GtkWidget *w;
	CfgList *cfg_list;
	EDVCore *core;

	if(fsimg == NULL)
	    return(-1);

	core = fsimg->core;
	cfg_list = core->cfg_list; 

	dither = FSImgGetDither(fsimg);
	w = fsimg->toplevel;
	window = w->window;
	style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
	    return(-1);

	/* Get the configured background color in RGBA */
	(void)FSImgGetBGColorRGBA(fsimg, bg_color);

	/* Fade out */
	if(fsimg->cur_frame != NULL)
	{
	    GList *glist = fsimg->cur_frame;
	    EDVFSImgFrame *frame = FSIMG_FRAME(glist->data);
	    if(frame != NULL)
	    {
		const edv_presentation_mode_transision trans =
		    EDV_GET_I(EDV_CFG_PARM_PRESENTATION_TRANSISION);
		FSImgTransision(
		    FSImgGetTransisionOpFromTransision(
			trans,
			FALSE			/* Out */
		    ),
		    window,
		    dither,
		    style,
		    frame->rgba,
		    fsimg->frame_width, fsimg->frame_height, -1,
		    EDV_GET_L(EDV_CFG_PARM_PRESENTATION_TRANSISION_INT) / 2l,
		    bg_color			/* 4 bytes RGBA */
		);
	    }
	}

	/* Reset the image size and the shown image size */
	fsimg->img_width = 0;
	fsimg->img_height = 0;
	fsimg->img_shown_width = 0;
	fsimg->img_shown_height = 0;

	/* Use size of current video mode? */
	m_cur = fsimg->cur_vidmode;
	if(m_cur != NULL)
	{
	    if(_width <= 0)
		_width = m_cur->viewport_width;
	    if(_height <= 0)
		_height = m_cur->viewport_height;
	}
	if((_width <= 0) || (_height <= 0))
	    return(-5);

	/* Find and nominate a new video mode with a suitable size */
	m_sel = NULL;
	for(glist = fsimg->vidmodes_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    m = GDK_VIDEO_MODE_INFO(glist->data);
	    if(m == NULL)
		continue;

	    /* Size too small? */
	    if((m->viewport_width < _width) ||
	       (m->viewport_height < _height)
	    )
		continue;

	    if(m_sel != NULL)
	    {
		/* This suitable video mode smaller than the last
		 * nominated one?
		 */
		if((m->viewport_width < m_sel->viewport_width) ||
		   (m->viewport_height < m_sel->viewport_height)
		)
		    m_sel = m;
	    }
	    else
	    {
		/* No video mode nominated yet, so nominate this one */
		m_sel = m;
	    }
	}
	/* If no video mode nominated then nominatee the largest video
	 * mode that is available
	 */
	if(m_sel == NULL)
	{
	    for(glist = fsimg->vidmodes_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		m = GDK_VIDEO_MODE_INFO(glist->data);
		if(m == NULL)
		    continue;

		if(m_sel != NULL)
		{
		    /* This video mode larger than the last nominated
		     * video mode?
		     */
		    if((m->viewport_width > m_sel->viewport_width) ||
		       (m->viewport_height > m_sel->viewport_height)
		    )
			m_sel = m;
		}
		else
		{   
		    /* No video mode nominated yet, so nominate this one */
		    m_sel = m;
		}
	    }
	}

	/* If there is still no video mode nominated then just use the
	 * current video mode, otherwise determine if we need to change
	 * video modes
	 */
	if(m_sel == NULL)
	    m_sel = gdk_video_mode_get_current(fsimg->vidmodes_list);
	else
	    need_change_vidmode = (m_sel != gdk_video_mode_get_current(fsimg->vidmodes_list)) ?
		TRUE : FALSE;
	if(m_sel == NULL)
	    return(-5);

	/* Get values of the new video mode and record the previous
	 * video mode
	 */
	fsimg->prev_vidmode = fsimg->cur_vidmode;
	fsimg->cur_vidmode = m_sel;
	win_width = m_sel->viewport_width;
	win_height = m_sel->viewport_height;
	if((win_width <= 0) || (win_height <= 0))
	    return(-2);

	/* Stop the frames draw timeout */
	fsimg->frame_next_toid = GTK_TIMEOUT_REMOVE(fsimg->frame_next_toid);

	/* Clear the current frames list */
	if(fsimg->frames_list != NULL)
	{
	    g_list_foreach(
		fsimg->frames_list, (GFunc)EDVFSImgFrameDelete, NULL
	    );
	    g_list_free(fsimg->frames_list);
	    fsimg->frames_list = NULL;
	    fsimg->cur_frame = fsimg->frames_list;
	    fsimg->frame_width = 0;
	    fsimg->frame_height = 0;
	}

	/* Recreate the pixmap */
	if(bg_color != NULL)
	{
	    GdkPixmap *pixmap;
	    guint8 *rgba = (guint8 *)g_malloc(
		win_width * win_height * 4 * sizeof(guint8)
	    );
	    if(rgba != NULL)
	    {
		guint32	*ptr = (guint32 *)rgba,
			*end = ptr + (win_width * win_height);
		while(ptr < end)
		    *ptr++ = *(const guint32 *)bg_color;
	    }

	    /* Recreate the window's GdkPixmap and put the window's   
	     * RGBA image data on it
	     */
	    GDK_PIXMAP_UNREF(fsimg->pixmap);
	    fsimg->pixmap = pixmap = gdk_pixmap_new(
		window, win_width, win_height, -1
	    );
	    if((pixmap != NULL) && (rgba != NULL))
		gdk_draw_rgb_32_image(
		    pixmap, style->black_gc,
		    0, 0, win_width, win_height,
		    dither,
		    (guchar *)rgba,
		    win_width * 4
		);

	    /* Delete the window's RGBA image data */
	    g_free(rgba);
	}

	/* Move and resize the FSImg window to the new size
	 *
	 * The widget needs to have its user position and size set for
	 * it's child widgets to configure correctly and the widget's
	 * window has to be moved and resized for the actual affect to
	 * take place for the window
	 */
	gtk_widget_set_usize(w, win_width, win_height);
	gtk_widget_set_uposition(w, 0, 0);
	gdk_window_move_resize(
	    window,
	    0, 0,
	    win_width, win_height
	);

	/* Map and raise the FSImg window */
	gtk_widget_show_raise(w);

	/* Grab the FSImg toplevel window to confine pointer and
	 * receive key events
	 */
	FSImgGrab(fsimg);

	/* Need to change video modes? */
	if(need_change_vidmode)
	{
	    /* Move pointer to center of window */
	    FSImgWarpPointer(
		window,
		win_width / 2, win_height / 2
	    );
	    /* Switch to the nominated video mode */
	    gdk_video_mode_switch(m_sel);
	}

	/* Viewport will be set when the "configure_event" signal is
	 * received
	 */

	/* Redraw since image changed */
	FSImgQueueDraw(fsimg);

	FSImgNavigatorUpdate(fsimg);

	return(0);
}


/*
 *	Opens the specified image and switches video modes as needed.
 *
 *	Returns values: 
 *
 *	0	Success
 *	-1	General error
 *	-2	Invalid value.   
 *	-3	Systems error
 *	-4	User aborted
 *	-5	No video mode available
 */
static gint FSImgOpen(
	EDVFSImg *fsimg,
	const gchar *path
)
{
	gint		status,
			width, height,
			bpl,
			nframes;
	guint8		bg_color[4];
	GList		*rgba_list,
			*delay_list;
	GdkWindow *window;
	GtkWidget *w;
	CfgList *cfg_list;
	EDVCore *core;

	if((fsimg == NULL) || STRISEMPTY(path))
	    return(-2);

	core = fsimg->core;
	cfg_list = core->cfg_list;

	FSImgHelpShow(fsimg, FALSE);
	FSImgMesg(fsimg, NULL);

	w = fsimg->toplevel;
	if(w == NULL)
	    return(-1);

	window = w->window;
	if(window == NULL)
	    return(-1);

	/* Get the configured background color in RGBA */
	(void)FSImgGetBGColorRGBA(fsimg, bg_color);

	/* Open the image and get the RGBA image data */
	rgba_list = edv_image_open_rgba(
	    path,
	    &width, &height,
	    &bpl,
	    &nframes,
	    &delay_list,
	    EDV_GET_B(EDV_CFG_PARM_PRESENTATION_OVERRIDE_IMAGE_BG_COLOR) ?
		NULL : bg_color,
	    NULL, NULL
	);
	if(rgba_list == NULL)
	{
	    gchar *msg;

	    FSImgClear(fsimg, -1, -1);
	    msg = g_strdup_printf(
		"Unable to open image:\n\n    %s",
		g_basename(path)
	    );
	    FSImgMesg(fsimg, msg);
	    g_free(msg);
	    edv_play_sound_error(fsimg->core);

	    g_list_free(delay_list);

	    return(-1);
	}

	/* Set the image data and switch video mode as needed */
	status = FSImgSetData(
	    fsimg,
	    width, height, bpl,
	    rgba_list,
	    delay_list,
	    bg_color				/* 4 bytes RGBA */
	);
	if(status != 0)
	{
	    gchar *msg;
	    switch(status)
	    {
	      case -1:
		msg = g_strdup(
		    "Error occured while displaying image data"
		);
		break;
	      case -2:
		msg = g_strdup(
		    "Invalid value in image data"
		);
		break;
	      case -3:
		msg = g_strdup(
		    "Memory allocation error"
		);
		break;
	      case -4:
		msg = g_strdup(
		    "User interrupted"
		);
		break;
	      case -5:
		msg = g_strdup(
		    "No video mode available to display image data"
		);
		break;
	      default:
		msg = g_strdup(
		    "Error occured while displaying image data"
		);
		break;
	    }
	    FSImgMesg(fsimg, msg);
	    g_free(msg);
	    edv_play_sound_error(fsimg->core);
	    FSImgClear(fsimg, -1, -1);
	}

	/* Reset the rotate position marker */
	fsimg->rotate_position = EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE;

	/* Delete the image data and delay list */
	g_list_foreach(rgba_list, (GFunc)g_free, NULL);
	g_list_free(rgba_list);
	g_list_free(delay_list);

	return(status);
}

/*
 *	Displays the next image on the FSImg.
 *
 *	Opens the next image and switches video modes as needed.
 */
static void FSImgNext(EDVFSImg *fsimg)
{
	gboolean image_opened = FALSE;
	gint i;
	const gchar	*name,
			*ext;
	GList *glist;
	GtkCList *clist;
	EDVVFSObject *obj;
	EDVCore *core;
	if(fsimg == NULL)
	    return;

	clist = GTK_CLIST(fsimg->clist);
	core = fsimg->core;

	FSImgSetBusy(fsimg, TRUE);
	FSImgNavigatorUpdate(fsimg);

	glist = clist->selection_end;

	/* Iterate from the next selected row until the last row is
	 * reached or an image is opened
	 */
	for(i = (glist != NULL) ? MAX(((gint)glist->data + 1), 0) : 0;
	    i < clist->rows;
	    i++
	)
	{
	    obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, i));
	    if(obj == NULL)
		continue;

	    name = obj->name;
	    if(name == NULL)
		continue;

	    ext = edv_name_get_extension(name);
	    if(!edv_image_is_supported_extension(ext))
		continue;

	    /* Open the image to the FSImg and switch video mode as
	     * needed
	     */
	    (void)FSImgOpen(
		fsimg,
		obj->path
	    );
	    image_opened = TRUE;

	    /* Select list row associated with this image */
	    gtk_clist_unselect_all(clist);
	    gtk_clist_select_row(clist, i, 0);

	    /* Update the information as needed */
	    FSImgInfoShow(fsimg, FSImgInfoShown(fsimg));

	    break;
	}

	/* Beep if no image was opened */
	if(!image_opened)
	    edv_play_sound_beep(core);

	FSImgSetBusy(fsimg, FALSE);
	FSImgNavigatorUpdate(fsimg);
}

/*
 *	Displays the previous image on the FSImg.
 *
 *	Opens the previous image and switches video modes as needed.
 */
static void FSImgPrev(EDVFSImg *fsimg)
{
	gboolean image_opened = FALSE;
	gint i;
	const gchar	*name,
			*ext;
	GList *glist;
	GtkCList *clist;
	EDVVFSObject *obj;
	EDVCore *core;
	if(fsimg == NULL)
	    return;

	clist = GTK_CLIST(fsimg->clist);
	core = fsimg->core;

	FSImgSetBusy(fsimg, TRUE);
	FSImgNavigatorUpdate(fsimg);

	glist = clist->selection_end;

	/* Iterate from the previous selected row until the first row
	 * is reached or an image is opened
	 */
	for(i = (glist != NULL) ?
		MIN(((gint)glist->data - 1), (clist->rows - 1)) : 0;
	    i >= 0;
	    i--
	)
	{
	    obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, i));
	    if(obj == NULL)
		continue;

	    name = obj->name;
	    if(name == NULL)
		continue;

	    ext = edv_name_get_extension(name);
	    if(!edv_image_is_supported_extension(ext))
		continue;

	    /* Open the image to the FSImg and switch video mode as
	     * needed
	     */
	    (void)FSImgOpen(
		fsimg,
		obj->path
	    );
	    image_opened = TRUE;

	    /* Select list row associated with this image */
	    gtk_clist_unselect_all(clist);
	    gtk_clist_select_row(clist, i, 0);

	    /* Update the information as needed */
	    FSImgInfoShow(fsimg, FSImgInfoShown(fsimg));

	    break;
	}

	/* Beep if no image was opened */
	if(!image_opened)
	    edv_play_sound_beep(core);

	FSImgSetBusy(fsimg, FALSE);
	FSImgNavigatorUpdate(fsimg);
}

/*
 *	Displays the first image on the FSImg.
 *
 *	Opens the first image and switches video modes as needed.
 */
static void FSImgFirst(EDVFSImg *fsimg)
{
	gboolean image_opened = FALSE;
	gint i;
	const gchar	*name,
			*ext;
	GtkCList *clist;
	EDVVFSObject *obj;
	EDVCore *core;
	if(fsimg == NULL)
	    return;

	clist = GTK_CLIST(fsimg->clist);
	core = fsimg->core;

	FSImgSetBusy(fsimg, TRUE);
	FSImgNavigatorUpdate(fsimg);

	/* Iterate from the first row until the last row is reached or
	 * an image is opened
	 */
	for(i = 0; i < clist->rows; i++)
	{
	    obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, i));
	    if(obj == NULL)
		continue;

	    name = obj->name;
	    if(name == NULL)
		continue;

	    ext = edv_name_get_extension(name);
	    if(!edv_image_is_supported_extension(ext))
		continue;

	    /* Open the image to the FSImg and switch video mode as
	     * needed
	     */
	    (void)FSImgOpen(
		fsimg,
		obj->path
	    );
	    image_opened = TRUE;

	    /* Select list row associated with this image */
	    gtk_clist_unselect_all(clist);
	    gtk_clist_select_row(clist, i, 0);

	    /* Update the information as needed */
	    FSImgInfoShow(fsimg, FSImgInfoShown(fsimg));

	    break;
	}

	/* Beep if no image was opened */
	if(!image_opened)
	    edv_play_sound_beep(core);

	FSImgSetBusy(fsimg, FALSE);
	FSImgNavigatorUpdate(fsimg);
}

/*
 *	Displays the last image on the FSImg.
 *
 *	Opens the last image and switches video modes as needed.
 */
static void FSImgLast(EDVFSImg *fsimg)
{
	gboolean image_opened = FALSE;
	gint i;
	const gchar	*name,
			*ext;
	GtkCList *clist;
	EDVVFSObject *obj;
	EDVCore *core;
	if(fsimg == NULL)
	    return;

	clist = GTK_CLIST(fsimg->clist);
	core = fsimg->core;

	FSImgSetBusy(fsimg, TRUE);
	FSImgNavigatorUpdate(fsimg);

	/* Iterate from the last row until the first row is reached or
	 * an image is opened
	 */
	for(i = clist->rows - 1; i >= 0; i--)
	{
	    obj = EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, i));
	    if(obj == NULL)
		continue;

	    name = obj->name;
	    if(name == NULL)
		continue;

	    ext = edv_name_get_extension(name);
	    if(!edv_image_is_supported_extension(ext))
		continue;

	    /* Open the image to the FSImg and switch video mode as
	     * needed
	     */
	    (void)FSImgOpen(
		fsimg,
		obj->path
	    );
	    image_opened = TRUE;

	    /* Select list row associated with this image */
	    gtk_clist_unselect_all(clist);
	    gtk_clist_select_row(clist, i, 0);

	    /* Update the information as needed */
	    FSImgInfoShow(fsimg, FSImgInfoShown(fsimg));

	    break;
	}

	/* Beep if no image was opened */
	if(!image_opened)
	    edv_play_sound_beep(core);

	FSImgSetBusy(fsimg, FALSE);
	FSImgNavigatorUpdate(fsimg);
}

/*
 *	Refreshes and reopens the image.
 */
static void FSImgRefresh(EDVFSImg *fsimg)
{
	EDVVFSObject *obj = FSImgCurrentObject(fsimg);
	if((obj != NULL) ? STRISEMPTY(obj->path) : TRUE)
	    return;

	FSImgSetBusy(fsimg, TRUE);
	FSImgNavigatorUpdate(fsimg);

	/* Reopen the image */
	(void)FSImgOpen(
	    fsimg,
	    obj->path
	);

	/* Update the information as needed */
	FSImgInfoShow(fsimg, FSImgInfoShown(fsimg));

	FSImgSetBusy(fsimg, FALSE);
	FSImgNavigatorUpdate(fsimg);
}

/*
 *	Rotates the image clockwise by 90 degrees.
 *
 *	The image will be reopened and the video mode will be switched
 *	as needed.
 */
static void FSImgRotateCW90(EDVFSImg *fsimg)
{
	gint		status,
			width, height,
			bpl,
			nframes,
			rot_width, rot_height,
			rot_bpl;
	guint8		*rgba,
			*rot_rgba,
			bg_color[4];
	GList		*glist,
			*rgba_list,
			*delay_list;
	GdkWindow *window;
	GtkWidget *w;
	CfgList *cfg_list;
	EDVVFSObject *obj;
	EDVCore *core;
	if(fsimg == NULL) 
	    return;

	FSImgHelpShow(fsimg, FALSE);
	FSImgMesg(fsimg, NULL);

	obj = FSImgCurrentObject(fsimg);
	if(obj == NULL)
	    return;

	if(STRISEMPTY(obj->path))
	    return;

	w = fsimg->toplevel;
	window = w->window;
	core = fsimg->core;
	cfg_list = core->cfg_list;

	/* Handle rotation alternates */
	switch(fsimg->rotate_position)
	{
	  case EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE:
	    break;

	  case EDV_PRESENTATION_MODE_ROTATE_POSITION_CW90:
	    fsimg->rotate_position = EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE;
	    FSImgRotateCW180(fsimg);
	    return;
	    break;

	  case EDV_PRESENTATION_MODE_ROTATE_POSITION_CCW90:
	    fsimg->rotate_position = EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE;
	    (void)FSImgOpen(
		fsimg,
		obj->path
	    );
	    FSImgInfoShow(fsimg, FSImgInfoShown(fsimg));
	    return;
	    break;

	  case EDV_PRESENTATION_MODE_ROTATE_POSITION_CW180:
	    fsimg->rotate_position = EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE;
	    FSImgRotateCCW90(fsimg);
	    return;
	    break;
	}

	/* Begin rotating */

	FSImgSetBusy(fsimg, TRUE);
	FSImgNavigatorUpdate(fsimg);

	/* Get the configured background color in RGBA */
	(void)FSImgGetBGColorRGBA(fsimg, bg_color);

	/* Open the image and get the RGBA image data */
	rgba_list = edv_image_open_rgba(
	    obj->path,
	    &width, &height,
	    &bpl,
	    &nframes,
	    &delay_list,
	    EDV_GET_B(EDV_CFG_PARM_PRESENTATION_OVERRIDE_IMAGE_BG_COLOR) ?
		NULL : bg_color,
	    NULL, NULL
	);
	if(rgba_list == NULL)
	{
	    gchar *msg = g_strdup_printf(
		"Unable to open image:\n\n    %s",
		obj->name
	    );
	    FSImgMesg(fsimg, msg);
	    g_free(msg);
	    edv_play_sound_error(core);
	    FSImgSetBusy(fsimg, FALSE);
	    FSImgNavigatorUpdate(fsimg);
	    return;
	}

	/* Calculate the rotated image size */
	rot_width = height;
	rot_height = width;
	rot_bpl = rot_width * 4 * sizeof(guint8);

	/* Rotate each frame */
	for(glist = rgba_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    rgba = (guint8 *)glist->data;

	    /* Allocate the rotated RGBA image data */
	    rot_rgba = (guint8 *)g_malloc(rot_bpl * rot_height);
	    if(rot_rgba == NULL)
		break;

	    /* Rotate */
	    GUIImageBufferRotateCW90(
		4,
		rgba, width, height, bpl,
		rot_rgba, rot_width, rot_height, rot_bpl
	    );
	    g_free(rgba);
	    glist->data = rot_rgba;
	}

	/* Set the rotated RGBA image data and switch video mode as needed */
	status = FSImgSetData(
	    fsimg,
	    rot_width, rot_height, rot_bpl,
	    rgba_list,
	    delay_list,
	    bg_color				/* 4 bytes RGBA */
	);

	/* Delete the image data */
	g_list_foreach(rgba_list, (GFunc)g_free, NULL);
	g_list_free(rgba_list);
	g_list_free(delay_list);

	/* Update the rotate position marker */
	fsimg->rotate_position = EDV_PRESENTATION_MODE_ROTATE_POSITION_CW90;

	/* Update the information as needed */
	FSImgInfoShow(fsimg, FSImgInfoShown(fsimg));

	FSImgSetBusy(fsimg, FALSE);
	FSImgNavigatorUpdate(fsimg);
}

/*
 *	Rotates the image counter-clockwise by 90 degrees.
 *
 *	The image will be reopened and the video mode will be switched
 *	as needed.
 */
static void FSImgRotateCCW90(EDVFSImg *fsimg)
{
	gint		status,
			width, height,
			bpl,
			nframes,
			rot_width, rot_height,
			rot_bpl;
	guint8		*rgba,
			*rot_rgba,
			bg_color[4];
	GList		*glist,
			*rgba_list,
			*delay_list;
	GdkWindow *window;
	GtkWidget *w;
	CfgList *cfg_list;
	EDVVFSObject *obj;
	EDVCore *core;
	if(fsimg == NULL)
	    return;

	FSImgHelpShow(fsimg, FALSE);
	FSImgMesg(fsimg, NULL);

	obj = FSImgCurrentObject(fsimg);
	if(obj == NULL)
	    return;

	if(STRISEMPTY(obj->path))
	    return;

	w = fsimg->toplevel;
	window = w->window;
	core = fsimg->core;
	cfg_list = core->cfg_list;

	/* Handle rotation alternates */
	switch(fsimg->rotate_position)
	{
	  case EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE:
	    break;

	  case EDV_PRESENTATION_MODE_ROTATE_POSITION_CW90:
	    fsimg->rotate_position = EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE;
	    (void)FSImgOpen(
		fsimg,
		obj->path
	    );
	    FSImgInfoShow(fsimg, FSImgInfoShown(fsimg));
	    return;
	    break;

	  case EDV_PRESENTATION_MODE_ROTATE_POSITION_CCW90:
	    fsimg->rotate_position = EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE;
	    FSImgRotateCW180(fsimg);
	    return;
	    break;

	  case EDV_PRESENTATION_MODE_ROTATE_POSITION_CW180:
	    fsimg->rotate_position = EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE;
	    FSImgRotateCW90(fsimg);
	    return;
	    break;
	}

	/* Begin rotating */

	FSImgSetBusy(fsimg, TRUE);
	FSImgNavigatorUpdate(fsimg);

	/* Get the configured background color in RGBA */
	(void)FSImgGetBGColorRGBA(fsimg, bg_color);

	/* Open the image and get the RGBA image data */
	rgba_list = edv_image_open_rgba(
	    obj->path,
	    &width, &height,
	    &bpl,
	    &nframes,
	    &delay_list,
	    EDV_GET_B(EDV_CFG_PARM_PRESENTATION_OVERRIDE_IMAGE_BG_COLOR) ?
		NULL : bg_color,
	    NULL, NULL
	);
	if(rgba_list == NULL)
	{
	    gchar *msg = g_strdup_printf(
		"Unable to open image:\n\n    %s",
		obj->name
	    );
	    FSImgMesg(fsimg, msg);
	    g_free(msg);
	    edv_play_sound_error(core);
	    FSImgSetBusy(fsimg, FALSE);
	    FSImgNavigatorUpdate(fsimg);
	    return;
	}

	/* Calculate the rotated image size */
	rot_width = height;
	rot_height = width;
	rot_bpl = rot_width * 4 * sizeof(guint8);

	/* Rotate each frame */
	for(glist = rgba_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    rgba = (guint8 *)glist->data;

	    /* Allocate the rotated RGBA image data */
	    rot_rgba = (guint8 *)g_malloc(rot_bpl * rot_height);
	    if(rot_rgba == NULL)
		break;

	    /* Rotate */
	    GUIImageBufferRotateCCW90(
		4,
		rgba, width, height, bpl,
		rot_rgba, rot_width, rot_height, rot_bpl
	    );
	    g_free(rgba);
	    glist->data = rot_rgba;
	}

	/* Set rotated RGBA image data and switch video mode as needed */
	status = FSImgSetData(
	    fsimg,
	    rot_width, rot_height, rot_bpl,
	    rgba_list,
	    delay_list,
	    bg_color				/* 4 bytes RGBA */
	);

	/* Delete the image data */
	g_list_foreach(rgba_list, (GFunc)g_free, NULL);
	g_list_free(rgba_list);
	g_list_free(delay_list);

	/* Update the rotate position marker */
	fsimg->rotate_position = EDV_PRESENTATION_MODE_ROTATE_POSITION_CCW90;

	/* Update the information as needed */
	FSImgInfoShow(fsimg, FSImgInfoShown(fsimg));

	FSImgSetBusy(fsimg, FALSE);
	FSImgNavigatorUpdate(fsimg);
}

/*
 *	Rotates the image clockwise by 180 degrees.
 *
 *	The image will be reopened and the video mode will be switched
 *	as needed.
 */
static void FSImgRotateCW180(EDVFSImg *fsimg)
{
	gint		status,
			width, height,
			bpl,
			nframes,
			rot_width, rot_height,
			rot_bpl;
	guint8		*rgba,
			*rot_rgba,
			bg_color[4];
	GList		*glist,
			*rgba_list,
			*delay_list;
	GdkWindow *window;
	GtkWidget *w;
	CfgList *cfg_list;
	EDVVFSObject *obj;
	EDVCore *core;
	if(fsimg == NULL)
	    return;

	FSImgHelpShow(fsimg, FALSE);
	FSImgMesg(fsimg, NULL);

	obj = FSImgCurrentObject(fsimg);
	if(obj == NULL)
	    return;

	if(STRISEMPTY(obj->path))
	    return;

	w = fsimg->toplevel;
	window = w->window;
	core = fsimg->core;
	cfg_list = core->cfg_list;

	/* Handle rotation alternates */
	switch(fsimg->rotate_position)
	{
	  case EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE:
	    break;

	  case EDV_PRESENTATION_MODE_ROTATE_POSITION_CW90:
	    fsimg->rotate_position = EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE;
	    FSImgRotateCCW90(fsimg);
	    return;
	    break;

	  case EDV_PRESENTATION_MODE_ROTATE_POSITION_CCW90:
	    fsimg->rotate_position = EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE;
	    FSImgRotateCW90(fsimg);
	    return;
	    break;

	  case EDV_PRESENTATION_MODE_ROTATE_POSITION_CW180:
	    fsimg->rotate_position = EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE;
	    (void)FSImgOpen(
		fsimg,
		obj->path
	    );
	    FSImgInfoShow(fsimg, FSImgInfoShown(fsimg));
	    return;
	}

	/* Begin rotating */

	FSImgSetBusy(fsimg, TRUE);
	FSImgNavigatorUpdate(fsimg);

	/* Get the configured background color in RGBA */
	(void)FSImgGetBGColorRGBA(fsimg, bg_color);

	/* Open the image and get the RGBA image data */
	rgba_list = edv_image_open_rgba(
	    obj->path,
	    &width, &height,
	    &bpl,
	    &nframes,
	    &delay_list,
	    EDV_GET_B(EDV_CFG_PARM_PRESENTATION_OVERRIDE_IMAGE_BG_COLOR) ?
		NULL : bg_color,
	    NULL, NULL
	);
	if(rgba_list == NULL)
	{
	    gchar *msg = g_strdup_printf(
		"Unable to open image:\n\n    %s",
		obj->name
	    );
	    FSImgMesg(fsimg, msg);
	    g_free(msg);
	    edv_play_sound_error(core);
	    FSImgSetBusy(fsimg, FALSE);
	    FSImgNavigatorUpdate(fsimg);
	    return;
	}

	/* Calculate the rotated image size */
	rot_width = width;
	rot_height = height;
	rot_bpl = rot_width * 4 * sizeof(guint8);

	/* Rotate each frame */
	for(glist = rgba_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    rgba = (guint8 *)glist->data;

	    /* Allocate the rotated RGBA image data */
	    rot_rgba = (guint8 *)g_malloc(rot_bpl * rot_height);
	    if(rot_rgba == NULL)
		break;

	    /* Rotate */
	    GUIImageBufferRotateCW180(
		4,
		rgba, width, height, bpl,
		rot_rgba, rot_width, rot_height, rot_bpl
	    );
	    g_free(rgba);
	    glist->data = rot_rgba;
	}

	/* Set rotated RGBA image data and switch video mode as needed */
	status = FSImgSetData(
	    fsimg,
	    rot_width, rot_height, rot_bpl,
	    rgba_list,
	    delay_list,
	    bg_color				/* 4 bytes RGBA */
	);

	/* Delete the image data */
	g_list_foreach(rgba_list, (GFunc)g_free, NULL);
	g_list_free(rgba_list);
	g_list_free(delay_list);

	/* Update the rotate position marker */
	fsimg->rotate_position = EDV_PRESENTATION_MODE_ROTATE_POSITION_CW180;

	/* Update the information as needed */
	FSImgInfoShow(fsimg, FSImgInfoShown(fsimg));

	FSImgSetBusy(fsimg, FALSE);
	FSImgNavigatorUpdate(fsimg);
}


/*
 *	Returns TRUE if help is being shown.
 */
static gboolean FSImgHelpShown(EDVFSImg *fsimg)
{
	return((fsimg != NULL) ? fsimg->show_help : FALSE);
}

/*
 *	Shows or hides the help.
 */
static void FSImgHelpShow(EDVFSImg *fsimg, const gboolean show)
{
	if(fsimg == NULL)
	    return;

	if(fsimg->show_help == show)
	    return;

	fsimg->show_help = show;
	FSImgQueueDraw(fsimg);
}


/*
 *	Sets the specified message and redraws.
 */
static void FSImgMesg(EDVFSImg *fsimg, const gchar *mesg)
{
	if(fsimg == NULL)
	    return;

	if((fsimg->mesg == NULL) && (mesg == NULL))
	    return;

	g_free(fsimg->mesg);
	fsimg->mesg = (mesg != NULL) ? g_strdup(mesg) : NULL;

	FSImgQueueDraw(fsimg);
}


/*
 *	Returns the shown information level.
 */
static gint FSImgInfoShown(EDVFSImg *fsimg)
{
	return((fsimg != NULL) ? fsimg->info_level : 0);
}

/*
 *	Shows or hides the information based on the specified info
 *	level.
 */
static void FSImgInfoShow(EDVFSImg *fsimg, const gint info_level)
{
	gint _info_level = info_level;
	const gchar *date_format;
	CfgList *cfg_list;
	EDVDateRelativity date_relativity;
	EDVCore *core;

	if(fsimg == NULL)
	    return;

	core = fsimg->core;
	cfg_list = core->cfg_list;
	date_relativity = (EDVDateRelativity)EDV_GET_I(
	    EDV_CFG_PARM_DATE_RELATIVITY
	);
	date_format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);

	/* Specified info level has "cycled"? */
	if(_info_level >= 5)
	    _info_level = 0;
	else if(_info_level < 0)
	    _info_level = 0;

	/* Set info level */
	fsimg->info_level = _info_level;

	/* Delete existing info */
	g_free(fsimg->info_text1);
	fsimg->info_text1 = NULL;
	g_free(fsimg->info_text2);
	fsimg->info_text2 = NULL;
	g_free(fsimg->info_text3);
	fsimg->info_text3 = NULL;

	/* Set new information? */
	if(_info_level >= 1)
	{
	    GdkVisual *vis = fsimg->visual;
	    const GdkVideoModeInfo *cur_vidmode = fsimg->cur_vidmode;
	    GtkCList *clist = GTK_CLIST(fsimg->clist);
	    EDVVFSObject *obj = FSImgCurrentObject(fsimg);

	    /* File name, image size, and file size */
	    if(obj != NULL)
	    {
		if((fsimg->img_shown_width != fsimg->img_width) ||
		   (fsimg->img_shown_height != fsimg->img_height)
		)
		    fsimg->info_text1 = g_strdup_printf(
			"%s\n%ix%i (%ix%i shown)\n%s %s",
			obj->name,
			fsimg->img_width, fsimg->img_height,
			fsimg->img_shown_width, fsimg->img_shown_height,
			edv_str_size_delim(obj->size),
			(obj->size == 1l) ? "byte" : "bytes"
		    );
		else
		    fsimg->info_text1 = g_strdup_printf(
			"%s\n%ix%i\n%s %s",
			obj->name,
			fsimg->img_width, fsimg->img_height,
			edv_str_size_delim(obj->size),
			(obj->size == 1l) ? "byte" : "bytes"
		    );
	    }
	    /* Date and image number */
	    if((clist != NULL) && (obj != NULL) && (_info_level >= 2))
	    {
		GList *glist = clist->selection_end;
		const gint i = (glist != NULL) ? (gint)glist->data : 0;
		const gulong t = obj->modify_time;
		gchar *s = (t > 0l) ?
		    edv_date_string_format(
			t,
			date_format, date_relativity
		    ) : NULL;
		fsimg->info_text2 = g_strdup_printf(
		    "%s\n%i/%i",
		    (s != NULL) ? s : "",
		    i + 1, clist->rows
		);
		g_free(s);
	    }
	    /* Display size */
	    if((vis != NULL) && (cur_vidmode != NULL) && (_info_level >= 3))
	    {
		const gchar *vis_name = NULL;
		switch(vis->type)
		{
		  case GDK_VISUAL_STATIC_GRAY:
		    vis_name = "StaticGrey";
		    break;
		  case GDK_VISUAL_GRAYSCALE:
		    vis_name = "GreyScale";
		    break;
		  case GDK_VISUAL_STATIC_COLOR:
		    vis_name = "StaticColor";
		    break;
		  case GDK_VISUAL_PSEUDO_COLOR:
		    vis_name = "PseduoColor";
		    break;
		  case GDK_VISUAL_TRUE_COLOR:
		    vis_name = "TrueColor";
		    break;
		  case GDK_VISUAL_DIRECT_COLOR:
		    vis_name = "DirectColor";
		    break;
		}
		fsimg->info_text3 = g_strdup_printf(
		    "Display: %ix%i %i bits\n%s",
		    cur_vidmode->viewport_width,
		    cur_vidmode->viewport_height,
		    vis->depth,
		    vis_name
		);
	    }
	}

	FSImgQueueDraw(fsimg);
}

/*
 *	Returns the map state of the list.
 */
static gboolean FSImgListShown(EDVFSImg *fsimg)
{
	GtkWidget *w = (fsimg != NULL) ? fsimg->clist_toplevel : NULL;
	return((w != NULL) ? GTK_WIDGET_MAPPED(w) : FALSE);
}

/*
 *	Shows or hides the list.
 */
static void FSImgListShow(EDVFSImg *fsimg, const gboolean show)
{
	GtkWidget *w = (fsimg != NULL) ? fsimg->clist_toplevel : NULL;
	if(w == NULL)
	    return;

	if(show)
	{
	    GList *glist;
	    GtkCList *clist = GTK_CLIST(fsimg->clist);

	    gtk_widget_show(w);

	    /* Move to selected row */
	    glist = (clist != NULL) ? clist->selection_end : NULL;
	    if(glist != NULL)
		gtk_clist_moveto(
		    clist,
		    (gint)glist->data, -1,	/* Row, column */
		    0.5f, 0.0f			/* Row, column */
		);
	}
	else
	{
	    gtk_widget_hide(w);

	    /* Need to grab the FSImg's toplevel window again whenever
	     * the GtkCList is unmapped because it will ungrab the
	     * pointer and loose confinement to the FSImg's toplevel
	     * window
	     */
	    FSImgGrab(fsimg);
	}

	FSImgNavigatorUpdate(fsimg);
}

/*
 *	Gets the list of images from the specified location.
 */
static void FSImgListGet(EDVFSImg *fsimg, const gchar *path)
{
	const gint border_minor = 2;
	gint		column,
			ncolumns;
	gulong block_size;
	const gchar *date_format;
	CfgList *cfg_list;
	GtkCList *clist;
	EDVSizeFormat size_format;
	EDVDateRelativity date_relativity;
	EDVCore *core;

	if(fsimg == NULL)
	    return;

	clist = GTK_CLIST(fsimg->clist);
	ncolumns = MAX(clist->columns, 1);
	core = fsimg->core;
	cfg_list = core->cfg_list;
	size_format = (EDVSizeFormat)EDV_GET_I(
	    EDV_CFG_PARM_SIZE_FORMAT
	);
	block_size = EDV_GET_UL(EDV_CFG_PARM_BLOCK_SIZE);
	date_relativity = (EDVDateRelativity)EDV_GET_I(
	    EDV_CFG_PARM_DATE_RELATIVITY
	);
	date_format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);


	/* Begin getting the list of images */

	gtk_clist_freeze(clist);

	/* Delete any existing items */
	gtk_clist_clear(clist);

	/* Get list of items from the Thumbs List */
	if(!STRISEMPTY(path))
	{
	    GList *names_list = edv_directory_list(
		path,
		TRUE,				/* Sorted */
		FALSE				/* Exclude notations */
	    );
	    if(names_list != NULL)
	    {
		gint	i,
			row;
		const gchar	*name,
				*ext;
		gchar	*full_path,
			**strv;
		GList *glist;
		EDVPixmap	*icon,
				*icon_opened,
				*icon_inaccessable,
				*icon_hidden;
		EDVVFSObject *obj;

		strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
		for(i = 0; i < ncolumns; i++)
		    strv[i] = "";

		for(glist = names_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    name = (const gchar *)glist->data;
		    if(name == NULL)
			continue;

		    /* Skip non-images */
		    ext = edv_name_get_extension(name);
		    if(!edv_image_is_supported_extension(ext))
		    {
			g_free(glist->data);
			continue;
		    }

		    /* Get the full path to this object */
		    full_path = edv_paths_join(
			path,
			name
		    );
		    if(full_path == NULL)
		    {
			g_free(glist->data);
			continue;
		    }

		    /* Get this object's local statistics */
		    obj = edv_vfs_object_lstat(full_path);
		    if(obj == NULL)
		    {
			g_free(full_path);
			g_free(glist->data);
			continue; 
		    }

		    /* Get the icon associated with this object */
		    (void)edv_match_object_icon(
			core->devices_list,
			core->mime_types_list,
			obj->type,
			obj->path,
			EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj),
			obj->permissions,
			EDV_ICON_SIZE_20,
			&icon,
			&icon_opened,
			&icon_inaccessable,
			&icon_hidden
		    );

		    /* Append a new row */
		    row = gtk_clist_append(clist, strv);
		    if(row < 0)
		    {
			(void)edv_pixmap_unref(icon);
			(void)edv_pixmap_unref(icon_opened);
			(void)edv_pixmap_unref(icon_inaccessable);
			(void)edv_pixmap_unref(icon_hidden);
			edv_vfs_object_delete(obj);
			g_free(full_path);
			g_free(glist->data);
			continue;
		    }

		    /* Set the row cells */
		    /* Name */
		    column = 0;
		    if(column < ncolumns)
		    {
			if(edv_pixmap_is_loaded(icon))
			    gtk_clist_set_pixtext(
				clist,
				row, column,
				obj->name,
				EDV_LIST_PIXMAP_TEXT_SPACING,
				icon->pixmap, icon->mask
			    );
			else
			    gtk_clist_set_text(
				clist,
				row, column,
				obj->name
			    );
			gtk_clist_set_shift(
			    clist,
			    row, column,
			    0, 0
			);
		    }
		    /* Size */
		    column = 1;
		    if(column < ncolumns) 
		    {
			const gchar *s = edv_str_size_format(
			    obj->size,
			    size_format,
			    block_size,
			    ',',
			    TRUE		/* Allow unit conversion */
			);
			gtk_clist_set_text(
			    clist,
			    row, column,
			    (s != NULL) ? s : ""
			);
			gtk_clist_set_shift(
			    clist,
			    row, column,
			    0, -border_minor
			);
		    }
		    /* Date Modified */
		    column = 2;
		    if(column < ncolumns)
		    {
			const gulong t = obj->modify_time;
			gchar *s = (t > 0l) ?
			    edv_date_string_format(
				t,
				date_format,
				date_relativity
			    ) : NULL;
			gtk_clist_set_text(
			    clist,
			    row, column,
			    (s != NULL ) ? s : ""
			);
			g_free(s);
		    }

		    /* Set this row's data as the object */
		    gtk_clist_set_row_data_full(
			clist,
			row,
			obj, (GtkDestroyNotify)edv_vfs_object_delete
		    );

		    (void)edv_pixmap_unref(icon);
		    (void)edv_pixmap_unref(icon_opened);
		    (void)edv_pixmap_unref(icon_inaccessable);
		    (void)edv_pixmap_unref(icon_hidden);

		    g_free(full_path);
		    g_free(glist->data);
		}

		g_free(strv);
		g_list_free(names_list);
	    }
	}

	gtk_clist_columns_autosize(clist);

	gtk_clist_thaw(clist); 
}


/*
 *	Returns the active state of the slideshow.
 */
static gboolean FSImgSlideshowActive(EDVFSImg *fsimg)
{
	if(fsimg == NULL)
	    return(FALSE);
	else
	    return((fsimg->slideshow_toid != 0) ? TRUE : FALSE);
}


/*
 *	Returns the map state of the control bar.
 */
static gboolean FSImgNavigatorShown(EDVFSImg *fsimg)
{
	GtkWidget *w = (fsimg != NULL) ? fsimg->navigator_toplevel : NULL;
	return((w != NULL) ? GTK_WIDGET_MAPPED(w) : FALSE);
}

/*
 *	Shows or hides the control bar.
 */
static void FSImgNavigatorShow(EDVFSImg *fsimg, const gboolean show)
{
	GtkWidget *w = (fsimg != NULL) ? fsimg->navigator_toplevel : NULL;
	if(w == NULL)
	    return;

	if(show)
	    gtk_widget_show(w);
	else
	    gtk_widget_hide(w);

	FSImgQueueDraw(fsimg);
	FSImgNavigatorUpdate(fsimg);
}

/*
 *	Updates the control bar's button mappings and sensitivities.
 */
static void FSImgNavigatorUpdate(EDVFSImg *fsimg)
{
	const guint	border_major = 5,
			btn_width = FSIMG_BTN_WIDTH;
	gboolean busy, list_mapped, slideshow;
	gint width, height;

	if(fsimg == NULL)
	    return;

	busy = (fsimg->busy_count > 0) ? TRUE : FALSE;
	list_mapped = FSImgListShown(fsimg);
	slideshow = FSImgSlideshowActive(fsimg);

	/* Get size of the current view port */
	FSImgCurrentViewportSize(
	    fsimg, &width, &height
	);

	/* Reduce width by the borders of the control bar's frame */
	width -= (2 * 3);

	if((width <= 0) || (height <= 0))
	    return;

	/* Begin mapping or unmapping buttons due to the width of the
	 * control bar
	 *
	 * Buttons layout:
	 * 
	 * [x]  [l]  [s]  [<<] {<] [>] [>>]   [i]  [rcw] [rccw] [cw180]  [?]
	 *
	 */

	/* Close */
	gtk_widget_show(fsimg->close_btn);

	/* List */
	if(width >= ((8 * btn_width) + (4 * border_major)))
	    gtk_widget_show(fsimg->list_btn);
	else
	    gtk_widget_hide(fsimg->list_btn);
	GTK_WIDGET_SET_SENSITIVE(
	    fsimg->list_btn,
	    (!busy && !slideshow) ? TRUE : FALSE
	);

	/* Slideshow */
	if(width >= ((8 * btn_width) + (4 * border_major)))
	    gtk_widget_show(fsimg->slideshow_btn);
	else
	    gtk_widget_hide(fsimg->slideshow_btn);
	GTK_WIDGET_SET_SENSITIVE(
	    fsimg->slideshow_btn,
	    (!busy && !slideshow && !list_mapped) ? TRUE : FALSE
	);

	/* First, Previous, Next, and Last */
	gtk_widget_show(fsimg->next_btns_toplevel);
	if(width >= ((6 * btn_width) + (2 * border_major)))
	{
	    gtk_widget_show(fsimg->first_btn);
	    gtk_widget_show(fsimg->last_btn);
	}
	else
	{
	    gtk_widget_hide(fsimg->first_btn);
	    gtk_widget_hide(fsimg->last_btn);
	}
	GTK_WIDGET_SET_SENSITIVE(
	    fsimg->first_btn,
	    (!busy && !slideshow && !list_mapped) ? TRUE : FALSE
	);
	GTK_WIDGET_SET_SENSITIVE(
	    fsimg->last_btn,
	    (!busy && !slideshow && !list_mapped) ? TRUE : FALSE
	);

	gtk_widget_show(fsimg->prev_btn);
	GTK_WIDGET_SET_SENSITIVE(
	    fsimg->prev_btn,
	    (!slideshow && !list_mapped) ? TRUE : FALSE
	);
	gtk_widget_show(fsimg->next_btn);
	GTK_WIDGET_SET_SENSITIVE(
	    fsimg->next_btn,
	    (!slideshow && !list_mapped) ? TRUE : FALSE
	);

	/* Info */
	if(width >= ((8 * btn_width) + (4 * border_major)))
	    gtk_widget_show(fsimg->info_btn);
	else
	    gtk_widget_hide(fsimg->info_btn);
	GTK_WIDGET_SET_SENSITIVE(
	    fsimg->info_btn,
	    (!busy && !slideshow && !list_mapped) ? TRUE : FALSE
	);

	/* Rotate */
	if(width >= ((11 * btn_width) + (5 * border_major)))
	    gtk_widget_show(fsimg->rot_btns_toplevel);
	else
	    gtk_widget_hide(fsimg->rot_btns_toplevel);
	GTK_WIDGET_SET_SENSITIVE(
	    fsimg->rot_btns_toplevel,
	    (!busy && !slideshow && !list_mapped) ? TRUE : FALSE
	);

	/* Help */
	gtk_widget_show(fsimg->help_btn);
}


/*
 *	Creates a new frame.
 */
static EDVFSImgFrame *EDVFSImgFrameNew(void)
{
	return(FSIMG_FRAME(g_malloc0(sizeof(EDVFSImgFrame))));
}

/*
 *	Deletes the frame.
 */
static void EDVFSImgFrameDelete(EDVFSImgFrame *frame)
{
	if(frame == NULL)
	    return;

	g_free(frame->rgba);
	g_free(frame);
}


/*
 *	Ungrabs the pointer as needed, unmaps the FSImg window, and
 *	restores the original video mode.
 */
static void FSImgRestore(EDVFSImg *fsimg)
{
	const GdkVideoModeInfo *m;
	GdkWindow *root = gdk_window_get_root();
	GtkWidget *w;
	if(fsimg == NULL)
	    return;

	/* Ungrab toplevel */
	w = fsimg->toplevel; 
	if(w != NULL)
	    gtk_grab_remove(w);

	/* Ungrab pointer as needed */
	if(gdk_pointer_is_grabbed())
	    gdk_pointer_ungrab(GDK_CURRENT_TIME);

	/* Unmap the fsimg */
	w = fsimg->toplevel;
	if(w != NULL)
	    gtk_widget_hide(w);

	/* Switch back to the original video mode */
	m = fsimg->orig_vidmode;
	if(gdk_video_mode_switch(m))
	{
	    /* Restore pointer position */
	    FSImgWarpPointer(
		root,
		fsimg->orig_pointer_x, fsimg->orig_pointer_y
	    );

	    /* Restore viewport position */
	    gdk_video_mode_set_viewport(
		fsimg->orig_viewport_x, fsimg->orig_viewport_y
	    );
	}
	else
	{
	    /* Failed to switch video modes, try to switch to the
	     * default video mode
	     */
	    GList *glist = fsimg->vidmodes_list;
	    while(glist != NULL)
	    {
		m = GDK_VIDEO_MODE_INFO(glist->data);
		if((m != NULL) ? (m->flags & GDK_VIDEO_MODE_DEFAULT) : FALSE)
		{
		    gdk_video_mode_switch(m);
		    break;
		}
		glist = g_list_next(glist);
	    }
	}
}

/*
 *	Creates a new FSImg.
 */
static EDVFSImg *FSImgNew(EDVCore *core)
{
	const gint	border_major = 5,
			btn_width = FSIMG_BTN_WIDTH,
			btn_height = FSIMG_BTN_HEIGHT;
	const guint8	icon_play_mask_16x16_data[] = {
		0x03, 0x00, 0x0f, 0x00, 0x3f, 0x00, 0xff, 0x00,
		0xff, 0x03, 0xff, 0x0f, 0xff, 0x3f, 0xff, 0xff,
		0xff, 0xff, 0xff, 0x3f, 0xff, 0x0f, 0xff, 0x03,
		0xff, 0x00, 0x3f, 0x00, 0x0f, 0x00, 0x03, 0x00
	},
			icon_pause_mask_16x16_data[] = {
		0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
		0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
		0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18,
		0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18
	};
	const gchar *wm_name, *wm_class;
	gchar *clist_heading[3];
	GdkModifierType mask;
	GdkFont *font;
	GdkColormap *colormap;
	GdkWindow	*window,
			*root = gdk_window_get_root();
	GdkVisual *vis;
	GdkGC *gc;
	GtkWidget	*w,
			*parent, *parent2, *parent3,
			*toplevel;
	GtkCList *clist;
	CfgList *cfg_list;
	EDVFSImg *fsimg;

	if(core == NULL)
	    return(NULL);

	cfg_list = core->cfg_list;
	wm_name = core->wm_name;
	wm_class = core->wm_class;

	/* Create the Full Screen Image Data */
	fsimg = EDV_FSIMG(g_malloc0(sizeof(EDVFSImg)));
	if(fsimg == NULL)
	    return(NULL);

	gtk_widget_push_visual(gdk_rgb_get_visual());
	gtk_widget_push_colormap(gdk_rgb_get_cmap());
	fsimg->toplevel = toplevel = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_widget_pop_visual();
	gtk_widget_pop_colormap();
#if 0
	fsimg->freeze_count = 0;
	fsimg->busy_count = 0;
	fsimg->main_loop_level = 0;
	fsimg->visual = NULL;
	fsimg->colormap = NULL;
	memset(&fsimg->color_fg, 0x00, sizeof(GdkColor));
	memset(&fsimg->color_bg, 0x00, sizeof(GdkColor));
	memset(&fsimg->color_shadow, 0x00, sizeof(GdkColor));
#endif
	fsimg->invert_fg = EDV_GET_B(
	    EDV_CFG_PARM_PRESENTATION_COLOR_FG_INVERT
	);
#if 0
	fsimg->font = NULL;
 	fsimg->pixmap = NULL;
	fsimg->gc = NULL;
#endif

	fsimg->core = core;

#if 0
	fsimg->info_level = 0;
	fsimg->info_text1 = NULL;
	fsimg->info_text2 = NULL;
	fsimg->info_text3 = NULL;
#endif
	fsimg->icon_play_mask = gdk_bitmap_create_from_data(
	    root, icon_play_mask_16x16_data,
	    16, 16
	);
	fsimg->icon_pause_mask = gdk_bitmap_create_from_data(
	    root, icon_pause_mask_16x16_data,
	    16, 16
	);
#if 0
	fsimg->img_width = 0;
	fsimg->img_height = 0;
	fsimg->img_shown_width = 0;
	fsimg->img_shown_height = 0;
#endif

#if 0
	fsimg->frame_width = 0;
	fsimg->frame_height = 0;
	fsimg->frames_list = NULL;
	fsimg->cur_frame = fsimg->frames_list;
	fsimg->frame_next_toid = 0;
#endif

	fsimg->vidmodes_list = gdk_video_modes_get();

	fsimg->orig_vidmode = fsimg->cur_vidmode = gdk_video_mode_get_current(
	    fsimg->vidmodes_list
	);
#if 0
	fsimg->prev_vidmode = NULL;
#endif

	gdk_video_mode_get_viewport(
	    &fsimg->orig_viewport_x, &fsimg->orig_viewport_y
	);
	gdk_window_get_pointer(
	    root,
	    &fsimg->orig_pointer_x, &fsimg->orig_pointer_y,
	    &mask
	);

#if 0
	fsimg->hide_pointer = FALSE;
	fsimg->hide_pointer_toid = 0;
#endif

	fsimg->slideshow_delay = (gulong)EDV_GET_L(
	    EDV_CFG_PARM_PRESENTATION_SLIDESHOW_DELAY) * 1000l;
	fsimg->slideshow_toid = 0;

	fsimg->rotate_position = EDV_PRESENTATION_MODE_ROTATE_POSITION_NONE;

#if 0
	fsimg->show_help = FALSE;
	fsimg->mesg = NULL;
#endif

	fsimg->freeze_count++;

	/* Toplevel GtkWindow */
	parent = w = toplevel;
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_widget_set_app_paintable(w, TRUE);
	if(!STRISEMPTY(wm_name) && !STRISEMPTY(wm_class))
	    gtk_window_set_wmclass(GTK_WINDOW(w), wm_name, wm_class);
	else
	    gtk_window_set_wmclass(
		GTK_WINDOW(w),
		EDV_PRESENTATION_MODE_WM_CLASS_WINDOW_NAME,
		PROG_NAME
	    );
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT | GTK_CAN_FOCUS);
	gtk_widget_add_events(
	    w,
	    GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK |
	    GDK_FOCUS_CHANGE_MASK |
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "configure_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "focus_in_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "focus_out_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(FSImgPointerMotionCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(FSImgPointerMotionCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(FSImgPointerMotionCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "enter_notify_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "leave_notify_event",
	    GTK_SIGNAL_FUNC(FSImgEventCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "draw",
	    GTK_SIGNAL_FUNC(FSImgDrawCB), fsimg
	);
	gtk_widget_realize(w);
	window = w->window;

	/* Get the visual */
	fsimg->visual = vis = gdk_window_get_visual(window);
	if(vis != NULL)
	    gdk_visual_ref(vis);

	/* Get the colormap */
	fsimg->colormap = colormap = GDK_COLORMAP_REF(gdk_window_get_colormap(window));
	if(colormap != NULL)
	{
	    GdkColor *ct;
	    const CfgColor *cs;

	    cs = EDV_GET_COLOR(EDV_CFG_PARM_PRESENTATION_COLOR_FG);
	    ct = &fsimg->color_fg;
	    if(cs != NULL)
		GDK_COLOR_SET_COEFF(ct, cs->r, cs->g, cs->b);
	    GDK_COLORMAP_ALLOC_COLOR(colormap, ct);

	    cs = EDV_GET_COLOR(EDV_CFG_PARM_PRESENTATION_COLOR_BG);
	    ct = &fsimg->color_bg;
	    if(cs != NULL)
		GDK_COLOR_SET_COEFF(ct, cs->r, cs->g, cs->b);
	    GDK_COLORMAP_ALLOC_COLOR(colormap, ct);

	    ct = &fsimg->color_shadow;
	    ct->red = 0x5fff;
	    ct->green = 0x5fff;
	    ct->blue = 0x5fff;
	    GDK_COLORMAP_ALLOC_COLOR(colormap, ct);
	}

	/* Create the font */
	font = gdk_font_load(
	    EDV_GET_S(EDV_CFG_PARM_PRESENTATION_FONT)
	);
#ifdef EDV_GDK_FONT_NAME_12
	if(font == NULL)
	    font = gdk_font_load(EDV_GDK_FONT_NAME_12);
#else
#warning "EDV_GDK_FONT_NAME_12 was not defined, fonts may not display correctly"
#endif
	if(font == NULL)
	    font = gdk_font_load("8x16");
	fsimg->font = font;

	/* Create GC */
	fsimg->gc = gc = gdk_gc_new(window);


	/* Main GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Create the list */
	fsimg->clist_toplevel = parent2 = w = gtk_event_box_new();
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	/* GtkScrolledWindow */
	parent3 = w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	/* GtkCList */
	clist_heading[0] = "Name";
	clist_heading[1] = "Size";
	clist_heading[2] = "Date Modified";
	fsimg->clist = w = gtk_clist_new_with_titles(
	    sizeof(clist_heading) / sizeof(gchar *),
	    clist_heading
	);
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
#if 0
/* Causes the loss of pointer grabs */
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(edv_clist_motion_event_cb), core
	);
#endif
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(FSImgListEventCB), fsimg
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(FSImgListEventCB), fsimg
	);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_realize(w);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_resizeable(clist, 0, TRUE); 
	gtk_clist_set_column_justification(
	    clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 1, TRUE);
	gtk_clist_set_column_resizeable(clist, 1, TRUE); 
	gtk_clist_set_column_justification(
	    clist, 1, GTK_JUSTIFY_RIGHT
	);
	gtk_clist_set_column_auto_resize(clist, 2, TRUE);
	gtk_clist_set_column_resizeable(clist, 2, TRUE); 
	gtk_clist_set_column_justification(
	    clist, 2, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_BROWSE);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_widget_show(w);


	/* Create control Bar */
	fsimg->navigator_toplevel = w = gtk_event_box_new();
	gtk_box_pack_end(GTK_BOX(parent), w, FALSE, FALSE, 0);
	parent = w;

	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* Hbox for buttons */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent2 = w;

	/* Close Button */
	fsimg->close_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_playerbtn_close_14x14_xpm
	);
	gtk_widget_set_usize(w, btn_width, btn_height);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(FSImgCloseCB), fsimg
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fermer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
	);
	gtk_widget_show(w);

	/* List Button */
	fsimg->list_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_playerbtn_eject_14x14_xpm
	);
	gtk_widget_set_usize(w, btn_width, btn_height);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(FSImgListCB), fsimg
	);
	gtk_widget_show(w);

	/* Slideshow Button */
	fsimg->slideshow_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_playerbtn_slideshow_14x14_xpm
	);
	gtk_widget_set_usize(w, btn_width, btn_height);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(FSImgSlideshowCB), fsimg
	);
	gtk_widget_show(w);

	/* Hbox for First, Previous, Next, and Last buttons */
	fsimg->next_btns_toplevel = w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* First Button */
	fsimg->first_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_playerbtn_rewind_14x14_xpm
	);
	gtk_widget_set_usize(w, btn_width, btn_height);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(FSImgFirstCB), fsimg
	);
	gtk_widget_show(w);

	/* Previous Button */
	fsimg->prev_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_playerbtn_playbackwards_14x14_xpm
	);
	gtk_widget_set_usize(w, btn_width, btn_height);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(FSImgPrevCB), fsimg
	);
	gtk_widget_show(w);

	/* Next Button */
	fsimg->next_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_playerbtn_play_14x14_xpm
	);
	gtk_widget_set_usize(w, btn_width, btn_height);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(FSImgNextCB), fsimg
	);
	gtk_widget_show(w);

	/* Last Button */
	fsimg->last_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_playerbtn_seekend_14x14_xpm
	);
	gtk_widget_set_usize(w, btn_width, btn_height);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(FSImgLastCB), fsimg
	);
	gtk_widget_show(w);

	/* Info Button */
	fsimg->info_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_playerbtn_info_14x14_xpm
	);
	gtk_widget_set_usize(w, btn_width, btn_height);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(FSImgInfoCB), fsimg
	);
	gtk_widget_show(w);

	/* Hbox for Rotate buttons */
	fsimg->rot_btns_toplevel = w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Rotate CCW 90 Button */
	fsimg->rot_ccw90_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_playerbtn_rotateccw90_14x14_xpm
	);
	gtk_widget_set_usize(w, btn_width, btn_height);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(FSImgRotateCCW90CB), fsimg
	);
	gtk_widget_show(w);

	/* Rotate CW 90 Button */
	fsimg->rot_cw90_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_playerbtn_rotatecw90_14x14_xpm
	);
	gtk_widget_set_usize(w, btn_width, btn_height);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(FSImgRotateCW90CB), fsimg
	);
	gtk_widget_show(w);

	/* Rotate CW 180 Button */
	fsimg->rot_cw180_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_playerbtn_rotatecw180_14x14_xpm
	);
	gtk_widget_set_usize(w, btn_width, btn_height);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(FSImgRotateCW180CB), fsimg
	);
	gtk_widget_show(w);

	/* Help Button */
	fsimg->help_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_playerbtn_help_14x14_xpm
	);
	gtk_widget_set_usize(w, btn_width, btn_height);
	gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(FSImgHelpCB), fsimg
	);
	GUISetWidgetTip(w,
#if defined(PROG_LANGUAGE_SPANISH)
"Ayuda"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aide"
#elif defined(PROG_LANGUAGE_GERMAN) 
"Hilfe"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aiuto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Hulp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ajuda"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Hjelp"
#else
"Help"
#endif
	);
	gtk_widget_show(w);

	fsimg->freeze_count--;

	return(fsimg);
}

/*
 *	Marks the FSImg as busy or ready.
 */
static void FSImgSetBusy(EDVFSImg *fsimg, const gboolean busy)
{
	GdkCursor *cursor;
	GtkWidget *w;
	EDVCore *core;

	if(fsimg == NULL)
	    return;

	core = fsimg->core;

	w = fsimg->toplevel;
	if(w != NULL)
	{
	    if(busy)
	    {
		/* Increase busy count */
		fsimg->busy_count++;

		/* If already busy then don't change anything */
		if(fsimg->busy_count > 1)
		    return;

		cursor = edv_get_cursor(core, EDV_CURSOR_CODE_BUSY);
	    }
	    else
	    {
		/* Reduce busy count */
		fsimg->busy_count--;
		if(fsimg->busy_count < 0)
		    fsimg->busy_count = 0;

		/* If still busy do not change anything */
		if(fsimg->busy_count > 0)
		    return;

		if(fsimg->hide_pointer)
		    cursor = edv_get_cursor(core, EDV_CURSOR_CODE_INVISIBLE);
		else
		    cursor = NULL;
	    }

	    /* Update toplevel window's cursor */
	    if(w->window != NULL)
	    {
		gdk_window_set_cursor(w->window, cursor);
		gdk_flush();
	    }
	}
}

/*
 *	Pops all main loop levels, switches to the original video mode,
 *	and deletes the FSImg.
 */
static void FSImgDeleteRestore(EDVFSImg *fsimg)
{
	if(fsimg == NULL)
	    return;

	/* Pop all main loop levels */
	while(fsimg->main_loop_level > 0)
	{
	    gtk_main_quit();
	    fsimg->main_loop_level--;
	}

	/* Ungrabs the pointer, unmap the fsimg window and restore the
	 * original video mode
	 */
	FSImgRestore(fsimg);

	/* Delete the fsimg */
	FSImgDelete(fsimg);
}

/*
 *	Deletes the FSImg.
 */
static void FSImgDelete(EDVFSImg *fsimg)
{
	GdkColormap *colormap;
	GtkCList *clist;

	if(fsimg == NULL)
	    return;

	colormap = fsimg->colormap;

	fsimg->frame_next_toid = GTK_TIMEOUT_REMOVE(fsimg->frame_next_toid);
	fsimg->hide_pointer_toid = GTK_TIMEOUT_REMOVE(fsimg->hide_pointer_toid);
	fsimg->slideshow_toid = GTK_TIMEOUT_REMOVE(fsimg->slideshow_toid);

	gtk_widget_hide(fsimg->toplevel);

	fsimg->freeze_count++;

	clist = GTK_CLIST(fsimg->clist);
	gtk_clist_freeze(clist); 
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist); 

	g_free(fsimg->mesg);

	g_free(fsimg->info_text1);
	g_free(fsimg->info_text2);
	g_free(fsimg->info_text3);

	(void)GDK_BITMAP_UNREF(fsimg->icon_play_mask);
	(void)GDK_BITMAP_UNREF(fsimg->icon_pause_mask);

	GTK_WIDGET_DESTROY(fsimg->clist);
	GTK_WIDGET_DESTROY(fsimg->clist_toplevel);

	GTK_WIDGET_DESTROY(fsimg->close_btn);
	GTK_WIDGET_DESTROY(fsimg->list_btn);
	GTK_WIDGET_DESTROY(fsimg->slideshow_btn);
	GTK_WIDGET_DESTROY(fsimg->first_btn);
	GTK_WIDGET_DESTROY(fsimg->prev_btn);
	GTK_WIDGET_DESTROY(fsimg->next_btn);
	GTK_WIDGET_DESTROY(fsimg->last_btn);
	GTK_WIDGET_DESTROY(fsimg->info_btn);
	GTK_WIDGET_DESTROY(fsimg->rot_cw90_btn);
	GTK_WIDGET_DESTROY(fsimg->rot_ccw90_btn);
	GTK_WIDGET_DESTROY(fsimg->rot_cw180_btn);
	GTK_WIDGET_DESTROY(fsimg->help_btn);
	GTK_WIDGET_DESTROY(fsimg->next_btns_toplevel);
	GTK_WIDGET_DESTROY(fsimg->rot_btns_toplevel);
	GTK_WIDGET_DESTROY(fsimg->navigator_toplevel);
	GTK_WIDGET_DESTROY(fsimg->toplevel);

	(void)GDK_GC_UNREF(fsimg->gc);
	(void)GDK_PIXMAP_UNREF(fsimg->pixmap);
	(void)GDK_FONT_UNREF(fsimg->font);
	GDK_COLORMAP_FREE_COLOR(colormap, &fsimg->color_fg);
	GDK_COLORMAP_FREE_COLOR(colormap, &fsimg->color_bg);
	GDK_COLORMAP_FREE_COLOR(colormap, &fsimg->color_shadow);
	(void)GDK_COLORMAP_UNREF(colormap);

	if(fsimg->visual != NULL)
	    gdk_visual_unref(fsimg->visual);
	gdk_video_modes_delete(fsimg->vidmodes_list);

	if(fsimg->frames_list != NULL)
	{
	    g_list_foreach(
		fsimg->frames_list,
		(GFunc)EDVFSImgFrameDelete,
		NULL
	    );
	    g_list_free(fsimg->frames_list);
	}

	fsimg->freeze_count--;

	g_free(fsimg);
}


/*
 *	Enters presentation mode from the specified Image Browser.
 *
 *	The Image Browser's first or selected image will be displayed,
 *	video modes will be switched as needed, and the pointer will be
 *	grabbed.
 *
 *	A main loop level will be pushed and this function will block
 *	until the user exits presentation mode.
 */
void EDVPresentationModeEnterFromImbr(
	EDVCore *core,
	EDVImageBrowser *imbr  
)
{
	gboolean image_opened = FALSE;
	gint		i,
			thumb_num;
	const gchar	*name,
			*ext;
	GList *glist;
	GtkWidget *toplevel;
	GtkCList *clist;
	CfgList *cfg_list;
	tlist_struct *tlist;
	EDVVFSObject *obj, *obj2;
	EDVFSImg *fsimg;

	if((core == NULL) || (imbr == NULL))
	    return;

	toplevel = imbr->toplevel;
	tlist = imbr->tlist;
	cfg_list = core->cfg_list;

	/* If video modes are not supported then presentation mode will
	 * not work
	 */
	if(!gdk_video_mode_is_supported())
	{
	    edv_play_sound_warning(core);
	    edv_message_warning(
#if defined(PROG_LANGUAGE_SPANISH)
"La Conmutacio'n Video Del Modo No Sostuvo",
"Incapaz de entrar el modo de la presentacio'n, la\n\
conmutacio'n video del modo no se sostiene.\n\
\n\
Su GUI necesita sostener la conmutacio'n video del modo\n\
y usted necesita compilar este programa con apoyo video\n\
de modo.",
#elif defined(PROG_LANGUAGE_FRENCH)
"La Commutation du mode Vido n'est pas supporte",
"Impossible de passer en mode prsentation, la\n\
commutation de mode vido n'est pas supporte.\n\
\n\
Votre GUI doit supporter la commutation du mode\n\
vido et vous devez compiler ce programme avec\n\
le support de mode vido.",
#elif defined(PROG_LANGUAGE_GERMAN)
"Videomodusschaltung Ist Nicht Unterstu'tzt",
"Unfa'hig, Vorstellungsmodus einzutragen, ist\n\
videomodusschaltung nicht unterstu'tzt.\n\
\n\
Ihr GUI muss Videomodusschaltung und Sie mu'ssen\n\
kompilieren dieses programm mit videomodusstu'tze\n\
unterstu'tzen.",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Commutare Di Modo Video Non E` Sostenuto",
"Incapace per entrare il modo di presentazione,\n\
commutare di modo video non e` sostenuto.\n\
\n\
Il suo GUI ha bisogno di sostenere commutare di modo video\n\
e lei ha bisogno di compilare questo programma col sostegno\n\
di modo video.",
#elif defined(PROG_LANGUAGE_DUTCH)
"Videomodus Schakelen Is Niet Gesteund",
"Onbekwaam om voorstelling modus binnengaan, is videomodus\n\
schakelen niet gesteund.\n\
\n\
Uw GUI moet steunen videomodus schakelen en u moet dit\n\
programma met videomodus steun verzamelen.",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Modo De Vi'deo Na~o Troca E' Apoiado",
"Incapaz de entrar modo de apresentac,a~o, modo de vi'deo\n\
na~o troca e' apoiado.\n\
\n\
Seu GUI necessita apoiar modo de vi'deo trocar e necessita\n\
compilar este programa com apoio de modo de vi'deo.",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"VideomodusOmkopling St?tter Ikke",
"Maktesls ga inn i framstillingsmodus,\n\
videomodusomkopling st'tter ikke.\n\
\n\
Deres GUI st'tter videomodusomkopling og de kompilerer dette\n\
programet med videomodusst'tte.",
#else
"Video Mode Switching Not Supported",
"Unable to enter presentation mode, video mode switching\n\
is not supported.\n\
\n\
Your GUI needs to support video mode switching and you need\n\
to compile this program with video mode support.",
#endif
		NULL, toplevel
	    );
	    return;
	}


	/* Create the FSImg */
	fsimg = FSImgNew(core);
	if(fsimg == NULL)
	    return;

	clist = GTK_CLIST(fsimg->clist);

	/* Get the list of images from the specified location */
	FSImgListGet(fsimg, edv_image_browser_get_location(imbr));

	/* Get last selected thumb or first thumb if none are selected */
	glist = tlist->selection_end;

	/* Start with the selected or first thumb and iterate until
	 * an image is opened or there are no more thumbs
	 */
	for(thumb_num = (glist != NULL) ? (gint)glist->data : 0;
	    thumb_num < tlist->total_thumbs;
	    thumb_num++
	)
	{
	    obj = EDV_VFS_OBJECT(TListGetThumbData(tlist, thumb_num));
	    if(obj == NULL)
		continue;

	    name = obj->name;
	    if(name == NULL)
		continue;

	    ext = edv_name_get_extension(name);
	    if(!edv_image_is_supported_extension(ext))
		continue;

	    /* Open the image and switch video mode as needed */
	    (void)FSImgOpen(
		fsimg,
		obj->path
	    );
	    image_opened = TRUE;
	    for(i = 0; i < clist->rows; i++)
	    {
		obj2 = EDV_VFS_OBJECT(gtk_clist_get_row_data(clist, i));
		if(obj2 == NULL)
		    continue;

		if(obj2->path == NULL)
		    continue;

		if(!strcmp((const char *)obj->path, (const char *)obj2->path))
		{
		    gtk_clist_unselect_all(clist);
		    gtk_clist_select_row(clist, i, 0);
		    break;
		}
	    }

	    /* Update the information as needed */
	    FSImgInfoShow(fsimg, FSImgInfoShown(fsimg));

	    break;
	}

	/* If no image was opened then clear the fsimg */
	if(!image_opened)
	{
	    FSImgClear(fsimg, -1, -1);
	    FSImgMesg(fsimg, "No images found");
	}

	/* Begin startup settings */

	/* Show Navigator? */
	FSImgNavigatorShow(
	    fsimg,
	    EDV_GET_B(EDV_CFG_PARM_PRESENTATION_SHOW_NAVIGATOR)
	);

	/* Hide pointer? */
	FSImgShowPointer(
	    fsimg,
	    (EDV_GET_L(EDV_CFG_PARM_PRESENTATION_HIDE_POINTER_DELAY) > 0l) ?
		FALSE : TRUE
	);

	/* Slideshow? */
	if(EDV_GET_B(EDV_CFG_PARM_PRESENTATION_SLIDESHOW) &&
	   (clist->rows >= 2)
	)
	{
	    (void)GTK_TIMEOUT_REMOVE(fsimg->slideshow_toid);
	    fsimg->slideshow_toid = gtk_timeout_add(
		MAX(fsimg->slideshow_delay, 1000l),
		FSImgSlideShowTOCB, fsimg
	    );
	}

	FSImgNavigatorUpdate(fsimg);

	/* Push a GTK main loop level and block */
	fsimg->main_loop_level++;
	gtk_main();

	/* The FSImg should have been deleted after leaving the main
	 * loop
	 */
	fsimg = NULL;
}


/*
 *	Enters presentation mode.
 *
 *	The first image at the specified location will be displayed, or
 *	if the specified location is an image then that image will be
 *	displayed.
 *
 *	Video modes will be switched as needed and the pointer will be
 *	grabbed.
 *
 *	A main loop level will be pushed and this function will block
 *	until the user exits presentation mode.
 */
void EDVPresentationModeEnter(EDVCore *core)
{





}
