#include <string.h>
#include <math.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "pie_chart.h"


typedef struct _PieChart		PieChart;
#define PIE_CHART(p)			((PieChart *)(p))
#define PIE_CHART_KEY			"/PieChart"

typedef struct _PieChartValue		PieChartValue;
#define PIE_CHART_VALUE(p)		((PieChartValue *)(p))
#define PIE_CHART_VALUE_KEY		"/PieChart/Value"


/*
 *	Value Flags:
 */
typedef enum {
	PIE_CHART_VALUE_REALIZED	= (1 << 0)
} pie_chart_value_flags;


/* Callbacks */
static void PieChartDestroyCB(gpointer data);
static void PieChartRealizeCB(GtkWidget *widget, gpointer data);
static gint PieChartConfigureEventCB(
	GtkWidget *widget, GdkEventConfigure *configure, gpointer data
);
static gint PieChartExposeEventCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
);

/* Drawing */
static void PieChartDrawArc(
	GdkDrawable *drawable,
	GdkGC *gc,
	const gint x, const gint y,
	const gint width, const gint height,
	const gfloat start_angle,		/* 0.0 to 1.0 */
	const gfloat end_angle,			/* 0.0 to 1.0 */
	const gboolean outline_circumferance,
	const gboolean outline_radius,
	const gboolean outline_height
);
static void PieChartDrawPieIterate(
	PieChart *pc,
	GdkDrawable *drawable,
	const GtkStateType state,
	GtkStyle *style,
	const gint x, const gint y,
	const gint width, const gint height,
	const gboolean outline_circumference,
	const gboolean outline_radius,
	const gboolean outline_height,
	const gboolean shaded,
	const gboolean shadow
);
static void PieChartDraw(PieChart *pc);

/* Realize Value */
static void PieChartValueRealize(
	PieChart *pc,
	GdkColor *c,
	const gchar *type_label,
	const gchar *value_label,
	GtkWidget **type_label_box_rtn,
	GtkWidget **value_label_box_rtn,
	GtkWidget **color_da_rtn
);
static void PieChartRealizeValues(PieChart *pc);

/* Pie Chart Value */
static PieChartValue *PieChartValueNew(void);
static void PieChartValueDelete(PieChartValue *v);
gint PieChartValueAdd(
	GtkWidget *w,
	GtkAdjustment *adj,
	GdkColor *c,
	const gchar *type_label,
	const gchar *value_label
);
void PieChartValueSet(
	GtkWidget *w,
	const gint value_num,
	GtkAdjustment *adj,
	GdkColor *c,
	const gchar *type_label,
	const gchar *value_label
);
void PieChartValueRemove(
	GtkWidget *w,
	const gint value_num
);
void PieChartClear(GtkWidget *w);

/* Pie Chart */
GtkWidget *PieChartNew(
	const pie_chart_flags flags,
	const gint width, const gint height,
	GtkAdjustment *adj,
	GdkColor *c,
	const gchar *title,
	const gchar *footer,
	const gchar *base_type_label,
	const gchar *base_value_label
);
void PieChartRealize(GtkWidget *w);
void PieChartQueueDraw(GtkWidget *w);


#define PIE_CHART_PIE_HEIGHT		10

#ifndef PI
# define PI				3.14159265
#endif

#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Pie Chart Value:
 */
struct _PieChartValue {

	pie_chart_value_flags	flags;

	PieChart	*pc;

	GtkAdjustment	*adj;			/* Only members lower and
						 * upper are used */
	GdkColor	color,
			shade_color;

	GtkWidget	*type_label_box,
			*value_label_box,
			*color_da;

	gchar		*type_label,
			*value_label;

};


/*
 *	Pie Chart:
 */
struct _PieChart {

	GtkWidget	*toplevel;
	gint		freeze_count;
	GdkColormap	*colormap;
	GdkGC		*gc;
	pie_chart_flags	flags;

	GtkWidget	*pie_chart_da,		/* Pie Chart GtkDrawingArea */
			*type_labels_vbox,
			*value_labels_vbox,
			*base_color_da,		/* Base value color GtkDrawingArea */
			*title_label,
			*footer_label;

	GdkPixmap	*pie_chart_pixmap;	/* Back buffer for the
						 * GtkDrawingArea */
	gint		width, height;		/* Size of pie_chart_pixmap */

	GtkAdjustment	*adj;			/* Only members lower and
						 * upper are used */
	GdkColor	color,
			shade_color,
			shadow_color;

	gchar		*base_type_label,
			*base_value_label;

	GList		*values_list;		/* GList of PieChartValue *
						 * values */
};


/*
 *	Toplevel GtkWidget "destroy" signal callback.
 */
static void PieChartDestroyCB(gpointer data)
{
	GdkColormap *colormap;
	PieChart *pc = PIE_CHART(data);
	if(pc == NULL)
	    return;

	pc->freeze_count++;

	colormap = pc->colormap;

	/* Delete all the values */
	if(pc->values_list != NULL)
	{
	    GList *glist;
	    PieChartValue *v;

	    for(glist = pc->values_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		v = PIE_CHART_VALUE(glist->data);
		if(v == NULL)
		    continue;

		/* Mark all the child GtkWidgets as destroyed before
		 * deleting this value
		 */
		v->type_label_box = NULL;
		v->value_label_box = NULL;
		v->color_da = NULL;

		/* Delete thie value */
		PieChartValueDelete(v);
	    }

	    /* Delete the values list */
	    g_list_free(pc->values_list);
	    pc->values_list = NULL;
	}

	/* Unref/delete the rest of the resources on the Pie Chart */
	(void)GTK_OBJECT_UNREF(GTK_OBJECT(pc->adj));

	(void)GDK_PIXMAP_UNREF(pc->pie_chart_pixmap);

	(void)GDK_GC_UNREF(pc->gc);

	if(colormap != NULL)
	{
	    GDK_COLORMAP_FREE_COLOR(colormap, &pc->color);
	    GDK_COLORMAP_FREE_COLOR(colormap, &pc->shade_color);
	    GDK_COLORMAP_FREE_COLOR(colormap, &pc->shadow_color);
	    (void)GDK_COLORMAP_UNREF(colormap);
	}

	g_free(pc->base_type_label);
	g_free(pc->base_value_label);

	pc->freeze_count--;

	g_free(pc);
}

/*
 *	Pie Chart GtkDrawingArea "realized" signal callback.
 */
static void PieChartRealizeCB(GtkWidget *widget, gpointer data)
{
	GdkColormap *colormap;
	GdkWindow *window;
	PieChart *pc = PIE_CHART(data);
	if((widget == NULL) || (pc == NULL))
	    return;

	window = widget->window;
	if(window == NULL)
	    return;

	/* Get the colormap */
	colormap = pc->colormap;
	if(colormap == NULL)
	    pc->colormap = colormap = GDK_COLORMAP_REF(gtk_widget_get_colormap(widget));

	/* Create the GC */
	if(pc->gc == NULL)
	    pc->gc = gdk_gc_new(window);

	/* Allocate the colors and create the base value label GtkWidgets? */
	if(!(pc->flags & PIE_CHART_REALIZED))
	{
	    /* Allocate the colors */
	    GDK_COLORMAP_ALLOC_COLOR(colormap, &pc->color);
	    GDK_COLORMAP_ALLOC_COLOR(colormap, &pc->shade_color);
	    GDK_COLORMAP_ALLOC_COLOR(colormap, &pc->shadow_color);

	    /* Realize the base value */
	    PieChartValueRealize(
		pc,
		&pc->color,
		pc->base_type_label,
		pc->base_value_label,
		NULL,
		NULL,
		&pc->base_color_da
	    );
	}

	/* Realize the rest of the values on the Pie Chart */
	PieChartRealizeValues(pc);

	/* Mark the Pie Chart as realized */
	pc->flags |= PIE_CHART_REALIZED;
}

/*
 *	GtkDrawingArea "configure_event" signal callback.
 */
static gint PieChartConfigureEventCB(
	GtkWidget *widget, GdkEventConfigure *configure, gpointer data
)
{
	gint width, height;
	GdkWindow *window;
	PieChart *pc = PIE_CHART(data);
	if((widget == NULL) || (configure == NULL) || (pc == NULL))
	    return(FALSE);

	window = configure->window;
	width = configure->width;
	height = configure->height;

	/* Recreate the Pie Chart GtkDrawingArea back buffer GdkPixmap */
	GDK_PIXMAP_UNREF(pc->pie_chart_pixmap);
	if((pc->flags & PIE_CHART_DOUBLE_BUFFER) &&
	   (width > 0) && (height > 0)
	)
	    pc->pie_chart_pixmap = gdk_pixmap_new(
		window,
		width, height,
		-1
	    );
	else
	    pc->pie_chart_pixmap = NULL;

	return(TRUE);
}

/*
 *	GtkDrawingArea "expose_event" signal callback.
 */
static gint PieChartExposeEventCB(
	GtkWidget *widget, GdkEventExpose *expose, gpointer data
)
{
	PieChart *pc = PIE_CHART(data);
	if((widget == NULL) || (pc == NULL))
	    return(FALSE);

	PieChartDraw(pc);

	return(TRUE);
}


/*
 *	Draws a segment of a pie arc.
 *
 *	The drawable specifies the GdkDrawable to draw to.
 *
 *	The x and y specifies the upper-left position of the pie arc.
 *
 *	The width and height specifies the maximum size of the pie
 *	arc.
 *
 *	The start_angle and end_angle specifies the segment to draw
 *	in units from 0.0 to 1.0 starting at the 12 o'clock position
 *	being 0.0.
 */
static void PieChartDrawArc(
	GdkDrawable *drawable,
	GdkGC *gc,
	const gint x, const gint y,
	const gint width, const gint height,
	const gfloat start_angle,		/* 0.0 to 1.0 */
	const gfloat end_angle,			/* 0.0 to 1.0 */
	const gboolean outline_circumferance,
	const gboolean outline_radius,
	const gboolean outline_height
)
{
	gfloat	_start_angle = start_angle,
		delta_angle;

	/* Calculate the delta angle */
	delta_angle = _start_angle - end_angle;
	if(delta_angle == 0.0f)
	    return;

	/* Outline radius? */
	if(outline_radius &&
	   ((_start_angle != 0.0f) || (end_angle != 1.0f))
	)
	{
	    gfloat	ostart_angle = _start_angle,
			oend_angle = end_angle;
	    gint	rx = (width / 2),
			ry = (height / 2),
			cx = x + rx,
			cy = y + ry;

	    /* Sanitize outline start and end angles */
	    while(ostart_angle > 1.0f)
		ostart_angle -= 1.0f;
	    while(ostart_angle < 0.0f)
		ostart_angle += 1.0f;

	    while(oend_angle > 1.0f)
		oend_angle -= 1.0f;
	    while(oend_angle < 0.0f)
		oend_angle += 1.0f;

	    gdk_draw_line(
		drawable,
		gc,
		cx, cy,
		cx + (rx * sin(ostart_angle * 2.0f * PI)),
		cy + (ry * -cos(ostart_angle * 2.0f * PI))
	    );
	    gdk_draw_line(
		drawable,
		gc,
		cx, cy,
		cx + (rx * sin(oend_angle * 2.0f * PI)),
		cy + (ry * -cos(oend_angle * 2.0f * PI))
	    );
	}

	/* Outline height? */
	if(outline_height)
	{
	    gfloat      ostart_angle = _start_angle,
			oend_angle = end_angle;
	    gint        rx = (width / 2),
			ry = (height / 2),
			cx = x + rx,
			cy = y + ry;

	    /* Sanitize outline start and end angles */
	    while(ostart_angle > 1.0f)
		ostart_angle -= 1.0f;
	    while(ostart_angle < 0.0f)
		ostart_angle += 1.0f;

	    while(oend_angle > 1.0f)
		oend_angle -= 1.0f;
	    while(oend_angle < 0.0f)
		oend_angle += 1.0f;

	    /* Draw height lines dividing pie segments */
	    if((ostart_angle > 0.25f) && (ostart_angle < 0.75f))
		gdk_draw_line(
		    drawable,
		    gc,
		    cx + (rx * sin(ostart_angle * 2.0f * PI)),
		    cy + (ry * -cos(ostart_angle * 2.0f * PI)),
		    cx + (rx * sin(ostart_angle * 2.0f * PI)),
		    cy + (ry * -cos(ostart_angle * 2.0f * PI)) +
			PIE_CHART_PIE_HEIGHT
		);
	    if((oend_angle > 0.25f) && (oend_angle < 0.75f))
		gdk_draw_line(
		    drawable,
		    gc,
		    cx + (rx * sin(oend_angle * 2.0f * PI)),
		    cy + (ry * -cos(oend_angle * 2.0f * PI)),
		    cx + (rx * sin(oend_angle * 2.0f * PI)),
		    cy + (ry * -cos(oend_angle * 2.0f * PI)) +
			PIE_CHART_PIE_HEIGHT
		);

	    /* Draw height lines at far left and right edges */
	    gdk_draw_line(
		drawable,
		gc,
		x, y + ry,
		x, y + ry + PIE_CHART_PIE_HEIGHT - 1
	    );
	    gdk_draw_line(
		drawable,
		gc,
		x + width, y + ry,
		x + width, y + ry + PIE_CHART_PIE_HEIGHT - 1
	    );
	}

	/* Convert angles to degrees and make it relative to the 3
	 * o'clock position
	 */
	_start_angle = 90.0f - (_start_angle * 360.0f);
	while(_start_angle < 0.0f)
	    _start_angle += 360.0f;
	while(_start_angle >= 360.0f)
	    _start_angle -= 360.0f;

	delta_angle *= 360.0f;

	gdk_draw_arc(
	    drawable,
	    gc,
	    !outline_circumferance,
	    x, y,
	    width, height,
	    (gint)(_start_angle * 64.0f),
	    (gint)(delta_angle * 64.0f)
	);
}

/*
 *	Draws one iteration of a complete pie disk.
 *
 *	The drawable specifies the GdkDrawable to draw to.
 *
 *	The x and y specifies the coordinates.
 *
 *	The width and height specifies the size of the pie disk.
 */
static void PieChartDrawPieIterate(
	PieChart *pc,
	GdkDrawable *drawable,
	const GtkStateType state,
	GtkStyle *style,
	const gint x, const gint y,
	const gint width, const gint height,
	const gboolean outline_circumference,
	const gboolean outline_radius,
	const gboolean outline_height,
	const gboolean shaded,
	const gboolean shadow
)
{
	const gboolean outline = (outline_circumference || outline_radius || outline_height) ?
	    TRUE : FALSE;
	gfloat values_range, total_range, values_range_coeff;
	GList *glist;
	GdkGC *gc;
	GtkAdjustment *adj;
	PieChartValue *v;

	if(outline)
	    gc = style->fg_gc[state];
	else
	    gc = pc->gc;
	if(gc == NULL)
	    return;

	/* Set foreground color only if not drawing for an outline */
	if(!outline)
	{
	    if(shaded)
		gdk_gc_set_foreground(gc, &pc->shade_color);
	    else if(shadow)
		gdk_gc_set_foreground(gc, &pc->shadow_color);
	    else
		gdk_gc_set_foreground(gc, &pc->color);
	    gdk_gc_set_fill(gc, GDK_SOLID);
	}

	/* Calculate total range of all values */
	values_range = 0.0f;
	for(glist = pc->values_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    v = PIE_CHART_VALUE(glist->data);
	    if(v == NULL)
		continue;

	    adj = v->adj;
	    values_range += (adj != NULL) ? (adj->upper - adj->lower) : 0.0f;
	}

	/* Calculate absolute total range */
	adj = pc->adj;
	total_range = (adj != NULL) ? (adj->upper - adj->lower) : 0.0f;

	/* Check if more of the values range is visible on the `lower'
	 * side
	 *
	 * Note that angles are in units from 0.0 to 1.0 starting at
	 * the 12 o'clock position being 0.0 going clockwise
	 */
	values_range_coeff = (total_range > 0.0f) ?
	    (values_range / total_range) : 0.0f;
	if(values_range_coeff < 0.5f)
	{
	    gfloat value_pos_coeff = 0.5 + (values_range_coeff / 2.0);
	    gfloat value_pos_delta;

	    /* Draw the base disk */
	    PieChartDrawArc(
		drawable,
		gc,
		x, y,
		width, height,
		0.0f, 1.0f,
		outline_circumference,
		outline_radius,
		outline_height
	    );
	    /* Draw each value only if this disk is not a shadow */
	    if(!shadow)
	    {
		/* Draw each value */
		for(glist = pc->values_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    v = PIE_CHART_VALUE(glist->data);
		    if(v == NULL)
			continue;

		    adj = v->adj;
		    if((adj != NULL) && (total_range > 0.0f))
			value_pos_delta = (adj->upper - adj->lower) /
			    total_range;
		    else
			value_pos_delta = 0.0f;

		    if(!outline)
		    {
			if(shaded)
			    gdk_gc_set_foreground(gc, &v->shade_color);
			else
			    gdk_gc_set_foreground(gc, &v->color);
			gdk_gc_set_fill(gc, GDK_SOLID);
		    }

		    PieChartDrawArc(
			drawable,
			gc,
			x, y,
			width, height,
			value_pos_coeff,
			value_pos_coeff - value_pos_delta,
			outline_circumference,
			outline_radius,
			outline_height
		    );

		    value_pos_coeff -= value_pos_delta;
		}
	    }
	}
	else
	{
	    gfloat	value_pos_coeff = 1.0f - (values_range_coeff / 2.0f),
			value_pos_delta;

	    /* Draw the base disk */
	    PieChartDrawArc(
		drawable,
		gc,
		x, y,
		width, height,
		0.0f, 1.0f,
		outline_circumference,
		outline_radius,
		outline_height
	    );

	    /* Draw each value only if this disk is not a shadow */
	    if(!shadow)
	    {
		/* Draw each value */
		for(glist = pc->values_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    v = PIE_CHART_VALUE(glist->data);
		    if(v == NULL)
			continue;

		    adj = v->adj;
		    if((adj != NULL) && (total_range > 0.0f))
			value_pos_delta = (adj->upper - adj->lower) /
			    total_range;
		    else
			value_pos_delta = 0.0f;

		    if(!outline)
		    {
			if(shaded)
			    gdk_gc_set_foreground(gc, &v->shade_color);
			else
			    gdk_gc_set_foreground(gc, &v->color);
			gdk_gc_set_fill(gc, GDK_SOLID);
		    }

		    PieChartDrawArc(
			drawable,
			gc,
			x, y,
			width, height,
			value_pos_coeff,
			value_pos_coeff + value_pos_delta,
			outline_circumference,
			outline_radius,
			outline_height
		    );

		    value_pos_coeff += value_pos_delta;
		}
	    }
	}
}

/*
 *	Draws the Pie Chart.
 */
static void PieChartDraw(PieChart *pc)
{
	gint		x, y,
			width, height,
			pwidth, pheight;
	GdkWindow *window;
	GdkDrawable *drawable;
	GdkGC *gc;
	GtkStateType state;
	GtkStyle *style;
	pie_chart_flags flags;
	GtkWidget *w = pc->pie_chart_da;
	if(!GTK_WIDGET_REALIZED(w) || !GTK_WIDGET_VISIBLE(w))
	    return;

	flags = pc->flags;
	window = w->window;
	state = GTK_WIDGET_STATE(w);
	style = gtk_widget_get_style(w);
	if((window == NULL) || (style == NULL))
	    return;

	gc = pc->gc;
	gdk_window_get_size(window, &width, &height);
	if((width <= 0) || (height <= 0) || (gc == NULL))
	    return;

	drawable = (GdkDrawable *)pc->pie_chart_pixmap;
	if(drawable == NULL)
	    drawable = (GdkDrawable *)window;


	/* Begin drawing the Pie Chart */

	/* Draw the background */
	gtk_style_apply_default_background(
	    style,
	    drawable,
	    FALSE,
	    state,
	    NULL,
	    0, 0,
	    width, height
	);

	/* Shadow */
	x = 4;
	y = PIE_CHART_PIE_HEIGHT + 2;
	pwidth = width - 6;
	pheight = height - PIE_CHART_PIE_HEIGHT - 4;
	if(flags & PIE_CHART_SHOW_SHADOW)
	    PieChartDrawPieIterate(
		pc,
		drawable,
		state,
		style,
		x, y,
		pwidth, pheight,
		FALSE,				/* Outline circumference */
		FALSE, FALSE,			/* Outline radius and height */
		FALSE,				/* Not shaded */
		TRUE				/* Is shadow */
	    );

	/* Base outline */
	x = 0;
	y = PIE_CHART_PIE_HEIGHT;
	if(flags & PIE_CHART_SHOW_OUTLINE)
	{
	    PieChartDrawPieIterate(
		pc,
		drawable,
		state,
		style,
		x, y,
		pwidth, pheight,
		FALSE,				/* Outline circumference */
		FALSE, FALSE,			/* Outline radius and height */
		TRUE,				/* Shaded */
		FALSE				/* Not shadow */
	    );
	    PieChartDrawPieIterate(
		pc,
		drawable,
		state,
		style,
		x, y,
		pwidth, pheight,
		TRUE,				/* Outline circumference */
		FALSE, FALSE,			/* Outline radius and height */
		TRUE,				/* Shaded */
		FALSE				/* Not shadow */
	    );
	    y--;
	}

	/* Sides */
	while(y > 0)
	{
	    PieChartDrawPieIterate(
		pc,
		drawable,
		state,
		style,
		x, y,
		pwidth, pheight,
		FALSE, 				/* Outline circumference */
		FALSE, FALSE,			/* Outline radius and height */
		TRUE,				/* Shaded */
		FALSE				/* Not shadow */
	    );
	    y--;
	}

	/* Top surface */
	y = 0;
	PieChartDrawPieIterate(
	    pc,
	    drawable,
	    state,
	    style,
	    x, y, pwidth, pheight,
	    FALSE,				/* Outline circumference */
	    FALSE, FALSE,			/* Outline radius and height */
	    FALSE,				/* Not shaded */
	    FALSE				/* Not shadow */
	);

	/* Top outline */
	if(flags & PIE_CHART_SHOW_OUTLINE)
	    PieChartDrawPieIterate(
		pc,
		drawable,
		state,
		style,
		x, y, pwidth, pheight,
		TRUE,				/* Outline circumference */
		TRUE, TRUE,			/* Outline radius and height */
		FALSE,				/* Not shaded */
		FALSE				/* Not shadow */
	    );

	/* Send the drawable to the if the drawable is not the window */
	if(drawable != (GdkDrawable *)window)
	    gdk_draw_pixmap(
		(GdkDrawable *)window,
		gc,
		drawable,
		0, 0,
		0, 0,
		width, height
	    );
}


/*
 *	Creates a new value label and parents it to the Pie Chart's
 *	value_labels_vbox.
 *
 *	The Pie Chart must be realized before calling this function.
 */
static void PieChartValueRealize(
	PieChart *pc,
	GdkColor *c,
	const gchar *type_label,
	const gchar *value_label,
	GtkWidget **type_label_box_rtn,
	GtkWidget **value_label_box_rtn,
	GtkWidget **color_da_rtn
)
{
	const gint	border_minor = 2;
	gint		da_width = 12,
			da_height = 12;
	GdkFont *font;
	GdkWindow *window;
	GtkStyle *style;
	GtkWidget *w, *parent, *parent2;
	GtkWidget *type_labels_vbox, *value_labels_vbox;

	/* Reset the returns */
	if(type_label_box_rtn != NULL)
	    *type_label_box_rtn = NULL;
	if(value_label_box_rtn != NULL)
	    *value_label_box_rtn = NULL;
	if(color_da_rtn != NULL)
	    *color_da_rtn = NULL;

	if(STRISEMPTY(type_label))
	    return;

	w = pc->toplevel;
	style = gtk_widget_get_style(w);
	if(style == NULL)
	    return;

	font = style->font;

	/* Get the parent */
	type_labels_vbox = parent = pc->type_labels_vbox;
	if(parent == NULL)
	    return;

	/* Labels GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent = w;
	if(type_label_box_rtn != NULL)
	    *type_label_box_rtn = w;

	/* Color GtkFrame */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Value Color GtkDrawingArea */
	if(font != NULL)
	{
	    da_height = MAX(font->ascent + font->descent - 2, 2);
	    da_width = da_height;
	}
	w = gtk_drawing_area_new();
	gtk_widget_set_usize(w, da_width, da_height);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	    gdk_window_set_background(
		window,
		c
	    );
	gtk_widget_show(w);
	if(color_da_rtn != NULL)
	    *color_da_rtn = w;

	/* Type GtkLabel */
	w = gtk_label_new(type_label);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Value GtkLabel */
	value_labels_vbox = parent = pc->value_labels_vbox;
	if((value_label != NULL) && (parent != NULL))
	{
	    w = gtk_hbox_new(FALSE, 0);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;
	    if(value_label_box_rtn != NULL)
		*value_label_box_rtn = w;

	    w = gtk_alignment_new(0.0f, 0.5f, 0.0f, 0.0f);
	    gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    w = gtk_label_new(value_label);
	    gtk_container_add(GTK_CONTAINER(parent2), w);
	    gtk_widget_show(w);
	}
}

/*
 *	Realizes all the values.
 *
 *	The Pie Chart must be realized before calling this function.
 *
 *	Values that are already realized will not be re-realized.
 */
static void PieChartRealizeValues(PieChart *pc)
{
	GList *glist;
	GdkColormap *colormap = pc->colormap;
	PieChartValue *v;

	/* Realize all the values */
	for(glist = pc->values_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    v = PIE_CHART_VALUE(glist->data);
	    if(v == NULL)
		continue;

	    if(v->flags & PIE_CHART_VALUE_REALIZED)
		continue;

	    /* Allocate the colors */
	    GDK_COLORMAP_ALLOC_COLOR(colormap, &v->color);
	    GDK_COLORMAP_ALLOC_COLOR(colormap, &v->shade_color);

	    /* Realize this value */
	    PieChartValueRealize(
		pc,
		&v->color,
		v->type_label,
		v->value_label,
		&v->type_label_box,
		&v->value_label_box,
		&v->color_da
	    );

	    /* Mark this value as realized */
	    v->flags |= PIE_CHART_VALUE_REALIZED;
	}
}


/*
 *	Creates a new Pie Chart Value.
 */
static PieChartValue *PieChartValueNew(void)
{
	return(PIE_CHART_VALUE(g_malloc0(sizeof(PieChartValue))));
}

/*
 *	Deletes the Pie Chart Value.
 */
static void PieChartValueDelete(PieChartValue *v)
{
	PieChart *pc;

	if(v == NULL)
	    return;

	pc = v->pc;

	/* Unref the GtkAdjustment */
	v->adj = (GtkAdjustment *)GTK_OBJECT_UNREF(GTK_OBJECT(v->adj));

	/* Delete the colors? */
	if(v->flags & PIE_CHART_VALUE_REALIZED)
	{
	    GdkColormap *colormap = (pc != NULL) ? pc->colormap : NULL;
	    if(colormap != NULL)
	    {
		GDK_COLORMAP_FREE_COLOR(colormap, &v->color);
		GDK_COLORMAP_FREE_COLOR(colormap, &v->shade_color);
	    }
	}

	GTK_WIDGET_DESTROY(v->type_label_box);
	GTK_WIDGET_DESTROY(v->value_label_box);

	g_free(v->type_label);
	g_free(v->value_label);

	g_free(v);
}


/*
 *	Adds a new value to the Pie Chart.
 */
gint PieChartValueAdd(
	GtkWidget *w,
	GtkAdjustment *adj,
	GdkColor *c,
	const gchar *type_label,
	const gchar *value_label
)
{
	gint i;
	GdkColor *tc;
	PieChartValue *v;
	PieChart *pc = PIE_CHART(GTK_OBJECT_GET_DATA(
	    w,
	    PIE_CHART_KEY
	));
	if((pc == NULL) || (adj == NULL) || (c == NULL))
	    return(-2);

	/* Create a new Pie Chart Value */
	v = PieChartValueNew();
	if(v == NULL)
	    return(-3);

	i = g_list_length(pc->values_list);

	/* Add the new Pie Chart Value to the values list */
	pc->values_list = g_list_append(
	    pc->values_list,
	    v
	);

	/* Set the Pie Chart that this value is on */
	v->pc = pc;

	/* Set the GtkAdjustment */
	v->adj = adj = (GtkAdjustment *)GTK_OBJECT_REF(GTK_OBJECT(adj));

	/* Set the new colors */
	memcpy(
	    &v->color,
	    c,
	    sizeof(GdkColor)
	);
	tc = &v->shade_color;
	tc->red		= c->red	/ 2;
	tc->green	= c->green	/ 2;
	tc->blue	= c->blue	/ 2;

	/* Set the labels */
	v->type_label = STRDUP(type_label);
	v->value_label = STRDUP(value_label);

	/* Realize the new value if the Pie Chart is realized */
	if(pc->flags & PIE_CHART_REALIZED)
	    PieChartRealizeValues(pc);

	PieChartQueueDraw(w);

	return(i);
}

/*
 *	Sets the value on the Pie Chart.
 */
void PieChartValueSet(
	GtkWidget *w,
	const gint value_num,
	GtkAdjustment *adj,
	GdkColor *c,
	const gchar *type_label,
	const gchar *value_label
)
{
	GdkColor *tc;
	PieChartValue *v;
	PieChart *pc = PIE_CHART(GTK_OBJECT_GET_DATA(
	    w,
	    PIE_CHART_KEY
	));
	if((pc == NULL) || (adj == NULL) || (c == NULL))
	    return;

	v = PIE_CHART_VALUE(g_list_nth_data(
	    pc->values_list,
	    value_num
	));
	if(v == NULL)
	    return;

	/* Set the new GtkAdkustment */
	v->adj = (GtkAdjustment *)GTK_OBJECT_UNREF(GTK_OBJECT(v->adj));
	v->adj = adj = GTK_ADJUSTMENT(GTK_OBJECT_REF(GTK_OBJECT(adj)));

	/* Delete the old colors? */
	if(v->flags & PIE_CHART_VALUE_REALIZED)
	{
	    GdkColormap *colormap = pc->colormap;
	    if(colormap != NULL)
	    {
		GDK_COLORMAP_FREE_COLOR(colormap, &v->color);
		GDK_COLORMAP_FREE_COLOR(colormap, &v->shade_color);
	    }
	}

	/* Set the new colors */
	memcpy(
	    &v->color,
	    c,
	    sizeof(GdkColor)
	);
	tc = &v->shade_color;
	tc->red		= c->red	/ 2;
	tc->green	= c->green	/ 2;
	tc->blue	= c->blue	/ 2;

	/* Allocate the new colors? */
	if(v->flags & PIE_CHART_VALUE_REALIZED)
	{
	    GdkColormap *colormap = pc->colormap;
	    if(colormap != NULL)
	    {
		GDK_COLORMAP_ALLOC_COLOR(colormap, &v->color);
		GDK_COLORMAP_ALLOC_COLOR(colormap, &v->shade_color);
	    }
	}

	/* Set the new label text */
	g_free(v->type_label);
	v->type_label = STRDUP(type_label);

	g_free(v->value_label);
	v->value_label = STRDUP(value_label);

	PieChartQueueDraw(w);
}

/*
 *	Removes the value from the Pie Chart.
 */
void PieChartValueRemove(
	GtkWidget *w,
	const gint value_num
)
{
	PieChartValue *v;
	PieChart *pc = PIE_CHART(GTK_OBJECT_GET_DATA(
	    w,
	    PIE_CHART_KEY
	));
	if(pc == NULL)
	    return;

	v = PIE_CHART_VALUE(g_list_nth_data(
	    pc->values_list,
	    value_num
	));
	if(v == NULL)
	    return;

	PieChartValueDelete(v);
	g_list_remove(pc->values_list, v);

	PieChartQueueDraw(w);
}

/*
 *	Removes all the values on the Pie Chart.
 */
void PieChartClear(GtkWidget *w)
{
	PieChart *pc = PIE_CHART(GTK_OBJECT_GET_DATA(
	    w,
	    PIE_CHART_KEY
	));
	if(pc == NULL)
	    return;

	if(pc->values_list != NULL)
	{
	    g_list_foreach(
		pc->values_list,
		(GFunc)PieChartValueDelete,
		NULL
	    );
	    g_list_free(pc->values_list);
	    pc->values_list = NULL;

	    PieChartQueueDraw(w);
	}
}


/*
 *	Creates a new Pie Chart.
 */
GtkWidget *PieChartNew(
	const pie_chart_flags flags,
	const gint width, const gint height,
	GtkAdjustment *adj,
	GdkColor *c,
	const gchar *title,
	const gchar *footer,
	const gchar *base_type_label,
	const gchar *base_value_label
)
{
	const gint	border_major = 5,
			border_minor = 2;
	GtkWidget *w, *parent, *parent2, *toplevel;
	PieChart *pc = PIE_CHART(g_malloc0(
	    sizeof(PieChart)
	));
	if(pc == NULL)
	    return(NULL);

	pc->toplevel = toplevel = gtk_vbox_new(FALSE, border_minor);
	pc->freeze_count = 0;
	pc->flags = flags;
	pc->flags &= ~PIE_CHART_REALIZED;
	pc->width = width;
	pc->height = height;
	pc->adj = adj = (GtkAdjustment *)GTK_OBJECT_REF(GTK_OBJECT(adj));

	if(c != NULL)
	{
	    GdkColor *tc;
	    tc = &pc->color;
	    memcpy(tc, c, sizeof(GdkColor));
	    tc = &pc->shade_color;
	    tc->red	= c->red	/ 2;
	    tc->green	= c->green	/ 2;
	    tc->blue	= c->blue	/ 2;
	    tc = &pc->shadow_color;
	    tc->red	= 0.5f * (gushort)-1;
	    tc->green	= 0.5f * (gushort)-1;
	    tc->blue	= 0.5f * (gushort)-1;
	}
	pc->base_type_label = STRDUP(base_type_label);
	pc->base_value_label = STRDUP(base_value_label);
	pc->values_list = NULL;

	pc->freeze_count++;

	/* Toplevel GtkVBox */
	w = toplevel;
	gtk_object_set_data_full(
	    GTK_OBJECT(w), PIE_CHART_KEY,
	    pc, PieChartDestroyCB
	);
	parent = w;

	/* Title */
	if(title != NULL)
	{
	    pc->title_label = w = gtk_label_new(title);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}

	/* Type and value labels GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Type labels GtkVBox */
	pc->type_labels_vbox = w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Center right column GtkHBox */
	w = gtk_hbox_new(TRUE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Value labels GtkVBox */
	pc->value_labels_vbox = w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_show(w);

	/* Pie Chart GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Pie Chart GtkDrawingArea */
	pc->pie_chart_da = w = gtk_drawing_area_new();
	gtk_widget_set_usize(w, width, height);
	gtk_widget_add_events(
	    w,
	    GDK_STRUCTURE_MASK | GDK_EXPOSURE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "realize",
	    GTK_SIGNAL_FUNC(PieChartRealizeCB), pc
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "configure_event",
	    GTK_SIGNAL_FUNC(PieChartConfigureEventCB), pc
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "expose_event",
	    GTK_SIGNAL_FUNC(PieChartExposeEventCB), pc
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_show(w);

	/* Footer */
	if(footer != NULL)
	{
	    pc->footer_label = w = gtk_label_new(footer);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	}

	pc->freeze_count--;

	return(pc->toplevel);
}

/*
 *	Realizes the Pie Chart.
 */
void PieChartRealize(GtkWidget *w)
{
	PieChart *pc = PIE_CHART(GTK_OBJECT_GET_DATA(
	    w,
	    PIE_CHART_KEY
	));
	if(pc == NULL)
	    return;

	/* Already realized? */
	if(pc->flags & PIE_CHART_REALIZED)
	    return;

	/* Realize the Pie Chart */
	gtk_widget_realize(pc->pie_chart_da);
}

/*
 *	Queues the Pie Chart to draw.
 */
void PieChartQueueDraw(GtkWidget *w)
{
	PieChart *pc = PIE_CHART(GTK_OBJECT_GET_DATA(
	    w,
	    PIE_CHART_KEY
	));
	if(pc == NULL)
	    return;

	gtk_widget_queue_draw(pc->pie_chart_da);
}
