#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>

#include "guiutils.h"
#include "pulist.h"
#include "time_prompt.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_recycle_bin_index.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "libendeavour2-base/edv_id.h"
#include "edv_ids_list.h"
#include "edv_date_format.h"
#include "edv_utils_gtk.h"
#include "obj_op_dlg.h"
#include "obj_op_dlg_op.h"
#include "edv_vfs_obj_op.h"
#include "edv_recycled_obj_op.h"
#include "edv_emit.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"


static gboolean EDVObjOpDlgCheckCopyMultipleToNonDirectory(
	const gint nsrc_objs,
	const gchar *tar_path,
	EDVVFSObject *tar_obj
);

void EDVObjOpDlgProcess(edv_obj_op_dlg_struct *d);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Checks if copying or moving multiple objects to a non-directory.
 *
 *	Returns TRUE if copying multiple objects to a non-directory.
 */
static gboolean EDVObjOpDlgCheckCopyMultipleToNonDirectory(
	const gint nsrc_objs,
	const gchar *tar_path,
	EDVVFSObject *tar_obj
)
{
	/* Not copying multiple objects? */
	if(nsrc_objs <= 1)
	    return(FALSE);

	/* Assume non-directory if the statistics are not available */
	if(tar_obj == NULL)
	    return(TRUE);

	/* Is the target a directory? */
	if(EDV_VFS_OBJECT_IS_DIRECTORY(tar_obj))
	    return(FALSE);

	return(TRUE);
}


/*
 *	Performs the operation based on the current values set
 *	on the Object Operations Dialog.
 *
 *	The values from the Object Operations Dialog's widgets and the
 *	list of objects will be collected and the operation specified
 *	by the Object Operations Dialog will be performed on each
 *	object.
 *
 *	The progress dialog will be mapped as needed as the operation
 *	is performed and any errors that occure during or after
 *	the operation will be displayed.
 *
 *	The Object Operations Dialog will then be unmapped and its
 *	values will be reset.
 */
void EDVObjOpDlgProcess(edv_obj_op_dlg_struct *d)
{
	const gboolean show_progress = TRUE;
	gboolean	yes_to_all = FALSE,
			recursive,
			dereference_links;
	gint status;
	gchar *tar_path;
	GList *objs_list;
	GtkWidget	*w,
			*toplevel;
	CfgList *cfg_list;
	EDVLocationType location_type;
	EDVVFSObject *tar_obj;
	edv_obj_op_dlg_op op;
	EDVCore *core;

	if(d == NULL)
	    return;

	toplevel = d->ref_toplevel;
	core = d->core;
	cfg_list = core->cfg_list;

	/* Check and warn if write protect is enabled */
	if(edv_check_master_write_protect(core, TRUE, toplevel))
	    return;

	/* Get the operation code that indicates which operation is
	 * to be performed and the location type
	 */
	op = d->op;
	location_type = d->location_type;

	/* Get the list of objects to operate on */
	objs_list = d->objs_list;

	/* Get the target value */
	w = d->target_entry;
	if((w != NULL) ? GTK_WIDGET_MAPPED(w) : FALSE)
	{
	    tar_path = edv_path_evaluate(
		d->src_dir,			/* Parent used for reference */
		gtk_entry_get_text(GTK_ENTRY(w))
	    );
	}
	else
	{
	    tar_path = NULL;
	}

	/* Get the target object's destination statistics */
	if(tar_path != NULL)
	    tar_obj = edv_vfs_object_stat(tar_path);
	else
	    tar_obj = NULL;

	/* Get the options if they are specified */
	recursive = GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    d->opt_recursive_check
	);
	dereference_links = GTK_TOGGLE_BUTTON_GET_ACTIVE(
	    d->opt_dereference_links_check
	);


	/* Unmap the Object Operations Dialog */
	EDVObjOpDlgUnmap(d);


	status = 0;

	/* Perform the operation
	 *
	 * Copy
	 */
	if(op == EDV_OBJ_OP_DLG_OP_COPY)
	{
	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    const gchar	*error_msg,
				*src_path;
		    GList	*glist,
				*new_paths_list;
		    EDVVFSObject *obj;

		    /* Check if copying multiple objects to a non directory */
		    if(EDVObjOpDlgCheckCopyMultipleToNonDirectory(
			g_list_length(objs_list),
			tar_path,
			tar_obj
		    ))
		    {
			edv_play_sound_warning(core);
			edv_message_warning(
			    "Copy Object Failed",
"Unable to copy multiple objects to a non-directory object",
			    NULL,
			    toplevel
			);
			status = -1;
			objs_list = NULL;	/* So we copy nothing below */
		    }

		    /* Copy each object */
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_VFS_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			src_path = obj->path;
			if(STRISEMPTY(src_path))
			    continue;

			/* Copy this object */
			new_paths_list = NULL;
			status = edv_vfs_object_op_copy(
			    core,
			    src_path,
			    tar_path,
			    &new_paths_list,
			    !dereference_links,	/* Archive */
			    toplevel,
			    show_progress,
			    TRUE,		/* Interactive */
			    &yes_to_all
			);

			/* Check for errors */
			error_msg = edv_vfs_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
			    /* Report the error */
			    edv_play_sound_error(core);
			    edv_message_error(
				"Copy Object Error",
				error_msg,
				NULL,
				toplevel
			    );
			}

			/* Does the source object no longer exist? */
			if(!edv_path_lexists(src_path))
			    edv_emit_vfs_object_removed(
				core,
				src_path
			    );

	                /* Report the new objects? */
	                if(new_paths_list != NULL)
	                {
	                    const gchar *path;
	                    GList *glist;
			    EDVVFSObject *obj;

	                    for(glist = new_paths_list;
	                        glist != NULL;
	                        glist = g_list_next(glist)
	                    )
	                    {
	                        path = (const gchar *)glist->data;
	                        if(STRISEMPTY(path))
	                            continue;

				obj = edv_vfs_object_lstat(path);
	                        if(obj != NULL)
				{
	                            edv_emit_vfs_object_added(
	                                core,
	                                path,
	                                obj
	                            );
				    edv_vfs_object_delete(obj);
				}
	                    }

	                    g_list_foreach(new_paths_list, (GFunc)g_free, NULL);
	                    g_list_free(new_paths_list);
			}

			/* User aborted? */
			if(status == -4)
			    break;
		    }
		}
		break;

	      case EDV_LOCATION_TYPE_RECYCLE_BIN:
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	    }
	}
	/* Move */
	else if(op == EDV_OBJ_OP_DLG_OP_MOVE)
	{
	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    const gchar	*error_msg,
				*src_path;
		    GList	*glist,
				*new_paths_list;
		    EDVVFSObject *obj;

		    /* Check if moving multiple objects to a non directory */
		    if(EDVObjOpDlgCheckCopyMultipleToNonDirectory(
			g_list_length(objs_list),
			tar_path,
			tar_obj
		    ))
		    {
			edv_play_sound_warning(core);
			edv_message_warning(
			    "Move Object Failed",
"Unable to move multiple objects to a non-directory object",
			    NULL,
			    toplevel
			);
			status = -1;
			objs_list = NULL;	/* So we move nothing below */
		    }

		    /* Move each object */
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_VFS_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			src_path = obj->path;
			if(STRISEMPTY(src_path))
			    continue;

			/* Move this object */
			new_paths_list = NULL;
			status = edv_vfs_object_op_move(
			    core,
			    src_path,
			    tar_path,
			    &new_paths_list,
			    !dereference_links,	/* Archive */
			    toplevel,
			    show_progress,
			    TRUE,		/* Interactive */
			    &yes_to_all
			);

			/* Check for errors */
			error_msg = edv_vfs_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
			    /* Report the error */
			    edv_play_sound_error(core);
			    edv_message_error(
				"Move Object Error",
				error_msg,
				NULL,
				toplevel
			    );
			}

			/* Does the source object no longer exist? */
			if(!edv_path_lexists(src_path))
			    edv_emit_vfs_object_removed(
				core,
				src_path
			    );

	                /* Report the new objects? */
	                if(new_paths_list != NULL)
	                {
	                    const gchar *path;
	                    GList *glist;
			    EDVVFSObject *obj;

	                    for(glist = new_paths_list;
	                        glist != NULL;
	                        glist = g_list_next(glist)
	                    )
	                    {
	                        path = (const gchar *)glist->data;
	                        if(STRISEMPTY(path))
	                            continue;

				obj = edv_vfs_object_lstat(path);
	                        if(obj != NULL)
				{
	                            edv_emit_vfs_object_added(
	                                core,
	                                path,
	                                obj
	                            );
				    edv_vfs_object_delete(obj);
				}
	                    }

	                    g_list_foreach(new_paths_list, (GFunc)g_free, NULL);
	                    g_list_free(new_paths_list);
			}

			/* User aborted? */
			if(status == -4)
			    break;
		    }
		}
		break;

	      case EDV_LOCATION_TYPE_RECYCLE_BIN:
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		break;
	    }
	}
	/* Link */
	else if(op == EDV_OBJ_OP_DLG_OP_LINK)
	{
	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    const gchar	*error_msg,
				*src_path;
		    GList	*glist,
				*new_paths_list;
		    EDVVFSObject *obj;

		    /* Check if linking multiple objects to a non directory */
		    if(EDVObjOpDlgCheckCopyMultipleToNonDirectory(
			g_list_length(objs_list),
			tar_path,
			tar_obj
		    ))
		    {
			edv_play_sound_warning(core);
			edv_message_warning(
			    "Link Object Failed",
"Unable to link multiple objects to a non-directory object",
			    NULL,
			    toplevel
			);
			status = -1;
			objs_list = NULL;	/* So we link nothing below */
		    }

		    /* Link each object */
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_VFS_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			src_path = obj->path;
			if(STRISEMPTY(src_path))
			    continue;

			/* Link this object */
			new_paths_list = NULL;
			status = edv_vfs_object_op_link(
			    core,
			    tar_path,		/* New link */
			    src_path,		/* Target */
			    &new_paths_list,
			    toplevel,
			    show_progress,
			    TRUE,		/* Interactive */
			    &yes_to_all
			);

			/* Check for errors */
			error_msg = edv_vfs_object_op_get_error(core);
			if(!STRISEMPTY(error_msg))
			{
			    /* Report the error */
			    edv_play_sound_error(core);
			    edv_message_error(
				"Link Object Error",
				error_msg,
				NULL,
				toplevel
			    );
			}

			/* Does the source object no longer exist? */
			if(!edv_path_lexists(src_path))
			    edv_emit_vfs_object_removed(
				core,
				src_path
			    );

	                /* Report the new objects? */
	                if(new_paths_list != NULL)
	                {
	                    const gchar *path;
	                    GList *glist;
			    EDVVFSObject *obj;

	                    for(glist = new_paths_list;
	                        glist != NULL;
	                        glist = g_list_next(glist)
	                    )
	                    {
	                        path = (const gchar *)glist->data;
	                        if(STRISEMPTY(path))
	                            continue;

				obj = edv_vfs_object_lstat(path);
	                        if(obj != NULL)
				{
	                            edv_emit_vfs_object_added(
	                                core,
	                                path,
	                                obj
	                            );
				    edv_vfs_object_delete(obj);
				}
	                    }

	                    g_list_foreach(new_paths_list, (GFunc)g_free, NULL);
	                    g_list_free(new_paths_list);
			}

			/* User aborted? */
			if(status == -4)
			    break;
		    }
		}
		break;

	      case EDV_LOCATION_TYPE_RECYCLE_BIN:
		/* Recycled objects cannot be linked */
		break;

	      case EDV_LOCATION_TYPE_ARCHIVE:
		/* Archive objects cannot be linked */
		break;
	    }
	}
	/* Change Permissions */
	else if(op == EDV_OBJ_OP_DLG_OP_CHMOD)
	{
	    EDVPermissionFlags permissions = 0x00000000;

	    /* Get the permissions value from the widgets */
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_ur_check))
		permissions |= EDV_PERMISSION_UR;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_uw_check))
		permissions |= EDV_PERMISSION_UW;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_ux_check))
		permissions |= EDV_PERMISSION_UX;

	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_gr_check))
		permissions |= EDV_PERMISSION_GR;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_gw_check))
		permissions |= EDV_PERMISSION_GW;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_gx_check))
		permissions |= EDV_PERMISSION_GX;

	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_or_check))
		permissions |= EDV_PERMISSION_OR;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_ow_check))
		permissions |= EDV_PERMISSION_OW;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_ox_check))
		permissions |= EDV_PERMISSION_OX;

	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_suid_check))
		permissions |= EDV_PERMISSION_SETUID;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_sgid_check))
		permissions |= EDV_PERMISSION_SETGID;
	    if(GTK_TOGGLE_BUTTON_GET_ACTIVE(d->target_sticky_check))
		permissions |= EDV_PERMISSION_STICKY;

	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    const gchar *error_msg;
		    GList	*glist,
				*paths_list,
				*modified_paths_list;
		    EDVVFSObject *obj;

		    /* Create the paths list */
		    paths_list = NULL;
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_VFS_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			paths_list = g_list_append(
			    paths_list,
			    STRDUP(obj->path)
			);
		    }

		    /* Change the permissions */
		    status = edv_vfs_object_op_chmod(
			core,
			paths_list,
			permissions,
			&modified_paths_list,
			toplevel,
			show_progress,
			TRUE,			/* Interactive */
			&yes_to_all,
			recursive,
			!dereference_links	/* Archive */
		    );

		    /* Check for errors */
		    error_msg = edv_vfs_object_op_get_error(core);
		    if(!STRISEMPTY(error_msg))
		    {
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
			    "Change Permissions Error",
			    error_msg,
			    NULL,
			    toplevel
			);
		    }

		    /* Notify about the objects being modified */
		    if(modified_paths_list != NULL)
		    {
			const gchar *path;
			GList *glist;
			EDVVFSObject *obj;

			for(glist = modified_paths_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    path = (const gchar *)glist->data;
			    if(path == NULL)
				continue;

			    obj = edv_vfs_object_lstat(path);
			    if(obj != NULL)
			    {
				edv_emit_vfs_object_modified(
				    core,
				    path,
				    path,
				    obj
				);
				edv_vfs_object_delete(obj);
			    }
			}

			/* Delete the modified paths list */
			g_list_foreach(modified_paths_list, (GFunc)g_free, NULL);
			g_list_free(modified_paths_list);
		    }

		    /* Delete the paths list */
		    if(paths_list != NULL)
		    {
			g_list_foreach(paths_list, (GFunc)g_free, NULL);
			g_list_free(paths_list);
		    }
		}
		break;

	      case EDV_LOCATION_TYPE_RECYCLE_BIN:
		if(objs_list != NULL)
		{
		    const gchar *error_msg;
		    GList	*glist,
				*indicies_list = NULL,
				*modified_indicies_list;
		    EDVRecycledObject *obj;

		    /* Create the indicies list */
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_RECYCLED_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			indicies_list = g_list_append(
			    indicies_list,
			    (gpointer)obj->index
			);
		    }

		    /* Change the permissions */
		    status = edv_recycled_object_op_chmod(
			core,
			indicies_list,
			permissions,
			&modified_indicies_list,
			toplevel,
			show_progress,
			TRUE,			/* Interactive */
			&yes_to_all
		    );

		    /* Check for errors */
		    error_msg = edv_recycled_object_op_get_error(core);
		    if(!STRISEMPTY(error_msg))
		    {
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
			    "Change Permissions Error",
			    error_msg,
			    NULL,
			    toplevel
			);
		    }

		    /* Notify about the objects being modified */
		    if(modified_indicies_list != NULL)
		    {
			guint index;
			GList *glist;

			for(glist = modified_indicies_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    index = (guint)glist->data;
			    if(index == 0)
				continue;

			    edv_emit_recycled_object_modified(
				core,
				index
			    );
			}

			/* Delete the modified indicies list */
			g_list_free(modified_indicies_list);
		    }

		    /* Delete the indicies list */
		    g_list_free(indicies_list);
		}
		break;

	      case EDV_LOCATION_TYPE_ARCHIVE:
		/* Archive object permissions cannot be changed */
		break;
	    }
	}
	/* Change Ownership */
	else if(op == EDV_OBJ_OP_DLG_OP_CHOWN)
	{
	    gint owner_id = 0, group_id = 0;
	    GtkWidget *w;

	    /* Get the owner id value from the widget */
	    w = d->target_owner_entry;
	    if(w != NULL)
		owner_id = edv_uid_name_to_uid(
		    core->uids_list,
		    gtk_entry_get_text(GTK_ENTRY(w))
		);

	    /* Get the group id value from the widget */
	    w = d->target_group_entry;
	    if(w != NULL)
		group_id = edv_gid_name_to_gid(
		    core->gids_list,
		    gtk_entry_get_text(GTK_ENTRY(w))
		);

	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    const gchar *error_msg;
		    GList *glist, *paths_list, *modified_paths_list;
		    EDVVFSObject *obj;

		    /* Create the paths list */
		    paths_list = NULL;
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_VFS_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			paths_list = g_list_append(
			    paths_list,
			    STRDUP(obj->path)
			);
		    }

		    /* Change the ownership */
		    status = edv_vfs_object_op_chown(
			core,
			paths_list,
			owner_id, group_id,
			&modified_paths_list,
			toplevel,
			show_progress,
			TRUE,			/* Interactive */
			&yes_to_all,
			recursive,
			!dereference_links	/* Archive */
		    );

		    /* Check for errors */
		    error_msg = edv_vfs_object_op_get_error(core);
		    if(!STRISEMPTY(error_msg))
		    {
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
			    "Change Ownership Error",
			    error_msg,
			    NULL,
			    toplevel
			);
		    }

		    /* Notify about the objects being modified */
		    if(modified_paths_list != NULL)
		    {
			const gchar *path;
			GList *glist;
			EDVVFSObject *obj;

			for(glist = modified_paths_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    path = (const gchar *)glist->data;
			    if(path == NULL)
				continue;

			    obj = edv_vfs_object_lstat(path);
			    if(obj != NULL)
			    {
				edv_emit_vfs_object_modified(
				    core,
				    path,
				    path,
				    obj
				);
				edv_vfs_object_delete(obj);
			    }
			}

			/* Delete the modified paths list */
			g_list_foreach(modified_paths_list, (GFunc)g_free, NULL);
			g_list_free(modified_paths_list);
		    }

		    /* Delete the paths list */
		    if(paths_list != NULL)
		    {
			g_list_foreach(paths_list, (GFunc)g_free, NULL);
			g_list_free(paths_list);
		    }
		}
		break;

	      case EDV_LOCATION_TYPE_RECYCLE_BIN:
		if(objs_list != NULL)
		{
		    const gchar *error_msg;
		    GList	*glist,
				*indicies_list = NULL,
				*modified_indicies_list;
		    EDVRecycledObject *obj;

		    /* Create the indicies list */
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_RECYCLED_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			indicies_list = g_list_append(
			    indicies_list,
			    (gpointer)obj->index
			);
		    }

		    /* Change the ownership */
		    status = edv_recycled_object_op_chown(
			core,
			indicies_list,
			owner_id, group_id,
			&modified_indicies_list,
			toplevel,
			show_progress,
			TRUE,			/* Interactive */
			&yes_to_all
		    );

		    /* Check for errors */
		    error_msg = edv_recycled_object_op_get_error(core);
		    if(!STRISEMPTY(error_msg))
		    {
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
			    "Change Ownership Error",
			    error_msg,
			    NULL,
			    toplevel
			);
		    }

		    /* Notify about the objects being modified */
		    if(modified_indicies_list != NULL)
		    {
			guint index;
			GList *glist;

			for(glist = modified_indicies_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    index = (guint)glist->data;
			    if(index == 0)
				continue;

			    edv_emit_recycled_object_modified(core, index);
			}

			/* Delete the modified indicies list */
			g_list_free(modified_indicies_list);
		    }

		    /* Delete the indicies list */
		    g_list_free(indicies_list);
		}
		break;

	      case EDV_LOCATION_TYPE_ARCHIVE:
		/* Archive object ownership cannot be changed */
		break;
	    }
	}
	/* Change Time Stamps */
	else if(op == EDV_OBJ_OP_DLG_OP_CHTIME)
	{
	    const gulong	access_time = time_prompt_get(d->target_atime_prompt),
				modify_time = time_prompt_get(d->target_mtime_prompt),
				delete_time = time_prompt_get(d->target_dtime_prompt);

	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    const gchar *error_msg;
		    GList	*glist,
				*paths_list,
				*modified_paths_list;
		    EDVVFSObject *obj;

		    /* Create the paths list */
		    paths_list = NULL;
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_VFS_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			paths_list = g_list_append(
			    paths_list,
			    STRDUP(obj->path)
			);
		    }

		    /* Change the time stamps */
		    status = edv_vfs_object_op_chtime(
			core,
			paths_list,
			access_time, modify_time,
			&modified_paths_list,
			toplevel,
			show_progress,
			TRUE,			/* Interactive */
			&yes_to_all,
			recursive,
			!dereference_links	/* Archive */
		    );

		    /* Check for errors */
		    error_msg = edv_vfs_object_op_get_error(core);
		    if(!STRISEMPTY(error_msg))
		    {
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
			    "Change Time Stamps Error",
			    error_msg,
			    NULL,
			    toplevel
			);
		    }

		    /* Notify about the objects being modified */
		    if(modified_paths_list != NULL)
		    {
			const gchar *path;
			GList *glist;
			EDVVFSObject *obj;

			for(glist = modified_paths_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    path = (const gchar *)glist->data;
			    if(path == NULL)
				continue;

			    obj = edv_vfs_object_lstat(path);
			    if(obj != NULL)
			    {
				edv_emit_vfs_object_modified(
				    core,
				    path,
				    path,
				    obj
				);
				edv_vfs_object_delete(obj);
			    }
			}

			/* Delete the modified paths list */
			g_list_foreach(modified_paths_list, (GFunc)g_free, NULL);
			g_list_free(modified_paths_list);
		    }

		    /* Delete the paths list */
		    if(paths_list != NULL)
		    {
			g_list_foreach(paths_list, (GFunc)g_free, NULL);
			g_list_free(paths_list);
		    }
		}
		break;

	      case EDV_LOCATION_TYPE_RECYCLE_BIN:
		if(objs_list != NULL)
		{
		    const gchar *error_msg;
		    GList	*glist,
				*indicies_list = NULL,
				*modified_indicies_list;
		    EDVRecycledObject *obj;

		    /* Create the indicies list */
		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			obj = EDV_RECYCLED_OBJECT(glist->data);
			if(obj == NULL)
			    continue;

			indicies_list = g_list_append(
			    indicies_list,
			    (gpointer)obj->index
			);
		    }

		    /* Change the time stamps */
		    status = edv_recycled_object_op_chtime(
			core,
			indicies_list,
			access_time, modify_time, delete_time,
			&modified_indicies_list,
			toplevel,
			show_progress,
			TRUE,			/* Interactive */
			&yes_to_all
		    );

		    /* Check for errors */
		    error_msg = edv_recycled_object_op_get_error(core);
		    if(!STRISEMPTY(error_msg))
		    {
			/* Report the error */
			edv_play_sound_error(core);
			edv_message_error(
			    "Change Time Stamps Error",
			    error_msg,
			    NULL,
			    toplevel
			);
		    }

		    /* Notify about the objects being modified */
		    if(modified_indicies_list != NULL)
		    {
			guint index;
			GList *glist;

			for(glist = modified_indicies_list;
			    glist != NULL;
			    glist = g_list_next(glist)
			)
			{
			    index = (guint)glist->data;
			    if(index == 0)
				continue;

			    edv_emit_recycled_object_modified(
				core,
				index
			    );
			}

			/* Delete the modified indicies list */
			g_list_free(modified_indicies_list);
		    }

		    /* Delete the indicies list */
		    g_list_free(indicies_list);
		}
		break;

	      case EDV_LOCATION_TYPE_ARCHIVE:
		/* Archive object time stamps cannot be modified */
		break;
	    }
	}
	else
	{
	    gchar *msg = g_strdup_printf(
"EDVObjOpDlgProcess(): Operation \"%i\" not supported.",
		op
	    );
	    edv_play_sound_error(core);
	    edv_message_error(
		"Internal Error",
		msg,
		NULL,
		toplevel
	    );
	    g_free(msg);
	    status = -2;
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Play the completed sound on success */
	if(status == 0)
	    edv_play_sound_completed(core);


	/* Delete the copy of the target path and its statistics */
	edv_vfs_object_delete(tar_obj);
	g_free(tar_path);

	/* Delete the source objects list and the source directory
	 * on the object operations dialog
	 */
	EDVObjOpDlgReset(d);
}
