#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "guiutils.h"
#include "pulist.h"
#include "time_prompt.h"
#include "fb.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_recycled_obj.h"
#include "libendeavour2-base/edv_archive_obj.h"
#include "libendeavour2-base/edv_id.h"
#include "edv_ids_list.h"
#include "edv_pixmap.h"
#include "edv_utils_gtk.h"
#include "obj_op_dlg.h"
#include "obj_op_dlg_op.h"
#include "endeavour2.h"

#include "config.h"

#include "images/icon_cancel_20x20.xpm"
#include "images/icon_folder_opened_20x20.xpm"
#include "images/icon_move_file_20x20.xpm"
#include "images/icon_copy_file_20x20.xpm"
#include "images/icon_link_20x20.xpm"
#include "images/icon_move_file_32x32.xpm"
#include "images/icon_copy_file_32x32.xpm"
#include "images/icon_link_32x32.xpm"
#include "images/icon_chmod_20x20.xpm"
#include "images/icon_chmod_32x32.xpm"
#include "images/icon_owned_20x20.xpm"
#include "images/icon_owned_32x32.xpm"
#include "images/icon_time_stamp_20x20.xpm"
#include "images/icon_time_stamp_32x32.xpm"


/* Callbacks */
static gint EDVObjOpDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVObjOpDlgProcessCB(GtkWidget *widget, gpointer data);
static void EDVObjOpDlgCancelCB(GtkWidget *widget, gpointer data);
static void EDVObjOpDlgBrowseCB(GtkWidget *widget, gpointer data);

static void EDVObjOpDlgOwnerMapPUListCB(
	GtkWidget *widget, gpointer data
);
static void EDVObjOpDlgGroupMapPUListCB(
	GtkWidget *widget, gpointer data
);

/* Object Operations Dialog */
static void EDVObjOpDlgCreatePermissionWidgets(
	GtkWidget *parent,		/* GtkVBox */
	GtkStyle *style,
	GtkWidget **ur_w,
	GtkWidget **uw_w,
	GtkWidget **ux_w,
	GtkWidget **gr_w,
	GtkWidget **gw_w,
	GtkWidget **gx_w,
	GtkWidget **or_w,
	GtkWidget **ow_w,
	GtkWidget **ox_w,
	GtkWidget **suid_w,
	GtkWidget **sgid_w,
	GtkWidget **sticky_w
);
edv_obj_op_dlg_struct *EDVObjOpDlgNew(EDVCore *core);
void EDVObjOpDlgReset(edv_obj_op_dlg_struct *d);
void EDVObjOpDlgSetBusy(edv_obj_op_dlg_struct *d, const gboolean busy);
gboolean EDVObjOpDlgIsMapped(edv_obj_op_dlg_struct *d);
void EDVObjOpDlgMapValues(
	edv_obj_op_dlg_struct *d,
	const edv_obj_op_dlg_op op,
	const EDVLocationType location_type,
	GList *objs_list,
	const gchar *src_dir,
	GtkWidget *toplevel
);
void EDVObjOpDlgUnmap(edv_obj_op_dlg_struct *d);
void EDVObjOpDlgDelete(edv_obj_op_dlg_struct *d);


#define OBJ_OP_DLG_USE_PERMISSION_CHECK_BUTTON	FALSE


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	"delete_event" signal callback.
 */
static gint EDVObjOpDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_obj_op_dlg_struct *d = EDV_OBJ_OP_DLG(data);
	if(d == NULL)
	    return(FALSE);

	if(d->freeze_count > 0)
	    return(TRUE);

	EDVObjOpDlgCancelCB(NULL, d);

	return(TRUE);
}

/*
 *	Process signal callback.
 */
static void EDVObjOpDlgProcessCB(GtkWidget *widget, gpointer data)
{
	edv_obj_op_dlg_struct *d = EDV_OBJ_OP_DLG(data);
	if(d == NULL)
	    return;

	EDVObjOpDlgProcess(d);
}

/*
 *	Cancel signal callback.
 */
static void EDVObjOpDlgCancelCB(GtkWidget *widget, gpointer data)
{
	edv_obj_op_dlg_struct *d = EDV_OBJ_OP_DLG(data);
	if(d == NULL)
	    return;

	/* Unmap the Object Operations Dialog */
	EDVObjOpDlgUnmap(d);         

	/* Delete the source objects list and the source directory
	 * on the object operations dialog
	 */
	EDVObjOpDlgReset(d);
}


/*
 *	Browse callback.
 */
static void EDVObjOpDlgBrowseCB(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gint		npaths = 0,
			nftypes = 0;
	gchar **paths_list = NULL;
	GtkWidget *toplevel;
	fb_type_struct	**ftypes_list = NULL,
			*ftype_rtn = NULL;
	GtkEntry *entry = (GtkEntry *)data;
	if((entry == NULL) || FileBrowserIsQuery())
	    return;

	toplevel = gtk_widget_get_toplevel(GTK_WIDGET(entry));

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for a target object */
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Target Object",
	    "Select", "Cancel",
	    gtk_entry_get_text(entry),          /* Startup path */
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(response && (npaths > 0))
	{
	    gchar *path = STRDUP(paths_list[npaths - 1]);
	    if(path != NULL)
	    {
		gtk_entry_set_text(entry, path);
		gtk_entry_set_position(entry, -1);
		g_free(path);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	/* Reset the file selector due to possible file related change */
	FileBrowserReset();
}


/*
 *      Maps the popup list for the owner entry.
 */
static void EDVObjOpDlgOwnerMapPUListCB(GtkWidget *widget, gpointer data)
{
	gint nitems, nitems_visible;
	const gchar *value;
	GtkWidget *pulist;
	GtkEntry *entry;
	EDVCore *core;
	edv_obj_op_dlg_struct *d = EDV_OBJ_OP_DLG(data);
	if(d == NULL)
	    return;

	entry = (GtkEntry *)d->target_owner_entry;
	if(entry == NULL)
	    return;

	core = d->core;

	pulist = core->users_pulist;
	if(PUListIsQuery(pulist))
	    return;

	nitems = PUListGetTotalItems(pulist);
	nitems_visible = MIN(15, nitems);

	/* Block input and get value */
	value = PUListMapQuery(
	    pulist,			/* Popup List */
	    gtk_entry_get_text(entry),	/* Initial Value */
	    nitems_visible,
	    PULIST_RELATIVE_BELOW,	/* Popup Relativity */
	    GTK_WIDGET(entry),		/* Relative Widget */
	    widget			/* Map Trigger Widget */
	);
	/* Got value? */
	if(value != NULL)
	    gtk_entry_set_text(entry, value);
}

/*
 *      Maps the popup list for the group entry.
 */
static void EDVObjOpDlgGroupMapPUListCB(GtkWidget *widget, gpointer data)
{
	gint nitems, nitems_visible;
	const gchar *value;
	GtkWidget *pulist;
	GtkEntry *entry;
	EDVCore *core;
	edv_obj_op_dlg_struct *d = EDV_OBJ_OP_DLG(data);
	if(d == NULL)
	    return;

	entry = (GtkEntry *)d->target_group_entry;
	if(entry == NULL)
	    return;

	core = d->core;

	pulist = core->groups_pulist;
	if(PUListIsQuery(pulist))
	    return;

	nitems = PUListGetTotalItems(pulist);
	nitems_visible = MIN(15, nitems);

	/* Block input and get value */
	value = PUListMapQuery(
	    pulist,				/* Popup List */
	    gtk_entry_get_text(entry),		/* Initial Value */
	    nitems_visible,
	    PULIST_RELATIVE_BELOW,		/* Popup Relativity */
	    GTK_WIDGET(entry),			/* Relative GtkWidget */
	    widget				/* Map Trigger GtkWidget */
	);
	/* Got value? */
	if(value != NULL)
	    gtk_entry_set_text(entry, value);
}


/*
 *	Creates the permission widgets.
 */
static void EDVObjOpDlgCreatePermissionWidgets(
	GtkWidget *parent,		/* GtkVBox */
	GtkStyle *style,
	GtkWidget **ur_w,
	GtkWidget **uw_w,
	GtkWidget **ux_w,
	GtkWidget **gr_w,
	GtkWidget **gw_w,
	GtkWidget **gx_w,
	GtkWidget **or_w,
	GtkWidget **ow_w,
	GtkWidget **ox_w,
	GtkWidget **suid_w,
	GtkWidget **sgid_w,
	GtkWidget **sticky_w
)
{
	const gint	border_major = 5,
			border_minor = 2;
	GtkWidget	*w,
			*parent2, *parent3;

	/* GtkHBox for the permissions GtkLabels */
	w = gtk_hbox_new(TRUE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;
	w = gtk_label_new("Owner");
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_label_new("Group");
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_label_new("Other");
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_show(w);

	/* GtkHBox for the permissions GtkCheckButtons */
	w = gtk_hbox_new(TRUE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

#define NEW_PERMISSION_TOGGLE_BUTTON(_label_,_tip_,_active_,_parent_)	{ \
 if(OBJ_OP_DLG_USE_PERMISSION_CHECK_BUTTON) {			\
  /* Create the permission button as a GtkCheckButton */	\
  w = gtk_check_button_new_with_label(_label_);			\
 } else {							\
  /* Create the permission button as a GtkToggleButton */	\
  GdkFont *font = style->font;					\
  const gint font_height = (font != NULL) ?			\
   (font->ascent + font->descent) : 0;				\
  w = gtk_toggle_button_new_with_label(_label_);		\
  if(font_height > 0)						\
   gtk_widget_set_usize(					\
    w,								\
    MIN(font_height + 5, 20), MIN(font_height + 5, 20)		\
   );								\
 }								\
 GTK_TOGGLE_BUTTON(w)->active = (_active_) ? TRUE : FALSE;	\
 gtk_box_pack_start(GTK_BOX(_parent_), w, FALSE, FALSE, 0);	\
 if((_tip_) != NULL)						\
  GUISetWidgetTip(w, (_tip_));					\
 gtk_widget_show(w);						\
}

	/* GtkHBox for the owner GtkCheckButtons */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Read GtkCheckButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "R",
"Check this to allow the owner of the object to read from it",
	    FALSE,
	    parent3
	);
	*ur_w = w;
	/* Write GtkCheckButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "W",
"Check this to allow the owner of the object to write to it",
	    FALSE,
	    parent3
	);
	*uw_w = w;
	/* Execute GtkCheckButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "X",
"Check this to allow the owner of the object to execute it.\
 If the object is a directory then this allows the owner to list\
 and access its contents",
	    FALSE,
	    parent3
	);
	*ux_w = w;

	/* GtkHBox for the Group GtkCheckButtons */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Read GtkCheckButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "R",
"Check this to allow any group member of the object to read from it",
	    FALSE,
	    parent3 
	);
	*gr_w = w;
	/* Write GtkCheckButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "W",
"Check this to allow any group member of the object to write to it",
	    FALSE,
	    parent3
	);
	*gw_w = w;
	/* Execute GtkCheckButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "X",
"Check this to allow any group member of the object to execute it.\
 If the object is a directory then this allows any group member to list\
 and access its contents",
	    FALSE,  
	    parent3
	);
	*gx_w = w;

	/* GtkHBox for the Other GtkCheckButtons */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Read GtkCheckButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "R",
"Check this to allow anyone to read from the object",
	    FALSE,
	    parent3
	);
	*or_w = w;
	/* Write GtkCheckButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "W",
"Check this to allow anyone to write to the object",
	    FALSE,
	    parent3
	);
	*ow_w = w;
	/* Execute GtkCheckButton */
	NEW_PERMISSION_TOGGLE_BUTTON(
	    "X",
"Check this to allow anyone to execute the object.\
 If the object is a directory then this allows anyone to list\
 and access its contents",
	    FALSE,  
	    parent3
	);
	*ox_w = w;

#undef NEW_PERMISSION_TOGGLE_BUTTON

	/* GtkHBox for the setuid, setgid, and sticky permission
	 * GtkCheckButtons
	 */
	w = gtk_hbox_new(TRUE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Setuid GtkCheckButton */
	*suid_w = w = gtk_check_button_new_with_label("SetUID");
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	GUISetWidgetTip(
	    w,
"Check this to set the Set User ID Bit on the object.\
 When this is set, the executed processes of the object will\
 inherit the object's owner permissions instead of the\
 executer's permissions"
	);
	gtk_widget_show(w);

	/* Setgid GtkCheckButton */
	*sgid_w = w = gtk_check_button_new_with_label("SetGID");
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	GUISetWidgetTip(
	    w,
"Check this to set the Set Group ID Bit on the object.\
 When this is set, the executed processes of the object will\
 inherit the object's group permissions instead of the\
 executer's permissions"
	);
	gtk_widget_show(w);

	/* Sticky GtkCheckButton */
	*sticky_w = w = gtk_check_button_new_with_label("Sticky");
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	GUISetWidgetTip(
	    w,
"Check this to set the Sticky Bit on the object.\
 When this is set on a directory, only owners of objects can\
 modify them. The affects of this bit on other types of objects\
 vary, see help"
	);
	gtk_widget_show(w);

}

/*
 *	Creates a new edv_obj_op_dlg_struct.
 */
edv_obj_op_dlg_struct *EDVObjOpDlgNew(EDVCore *core)
{
	const gint	border_major = 5,
			border_minor = 2;
	GdkWindow *window;
	GtkStyle *style;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4,
			*parent5, *parent6, *parent7,
			*toplevel;
	edv_obj_op_dlg_struct *d;

	if(core == NULL)
	    return(NULL);

	d = EDV_OBJ_OP_DLG(g_malloc0(sizeof(edv_obj_op_dlg_struct)));
	if(d == NULL)
	    return(NULL);

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
	d->processing = FALSE;
	d->freeze_count = 0;
	d->busy_count = 0;
	d->core = core;

	d->ref_toplevel = NULL;
	d->op = EDV_OBJ_OP_DLG_OP_MOVE;
	d->location_type = EDV_LOCATION_TYPE_VFS;
	d->objs_list = NULL;
	d->src_dir = NULL;
	d->arch_path = NULL;

	d->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	}
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVObjOpDlgDeleteEventCB), d
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	style = gtk_widget_get_style(w);
	parent = w;


	/* Main GtkVBox */
	d->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Icon & Label GtkHBox */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, border_major);
	gtk_widget_show(w);
	parent2 = w;


	/* Icon GtkVBox */
	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent3 = w;

	/* Icon GtkFixed */
	d->icon_fixed = w = gtk_fixed_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, FALSE, 0);
	gtk_widget_realize(w);
	gtk_widget_show(w);

	/* Icon GtkPixmap will be created later */
	d->icon_pm = NULL;


	/* Label & Operation-Specific Widgets GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, border_major);
	gtk_widget_show(w);
	parent3 = w;


	/* Label GtkHBox */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* GtkLabel */
	d->label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Operation-Specific GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	/* Label GtkAlignment */
	w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Label */
	w = gtk_label_new("To:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_container_add(GTK_CONTAINER(parent5), w);
	gtk_widget_show(w);


	/* Target GtkEntry */
	d->target_entry = w = gtk_entry_new();
	gtk_widget_set_usize(w, 350, -1);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "activate",
	    GTK_SIGNAL_FUNC(EDVObjOpDlgProcessCB), d
	);
	edv_entry_set_dnd(core, w);
	edv_entry_set_complete_path(core, w);
	GUIEditableEndowPopupMenu(w, 0);

	/* Target Browse GtkButton */
	d->target_browse_btn = w = GUIButtonPixmap(
	    (guint8 **)icon_folder_opened_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVObjOpDlgBrowseCB), d->target_entry
	);
	GUISetWidgetTip(
	    w,
	    "Browse"
	);

	/* Create widgets for the Change Permissions operation */

	/* Parent GtkVBox */
	d->target_chmod_parent = w = gtk_vbox_new(FALSE, border_minor);
	gtk_widget_set_usize(w, 300, -1);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	parent5 = w;

	/* Create the Change Permission widgets */
	EDVObjOpDlgCreatePermissionWidgets(
	    parent5, style,
	    &d->target_ur_check,
	    &d->target_uw_check,
	    &d->target_ux_check,
	    &d->target_gr_check,
	    &d->target_gw_check,
	    &d->target_gx_check,
	    &d->target_or_check,
	    &d->target_ow_check,
	    &d->target_ox_check,
	    &d->target_suid_check,
	    &d->target_sgid_check,
	    &d->target_sticky_check
	);


	/* Create widgets for the Change Ownership operation */

	/* Parent GtkVBox */
	d->target_chown_parent = w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	parent5 = w;

	/* Owner GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	/* Owner GtkLabel */
	w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	gtk_widget_set_usize(w, 60, -1);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent7 = w;
	w = gtk_label_new("Owner:");
	gtk_container_add(GTK_CONTAINER(parent7), w);
	gtk_widget_show(w);
	/* Owner entry and map button hbox */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent7 = w;
	/* Owner GtkEntry */
 	d->target_owner_entry = w = gtk_entry_new();
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(w, 250, -1);
	gtk_box_pack_start(GTK_BOX(parent7), w, TRUE, TRUE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "activate",
	    GTK_SIGNAL_FUNC(EDVObjOpDlgProcessCB), d
	);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);
	/* Owner Popup List Map Button */
	d->target_owner_btn = w = PUListNewMapButtonArrow(
	    GTK_ARROW_DOWN, GTK_SHADOW_OUT,
	    EDVObjOpDlgOwnerMapPUListCB, d
	);
	gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Group GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;
	/* Group GtkLabel */
	w = gtk_alignment_new(1.0f, 0.5f, 0.0f, 0.0f);
	gtk_widget_set_usize(w, 60, -1);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent7 = w;
	w = gtk_label_new("Group:");
	gtk_container_add(GTK_CONTAINER(parent7), w);
	gtk_widget_show(w);
	/* Group entry and popup list GtkHBox */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent7 = w;
	/* Group GtkEntry */
	d->target_group_entry = w = gtk_entry_new();
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(w, 250, -1);
	gtk_box_pack_start(GTK_BOX(parent7), w, TRUE, TRUE, 0);
	gtk_signal_connect(                                    
	    GTK_OBJECT(w), "activate",
	    GTK_SIGNAL_FUNC(EDVObjOpDlgProcessCB), d
	);
	GUIEditableEndowPopupMenu(w, 0);
	gtk_widget_show(w);
	/* Group Popup List Map Button */
	d->target_group_btn = w = PUListNewMapButtonArrow(
	    GTK_ARROW_DOWN, GTK_SHADOW_OUT,
	    EDVObjOpDlgGroupMapPUListCB, d
	);
	gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Create the widgets for the Change Time Stamps operation */

	/* Parent GtkVBox */
	d->target_time_parent = w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);    
	parent5 = w;

	/* Access TimePrompt */
	d->target_atime_prompt = w = time_prompt_new(
	    GTK_ORIENTATION_HORIZONTAL,
	    "Access"
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	time_prompt_allow_set_current_time(w, TRUE);
	gtk_widget_show(w);

	/* Modify TimePrompt */
	d->target_mtime_prompt = w = time_prompt_new(
	    GTK_ORIENTATION_HORIZONTAL,
	    "Modify"
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	time_prompt_allow_set_current_time(w, TRUE);
	gtk_widget_show(w);

	/* Delete Time GtkHBox */
	d->target_dtime_prompt = w = time_prompt_new(
	    GTK_ORIENTATION_HORIZONTAL,
	    "Delete"
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	time_prompt_allow_set_current_time(w, TRUE);
	gtk_widget_show(w);


	/* Create the Options widgets */

	/* Parent GtkVBox */
	d->opt_parent = w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	parent4 = w;

	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);    
	gtk_widget_show(w);
	parent5 = w;

	/* Recursive */
	d->opt_recursive_check = w = gtk_check_button_new_with_label(
	    "Recursive"
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	GTK_TOGGLE_BUTTON_SET_ACTIVE(w, FALSE);
	GUISetWidgetTip(
	    w,
"Check this to recurse into subdirectories"
	);

	/* Dereference Links */
	d->opt_dereference_links_check = w = gtk_check_button_new_with_label(
	    "Dereference Links"
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	GTK_TOGGLE_BUTTON_SET_ACTIVE(w, FALSE);
	GUISetWidgetTip(
	    w,
"Check this to operate on the links' targets instead\
 of the links themselves. When copying links, this will copy\
 the object that the link references instead of the links themselves.\
 When moving links across physical devices, this will copy\
 the object that the link references and the link (not its\
 referenced object) will be removed. The object's original\
 permissions, ownership, and time stamps will not be preserved."
	);


	/* GtkSeparator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Move GtkButton */
	d->move_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_move_file_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Mueva"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le Mouvement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Bewegung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Movimento"
#elif defined(PROG_LANGUAGE_DUTCH)
"Beweging"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mova"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Trekk"
#else
"Move"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVObjOpDlgProcessCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_m, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_m);

	/* Copy GtkButton */
	d->copy_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_copy_file_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"La Copia"
#elif defined(PROG_LANGUAGE_FRENCH)
"Copie"
#elif defined(PROG_LANGUAGE_GERMAN)
"Kopie"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Copia"
#elif defined(PROG_LANGUAGE_DUTCH)
"Kopie"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Cpia"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kopi"
#else
"Copy"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVObjOpDlgProcessCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);

	/* Link GtkButton */
	d->link_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_link_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Enlace"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le Lien"
#elif defined(PROG_LANGUAGE_GERMAN)
"Kettenglied"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Maglia"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schakel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Elo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Ledd"
#else
"Link"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVObjOpDlgProcessCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_l, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"         
	);
	GUIButtonLabelUnderline(w, GDK_l);

	/* Change Permissions GtkButton */
	d->chmod_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_chmod_20x20_xpm, "ChMod", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVObjOpDlgProcessCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_h, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_h);

	/* Change Ownership GtkButton */
	d->chown_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_owned_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Propio"
#elif defined(PROG_LANGUAGE_FRENCH)
"Propre"
#elif defined(PROG_LANGUAGE_GERMAN)
"Eigen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Proprio"
#elif defined(PROG_LANGUAGE_DUTCH)
"Eigen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prprio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Egen"
#else
"ChOwn"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVObjOpDlgProcessCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_h, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_h);

	/* Change Time Stamps GtkButton */
	d->chtime_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_time_stamp_20x20_xpm,
	    "ChTime",
	    NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVObjOpDlgProcessCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_s, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_s);

	/* Cancel GtkButton */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cancela"
#elif defined(PROG_LANGUAGE_FRENCH)
"Annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Heben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Annulla"
#elif defined(PROG_LANGUAGE_DUTCH)
"Annuleer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cancelamento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kanseller"
#else
"Cancel"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVObjOpDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"         
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);

	d->freeze_count--;

	return(d);
}


/*
 *	Clears the objects list and source directory.
 */
void EDVObjOpDlgReset(edv_obj_op_dlg_struct *d)
{
	if(d == NULL)
	    return;

	/* Delete the objects list */
	if(d->objs_list != NULL)
	{
	    switch(d->location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		g_list_foreach(
		    d->objs_list, (GFunc)edv_vfs_object_delete, NULL
		);
		break;
	      case EDV_LOCATION_TYPE_RECYCLE_BIN:
		g_list_foreach(
		    d->objs_list, (GFunc)edv_recycled_object_delete, NULL
		);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		g_list_foreach(
		    d->objs_list, (GFunc)edv_archive_object_delete, NULL
		);
		break;
	    }
	    g_list_free(d->objs_list);
	    d->objs_list = NULL;
	}

	/* Delete the source directory */
	g_free(d->src_dir);
	d->src_dir = NULL;

	/* Delete the archive path */
	g_free(d->arch_path);
	d->arch_path = NULL;
}

/*
 *	Sets the Object Operations Dialog as busy or ready.
 */
void EDVObjOpDlgSetBusy(edv_obj_op_dlg_struct *d, const gboolean busy)
{
	GdkCursor *cursor;
	GtkWidget *w;
	EDVCore *core;

	if(d == NULL)
	    return;

	w = d->toplevel;
	core = d->core;

	if(w != NULL)
	{
	    if(busy)
	    {
		/* Increase busy count */
		d->busy_count++;

		/* If already busy then don't change anything */
		if(d->busy_count > 1)
		    return;

		cursor = edv_get_cursor(core, EDV_CURSOR_CODE_BUSY);
	    }
	    else
	    {
		/* Reduce busy count */
		d->busy_count--;
		if(d->busy_count < 0)
		    d->busy_count = 0;

		/* If still busy do not change anything */
		if(d->busy_count > 0)
		    return;

		cursor = NULL;  /* Use default cursor */
	    }

	    /* Update toplevel window's cursor */
	    if(w->window != NULL)
	    {
		gdk_window_set_cursor(w->window, cursor);
		gdk_flush();
	    }
	}
}

/*
 *	Checks if the Object Operations Dialog is mapped.
 */
gboolean EDVObjOpDlgIsMapped(edv_obj_op_dlg_struct *d)
{
	if(d == NULL)
	    return(FALSE);

	return(GTK_WIDGET_MAPPED(d->toplevel));
}

/*
 *	Maps the Object Operations Dialog.
 *
 *	The d specifies the Object Operations Dialog.
 *
 *	The op specifies the operation.
 *
 *	The location_type specifies the location type.
 *
 *	The objs_list specifies the GList of objects to operate
 *	on. The data type is determined by the location_type,
 *	where:
 *
 *	EDV_LOCATION_TYPE_VFS		EDVVFSObject *
 *	EDV_LOCATION_TYPE_RECYCLE_BIN	EDVRecycledObject *
 *	EDV_LOCATION_TYPE_ARCHIVE	EDVArchiveObject *
 *
 *	This list will be coppied and will not be modified by this
 *	function.
 *
 *	The src_dir specifies the initial target location.
 *
 *	The toplevel specifies the relative toplevel GtkWindow.
 */
void EDVObjOpDlgMapValues(
	edv_obj_op_dlg_struct *d,
	const edv_obj_op_dlg_op op,
	const EDVLocationType location_type,
	GList *objs_list,
	const gchar *src_dir,
	GtkWidget *toplevel
)
{
	gboolean	need_warn_chmod_link = FALSE,
			need_warn_chtime_link = FALSE,
			map_browse_btn = FALSE;
	gint nobjs;
	const gchar *title = NULL;
	gchar *msg = NULL;
	const gchar *src_obj_full_path = NULL, *src_obj_name = NULL;
	guint8 **icon_data = NULL;
	const gchar *icon_name = NULL;
	EDVCore *core;
	GtkWidget *w, *parent, *button = NULL, *op_parent = NULL;

	if(d == NULL)
	    return;

	core = d->core;

	/* Clear the objects list and source directory */
	EDVObjOpDlgReset(d);

	/* Set the operation code */
	d->op = op;

	/* Set the location type */
	d->location_type = location_type;

	/* If there is exactly one object in the source object list
	 * then get its full path and name
	 *
	 * Otherwise set the single object full path and name to NULL
	 */
	nobjs = g_list_length(objs_list);
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    if(nobjs == 1)
	    {
		EDVVFSObject *obj = EDV_VFS_OBJECT(
		    g_list_nth_data(objs_list, 0)
		);
		if(obj != NULL)
		{
		    src_obj_name = obj->name;
		    src_obj_full_path = obj->path;
		}
	    }
	    break;
	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
	    if(nobjs == 1)
	    {
		EDVRecycledObject *obj = EDV_RECYCLED_OBJECT(
		    g_list_nth_data(objs_list, 0)
		);
		if(obj != NULL)
		{
		    src_obj_name = obj->name;
		    src_obj_full_path = obj->original_path;
		}
	    }
	    break;
	  case EDV_LOCATION_TYPE_ARCHIVE:
	    if(nobjs == 1)
	    {
		EDVArchiveObject *obj = EDV_ARCHIVE_OBJECT(
		     g_list_nth_data(objs_list, 0)
		);
		if(obj != NULL)
		{
		    src_obj_name = obj->name;
		    src_obj_full_path = obj->path;
		}
	    }
	    break;
	}

#define MAP_OPTIONS_WIDGETS(_b_,_recursive_,_dereference_links_) {	\
 if(_b_)						\
  gtk_widget_show(d->opt_parent);			\
 else							\
  gtk_widget_hide(d->opt_parent);			\
							\
 if(_recursive_)					\
  gtk_widget_show(d->opt_recursive_check);		\
 else							\
  gtk_widget_hide(d->opt_recursive_check);		\
							\
 if(_dereference_links_)				\
  gtk_widget_show(d->opt_dereference_links_check);	\
 else							\
  gtk_widget_hide(d->opt_dereference_links_check);	\
}

	/* Update the widgets based on the operation */
	switch(op)
	{
	  case EDV_OBJ_OP_DLG_OP_MOVE:
	    title =
#if defined(PROG_LANGUAGE_SPANISH)
"Mueva"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le Mouvement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Bewegung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Movimento"
#elif defined(PROG_LANGUAGE_DUTCH)
"Beweging"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mova"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Trekk"
#else
"Move"
#endif
	    ;
	    icon_data = (guint8 **)icon_move_file_32x32_xpm;
	    icon_name = "icon_move_file_32x32_xpm";
	    if(src_obj_name != NULL)
		msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Mueva:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le Mouvement:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Bewegung:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Movimento:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Beweging:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mova:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Trekk:\n\
\n\
    %s\n"
#else
"Move:\n\
\n\
    %s\n"
#endif
		    , src_obj_name
		);
	    else
		msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Mueva %i objetos\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dplacer %i objets\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Bewegen sie %i objekte\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Muovere %i oggetti\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Beweeg %i voorwerpen\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mova %i objetos\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Flytt %i objekt\n"
#else
"Move %i objects\n"
#endif
		    , nobjs
		);
	    button = d->move_btn;
	    op_parent = d->target_entry;
	    map_browse_btn = TRUE;
	    MAP_OPTIONS_WIDGETS(TRUE, FALSE, TRUE);
	    break;

	  case EDV_OBJ_OP_DLG_OP_COPY:
	    title =
#if defined(PROG_LANGUAGE_SPANISH)
"La Copia"
#elif defined(PROG_LANGUAGE_FRENCH)
"Copie"
#elif defined(PROG_LANGUAGE_GERMAN)
"Kopie"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Copia"
#elif defined(PROG_LANGUAGE_DUTCH)
"Kopie"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Cpia"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kopi"
#else
"Copy"
#endif
	    ;
	    icon_data = (guint8 **)icon_copy_file_32x32_xpm;
	    icon_name = "icon_copy_file_32x32_xpm";
	    if(src_obj_name != NULL)
		msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"La Copia:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Copie:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Kopie:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Copia:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Kopie:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Cpia:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kopi:\n\
\n\
    %s\n"
#else
"Copy:\n\
\n\
    %s\n"
#endif
		    , src_obj_name
		);
	    else
		msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Copie %i objetos\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"La copie %i objets\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Kopieren sie %i objekte\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Copiare %i oggetti\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Kopieer %i voorwerpen\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Copie %i objetos\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kopier %i objekt\n"
#else
"Copy %i objects\n"
#endif
		    , nobjs
		);
	    button = d->copy_btn;
	    op_parent = d->target_entry;
	    map_browse_btn = TRUE;
	    MAP_OPTIONS_WIDGETS(TRUE, FALSE, TRUE);
	    break;

	  case EDV_OBJ_OP_DLG_OP_LINK:
	    title =
#if defined(PROG_LANGUAGE_SPANISH)
"Enlace"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le Lien"
#elif defined(PROG_LANGUAGE_GERMAN)
"Kettenglied"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Maglia"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schakel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Elo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Ledd"
#else
"Link"
#endif
	    ;
	    icon_data = (guint8 **)icon_link_32x32_xpm;
	    icon_name = "icon_link_32x32_xpm";
	    if(src_obj_name != NULL)
		msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"El enlace:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le Lien:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Kettenglied:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Maglia:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Schakel:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Elo:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Ledd:\n\
\n\
    %s\n"
#else
"Link:\n\
\n\
    %s\n"
#endif
		    , src_obj_name
		);
	    else
		msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Trabe %i objetos\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le lien %i objets\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verknpfen sie %i objekte\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Concatenare %i oggetti\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verbind %i voorwerpen\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ligue %i objetos\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forbind %i objekt\n"
#else
"Link %i objects\n"
#endif
		    , nobjs
		);
	    button = d->link_btn;
	    op_parent = d->target_entry;
	    map_browse_btn = TRUE;
	    MAP_OPTIONS_WIDGETS(FALSE, FALSE, FALSE);
	    break;

	  case EDV_OBJ_OP_DLG_OP_CHMOD:
	    title =
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie Los Permisos"
#elif defined(PROG_LANGUAGE_FRENCH)
"Changer Des Permissions"
#elif defined(PROG_LANGUAGE_GERMAN)
"ndern Sie Erlaubnis"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cambiare I Permessi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verandeer Toestemmingen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude Permisses"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forandr Permissions"
#else
"Change Permissions"
#endif
	    ;
	    icon_data = (guint8 **)icon_chmod_32x32_xpm;
	    icon_name = "icon_chmod_32x32_xpm";
	    if(src_obj_name != NULL)
		msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie los permisos de:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Changer des permissions de:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"ndern sie erlaubnis von:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cambiare i permessi di:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verandeer toestemmingen van:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude permisses de:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forandr tillatelse av:\n\
\n\
    %s\n"
#else
"Change the permissions of:\n\
\n\
    %s\n"
#endif
		    , src_obj_name
		);
	    else
		msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie los permisos de %i objetos\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Changer des permissions de %i objets\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"ndern sie erlaubnis von %i objekten\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cambiare i permessi di %i oggetti\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verandeer toestemmingen van %i voorwerpen\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude permisses de %i objetos\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forandr tillatelse av %i objekt\n"
#else
"Change the permissions of %i objects\n"
#endif
		    , nobjs
		);
	    button = d->chmod_btn;
	    op_parent = d->target_chmod_parent;
	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		MAP_OPTIONS_WIDGETS(TRUE, TRUE, TRUE);
		break;
	      case EDV_LOCATION_TYPE_RECYCLE_BIN:
		MAP_OPTIONS_WIDGETS(FALSE, FALSE, FALSE);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		MAP_OPTIONS_WIDGETS(FALSE, FALSE, FALSE);
		break;
	    }
	    break;

	  case EDV_OBJ_OP_DLG_OP_CHOWN:
	    title =
#if defined(PROG_LANGUAGE_SPANISH)
"Propio"
#elif defined(PROG_LANGUAGE_FRENCH)
"Propre"
#elif defined(PROG_LANGUAGE_GERMAN)
"Eigen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Proprio"
#elif defined(PROG_LANGUAGE_DUTCH)
"Eigen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prprio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Egen"
#else
"Change Ownership"
#endif
	    ;
	    icon_data = (guint8 **)icon_owned_32x32_xpm;
	    icon_name = "icon_owned_32x32_xpm";
	    if(src_obj_name != NULL)
		msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie la propiedad de:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Changer la possession de:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"ndern sie eigentumsrecht von:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cambiare la propriet di:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verandeer eigendomsrecht van:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude posse de:\n\
\n\
    %s\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forandr eiendomsrett av:\n\
\n\
    %s\n"
#else
"Change the ownership of:\n\
\n\
    %s\n"
#endif
		    , src_obj_name
		);
	    else
		msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie la propiedad de %i objetos\n"
#elif defined(PROG_LANGUAGE_FRENCH)
"Changer la possession de %i objetos\n"
#elif defined(PROG_LANGUAGE_GERMAN)
"ndern sie eigentumsrecht von %i objetos\n"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Cambiare la propriet di %i objetos\n"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verandeer eigendomsrecht van %i objetos\n"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude posse de %i objetos\n"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forandr eiendomsrett av %i objetos\n"
#else
"Change the ownership of %i objects\n"
#endif
		    , nobjs
		);
	    button = d->chown_btn;
	    op_parent = d->target_chown_parent;
	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		MAP_OPTIONS_WIDGETS(TRUE, TRUE, TRUE);
		break;
	      case EDV_LOCATION_TYPE_RECYCLE_BIN:
		MAP_OPTIONS_WIDGETS(FALSE, FALSE, FALSE);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		MAP_OPTIONS_WIDGETS(FALSE, FALSE, FALSE);
		break;
	    }
	    break;

	  case EDV_OBJ_OP_DLG_OP_CHTIME:
	    title = "Change Time Stamps";
	    icon_data = (guint8 **)icon_time_stamp_32x32_xpm;
	    icon_name = "icon_time_stamp_32x32_xpm";
	    if(src_obj_name != NULL)
		msg = g_strdup_printf(
"Change the time stamps of:\n\
\n\
    %s\n\
\n",
		    src_obj_name
		);
	    else
		msg = g_strdup_printf(
"Change the time stamps of %i objects\n"
		    , nobjs
		);
	    button = d->chtime_btn;
	    op_parent = d->target_time_parent;
	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		gtk_widget_show(d->target_atime_prompt);
		gtk_widget_show(d->target_mtime_prompt);
		gtk_widget_hide(d->target_dtime_prompt);
		MAP_OPTIONS_WIDGETS(TRUE, TRUE, TRUE);
		break;
	      case EDV_LOCATION_TYPE_RECYCLE_BIN:
		gtk_widget_show(d->target_atime_prompt);
		gtk_widget_show(d->target_mtime_prompt);
		gtk_widget_show(d->target_dtime_prompt);
		MAP_OPTIONS_WIDGETS(FALSE, FALSE, FALSE);
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		gtk_widget_show(d->target_atime_prompt);
		gtk_widget_show(d->target_mtime_prompt);
		gtk_widget_hide(d->target_dtime_prompt);
		MAP_OPTIONS_WIDGETS(FALSE, FALSE, FALSE);
		break;
	    }
	    break;
	}
#undef MAP_OPTIONS_WIDGETS

	/* Update the icon */
	parent = d->icon_fixed;
	if((icon_data != NULL) && (parent != NULL))
	{
	    /* Load the EDVPixmap from the icon data */
	    EDVPixmap *icon = edv_load_pixmap_from_data(
		core,
		icon_data,
		icon_name
	    );
	    if(edv_pixmap_is_loaded(icon))
	    {
		const gint	width = icon->width,
				height = icon->height;

		/* Create/set the icon GtkPixmap */
		w = d->icon_pm;
		if(w != NULL)
		{
		    edv_pixmap_set_gtk_pixmap(icon, w);
		}
		else
		{
		    d->icon_pm = w = edv_pixmap_new_gtk_pixmap(icon);
		    if(w != NULL)
		    {
			gtk_fixed_put(GTK_FIXED(parent), w, 0, 0);
			gtk_widget_show(w);
		    }
		}

		/* Adjust size of fixed widget to fit pixmap */
		gtk_widget_set_usize(parent, width, height);
		gtk_widget_queue_resize(parent);
		gtk_widget_shape_combine_mask(
		    parent,
		    icon->mask,
		    0, 0
		);

	    }
	    (void)edv_pixmap_unref(icon);

	    /* Update the toplevel GdkWindow's WM icon */
	    w = d->toplevel;
	    if(w != NULL)
		GUISetWMIcon(w->window, (guint8 **)icon_data);
	}

	/* Update the label */
	if(msg != NULL)
	{
	    w = d->label;
	    if(w != NULL)
		gtk_label_set_text(GTK_LABEL(w), msg);

	    g_free(msg);
	    msg = NULL;
	}


	/* Set the source directory */
	g_free(d->src_dir);
	d->src_dir = STRDUP(src_dir);

	/* Set the target GtkEntry's value if it is the operation
	 * widget?
	 */
	w = d->target_entry;
	if((w != NULL) && (w == op_parent))
	{
	    gtk_entry_set_text(
		GTK_ENTRY(w),
		(src_dir != NULL) ? src_dir : ""
	    );
	    if(GTK_WIDGET_CAN_FOCUS(w))
		gtk_widget_grab_focus(w);
	    if(GTK_WIDGET_CAN_DEFAULT(w))
		gtk_widget_grab_default(w);
	}

	/* Copy the objects list */
	switch(location_type)
	{
	  case EDV_LOCATION_TYPE_VFS:
	    if(objs_list != NULL)
	    {
		GList *glist;
		EDVVFSObject *obj;

		for(glist = objs_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    obj = EDV_VFS_OBJECT(glist->data);
		    if(obj == NULL)
			continue;

		    obj = edv_vfs_object_copy(obj);
		    if(obj == NULL)
			continue;

		    d->objs_list = g_list_append(d->objs_list, obj);
		}
	    }
	    break;

	  case EDV_LOCATION_TYPE_RECYCLE_BIN:
	    if(objs_list != NULL)
	    {
		GList *glist;
		EDVRecycledObject *obj;

		for(glist = objs_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    obj = EDV_RECYCLED_OBJECT(glist->data);
		    if(obj == NULL)
			continue;

		    obj = edv_recycled_object_copy(obj);
		    if(obj == NULL)
			continue;

		    d->objs_list = g_list_append(d->objs_list, obj);
		}
	    }
	    break;

	  case EDV_LOCATION_TYPE_ARCHIVE:
	    if(objs_list != NULL)
	    {
		GList *glist;
		EDVArchiveObject *obj;

		for(glist = objs_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    obj = EDV_ARCHIVE_OBJECT(glist->data);
		    if(obj == NULL)
			continue;

		    obj = edv_archive_object_copy(obj);
		    if(obj == NULL)
			continue;

		    d->objs_list = g_list_append(d->objs_list, obj);
		}
	    }
	    break;
	}

	/* Is the operation to change permissions? */
	w = d->target_chmod_parent;
	if((w != NULL) && (w == op_parent))
	{
	    /* Set the initial values for the change permission
	     * widgets
	     */
	    EDVPermissionFlags permissions = 0x00000000;

	    /* Iterate through the objects list and collect the
	     * permission values that are common to all of them
	     */
	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    GList *glist;
		    EDVVFSObject *ref_obj;

		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			ref_obj = EDV_VFS_OBJECT(glist->data);
			if(ref_obj == NULL)
			    continue;

			/* Do a quick check to see if the object to change
			 * permissions of is a symbolic link
			 *
			 * If this is a symbolic link then a warning needs to
			 * be issued about changing a symbolic link's
			 * permissions that will actually change its
			 * target permissions
			 */
			if(ref_obj->type == EDV_OBJECT_TYPE_LINK)
			    need_warn_chmod_link = TRUE;

			/* Gather the permissions from this object */
			permissions |= ref_obj->permissions;
		    }
		}
		break;
	      case EDV_LOCATION_TYPE_RECYCLE_BIN:
		if(objs_list != NULL)
		{
		    GList *glist;
		    EDVRecycledObject *ref_obj;

		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			ref_obj = EDV_RECYCLED_OBJECT(glist->data);
			if(ref_obj == NULL)
			    continue;

			/* Do a quick check to see if the object to change
			 * permissions of is a symbolic link
			 *
			 * If this is a symbolic link then a warning needs to
			 * be issued about changing a symbolic link's
			 * permissions that will actually change its
			 * target permissions
			 */
			if(ref_obj->type == EDV_OBJECT_TYPE_LINK)
			    need_warn_chmod_link = TRUE;

			/* Gather the permissions from this object */
			permissions |= ref_obj->permissions;
		    }
		}
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		if(objs_list != NULL)
		{
		    GList *glist;
		    EDVArchiveObject *ref_obj;

		    for(glist = objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			ref_obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(ref_obj == NULL)
			    continue;

			/* Do a quick check to see if the object to change
			 * permissions of is a symbolic link
			 *
			 * If this is a symbolic link then a warning needs to
			 * be issued about changing a symbolic link's
			 * permissions that will actually change its
			 * target permissions
			 */
			if(ref_obj->type == EDV_OBJECT_TYPE_LINK)
			    need_warn_chmod_link = TRUE;

			/* Gather the permissions from this object */
			permissions |= ref_obj->permissions;
		    }
		}
		break;
	    }

	    /* Set the initial values for the change permission
	     * widgets based on the collected permission values
	     */

	    /* User RWX */
	    GTK_TOGGLE_BUTTON_SET_ACTIVE(
		d->target_ur_check,
		(permissions & EDV_PERMISSION_UR) ? TRUE : FALSE
	    );
	    if(GTK_WIDGET_CAN_FOCUS(d->target_ur_check))
		gtk_widget_grab_focus(d->target_ur_check);
	    if(GTK_WIDGET_CAN_DEFAULT(d->target_ur_check))
		gtk_widget_grab_default(d->target_ur_check);
	    GTK_TOGGLE_BUTTON_SET_ACTIVE(
		d->target_uw_check,
		(permissions & EDV_PERMISSION_UW) ? TRUE : FALSE
	    );
	    GTK_TOGGLE_BUTTON_SET_ACTIVE(
		d->target_ux_check,
		(permissions & EDV_PERMISSION_UX) ? TRUE : FALSE
	    );

	    /* Group RWX */
	    GTK_TOGGLE_BUTTON_SET_ACTIVE(
		d->target_gr_check,
		(permissions & EDV_PERMISSION_GR) ? TRUE : FALSE
	    );
	    GTK_TOGGLE_BUTTON_SET_ACTIVE(
		d->target_gw_check,
		(permissions & EDV_PERMISSION_GW) ? TRUE : FALSE
	    );
	    GTK_TOGGLE_BUTTON_SET_ACTIVE(
		d->target_gx_check,
		(permissions & EDV_PERMISSION_GX) ? TRUE : FALSE
	    );

	    /* Other RWX */
	    GTK_TOGGLE_BUTTON_SET_ACTIVE(
		d->target_or_check,
		(permissions & EDV_PERMISSION_OR) ? TRUE : FALSE
	    );
	    GTK_TOGGLE_BUTTON_SET_ACTIVE(
		d->target_ow_check,
		(permissions & EDV_PERMISSION_OW) ? TRUE : FALSE
	    );
	    GTK_TOGGLE_BUTTON_SET_ACTIVE(
		d->target_ox_check,
		(permissions & EDV_PERMISSION_OX) ? TRUE : FALSE
	    );

	    /* Setuid, setgid, and sticky */
	    GTK_TOGGLE_BUTTON_SET_ACTIVE(
		d->target_suid_check,
		(permissions & EDV_PERMISSION_SETUID) ? TRUE : FALSE
	    );
	    GTK_TOGGLE_BUTTON_SET_ACTIVE(
		d->target_sgid_check,
		(permissions & EDV_PERMISSION_SETGID) ? TRUE : FALSE
	    );
	    GTK_TOGGLE_BUTTON_SET_ACTIVE(
		d->target_sticky_check,
		(permissions & EDV_PERMISSION_STICKY) ? TRUE : FALSE
	    );
	}

	/* Is the operation to change ownership? */
	w = d->target_chown_parent;
	if((w != NULL) && (w == op_parent))
	{
	    /* Set the initial values for the owner and group widgets */
	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    EDVVFSObject *ref_obj = EDV_VFS_OBJECT(
			g_list_nth_data(objs_list, (guint)(nobjs - 1))
		    );

		    /* Owner */
		    w = d->target_owner_entry;
		    if(w != NULL)
		    {
			gchar *owner = edv_uid_uid_to_name(
	                    core->uids_list,
	                    (ref_obj != NULL) ?
	                        ref_obj->owner_id : core->effective_user_id,
	                    NULL
	                );
			gtk_entry_set_text(GTK_ENTRY(w), (owner != NULL) ? owner : "");
			g_free(owner);
			if(GTK_WIDGET_CAN_FOCUS(w))
			    gtk_widget_grab_focus(w);
			if(GTK_WIDGET_CAN_DEFAULT(w))
			    gtk_widget_grab_default(w);
		    }
		    /* Group */
		    w = d->target_group_entry;
		    if(w != NULL)
		    {
			gchar *group = edv_gid_gid_to_name(
	                    core->gids_list,
	                    (ref_obj != NULL) ?
	                        ref_obj->group_id : core->effective_group_id,
			    NULL
			);
			gtk_entry_set_text(GTK_ENTRY(w), (group != NULL) ? group : "");
			g_free(group);
		    }
		}
		break;
	      case EDV_LOCATION_TYPE_RECYCLE_BIN:
		if(objs_list != NULL)
		{
		    EDVRecycledObject *ref_obj = EDV_RECYCLED_OBJECT(
			g_list_nth_data(objs_list, (guint)(nobjs - 1))
		    );

		    /* Owner */
		    w = d->target_owner_entry;
		    if(w != NULL)
		    {
			gchar *owner = edv_uid_uid_to_name(
	                    core->uids_list,
	                    (ref_obj != NULL) ?
	                        ref_obj->owner_id : core->effective_user_id,
	                    NULL
	                );
			gtk_entry_set_text(GTK_ENTRY(w), (owner != NULL) ? owner : "");
			g_free(owner);
			if(GTK_WIDGET_CAN_FOCUS(w))
			    gtk_widget_grab_focus(w);
			if(GTK_WIDGET_CAN_DEFAULT(w))
			    gtk_widget_grab_default(w);
		    }
		    /* Group */
		    w = d->target_group_entry;
		    if(w != NULL)
		    {
			gchar *group = edv_gid_gid_to_name(
	                    core->gids_list,
	                    (ref_obj != NULL) ?
	                        ref_obj->group_id : core->effective_group_id,
			    NULL
			);
			gtk_entry_set_text(GTK_ENTRY(w), (group != NULL) ? group : "");
			g_free(group);
		    }
		}
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		if(objs_list != NULL)
		{
		    EDVArchiveObject *ref_obj = EDV_ARCHIVE_OBJECT(
			g_list_nth_data(objs_list, (guint)(nobjs - 1))
		    );

		    /* Owner */
		    w = d->target_owner_entry;
		    if(w != NULL)
		    {
			gchar *owner = (ref_obj != NULL) ?
			    STRDUP(ref_obj->owner_name) : NULL;
			gtk_entry_set_text(GTK_ENTRY(w), (owner != NULL) ? owner : "");
			g_free(owner);
			if(GTK_WIDGET_CAN_FOCUS(w))
			    gtk_widget_grab_focus(w);
			if(GTK_WIDGET_CAN_DEFAULT(w))
			    gtk_widget_grab_default(w);
		    }
		    /* Group */
		    w = d->target_group_entry;
		    if(w != NULL)
		    {
			gchar *group = (ref_obj != NULL) ?
			    STRDUP(ref_obj->group_name) : NULL;
			gtk_entry_set_text(GTK_ENTRY(w), (group != NULL) ? group : "");
			g_free(group);
		    }
		}
		break;
	    }
	}

	/* Is the operation to change time stamps? */
	w = d->target_time_parent;
	if((w != NULL) && (w == op_parent))
	{
	    /* Set the initial time stamp widget values to the time
	     * stamps of the specified object
	     */
	    gulong	access_time = 0l,
			modify_time = 0l,
			change_time = 0l,
			delete_time = 0l;

	    switch(location_type)
	    {
	      case EDV_LOCATION_TYPE_VFS:
		if(objs_list != NULL)
		{
		    GList *glist;
		    EDVVFSObject *ref_obj = EDV_VFS_OBJECT(
			g_list_nth_data(objs_list, (guint)(nobjs - 1))
		    );
		    if(ref_obj != NULL)
		    {
			access_time = ref_obj->access_time;
			modify_time = ref_obj->modify_time;
			change_time = ref_obj->change_time;
		    }
		    /* Check if we need to warn about links */
		    for(glist = d->objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			ref_obj = EDV_VFS_OBJECT(glist->data);
			if(ref_obj == NULL)
			    continue;

			if(ref_obj->type == EDV_OBJECT_TYPE_LINK)
			    need_warn_chtime_link = TRUE;
		    }
		}
		break;
	      case EDV_LOCATION_TYPE_RECYCLE_BIN:
		if(objs_list != NULL)
		{
		    GList *glist;
		    EDVRecycledObject *ref_obj = EDV_RECYCLED_OBJECT(
			g_list_nth_data(objs_list, (guint)(nobjs - 1))
		    );
		    if(ref_obj != NULL)
		    {
			access_time = ref_obj->access_time;
			modify_time = ref_obj->modify_time;
			change_time = ref_obj->change_time;
			delete_time = ref_obj->deleted_time;
		    }
		    /* Check if we need to warn about links */
		    for(glist = d->objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			ref_obj = EDV_RECYCLED_OBJECT(glist->data);
			if(ref_obj == NULL)
			    continue;

			if(ref_obj->type == EDV_OBJECT_TYPE_LINK)
			    need_warn_chtime_link = TRUE;
		    }
		}
		break;
	      case EDV_LOCATION_TYPE_ARCHIVE:
		if(objs_list != NULL)
		{
		    GList *glist;
		    EDVArchiveObject *ref_obj = EDV_ARCHIVE_OBJECT(
			g_list_nth_data(objs_list, (guint)(nobjs - 1))
		    );
		    if(ref_obj != NULL)
		    {
			access_time = ref_obj->access_time;
			modify_time = ref_obj->modify_time;
			change_time = ref_obj->change_time;
		    }
		    /* Check if we need to warn about links */
		    for(glist = d->objs_list;
			glist != NULL;
			glist = g_list_next(glist)
		    )
		    {
			ref_obj = EDV_ARCHIVE_OBJECT(glist->data);
			if(ref_obj == NULL)
			    continue;

			if(ref_obj->type == EDV_OBJECT_TYPE_LINK)
			    need_warn_chtime_link = TRUE;
		    }
		}
		break;
	    }

	    /* Set the time stamp widget values to the time stamps
	     * of the reference object
	     *
	     * Access time
	     */
	    if(access_time > 0l)
		time_prompt_set(
		    d->target_atime_prompt,
		    access_time
		);
	    /* Modify Time */
	    if(modify_time > 0l)
		time_prompt_set(
		    d->target_mtime_prompt,
		    modify_time
		);
	    /* Delete Time */
	    if(delete_time > 0l)
		time_prompt_set(
		    d->target_dtime_prompt,
		    delete_time
		);
	}



	/* Unmap all the buttons */
	w = d->move_btn;
	if(w != NULL)
	    gtk_widget_hide(w);
	w = d->copy_btn;
	if(w != NULL)
	    gtk_widget_hide(w);
	w = d->link_btn;
	if(w != NULL)
	    gtk_widget_hide(w);
	w = d->chmod_btn;
	if(w != NULL)
	    gtk_widget_hide(w);
	w = d->chown_btn;
	if(w != NULL)
	    gtk_widget_hide(w);
	w = d->chtime_btn;
	if(w != NULL)
	    gtk_widget_hide(w);

	/* Check if button to be mapped is valid and map it */
	if(button != NULL)
	    gtk_widget_show(button);


	/* Unmap all the target widgets */
	w = d->target_entry;
	if(w != NULL)
	    gtk_widget_hide(w);
	w = d->target_chmod_parent;
	if(w != NULL)
	    gtk_widget_hide(w);
	w = d->target_chown_parent;
	if(w != NULL)
	    gtk_widget_hide(w);
	w = d->target_time_parent;
	if(w != NULL)
	    gtk_widget_hide(w);

	/* Check if the target widget to be mapped is valid and map it */
	if(op_parent != NULL)
	    gtk_widget_show(op_parent);

	/* Map or unmap the browse button */
	w = d->target_browse_btn;
	if(w != NULL)
	{
	    if(map_browse_btn)
		gtk_widget_show(w);
	    else
		gtk_widget_hide(w);
	}

	/* Need to print warning about changing permissions on link
	 * objects?
	 */
	if(need_warn_chmod_link)
	{
	    edv_play_sound_warning(core);
	    edv_message_warning(
"Changing Permissions Warning",
"Changing the permissions on a link will effectively\n\
change the permissions of its target object.",
"One or more of the selected object(s) are of type\n\
link. Links do not have permissions, instead, their\n\
permissions are determined by the permissions of the\n\
target object. So changing the permissions on a link\n\
will effectively change the permissions of the target\n\
object.",
		toplevel
	    );
	}
	if(need_warn_chtime_link)
	{
	    edv_play_sound_warning(core);
	    edv_message_warning(
"Changing Time Stamps Warning",
"Changing the time stamps on a link will effectively\n\
change the time stamps of its target object.",
"One or more of the selected object(s) are of type\n\
link. Links do not have time stamps, instead, their\n\
time stamps are determined by the time stamps of the\n\
target object. So changing the time stamps on a link\n\
will effectively change the time stamps of the target\n\
object.",
		toplevel
	    );
	}


	w = d->toplevel;
	if(w != NULL)
	{
	    /* Update title? */
	    if(title != NULL)
		gtk_window_set_title(GTK_WINDOW(w), title);

	    /* If given toplevel is not NULL then set transient for */
	    if((toplevel != NULL) ? GTK_IS_WINDOW(toplevel) : FALSE)
	    {
		gtk_window_set_transient_for(
		    GTK_WINDOW(w), GTK_WINDOW(toplevel)
		);
		d->ref_toplevel = toplevel;
	    }

	    /* Map the object operations dialog */
	    gtk_widget_show_raise(w);
	}
}

/*
 *	Unmaps the Object Operations Dialog.
 */
void EDVObjOpDlgUnmap(edv_obj_op_dlg_struct *d)
{
	GtkWidget *w;

	if(d == NULL)
	    return;

	w = d->toplevel;
	gtk_widget_hide(w);
	gtk_window_set_transient_for(GTK_WINDOW(w), NULL);
	d->ref_toplevel = NULL;
}

/*
 *	Deletes the Object Operations Dialog.
 */
void EDVObjOpDlgDelete(edv_obj_op_dlg_struct *d)
{
	if(d == NULL)
	    return;

	EDVObjOpDlgUnmap(d);

	/* Delete the object list and the source directory */
	EDVObjOpDlgReset(d);

	d->ref_toplevel = NULL;		/* Shared */

	d->freeze_count++;

#if 0
	GTK_WIDGET_DESTROY(d->icon_pm);
	GTK_WIDGET_DESTROY(d->icon_fixed);

	GTK_WIDGET_DESTROY(d->target_entry);
	GTK_WIDGET_DESTROY(d->target_browse_btn);

	GTK_WIDGET_DESTROY(d->target_ur_check);
	GTK_WIDGET_DESTROY(d->target_uw_check);
	GTK_WIDGET_DESTROY(d->target_ux_check);
	GTK_WIDGET_DESTROY(d->target_gr_check);
	GTK_WIDGET_DESTROY(d->target_gw_check);
	GTK_WIDGET_DESTROY(d->target_gx_check);
	GTK_WIDGET_DESTROY(d->target_or_check);
	GTK_WIDGET_DESTROY(d->target_ow_check);
	GTK_WIDGET_DESTROY(d->target_ox_check);
	GTK_WIDGET_DESTROY(d->target_suid_check);
	GTK_WIDGET_DESTROY(d->target_sgid_check);
	GTK_WIDGET_DESTROY(d->target_sticky_check);
	GTK_WIDGET_DESTROY(d->target_chmod_parent);

	GTK_WIDGET_DESTROY(d->target_owner_entry);
	GTK_WIDGET_DESTROY(d->target_owner_btn);
	GTK_WIDGET_DESTROY(d->target_group_entry);
	GTK_WIDGET_DESTROY(d->target_group_btn);
	GTK_WIDGET_DESTROY(d->target_chown_parent);

	GTK_WIDGET_DESTROY(d->target_atime_year_spin);
	GTK_WIDGET_DESTROY(d->target_atime_month_pulistbox);
	GTK_WIDGET_DESTROY(d->target_atime_day_spin);
	GTK_WIDGET_DESTROY(d->target_atime_hours_spin);
	GTK_WIDGET_DESTROY(d->target_atime_minutes_spin);
	GTK_WIDGET_DESTROY(d->target_atime_seconds_spin);
	GTK_WIDGET_DESTROY(d->target_atime_current_btn);
	GTK_WIDGET_DESTROY(d->target_mtime_year_spin);
	GTK_WIDGET_DESTROY(d->target_mtime_month_pulistbox);
	GTK_WIDGET_DESTROY(d->target_mtime_day_spin);
	GTK_WIDGET_DESTROY(d->target_mtime_hours_spin);
	GTK_WIDGET_DESTROY(d->target_mtime_minutes_spin);
	GTK_WIDGET_DESTROY(d->target_mtime_seconds_spin);
	GTK_WIDGET_DESTROY(d->target_mtime_current_btn);
	GTK_WIDGET_DESTROY(d->target_dtime_month_pulistbox);
	GTK_WIDGET_DESTROY(d->target_dtime_day_spin);
	GTK_WIDGET_DESTROY(d->target_dtime_hours_spin);
	GTK_WIDGET_DESTROY(d->target_dtime_minutes_spin);
	GTK_WIDGET_DESTROY(d->target_dtime_seconds_spin);
	GTK_WIDGET_DESTROY(d->target_dtime_current_btn);
	GTK_WIDGET_DESTROY(d->target_time_parent);

	GTK_WIDGET_DESTROY(d->opt_recursive_check);
	GTK_WIDGET_DESTROY(d->opt_dereference_links_check);
	GTK_WIDGET_DESTROY(d->opt_parent);

	GTK_WIDGET_DESTROY(d->move_btn);
	GTK_WIDGET_DESTROY(d->copy_btn);
	GTK_WIDGET_DESTROY(d->link_btn);
	GTK_WIDGET_DESTROY(d->chmod_btn);
	GTK_WIDGET_DESTROY(d->chown_btn);
	GTK_WIDGET_DESTROY(d->chtime_btn);
	GTK_WIDGET_DESTROY(d->cancel_btn);
#endif
	gtk_widget_destroy(d->toplevel);
	gtk_accel_group_unref(d->accelgrp);

	d->freeze_count--;

	g_free(d);
}
