#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "../include/string.h"

#include "cfg.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fb.h"
#include "progressdialog.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_date_format.h"
#include "edv_pixmap.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_utils_gtk.h"
#include "edv_list_cb.h"
#include "mime_type_import_options_dlg.h"
#include "mime_type_edit_dlg.h"
#include "mime_types_list_win.h"
#include "edv_cb.h"
#include "edv_emit.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_search_20x20.xpm"
#include "images/icon_add_20x20.xpm"
#include "images/icon_edit_20x20.xpm"
#include "images/icon_remove_20x20.xpm"
#include "images/icon_save_20x20.xpm"
#include "images/icon_save_32x32.xpm"
#include "images/icon_import_20x20.xpm"
#include "images/icon_import_32x32.xpm"
#include "images/icon_export_20x20.xpm"
#include "images/icon_export_32x32.xpm"
#include "images/icon_close_20x20.xpm"

#include "images/icon_mimetypes_32x32.xpm"
#include "images/icon_mimetypes_48x48.xpm"

#include "images/icon_system_object_20x20.xpm"
#include "images/icon_file_extension_20x20.xpm"
#include "images/icon_executable_20x20.xpm"
#include "images/icon_unique_object_20x20.xpm"


/* Callbacks */
static gint EDVMimeTypesListProgressCB(
	gpointer data, const gulong pos, const gulong max
);

static gint EDVMimeTypesListWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static gint EDVMimeTypesListWinKeyEventCB(
	GtkWidget *widget, GdkEventKey *key, gpointer data
);
static gint EDVMimeTypesListWinButtonPressEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
);
static void EDVMimeTypesListWinDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void EDVMimeTypesListWinDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void EDVMimeTypesListWinDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);
static void EDVMimeTypesListWinSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void EDVMimeTypesListWinUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void EDVMimeTypesListWinFindCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypesListWinAddCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypesListWinEditCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypesListWinRemoveCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypesListWinShiftUpCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypesListWinShiftDownCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypesListWinSaveCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypesListWinImportCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypesListWinExportCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypesListWinCloseCB(
	GtkWidget *widget, gpointer data
);

void EDVMimeTypesListWinReconfiguredNotifyCB(
	edv_mime_types_list_win_struct *lw
);
void EDVMimeTypesListWinMimeTypeAddedCB(
	edv_mime_types_list_win_struct *lw,
	const gint mt_num, EDVMIMEType *m
);
void EDVMimeTypesListWinMimeTypeModifiedCB(
	edv_mime_types_list_win_struct *lw,
	const gint mt_num, EDVMIMEType *m
);
void EDVMimeTypesListWinMimeTypeRemovedCB(
	edv_mime_types_list_win_struct *lw,
	const gint mt_num
);

/* Utilities */
void EDVMimeTypesListWinSelect(
	edv_mime_types_list_win_struct *lw,
	const gint i
);
gint EDVMimeTypesListWinSelectByType(
	edv_mime_types_list_win_struct *lw,
	const gchar *type
);

/* Display */
static void EDVMimeTypesListWinUpdateDisplay(
	edv_mime_types_list_win_struct *lw,
	const gint mt_num, EDVMIMEType *m
);

/* List & Items */
static void EDVMimeTypesListWinSetRow(
	edv_mime_types_list_win_struct *lw,
	const gint row,
	EDVMIMEType *m
);
void EDVMimeTypesListWinGetList(
	edv_mime_types_list_win_struct *lw,
	const gboolean verbose
);

/* Operations */
void EDVMimeTypesListWinFind(edv_mime_types_list_win_struct *lw);
void EDVMimeTypesListWinAdd(edv_mime_types_list_win_struct *lw);
void EDVMimeTypesListWinEditMimeType(
	edv_mime_types_list_win_struct *lw,
	const gint mt_num
);
void EDVMimeTypesListWinEdit(edv_mime_types_list_win_struct *lw);
void EDVMimeTypesListWinRemove(
	edv_mime_types_list_win_struct *lw,
	const gboolean confirm
);
void EDVMimeTypesListWinShiftUp(edv_mime_types_list_win_struct *lw);
void EDVMimeTypesListWinShiftDown(edv_mime_types_list_win_struct *lw);
void EDVMimeTypesListWinSave(edv_mime_types_list_win_struct *lw);
void EDVMimeTypesListWinImport(edv_mime_types_list_win_struct *lw);
void EDVMimeTypesListWinExport(edv_mime_types_list_win_struct *lw);
void EDVMimeTypesListWinClose(edv_mime_types_list_win_struct *lw);

/* MIME Types List */
edv_mime_types_list_win_struct *EDVMimeTypesListWinNew(EDVCore *core);
void EDVMimeTypesListWinUpdateWidgets(edv_mime_types_list_win_struct *lw);
void EDVMimeTypesListWinSetBusy(
	edv_mime_types_list_win_struct *lw,
	const gboolean busy
);
gboolean EDVMimeTypesListWinIsMapped(edv_mime_types_list_win_struct *lw);
void EDVMimeTypesListWinMap(edv_mime_types_list_win_struct *lw);
void EDVMimeTypesListWinUnmap(edv_mime_types_list_win_struct *lw);
void EDVMimeTypesListWinDelete(edv_mime_types_list_win_struct *lw);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


#define EDV_MIME_TYPES_LIST_WIN_WIDTH	500
#define EDV_MIME_TYPES_LIST_WIN_HEIGHT	480

#define EDV_MIME_TYPES_LIST_WIN_TITLE	"MIME Types"


#define EDV_MIME_TYPES_FORMAT_DESC_MAILCAP		\
 "Mailcap"
#define EDV_MIME_TYPES_FORMAT_DESC_MEDIA_TYPES		\
 "Media Types"
#define EDV_MIME_TYPES_FORMAT_DESC_ENDEAVOUR		\
 "Endeavour MIME Types"


/*
 *	MIME Types List Window Progress callback.
 *
 *	For open, save, import, and export progress.
 */
static gint EDVMimeTypesListProgressCB(
	gpointer data, const gulong pos, const gulong max
)
{
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if(lw == NULL)
		return(1);

	if(ProgressDialogIsQuery())
	{
		if(ProgressDialogStopCount() > 0)
			return(1);

		if(max > 0l)
			ProgressDialogUpdate(
				NULL, NULL, NULL, NULL,
				(gfloat)pos / (gfloat)max,
				EDV_PROGRESS_BAR_NTICKS, TRUE
			);
		else
			ProgressDialogUpdateUnknown(
				NULL, NULL, NULL, NULL,
				TRUE
			);
	}

	return(0);
}


/*
 *	MIME Types List Window GtkWindow "delete_event" signal callback.
 */
static gint EDVMimeTypesListWinDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if(lw == NULL)
		return(TRUE);

	if(lw->freeze_count > 0)
		return(TRUE);

	EDVMimeTypesListWinClose(lw);

	return(TRUE);
}

/*
 *	MIME Types List Window "key_press_event" or "key_release_event"
 *	signal callback.
 */
static gint EDVMimeTypesListWinKeyEventCB(
	GtkWidget *widget, GdkEventKey *key, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	guint keyval, state;
	gboolean press;
	EDVCore *core;
	GtkCList *clist;
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if((widget == NULL) || (key == NULL) || (lw == NULL))
		return(status);

	if(lw->freeze_count > 0)
		return(status);

	clist = GTK_CLIST(lw->mimetypes_clist);
	core = lw->core;

	etype = key->type;
	press = (etype == GDK_KEY_PRESS) ? TRUE : FALSE;
	keyval = key->keyval;
	state = key->state;

/* Stop emit of signal */
#define DO_STOP_KEY_SIGNAL_EMIT	{		\
 gtk_signal_emit_stop_by_name(			\
  GTK_OBJECT(widget),				\
  press ?					\
   "key_press_event" : "key_release_event"	\
 );						\
}

	/* Begin checking to see which widget this event is for */

	/* MIME Types clist */
	if(widget == lw->mimetypes_clist)
	{
		/* Get the last selected row */
		gint row = edv_clist_get_selected_last(clist, NULL);

		/* Handle by key value */
		switch(keyval)
		{
		  case GDK_Return:
		  case GDK_KP_Enter:
		  case GDK_ISO_Enter:
		  case GDK_3270_Enter:
			if(press)
			{
				EDVMimeTypesListWinEdit(lw);
			}
			DO_STOP_KEY_SIGNAL_EMIT
			status = TRUE;
			break;

		  case GDK_Delete:
			if(press)
			{
				EDVMimeTypesListWinRemove(
					lw,
					TRUE			/* Confirm */
				);
			}
			DO_STOP_KEY_SIGNAL_EMIT
			status = TRUE;
			break;

		  case GDK_space:
		  case GDK_KP_Space:
			row = clist->focus_row;
			if((row >= 0) && (row < clist->rows) && press)
			{
				gboolean already_selected = FALSE;

				/* Check if this row is already selected */
				GList *glist = clist->selection;
				while(glist != NULL)
				{
					if(row == (gint)glist->data)
					{
						already_selected = TRUE;
						break;
					}
					glist = g_list_next(glist);
				}

				gtk_clist_freeze(clist);
				if(already_selected)
					gtk_clist_unselect_row(clist, row, 0);
				else
					gtk_clist_select_row(clist, row, 0);
				gtk_clist_thaw(clist);
			}
			DO_STOP_KEY_SIGNAL_EMIT
			status = TRUE;
			break;
		}
	}

	return(status);
#undef DO_STOP_KEY_SIGNAL_EMIT
}

/*
 *	MIME Types List Window "button_press_event" or 
 *	"button_release_event" signal callback.
 */
static gint EDVMimeTypesListWinButtonPressEventCB(
	GtkWidget *widget, GdkEventButton *button, gpointer data
)
{
	gint status = FALSE;
	gint etype;
	GtkCList *clist;
	CfgItem *cfg_list;
	EDVCore *core;
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if((widget == NULL) || (button == NULL) || (lw == NULL))
		return(status);

	if(lw->freeze_count > 0)
		return(status);

	clist = GTK_CLIST(lw->mimetypes_clist);
	core = lw->core;
	cfg_list = core->cfg_list;
	etype = button->type;

	if(widget == lw->mimetypes_clist)
	{
		gint row, column;

		/* Handle by the event type */
		switch(etype)
		{
		  case GDK_BUTTON_PRESS:
			switch(button->button)
			{
			  case GDK_BUTTON3:
				/* Find the row and column that this button press
				 * occured over
				 */
				if(!gtk_clist_get_selection_info(
					clist,
					button->x, button->y,
					&row, &column
				))
				{
					row = -1;
					column = 0;
				}

				/* Select before mapping the menu? */
				if(EDV_GET_B(EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS) &&
				   (row >= 0) && (row < clist->rows)
				)
				{
					/* Select the row that the button was pressed over
					 * if no key modifiers are held then this will also
					 * unselect all previously selected rows
					 */
					gtk_clist_freeze(clist);
					if(!(button->state & GDK_CONTROL_MASK) &&
					   !(button->state & GDK_SHIFT_MASK)
					)
						gtk_clist_unselect_all(clist);
					clist->focus_row = row;
					gtk_clist_select_row(clist, row, 0);
					gtk_clist_thaw(clist);
				}

				/* Update the menus and then map the right-click
				 * menu
				 */
				EDVMimeTypesListWinUpdateWidgets(lw);
				gtk_menu_popup(
					GTK_MENU(lw->menu),
					NULL, NULL,
					NULL, NULL,
					button->button, button->time
				);

				status = TRUE;
				break;
			}
			break;
		}
	}

	return(status);
}

/*
 *	DND "drag_data_get" signal callback.
 */
static void EDVMimeTypesListWinDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	EDVCore *core;
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if((widget == NULL) || (dc == NULL) || (lw == NULL))
		return;

	if(lw->freeze_count > 0)
		return;

	core = lw->core;

	if(widget == lw->mimetypes_clist)
	{
		GtkCList *clist = GTK_CLIST(lw->mimetypes_clist);

		if(clist->selection != NULL)
		{
			gint buf_len;
			guint8 *buf = edv_mime_types_list_encode_buffer(
				core->mime_types_list,
				clist->selection,
				&buf_len
			);
			if(buf != NULL)
			{
				gtk_selection_data_set(
					selection_data,
					GDK_SELECTION_TYPE_STRING,
					8,		/* Bits Per Character */
					buf,		/* Data */
					buf_len		/* Length */
				);
				data_sent = TRUE;
				g_free(buf);
			}
		}
	}

	if(!data_sent)
	{
		const gchar *s = "Error";
		gtk_selection_data_set(
			selection_data,
			GDK_SELECTION_TYPE_STRING,
			8,			/* Bits Per Character */
			s,			/* Data */
			STRLEN(s)		/* Length */
		);
	}
}

/*
 *	DND "drag_data_received" signal callback.
 */
static void EDVMimeTypesListWinDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data                                         
)
{
	gint row = -1, column = 0;
	EDVCore *core;
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if((widget == NULL) || (selection_data == NULL) || (lw == NULL))
		return;

	if(lw->freeze_count > 0)
		return;

	if((selection_data->data == NULL) ||
	   (selection_data->length <= 0)
	)
		return;

	core = lw->core;

	/* Find the dropped on row and column */
	if(GTK_IS_CLIST(widget))
	{
		GtkCList *clist = GTK_CLIST(widget);
		gtk_clist_get_selection_info(
			clist,
			x,
			y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
				clist->column_title_area.height +
				clist->column_title_area.y : 0),
			&row, &column
		);
	}

	if(widget == lw->mimetypes_clist)
	{
		if(info == EDV_DND_INFO_MIME_TYPE)
		{
			gint	cur_insert_pos = row,
					mt_num;
			GList	*glist,
					*mime_types_list_dde = edv_mime_types_list_decode_buffer(
				core->mime_types_list,
				(const guint8 *)selection_data->data,
				selection_data->length
			);
			EDVMIMEType *m;

			/* Add each new MIME Type */
			for(glist = mime_types_list_dde;
				glist != NULL;
				glist = g_list_next(glist)
			)
			{
				m = EDV_MIME_TYPE(glist->data);
				if(m == NULL)
					continue;

				/* Append/insert this MIME Type into the list */
				if(cur_insert_pos < 0)
				{
					/* Append this MIME Type to the list */
					mt_num = g_list_length(core->mime_types_list);
					core->mime_types_list = g_list_append(
						core->mime_types_list,
						m
					);
				}
				else
				{
					/* Insert this MIME Type into the list */
					mt_num = cur_insert_pos;
					core->mime_types_list = g_list_insert(
						core->mime_types_list,
						m,
						cur_insert_pos
					);
					cur_insert_pos++;	/* Increment the insert position */
				}			
				glist->data = NULL;		/* Transfered */

				/* Notify about this MIME Type being added */
				edv_emit_mime_type_added(
					core,
					mt_num,
					m
				);
			}

			if(mime_types_list_dde != NULL)
			{
				g_list_foreach(
					mime_types_list_dde,
					(GFunc)edv_mime_type_delete,
					NULL
				);
				g_list_free(mime_types_list_dde);
			}
		}
	}
}

/*
 *	DND "drag_data_delete" signal callback.
 */
static void EDVMimeTypesListWinDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	EDVCore *core;
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if((widget == NULL) || (dc == NULL) || (lw == NULL))
		return;

	if(lw->freeze_count > 0)
		return;

	core = lw->core;

	if(widget == lw->mimetypes_clist)
	{
		GtkCList *clist = GTK_CLIST(widget);
		if(clist->selection != NULL)
		{
			gint mt_num;
			EDVMIMEType *m;

			/* Remove all the selected MIME Types */
			while(clist->selection != NULL)
			{
				mt_num = (gint)clist->selection->data;
				m = EDV_MIME_TYPE(g_list_nth_data(
					core->mime_types_list,
					(guint)mt_num
				));
				if(m == NULL)
					break;

				/* Remove this MIME Type */
				core->mime_types_list = g_list_remove(
					core->mime_types_list,
					m
				);

				/* Notify about the removal of this MIME Type,
				 * the callback will the row from the GtkCList
				 */
				edv_emit_mime_type_removed(core, mt_num);
			}
		}
	}
}

/*
 *	GtkCList "select_row" signal callback.
 */
static void EDVMimeTypesListWinSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	EDVCore *core;
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if((clist == NULL) || (lw == NULL))
		return;

	if(lw->freeze_count > 0)
		return;

	core = lw->core;

	/* If the selected row is fully visible then scroll to it */
	if(gtk_clist_row_is_visible(clist, row) !=
		GTK_VISIBILITY_FULL
	)
		gtk_clist_moveto(
			clist,
			row, -1,			/* Row, column */
			0.5f, 0.0f			/* Row, column */
		);

	/* Set the DND drag icon */
	GUIDNDSetDragIconFromCListSelection(clist);

	/* Update the displayed MIME Type */
	if(row > -1)
		EDVMimeTypesListWinUpdateDisplay(
			lw,
			row,
			EDV_MIME_TYPE(g_list_nth_data(
				core->mime_types_list,
				(guint)row
			))
		);

	EDVMimeTypesListWinUpdateWidgets(lw);

	/* Double click? */
	if(event != NULL)
	{
		if(event->type == GDK_2BUTTON_PRESS)
		{
			/* Edit */
			EDVMimeTypesListWinEdit(lw);
		}
	}
}

/*
 *	GtkCList "unselect_row" signal callback.
 */
static void EDVMimeTypesListWinUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	EDVCore *core;
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if((clist == NULL) || (lw == NULL))
		return;

	if(lw->freeze_count > 0)
		return;

	core = lw->core;

	/* Clear the displayed MIME Type */
	EDVMimeTypesListWinUpdateDisplay(lw, -1, NULL);

	EDVMimeTypesListWinUpdateWidgets(lw);
}

/*
 *	Find callback.
 */
static void EDVMimeTypesListWinFindCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	EDVMimeTypesListWinFind(lw);
}

/*
 *	Add callback.
 */
static void EDVMimeTypesListWinAddCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	EDVMimeTypesListWinAdd(lw);
}

/*
 *	Edit callback.
 */
static void EDVMimeTypesListWinEditCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	EDVMimeTypesListWinEdit(lw);
}

/*
 *	Remove callback.
 */
static void EDVMimeTypesListWinRemoveCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	EDVMimeTypesListWinRemove(
		lw,
		TRUE				/* Confirm */
	);
}

/*
 *	Shift up callback.
 */
static void EDVMimeTypesListWinShiftUpCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	EDVMimeTypesListWinShiftUp(lw);
}

/*
 *	Shift down callback.
 */
static void EDVMimeTypesListWinShiftDownCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	EDVMimeTypesListWinShiftDown(lw);
}


/*
 *	Save callback.
 */
static void EDVMimeTypesListWinSaveCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	EDVMimeTypesListWinSave(lw);
}

/*
 *	Import callback.
 */
static void EDVMimeTypesListWinImportCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	EDVMimeTypesListWinImport(lw);
}

/*
 *	Export callback.
 */
static void EDVMimeTypesListWinExportCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	EDVMimeTypesListWinExport(lw);
}


/*
 *	Close callback.
 */
static void EDVMimeTypesListWinCloseCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(data);
	if(lw == NULL)
		return;

	if(lw->freeze_count > 0)
		return;

	EDVMimeTypesListWinClose(lw);
}


/*
 *	Reconfigured notify callback.
 */
void EDVMimeTypesListWinReconfiguredNotifyCB(
	edv_mime_types_list_win_struct *lw
)
{
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkWidget *w;
	const CfgItem *cfg_list;
	EDVCore *core;

	if(lw == NULL)
		return;

	core = lw->core;
	cfg_list = core->cfg_list;
	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	/* Update RC styles */
	w = lw->toplevel;
	if((w != NULL) && (standard_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	w = lw->mimetypes_clist;
	if((w != NULL) && (lists_rcstyle != NULL))
		gtk_widget_modify_style_recursive(w, lists_rcstyle);

	EDVMimeTypesListWinUpdateWidgets(lw);
}

/*
 *	MIME Type added callback.
 */
void EDVMimeTypesListWinMimeTypeAddedCB(
	edv_mime_types_list_win_struct *lw,
	const gint mt_num, EDVMIMEType *m
)
{
	gint i, row, new_row;
	gchar **strv;
	GtkCList *clist;
	EDVCore *core;

	if((lw == NULL) || (m == NULL))
		return;

	clist = GTK_CLIST(lw->mimetypes_clist);
	core = lw->core;

	row = mt_num;

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(clist->columns * sizeof(gchar *));
	for(i = 0; i < clist->columns; i++)
		strv[i] = "";

	/* Append/insert te new new row */
	if(row >= clist->rows)
		new_row = gtk_clist_append(clist, strv);
	else
		new_row = gtk_clist_insert(clist, row, strv);

	/* Delete the row cell values */
	g_free(strv);

	/* Failed to add a new row? */
	if(new_row < 0)
		return;

	/* Set the new row's values */
	EDVMimeTypesListWinSetRow(lw, new_row, m);

	EDVMimeTypesListWinUpdateWidgets(lw);
}

/*
 *	MIME Type modified callback.
 */
void EDVMimeTypesListWinMimeTypeModifiedCB(
	edv_mime_types_list_win_struct *lw,
	const gint mt_num, EDVMIMEType *m
)
{
	GtkCList *clist;
	EDVCore *core;

	if((lw == NULL) || (m == NULL))
		return;

	clist = GTK_CLIST(lw->mimetypes_clist);
	core = lw->core;

	/* Check if the MIME Type index is valid */
	if((mt_num > -1) && (mt_num < clist->rows))
	{
		const gint sel_row = edv_clist_get_selected_last(clist, NULL);

		/* Update the row values */
		EDVMimeTypesListWinSetRow(
			lw,
			mt_num,
			m
		);
		if(mt_num == sel_row)
		{
			/* Update the displayed MIME Type */
			EDVMimeTypesListWinUpdateDisplay(lw, mt_num, m);
		}

		EDVMimeTypesListWinUpdateWidgets(lw);
	}
}

/*
 *	MIME Type removed callback.
 */
void EDVMimeTypesListWinMimeTypeRemovedCB(
	edv_mime_types_list_win_struct *lw,
	const gint mt_num
)
{
	gint row, sel_row;
	GtkCList *clist;
	EDVCore *core;

	if(lw == NULL)
		return;

	clist = GTK_CLIST(lw->mimetypes_clist);
	core = lw->core;

	row = mt_num;

	/* Get the current selected row */
	sel_row = edv_clist_get_selected_last(clist, NULL);

	/* Row exists? */
	if((row >= 0) && (row < clist->rows))
	{
		/* Remove the row */
		gtk_clist_remove(clist, row);

		if(mt_num == sel_row)
		{
			/* Update the displayed MIME Type */
			EDVMimeTypesListWinUpdateDisplay(lw, mt_num, NULL);
		}

		EDVMimeTypesListWinUpdateWidgets(lw);
	}

	/* Forward this signal to the MIME Type Edit Dialog */
	EDVMimeTypeEditDlgMimeTypeRemovedCB(lw->edit_dlg, mt_num);
}


/*
 *	Selects the MIME Type by index.
 *
 *	The i specifies the MIME Type's index.
 */
void EDVMimeTypesListWinSelect(
	edv_mime_types_list_win_struct *lw,
	const gint i
)
{
	GtkCList *clist;

	if(lw == NULL)
		return;

	clist = GTK_CLIST(lw->mimetypes_clist);
	if((i >= 0) && (i < clist->rows))
	{
		gtk_clist_freeze(clist);
		gtk_clist_unselect_all(clist);
		gtk_clist_select_row(clist, i, 0);
		gtk_clist_thaw(clist);
	}
}

/*
 *	Selects the MIME Type by type.
 *
 *	The type specifies the MIME Type's type.
 *
 *	Returns the index of the matched MIME Type of -1 on failed
 *	match.
 */
gint EDVMimeTypesListWinSelectByType(
	edv_mime_types_list_win_struct *lw,
	const gchar *type
)
{
	gint i;
	GList *glist;
	EDVMIMEType *m;
	EDVCore *core;

	if((lw == NULL) || STRISEMPTY(type))
		return(-2);

	core = lw->core;

	/* Iterate through the MIME Types list */
	for(glist = core->mime_types_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
			continue;

		if(m->type == NULL)
			continue;

		if(!g_strcasecmp(m->type, type))
		{
			EDVMimeTypesListWinSelect(lw, i);
			return(i);
		}
	}

	return(-1);
}


/*
 *	Updates the MIME Types List Window's displayed MIME Type.
 *
 *	This will destroy the display_client widget and create a
 *	new one if mt_ptr is not NULL.
 */
static void EDVMimeTypesListWinUpdateDisplay(
	edv_mime_types_list_win_struct *lw,
	const gint mt_num, EDVMIMEType *m
)
{
	const gint	border_major = 5,
					border_minor = 2;
	gint font_size;
	const gchar *date_format, *font_encoding;
	gchar	*font_name_h1_bold,
			*font_name_h1,
			*font_name_h2_bold,
			*font_name_h2;
	GdkFont *font;
	GtkRcStyle *rcstyle, *standard_rcstyle;
	GtkStyle *style;
	GtkWidget *w, *parent, *parent2, *parent3, *parent4;
	const CfgItem *cfg_list;
	EDVDateRelativity date_relativity;
	EDVCore *core;

	if(lw == NULL)
		return;

	core = lw->core;
	cfg_list = core->cfg_list;
	standard_rcstyle = core->standard_rcstyle;

	/* Get previous display client GtkVBox and destroy it */
	if(lw->display_client != NULL)
	{
		GTK_WIDGET_DESTROY(lw->display_icon);
		lw->display_icon = NULL;
		GTK_WIDGET_DESTROY(lw->display_edit_btn);
		lw->display_edit_btn = NULL;
		GTK_WIDGET_DESTROY(lw->display_client);
		lw->display_client = NULL;
	}

	/* Get the display parent widget */
	parent = lw->display_parent;
	if(parent == NULL)
		return;

	style = gtk_widget_get_style(parent);
	if(style == NULL)
		return;

	font = style->font;
	font_size = GDK_FONT_GET_FONT_NAME_SIZE(font);
	if(font_size < 3)
		font_size = 3;

#if defined(PROG_LANGUAGE_POLISH)
	font_encoding = "iso8859-2";
#else
	font_encoding = "iso8859-1";
#endif

	font_name_h1_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
		font_size + 2,
		font_encoding
	);
	font_name_h1 = g_strdup_printf(
"-adobe-helvetica-medium-r-normal-*-%i-*-*-*-*-*-%s",
		font_size + 2,
		font_encoding
	);
	font_name_h2_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
		font_size,
		font_encoding
	);
	font_name_h2 = g_strdup_printf(
"-adobe-helvetica-medium-r-normal-*-%i-*-*-*-*-*-%s",
		font_size,
		font_encoding
	);

#define CLEANUP_RETURN	{		\
 g_free(font_name_h1_bold);		\
 g_free(font_name_h1);			\
 g_free(font_name_h2_bold);		\
 g_free(font_name_h2);			\
									\
 return;				\
}

	/* Do not continue updating if the specified MIME Type is NULL */
	if(m == NULL)
	{
		CLEANUP_RETURN;
	}

	/* Get the configuration */
	date_relativity = (EDVDateRelativity)EDV_GET_I(
		EDV_CFG_PARM_DATE_RELATIVITY
	);
	date_format = EDV_GET_S(EDV_CFG_PARM_DATE_FORMAT);


	/* Realize the MIME Type since we need to use its icons */
	edv_mime_type_realize(m, FALSE);

	/* Create new client hbox for multiple columns */
	lw->display_client = w = gtk_hbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent = w;

	/* Left column vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	if(TRUE)
	{
		gboolean has_handler = FALSE;
		gchar	*s,
					*s2;
		const gchar *class_str = NULL;
		EDVPixmap *class_icon = NULL;

		/* Hbox for class heading label and handler/command */
		w = gtk_hbox_new(FALSE, border_minor);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;

		/* Get class heading icon and label */
		switch(m->mt_class)
		{
		  case EDV_MIME_TYPE_CLASS_SYSTEM:
			class_icon = lw->system_object_icon;
			class_str =
#if defined(PROG_LANGUAGE_SPANISH)
"El Tipo Del Objeto Del Sistema"
#elif defined(PROG_LANGUAGE_FRENCH)
"Le Type D'Objet De Systme"
#else
"System Object Type"
#endif
			;
			break;

		  case EDV_MIME_TYPE_CLASS_FORMAT:
			class_icon = lw->file_format_icon;
			class_str =
#if defined(PROG_LANGUAGE_SPANISH)
"Archive Formato"
#elif defined(PROG_LANGUAGE_FRENCH)
"Classer Le Format"
#else
"File Format"
#endif
			;
			has_handler = TRUE;
			break;

		  case EDV_MIME_TYPE_CLASS_PROGRAM:
			class_icon = lw->program_icon;
			class_str =
#if defined(PROG_LANGUAGE_SPANISH)
"La Aplicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Application"
#else
"Application"
#endif
			;
			has_handler = TRUE;
			break;

		  case EDV_MIME_TYPE_CLASS_UNIQUE:
			class_icon = lw->unique_object_icon;
			class_str =
#if defined(PROG_LANGUAGE_SPANISH)
"El Objeto"
#elif defined(PROG_LANGUAGE_FRENCH)
"Objet"
#else
"Object"
#endif
			;
			has_handler = TRUE;
			break;
		}
		/* Class heading icon */
		w = edv_pixmap_new_gtk_pixmap(class_icon);
		if(w != NULL)
		{
			gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
			gtk_widget_show(w);
		}
		/* Class heading label */
		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(font_name_h1);
		w = gtk_label_new(
			(class_str != NULL) ? class_str :
#if defined(PROG_LANGUAGE_SPANISH)
"*La Clase No Apoyada*"
#elif defined(PROG_LANGUAGE_FRENCH)
"*Classe Non Soutenue*"
#else
"*Unsupported Class*"
#endif
		);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_modify_style(w, rcstyle);
		GTK_RC_STYLE_UNREF(rcstyle);
		gtk_widget_show(w);

		/* Handler/command label, determine by handler */
		switch(m->handler)
		{
		  case EDV_MIME_TYPE_HANDLER_COMMAND:
			if((m->commands_list != NULL) && has_handler)
			{
				if(m->mt_class == EDV_MIME_TYPE_CLASS_PROGRAM)
				{
					s = s2 = NULL;
				}
				else
				{
					const gint	chars_visible = 40,
									ncmds = g_list_length(m->commands_list);
					const gchar *name;
					gchar *s_tmp;
				    gint len;
					GList *glist;
					EDVMIMETypeCommand *cmd;

					s = g_strdup_printf(
						"%s: ",
						(ncmds == 1) ? "Command" : "Commands"
					);

					s2 = STRDUP("");
				    for(glist = m->commands_list;
						glist != NULL;
						glist = g_list_next(glist)
					)
					{
						cmd = EDV_MIME_TYPE_COMMAND(glist->data);
						if(cmd == NULL)
							continue;

						name = cmd->name;
						if(name == NULL)
							continue;

						s_tmp = g_strconcat(
							s2,
							name,
							(g_list_next(glist) != NULL) ?
								", " : "",
							NULL
						);
						g_free(s2);
						s2 = s_tmp;

						len = STRLEN(s2);
						if((len > chars_visible) && (len > 3))
						{
							s2[chars_visible - 3] = '\0';
							s_tmp = g_strconcat(s2, "...", NULL);
							g_free(s2);
							s2 = s_tmp;
							break;
						}
					}
				}
			}
			else
			{
				s = s2 = NULL;
			}
			break;

		  case EDV_MIME_TYPE_HANDLER_EDV_ARCHIVER:
			if(has_handler)
			{
				s = STRDUP("Handled By: ");
				s2 = STRDUP("Archiver");
			}
			else
			{
				s = s2 = NULL;
			}
			break;

		  case EDV_MIME_TYPE_HANDLER_EDV_IMAGE_BROWSER:
			if(has_handler)
			{
				s = STRDUP("Handled By: ");
				s2 = STRDUP("Image Browser");
			}
			else
			{
				s = s2 = NULL;
			}
			break;

		  case EDV_MIME_TYPE_HANDLER_EDV_RECYCLE_BIN:
			if(has_handler)
			{
				s = STRDUP("Handled By: ");
				s2 = STRDUP("Recycle Bin");
			}
			else
			{
				s = s2 = NULL;
			}
			break;

		  default:
			s = s2 = NULL;
			break;
		}
		if(s != NULL)
		{
			rcstyle = gtk_rc_style_new();
			rcstyle->font_name = STRDUP(font_name_h2_bold);
			w = gtk_label_new(s2);
			gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
			gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
			gtk_widget_modify_style(w, rcstyle);
			gtk_widget_show(w);
			GTK_RC_STYLE_UNREF(rcstyle)

			rcstyle = gtk_rc_style_new();
			rcstyle->font_name = STRDUP(font_name_h2);
			w = gtk_label_new(s);
			gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
			gtk_widget_modify_style(w, rcstyle);
			gtk_widget_show(w);
			GTK_RC_STYLE_UNREF(rcstyle)
		}
		g_free(s);
		g_free(s2);
	}


	/* Icon and Type GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Icon */
	if(TRUE)
	{
		EDVPixmap *icon = edv_pixmap_ref(m->medium_icon[
			EDV_MIME_TYPE_ICON_STATE_STANDARD
		]);
		if(!edv_pixmap_is_loaded(icon))
		{
			(void)edv_pixmap_unref(icon);
			icon = edv_pixmap_ref(m->small_icon[
				EDV_MIME_TYPE_ICON_STATE_STANDARD
			]);
		}

		lw->display_icon = w = edv_pixmap_new_gtk_pixmap(icon);
		if(w != NULL)
		{
			gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
			gtk_widget_show(w);
		}

		(void)edv_pixmap_unref(icon);
	}
	/* Type label */
	rcstyle = gtk_rc_style_new();
	rcstyle->font_name = STRDUP(font_name_h1_bold);
	w = gtk_label_new(
		(m->type != NULL) ? m->type : "(null)"
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_modify_style(w, rcstyle);
	gtk_widget_show(w);
	GTK_RC_STYLE_UNREF(rcstyle)

	/* Edit button */
	w = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_end(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;
	lw->display_edit_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_edit_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Redacte"
#elif defined(PROG_LANGUAGE_FRENCH)
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#else
"Edit"
#endif
		"...", NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinEditCB), lw
	);
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(w, standard_rcstyle);
	gtk_widget_show(w);

	/* Value available? */
	if(!STRISEMPTY(m->value))
	{
		const gchar	*value_label = "",
					*value = m->value;

		/* HBox for value line */
		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;

		/* Value label */
		switch(m->mt_class)
		{
		  case EDV_MIME_TYPE_CLASS_SYSTEM:
			break;
		  case EDV_MIME_TYPE_CLASS_FORMAT:
			value_label = (strchr(value, ' ') != NULL) ?
				"Extensions: " : "Extension: ";
			break;
		  case EDV_MIME_TYPE_CLASS_PROGRAM:
			value_label = "Location: ";
			break;
		  case EDV_MIME_TYPE_CLASS_UNIQUE:
			value_label = "Location: ";
			break;
		}
		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(font_name_h2);
		w = gtk_label_new(value_label);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_modify_style(w, rcstyle);
		gtk_widget_show(w);
		GTK_RC_STYLE_UNREF(rcstyle)

		/* Value */
		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(font_name_h2);
		w = gtk_label_new(value);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_modify_style(w, rcstyle);
		gtk_widget_show(w);
		GTK_RC_STYLE_UNREF(rcstyle)
	}

	/* Description available? */
	if(!STRISEMPTY(m->description))
	{
		/* HBox for description line */
		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;

		/* Description label */
		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(font_name_h2);
		w = gtk_label_new(m->description);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_modify_style(w, rcstyle);
		gtk_widget_show(w);
		GTK_RC_STYLE_UNREF(rcstyle)
	}

	/* Last modified */
	if(!m->read_only)
	{
		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;

		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(font_name_h2);
		/* Prefix */
		w = gtk_label_new(
"Last Modified:"
		);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_modify_style(w, rcstyle);
		gtk_widget_show(w);
		/* Modified Date */
		if(m->modify_time > 0)
		{
		    gchar *s = edv_date_string_format(
			m->modify_time,
			date_format, date_relativity
		    );
		    w = gtk_label_new((s != NULL) ? s : "");
		    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		    gtk_widget_modify_style(w, rcstyle);
		    gtk_widget_show(w);
		    g_free(s);
		}

		GTK_RC_STYLE_UNREF(rcstyle);
	}

	CLEANUP_RETURN;
#undef CLEANUP_RETURN
}


/*
 *	Sets the MIME Types GtkCList row values.
 *
 *	The row specifies the row on the MIME Types GtkCList.
 *
 *	The m specifies the MIME Type.
 */
static void EDVMimeTypesListWinSetRow(
	edv_mime_types_list_win_struct *lw,
	const gint row,
	EDVMIMEType *m
)
{
	gint		mt_class,
					column,
					ncolumns;
	const gchar	*type,
					*value;
	GtkCList *clist;
	EDVPixmap	*mt_icon,
					*class_icon;
	EDVMIMETypeIconState mt_icon_state;
	EDVCore *core;

	if((lw == NULL) || (m == NULL))
		return;

	clist = GTK_CLIST(lw->mimetypes_clist);
	ncolumns = clist->columns;
	core = lw->core;
	if((row < 0) || (row >= clist->rows))
		return;

	/* Realize this MIME Type since we need to use its icons */
	edv_mime_type_realize(m, FALSE);

	/* Get the MIME Type's class */
	mt_class = m->mt_class;

	/* Get the values based on MIME Type's class */
	type = NULL;
	value = NULL;
	mt_icon = NULL;
	class_icon = NULL;
	switch(mt_class)
	{
	  case EDV_MIME_TYPE_CLASS_SYSTEM:
		type = m->type;
		value = "";
		mt_icon_state = EDV_MIME_TYPE_ICON_STATE_STANDARD;
		mt_icon = edv_pixmap_ref(m->small_icon[mt_icon_state]);
		class_icon = edv_pixmap_ref(lw->system_object_icon);
		break;

	  case EDV_MIME_TYPE_CLASS_FORMAT:
		type = m->type;
		value = m->value;
		mt_icon_state = EDV_MIME_TYPE_ICON_STATE_STANDARD;
		mt_icon = edv_pixmap_ref(m->small_icon[mt_icon_state]);
		class_icon = edv_pixmap_ref(lw->file_format_icon);
		break;

	  case EDV_MIME_TYPE_CLASS_PROGRAM:
		type = m->type;
		value = m->value;
		mt_icon_state = EDV_MIME_TYPE_ICON_STATE_STANDARD;
		mt_icon = edv_pixmap_ref(m->small_icon[mt_icon_state]);
		class_icon = edv_pixmap_ref(lw->program_icon);
		break;

	  case EDV_MIME_TYPE_CLASS_UNIQUE:
		type = m->type;
		value = m->value;
		mt_icon_state = EDV_MIME_TYPE_ICON_STATE_STANDARD;
		mt_icon = edv_pixmap_ref(m->small_icon[mt_icon_state]);
		class_icon = edv_pixmap_ref(lw->unique_object_icon);
		break;
	}


	/* Column 0 - Type */
	column = 0;
	if(column < ncolumns)
	{
		if((mt_icon != NULL) ? (mt_icon->pixmap != NULL) : FALSE)
			gtk_clist_set_pixtext(
				clist,
				row, column,
				(type != NULL) ? type : "",
				EDV_LIST_PIXMAP_TEXT_SPACING,
				mt_icon->pixmap, mt_icon->mask
			);
		else
			gtk_clist_set_text(
				clist,
				row, column,
				(type != NULL) ? type : ""
			);
	}

	/* Column 1 - Value */
	column = 1;
	if(column < ncolumns)
	{
		gtk_clist_set_text(
			clist,
			row, column,
			(value != NULL) ? value : ""
		);
	}

	/* Column 2 - Class */
	column = 2;
	if(column < ncolumns)
	{
		if((class_icon != NULL) ? (class_icon->pixmap != NULL) : FALSE)
			gtk_clist_set_pixmap(
				clist,
				row, column,
				class_icon->pixmap, class_icon->mask
			);
		else
			gtk_clist_set_text(
				clist,
				row, column,
				""
			);
	}

	(void)edv_pixmap_unref(mt_icon);
	(void)edv_pixmap_unref(class_icon);
}

/*
 *	Gets the MIME Types List.
 */
void EDVMimeTypesListWinGetList(
	edv_mime_types_list_win_struct *lw,
	const gboolean verbose
)
{
	gboolean user_aborted;
	gint		i,
					nmimetypes,
					row,
					ncolumns,
					progress_update_mod;
	gchar **strv;
	GList *glist;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVMIMEType *m;
	EDVCore *core;

	if(lw == NULL)
		return;

	if(verbose && ProgressDialogIsQuery())
		return;

	lw->freeze_count++;

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->mimetypes_clist);
	core = lw->core;
	ncolumns = clist->columns;

	/* Allocate the row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
		strv[i] = "";

	/* Map the progress dialog? */
	if(verbose)
	{
		ProgressDialogSetTransientFor(toplevel);
		ProgressDialogMap(
			"Loading MIME Types",
			"Loading MIME Types...",
			(const guint8 **)icon_mimetypes_32x32_xpm,
			"Stop"
		);
		gdk_flush();
	}

	gtk_clist_freeze(clist);

	/* Remove the existing rows and clear the display */
	gtk_clist_clear(clist);
	EDVMimeTypesListWinUpdateDisplay(lw, -1, NULL);

	/* Iterate through all the MIME Types list and add each
	 * MIME Type to the GtkCList
	 */
	user_aborted = FALSE;
	nmimetypes = g_list_length(core->mime_types_list);
	progress_update_mod = MAX(nmimetypes / 100, 1);
	for(glist = core->mime_types_list, i = 0;
		glist != NULL;
		glist = g_list_next(glist), i++
	)
	{
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
			continue;

		/* Append a new row and set the new row's values */
		row = gtk_clist_append(clist, strv);
		if(row < 0)
			break;

		EDVMimeTypesListWinSetRow(lw, row, m);

		/* Report progress and check for user abort */
		if(verbose && ((i % progress_update_mod) == 0))
		{
			if(EDVMimeTypesListProgressCB(
				lw, (gulong)(i + 1), (gulong)nmimetypes
			))
			{
				user_aborted = TRUE;
				break;
			}
		}
	}

	gtk_clist_thaw(clist);

	/* Delete the row cell values */
	g_free(strv);

	EDVMimeTypesListWinUpdateWidgets(lw);

	/* Unmap the progress dialog? */
	if(verbose)
	{
		ProgressDialogBreakQuery(TRUE);
		ProgressDialogSetTransientFor(NULL);
	}

	lw->freeze_count--;
}


/*
 *	Find.
 */
void EDVMimeTypesListWinFind(edv_mime_types_list_win_struct *lw)
{
	gint find_itterations = 0;
	gint row, column, sel_row;
	const gchar *find_str;
	GtkEntry *entry;
	GtkCList *clist;

	gchar *cell_text;
	guint8 spacing;
	GdkPixmap *pixmap;
	GdkBitmap *mask;

	if(lw == NULL)
		return;

	entry = GTK_ENTRY(lw->find_entry);
	clist = GTK_CLIST(lw->mimetypes_clist);

	/* Get the search string */
	find_str = gtk_entry_get_text(entry);
	if(STRISEMPTY(find_str))
		return;

	/* Get the selected row + 1 or 0 if none */
	sel_row = edv_clist_get_selected_last(clist, NULL);
	if(sel_row < 0)
		sel_row = 0;
	else
		sel_row++;

	/* Iterate through rows, checking each Type column to see if
	 * there are any partial matches
	 */
	do
	{
		/* Iterate from selected row to end */
		for(row = sel_row; row < clist->rows; row++)
		{
			/* Iterate through all cells on this row */
			for(column = 0; column < clist->columns; column++)
			{
				/* Begin fetching current cell's text by its type */
				cell_text = NULL;
				switch(gtk_clist_get_cell_type(clist, row, column))
				{
				  case GTK_CELL_TEXT:
					gtk_clist_get_text(clist, row, column, &cell_text);
					break;
				  case GTK_CELL_PIXTEXT:
					gtk_clist_get_pixtext(
						clist, row, column, &cell_text,
						&spacing, &pixmap, &mask
					);
					break;
				  case GTK_CELL_PIXMAP:
				  case GTK_CELL_WIDGET:
				  case GTK_CELL_EMPTY:
					break;
				}
				/* Got cell text? */
				if(!STRISEMPTY(cell_text))
				{
					/* Find string found inside cell text string? */
					if(strcasestr(cell_text, find_str))
						break;
				}
			}
			/* If column itteration broke before all columns were
			 * iterated through then that implies a match was made
			 */
			if(column < clist->columns)
				break;
		}
		/* Got match? */
		if(row < clist->rows)
			break;
		else
			find_itterations++;

		/* Reset sel_row to 0 so that find starts at beginning */
		sel_row = 0;

	} while(find_itterations < 2);

	/* Got match? */
	if(row < clist->rows)
	{
		/* Select new matched row */
		gtk_clist_unselect_all(clist);
		gtk_clist_select_row(clist, row, 0);
	}
}

/*
 *	Add.
 */
void EDVMimeTypesListWinAdd(edv_mime_types_list_win_struct *lw)
{
	gint row, mt_num;
	EDVMIMEType *m;
	GtkCList *clist;
	EDVCore *core;

	if(lw == NULL)
		return;

	clist = GTK_CLIST(lw->mimetypes_clist);
	core = lw->core;

	EDVMimeTypesListWinSetBusy(lw, TRUE);

	/* Create a new MIME Type */
	m = edv_mime_type_new_values();
	if(m == NULL)
	{
		EDVMimeTypesListWinSetBusy(lw, FALSE);
		return;
	}

	m->mt_class = EDV_MIME_TYPE_CLASS_FORMAT;
	m->type = STRDUP("misc/untitled");
	m->description = STRDUP("New MIME Type");

	/* Get the last selected MIME Type */
	row = edv_clist_get_selected_last(clist, NULL);

	/* Append/insert the new MIME Type into the list */
	if(row < 0)
	{
		/* Append the new MIME Type to the list */
		mt_num = g_list_length(core->mime_types_list);
		core->mime_types_list = g_list_append(
			core->mime_types_list,
			m
		);
	}
	else
	{
		/* Insert the new MIME Type into the list */
		mt_num = row;
		core->mime_types_list = g_list_insert(
			core->mime_types_list,
			m,
			row
		);
	}

	/* Notify about this MIME Type being added, the callbacks
	 * will add it to the GtkCList
	 */
	edv_emit_mime_type_added(
		core,
		mt_num,
		m
	);

	/* Select the new MIME Type */
	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_select_row(
		clist,
		mt_num,
		0
	);
	gtk_clist_thaw(clist);

	/* Map the MIME Types Edit Dialog with the new MIME Type */
	EDVMimeTypesListWinEditMimeType(
		lw,
		mt_num
	);

	EDVMimeTypesListWinSetBusy(lw, FALSE);
}

/*
 *	Edit.
 *
 *	The mt_num specifies the MIME Type to edit.
 */
void EDVMimeTypesListWinEditMimeType(
	edv_mime_types_list_win_struct *lw,
	const gint mt_num
)
{
	edv_mime_type_edit_dlg_struct *d;
	EDVCore *core;

	if(lw == NULL)
		return;

	core = lw->core;

	d = lw->edit_dlg;
	if(d == NULL)
		lw->edit_dlg = d = EDVMimeTypeEditDlgNew(core);
	if(d == NULL)
		return;

	edv_center_window_to_window(lw->toplevel, d->toplevel);
	EDVMimeTypeEditDlgMap(d);
	EDVMimeTypeEditDlgGetValues(d, mt_num);
	EDVMimeTypeEditDlgResetHasChanges(d, FALSE);
}

/*
 *	Edit.
 */
void EDVMimeTypesListWinEdit(edv_mime_types_list_win_struct *lw)
{
	gint row, mt_num;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVMIMEType *m;
	EDVCore *core;

	if(lw == NULL)
		return;

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->mimetypes_clist);
	core = lw->core;

	EDVMimeTypesListWinSetBusy(lw, TRUE);

	/* Get the last selected MIME Type */
	row = edv_clist_get_selected_last(clist, NULL);
	mt_num = row;
	m = EDV_MIME_TYPE(g_list_nth_data(
		core->mime_types_list,
		(guint)mt_num
	));
	if(m == NULL)
	{
		EDVMimeTypesListWinSetBusy(lw, FALSE);
		return;
	}

	/* Map the MIME Types Edit Dialog with this MIME Type */
	EDVMimeTypesListWinEditMimeType(lw, mt_num);

	EDVMimeTypesListWinSetBusy(lw, FALSE);
}

/*
 *	Remove.
 *
 *	If confirm is TRUE then the user will be asked for
 *	confirmation of removal.
 */
void EDVMimeTypesListWinRemove(
	edv_mime_types_list_win_struct *lw,
	const gboolean confirm
)
{
	gint mt_num, nselected;
	EDVMIMEType *m;
	GList *glist, *sel_list;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVCore *core;

	if(lw == NULL)
		return;

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->mimetypes_clist);
	core = lw->core;

	EDVMimeTypesListWinSetBusy(lw, TRUE);

	/* Nothing to remove? */
	if(clist->selection == NULL)
	{
		EDVMimeTypesListWinSetBusy(lw, FALSE);
		return;
	}

	/* Check if any of the selected MIME Types may not be removed */
	for(glist = clist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		mt_num = (gint)glist->data;
		m = EDV_MIME_TYPE(g_list_nth_data(
			core->mime_types_list,
			(guint)mt_num
		));
		if(m == NULL)
			continue;

		/* Is this MIME Type read-only? */
		if(m->read_only)
		{
			gchar *msg = g_strdup_printf(
"MIME Type \"%s\"\n\
may not be removed, because it was either created\n\
internally by %s or opened from a global\n\
configuration.",
				m->type,
				PROG_NAME
			);
			edv_play_sound_warning(core);
			edv_message_warning(
				"Remove Failed",
				msg,
				NULL,
				toplevel
			);
			g_free(msg);
			EDVMimeTypesListWinSetBusy(lw, FALSE);
			return;
		}
	}

	/* Create a list of selected MIME Type pointers */
	sel_list = NULL;
	for(glist = clist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		mt_num = (gint)glist->data;
		m = EDV_MIME_TYPE(g_list_nth_data(
			core->mime_types_list,
			(guint)mt_num
		));
		if(m == NULL)
			continue;

		sel_list = g_list_append(
			sel_list,
			m
		);
	}
	if(sel_list == NULL)
	{
		EDVMimeTypesListWinSetBusy(lw, FALSE);
		return;
	}

	/* Get the pointer to the first selected MIME Type if and only
	 * if exactly one MIME Type is selected
	 */
	nselected = g_list_length(sel_list);
	mt_num = -1;
	m = (nselected == 1) ?
		EDV_MIME_TYPE(sel_list->data) : NULL;

	/* Confirm remove? */
	if(confirm)
	{
		gint response;
		gchar *msg;

		if((m != NULL) ? (m->type != NULL) : FALSE)
			msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Usted est seguro que usted quiere quitar el MIME Type \"%s\"?"
#elif defined(PROG_LANGUAGE_FRENCH)
"Etes-vous sr que vous voulez enlever le MIME Type \"%s\"?"
#elif defined(PROG_LANGUAGE_GERMAN)
"Sind Sie sicher Sie der MIME Type \"%s\" wollen herausnehmen?"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Lei sono sicuro che lei vuole togliere lo MIME Type \"%s\"?"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bent u zeker u de MIME Type \"%s\" wil verwijderen?"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Esto seguro quer retirar o MIME Type \"%s\"?"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Er De sikker De fjerner MIME Type \"%s\"?"
#else
"Remove MIME Type \"%s\"?"
#endif
				,
				m->type
			);
		else
			msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Usted est seguro que usted quiere quitar %i MIME Types?"
#elif defined(PROG_LANGUAGE_FRENCH)
"Etes-vous sr que vous voulez enlever %i MIME Types?"
#elif defined(PROG_LANGUAGE_GERMAN)
"Sind sie sicher sie %i MIME Types wollen herausnehmen?"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Lei sono sicuro che lei vuole togliere %i MIME Types?"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bent u zeker u %i MIME Types wil verwijderen?"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Esto seguro quer retirar %i MIME Types?"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Er De sikker De fjerner %i MIME Types?"
#else
"Remove %i selected MIME Types?"
#endif
				,
				nselected
			);
		edv_play_sound_question(core);
		CDialogSetTransientFor(toplevel);
		response = CDialogGetResponse(
#if defined(PROG_LANGUAGE_SPANISH)
"Confirme Quite"
#elif defined(PROG_LANGUAGE_FRENCH)
"Confirmer Enlever"
#elif defined(PROG_LANGUAGE_GERMAN)
"Besttigen Sie Herausnimmt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Confermare Togliere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bevestiig Verwijdert"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Confirme Retira"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Bekreft Fjerner"
#else
"Confirm Remove"
#endif
			, msg, NULL,
			CDIALOG_ICON_QUESTION,
			CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
			CDIALOG_BTNFLAG_NO
		);
		g_free(msg);
		CDialogSetTransientFor(NULL);
		if(response != CDIALOG_RESPONSE_YES)
		{
			g_list_free(sel_list);
			EDVMimeTypesListWinSetBusy(lw, FALSE);
			return;
		}
	}

	/* Begin removing the selected MIME Types */
	while(clist->selection != NULL)
	{
		mt_num = (gint)clist->selection->data;
		m = EDV_MIME_TYPE(g_list_nth_data(
			core->mime_types_list,
			(guint)mt_num
		));
		if(m == NULL)
			break;

		/* Remove this MIME Type */
		core->mime_types_list = g_list_remove(
			core->mime_types_list,
			m
		);

		/* Notify about this MIME Type being removed, the
		 * callbacks will remove the corresponding GtkCList
		 * row
		 */
		edv_emit_mime_type_removed(core, mt_num);
	}

	/* Delete the selected MIME Types list */
	g_list_free(sel_list);

	EDVMimeTypesListWinSetBusy(lw, FALSE);
	edv_play_sound_completed(core);
}

/*
 *	Shift up.
 */
void EDVMimeTypesListWinShiftUp(edv_mime_types_list_win_struct *lw)
{
	gint i, n;
	GtkCList *clist;
	EDVCore *core;

	if(lw == NULL)
		return;

	clist = GTK_CLIST(lw->mimetypes_clist);
	core = lw->core;

	EDVMimeTypesListWinSetBusy(lw, TRUE);

	/* Nothing to shift? */
	if((clist->selection == NULL) || (clist->rows < 2))
	{
		EDVMimeTypesListWinSetBusy(lw, FALSE);
		return;
	}

	/* Check if the first item is already at the top */
	i = 0;
	if(g_list_find(clist->selection, (gpointer)i) != NULL)
	{
		EDVMimeTypesListWinSetBusy(lw, FALSE);
		return;
	}

	/* Shift each selected MIME Type up */
	n = clist->rows;
	for(i = 1; i < n; i++)
	{
		if(g_list_find(clist->selection, (gpointer)i) != NULL)
		{
			gpointer data;

			/* Shift this MIME Type up */
			GList *glist1 = g_list_nth(
				core->mime_types_list,
				(guint)(i - 1)
			);
			GList *glist2 = g_list_nth(
				core->mime_types_list,
				(guint)i
			);
			if((glist1 == NULL) || (glist2 == NULL))
				break;

			data = glist1->data;
			glist1->data = glist2->data;
			glist2->data = data;

			/* Shift this row up */
			gtk_clist_freeze(clist);
			gtk_clist_swap_rows(
				clist,
				i - 1,
				i
			);
			gtk_clist_thaw(clist);

			/* Notify about the shift */
			edv_emit_mime_type_modified(
				core,
				i - 1,
				EDV_MIME_TYPE(glist1->data)
			);
			edv_emit_mime_type_modified(
				core,
				i,
				EDV_MIME_TYPE(glist2->data)
			);
		}
	}

	EDVMimeTypesListWinSetBusy(lw, FALSE);
}

/*
 *	Shift down.
 */
void EDVMimeTypesListWinShiftDown(edv_mime_types_list_win_struct *lw)
{
	gint i, n;
	GtkCList *clist;
	EDVCore *core;

	if(lw == NULL)
		return;

	clist = GTK_CLIST(lw->mimetypes_clist);
	core = lw->core;

	EDVMimeTypesListWinSetBusy(lw, TRUE);

	/* Nothing to shift? */
	if((clist->selection == NULL) || (clist->rows < 2))
	{
		EDVMimeTypesListWinSetBusy(lw, FALSE);
		return;
	}

	/* Check if the last item is already at the bottom */
	i = clist->rows - 1;
	if(g_list_find(clist->selection, (gpointer)i) != NULL)
	{
		EDVMimeTypesListWinSetBusy(lw, FALSE);
		return;
	}

	/* Shift each selected MIME Type down */
	n = clist->rows;
	for(i = n - 2; i >= 0; i--)
	{
		if(g_list_find(clist->selection, (gpointer)i) != NULL)
		{
			gpointer data;

			/* Shift this MIME Type down */
			GList *glist1 = g_list_nth(
				core->mime_types_list,
				(guint)i
			);
			GList *glist2 = g_list_nth(
				core->mime_types_list,
				(guint)(i + 1)
			);
			if((glist1 == NULL) || (glist2 == NULL))
				break;

			data = glist1->data;
			glist1->data = glist2->data;
			glist2->data = data;

			/* Shift this row down */
			gtk_clist_freeze(clist);
			gtk_clist_swap_rows(
				clist,
				i,
				i + 1
			);
			gtk_clist_thaw(clist);

			/* Notify about the shift */
			edv_emit_mime_type_modified(
				core,
				i,
				EDV_MIME_TYPE(glist1->data)
			);
			edv_emit_mime_type_modified(
				core,
				i + 1,
				EDV_MIME_TYPE(glist2->data)
			);
		}
	}

	EDVMimeTypesListWinSetBusy(lw, FALSE);
}

/*
 *	Save.
 */
void EDVMimeTypesListWinSave(edv_mime_types_list_win_struct *lw)
{
	const gchar *s;
	GtkWidget *toplevel;
	const CfgItem *cfg_list;
	EDVCore *core;

	if(lw == NULL)
		return;

	toplevel = lw->toplevel;
	core = lw->core;
	cfg_list = core->cfg_list;

	EDVMimeTypesListWinSetBusy(lw, TRUE);

	/* Get the MIME Types file path */
	s = EDV_GET_S(EDV_CFG_PARM_FILE_MIME_TYPES);
	if(!STRISEMPTY(s))
	{
		gboolean object_existed;
		gchar	*msg,
					*path = STRDUP(s);
		EDVVFSObject *obj;

		/* Map the progress dialog */
		msg = g_strdup_printf(
"Saving MIME Types to:\n\
\n\
    %s\n",
			path
		);
		ProgressDialogSetTransientFor(toplevel);
		ProgressDialogMap(
			"Saving MIME Types",
			msg,
			(const guint8 **)icon_save_32x32_xpm,
			"Stop"
		);
		g_free(msg);
		gdk_flush();

		/* Does the MIME Types file already exists? */
		object_existed = edv_path_lexists(path);

		/* Save the MIME Types */
		edv_mime_types_list_file_save(
			core->mime_types_list,
			path,
			FALSE,				/* Do not include read only
											 * MIME Types */
			EDVMimeTypesListProgressCB, lw
		);

		/* Notify about the MIME Types file being saved */
		obj = edv_vfs_object_stat(path);
		if(obj != NULL)
		{
			if(object_existed)
				edv_emit_vfs_object_modified(
					core,
					path,
					path,
					obj
				);
			else
				edv_emit_vfs_object_added(
					core,
					path,
					obj
				);
			edv_vfs_object_delete(obj);
		}

		g_free(path);
	}

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	EDVMimeTypesListWinUpdateWidgets(lw);

	EDVMimeTypesListWinSetBusy(lw, FALSE);
}

/*
 *	Import.
 */
void EDVMimeTypesListWinImport(edv_mime_types_list_win_struct *lw)
{
	gboolean response;
	gint		mt_num,
					npaths = 0,
					nftypes = 0;
	gchar **paths_list = NULL;
	GtkWidget *toplevel;
	GtkCList *clist;
	fb_type_struct	**ftypes_list = NULL,
					*ftype_rtn = NULL;
	EDVCore *core;

	if((lw == NULL) || FileBrowserIsQuery())
		return;

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->mimetypes_clist);
	core = lw->core;

	/* Get selected MIME Type index number that corresponds to the
	 * last selected row index number on the MIME Types clist
	 */
	mt_num = edv_clist_get_selected_last(clist, NULL);

	EDVMimeTypesListWinSetBusy(lw, TRUE);

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*mailcap",
		EDV_MIME_TYPES_FORMAT_DESC_MAILCAP
	);
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*media.types *mime.types",
		EDV_MIME_TYPES_FORMAT_DESC_MEDIA_TYPES
	);
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		".ini",
		EDV_MIME_TYPES_FORMAT_DESC_ENDEAVOUR
	);

	/* Query the user for the MIME Types file to import */
	lw->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Import MIME Types",
		"Import", "Cancel",
		NULL,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);
	lw->freeze_count--;

	/* Got user response? */
	if(response && (npaths > 0))
	{
		const gchar *ftype_name = (ftype_rtn != NULL) ?
			ftype_rtn->name : NULL;
		if(ftype_name != NULL)
		{
			gint i;
			gchar *path;
			for(i = 0; i < npaths; i++)
			{
				path = STRDUP(paths_list[i]);
				if(path == NULL)
					continue;

#define MAP_PROGRESS(_path_,_toplevel_)	{	\
 gchar *msg = g_strdup_printf(			\
"Importing MIME Types from:\n\
\n\
    %s\n",					\
  (_path_)					\
 );						\
 ProgressDialogSetTransientFor(_toplevel_);	\
 ProgressDialogMap(				\
  "Importing MIME Types",			\
  msg,						\
  (const guint8 **)icon_import_32x32_xpm,	\
  "Stop"					\
 );						\
 g_free(msg);					\
}

				/* Begin importing MIME Types based on selected
				 * file type
				 *
				 * Mailcap
				 */
				if(!g_strcasecmp(ftype_name, EDV_MIME_TYPES_FORMAT_DESC_MAILCAP))
				{
					gboolean	update,
									only_newer;
				    if(EDVMimeTypesMailcapImportOptsGetResponse(
						core,
						toplevel,
						path,
						&update,
						&only_newer
					))
					{
						MAP_PROGRESS(path, toplevel);
						core->mime_types_list = edv_mime_types_list_import_mailcap(
							core->mime_types_list,
							path,
							mt_num,		/* Insert index */
							update,
							only_newer,
							EDVMimeTypesListProgressCB, lw,
							edv_mime_type_added_cb, core,
							edv_mime_type_modified_cb, core
						);
					}
				}
				/* Media Types */
				else if(!g_strcasecmp(ftype_name, EDV_MIME_TYPES_FORMAT_DESC_MEDIA_TYPES))
				{
					gboolean	update,
									only_newer;
					if(EDVMimeTypesMediaTypesImportOptsGetResponse(
						core,
						toplevel,
						path,
						&update,
						&only_newer
					))
					{
						MAP_PROGRESS(path, toplevel);
						core->mime_types_list = edv_mime_types_list_import_media_types(
							core->mime_types_list,
							path,
							mt_num,		/* Insert index */
							update,
							only_newer,
							EDVMimeTypesListProgressCB, lw,
							edv_mime_type_added_cb, core,
							edv_mime_type_modified_cb, core
						);
					}
				}
				/* Endeavour MIME Types */
				else if(!g_strcasecmp(ftype_name, EDV_MIME_TYPES_FORMAT_DESC_ENDEAVOUR))
				{
					gboolean	update,
									only_newer;
					if(EDVMIMETypesEndeavourImportOptsGetResponse(
						core,
						toplevel,
						path,
						&update,
						&only_newer
					))
					{
						MAP_PROGRESS(path, toplevel);
						core->mime_types_list = edv_mime_types_list_file_open(
							core->mime_types_list,
							path,
							mt_num,		/* Insert index */
							update,
							only_newer,
							EDVMimeTypesListProgressCB, lw,
							edv_mime_type_added_cb, core,
							edv_mime_type_modified_cb, core,
							FALSE
						);
					}
				}
#undef MAP_PROGRESS
				g_free(path);
			}
			edv_play_sound_completed(core);
		}
	}	/* Got user response? */

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	EDVMimeTypesListWinUpdateWidgets(lw);

	EDVMimeTypesListWinSetBusy(lw, FALSE);
}

/*
 *	Export.
 */
void EDVMimeTypesListWinExport(edv_mime_types_list_win_struct *lw)
{
	gboolean response;
	gint		nftypes = 0,
					npaths = 0,
					mt_num;
	gchar **paths_list = NULL;
	GList		*glist,
					*selected_mime_types_list;
	GtkWidget *toplevel;
	GtkCList *clist;
	fb_type_struct	**ftypes_list = NULL,
					*ftype_rtn = NULL;
	EDVMIMEType *m;
	EDVCore *core;

	if((lw == NULL) || FileBrowserIsQuery())
		return;

	EDVMimeTypesListWinSetBusy(lw, TRUE);

	toplevel = lw->toplevel;
	clist = GTK_CLIST(lw->mimetypes_clist);
	core = lw->core;

	/* Create a list of selected MIME Types to export */
	glist = clist->selection;
	selected_mime_types_list = NULL;
	for(glist = clist->selection;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		mt_num = (gint)glist->data;
		m = EDV_MIME_TYPE(g_list_nth_data(
			core->mime_types_list,
			(guint)mt_num
		));
		if(m == NULL)
			continue;

		selected_mime_types_list = g_list_append(
			selected_mime_types_list,
			edv_mime_type_copy(m)
		);
	}
	/* No MIME Types selected? */
	if(selected_mime_types_list == NULL)
	{
		EDVMimeTypesListWinSetBusy(lw, FALSE);
		return;
	}

	/* Create the file types list */
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*mailcap",
		EDV_MIME_TYPES_FORMAT_DESC_MAILCAP
	);
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		"*media.types *mime.types",
		EDV_MIME_TYPES_FORMAT_DESC_MEDIA_TYPES
	);
	FileBrowserTypeListNew(
		&ftypes_list, &nftypes,
		".ini",
		EDV_MIME_TYPES_FORMAT_DESC_ENDEAVOUR
	);

	/* Query the user for the MIME Types file to export */
	lw->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
		"Export MIME Types",
		"Export", "Cancel",
		NULL,
		ftypes_list, nftypes,
		&paths_list, &npaths,
		&ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);
	lw->freeze_count--;

	/* Got user response? */
	if(response && (npaths > 0))
	{
		gchar *new_path = STRDUP(paths_list[npaths - 1]);
		const gchar *ftype_name = (ftype_rtn != NULL) ?
			ftype_rtn->name : NULL;
		if((new_path != NULL) && (ftype_name != NULL))
		{
			EDVVFSObject *obj;
			gboolean object_existed = FALSE;
			gint response = CDIALOG_RESPONSE_YES;

			gchar *s = FileBrowserTypeCompleteExtension(
				new_path,
				ftype_rtn,
				FALSE			/* Do not replace if specified */
			);
			if(s != NULL)
			{
				g_free(new_path);
				new_path = s;
			}

			/* File already exists? */
			if(edv_path_lexists(new_path))
			{
				gchar *msg = g_strdup_printf(
"Overwrite existing file:\n\
\n\
    %s",
					new_path
				);
				edv_play_sound_warning(core);
				CDialogSetTransientFor(toplevel);
				response = CDialogGetResponse(
					"Confirm Overwrite",
					msg,
					NULL,
					CDIALOG_ICON_WARNING,
					CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
					CDIALOG_BTNFLAG_NO
				);
				g_free(msg);
				CDialogSetTransientFor(NULL);

				object_existed = TRUE;
			}
			if(response == CDIALOG_RESPONSE_YES)
			{
				/* Map the progress dialog */
				gchar *msg = g_strdup_printf(
"Exporting MIME Types to:\n\
\n\
    %s\n",
					new_path
				);
				ProgressDialogSetTransientFor(toplevel);
				ProgressDialogMap(
					"Exporting MIME Types",
					msg,
					(const guint8 **)icon_export_32x32_xpm,
					"Stop"
				);
				g_free(msg);
				gdk_flush();

				/* Export the MIME Types based on the selected file
				 * type
				 */
				/* Mailcap */
				if(!g_strcasecmp(ftype_name, EDV_MIME_TYPES_FORMAT_DESC_MAILCAP))
				{
					edv_mime_types_list_export_mailcap(
						selected_mime_types_list,
						new_path,
						TRUE,		/* Include read_only MIME Types */
						EDVMimeTypesListProgressCB, lw
					);
				}
				/* Media Types */
				else if(!g_strcasecmp(ftype_name, EDV_MIME_TYPES_FORMAT_DESC_MEDIA_TYPES))
				{
				    edv_mime_types_list_export_media_types(
						selected_mime_types_list,
						new_path,
						TRUE,		/* Include read_only MIME Types */
						EDVMimeTypesListProgressCB, lw
					);
				}
				/* Endeavour */
				else if(!g_strcasecmp(ftype_name, EDV_MIME_TYPES_FORMAT_DESC_ENDEAVOUR))
				{
				    edv_mime_types_list_file_save(
						selected_mime_types_list,
						new_path,
						TRUE,		/* Include read_only MIME Types */
						EDVMimeTypesListProgressCB, lw
					);
				}
				edv_play_sound_completed(core);

				/* Notify object added or modified */
				obj = edv_vfs_object_stat(new_path);
				if(obj != NULL)
				{
					if(object_existed)
						edv_emit_vfs_object_modified(
							core,
							new_path,
							new_path,
							obj
						);
					else
						edv_emit_vfs_object_added(
							core,
							new_path,
							obj
						);
					edv_vfs_object_delete(obj);
				}
			}

			g_free(new_path);
		}
	}       /* Got user response? */

	/* Unmap the progress dialog */
	ProgressDialogBreakQuery(TRUE);
	ProgressDialogSetTransientFor(NULL);

	/* Delete the file types list and reset the file browser */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);
	FileBrowserReset();

	EDVMimeTypesListWinUpdateWidgets(lw);

	/* Delete the selected MIME Types list */
	g_list_foreach(
		selected_mime_types_list,
		(GFunc)edv_mime_type_delete,
		NULL
	);
	g_list_free(selected_mime_types_list);

	EDVMimeTypesListWinSetBusy(lw, FALSE);
}

/*
 *	Close.
 */
void EDVMimeTypesListWinClose(edv_mime_types_list_win_struct *lw)
{
	if(lw == NULL)
		return;

	EDVMimeTypesListWinUnmap(lw);
}


/*
 *	Creates a new MIME Types List Window.
 */
edv_mime_types_list_win_struct *EDVMimeTypesListWinNew(EDVCore *core)
{
	const gint	border_major = 5,
					border_minor = 2;
	const gchar *wm_name, *wm_class;
	gchar *heading[3];
	gpointer entry_rtn;
	GdkWindow *window;
	GtkRcStyle *standard_rcstyle, *lists_rcstyle;
	GtkAccelGroup *accelgrp;
	GtkStyle *style;
	GtkWidget	*w, *menu,
					*parent, *parent2, *parent3, *parent4,
					*toplevel;
	GtkCList *clist;
	const GtkTargetEntry dnd_tar_types[] = {
{EDV_DND_TARGET_MIME_TYPE,		0,	EDV_DND_INFO_MIME_TYPE}
	};
	const GtkTargetEntry dnd_src_types[] = {
{EDV_DND_TARGET_MIME_TYPE,		0,	EDV_DND_INFO_MIME_TYPE}
	};
	edv_mime_types_list_win_struct *lw = EDV_MIME_TYPES_LIST_WIN(
		g_malloc0(sizeof(edv_mime_types_list_win_struct))
	);
	if((lw == NULL) || (core == NULL))
	{
		g_free(lw);
		return(NULL);
	}

	standard_rcstyle = core->standard_rcstyle;
	lists_rcstyle = core->lists_rcstyle;

	wm_name = core->wm_name;
	wm_class = core->wm_class;

	lw->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	lw->accelgrp = accelgrp = gtk_accel_group_new();
	lw->busy_count = 0;
	lw->freeze_count = 0;
	lw->core = core;

	lw->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	if(!STRISEMPTY(wm_name) && !STRISEMPTY(wm_class))
		gtk_window_set_wmclass(GTK_WINDOW(w), wm_name, wm_class);
	else
		gtk_window_set_wmclass(
			GTK_WINDOW(w),
			EDV_MIME_TYPES_LIST_WM_CLASS_WINDOW_NAME,
			PROG_NAME
		);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_window_set_title(GTK_WINDOW(w), EDV_MIME_TYPES_LIST_WIN_TITLE);
	gtk_widget_set_usize(w, EDV_MIME_TYPES_LIST_WIN_WIDTH, EDV_MIME_TYPES_LIST_WIN_HEIGHT);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
		gdk_window_set_decorations(
			window,
			GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
				GDK_DECOR_MINIMIZE
		);
		gdk_window_set_functions(
			window,
			GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
		);
		GUISetWMIcon(
			window,
			(guint8 **)icon_mimetypes_48x48_xpm
		);
	}
	gtk_signal_connect(
		GTK_OBJECT(w), "delete_event",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinDeleteEventCB), lw
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	style = gtk_widget_get_style(w);
	parent = w;

	/* Load the icons */
	lw->system_object_icon = edv_load_pixmap_from_data(
		core,
		(edv_pixmap_data *)icon_system_object_20x20_xpm,
		"icon_system_object_20x20_xpm"
	);
	lw->file_format_icon = edv_load_pixmap_from_data(
		core,
		(edv_pixmap_data *)icon_file_extension_20x20_xpm,
		"icon_file_extension_20x20_xpm"
	);
	lw->program_icon = edv_load_pixmap_from_data(
		core,
		(edv_pixmap_data *)icon_executable_20x20_xpm,
		"icon_executable_20x20_xpm"
	);
	lw->unique_object_icon = edv_load_pixmap_from_data(
		core,
		(edv_pixmap_data *)icon_unique_object_20x20_xpm,
		"icon_unique_object_20x20_xpm"
	);


	/* Main GtkVBox */
	lw->main_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Find GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	w = GUIPromptBar(
		(guint8 **)icon_search_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"El Hallazgo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Dcouverte"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fund"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Trovare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Vondst"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ache"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Funn"
#else
"Find"
#endif
		":", NULL, &entry_rtn
	);
	lw->find_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
		GTK_OBJECT(w), "activate",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinFindCB), lw
	);
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);


	/* MIME Types GtkCList & Buttons GtkHbox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Scrolled window for MIME Types clist */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
		GTK_SCROLLED_WINDOW(w),
		GTK_POLICY_AUTOMATIC,
		GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* MIME Types clist */
#if defined(PROG_LANGUAGE_SPANISH)
	heading[0] = "De Tipo MIME";
	heading[1] = "El Valor";
	heading[2] = "";
#elif defined(PROG_LANGUAGE_FRENCH)
	heading[0] = "MIME Le Type";
	heading[1] = "Valeur";
	heading[2] = "";
#elif defined(PROG_LANGUAGE_GERMAN)
	heading[0] = "MIME Typ";
	heading[1] = "Wert";
	heading[2] = "";
#elif defined(PROG_LANGUAGE_ITALIAN)
	heading[0] = "Il Tipo Di MIME";
	heading[1] = "Il Valore";
	heading[2] = "";
#elif defined(PROG_LANGUAGE_DUTCH)
	heading[0] = "MIME Type";
	heading[1] = "Waarde";
	heading[2] = "";
#elif defined(PROG_LANGUAGE_PORTUGUESE)
	heading[0] = "MIME Tipo";
	heading[1] = "O Valor";
	heading[2] = "";
#elif defined(PROG_LANGUAGE_NORWEGIAN)
	heading[0] = "MIME Type";
	heading[1] = "Verdi";
	heading[2] = "";
#else
	heading[0] = "MIME Type";
	heading[1] = "Value";
	heading[2] = "";		/* Class */
#endif
	lw->mimetypes_clist = w = gtk_clist_new_with_titles(
		sizeof(heading) / sizeof(gchar *),
		heading
	);
	clist = GTK_CLIST(w);
	gtk_widget_set_usize(w, -1, EDV_MIME_TYPES_LIST_WIN_HEIGHT * 0.5f);
	gtk_widget_add_events(
		w,
		GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(edv_clist_motion_event_cb), core
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinKeyEventCB), lw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinKeyEventCB), lw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinButtonPressEventCB), lw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "select_row",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinSelectRowCB), lw
	);
	gtk_signal_connect(
		GTK_OBJECT(w), "unselect_row",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinUnselectRowCB), lw
	);
	gtk_container_add(GTK_CONTAINER(parent3), w);
	gtk_widget_realize(w);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	gtk_clist_set_column_width(clist, 0, 160);
	gtk_clist_set_column_justification(   
		clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_width(clist, 1, 150);
	gtk_clist_set_column_justification(   
		clist, 1, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_width(clist, 2, 20);
	gtk_clist_set_column_justification(   
		clist, 2, GTK_JUSTIFY_CENTER
	);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	/* Set up DND for the contents list */
	GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE,	/* Actions */
		GDK_BUTTON1_MASK,			/* Buttons */
		NULL,
		EDVMimeTypesListWinDragDataGetCB,
		EDVMimeTypesListWinDragDataDeleteCB,
		NULL,
		lw
	);
	GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE,	/* Actions */
		GDK_ACTION_MOVE,			/* Default action if same */
		GDK_ACTION_MOVE,			/* Default action */
		EDVMimeTypesListWinDragDataReceivedCB,
		lw,
		TRUE				/* Highlight */
	);
	gtk_widget_show(w);


	/* Buttons GtkVBox */
	w = gtk_vbox_new(FALSE, 2 * border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Vbox for add, edit, and remove set of buttons */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Add button */
	lw->add_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_add_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Agregue"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ajouter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fgen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aggiungere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toevoeg"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Adicione"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfy"
#else
"Add"
#endif
		"...", NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinAddCB), lw
	);
	gtk_accel_group_add(
		accelgrp, GDK_a, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_a);
	gtk_widget_show(w);

	/* Edit button */
	lw->edit_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_edit_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Redacte"
#elif defined(PROG_LANGUAGE_FRENCH)
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#else
"Edit"
#endif
		"...", NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinEditCB), lw
	);
	gtk_accel_group_add(
		accelgrp, GDK_e, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_e);
	gtk_widget_show(w);

	/* Remove button */
	lw->remove_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_remove_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Quite"
#elif defined(PROG_LANGUAGE_FRENCH)
"Enlever"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nehmen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Togliere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verwijdeer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Retire"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fjern"
#else
"Remove"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinRemoveCB), lw
	);
	gtk_accel_group_add(
		accelgrp, GDK_r, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_r);
	gtk_widget_show(w);


	/* GtkVBox for the Shift Buttons */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Shift Up Button */
	lw->up_btn = w = GUIButtonArrowLabelH(
		GTK_ARROW_UP, 20, 20,
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie Arriba"
#elif defined(PROG_LANGUAGE_FRENCH)
"Changer Haut"
#elif defined(PROG_LANGUAGE_GERMAN)
"Spostare Su"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Spostare Su"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verplaats Op"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude Cima"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forskyv Opp"
#else
"Shift Up"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinShiftUpCB), lw
	);
	gtk_accel_group_add(
		accelgrp, GDK_u, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_u);
	gtk_widget_show(w);
	gtk_box_set_child_packing(
		GTK_BOX(GUIButtonGetMainBox(w)),
		GUIButtonGetArrow(w),
		FALSE, FALSE, 0, GTK_PACK_START
	);

	/* Shift Down Button */
	lw->down_btn = w = GUIButtonArrowLabelH(
		GTK_ARROW_DOWN, 20, 20,
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie Abajo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Changer Bas"
#elif defined(PROG_LANGUAGE_GERMAN)
"Spostare Gi"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Spostare Gi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verplaats Beneden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude Baixo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forskyv Ned"
#else
"Shift Down"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinShiftDownCB), lw
	);
	gtk_accel_group_add(
		accelgrp, GDK_d, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_d);
	gtk_widget_show(w);
	gtk_box_set_child_packing(
		GTK_BOX(GUIButtonGetMainBox(w)),
		GUIButtonGetArrow(w),
		FALSE, FALSE, 0, GTK_PACK_START
	);

	/* Vbox for Save, Import, and Export Buttons */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Save Button */
	lw->save_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_save_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Salve"
#elif defined(PROG_LANGUAGE_FRENCH)
"Epargner"
#elif defined(PROG_LANGUAGE_GERMAN)
"Auer"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Risparmiare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Red"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Poupe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Untatt"
#else
"Save"
#endif
		, NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinSaveCB), lw
	);
	gtk_accel_group_add(
		accelgrp, GDK_s, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_s);
	gtk_widget_show(w);

	/* Import Button */
	lw->import_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_import_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Importacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Importation"
#elif defined(PROG_LANGUAGE_GERMAN)
"Import"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Importazione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Import"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Importe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Import"
#else
"Import"
#endif
		"...", NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinImportCB), lw
	);
	gtk_accel_group_add(
		accelgrp, GDK_i, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_i);
	gtk_widget_show(w);

	/* Export Button */
	lw->export_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_export_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Exportacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Exportation"
#elif defined(PROG_LANGUAGE_GERMAN)
"Export"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Esportazione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Export"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Exportao"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Eksport"
#else
"Export"
#endif
		"...", NULL
	);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinExportCB), lw
	);
	gtk_accel_group_add(
		accelgrp, GDK_p, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_p);
	gtk_widget_show(w);



	/* Horizontal separator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Parent vbox for displaying selected MIME Type */
	lw->display_parent = w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Client vbox that is parented to the display_parent, leave
	 * this NULL, it'll be created when needed
	 */
	lw->display_client = NULL;


	/* Horizontal separator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Hbox for buttons */
	w = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Close button */
	lw->close_btn = w = GUIButtonPixmapLabelH(
		(guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fin"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
		, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
		w,
		GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
		GTK_OBJECT(w), "clicked",
		GTK_SIGNAL_FUNC(EDVMimeTypesListWinCloseCB), lw
	);
	gtk_accel_group_add(
		accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
		accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	gtk_widget_show(w);


	/* Right-Click GtkMenu */
	lw->menu = menu = GUIMenuCreate();
	if(menu != NULL)
	{
		guint accel_key, accel_mods;
		guint8 **icon;
		const gchar *label;
		gpointer data = lw;
		void (*func_cb)(GtkWidget *, gpointer);

#define ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_LABEL,			\
  accelgrp,					\
  icon,						\
  label,					\
  accel_key, accel_mods,			\
  func_cb, data					\
 );						\
}
#define ADD_MENU_ITEM_SEPARATOR	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL, NULL,					\
  0, 0,						\
  NULL, NULL					\
 );						\
}
		icon = (guint8 **)icon_add_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Agregue"
#elif defined(PROG_LANGUAGE_FRENCH)
"Ajouter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Fgen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Aggiungere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toevoeg"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Adicione"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Tilfy"
#else
"Add"
#endif
			"...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = EDVMimeTypesListWinAddCB;
		ADD_MENU_ITEM_LABEL
		lw->add_mi = w;

		icon = (guint8 **)icon_edit_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Redacte"
#elif defined(PROG_LANGUAGE_FRENCH)
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#else
"Edit"
#endif
			"...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = EDVMimeTypesListWinEditCB;
		ADD_MENU_ITEM_LABEL
		lw->edit_mi = w;

		icon = (guint8 **)icon_remove_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Quite"
#elif defined(PROG_LANGUAGE_FRENCH)
"Enlever"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nehmen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Togliere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verwijdeer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Retire"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Fjern"
#else
"Remove"
#endif
			;
		accel_key = 0;
		accel_mods = 0;
		func_cb = EDVMimeTypesListWinRemoveCB;
		ADD_MENU_ITEM_LABEL
		lw->remove_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		icon = NULL;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie Arriba"
#elif defined(PROG_LANGUAGE_FRENCH)
"Changer En Haut"
#elif defined(PROG_LANGUAGE_GERMAN)
"Spostare Su"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Spostare Su"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verplaats Op"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude Para Cima"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forskyv Opp"
#else
"Shift Up"
#endif
			;
		accel_key = 0;
		accel_mods = 0;
		func_cb = EDVMimeTypesListWinShiftUpCB;
		ADD_MENU_ITEM_LABEL
		lw->up_mi = w;

		icon = NULL;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie Hacia Abajo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Changer En Bas"
#elif defined(PROG_LANGUAGE_GERMAN)
"Spostare Gi"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Spostare Gi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verplaats Beneden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude Para Baixo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forskyv Ned"
#else
"Shift Down"
#endif
			;
		accel_key = 0;
		accel_mods = 0;
		func_cb = EDVMimeTypesListWinShiftDownCB;
		ADD_MENU_ITEM_LABEL
		lw->down_mi = w;

		ADD_MENU_ITEM_SEPARATOR

		icon = (guint8 **)icon_save_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Salve"
#elif defined(PROG_LANGUAGE_FRENCH)
"Epargner"
#elif defined(PROG_LANGUAGE_GERMAN)
"Auer"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Risparmiare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Red"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Poupe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Untatt"
#else
"Save"
#endif
			;
		accel_key = 0;
		accel_mods = 0;
		func_cb = EDVMimeTypesListWinSaveCB;
		ADD_MENU_ITEM_LABEL
		lw->save_mi = w;

		icon = (guint8 **)icon_import_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Importacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Importation"
#elif defined(PROG_LANGUAGE_GERMAN)
"Import"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Importazione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Import"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Importe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Import"
#else
"Import"
#endif
			"...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = EDVMimeTypesListWinImportCB;
		ADD_MENU_ITEM_LABEL
		lw->import_mi = w;

		icon = (guint8 **)icon_export_20x20_xpm;
		label =
#if defined(PROG_LANGUAGE_SPANISH)
"Exportacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Exportation"
#elif defined(PROG_LANGUAGE_GERMAN)
"Export"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Esportazione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Export"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Exportao"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Eksport"
#else
"Export"
#endif
			"...";
		accel_key = 0;
		accel_mods = 0;
		func_cb = EDVMimeTypesListWinExportCB;
		ADD_MENU_ITEM_LABEL
		lw->export_mi = w;



#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_SEPARATOR
	}


	/* Set initial RC styles */
	if(standard_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			lw->toplevel, standard_rcstyle
		);
	if(lists_rcstyle != NULL)
		gtk_widget_modify_style_recursive(
			lw->mimetypes_clist, lists_rcstyle
		);


	EDVMimeTypesListWinUpdateWidgets(lw);

	lw->freeze_count--;

	return(lw);
}

/*
 *	Updates the MIME Types List Window's widgets to reflect current
 *	values.
 */
void EDVMimeTypesListWinUpdateWidgets(edv_mime_types_list_win_struct *lw)
{
	gboolean sensitive;
	gint nrows, sel_row, nselected;
	GtkCList *clist;

	if(lw == NULL)
		return;

	lw->freeze_count++;

	/* Get selected row on clist */
	clist = GTK_CLIST(lw->mimetypes_clist);
	nrows = clist->rows;
	nselected = g_list_length(clist->selection);
	sel_row = edv_clist_get_selected_last(clist, NULL);

	/* Add */
	sensitive = (nselected <= 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(lw->add_btn, sensitive);
	gtk_widget_set_sensitive(lw->add_mi, sensitive);

	/* Edit */
	sensitive = (nselected == 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(lw->edit_btn, sensitive);
	gtk_widget_set_sensitive(lw->edit_mi, sensitive);

	/* Remove */
	sensitive = (nselected >= 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(lw->remove_btn, sensitive);
	gtk_widget_set_sensitive(lw->remove_mi, sensitive);

	/* Shift Up */
	sensitive = ((g_list_find(clist->selection, (gpointer)0) == NULL) &&
		(nselected > 0)) ? TRUE : FALSE;
	gtk_widget_set_sensitive(lw->up_btn, sensitive);
	gtk_widget_set_sensitive(lw->up_mi, sensitive);

	/* Shift Down */
	sensitive = ((g_list_find(clist->selection, (gpointer)(nrows - 1)) == NULL) &&
		(nselected > 0)) ? TRUE : FALSE;
	gtk_widget_set_sensitive(lw->down_btn, sensitive);
	gtk_widget_set_sensitive(lw->down_mi, sensitive);

	/* Import */
	sensitive = (nselected <= 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(lw->import_mi, sensitive);
	gtk_widget_set_sensitive(lw->import_btn, sensitive);

	/* Export */
	sensitive = (nselected >= 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(lw->export_mi, sensitive);
	gtk_widget_set_sensitive(lw->export_btn, sensitive);

	lw->freeze_count--;
}

/*
 *	Sets the MIME Types List Window as busy or ready.
 */
void EDVMimeTypesListWinSetBusy(
	edv_mime_types_list_win_struct *lw,
	const gboolean busy
)
{
	GdkCursor *cursor;
	GtkWidget *w;
	EDVCore *core;

	if(lw == NULL)
		return;

	core = lw->core;

	w = lw->toplevel;
	if(w != NULL)
	{
		if(busy)
		{
			/* Increase busy count */
			lw->busy_count++;

			/* If already busy then don't change anything */
			if(lw->busy_count > 1)
				return;

			cursor = edv_get_cursor(core, EDV_CURSOR_CODE_BUSY);
		}
		else
		{
			/* Reduce busy count */
			lw->busy_count--;
			if(lw->busy_count < 0)
				lw->busy_count = 0;

			/* If still busy do not change anything */
			if(lw->busy_count > 0)
				return;

			cursor = NULL;		/* Use the default cursor */
		}

		/* Update toplevel window's cursor */
		if(w->window != NULL)
		{
			gdk_window_set_cursor(w->window, cursor);
			gdk_flush();
		}
	}
}

/*
 *	Checks if the MIME Types List Window is mapped.
 */
gboolean EDVMimeTypesListWinIsMapped(edv_mime_types_list_win_struct *lw)
{
	if(lw == NULL)
		return(FALSE);

	return(GTK_WIDGET_MAPPED(lw->toplevel));
}

/*
 *	Maps the MIME Types List Window.
 */
void EDVMimeTypesListWinMap(edv_mime_types_list_win_struct *lw)
{
	if(lw == NULL)
		return;

	gtk_widget_show_raise(lw->toplevel);
	gtk_widget_grab_focus(lw->mimetypes_clist);
}

/*
 *	Unmaps the MIME Types List Window.
 */
void EDVMimeTypesListWinUnmap(edv_mime_types_list_win_struct *lw)
{
	GtkCList *clist;

	if(lw == NULL)
		return;

	gtk_widget_hide(lw->toplevel);

	clist = GTK_CLIST(lw->mimetypes_clist);
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);
}

/*
 *	Delete the MIME Types List Window.
 */
void EDVMimeTypesListWinDelete(edv_mime_types_list_win_struct *lw)
{
	GtkCList *clist;

	if(lw == NULL)
		return;

	/* Delete the MIME Type Edit dialog */
	EDVMimeTypeEditDlgDelete(lw->edit_dlg);
	lw->edit_dlg = NULL;

	EDVMimeTypesListWinUnmap(lw);

	clist = GTK_CLIST(lw->mimetypes_clist);
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);

	lw->freeze_count++;

	gtk_widget_destroy(lw->menu);

	gtk_widget_destroy(lw->toplevel);
	gtk_accel_group_unref(lw->accelgrp);

	(void)edv_pixmap_unref(lw->system_object_icon);
	(void)edv_pixmap_unref(lw->file_format_icon);
	(void)edv_pixmap_unref(lw->program_icon);
	(void)edv_pixmap_unref(lw->unique_object_icon);

	lw->freeze_count--;

	g_free(lw);
}
