#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#ifdef __SOLARIS__
# include "../include/os.h"
#endif

#include "guiutils.h"
#include "cfg.h"
#include "edv_types.h"
#include "mime_type_import_options_dlg.h"
#include "endeavour2.h"
#include "libendeavour2-base/edv_utils.h"
#include "edv_utils_gtk.h"
#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_import_20x20.xpm"
/* #include "images/icon_export_20x20.xpm" */
#include "images/icon_cancel_20x20.xpm"

#include "images/icon_import_32x32.xpm"
/* #include "images/icon_export_32x32.xpm" */


/*
 *	Dialog Common:
 */
typedef struct {
	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gint		freeze_count;
	EDVCore	*core;
	GtkWidget	*main_vbox,
			*client_vbox,
			*ok_btn,
			*cancel_btn;
	gchar		*path;		/* Path to object to be imported/exported */
	gboolean	got_response;
} Dlg;
#define DLG(p)			((Dlg *)(p))


/*
 *	Mailcap Import Options Dialog:
 */
typedef struct {
	Dlg		d;
	GtkWidget	*update_check,
			*only_newer_check;
} MailcapImportDlg;
#define MAILCAP_IMPORT_DLG(p)	((MailcapImportDlg *)(p))


/*
 *	Media Types Import Options Dialog:
 */
typedef struct {
	Dlg		d;
	GtkWidget       *update_check,
			*only_newer_check;
} MediaTypesImportDlg;
#define MEDIA_TYPES_IMPORT_DLG(p)	((MediaTypesImportDlg *)(p))


/*
 *	MIME Types Import Options Dialog:
 */
typedef struct {
	Dlg		d;
	GtkWidget	*update_check,
			*only_newer_check;
} MIMETypesImportDlg;
#define MIME_TYPES_IMPORT_DLG(p)	((MIMETypesImportDlg *)(p))


/* Dialog Common Callbacks */
static gint EDVDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVDlgOKCB(GtkWidget *widget, gpointer data);
static void EDVDlgCancelCB(GtkWidget *widget, gpointer data);

/* Dialog Common */
static Dlg *EDVDlgNew(
	gint size, gint width, gint height,
	const gchar *title, guint8 **icon_data,
	const gchar *ok_label, guint8 **ok_icon_data,
	const gchar *cancel_label, guint8 **cancel_icon_data,
	GtkWidget *ref_toplevel,
	EDVCore *core, const gchar *path
);
static void EDVDlgDelete(Dlg *d);


/* Mailcap Options Querying */
gboolean EDVMimeTypesMailcapImportOptsGetResponse(
	EDVCore *core,
	GtkWidget *ref_toplevel,
	const gchar *path,
	gboolean *update,
	gboolean *only_newer
);

/* Media Types Options Querying */
gboolean EDVMimeTypesMediaTypesImportOptsGetResponse(
	EDVCore *core,
	GtkWidget *ref_toplevel,
	const gchar *path,
	gboolean *update,
	gboolean *only_newer
);

/* Endeavour Options Querying */
gboolean EDVMIMETypesEndeavourImportOptsGetResponse(
	EDVCore *core,
	GtkWidget *ref_toplevel,
	const gchar *path,
	gboolean *update,
	gboolean *only_newer
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Dialog toplevel GtkWindow "delete_event" signal callback.
 */
static gint EDVDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	EDVDlgCancelCB(widget, data);
	return(TRUE);
}

/*
 *	Dialog OK callback.
 */
static void EDVDlgOKCB(GtkWidget *widget, gpointer data)
{
	Dlg *d = DLG(data);
	if(d == NULL)
	    return;

	d->got_response = TRUE;
	gtk_main_quit();
}

/*
 *	Dialog Cancel callback.
 */
static void EDVDlgCancelCB(GtkWidget *widget, gpointer data)
{
	Dlg *d = DLG(data);
	if(d == NULL)
	    return;

	d->got_response = FALSE;
	gtk_main_quit();
}


/*
 *	Creates a new Dialog.
 */
static Dlg *EDVDlgNew(
	gint size, gint width, gint height,
	const gchar *title, guint8 **icon_data,
	const gchar *ok_label, guint8 **ok_icon_data,
	const gchar *cancel_label, guint8 **cancel_icon_data,
	GtkWidget *ref_toplevel,
	EDVCore *core, const gchar *path
)
{
	const gint	border_major = 5;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*parent, *parent2,
			*toplevel;
	Dlg *d = (size > 0) ? DLG(g_malloc0(size)) : NULL;
	if(d == NULL)
	    return(NULL);

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
	d->freeze_count = 0;
	d->core = core;
	d->path = STRDUP(path);
	d->got_response = FALSE;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	if(title != NULL)
	    gtk_window_set_title(GTK_WINDOW(w), title);
	gtk_widget_set_usize(w, width, height);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, icon_data);
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVDlgDeleteEventCB), d
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	if((ref_toplevel != NULL) ?
	    GTK_IS_WINDOW(ref_toplevel) : FALSE
	)
	{
	    gtk_window_set_modal(GTK_WINDOW(w), TRUE);
	    gtk_window_set_transient_for(
		GTK_WINDOW(w), GTK_WINDOW(ref_toplevel)
	    );
	}
	parent = w;

	/* Main vbox */
	d->main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* Client vbox */
	d->client_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons hbox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* OK button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    ok_icon_data, ok_label, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT | GTK_HAS_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDlgOKCB), d
	);
	gtk_widget_show(w);

	/* Cancel button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    cancel_icon_data, cancel_label, NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_widget_show(w);

	/* Grab focus and default for the OK button */
	w = d->ok_btn;
	gtk_widget_grab_default(w);
	gtk_widget_grab_focus(w);

	return(d);
}

/*
 *	Deletes the Dialog.
 */
static void EDVDlgDelete(Dlg *d)
{
	if(d == NULL)
	    return;

	gtk_widget_hide(d->toplevel);
	gtk_window_set_modal(GTK_WINDOW(d->toplevel), FALSE);
	gtk_window_set_transient_for(GTK_WINDOW(d->toplevel), NULL);
	GTK_WIDGET_DESTROY(d->ok_btn);
	GTK_WIDGET_DESTROY(d->cancel_btn);
	GTK_WIDGET_DESTROY(d->client_vbox);
	GTK_WIDGET_DESTROY(d->main_vbox);
	GTK_WIDGET_DESTROY(d->toplevel);
	GTK_ACCEL_GROUP_UNREF(d->accelgrp);
	g_free(d->path);
	g_free(d);
}


/*
 *	Queries user for Mailcap Import options.
 *
 *	Returns TRUE if the user response with OK or FALSE if the
 *	user response with Cancel.
 */
gboolean EDVMimeTypesMailcapImportOptsGetResponse(
	EDVCore *core,
	GtkWidget *ref_toplevel,
	const gchar *path,
	gboolean *update,
	gboolean *only_newer
)
{
	const gint	border_major = 5,
			border_minor = 2,
			width = 350,
			height = -1;
	gboolean	got_response = FALSE;
	GtkAccelGroup *accelgrp;
	GtkStyle *style;
	GtkWidget *toplevel, *client_vbox;
	CfgItem *cfg_list;
	MailcapImportDlg *d;

	if(core == NULL)
	    return(got_response);

	cfg_list = core->cfg_list;

	/* Get configuration values */
	if(update != NULL)
	    *update = EDV_GET_B(EDV_CFG_PARM_MAILCAP_IMPORT_UPDATE);
	if(only_newer != NULL)
	    *only_newer = EDV_GET_B(EDV_CFG_PARM_MAILCAP_IMPORT_ONLY_NEWER);

	/* Create new Dialog */
	d = MAILCAP_IMPORT_DLG(EDVDlgNew(
	    sizeof(MailcapImportDlg), width, height,
	    "Import Mailcap Options", (guint8 **)icon_import_32x32_xpm,
	    "Import", (guint8 **)icon_import_20x20_xpm,
	    "Cancel", (guint8 **)icon_cancel_20x20_xpm,
	    ref_toplevel,
	    core, path
	));
	if(d == NULL)
	    return(got_response);

	toplevel = DLG(d)->toplevel;
	accelgrp = DLG(d)->accelgrp;
	client_vbox = DLG(d)->client_vbox;
	style = gtk_widget_get_style(toplevel);

	/* Create the options widgets */
	if(client_vbox != NULL)
	{
	    gint font_size;
	    const gchar *font_encoding;
	    gchar *font_name_h1_bold;
	    GdkFont *font;
	    GtkStateType state;
	    GtkRcStyle *rcstyle;
	    GtkStyle *style = gtk_widget_get_style(toplevel);
	    GtkWidget *w, *parent = client_vbox, *parent2, *parent3;

	    /* Get the base font size */
	    font = style->font;
	    font_size = GDK_FONT_GET_FONT_NAME_SIZE(font);
	    if(font_size < 3)
		font_size = 3;

	    /* Format the font names */
#if defined(PROG_LANGUAGE_POLISH)
	    font_encoding = "iso8859-2";
#else
	    font_encoding = "iso8859-1";
#endif

            font_name_h1_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
                font_size + 2,
                font_encoding
            );

	    /* OK button accel keys */
	    w = DLG(d)->ok_btn;
	    gtk_accel_group_add(
		accelgrp, GDK_i, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    GUIButtonLabelUnderline(w, GDK_i);

	    /* Cancel button accel keys */
	    w = DLG(d)->cancel_btn;
	    gtk_accel_group_add(
		accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    gtk_accel_group_add(
		accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    GUIButtonLabelUnderline(w, GDK_c);

	    /* Heading */
	    if(!STRISEMPTY(path))
	    {
		w = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
		gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
		gtk_widget_show(w); 
		parent2 = w;

		rcstyle = gtk_rc_style_new();
		state = GTK_STATE_NORMAL;
		rcstyle->color_flags[state] = GTK_RC_BG | GTK_RC_BASE;
		memcpy(
		    &rcstyle->bg[state],
		    &style->base[state],
		    sizeof(GdkColor)
		);
		memcpy(
		    &rcstyle->base[state], 
		    &style->base[state],
		    sizeof(GdkColor)
		);
		w = gtk_event_box_new();
		gtk_container_add(GTK_CONTAINER(parent2), w);
		gtk_widget_modify_style(w, rcstyle);
		gtk_widget_show(w);
		GTK_RC_STYLE_UNREF(rcstyle);
		parent2 = w;

		w = gtk_vbox_new(FALSE, border_major);
		gtk_container_add(GTK_CONTAINER(parent2), w);
		gtk_container_border_width(GTK_CONTAINER(w), border_major);
		gtk_widget_show(w);
		parent2 = w;

		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;

		w = edv_new_pixmap_widget(
		    core,
		    (guint8 **)icon_import_32x32_xpm,
		    "icon_import_32x32_xpm"
		);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);

		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(font_name_h1_bold);
		w = gtk_label_new(g_basename(path));
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_modify_style(w, rcstyle);
		GTK_RC_STYLE_UNREF(rcstyle);
		gtk_widget_show(w);

		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;
		w = gtk_label_new(
"Mailcap files specify command(s) to view files\n\
associated with MIME Types."
		);
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);

		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;
		w = gtk_label_new(
"Mailcap files do not contain information on extensions,\n\
the extensions need to be obtained by importing the\n\
respective Media Types file afterwards."
		);
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);

		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;
		w = gtk_label_new(
"If you want to update the MIME Types with the ones\n\
from this file then check \"Update\"."
		);
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	    }

	    /* GtkVBox for option widgets */
	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Update */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* GtkCheckButton */
	    d->update_check = w = gtk_check_button_new_with_label(
		"Update"
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    GUISetWidgetTip(w,
"Check this to update MIME Types if they already exist and only\
 add new MIME Types if they do not exist, otherwise all MIME\
 Types will be add (which may result with duplicate MIME Types)"
	    );
	    gtk_widget_show(w);

	    /* Only Newer */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* Check button */
	    d->only_newer_check = w = gtk_check_button_new_with_label(
		"Only Newer"
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    GUISetWidgetTip(w,
"Check this to only import the MIME Types that are newer than\
 the existing ones or if it does not already exist"
	    );
	    gtk_widget_show(w);


	    /* Begin setting initial values */

	    /* Update */
	    if(update != NULL)
		gtk_toggle_button_set_active(
		    GTK_TOGGLE_BUTTON(d->update_check), *update
		);
	    /* Only Newer */
	    if(only_newer != NULL)
		gtk_toggle_button_set_active(
		    GTK_TOGGLE_BUTTON(d->only_newer_check), *only_newer
		);

	    g_free(font_name_h1_bold);
	}

	/* Map the dialog */
	gtk_widget_show(toplevel);

	/* Wait for user response */
	gtk_main();

	/* Get user response */
	got_response = DLG(d)->got_response;

	/* Update returns if the user responded with ok */
	if(got_response)
	{
	    /* Update */
	    if(update != NULL)
	    {
		*update = GTK_TOGGLE_BUTTON_GET_ACTIVE(
		    d->update_check
		);
		EDV_SET_B(
		    EDV_CFG_PARM_MAILCAP_IMPORT_UPDATE,
		    *update
		);
	    }
	    /* Only Newer */
	    if(only_newer != NULL)
	    { 
		*only_newer = GTK_TOGGLE_BUTTON_GET_ACTIVE(
		    d->only_newer_check
		);
		EDV_SET_B(
		    EDV_CFG_PARM_MAILCAP_IMPORT_ONLY_NEWER,
		    *only_newer
		);
	    }
	}

	/* Delete the Dialog */
	EDVDlgDelete(DLG(d));

	return(got_response);
}


/*
 *	Queries user for Media Types Import options.
 *
 *	Returns TRUE if the user response with OK or FALSE if the
 *	user response with Cancel.
 */
gboolean EDVMimeTypesMediaTypesImportOptsGetResponse(
	EDVCore *core,
	GtkWidget *ref_toplevel,
	const gchar *path,
	gboolean *update,
	gboolean *only_newer
)
{
	const gint	border_major = 5,
			border_minor = 2, 
			width = 350,
			height = -1;
	gboolean	got_response = FALSE;
	GtkAccelGroup *accelgrp;
	GtkStyle *style;
	GtkWidget *toplevel, *client_vbox;
	CfgItem *cfg_list;
	MediaTypesImportDlg *d;

	if(core == NULL)
	    return(got_response);

	cfg_list = core->cfg_list;

	/* Get configuration values */
	if(update != NULL)
	    *update = EDV_GET_B(EDV_CFG_PARM_MEDIA_TYPES_IMPORT_UPDATE);
	if(only_newer != NULL)
	    *only_newer = EDV_GET_B(EDV_CFG_PARM_MEDIA_TYPES_IMPORT_ONLY_NEWER);

	/* Create new Dialog */
	d = MEDIA_TYPES_IMPORT_DLG(EDVDlgNew(
	    sizeof(MediaTypesImportDlg), width, height,
	    "Import Media Types Options", (guint8 **)icon_import_32x32_xpm,
	    "Import", (guint8 **)icon_import_20x20_xpm,
	    "Cancel", (guint8 **)icon_cancel_20x20_xpm,
	    ref_toplevel,
	    core, path
	));
	if(d == NULL)
	    return(got_response);

	toplevel = DLG(d)->toplevel;
	accelgrp = DLG(d)->accelgrp;
	client_vbox = DLG(d)->client_vbox;
	style = gtk_widget_get_style(toplevel);

	/* Create options widgets */
	if(client_vbox != NULL)
	{
	    gint font_size;
	    const gchar *font_encoding;
	    gchar *font_name_h1_bold;
	    GdkFont *font;
	    GtkStateType state;
	    GtkRcStyle *rcstyle;
	    GtkStyle *style = gtk_widget_get_style(toplevel);
	    GtkWidget *w, *parent = client_vbox, *parent2, *parent3;

	    /* Get the base font size */
	    font = style->font;
	    font_size = GDK_FONT_GET_FONT_NAME_SIZE(font);
	    if(font_size < 3)
		font_size = 3;

	    /* Format the font names */
#if defined(PROG_LANGUAGE_POLISH)
	    font_encoding = "iso8859-2";
#else
	    font_encoding = "iso8859-1";
#endif

            font_name_h1_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
                font_size + 2,
                font_encoding
            );

	    /* OK button accel keys */
	    w = DLG(d)->ok_btn;
	    gtk_accel_group_add(
		accelgrp, GDK_i, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    GUIButtonLabelUnderline(w, GDK_i);

	    /* Cancel button accel keys */
	    w = DLG(d)->cancel_btn;
	    gtk_accel_group_add(
		accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    gtk_accel_group_add(
		accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    GUIButtonLabelUnderline(w, GDK_c);

	    /* Heading */
	    if(!STRISEMPTY(path))
	    {
		w = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
		gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent2 = w;

		rcstyle = gtk_rc_style_new();
		state = GTK_STATE_NORMAL;
		rcstyle->color_flags[state] = GTK_RC_BG | GTK_RC_BASE;
		memcpy(
		    &rcstyle->bg[state],
		    &style->base[state],
		    sizeof(GdkColor)
		);
		memcpy(
		    &rcstyle->base[state],
		    &style->base[state],
		    sizeof(GdkColor)
		);
		w = gtk_event_box_new();
		gtk_container_add(GTK_CONTAINER(parent2), w);
		gtk_widget_modify_style(w, rcstyle);
		gtk_widget_show(w);
		GTK_RC_STYLE_UNREF(rcstyle);
		parent2 = w;

		w = gtk_vbox_new(FALSE, border_major);
		gtk_container_add(GTK_CONTAINER(parent2), w);
		gtk_container_border_width(GTK_CONTAINER(w), border_major);
		gtk_widget_show(w);
		parent2 = w;

		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;

		w = edv_new_pixmap_widget(
		    core,
		    (guint8 **)icon_import_32x32_xpm,
		    "icon_import_32x32_xpm"
		);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);

		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(font_name_h1_bold);
		w = gtk_label_new(g_basename(path));
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_modify_style(w, rcstyle);
		GTK_RC_STYLE_UNREF(rcstyle);
		gtk_widget_show(w);

		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;
		w = gtk_label_new(
"Media Types files specify extension(s) associated\n\
with MIME Types."
		);
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);

		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;
		w = gtk_label_new(
"Media Types files do not specify any additional\n\
information, it is recommended that you import this\n\
Media Types file last."
		);
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);

		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;
		w = gtk_label_new(
"If you want to update the MIME Types with the ones\n\
from this file then check \"Update\"."
		);
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	    }

	    /* Vbox for option widgets */
	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Update */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* Check button */
	    d->update_check = w = gtk_check_button_new_with_label(
		"Update"
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    GUISetWidgetTip(w,
"Check this to update MIME Types if they already exist and only\
 add new MIME Types if they do not exist, otherwise all MIME\
 Types will be add (which may result with duplicate MIME Types)"
	    );
	    gtk_widget_show(w);

	    /* Only Newer */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* Check button */
	    d->only_newer_check = w = gtk_check_button_new_with_label(
		"Only Newer"
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    GUISetWidgetTip(w,
"Check this to only import the MIME Types that are newer than\
 the existing ones or if it does not already exist"
	    );
	    gtk_widget_show(w);


	    /* Begin setting initial values */

	    /* Update */
	    if(update != NULL)
		gtk_toggle_button_set_active(
		    GTK_TOGGLE_BUTTON(d->update_check), *update
		);
	    /* Only Newer */
	    if(only_newer != NULL)
		gtk_toggle_button_set_active(
		    GTK_TOGGLE_BUTTON(d->only_newer_check), *only_newer
		);

	    g_free(font_name_h1_bold);
	}

	/* Map the dialog */
	gtk_widget_show(toplevel);

	/* Wait for user response */
	gtk_main();

	/* Get user response */
	got_response = DLG(d)->got_response;

	/* Update returns if the user responded with ok */
	if(got_response)
	{
	    /* Update */
	    if(update != NULL)
	    { 
		*update = GTK_TOGGLE_BUTTON_GET_ACTIVE(
		    d->update_check
		);
		EDV_SET_B(
		    EDV_CFG_PARM_MEDIA_TYPES_IMPORT_UPDATE,
		    *update
		);
	    }
	    /* Only Newer */
	    if(only_newer != NULL)
	    {
		*only_newer = GTK_TOGGLE_BUTTON_GET_ACTIVE(
		    d->only_newer_check
		);
		EDV_SET_B(
		    EDV_CFG_PARM_MEDIA_TYPES_IMPORT_ONLY_NEWER,
		    *only_newer
		);
	    }
	}

	/* Delete the Dialog */
	EDVDlgDelete(DLG(d));

	return(got_response);
}

/*
 *	Queries user for Endeavour Import options.
 *
 *	Returns TRUE if the user response with OK or FALSE if the
 *	user response with Cancel.
 */
gboolean EDVMIMETypesEndeavourImportOptsGetResponse(
	EDVCore *core,
	GtkWidget *ref_toplevel,
	const gchar *path,
	gboolean *update,
	gboolean *only_newer
)
{
	const gint	border_major = 5,
			border_minor = 2, 
			width = 350,
			height = -1;
	gboolean	got_response = FALSE;
	GtkAccelGroup *accelgrp;
	GtkStyle *style;
	GtkWidget *toplevel, *client_vbox;
	CfgItem *cfg_list;
	MIMETypesImportDlg *d;

	if(core == NULL)
	    return(got_response);

	cfg_list = core->cfg_list;

	/* Get the configuration values */
	if(update != NULL)
	    *update = EDV_GET_B(EDV_CFG_PARM_MIME_TYPES_IMPORT_UPDATE);
	if(only_newer != NULL)
	    *only_newer = EDV_GET_B(EDV_CFG_PARM_MIME_TYPES_IMPORT_ONLY_NEWER);

	/* Create new Dialog */
	d = MIME_TYPES_IMPORT_DLG(EDVDlgNew(
	    sizeof(MediaTypesImportDlg), width, height,
	    "Import MIME Types Options", (guint8 **)icon_import_32x32_xpm,
	    "Import", (guint8 **)icon_import_20x20_xpm,
	    "Cancel", (guint8 **)icon_cancel_20x20_xpm,
	    ref_toplevel,
	    core, path
	));
	if(d == NULL)
	    return(got_response);

	toplevel = DLG(d)->toplevel;
	accelgrp = DLG(d)->accelgrp;
	client_vbox = DLG(d)->client_vbox;
	style = gtk_widget_get_style(toplevel);

	/* Create the option widgets */
	if(client_vbox != NULL)
	{
	    gint font_size;
	    const gchar *font_encoding;
	    gchar *font_name_h1_bold;
	    GdkFont *font;
	    GtkStateType state;
	    GtkRcStyle *rcstyle;
	    GtkStyle *style = gtk_widget_get_style(toplevel);
	    GtkWidget *w, *parent = client_vbox, *parent2, *parent3;

	    /* Get the base font size */
	    font = style->font;
	    font_size = GDK_FONT_GET_FONT_NAME_SIZE(font);
	    if(font_size < 3)
		font_size = 3;

	    /* Format the font names */
#if defined(PROG_LANGUAGE_POLISH)
	    font_encoding = "iso8859-2";
#else
	    font_encoding = "iso8859-1";
#endif

            font_name_h1_bold = g_strdup_printf(
"-adobe-helvetica-bold-r-normal-*-%i-*-*-*-*-*-%s",
                font_size + 2,
                font_encoding
            );

	    /* OK button accel keys */
	    w = DLG(d)->ok_btn;
	    gtk_accel_group_add(
		accelgrp, GDK_i, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    GUIButtonLabelUnderline(w, GDK_i);

	    /* Cancel button accel keys */
	    w = DLG(d)->cancel_btn;
	    gtk_accel_group_add(
		accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    gtk_accel_group_add(
		accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
		GTK_OBJECT(w), "clicked"
	    );
	    GUIButtonLabelUnderline(w, GDK_c);

	    /* Heading */
	    if(!STRISEMPTY(path))
	    {
		w = gtk_frame_new(NULL);
		gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
		gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent2 = w;

		rcstyle = gtk_rc_style_new();
		state = GTK_STATE_NORMAL;
		rcstyle->color_flags[state] = GTK_RC_BG | GTK_RC_BASE;
		memcpy(
		    &rcstyle->bg[state],
		    &style->base[state],
		    sizeof(GdkColor)
		);
		memcpy(
		    &rcstyle->base[state],
		    &style->base[state],
		    sizeof(GdkColor)
		);
		w = gtk_event_box_new();
		gtk_container_add(GTK_CONTAINER(parent2), w);
		gtk_widget_modify_style(w, rcstyle);
		gtk_widget_show(w);
		GTK_RC_STYLE_UNREF(rcstyle);
		parent2 = w;

		w = gtk_vbox_new(FALSE, border_major);
		gtk_container_add(GTK_CONTAINER(parent2), w);
		gtk_container_border_width(GTK_CONTAINER(w), border_major);
		gtk_widget_show(w);
		parent2 = w;

		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;

		w = edv_new_pixmap_widget(
		    core,
		    (guint8 **)icon_import_32x32_xpm,
		    "icon_import_32x32_xpm"
		);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);

		rcstyle = gtk_rc_style_new();
		rcstyle->font_name = STRDUP(font_name_h1_bold);
		w = gtk_label_new(g_basename(path));
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_modify_style(w, rcstyle);
		GTK_RC_STYLE_UNREF(rcstyle);
		gtk_widget_show(w);

		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;
		w = gtk_label_new(
"To update your MIME Types with the ones from this file,\n\
check \"Update\". The MIME Types from this file that you\n\
do not have already will always be added as usual."
		);
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);

		w = gtk_hbox_new(FALSE, border_major);
		gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
		parent3 = w;
		w = gtk_label_new(
"To update only the MIME Types that are newer on this\n\
file than the ones you have, check \"Update\" and\n\
\"Only Newer\"."
		);
		gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
		gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
		gtk_widget_show(w);
	    }

	    /* GtkVBox for the option widgets */
	    w = gtk_vbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent2 = w;

	    /* Update */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* GtkCheckButton */
	    d->update_check = w = gtk_check_button_new_with_label(
		"Update"
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    GUISetWidgetTip(w,
"Check this to update MIME Types if they already exist and only\
 add new MIME Types if they do not exist, otherwise all MIME\
 Types will be add (which may result with duplicate MIME Types)"
	    );
	    gtk_widget_show(w);

	    /* Only Newer */
	    w = gtk_hbox_new(FALSE, border_minor);
	    gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	    gtk_widget_show(w);
	    parent3 = w;
	    /* GtkCheckButton */
	    d->only_newer_check = w = gtk_check_button_new_with_label(
		"Only Newer"
	    );
	    gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	    GUISetWidgetTip(w,
"Check this to only import the MIME Types that are newer than\
 the existing ones or if it does not already exist"
	    );
	    gtk_widget_show(w);


	    /* Begin setting initial values */

	    /* Update */
	    if(update != NULL)
		gtk_toggle_button_set_active(
		    GTK_TOGGLE_BUTTON(d->update_check), *update
		);
	    /* Only Newer */
	    if(only_newer != NULL)
		gtk_toggle_button_set_active(
		    GTK_TOGGLE_BUTTON(d->only_newer_check), *only_newer
		);

	    g_free(font_name_h1_bold);
	}

	/* Map the dialog */
	gtk_widget_show(toplevel);

	/* Wait for user response */
	gtk_main();

	/* Get user response */
	got_response = DLG(d)->got_response;

	/* Update returns if the user responded with ok */
	if(got_response)
	{
	    /* Update */
	    if(update != NULL)
	    { 
		*update = GTK_TOGGLE_BUTTON_GET_ACTIVE(
		    d->update_check
		);
		EDV_SET_B(
		    EDV_CFG_PARM_MIME_TYPES_IMPORT_UPDATE,
		    *update
		);
	    }
	    /* Only Newer */
	    if(only_newer != NULL)
	    {
		*only_newer = GTK_TOGGLE_BUTTON_GET_ACTIVE(
		    d->only_newer_check
		);
		EDV_SET_B(
		    EDV_CFG_PARM_MIME_TYPES_IMPORT_ONLY_NEWER,
		    *only_newer
		);
	    }
	}

	/* Delete the Dialog */
	EDVDlgDelete(DLG(d));

	return(got_response);
}
