#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include "url.h"
#include "cfg.h"

#include "guiutils.h"
#include "pulist.h"
#include "icon_picker.h"
#include "icon_sel_dlg.h"
#include "cdialog.h"
#include "fb.h"

#include "edv_types.h"
#include "libendeavour2-base/edv_utils.h"
#include "libendeavour2-base/edv_path.h"
#include "libendeavour2-base/edv_vfs_obj.h"
#include "libendeavour2-base/edv_vfs_obj_stat.h"
#include "edv_pixmap.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "edv_obj_info_match.h"
#include "edv_utils_gtk.h"
#include "edv_list_cb.h"
#include "mime_type_edit_dlg.h"
#include "edv_emit.h"
#include "endeavour2.h"

#include "edv_cfg_list.h"
#include "config.h"

#include "images/icon_add_20x20.xpm"
#include "images/icon_edit_20x20.xpm"
#include "images/icon_remove_20x20.xpm"
#include "images/icon_folder_opened_20x20.xpm"
#include "images/icon_ok_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_mimetypes_48x48.xpm"
#include "images/icon_file_extension_20x20.xpm"
#include "images/icon_terminal_run2_20x20.xpm"


typedef struct _CmdDlg			CmdDlg;
#define CMD_DLG(p)			((CmdDlg *)(p))


/* Callbacks */
static gint EDVMimeTypeEditDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void EDVMimeTypeEditDlgAnyChangedCB(GtkWidget *widget, gpointer data);
static void EDVMimeTypeEditDlgClassChangedCB(
	GtkWidget *widget, const gint i, gpointer data
);
static void EDVMimeTypeEditDlgHandlerChangedCB(
	 GtkWidget *widget, const gint i, gpointer data
);
static gint EDVMimeTypeEditDlgCommandsCListEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data 
);
static void EDVMimeTypeEditDlgCommandsCListDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void EDVMimeTypeEditDlgCommandsCListDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void EDVMimeTypeEditDlgCommandsCListDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
);
static void EDVMimeTypeEditDlgCommandsCListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void EDVMimeTypeEditDlgCommandsCListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void EDVMimeTypeEditDlgCommandAddCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgCommandEditCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgCommandRemoveCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgCommandShiftUpCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgCommandShiftDownCB(
	GtkWidget *widget, gpointer data
);

static void EDVMimeTypeEditDlgIconPickerIconSwitchedCB(
	GtkWidget *w,
	const gint icon_num, IconPickerIcon *icon,
	gpointer data
);

static void EDVMimeTypeEditDlgOKCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgApplyCB(
	GtkWidget *widget, gpointer data
);
static void EDVMimeTypeEditDlgCancelCB(
	GtkWidget *widget, gpointer data
);

void EDVMimeTypeEditDlgMimeTypeRemovedCB(
	edv_mime_type_edit_dlg_struct *d,
	const gint mt_num
);

/* Command Dialog Callbacks */
static gint CmdDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
);
static void CmdDlgChangedCB(GtkWidget *widget, gpointer data);
static void CmdDlgBrowseCommandCB(GtkWidget *widget, gpointer data);
static void CmdDlgBrowseShellCommandCB(GtkWidget *widget, gpointer data);
static void CmdDlgCListDragDataGet(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
);
static void CmdDlgCListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void CmdDlgCListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
);
static void CmdDlgTokensCB(GtkWidget *widget, gpointer data);
static void CmdDlgInsertTokenCB(GtkWidget *widget, gpointer data);
static void CmdDlgOKCB(GtkWidget *widget, gpointer data);
static void CmdDlgCancelCB(GtkWidget *widget, gpointer data);
static void CmdDlgCloseCB(GtkWidget *widget, gpointer data);

/* Utilities */
static EDVMIMETypeClass EDVMimeTypeEditDlgGetClass(edv_mime_type_edit_dlg_struct *d);
static EDVMIMETypeHandler EDVMimeTypeEditDlgGetHandler(edv_mime_type_edit_dlg_struct *d);
static gint EDVMimeTypeEditDlgCheckMIMEType(
	EDVCore *core,
	GList *mime_typeS_list,
	const gint mt_num,
	const gboolean verbose,
	GtkWidget *toplevel
);

/* Get/set values */
void EDVMimeTypeEditDlgGetValues(
	edv_mime_type_edit_dlg_struct *d,
	const gint mt_num
);
void EDVMimeTypeEditDlgSetValues(
	edv_mime_type_edit_dlg_struct *d,
	const gint mt_num,
	const gboolean verbose
);

/* Command Dialog */
static CmdDlg *CmdDlgNew(
	EDVCore *core,
	GtkWidget *ref_toplevel,
	EDVMIMETypeCommand *cmd,
	const gboolean is_new
);
static void CmdDlgDelete(CmdDlg *d);

/* Add, Edit, Remove, Shift Commands */
static void EDVMimeTypeEditDlgCommandAdd(edv_mime_type_edit_dlg_struct *d);
static gboolean EDVMimeTypeEditDlgCommandEdit(
	edv_mime_type_edit_dlg_struct *d,
	const gboolean is_new
);
static void EDVMimeTypeEditDlgCommandRemove(
	edv_mime_type_edit_dlg_struct *d,
	const gboolean confirm
);
static void EDVMimeTypeEditDlgCommandShiftUp(edv_mime_type_edit_dlg_struct *d);
static void EDVMimeTypeEditDlgCommandShiftDown(edv_mime_type_edit_dlg_struct *d);

/* Commands List & Items */
static gint EDVMimeTypeEditDlgCommandsListInsert(
	edv_mime_type_edit_dlg_struct *d,
	const gint cmd_num,
	EDVMIMETypeCommand *cmd
);
static gint EDVMimeTypeEditDlgCommandsListAppend(
	edv_mime_type_edit_dlg_struct *d,
	EDVMIMETypeCommand *cmd
);
static void EDVMimeTypeEditDlgCommandsListSet(
	edv_mime_type_edit_dlg_struct *d,
	const gint cmd_num,
	EDVMIMETypeCommand *cmd
);

/* MIME Type Edit Dialog */
edv_mime_type_edit_dlg_struct *EDVMimeTypeEditDlgNew(EDVCore *core);
void EDVMimeTypeEditDlgResetHasChanges(
	edv_mime_type_edit_dlg_struct *d,
	const gboolean has_changes
);
void EDVMimeTypeEditDlgUpdateMenus(edv_mime_type_edit_dlg_struct *d);
gboolean EDVMimeTypeEditDlgIsMapped(edv_mime_type_edit_dlg_struct *d);
void EDVMimeTypeEditDlgMap(edv_mime_type_edit_dlg_struct *d);
void EDVMimeTypeEditDlgUnmap(edv_mime_type_edit_dlg_struct *d);
void EDVMimeTypeEditDlgDelete(edv_mime_type_edit_dlg_struct *d);


/*
 *	Command Dialog:
 */
struct _CmdDlg {
	GtkWidget	*toplevel;
	GtkAccelGroup	*accelgrp;
	gint		freeze_count;
	gboolean	has_changes,
			got_response;
	EDVCore	*core;
	GtkWidget	*name_entry,
			*command_entry,
			*tokens_btn,
			*tokens_box,
			*tokens_clist,
			*tokens_insert_btn,
			*shell_command_entry,
			*run_in_terminal_check,
			*ok_btn,
			*cancel_btn,
			*close_btn;
};


#define EDV_MIME_TYPE_EDIT_DLG_TITLE	"MIME Type"

#define EDV_MIME_TYPE_EDIT_DLG_WIDTH	500
#define EDV_MIME_TYPE_EDIT_DLG_HEIGHT	-1


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	GtkWindow "delete_event" signal callback.
 */
static gint EDVMimeTypeEditDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return(TRUE);

	EDVMimeTypeEditDlgCancelCB(widget, d);

	return(TRUE);
}

/*
 *	Any GtkWidget "changed" signal callback.
 */
static void EDVMimeTypeEditDlgAnyChangedCB(GtkWidget *widget, gpointer data)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	if(!d->has_changes)
	    EDVMimeTypeEditDlgResetHasChanges(d, TRUE);

	d->freeze_count--;
}

/*
 *	Class popup list box "changed" signal callback.
 *
 *	This will update the value_label and value_entry widgets.
 */
static void EDVMimeTypeEditDlgClassChangedCB(
	 GtkWidget *widget, const gint i, gpointer data
)
{
	gboolean sensitive = TRUE;
	const gchar *value_label_str = "";
	GtkWidget *pulist;
	EDVMIMETypeClass mt_class;
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((widget == NULL) || (d == NULL))
	    return;

	pulist = PUListBoxGetPUList(widget);
	if(pulist == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	/* Get the new class */
	mt_class = (EDVMIMETypeClass)PUListGetItemData(pulist, i);

	/* Update the widgets depending on class */
	switch(mt_class)
	{
	  case EDV_MIME_TYPE_CLASS_SYSTEM:
	    sensitive = FALSE;
	    value_label_str =
#if defined(PROG_LANGUAGE_SPANISH)
"El Valor"
#elif defined(PROG_LANGUAGE_FRENCH)
"Valeur"
#elif defined(PROG_LANGUAGE_GERMAN)
"Wert"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Valore"
#elif defined(PROG_LANGUAGE_DUTCH)
"Waarde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Valor"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Verdi"
#else
"Value"
#endif
		":";
	    break;

	  case EDV_MIME_TYPE_CLASS_FORMAT:
	    sensitive = TRUE;
	    value_label_str =
#if defined(PROG_LANGUAGE_SPANISH)
"Las Extensiones"
#elif defined(PROG_LANGUAGE_FRENCH)
"Extensions"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verlngerungen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Le Estensioni"
#elif defined(PROG_LANGUAGE_DUTCH)
"Uitbreidingen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"As Extenses"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Utvidelser"
#else
"Extensions"
#endif
		":";
	    break;

	  case EDV_MIME_TYPE_CLASS_PROGRAM:
	    sensitive = TRUE;
	    value_label_str =
#if defined(PROG_LANGUAGE_SPANISH)
"La Ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ort"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plaats"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Plassering"
#else
"Location"
#endif
		":";
	    break;

	  case EDV_MIME_TYPE_CLASS_UNIQUE:
	    sensitive = TRUE;
	    value_label_str =
#if defined(PROG_LANGUAGE_SPANISH)
"La Ubicacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Emplacement"
#elif defined(PROG_LANGUAGE_GERMAN)
"Ort"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Posizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Plaats"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Localidade"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Plassering"
#else
"Location"
#endif
		":";
	    break;
	}

	gtk_label_set_text(GTK_LABEL(d->value_label), value_label_str);

	gtk_widget_set_sensitive(d->value_entry, sensitive);

	gtk_widget_set_sensitive(d->handler_pulistbox, sensitive);

	gtk_widget_set_sensitive(d->commands_clist_box, sensitive);

	d->freeze_count--;

	EDVMimeTypeEditDlgAnyChangedCB(widget, data);
}

/*
 *	Handler popup list box "changed" signal callback.
 *
 *	This will update the sensitivity or mapping of some widgets
 *	such as the command editable clist.
 */
static void EDVMimeTypeEditDlgHandlerChangedCB(
	GtkWidget *widget, const gint i, gpointer data
)
{
	gboolean sensitive = FALSE;
	GtkWidget *pulist;
	EDVMIMETypeClass mt_class;
	EDVMIMETypeHandler handler;
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((widget == NULL) || (d == NULL))
	    return;

	pulist = PUListBoxGetPUList(widget);
	if(pulist == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->freeze_count++;

	/* Get the current class and handler values */
	mt_class = EDVMimeTypeEditDlgGetClass(d);
	handler = (EDVMIMETypeHandler)PUListGetItemData(pulist, i);

	/* Update widgets depending on class */
	switch(handler)
	{
	  case EDV_MIME_TYPE_HANDLER_COMMAND:
	    switch(mt_class)
	    {
	      case EDV_MIME_TYPE_CLASS_SYSTEM:
		sensitive = FALSE;
		break;
	      case EDV_MIME_TYPE_CLASS_FORMAT:
	      case EDV_MIME_TYPE_CLASS_PROGRAM:
	      case EDV_MIME_TYPE_CLASS_UNIQUE:
		sensitive = TRUE;
		break;
	    }
	    break;
	  case EDV_MIME_TYPE_HANDLER_EDV_ARCHIVER:
	  case EDV_MIME_TYPE_HANDLER_EDV_IMAGE_BROWSER:
	  case EDV_MIME_TYPE_HANDLER_EDV_RECYCLE_BIN:
	    sensitive = FALSE;
	    break;
	}

	gtk_widget_set_sensitive(d->commands_clist_box, sensitive);

	d->freeze_count--;

	EDVMimeTypeEditDlgAnyChangedCB(widget, data);
}

/*
 *	Commands GtkCList event signal callback.
 */
static gint EDVMimeTypeEditDlgCommandsCListEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data 
)
{
	gint status = FALSE;
	gboolean key_press;
	gint row, column;
	GdkEventKey *key;
	GdkEventButton *button;
	GtkCList *clist;
	CfgItem *cfg_list;
	EDVCore *core;
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((widget == NULL) || (event == NULL) || (d == NULL))
	    return(status);

	if(d->freeze_count > 0)
	    return(status);

	clist = GTK_CLIST(widget);
	core = d->core;
	cfg_list = core->cfg_list;
			   
	switch((gint)event->type)
	{
	  case GDK_KEY_PRESS:
	  case GDK_KEY_RELEASE:
	    key = (GdkEventKey *)event;
	    key_press = (key->type == GDK_KEY_PRESS) ? TRUE : FALSE;
	    switch(key->keyval)
	    {
	      case GDK_Return:
		if(key_press)
		    EDVMimeTypeEditDlgCommandEdit(d, FALSE);
		status = TRUE;
		break;

	      case GDK_Delete:
		if(key_press)
		    EDVMimeTypeEditDlgCommandRemove(d, TRUE);
		status = TRUE;
		break;
	    }
	    break;

	  case GDK_BUTTON_PRESS:
	    button = (GdkEventButton *)event;
	    switch(button->button)
	    {
	      case GDK_BUTTON3:
		/* Find the row and column that this button press
		 * occured over
		 */
		if(!gtk_clist_get_selection_info(
		    clist,
		    button->x, button->y,
		    &row, &column
		))
		{
		    row = -1;
		    column = 0;
		}

		/* Select before mapping the menu? */
		if(EDV_GET_B(EDV_CFG_PARM_RIGHT_CLICK_MENU_SELECTS) &&
		   (row >= 0) && (row < clist->rows)
		)
		{
		    gtk_clist_freeze(clist);
		    gtk_clist_select_row(clist, row, column);
		    gtk_clist_thaw(clist);
		}

		/* Update the menus and then map the right-click menu */
		EDVMimeTypeEditDlgUpdateMenus(d);
		gtk_menu_popup(
		    GTK_MENU(d->commands_menu),
		    NULL, NULL,
		    NULL, NULL,
		    button->button, button->time
		);
		status = TRUE;
		break;
	    }
	    break;

	  case GDK_BUTTON_RELEASE:
	    break;
	}

	return(status);
}


/*
 *	Commands GtkCList "drag_data_get" signal callback.
 */
static void EDVMimeTypeEditDlgCommandsCListDragDataGetCB(
	GtkWidget *widget, GdkDragContext *dc,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gboolean data_sent = FALSE;
	gint row;
	GList *glist, *cmds_list;
	GtkCList *clist;
	EDVMIMETypeCommand *cmd;
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((dc == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	clist = GTK_CLIST(d->commands_clist);

	/* Create a list of selected commands */
	cmds_list = NULL;
	for(glist = clist->selection;
	    glist != NULL;
	    glist = g_list_next(glist)
	)                             
	{
	    row = (gint)glist->data;
	    cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, row));
	    if(cmd == NULL)
		continue;

	    cmds_list = g_list_append(
		cmds_list,
		cmd				/* Reference */
	    );
	}                                           
	if(cmds_list != NULL)
	{
	    gint buf_len;
	    guint8 *buf = edv_mime_type_commands_list_encode_buffer(
		cmds_list, &buf_len
	    );

	    g_list_free(cmds_list);

	    if(buf != NULL)
	    {
		gtk_selection_data_set(
		    selection_data,
		    GDK_SELECTION_TYPE_STRING,
		    8,				/* Bits Per Character */
		    buf,			/* Data */
		    buf_len			/* Length */
		);
		data_sent = TRUE;
		g_free(buf);
	    }
	}

	if(!data_sent)
	{
	    const gchar *s = "Error";
	    gtk_selection_data_set(
		selection_data,
		GDK_SELECTION_TYPE_STRING,
		8,				/* Bits Per Character */
		s,				/* Data */
		STRLEN(s)			/* Length */
	    );
	}
}

/*
 *	Commands GtkCList "drag_data_received" signal callback.
 */
static void EDVMimeTypeEditDlgCommandsCListDragDataReceivedCB(
	GtkWidget *widget, GdkDragContext *dc,
	gint x, gint y,
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data
)
{
	gint row = -1, column = 0;
	GtkCList *clist;
	EDVCore *core;
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((dc == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	clist = GTK_CLIST(d->commands_clist);
	core = d->core;

	/* Find the dropped on row and column */
	gtk_clist_get_selection_info(
	    clist,
	    x,    
	    y - ((clist->flags & GTK_CLIST_SHOW_TITLES) ?
		clist->column_title_area.height +
		clist->column_title_area.y : 0),
	    &row, &column
	);

	/* Handle by the target type
	 *
	 * MIME Type Command
	 */
	if(info == EDV_DND_INFO_MIME_TYPE_COMMAND)
	{
	    /* Get the commands list and insert it to the GtkCList */
	    gint	insert_row = row,
			new_row;
	    GList	*glist,
			*cmds_list = edv_mime_type_commands_list_decode_buffer(
		selection_data->data, selection_data->length
	    );
	    EDVMIMETypeCommand *cmd;

	    /* Insert each command to the GtkCList */
	    gtk_clist_freeze(clist);
	    for(glist = cmds_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		cmd = EDV_MIME_TYPE_COMMAND(glist->data);
		if(cmd == NULL)
		    continue;

		if(insert_row > -1)
		{
		    new_row = EDVMimeTypeEditDlgCommandsListInsert(
			d,
			insert_row,
			cmd			/* Transfered */
		    );
		    glist->data = NULL;
		    insert_row++;
		}
		else
		{
		    new_row = EDVMimeTypeEditDlgCommandsListAppend(
			d,
			cmd			/* Transfered */
		    );
		    glist->data = NULL;
		}
		if(new_row < 0)
		    break;
	    }
	    gtk_clist_thaw(clist);

	    /* Delete the parsed commands list */
	    if(cmds_list != NULL)
	    {
		g_list_foreach(
		    cmds_list, (GFunc)edv_mime_type_command_delete, NULL
		);
		g_list_free(cmds_list);
	    }

	    EDVMimeTypeEditDlgAnyChangedCB(widget, data);
	    EDVMimeTypeEditDlgUpdateMenus(d);
	}
	/* MIME Type */
	else if(info == EDV_DND_INFO_MIME_TYPE)
	{
	    /* Get the commands list and insert it to the GtkCList */
	    gint	insert_row = row,
			new_row;
	    GList	*glist,
			*mt_list = edv_mime_types_list_decode_buffer(
		core->mime_types_list,
		selection_data->data, selection_data->length
	    );
	    EDVMIMEType *m;
	    EDVMIMETypeCommand *cmd;

	    /* Iterate through the parsed MIME Types list */
	    gtk_clist_freeze(clist);
	    for(glist = mt_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
		    continue;

		/* Create a new command */
		cmd = edv_mime_type_command_new();
		if((clist->rows == 0) || (insert_row == 0))
		    cmd->name = STRDUP("Default");
		else
		    cmd->name = g_strdup_printf(
			"Command #%i",
			(insert_row > -1) ?
			    (insert_row + 1) : (clist->rows + 1)
		    );
		cmd->command = STRDUP(m->type);
/*		cmd->shell_command = NULL; */	/* No shell command */

		/* Insert the new command to the GtkCList */
		if(insert_row > -1)
		{
		    new_row = EDVMimeTypeEditDlgCommandsListInsert(
			d,
			insert_row,
			cmd			/* Transfered */
		    );
		    cmd = NULL;
		    insert_row++;
		}
		else
		{
		    new_row = EDVMimeTypeEditDlgCommandsListAppend(
			d,
			cmd			/* Transfered */
		    );
		    cmd = NULL;
		}
		if(new_row < 0)
		    break;
	    }
	    gtk_clist_thaw(clist);

	    /* Delete the parsed MIME Types list */
	    if(mt_list != NULL)
	    {
		g_list_foreach(
		    mt_list, (GFunc)edv_mime_type_delete, NULL
		);
		g_list_free(mt_list);
	    }

	    EDVMimeTypeEditDlgAnyChangedCB(widget, data);
	    EDVMimeTypeEditDlgUpdateMenus(d);
	}
	/* String */
	else if((info == EDV_DND_INFO_TEXT_PLAIN) ||
		(info == EDV_DND_INFO_TEXT_URI_LIST) ||
		(info == EDV_DND_INFO_STRING)
	)
	{
	    /* Get the commands list and insert it to the GtkCList */
	    gint	insert_row = row,
			new_row;
	    GList	*glist,
			*urls_list = url_decode(
		(const guint8 *)selection_data->data,
		selection_data->length
	    );
	    URLStruct *url;
	    EDVMIMETypeCommand *cmd;

	    /* Iterate through the parsed URLs list */
	    gtk_clist_freeze(clist);
	    for(glist = urls_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		url = URL(glist->data);
		if(url == NULL)
		    continue;

		/* Create a new command */
		cmd = edv_mime_type_command_new();
		if((clist->rows == 0) || (insert_row == 0))
		    cmd->name = STRDUP("Default");
		else
		    cmd->name = g_strdup_printf(
			"Command #%i",
			(insert_row > -1) ?
			    (insert_row + 1) : (clist->rows + 1)
		    );
		cmd->command = g_strconcat(
		    url->path,
		    " \"%p\"",
		    NULL
		);
/*		cmd->shell_command = NULL; */	/* No shell command */

		/* Insert the new command to the GtkCList */
		if(insert_row > -1)
		{
		    new_row = EDVMimeTypeEditDlgCommandsListInsert(
			d,
			insert_row,
			cmd			/* Transfered */
		    );
		    cmd = NULL;
		    insert_row++;
		}
		else
		{
		    new_row = EDVMimeTypeEditDlgCommandsListAppend(
			d,
			cmd			/* Transfered */
		    );
		    cmd = NULL;
		}
		if(new_row < 0)
		    break;
	    }
	    gtk_clist_thaw(clist);

	    /* Delete the URLs list */
	    if(urls_list != NULL)
	    {
		g_list_foreach(
		    urls_list, (GFunc)url_delete, NULL
		);
		g_list_free(urls_list);
	    }

	    EDVMimeTypeEditDlgAnyChangedCB(widget, data);
	    EDVMimeTypeEditDlgUpdateMenus(d);
	}
}

/*
 *	Commands GtkCList "drag_data_delete" signal callback.
 */
static void EDVMimeTypeEditDlgCommandsCListDragDataDeleteCB(
	GtkWidget *widget, GdkDragContext *dc, gpointer data
)
{
	GtkCList *clist;
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((dc == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	clist = GTK_CLIST(d->commands_clist);

	/* Delete the selected commands */
	gtk_clist_freeze(clist);
	while(clist->selection != NULL)
	    gtk_clist_remove(
		clist,
		(gint)clist->selection->data
	    );
	gtk_clist_columns_autosize(clist);
	gtk_clist_thaw(clist);
}

/*
 *	Commands GtkCList "select_row" signal callback.
 */
static void EDVMimeTypeEditDlgCommandsCListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((clist == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	/* If the selected row is fully visible then scroll to it */
	if(gtk_clist_row_is_visible(clist, row) !=
	   GTK_VISIBILITY_FULL
	)
	    gtk_clist_moveto(
		clist,
		row, -1,			/* Row, column */
		0.5f, 0.0f			/* Row, column */
	    );

	/* Set the DND drag icon */
	GUIDNDSetDragIconFromCListSelection(clist);

	EDVMimeTypeEditDlgUpdateMenus(d);

	/* Double click? */
	if(event != NULL)
	{
	    if(event->type == GDK_2BUTTON_PRESS)
	    {
		/* Edit */
		EDVMimeTypeEditDlgCommandEdit(
		    d,
		    FALSE				/* Not new */
		);
	    }
	}
}

/*
 *	Commands GtkCList "unselect_row" signal callback.
 */
static void EDVMimeTypeEditDlgCommandsCListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((clist == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgUpdateMenus(d);
}


/*
 *	Command add callback.
 */
static void EDVMimeTypeEditDlgCommandAddCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;
	
	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgCommandAdd(d);
}

/*
 *	Command edit callback.
 */
static void EDVMimeTypeEditDlgCommandEditCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;
	
	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgCommandEdit(d, FALSE);
}

/*
 *	Command remove callback.
 */
static void EDVMimeTypeEditDlgCommandRemoveCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;
	
	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgCommandRemove(d, TRUE);
}

/*
 *	Command shift up callback.
 */
static void EDVMimeTypeEditDlgCommandShiftUpCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;
	
	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgCommandShiftUp(d);
}

/*
 *	Command shift down callback.
 */
static void EDVMimeTypeEditDlgCommandShiftDownCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;
	
	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgCommandShiftDown(d);
}


/*
 *	Icon Picker icon switched signal callback.
 */
static void EDVMimeTypeEditDlgIconPickerIconSwitchedCB(
	GtkWidget *w,
	const gint icon_num, IconPickerIcon *icon,
	gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if((w == NULL) || (d == NULL))
	    return;

}


/*
 *	OK Callback.
 */
static void EDVMimeTypeEditDlgOKCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgSetValues(
	    d,
	    d->mt_num,
	    TRUE				/* Verbose */
	);

	EDVMimeTypeEditDlgUnmap(d);
}

/*
 *	Apply callback.
 */
static void EDVMimeTypeEditDlgApplyCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgSetValues(
	    d,
	    d->mt_num,
	    TRUE				/* Verbose */
	);
}

/*
 *	Cancel callback.
 */
static void EDVMimeTypeEditDlgCancelCB(
	GtkWidget *widget, gpointer data
)
{
	edv_mime_type_edit_dlg_struct *d = EDV_MIME_TYPE_EDIT_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	EDVMimeTypeEditDlgUnmap(d);
}


/*
 *      Notifies the given MIME Types list window that the given MIME Type
 *      has been removed.
 */
void EDVMimeTypeEditDlgMimeTypeRemovedCB(
	edv_mime_type_edit_dlg_struct *d,
	const gint mt_num
)
{
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	/* If the removed MIME Type is the MIME Type that this edit window
	 * is displaying values for then this edit window needs to be
	 * unmapped
	 */
	if(d->mt_num == mt_num)
	{
	    d->mt_num = -1;
	    EDVMimeTypeEditDlgUnmap(d);
	}
}


/*
 *	Command dialog "delete_event" signal callback.
 */
static gint CmdDlgDeleteEventCB(
	GtkWidget *widget, GdkEvent *event, gpointer data
)
{
	CmdDlgCloseCB(widget, data);
	return(TRUE);
}

/*
 *	Command dialog GtkCList "drag_data_get" signal callback.
 */
static void CmdDlgCListDragDataGet(
	GtkWidget *widget, GdkDragContext *dc,           
	GtkSelectionData *selection_data, guint info, guint t,
	gpointer data                                         
)
{
	gint row;
	gchar *s;
	GList *glist;
	GtkCList *clist;
	CmdDlg *d = CMD_DLG(data);
	if((widget == NULL) || (dc == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	clist = GTK_CLIST(d->tokens_clist);
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;

	s = g_strconcat(
	    "\"",
	    (const gchar *)gtk_clist_get_row_data(clist, row),
	    "\"",
	    NULL
	);
	if(s == NULL)
	    s = STRDUP(" ");

	gtk_selection_data_set(
	    selection_data,
	    GDK_SELECTION_TYPE_STRING,
	    8,				/* Bits Per Character */
	    s,				/* Data */
	    STRLEN(s)			/* Length */
	);

	g_free(s);
}

/*
 *	Command dialog GtkCList "select_row" signal callback.
 */
static void CmdDlgCListSelectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	CmdDlg *d = CMD_DLG(data);
	if((clist == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	/* If the selected row is fully visible then scroll to it */
	if(gtk_clist_row_is_visible(clist, row) !=
	   GTK_VISIBILITY_FULL
	)
	    gtk_clist_moveto(
		clist,
		row, -1,			/* Row, column */
		0.5f, 0.0f			/* Row, column */
	    );

	/* Set the DND drag icon */
	GUIDNDSetDragIconFromCListSelection(clist);

	/* Double click? */
	if(event != NULL)
	{
	    if(event->type == GDK_2BUTTON_PRESS)
	    {
		/* Insert the selected character into the command */
		const gchar *token = (gchar *)gtk_clist_get_row_data(clist, row);
		if(token != NULL)
		{
		    GtkEditable *editable = GTK_EDITABLE(d->command_entry);
		    gint position = gtk_editable_get_position(editable);
		    gchar *s = g_strconcat(
			"\"", token, "\"", NULL
		    );
		    gtk_editable_insert_text(
			editable,
			s,
			STRLEN(s),
			&position
		    );
		    g_free(s);
		}
	    }
	}

	gtk_widget_set_sensitive(d->tokens_insert_btn, TRUE);
}

/*
 *	Command dialog GtkCList "unselect_row" signal callback.
 */
static void CmdDlgCListUnselectRowCB(
	GtkCList *clist, gint row, gint column, GdkEvent *event,
	gpointer data
)
{
	CmdDlg *d = CMD_DLG(data);
	if((clist == NULL) || (d == NULL))
	    return;

	if(d->freeze_count > 0)
	    return;

	gtk_widget_set_sensitive(d->tokens_insert_btn, FALSE);
}

/*
 *	Command dialog any GtkWidget "changed" signal callback.
 */
static void CmdDlgChangedCB(GtkWidget *widget, gpointer data)
{
	CmdDlg *d = CMD_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	if(!d->has_changes)
	{
	    gtk_widget_set_sensitive(d->ok_btn, TRUE);
	    gtk_widget_hide(d->close_btn);
	    gtk_widget_show(d->cancel_btn);
	    d->has_changes = TRUE;
	}
}

/*
 *	Command dialog browse command callback.
 */
static void CmdDlgBrowseCommandCB(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gint nftypes = 0, npaths = 0;
	gchar *cur_path = NULL, **paths_list = NULL;
	const gchar *cur_cmd;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	CmdDlg *d = CMD_DLG(data);
	if((d == NULL) || FileBrowserIsQuery())
	    return;

	if(d->freeze_count > 0)
	    return;

	toplevel = d->toplevel;

	/* Get the current command */
	cur_cmd = gtk_entry_get_text(GTK_ENTRY(d->command_entry));
	if(cur_cmd != NULL)
	{
	    /* Does it start with an absolute path? */
	    if(*cur_cmd == G_DIR_SEPARATOR)
	    {
		/* Get the first argument, which should specify the
		 * program
		 */
		gchar *s;
		(void)edv_strarg(
		    cur_cmd,
		    &s,
		    FALSE,			/* Do not parse escapes */
		    TRUE			/* Parse quotes */
		);
		if(s != NULL)
		{
		    /* Get the parent directory of the first argument */
		    cur_path = g_dirname(s);
		    g_free(s);
		}
	    }
	}
	/* Ise the default path? */
	if(STRISEMPTY(cur_path))
	{
	    g_free(cur_path);
	    cur_path = g_strdup(PATH_USR_BIN);
	}

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for the new location */
	d->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Program",
	    "Select", "Cancel",
	    cur_path,
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	g_free(cur_path);
	FileBrowserSetTransientFor(NULL);
	d->freeze_count--;

	/* Got user response? */
	if(response)
	{
	    const gchar *path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(path))
	    {
		gtk_entry_set_text(
		    GTK_ENTRY(d->command_entry),
		    path
		);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);
}

/*
 *	Command dialog browse shell command callback.
 */
static void CmdDlgBrowseShellCommandCB(GtkWidget *widget, gpointer data)
{
	gboolean response;
	gint nftypes = 0, npaths = 0;
	gchar *cur_path = NULL, **paths_list = NULL;
	const gchar *cur_cmd;
	GtkWidget *toplevel;
	fb_type_struct **ftypes_list = NULL, *ftype_rtn = NULL;
	CmdDlg *d = CMD_DLG(data);
	if((d == NULL) || FileBrowserIsQuery())
	    return;

	if(d->freeze_count > 0)
	    return;

	toplevel = d->toplevel;

	/* Get the current shell command */
	cur_cmd = gtk_entry_get_text(GTK_ENTRY(d->shell_command_entry));
	if(cur_cmd != NULL)
	{
	    /* Does it start with an absolute path? */
	    if(*cur_cmd == G_DIR_SEPARATOR)
	    {
		/* Get the first argument, which should specify the
		 * program
		 */
		gchar *s;
		(void)edv_strarg(
		    cur_cmd,
		    &s,
		    FALSE,			/* Do not parse escapes */
		    TRUE			/* Parse quotes */
		);
		if(s != NULL)
		{
		    /* Get the parent directory of the first argument */
		    cur_path = g_dirname(s);
		    g_free(s);
		}
	    }
	}
	/* Ise the default path? */
	if(STRISEMPTY(cur_path))
	{
	    g_free(cur_path);
	    cur_path = g_strdup(PATH_BIN);
	}

	/* Create the file types list */
	FileBrowserTypeListNew(
	    &ftypes_list, &nftypes,
	    "*.*", "All Files"
	);

	/* Query the user for the new location */
	d->freeze_count++;
	FileBrowserSetTransientFor(toplevel);
	response = FileBrowserGetResponse(
	    "Select Program",
	    "Select", "Cancel",
	    cur_path,
	    ftypes_list, nftypes,
	    &paths_list, &npaths,
	    &ftype_rtn
	);
	g_free(cur_path);
	FileBrowserSetTransientFor(NULL);
	d->freeze_count--;

	/* Got user response? */
	if(response)
	{
	    const gchar *path = (npaths > 0) ?
		paths_list[0] : NULL;
	    if(!STRISEMPTY(path))
	    {
		gtk_entry_set_text(
		    GTK_ENTRY(d->shell_command_entry),
		    path
		);
	    }
	}

	/* Delete the file types list */
	FileBrowserDeleteTypeList(ftypes_list, nftypes);
}

/*
 *	Command dialog tokens callback.
 */
static void CmdDlgTokensCB(GtkWidget *widget, gpointer data)
{
	GtkWidget *w;
	CmdDlg *d = CMD_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	w = d->tokens_box;
	if(GTK_WIDGET_MAPPED(w))
	{
	    GUIButtonPixmapUpdate(
		d->tokens_btn,
		NULL, "Tokens >>>"
	    );
	    gtk_widget_hide(w);
	}
	else
	{
	    GUIButtonPixmapUpdate(
		d->tokens_btn,
		NULL, "Tokens <<<"
	    );
	    gtk_widget_show(w);
	}
}

/*
 *	Command dialog insert token callback.
 */
static void CmdDlgInsertTokenCB(GtkWidget *widget, gpointer data)
{
	gint row;
	const gchar *token;
	GList *glist;
	GtkCList *clist;
	CmdDlg *d = CMD_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	clist = GTK_CLIST(d->tokens_clist);
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;
	token = (gchar *)gtk_clist_get_row_data(clist, row);
	if(token != NULL)
	{
	    GtkEditable *editable = GTK_EDITABLE(d->command_entry);
	    gint position = gtk_editable_get_position(editable);
	    gchar *s = g_strconcat(
		"\"", token, "\"", NULL
	    );
	    gtk_editable_insert_text(
		editable,
		s,
		STRLEN(s),
		&position
	    );
	    g_free(s);
	}
}

/*
 *	Command dialog ok callback.
 */
static void CmdDlgOKCB(GtkWidget *widget, gpointer data)
{
	CmdDlg *d = CMD_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->got_response = TRUE;
	gtk_main_quit();
}

/*
 *	Command dialog cancel callback.
 */
static void CmdDlgCancelCB(GtkWidget *widget, gpointer data)
{
	CmdDlg *d = CMD_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->got_response = FALSE;
	gtk_main_quit();
}

/*
 *	Command dialog close callback.
 */
static void CmdDlgCloseCB(GtkWidget *widget, gpointer data)
{
	CmdDlg *d = CMD_DLG(data);
	if(d == NULL)
	    return;

	if(d->freeze_count > 0)
	    return;

	d->got_response = FALSE;
	gtk_main_quit();
}


/*
 *	Returns the current MIME Type class value set on the dialog.
 */
static EDVMIMETypeClass EDVMimeTypeEditDlgGetClass(edv_mime_type_edit_dlg_struct *d)
{
	if(d == NULL)
	    return(EDV_MIME_TYPE_CLASS_SYSTEM);

	return((EDVMIMETypeClass)PUListBoxGetSelectedData(
	    d->class_pulistbox
	));
}

/*
 *	Returns the current MIME Type handler value set on the dialog.
 */
static EDVMIMETypeHandler EDVMimeTypeEditDlgGetHandler(edv_mime_type_edit_dlg_struct *d)
{
	if(d == NULL)
	    return(EDV_MIME_TYPE_HANDLER_COMMAND);

	return((EDVMIMETypeHandler)PUListBoxGetSelectedData(
	    d->handler_pulistbox
	));
}

/*
 *	Checks all of the specified MIME Type's values to see if they
 *	are any problems
 *
 *	Returns non-zero if there was a problem.
 */
static gint EDVMimeTypeEditDlgCheckMIMEType(
	EDVCore *core,
	GList *mime_types_list,
	const gint mt_num,
	const gboolean verbose,
	GtkWidget *toplevel
)
{
	gint status = 0;
	const gchar *type, *val, *desc;
	EDVMIMETypeClass mt_class;
	EDVMIMETypeHandler handler;
	const gchar *title =
#if defined(PROG_LANGUAGE_SPANISH)
"MIME Type Advertencia"
#elif defined(PROG_LANGUAGE_FRENCH)
"MIME Type Avertissement"
#elif defined(PROG_LANGUAGE_GERMAN)
"MIME Type Warnen"
#elif defined(PROG_LANGUAGE_ITALIAN)
"MIME Type L'avvertimento"
#elif defined(PROG_LANGUAGE_DUTCH)
"MIME Type Waarschuwen"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"MIME Type Aviso"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"MIME Type Varsling"
#else
"MIME Type Warning"
#endif
	;
	EDVMIMEType *mt;

	if((core == NULL) || (mime_types_list == NULL))
	    return(status);

	mt = EDV_MIME_TYPE(g_list_nth_data(
	    mime_types_list,
	    (guint)mt_num
	));
	if(mt == NULL)
	{
	    status = -1;
	    if(verbose)
	    {
		gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"MIME Type con el ndice #%i no est en la lista"
#elif defined(PROG_LANGUAGE_FRENCH)
"MIME Type avec l'index %i n'est pas dans la liste"
#elif defined(PROG_LANGUAGE_GERMAN)
"MIME Type mit index #%i ist nicht in der liste"
#elif defined(PROG_LANGUAGE_ITALIAN)
"MIME Type con #%i di indice non  nell'elenco"
#elif defined(PROG_LANGUAGE_DUTCH)
"MIME Type met index #%i is niet in de lijst"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"MIME Type com #%i de ndice no est na lista"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"MIME Type med indeks #%i er ikke i listen"
#else
"The MIME Type with index #%i is not in the list"
#endif
		    , mt_num
		);
		edv_play_sound_error(core);
		edv_message_error(title, msg, NULL, toplevel);
		g_free(msg);
	    }
	    return(status);
	}

	/* Get MIME Type's values */
	mt_class = mt->mt_class;
	type = mt->type;
	val = mt->value;
	desc = mt->description;
	handler = mt->handler;

	/* Check the MIME Type's type and value
	 *
	 * Handle by the MIME Type class
	 */
	switch(mt_class)
	{
	  case EDV_MIME_TYPE_CLASS_SYSTEM:
	    /* Is the Type defined? */
	    if(STRISEMPTY(type))
	    {
		status = -2;
		if(verbose)
		{
		    gchar *msg = STRDUP(
"This MIME Type's Type was not set, there will be no way to\n\
identify it."
		    );
		    edv_play_sound_warning(core);
		    edv_message_warning(title, msg, NULL, toplevel);
		    g_free(msg);
		}
	    }
	    /* Value is ignored */
	    break;

	  case EDV_MIME_TYPE_CLASS_FORMAT:
	    /* Is the Type defined? */
	    if(STRISEMPTY(type))
	    {
		status = -2;
		if(verbose)
		{
		    gchar *msg = STRDUP(
"This MIME Type's Type was not set, there will be no way to\n\
identify it."
		    );
		    edv_play_sound_warning(core);
		    edv_message_warning(title, msg, NULL, toplevel);
		    g_free(msg);
		}
	    }
	    /* Is the Value defined? */
	    if(STRISEMPTY(val))
	    {
		status = -2;
		if(verbose)
		{
		    gchar *msg = STRDUP(
"This MIME Type's Value was not set, this MIME Type will not\n\
identify any formats."
		    );
		    edv_play_sound_warning(core);
		    edv_message_warning(title, msg, NULL, toplevel);
		    g_free(msg);
		}
	    }
	    /* Check the handler
	     *
	     * Command?
	     */
	    if(handler == EDV_MIME_TYPE_HANDLER_COMMAND)
	    {
		const gchar *name, *command;
		GList *glist;
		EDVMIMETypeCommand *cmd;

		/* Check all the commands */
		for(glist = mt->commands_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
		    cmd = EDV_MIME_TYPE_COMMAND(glist->data);
		    if(cmd == NULL)
			continue;

		    name = cmd->name;
		    command = cmd->command;
		    if(STRISEMPTY(command))
			continue;

		    /* Reference to another MIME Type? */
		    if(*command != G_DIR_SEPARATOR)
		    {
			/* Check if the referenced MIME Type exists */
			EDVMIMEType *m_ref = edv_mime_types_list_match_type(
			    mime_types_list,
			    NULL,
			    command,
			    TRUE		/* Case sensitive */
			);
			if(m_ref == NULL)
			{
			    /* Referenced MIME Type was not found */
			    status = -2;
			    if(verbose)
			    {
			        gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Ordene reffers de \"%s\" a un MIME Type\n\
inexistente llam \"%s\"."
#elif defined(PROG_LANGUAGE_FRENCH)
"la commande \"%s\" fait rfrence  un MIME Type\n\
inexistant nomm \"%s\"."
#elif defined(PROG_LANGUAGE_GERMAN)
"Befehlen Sie \"%s\" reffers zu einem nicht\n\
vorhandenen MIME Type hat gerufen \"%s\"."
#elif defined(PROG_LANGUAGE_ITALIAN)
"Comandare il reffers di \"%s\" a un MIME Type\n\
non-esistente ha chiamato \"%s\"."
#elif defined(PROG_LANGUAGE_DUTCH)
"Beveel aan een onbestaande MIME Type\n\
\"%s\" reffers riep \"%s\"."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Comande reffers de \"%s\" a um MIME Type\n\
inexistente chamou \"%s\"."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kommander \"%s\" reffers til en\n\
ikke-eksisterende MIME Type kalt \"%s\"."
#else
"This MIME Type's command \"%s\" refers to another MIME Type:\n\
\n\
    %s\n\
\n\
Which could not be found."
#endif
				    ,
				    name,
				    command
				);
				edv_play_sound_warning(core);
				edv_message_warning(title, msg, NULL, toplevel);
			        g_free(msg);
			    }
			}
		    }
		    else
		    {
			/* Check if the program exists */
			gchar *prog;
			const gchar *s;

			/* Get the program from the command */
			s = (const gchar *)strpbrk(
			    (const char *)command, " \t"
			);
			if(s != NULL)
			{
			    const gint len = s - command;
			    prog = (gchar *)g_malloc((len + 1) * sizeof(gchar));
			    if(len > 0)
				memcpy(prog, command, len * sizeof(gchar));
			    prog[len] = '\0';
			}
			else
			{
			    prog = STRDUP(command);
			}

			/* Got the program from the command? */
			if(prog != NULL)
			{
			    /* Check if the program does not exist */
			    if(!edv_path_exists(prog))
			    {
				const gint error_code = (gint)errno;
				status = -2;
				if(verbose)
				{
				    gchar *msg = g_strdup_printf(
"This MIME Type's command \"%s\" refers to a program:\n\
\n\
    %s\n\
\n\
Which could not be found, %s.",
					name,
					prog,
					g_strerror(error_code)
				    );
				    edv_play_sound_warning(core);
				    edv_message_warning(title, msg, NULL, toplevel);
				    g_free(msg);
				}
			    }

			    g_free(prog);
			}
		    }
		}	/* Check all the commands */ 
	    }
	    break;

	  case EDV_MIME_TYPE_CLASS_PROGRAM:
	    /* Is the Type defined? */
	    if(STRISEMPTY(type))
	    {
		status = -2;
		if(verbose)
		{
		    gchar *msg = STRDUP(
"This MIME Type's Type was not set, there will be no way to\n\
identify it."
		    );
		    edv_play_sound_warning(core);
		    edv_message_warning(title, msg, NULL, toplevel);
		    g_free(msg);
		}
	    }
	    /* Is the Value defined and does it refer to a valid
	     * program?
	     */
	    if(!STRISEMPTY(val))
	    {
		if(!edv_path_exists(val))
		{
		    const gint error_code = (gint)errno;
		    status = -1;
		    if(verbose)
		    {
			gchar *msg = g_strdup_printf(
"Unable to find the location of the program:\n\
\n\
    %s\n\
\n\
%s.",
			    val,
			    g_strerror(error_code)
			);
			edv_play_sound_warning(core);
			edv_message_warning(title, msg, NULL, toplevel);
			g_free(msg);
		    }
		}
	    }
	    else
	    {
		status = -2;
		if(verbose)
		{
		    gchar *msg;
#if defined(PROG_LANGUAGE_SPANISH)
		    msg = STRDUP(
"La ubicacin de objeto de aplicacin no conjunto"
		    );
#elif defined(PROG_LANGUAGE_FRENCH)
		    msg = STRDUP(
"L'emplacement d'objet d'application pas srie"
		    );
#else
		    msg = STRDUP(
"The location of the application object was not set."
		    );
#endif
		    edv_play_sound_warning(core);
		    edv_message_warning(title, msg, NULL, toplevel);
		    g_free(msg);
		}
	    }
	    /* Check the handler
	     *
	     * Command?
	     */
	    if(handler == EDV_MIME_TYPE_HANDLER_COMMAND)
	    {
		/* At least one command defined? */
		if(mt->commands_list != NULL)
		{
		    const gchar *s, *name, *command;
		    gchar *prog;
		    GList *glist;
		    EDVMIMETypeCommand *cmd;

		    /* Check all the commands */
		    for(glist = mt->commands_list;
		        glist != NULL;
		        glist = g_list_next(glist)
		    )
		    {
			cmd = EDV_MIME_TYPE_COMMAND(glist->data);
			if(cmd == NULL)
			    continue;

			name = cmd->name;
			command = cmd->command;
			if(STRISEMPTY(command))
			    continue;

			/* Check if the program exists, get the program
			 * from the command
			 */
			s = (const gchar *)strpbrk(
			    (const char *)command, " \t"
			);
			if(s != NULL)
			{
			    const gint len = s - command;
			    prog = (gchar *)g_malloc((len + 1) * sizeof(gchar));
			    if(len > 0)
				memcpy(prog, command, len * sizeof(gchar));
			    prog[len] = '\0';
			}
			else
			{
			    prog = STRDUP(command);
			}

			/* Got the program from the command? */
			if(prog != NULL)
			{
			    /* Check if the program does not exist */
			    if(!edv_path_exists(prog))
			    {
				const gint error_code = (gint)errno;
				status = -2;
				if(verbose)
				{
				    gchar *msg = g_strdup_printf(
"This MIME Type's command \"%s\" refers to a program:\n\
\n\
    %s\n\
\n\
Which could not be found, %s.",
					name,
					prog,
					g_strerror(error_code)
				    );
				    edv_play_sound_warning(core);
				    edv_message_warning(title, msg, NULL, toplevel);
				    g_free(msg);
				}
			    }

			    g_free(prog);
			}
		    }	/* Check all the commands */
		}
		else
		{
		    status = -2;
		    if(verbose)
		    {
			gchar *msg;
#if defined(PROG_LANGUAGE_SPANISH)
			msg = STRDUP(
"Ninguna orden se ha definido en esta clase de la\n\
aplicacin MIME Type, este MIME Type no ser capaz de\n\
ejecutar la aplicacin reffers a."
			);
#elif defined(PROG_LANGUAGE_FRENCH)
			msg = STRDUP(
"Aucune commande n'a t dfinie pour cette classe d'application MIME Type\n\
, le MIME Type ne pourra pas donc\n\
excuter l'application auquel il se rfre."
			);
#else
			msg = STRDUP(
"No command has been defined on this application\n\
class MIME Type, this MIME Type will not be able to\n\
execute the application that it refers to."
			);
#endif
			edv_play_sound_warning(core);
			edv_message_warning(title, msg, NULL, toplevel);
			g_free(msg);
		    }
		}
	    }
	    break;

	  case EDV_MIME_TYPE_CLASS_UNIQUE:
	    /* Is the Type defined? */
	    if(STRISEMPTY(type))
	    {
		status = -2;
		if(verbose)
		{
		    gchar *msg = STRDUP(
"This MIME Type's Type was not set, there will be no way to\n\
identify it."
		    );
		    edv_play_sound_warning(core);
		    edv_message_warning(title, msg, NULL, toplevel);
		    g_free(msg);
		}
	    }
	    /* Is the Value defined and does it refer to a valid
	     * object?
	     */
	    if(!STRISEMPTY(val))
	    {
		if(!edv_path_exists(val))
		{
		    const gint error_code = (gint)errno;
		    status = -1;
		    if(verbose)
		    {
			gchar *msg = g_strdup_printf(
"Unable to find the location of the specific object:\n\
\n\
    %s\n\
\n\
%s.",
			    val,
			    g_strerror(error_code)
			);
			edv_play_sound_warning(core);
			edv_message_warning(
			    title,
			    msg,
			    NULL,
			    toplevel
			);
			g_free(msg);
		    }
		}
	    }
	    else
	    {
		status = -2;
		if(verbose)
		{
		    gchar *msg;

#if defined(PROG_LANGUAGE_SPANISH)
		    msg = STRDUP(
"La ubicacin de objeto extraordinario no conjunto"
		    );
#elif defined(PROG_LANGUAGE_FRENCH)
		    msg = STRDUP(
"L'emplacement de l'objet spcifique n'est pas dfini"
		    );
#else
		    msg = STRDUP(
"The location of the specific object was not set."
		    );
#endif
		    edv_play_sound_warning(core);
		    edv_message_warning(
			title,
			msg,
			NULL,
			toplevel
		    );
		    g_free(msg);
		}
	    }
	    break;

	  default:
	    status = -2;
	    if(verbose)
	    {
		gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"Cdigo `%i' no apoyado de clase de MIME Type."
#elif defined(PROG_LANGUAGE_FRENCH)
"Le code de la classe MIME Type `%i' n'est pas support."
#else
"Unsupported MIME Type class code `%i'."
#endif
		    , mt_class
		);
		edv_play_sound_error(core);
		edv_message_error(
		    title,
		    msg,
		    NULL,
		    toplevel
		);
		g_free(msg);
	    }
	    break;
	}

	return(status);
}

/*
 *	Gets the values from the MIME Type and sets them to the
 *	dialog's widgets.
 *
 *	The mt_num specifies the MIME Type.
 */
void EDVMimeTypeEditDlgGetValues(
	edv_mime_type_edit_dlg_struct *d,
	const gint mt_num
)
{
	const gint nicon_states = EDV_MIME_TYPE_TOTAL_ICON_STATES;
	gint i;
	const gchar *icon_state_names[] = EDV_MIME_TYPE_ICON_STATE_NAMES;
	GList *icons_list;
	GtkWidget	*w,
			*pulist;
	GtkCList *clist;
	IconPickerIcon *icon;
	EDVPixmap *p;
	EDVMIMEType *mt;
	EDVCore *core;

	if(d == NULL)
	    return;

	core = d->core;

	/* Get the MIME Type */
	mt = EDV_MIME_TYPE(g_list_nth_data(
	    core->mime_types_list,
	    (guint)mt_num
	));
	if(mt == NULL)
	    return;

	/* Record the MIME Type's index */
	d->mt_num = mt_num;

	/* Class */
	w = d->class_pulistbox;
	pulist = PUListBoxGetPUList(w);
	if(pulist != NULL)
	{
	    gint i;
	    const gint n = PUListGetTotalItems(pulist);
	    EDVMIMETypeClass mt_class;

	    for(i = 0; i < n; i++)
	    {
		mt_class = (EDVMIMETypeClass)PUListGetItemData(
		    pulist,
		    i
		);
		if(mt_class == mt->mt_class)
		{
		    PUListBoxSelect(
			w,
			i
		    );
		    EDVMimeTypeEditDlgClassChangedCB(w, i, d);
		    break;
		}
	    }
	}

	/* Type */
	gtk_entry_set_text(
	    GTK_ENTRY(d->type_entry),
	    (mt->type != NULL) ? mt->type : ""
	);

	/* Value */
	gtk_entry_set_text(
	    GTK_ENTRY(d->value_entry),
	    (mt->value != NULL) ? mt->value : ""
	);

	/* Description */
	gtk_entry_set_text(
	    GTK_ENTRY(d->description_entry),
	    (mt->description != NULL) ? mt->description : ""
	);

	/* Handler */
	w = d->handler_pulistbox;
	pulist = PUListBoxGetPUList(w);
	if(pulist != NULL)
	{
	    gint i;
	    const gint n = PUListGetTotalItems(pulist);
	    EDVMIMETypeHandler handler;

	    for(i = 0; i < n; i++)
	    {
		handler = (EDVMIMETypeHandler)PUListGetItemData(
		    pulist,
		    i
		);
		if(handler == mt->handler)
		{
		    PUListBoxSelect(
			w,
			i
		    );
		    EDVMimeTypeEditDlgHandlerChangedCB(w, i, d);
		    break;
		}
	    }
	}

	/* Commands */
	clist = GTK_CLIST(d->commands_clist);
	if(clist != NULL)
	{
	    gint row;
	    GList *glist;
	    EDVMIMETypeCommand *cmd;

	    gtk_clist_freeze(clist);

	    /* Clear the list */
	    gtk_clist_clear(clist);

	    /* Append the commands */
	    for(glist = mt->commands_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		cmd = EDV_MIME_TYPE_COMMAND(glist->data);
		if(cmd == NULL)
		    continue;

		row = EDVMimeTypeEditDlgCommandsListAppend(
		    d,
		    edv_mime_type_command_copy(cmd)
		);
	    }

	    gtk_clist_thaw(clist);
	}


	/* Small icons */
	w = d->icon_small_ip;
	icons_list = NULL;
	for(i = 0; i < nicon_states; i++)
	{
	    icon = icon_picker_icon_new();
	    if(icon == NULL)
		break;

	    icon->label = STRDUP(icon_state_names[i]);
	    icon->path = STRDUP(mt->small_icon_path[i]);
	    p = edv_pixmap_ref(mt->small_icon[i]);
	    if(edv_pixmap_is_loaded(p))
	    {
		icon->pixmap = GDK_PIXMAP_REF(p->pixmap);
		icon->mask = GDK_BITMAP_REF(p->mask);
		icon->width = p->width;
		icon->height = p->height;
	    }
	    (void)edv_pixmap_unref(p);
	    icons_list = g_list_append(
		icons_list,
		icon
	    );
	}
	icon_picker_set_icons_list(w, icons_list);
	icon_picker_set_has_changes(w, FALSE);

	/* Medium icons */
	w = d->icon_medium_ip;
	icons_list = NULL;
	for(i = 0; i < nicon_states; i++)
	{
	    icon = icon_picker_icon_new();
	    if(icon == NULL)
		break;

	    icon->label = STRDUP(icon_state_names[i]);
	    icon->path = STRDUP(mt->medium_icon_path[i]);
	    p = edv_pixmap_ref(mt->medium_icon[i]);
	    if(edv_pixmap_is_loaded(p))
	    {
		icon->pixmap = GDK_PIXMAP_REF(p->pixmap);
		icon->mask = GDK_BITMAP_REF(p->mask);
		icon->width = p->width;
		icon->height = p->height;
	    }
	    (void)edv_pixmap_unref(p);
	    icons_list = g_list_append(
		icons_list,
		icon
	    );
	}
	icon_picker_set_icons_list(w, icons_list);
	icon_picker_set_has_changes(w, FALSE);

	/* Large icons */
	w = d->icon_large_ip;
	icons_list = NULL;
	for(i = 0; i < nicon_states; i++)
	{
	    icon = icon_picker_icon_new();
	    if(icon == NULL)
		break;

	    icon->label = STRDUP(icon_state_names[i]);
	    icon->path = STRDUP(mt->large_icon_path[i]);
	    p = edv_pixmap_ref(mt->large_icon[i]);
	    if(edv_pixmap_is_loaded(p))
	    {
		icon->pixmap = GDK_PIXMAP_REF(p->pixmap);
		icon->mask = GDK_BITMAP_REF(p->mask);
		icon->width = p->width;
		icon->height = p->height;
	    }
	    (void)edv_pixmap_unref(p);
	    icons_list = g_list_append(
		icons_list,
		icon
	    );
	}
	icon_picker_set_icons_list(w, icons_list);
	icon_picker_set_has_changes(w, FALSE);


	EDVMimeTypeEditDlgUpdateMenus(d);
}

/*
 *	Sets the values from the dialog's widgets to the MIME Type.
 *
 *	The mt_num specifies the MIME Type.
 */
void EDVMimeTypeEditDlgSetValues(
	edv_mime_type_edit_dlg_struct *d,
	const gint mt_num,
	const gboolean verbose
)
{
	const gint nicon_states = EDV_MIME_TYPE_TOTAL_ICON_STATES;
	GList *icons_list;
	GtkWidget *toplevel;
	GtkEntry *entry;
	GtkCList *clist;
	EDVMIMEType *m;
	EDVCore *core;

	if(d == NULL)
	    return;

	toplevel = d->toplevel;
	core = d->core;

	/* Get the MIME Type being edited */
	m = EDV_MIME_TYPE(g_list_nth_data(
	    core->mime_types_list,
	    (guint)mt_num
	));

	/* Does the MIME Type no longer exist? */
	if(m == NULL)
	{
	    if(verbose)
	    {
		edv_play_sound_error(core);
		edv_message_error(
#if defined(PROG_LANGUAGE_SPANISH)
"Aplique Fallado",
"Este MIME Type no ms largo existe, estos valores se tirarn.",
#elif defined(PROG_LANGUAGE_FRENCH)
"Appliquer Echou",
"Ce MIME Type n'existe plu, les valeurs seront rejetes.",
#elif defined(PROG_LANGUAGE_GERMAN)
"Verwenden Sie Versagt".
"Dieser MIME Type werden nicht mehr, diese werte\n\
ausgeschieden werden existiert.",
#elif defined(PROG_LANGUAGE_ITALIAN)
"Applicare Fallito".
"Questo MIME Type non pi esistere, questi valori saranno scartati.",
#elif defined(PROG_LANGUAGE_DUTCH)
"Pas Geverzuimenene Toe".
"Deze MIME Type zullen, deze waarde niet langer weggegooid\n\
worden bestaat.",
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aplique Fracassado".
"Este MIME Type no existe mais, estes valores sero descartados.",
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjeld Sviktet".
"Denne MIME Type finnes ikke mer, disse verdiene kastet.",
#else
"Apply Failed",
"This MIME Type no longer exists, these values will be discarded.",
#endif
		    NULL,
		    toplevel
		);
	    }
	    return;
	}

	/* MIME Type marked as read only? */
	if(m->read_only)
	{
	    if(verbose)
	    {
		gchar *msg = g_strdup_printf(
#if defined(PROG_LANGUAGE_SPANISH)
"MIME Type \"%s\" no se puede redactar, porque era ni\n\
creado internamente ni cargado de una configuracin global."
#elif defined(PROG_LANGUAGE_FRENCH)
"Le Type MIME \"%s\" ne peut pas tre dit, parce qu'il\n\
ou a t soit cr intrieurement soit charg par la configuration\n\
globale."
#elif defined(PROG_LANGUAGE_GERMAN)
"MIME Type \"%s\" kann nicht redigiert werden, weil es\n\
war, entweder geschaffen inner oder kann von einer globalen\n\
konfiguration beladen werden."
#elif defined(PROG_LANGUAGE_ITALIAN)
"MIME Type \"%s\" non pu essere redatto, perch era sia\n\
creato internamente o caricato da una configurazione globale."
#elif defined(PROG_LANGUAGE_DUTCH)
"MIME Type \"%s\" kun niet is geredigeerd, omdat het ofwel\n\
gecreerde inwendig ofwel gelaade van een globale\n\
configuratie was."
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"\"%s\" de MIME Type no pode ser editado, porque ele\n\
qualquer um foi criado internamente nem foi carregado de uma\n\
configurao global."
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"MIME Type \"%s\" redigert ikke, fordi det var hver\n\
skape innvendig eller lastet fra en global konfigurasjon."
#else
"MIME Type \"%s\" cannot be edited, because it\n\
was either created internally or loaded from a global\n\
configuration."
#endif
		    , m->type
		);
		edv_play_sound_warning(core);
		edv_message_warning(
#if defined(PROG_LANGUAGE_SPANISH)
"Aplique Fallado"
#elif defined(PROG_LANGUAGE_FRENCH)
"appliquer Echou"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verwenden Sie Versagt"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Applicare Fallito"
#elif defined(PROG_LANGUAGE_DUTCH)
"Pas Geverzuimenene Toe"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aplique Fracassado"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjeld Sviktet"
#else
"Apply Failed"
#endif
		    , msg, NULL,
		    toplevel
		);
		g_free(msg);
	    }
	    return;
	}


	/* Begin applying the values */

	/* Class */
	m->mt_class = EDVMimeTypeEditDlgGetClass(d);

	/* Type */
	entry = GTK_ENTRY(d->type_entry);
	if(entry != NULL)
	{
	    const gchar *s = gtk_entry_get_text(entry);
	    if(s != NULL)
	    {
		g_free(m->type);
		m->type = STRDUP(s);
	    }
	}

	/* Value */
	entry = GTK_ENTRY(d->value_entry);
	if(entry != NULL)
	{
	    const gchar *s = gtk_entry_get_text(entry);
	    if(s != NULL)
	    {
		g_free(m->value);
		m->value = STRDUP(s);
	    }
	}

	/* Description */
	entry = GTK_ENTRY(d->description_entry);
	if(entry != NULL)
	{
	    const gchar *s = gtk_entry_get_text(entry);
	    if(s != NULL)
	    {
		g_free(m->description);
		m->description = STRDUP(s);
	    }
	}

	/* Handler */
	m->handler = EDVMimeTypeEditDlgGetHandler(d);

	/* Commands */
	clist = GTK_CLIST(d->commands_clist);
	if(clist != NULL)
	{
	    gint i, n;
	    EDVMIMETypeCommand *cmd;

	    /* Delete the old commands */
	    if(m->commands_list != NULL)
	    {
		g_list_foreach(
		    m->commands_list,
		    (GFunc)edv_mime_type_command_delete,
		    NULL
		);
		g_list_free(m->commands_list);
		m->commands_list = NULL;
	    }

	    /* Add the new commands */
	    n = clist->rows;
	    for(i = 0; i < n; i++)
	    {
		cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(
		    clist,
		    i
		));
		if(cmd == NULL)
		    continue;

		m->commands_list = g_list_append(
		    m->commands_list,
		    edv_mime_type_command_copy(cmd)
		);
	    }
	}


	/* Small icon */
	icons_list = icon_picker_get_icons_list(d->icon_small_ip);
	if(icons_list != NULL)
	{
	    gint i;
	    GList *glist;
	    IconPickerIcon *icon;

	    for(glist = icons_list, i = 0;
		(glist != NULL) && (i < nicon_states);
		glist = g_list_next(glist), i++
	    )
	    {
		icon = ICON_PICKER_ICON(glist->data);
		if(icon == NULL)
		    continue;

		g_free(m->small_icon_path[i]);
		m->small_icon_path[i] = STRDUP(icon->path);
	    }
	}

	/* Medium icon */
	icons_list = icon_picker_get_icons_list(d->icon_medium_ip);
	if(icons_list != NULL)
	{
	    gint i;
	    GList *glist;
	    IconPickerIcon *icon;

	    for(glist = icons_list, i = 0;
		(glist != NULL) && (i < nicon_states);
		glist = g_list_next(glist), i++
	    )
	    {
		icon = ICON_PICKER_ICON(glist->data);
		if(icon == NULL)
		    continue;

		g_free(m->medium_icon_path[i]);
		m->medium_icon_path[i] = STRDUP(icon->path);
	    }
	}

	/* Large icon */
	icons_list = icon_picker_get_icons_list(d->icon_large_ip);
	if(icons_list != NULL)
	{
	    gint i;
	    GList *glist;
	    IconPickerIcon *icon;

	    for(glist = icons_list, i = 0;
		(glist != NULL) && (i < nicon_states);
		glist = g_list_next(glist), i++
	    )
	    {
		icon = ICON_PICKER_ICON(glist->data);
		if(icon == NULL)
		    continue;

		g_free(m->large_icon_path[i]);
		m->large_icon_path[i] = STRDUP(icon->path);
	    }
	}

	/* Update the time stamps */
	m->access_time =
	m->modify_time =
	m->change_time = time(NULL);


	/* Check all of the MIME Type's values to see if they are any
	 * problems
	 */
	EDVMimeTypeEditDlgCheckMIMEType(
	    core,
	    core->mime_types_list,
	    mt_num,
	    TRUE,				/* Verbose */
	    toplevel
	);


	/* Clear the has changes flag */
	EDVMimeTypeEditDlgResetHasChanges(d, FALSE);


	/* Re-realize this MIME Type */
	edv_mime_type_realize(m, TRUE);

	/* Notify about this MIME Type being modified */
	edv_emit_mime_type_modified(core, mt_num, m);
}


/*
 *	Creates a new command dialog.
 */
static CmdDlg *CmdDlgNew(
	EDVCore *core,
	GtkWidget *ref_toplevel,
	EDVMIMETypeCommand *cmd,
	const gboolean is_new
)
{
	const gint      border_major = 5,
			border_minor = 2;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4,
			*parent5, *parent6,
			*toplevel;
	GtkCList *clist;
	CmdDlg *d = CMD_DLG(g_malloc0(sizeof(CmdDlg)));
	if(d == NULL)
	    return(NULL);

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
	d->freeze_count = 0;
	d->has_changes = FALSE;
	d->got_response = FALSE;
	d->core = core;

	d->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, TRUE);
	gtk_window_set_title(GTK_WINDOW(w), "MIME Type Command");
	gtk_widget_set_usize(w, 500, -1);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	}
	gtk_widget_add_events(
	    w,
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(CmdDlgDeleteEventCB), d
	);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
/*	gtk_window_set_modal(GTK_WINDOW(w), TRUE); */
	gtk_window_set_transient_for(
	    GTK_WINDOW(w), GTK_WINDOW(ref_toplevel)
	);
	parent = w;

	/* Main GtkVBox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* Name */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new("Name:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	d->name_entry = w = gtk_entry_new();
	gtk_widget_set_usize(w, 200, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_entry_set_text(
	    GTK_ENTRY(w),
	    (cmd->name != NULL) ? cmd->name : ""
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(CmdDlgChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
"Enter the name of this command"
	);
	gtk_widget_grab_focus(w);
	gtk_widget_show(w);

	/* Command */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_label_new("Command:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	d->command_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_entry_set_text(
	    GTK_ENTRY(w),
	    (cmd->command != NULL) ? cmd->command : ""
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(CmdDlgChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	edv_entry_set_complete_path(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
 "Enter the command or a MIME Type reference.\
 All commands start with a '/' character and MIME Type\
 references do not. All commands may contain shell metacharacters\
 (such as | & ; ( ) < >) and tokens for substitutions (click on\
 \"Tokens\" to see the complete list of tokens).\
 MIME Type references should be the exact name\
 (which is the type value) of the referenced MIME Type."
	);
	gtk_widget_show(w);

	/* Browse GtkButton */
	w = GUIButtonPixmap(
	    (guint8 **)icon_folder_opened_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(CmdDlgBrowseCommandCB), d
	);
	GUISetWidgetTip(w, "Browse");
	gtk_widget_show(w);

	/* Tokens GtkButton */
	d->tokens_btn = w = GUIButtonPixmapLabelH(
	    NULL, "Tokens >>>", NULL
	);
	gtk_widget_set_usize(
	    w,
	    -1, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(CmdDlgTokensCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_t, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_t);
	GUISetWidgetTip(
	    w,
"Click this to display the list of tokens that you can use"
	);
	gtk_widget_show(w);

	/* Tokens GtkVBox */
	d->tokens_box = w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	parent4 = w;

	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Tokens GtkCList */
	d->tokens_clist = w = gtk_clist_new(2);
	clist = GTK_CLIST(w);
	gtk_widget_set_usize(w, -1, 150);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(  
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(edv_clist_motion_event_cb), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(CmdDlgCListSelectRowCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "unselect_row",
	    GTK_SIGNAL_FUNC(CmdDlgCListUnselectRowCB), d
	);
	gtk_container_add(GTK_CONTAINER(parent5), w);
	gtk_widget_realize(w);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_resizeable(clist, 0, FALSE);
	gtk_clist_set_column_justification(
	    clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 1, TRUE);
	gtk_clist_set_column_resizeable(clist, 1, FALSE);
	gtk_clist_set_column_justification(
	    clist, 1, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_SINGLE);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	if(w != NULL)
	{
	    const GtkTargetEntry dnd_src_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING}
	    };
	    GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY,		/* Actions */
		GDK_BUTTON1_MASK,		/* Buttons */
		NULL,
		CmdDlgCListDragDataGet,
		NULL,
		NULL,
		d
	    );
	}
	gtk_widget_show(w);

#define ADD_LINE(_s1_,_s2_)	{		\
 gint row;					\
 gchar *strv[2];				\
 strv[0] = (_s1_);				\
 strv[1] = (_s2_);				\
 row = gtk_clist_append(clist, strv);		\
 if(row > -1) {					\
  gtk_clist_set_row_data_full(			\
   clist, row,					\
   STRDUP(_s1_), (GtkDestroyNotify)g_free	\
  );						\
 }						\
}

	gtk_clist_freeze(clist);
	ADD_LINE(
"%cwd", "Current working directory."
	);
	ADD_LINE(
"%display", "Calling process' display address."
	);
	ADD_LINE(
"%first_name", "Name of the first selected object (without the full path)."
	);
	ADD_LINE(
"%first_name_wo_ext", "Name of the first selected object (without the full path or extension)."
	);
	ADD_LINE(
"%first_path", "Full path to the first selected object."
	);
	ADD_LINE(
"%first_path_wo_ext", "Full path to the first selected object without its extension."
	);
	ADD_LINE(
"%home", "User's home directory."
	);
	ADD_LINE(
"%name", "Name of the last selected object (without the full path)."
	);
	ADD_LINE(
"%name_wo_ext", "Name of the last selected object (without the full path or extension)."
	);
	ADD_LINE(
"%names", "Name(s) of all the selected object(s) (without the full path)."
	);
	ADD_LINE(
"%path", "Full path to the last selected object."
	);
	ADD_LINE(
"%path_wo_ext", "Full path to the last selected object without its extension."
	);
	ADD_LINE(
"%paths", "Full path(s) to all the selected object(s)."
	);
	ADD_LINE(
"%pid", "Calling process' process ID."
	);
	ADD_LINE(
"%pe", "Same as %path_wo_ext."
	);
	ADD_LINE(
"%p", "Same as %path."
	);
	ADD_LINE(
"%s", "Same as %path."
	);
#undef ADD_LINE
	gtk_clist_columns_autosize(clist);
	gtk_clist_thaw(clist);

	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	w = gtk_label_new(
"Remember to put double quotes (\") around tokens so that the\n\
substituted values which have spaces don't get mistaken for\n\
multiple arguments (ie \"%paths\")."
	);
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Insert Token GtkButton */
	w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_end(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;

	d->tokens_insert_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_add_20x20_xpm,
	    "Insert",
	    NULL
	);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(CmdDlgInsertTokenCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_i, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_i);
	GUISetWidgetTip(
	    w,
"Click this to insert the selected token into the command"
	);
	gtk_widget_set_sensitive(w, FALSE);
	gtk_widget_show(w);


	/* Shell Command GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	w = gtk_label_new("Shell Command:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Shell Command GtkEntry */
	d->shell_command_entry = w = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_entry_set_text(
	    GTK_ENTRY(w),
	    (cmd->shell_command != NULL) ? cmd->shell_command : ""
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(CmdDlgChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	edv_entry_set_complete_path(core, w);
	GUIEditableEndowPopupMenu(w, 0);
	GUISetWidgetTip(
	    w,
 "Enter the shell command and any shell-specific arguments or\
 leave this blank to use the default shell (the default shell\
 can be set in Settings->Options->Programs)."
	);
	gtk_widget_show(w);

	/* Browse Shell Command GtkButton */
	w = GUIButtonPixmap(
	    (guint8 **)icon_folder_opened_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(CmdDlgBrowseShellCommandCB), d
	);
	GUISetWidgetTip(w, "Browse");
	gtk_widget_show(w);

	/* Options Buttons GtkHBox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Run In Terminal GtkCheckButton */
	d->run_in_terminal_check = w = gtk_check_button_new_with_label(
	    "Run In Terminal"
	);
	GTK_TOGGLE_BUTTON_SET_ACTIVE(
	    w,
	    (cmd->flags & EDV_MIME_TYPE_COMMAND_RUN_IN_TERMINAL) ? TRUE : FALSE
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(CmdDlgChangedCB), d
	);
	GUISetWidgetTip(
	    w,
"Check this to run this command in a terminal.\
 To set the \"Run In Terminal\" command, go to\
 Settings->Options->Programs."
	);
	gtk_widget_show(w);


	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* OK button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm, "Ok", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(CmdDlgOKCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);
	gtk_widget_set_sensitive(w, is_new);
	gtk_widget_show(w);

	/* Cancel button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm, "Cancel", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(CmdDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	if(is_new)
	    gtk_widget_show(w);

	/* Close button */
	d->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm, "Close", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(CmdDlgCloseCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);
	if(!is_new)
	    gtk_widget_show(w);

	d->freeze_count--;

	gtk_widget_show_raise(d->toplevel);

	return(d);
}

/*
 *	Creates a new command dialog.
 */
static void CmdDlgDelete(CmdDlg *d)
{
	GtkCList *clist;

	if(d == NULL)
	    return;

	gtk_widget_hide(d->toplevel);
/*	gtk_window_set_modal(GTK_WINDOW(d->toplevel), FALSE); */
	gtk_window_set_transient_for(GTK_WINDOW(d->toplevel), NULL);

	d->freeze_count++;

	clist = GTK_CLIST(d->tokens_clist);
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);

	gtk_widget_destroy(d->toplevel);
	gtk_accel_group_unref(d->accelgrp);

	d->freeze_count--;

	g_free(d);
}


/*
 *	Add a new command.
 */
static void EDVMimeTypeEditDlgCommandAdd(edv_mime_type_edit_dlg_struct *d)
{
	gint row, new_row;
	GList *glist;
	GtkCList *clist;
	CfgItem *cfg_list;
	EDVMIMETypeCommand *cmd;
	EDVCore *core;

	if(d == NULL)
	    return;

	core = d->core;
	cfg_list = core->cfg_list;
	clist = GTK_CLIST(d->commands_clist);
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;

	/* Create a new command */
	cmd = edv_mime_type_command_new();
	if(cmd == NULL)
	    return;

	/* Set the default name */
	g_free(cmd->name);
	if((row == 0) || (clist->rows == 0))
	    cmd->name = STRDUP("Default");
	else if(row > -1)
	    cmd->name = g_strdup_printf(
		"Command #%i",
		row + 1
	    );
	else
	    cmd->name = g_strdup_printf(
		"Command #%i",
		clist->rows + 1
	    );

	/* Set the default command */
	g_free(cmd->command);
	cmd->command = STRDUP(
	    EDV_GET_S(EDV_CFG_PARM_PROG_DEF_VIEWER)
	);

	/* Insert the new command */
	new_row = EDVMimeTypeEditDlgCommandsListInsert(
	    d,
	    row,
	    cmd
	);
	cmd = NULL;
	if(new_row < 0)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_select_row(clist, new_row, 0);
	gtk_clist_thaw(clist);

	EDVMimeTypeEditDlgAnyChangedCB(d->command_add_btn, d);
	EDVMimeTypeEditDlgUpdateMenus(d);

	/* Edit the new item */
	if(!EDVMimeTypeEditDlgCommandEdit(d, TRUE))
	{
	    EDVMimeTypeEditDlgCommandRemove(d, FALSE);
	}
}

/*
 *	Edit the selected command.
 */
static gboolean EDVMimeTypeEditDlgCommandEdit(
	edv_mime_type_edit_dlg_struct *d,
	const gboolean is_new
)
{
	gboolean status, sensitive;
	gint row, cmd_num;
	GList *glist;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVMIMETypeCommand *cmd;
	EDVCore *core;
	CmdDlg *cmddlg;

	if(d == NULL)
	    return(FALSE);

	toplevel = d->toplevel;
	core = d->core;
	clist = GTK_CLIST(d->commands_clist);
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;
	if(row < 0)
	    return(FALSE);

	cmd_num = row;
	cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, row));
	if(cmd == NULL)
	    return(FALSE);

	/* Create a new command dialog and query the user */
	cmddlg = CmdDlgNew(
	    core,
	    toplevel,
	    cmd,
	    is_new
	);
	if(cmddlg == NULL)
	    return(FALSE);

	d->freeze_count++;

	sensitive = FALSE;
	gtk_widget_set_sensitive(d->command_add_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_add_mi, sensitive);
	gtk_widget_set_sensitive(d->command_edit_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_edit_mi, sensitive);
	gtk_widget_set_sensitive(d->command_remove_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_remove_mi, sensitive);
	gtk_widget_set_sensitive(d->command_shift_up_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_shift_up_mi, sensitive);
	gtk_widget_set_sensitive(d->command_shift_down_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_shift_down_mi, sensitive);

	/* Wait for user response */
	gtk_main();

	d->freeze_count--;

	if(cmddlg->got_response)
	{
	    /* Create a new command with the new values */
	    EDVMIMETypeCommand *cmd = edv_mime_type_command_new();
	    if(cmd != NULL)
	    {
		const gchar *s = gtk_entry_get_text(GTK_ENTRY(cmddlg->name_entry));
		if(!STRISEMPTY(s))
		    cmd->name = g_strdup(s);

		s = gtk_entry_get_text(GTK_ENTRY(cmddlg->command_entry));
		if(!STRISEMPTY(s))
		    cmd->command = g_strdup(s);

		s = gtk_entry_get_text(GTK_ENTRY(cmddlg->shell_command_entry));
		if(!STRISEMPTY(s))
		    cmd->shell_command = g_strdup(s);

		if(GTK_TOGGLE_BUTTON_GET_ACTIVE(cmddlg->run_in_terminal_check))
		    cmd->flags |= EDV_MIME_TYPE_COMMAND_RUN_IN_TERMINAL;

		EDVMimeTypeEditDlgCommandsListSet(
		    d,
		    cmd_num,
		    cmd
		);
/*		cmd = NULL; */
	    }

	    EDVMimeTypeEditDlgAnyChangedCB(d->command_edit_btn, d);

	    status = TRUE;
	} 
	else
	{
	    status = FALSE;
	}

	/* Delete the command dialog */
	CmdDlgDelete(cmddlg);

	EDVMimeTypeEditDlgUpdateMenus(d);

	return(status);
}

/*
 *	Remove the selected handler(s).
 */
static void EDVMimeTypeEditDlgCommandRemove(
	edv_mime_type_edit_dlg_struct *d,
	const gboolean confirm
)
{
	gint row;
	GList *glist, *cmds_list;
	GtkWidget *toplevel;
	GtkCList *clist;
	EDVMIMETypeCommand *cmd;

	if(d == NULL)
	    return;

	clist = GTK_CLIST(d->commands_clist);
	toplevel = gtk_widget_get_toplevel(d->toplevel);

	/* Create a list of selected commands */
	cmds_list = NULL;
	glist = clist->selection;
	while(glist != NULL)
	{
	    row = (gint)glist->data;
	    cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, row));
	    if(cmd != NULL)
		cmds_list = g_list_append(cmds_list, cmd);
	    glist = g_list_next(glist);
	}

	/* Nothing to remove? */
	if(cmds_list == NULL)
	    return;

	/* Confirm remove? */
	if(confirm)
	{
	    const gint ncmds = g_list_length(cmds_list);
	    gint response;
	    gchar *msg;
	    if(ncmds == 1)
	    {
		cmd = EDV_MIME_TYPE_COMMAND(cmds_list->data);
		msg = g_strdup_printf(
"Remove command \"%s\"?",
		    (cmd != NULL) ? cmd->name : "(null)"
		);
	    }     
	    else
	    {
		msg = g_strdup_printf(
"Remove %i selected commands?",
		    ncmds
		);
	    }
	    CDialogSetTransientFor(toplevel);
	    response = CDialogGetResponse(
		"Confirm Remove",
		msg,
		NULL,
		CDIALOG_ICON_QUESTION, 
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_NO
	    );
	    g_free(msg);
	    CDialogSetTransientFor(NULL);
		   
	    if(response != CDIALOG_RESPONSE_YES)
	    {
		g_list_free(cmds_list);
		return;
	    }
	}    
	 
	/* Begin removing the selected commands */
	gtk_clist_freeze(clist);
	for(glist = cmds_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    cmd = EDV_MIME_TYPE_COMMAND(glist->data);
	    row = gtk_clist_find_row_from_data(clist, cmd);
	    if(row < 0)
		continue;

	    EDVMimeTypeEditDlgAnyChangedCB(d->command_remove_btn, d);

	    gtk_clist_remove(clist, row);
	}
	gtk_clist_columns_autosize(clist);
	gtk_clist_thaw(clist);

	g_list_free(cmds_list);

	EDVMimeTypeEditDlgUpdateMenus(d);
}

/*
 *	Shift the selected handler(s) up.
 */
static void EDVMimeTypeEditDlgCommandShiftUp(edv_mime_type_edit_dlg_struct *d)
{
	gint i, n, row;
	GList *glist, *cmds_list;
	GtkCList *clist;
	EDVMIMETypeCommand *cmd;

	if(d == NULL)
	    return;

	clist = GTK_CLIST(d->commands_clist);

	/* Create a list of selected commands */
	cmds_list = NULL;
	glist = clist->selection;
	while(glist != NULL)
	{
	    row = (gint)glist->data;
	    cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, row));
	    if(cmd != NULL)
		cmds_list = g_list_append(cmds_list, cmd);
	    glist = g_list_next(glist);
	}

	/* Nothing to shift? */
	if(cmds_list == NULL)
	    return;

	n = clist->rows;

	/* First row selected? */
	i = 0;
	cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, i));
	if(g_list_find(cmds_list, cmd) != NULL)
	{
	    /* The first row may not be shifted up any further */
	    g_list_free(cmds_list);
	    return;
	}

	/* Begin shifting the selected rows up */
	gtk_clist_freeze(clist);
	for(i = 1; i < n; i++)
	{
	    cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, i));
	    if(g_list_find(cmds_list, cmd) != NULL)
	    {
		gtk_clist_swap_rows(clist, i, i - 1);
		EDVMimeTypeEditDlgAnyChangedCB(d->command_shift_up_btn, d);
	    }                    
	}
	gtk_clist_thaw(clist);

	g_list_free(cmds_list);

	EDVMimeTypeEditDlgUpdateMenus(d);
}

/*
 *	Shift the selected handler(s) down.
 */
static void EDVMimeTypeEditDlgCommandShiftDown(edv_mime_type_edit_dlg_struct *d)
{
	gint i, n, row;
	GList *glist, *cmds_list;
	GtkCList *clist;
	EDVMIMETypeCommand *cmd;

	if(d == NULL)
	    return;

	clist = GTK_CLIST(d->commands_clist);

	/* Create a list of selected commands */
	cmds_list = NULL;
	glist = clist->selection;
	while(glist != NULL)
	{
	    row = (gint)glist->data;
	    cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, row));
	    if(cmd != NULL)
		cmds_list = g_list_append(cmds_list, cmd);
	    glist = g_list_next(glist);
	}

	/* Nothing to shift? */
	if(cmds_list == NULL)
	    return;

	n = clist->rows;

	/* Last row selected? */
	i = n - 1;
	cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, i));
	if(g_list_find(cmds_list, cmd) != NULL)
	{
	    /* The last row may not be shifted down any further */
	    g_list_free(cmds_list);
	    return;
	}

	/* Begin shifting the selected rows down */
	gtk_clist_freeze(clist);
	for(i = n - 2; i >= 0; i--)
	{
	    cmd = EDV_MIME_TYPE_COMMAND(gtk_clist_get_row_data(clist, i));
	    if(g_list_find(cmds_list, cmd) != NULL)
	    {
		gtk_clist_swap_rows(clist, i, i + 1);
		EDVMimeTypeEditDlgAnyChangedCB(d->command_shift_down_btn, d);
	    }
	}
	gtk_clist_thaw(clist);

	g_list_free(cmds_list);

	EDVMimeTypeEditDlgUpdateMenus(d);
}


/*
 *	Inserts a new handler item to the list.
 *
 *	The cmd_num specifies the insert index.
 *
 *	The cmd specifies the command that will be transfered and
 *	should not be referenced again after this call.
 *
 *	Returns the new command index or -1 on error.
 */
static gint EDVMimeTypeEditDlgCommandsListInsert(
	edv_mime_type_edit_dlg_struct *d,
	const gint cmd_num,
	EDVMIMETypeCommand *cmd
)
{
	gint i, row, new_row, ncolumns;
	gchar **strv;
	GtkCList *clist;

	if(d == NULL)
	{
	    edv_mime_type_command_delete(cmd);
	    return(-1);
	}

	clist = GTK_CLIST(d->commands_clist);
	ncolumns = MAX(clist->columns, 1);
	row = cmd_num;

	gtk_clist_freeze(clist);

	/* Allocate row cell values */
	strv = (gchar **)g_malloc(ncolumns * sizeof(gchar *));
	for(i = 0; i < ncolumns; i++)
	    strv[i] = "";

	/* Insert or append row */
	if(row > -1)
	    new_row = gtk_clist_insert(clist, row, strv);
	else
	    new_row = gtk_clist_append(clist, strv);

	/* Delete row cell values */
	g_free(strv);

	/* Failed to create new row? */
	if(new_row < 0)
	{
	    gtk_clist_thaw(clist);
	    edv_mime_type_command_delete(cmd);
	    return(-1);
	}              

	/* Set the row values */
	EDVMimeTypeEditDlgCommandsListSet(d, new_row, cmd);

	gtk_clist_thaw(clist);

	return(new_row);
}

/*
 *	Appends a new handler item to the list.
 *
 *	The cmd specifies the command that will be transfered and
 *	should not be referenced again after this call.
 *
 *	Returns the new command index or -1 on error.
 */
static gint EDVMimeTypeEditDlgCommandsListAppend(
	edv_mime_type_edit_dlg_struct *d,
	EDVMIMETypeCommand *cmd
)
{
	return(EDVMimeTypeEditDlgCommandsListInsert(
	    d, -1, cmd
	));
}

/*
 *	Sets the commands list's row cell values and row data.
 *
 *	The cmd_num specifies the command and row index.
 *
 *	The cmd specifies the command who's values will be used to
 *	set the row's cell values and be set as the row's data.
 *	This command must not be referenced again after this call.
 */
static void EDVMimeTypeEditDlgCommandsListSet(
	edv_mime_type_edit_dlg_struct *d,
	const gint cmd_num,
	EDVMIMETypeCommand *cmd
)
{
	gint row, nrows, column, ncolumns;
	GtkCList *clist;
	EDVCore *core;

	if(d == NULL)
	{
	    edv_mime_type_command_delete(cmd);
	    return;
	}

	clist = GTK_CLIST(d->commands_clist);
	ncolumns = clist->columns;
	nrows = clist->rows;
	core = d->core;

	row = cmd_num;
	if((row < 0) || (row >= nrows))
	{
	    edv_mime_type_command_delete(cmd);
	    return;
	}

	gtk_clist_freeze(clist);

	/* Name */	
	column = 0;
	if(column < ncolumns)
	{
	    const gchar *text = (cmd != NULL) ? cmd->name : NULL;
	    if(text == NULL)
		text = "";
	    gtk_clist_set_text(
		clist,
		row, column,
		text
	    );
	}

	/* Command */
	column = 1;
	if(column < ncolumns)
	{
	    EDVPixmap *icon = NULL;
	    const gchar *text = (cmd != NULL) ? cmd->command : NULL;
	    if(text == NULL)
		text = "";

	    /* Get the icon from the command */
	    if(cmd != NULL)
	    {
		if(!STRISEMPTY(cmd->command))
		{
		    const gchar *command = cmd->command;

		    /* Is the command a reference to another MIME Type? */
		    if(*command != G_DIR_SEPARATOR)
		    {
			icon = edv_pixmap_ref(d->mime_type_icon);
		    }
		    else
		    {
			/* Run in terminal? */
			if(cmd->flags & EDV_MIME_TYPE_COMMAND_RUN_IN_TERMINAL)
			{
			    icon = edv_pixmap_ref(d->run_in_terminal_icon);
			}
			else
			{
			    /* All else get the icon for the program in
			     * the command
			     */
			    gchar *prog;
			    (void)edv_strarg(
				command,
				&prog,
				TRUE,		/* Parse escapes */
				TRUE		/* Parse quotes */
			    );
			    if(prog != NULL)
			    {
				EDVVFSObject *obj = edv_vfs_object_lstat(prog);
				if(obj != NULL)
				{
				    EDVPixmap *icon_hidden;
				    (void)edv_match_object_icon(
					core->devices_list,
					core->mime_types_list,
					obj->type,
					obj->path,
					EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj),
					obj->permissions,
					EDV_ICON_SIZE_20,
					&icon,
					NULL,
					NULL,
					&icon_hidden
				    );
				    if(edv_path_is_hidden(obj->path))
				    {
					if(edv_pixmap_is_loaded(icon_hidden))
					{
					    (void)edv_pixmap_unref(icon);
					    icon = edv_pixmap_ref(icon_hidden);
					}
				    }
				    (void)edv_pixmap_unref(icon_hidden);
				    edv_vfs_object_delete(obj);
				}
				g_free(prog);
			    }
			}
		    }
		}
	    }	/* Get the icon from the command */
	    if(edv_pixmap_is_loaded(icon))
		gtk_clist_set_pixtext(
		    clist, 
		    row, column,
		    text,
		    EDV_LIST_PIXMAP_TEXT_SPACING,
		    icon->pixmap, icon->mask
		);
	    else
		gtk_clist_set_text(
		    clist,
		    row, column,
		    text
		);

	    (void)edv_pixmap_unref(icon);
	}

	/* Set this row's data as the command */
	gtk_clist_set_row_data_full(
	    clist,
	    row,
	    cmd, (GtkDestroyNotify)edv_mime_type_command_delete
	);

	gtk_clist_columns_autosize(clist);

	gtk_clist_thaw(clist);
}


/*
 *	Creates a new MIME Types Edit Dialog.
 */
edv_mime_type_edit_dlg_struct *EDVMimeTypeEditDlgNew(EDVCore *core)
{
	const gint	border_major = 5,
			border_minor = 2;
	gulong load_int;
	gchar		*default_icons_path,
			*heading[2];
	gpointer	entry_rtn,
			label_rtn;
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget	*w,
			*parent, *parent2, *parent3, *parent4,
			*parent5, *parent6, *parent7, *parent8,
			*toplevel,
			*pulist,
			*icon_sel_dlg;
	GtkCList *clist;
	CfgList *cfg_list;
	IconPickerFlags ip_flags;
	EDVListsPointerOpButton2 lists_pointer_op_button2;
	edv_mime_type_edit_dlg_struct *d;

	if(core == NULL)
	    return(NULL);

	cfg_list = core->cfg_list;

	/* Create the MIME Types Edit Dialog */
	d = EDV_MIME_TYPE_EDIT_DLG(
	    g_malloc0(sizeof(edv_mime_type_edit_dlg_struct))
	);
	if(d == NULL)
	    return(NULL);

	d->toplevel = toplevel = gtk_window_new(GTK_WINDOW_DIALOG);
	d->accelgrp = accelgrp = gtk_accel_group_new();
	d->freeze_count = 0;
	d->busy_count = 0;
	d->has_changes = FALSE;
	d->core = core;
	d->mt_num = -1;
	d->browse_path_rtn = NULL;

	d->freeze_count++;

	/* Toplevel GtkWindow */
	w = toplevel;
	gtk_window_set_wmclass(
	    GTK_WINDOW(w), "dialog", PROG_NAME
	);
	gtk_window_set_policy(GTK_WINDOW(w), FALSE, FALSE, FALSE);
	gtk_window_set_title(GTK_WINDOW(w), EDV_MIME_TYPE_EDIT_DLG_TITLE);
	gtk_widget_set_usize(w, EDV_MIME_TYPE_EDIT_DLG_WIDTH, EDV_MIME_TYPE_EDIT_DLG_HEIGHT);
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(
		window,
		(guint8 **)icon_mimetypes_48x48_xpm
	    );
	}
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgDeleteEventCB), d
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;

	/* Load the icons */
	d->mime_type_icon = edv_load_pixmap_from_data(
	    core,
	    (edv_pixmap_data *)icon_file_extension_20x20_xpm,
	    "icon_file_extension_20x20_xpm"
	);
	d->run_in_terminal_icon = edv_load_pixmap_from_data(
	    core,
	    (edv_pixmap_data *)icon_terminal_run2_20x20_xpm,
	    "icon_terminal_run2_20x20_xpm"
	);

	/* Main GtkVBox */
	d->main_vbox = w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* GtkHBox to make two columns */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Left column GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Class GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"La Clase"
#elif defined(PROG_LANGUAGE_FRENCH)
"Classe"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klasse"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Classe"
#elif defined(PROG_LANGUAGE_DUTCH)
"Klas"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Classe"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Klasse"
#else
"Class"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* Class Popup List Box */
	d->class_pulistbox = w = PUListBoxNew(
	    -1, -1
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	pulist = PUListBoxGetPUList(w);
	if(core->run_flags & EDV_RUN_SAFE_MODE)
	    PUListSetShadowStyle(pulist, PULIST_SHADOW_NONE);
	PUListBoxSetChangedCB(
	    w,
	    EDVMimeTypeEditDlgClassChangedCB, d
	);
	if(pulist != NULL)
	{
	    gint i;

	    i = PUListAddItem(
		pulist,
		"System Object Type"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_CLASS_SYSTEM);
	    i = PUListAddItem(
		pulist,
		"File Format"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_CLASS_FORMAT);
	    i = PUListAddItem(
		pulist,
		"Application"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_CLASS_PROGRAM);
	    i = PUListAddItem(
		pulist,
		"Specific Object"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_CLASS_UNIQUE);

	    PUListBoxSetLinesVisible(w, i + 1);
	}
	gtk_widget_show(w);


	/* Identification GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Identificacin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Identification"
#elif defined(PROG_LANGUAGE_GERMAN)
"Identifikation"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Identificazione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Identificatie"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Identificao"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Identification"
#else
"Identification"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Type GtkEntry */
	w = GUIPromptBar(
	    NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"El Tipo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Type"
#elif defined(PROG_LANGUAGE_GERMAN)
"Typ"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Tipo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Type"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Tipo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Type"
#else
"Type"
#endif
	    ":", NULL, &entry_rtn
	);
	d->type_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgAnyChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);

	/* Value GtkEntry */
	w = GUIPromptBar(
	    NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"El Valor"
#elif defined(PROG_LANGUAGE_FRENCH)
"Valeur"
#elif defined(PROG_LANGUAGE_GERMAN)
"Wert"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Valore"
#elif defined(PROG_LANGUAGE_DUTCH)
"Waarde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Valor"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Verdi"
#else
"Value"
#endif
	    ":", &label_rtn, &entry_rtn
	);
	d->value_label = (GtkWidget *)label_rtn;
	d->value_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgAnyChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);


	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Description GtkEntry */
	w = GUIPromptBar(
	    NULL,
#if defined(PROG_LANGUAGE_SPANISH)
"La Descripcin"
#elif defined(PROG_LANGUAGE_FRENCH)
"Description"
#elif defined(PROG_LANGUAGE_GERMAN)
"Beschreibung"
#elif defined(PROG_LANGUAGE_ITALIAN)
"La Descrizione"
#elif defined(PROG_LANGUAGE_DUTCH)
"Beschrijving"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"A Descrio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Beskrivelse"
#else
"Description"
#endif
	    ":", NULL, &entry_rtn
	);
	d->description_entry = (GtkWidget *)entry_rtn;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	gtk_signal_connect(
	    GTK_OBJECT(w), "changed",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgAnyChangedCB), d
	);
	edv_entry_set_dnd(core, w);
	GUIEditableEndowPopupMenu(w, 0);


	/* Handler GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Tratante"
#elif defined(PROG_LANGUAGE_FRENCH)
"Agent"
#elif defined(PROG_LANGUAGE_GERMAN)
"Behandler"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Addestratore"
#elif defined(PROG_LANGUAGE_DUTCH)
"Handler"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"O Manipulador"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Behandler"
#else
"Handler"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Handle By GtkHBox */
	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;

	/* Label */
	w = gtk_label_new("Handle By:");
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Handle By Popup List Box */
	d->handler_pulistbox = w = PUListBoxNew(
	    -1, -1
	);
	gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, TRUE, 0);
	pulist = PUListBoxGetPUList(w);
	if(core->run_flags & EDV_RUN_SAFE_MODE)
	    PUListSetShadowStyle(pulist, PULIST_SHADOW_NONE);
	PUListBoxSetChangedCB(
	    w,
	    EDVMimeTypeEditDlgHandlerChangedCB, d
	);
	if(pulist != NULL)
	{
	    gint i;

	    i = PUListAddItem(
		pulist,
		"Command"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_HANDLER_COMMAND);
	    i = PUListAddItem(
		pulist,
		"Endeavour Archiver"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_HANDLER_EDV_ARCHIVER);
	    i = PUListAddItem(
		pulist,
		"Endeavour Image Browser"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_HANDLER_EDV_IMAGE_BROWSER);
	    i = PUListAddItem(
		pulist,
		"Endeavour Recycle Bin"
	    );
	    PUListSetItemData(pulist, i, (gpointer)EDV_MIME_TYPE_HANDLER_EDV_RECYCLE_BIN);

	    PUListBoxSetLinesVisible(w, i + 1);
	}
	gtk_widget_show(w);

	/* Handler GtkHBox */
	d->commands_clist_box = w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent6 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent6), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent7 = w;

	/* Create the GtkScrolledWindow for the GtkCList */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent8 = w;

	/* Commands GtkCList */
#if defined(PROG_LANGUAGE_SPANISH)
heading[0] = "El Nombre";
heading[1] = "La Orden";
#elif defined(PROG_LANGUAGE_FRENCH)
heading[0] = "Nom";
heading[1] = "Commande";
#elif defined(PROG_LANGUAGE_GERMAN)
heading[0] = "Name";
heading[1] = "Befehl";
#elif defined(PROG_LANGUAGE_ITALIAN)
heading[0] = "Il Nome";
heading[1] = "Il Comando";
#elif defined(PROG_LANGUAGE_DUTCH)
heading[0] = "Naam";
heading[1] = "Bevel";
#elif defined(PROG_LANGUAGE_PORTUGUESE)
heading[0] = "O Nome";
heading[1] = "O Comando";
#elif defined(PROG_LANGUAGE_NORWEGIAN)
heading[0] = "Navn";
heading[1] = "Kommando";
#else
heading[0] = "Name";
heading[1] = "Command";
#endif
	d->commands_clist = w = gtk_clist_new_with_titles(
	    sizeof(heading) / sizeof(gchar *),
	    heading
	);
	clist = GTK_CLIST(w);
	gtk_widget_set_usize(w, -1, 150);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(edv_clist_key_event_cb), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(edv_clist_button_event_cb), core
	);
	gtk_signal_connect(  
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(edv_clist_motion_event_cb), core
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandsCListEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandsCListEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandsCListEventCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandsCListSelectRowCB), d
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "unselect_row",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandsCListUnselectRowCB), d
	);
	gtk_container_add(GTK_CONTAINER(parent8), w);
	gtk_widget_realize(w);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_clist_column_titles_passive(clist);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_resizeable(clist, 0, TRUE);
	gtk_clist_set_column_justification(
	    clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_column_auto_resize(clist, 1, TRUE);
	gtk_clist_set_column_resizeable(clist, 1, TRUE);
	gtk_clist_set_column_justification(
	    clist, 1, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	gtk_clist_set_row_height(clist, EDV_LIST_ROW_SPACING);
	if(w != NULL)
	{
	    const GtkTargetEntry dnd_tar_types[] = {
{GUI_TARGET_NAME_TEXT_PLAIN,	0,	EDV_DND_INFO_TEXT_PLAIN},
{GUI_TARGET_NAME_TEXT_URI_LIST,	0,	EDV_DND_INFO_TEXT_URI_LIST},
{GUI_TARGET_NAME_STRING,	0,	EDV_DND_INFO_STRING},
{EDV_DND_TARGET_MIME_TYPE,	0,	EDV_DND_INFO_MIME_TYPE},
{EDV_DND_TARGET_MIME_TYPE_COMMAND, 0,	EDV_DND_INFO_MIME_TYPE_COMMAND}
	    };
	    const GtkTargetEntry dnd_src_types[] = {
{EDV_DND_TARGET_MIME_TYPE_COMMAND, 0,	EDV_DND_INFO_MIME_TYPE_COMMAND}
	    };
	    GUIDNDSetSrc(
		w,
		dnd_src_types,
		sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE,	/* Actions */
		GDK_BUTTON1_MASK,			/* Buttons */
		NULL,
		EDVMimeTypeEditDlgCommandsCListDragDataGetCB,
		EDVMimeTypeEditDlgCommandsCListDragDataDeleteCB,
		NULL,
		d
	    );
	    GUIDNDSetTar(
		w,
		dnd_tar_types,
		sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
		GDK_ACTION_COPY | GDK_ACTION_MOVE,	/* Actions */
		GDK_ACTION_MOVE,		/* Default action if same */
		GDK_ACTION_COPY,		/* Default action */
		EDVMimeTypeEditDlgCommandsCListDragDataReceivedCB,
		d,
		TRUE				/* Highlight */
	    );
	}
	gtk_widget_show(w);

	/* buttons GtkVBox */
	w = gtk_vbox_new(FALSE, 2 * border_major);
	gtk_box_pack_start(GTK_BOX(parent6), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent7 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent8 = w;

	/* Add button */
	d->command_add_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_add_20x20_xpm,
	    "Add",
	    NULL
	);
	gtk_widget_set_usize(
	    w, GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent8), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandAddCB), d
	);
	GUISetWidgetTip(
	    w,
	    "Add a new command"
	);
	gtk_widget_show(w);

	/* Edit button */
	d->command_edit_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_edit_20x20_xpm,
	    "Edit",
	    NULL
	);
	gtk_widget_set_usize(
	    w, GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent8), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandEditCB), d
	);
	GUISetWidgetTip(
	    w,
	    "Edit the selected command"
	);
	gtk_widget_show(w);

	/* Remove button */
	d->command_remove_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_remove_20x20_xpm,
	    "Remove",
	    NULL
	);
	gtk_widget_set_usize(
	    w, GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent8), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandRemoveCB), d
	);
	GUISetWidgetTip(
	    w,
	    "Remove the selected command(s)"
	);
	gtk_widget_show(w);

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent7), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent8 = w;

	/* Shift up button */
	d->command_shift_up_btn = w = GUIButtonArrowLabelH(
	    GTK_ARROW_UP, 20, 20,
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie Arriba"
#elif defined(PROG_LANGUAGE_FRENCH) 
"Changer Haut"
#elif defined(PROG_LANGUAGE_GERMAN)
"Spostare Su"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Spostare Su"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verplaats Op"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude Cima"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forskyv Opp"
#else
"Shift Up"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w, 
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent8), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandShiftUpCB), d
	);
	GUISetWidgetTip(
	    w,
	    "Shift the selected command(s) up"
	);
	gtk_widget_show(w);

	/* Shift down button */
	d->command_shift_down_btn = w = GUIButtonArrowLabelH(
	    GTK_ARROW_DOWN, 20, 20,
#if defined(PROG_LANGUAGE_SPANISH)
"Cambie Abajo"
#elif defined(PROG_LANGUAGE_FRENCH) 
"Changer Bas"
#elif defined(PROG_LANGUAGE_GERMAN)
"Spostare Gi"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Spostare Gi"
#elif defined(PROG_LANGUAGE_DUTCH)
"Verplaats Beneden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mude Baixo"      
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Forskyv Ned"
#else       
"Shift Down"
#endif
	    , NULL
	);
	gtk_widget_set_usize(
	    w, 
	    GUI_BUTTON_HLABEL_WIDTH, GUI_BUTTON_HLABEL_HEIGHT
	);
	gtk_box_pack_start(GTK_BOX(parent8), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCommandShiftDownCB), d
	);
	GUISetWidgetTip(
	    w,
	    "Shift the selected command(s) down"
	);
	gtk_widget_show(w);


	/* Create the commands menu */
#define ADD_MENU_ITEM_LABEL	{		\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_LABEL,			\
  accelgrp,					\
  icon,						\
  label,					\
  accel_key, accel_mods,			\
  func_cb, menu_d				\
 );						\
}
#define ADD_MENU_ITEM_SEPARATOR		{	\
 w = GUIMenuItemCreate(				\
  menu,						\
  GUI_MENU_ITEM_TYPE_SEPARATOR,			\
  NULL,						\
  NULL,						\
  NULL,						\
  0, 0,						\
  NULL, NULL					\
 );						\
}
	d->commands_menu = w = GUIMenuCreate();
	if(w != NULL)
	{
	    GtkAccelGroup *accelgrp = NULL;
	    GtkWidget *menu = w;
	    guint8 **icon;
	    const gchar *label;
	    guint accel_key, accel_mods;
	    void (*func_cb)(GtkWidget *w, gpointer);
	    gpointer menu_d = d;

	    icon = (guint8 **)icon_add_20x20_xpm;
	    label = "Add...";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVMimeTypeEditDlgCommandAddCB;
	    ADD_MENU_ITEM_LABEL
	    d->commands_menu_add_mi = w;

	    icon = (guint8 **)icon_edit_20x20_xpm;
	    label = "Edit...";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVMimeTypeEditDlgCommandEditCB;
	    ADD_MENU_ITEM_LABEL
	    d->commands_menu_edit_mi = w;

	    icon = (guint8 **)icon_remove_20x20_xpm;
	    label = "Remove";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVMimeTypeEditDlgCommandRemoveCB;
	    ADD_MENU_ITEM_LABEL
	    d->commands_menu_remove_mi = w;

	    ADD_MENU_ITEM_SEPARATOR

	    icon = NULL;
	    label = "Shift Up";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVMimeTypeEditDlgCommandShiftUpCB;
	    ADD_MENU_ITEM_LABEL
	    d->commands_menu_shift_up_mi = w;

	    icon = NULL;
	    label = "Shift Down";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = EDVMimeTypeEditDlgCommandShiftDownCB;
	    ADD_MENU_ITEM_LABEL
	    d->commands_menu_shift_down_mi = w;
	}
#undef ADD_MENU_ITEM_LABEL
#undef ADD_MENU_ITEM_SEPARATOR


	/* Right Side GtkVBox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Icons GtkFrame */
	w = gtk_frame_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Iconos"
#elif defined(PROG_LANGUAGE_FRENCH)
"Icne"
#elif defined(PROG_LANGUAGE_GERMAN)
"Abbilder"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Le Icone"
#elif defined(PROG_LANGUAGE_DUTCH)
"Beelden"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Os cones"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Ikoner"
#else
"Icons"
#endif
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Icon Pickers GtkVBox */
	w = gtk_vbox_new(FALSE, 2 * border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* Set the Icon Picker flags */
	ip_flags = ICON_PICKER_CAN_CLEAR_ICON |
	    ICON_PICKER_ACCEPT_SMALLER_SIZE |
	    ICON_PICKER_ACCEPT_LARGER_SIZE |
	    ICON_PICKER_WARN_DIFFERENT_SIZE |
	    ICON_PICKER_ACCEPT_DND_DROP |
	    ICON_PICKER_PROVIDE_DND_DRAG;
	if(EDV_GET_B(EDV_CFG_PARM_LISTS_DOUBLE_BUFFER))
	    ip_flags |= ICON_PICKER_DOUBLE_BUFFER;
	if(EDV_GET_B(EDV_CFG_PARM_SHOW_TEXTTIPS))
	    ip_flags |= ICON_PICKER_SHOW_TEXTTIPS;

	lists_pointer_op_button2 = (EDVListsPointerOpButton2)EDV_GET_I(
	    EDV_CFG_PARM_LISTS_POINTER_OP_BUTTON2
	);

	/* Load images interval (in milliseconds) */
	load_int = edv_get_interval_from_load_images_priority(
	    (EDVListsLoadImagesPriority)EDV_GET_I(
		EDV_CFG_PARM_LISTS_LOAD_IMAGES_PRIORITY
	    ) 
	);

	/* Default icons location */
	default_icons_path = g_strconcat(
	    EDV_GET_S(EDV_CFG_PARM_DIR_GLOBAL),
	    G_DIR_SEPARATOR_S,
	    EDV_NAME_ICONS_SUBDIR,
	    NULL
	);

	/* Icon Selector Dialog */
	d->icon_sel_dlg = icon_sel_dlg = IconSelDlgNew(
	    ((ip_flags & ICON_PICKER_DOUBLE_BUFFER) ?
		ICON_SEL_DLG_DOUBLE_BUFFER : 0) |
	    ((ip_flags & ICON_PICKER_SHOW_TEXTTIPS) ?
		ICON_SEL_DLG_SHOW_TEXTTIPS : 0) |
	    ((lists_pointer_op_button2 == EDV_LISTS_POINTER_OP_BUTTON2_SCROLL_XY) ?
		ICON_SEL_DLG_ENABLE_DRAG_SCROLL : 0),
	    GTK_ORIENTATION_HORIZONTAL,
	    -1,					/* Default list height */
	    50, 50,				/* Thumb size */
	    0,					/* Thumb border */
	    default_icons_path,
	    load_int
	);

	g_free(default_icons_path);

	/* Large Icon Picker */
	d->icon_large_ip = w = icon_picker_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Grande"
#elif defined(PROG_LANGUAGE_FRENCH)
"Grand"
#elif defined(PROG_LANGUAGE_GERMAN)
"Gro"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Grande"
#elif defined(PROG_LANGUAGE_DUTCH)
"Groot"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Grande"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Stor"
#else
"Large"
#endif
	    " 48x48:",
	    ip_flags,
	    ICON_PICKER_ICON_POSITION_CENTER,
	    48, 48,				/* Icon size */
	    54, 54,				/* Display size */
	    80,					/* Label width */
	    icon_sel_dlg
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	icon_picker_set_switched_icon_cb(
	    w,
	    EDVMimeTypeEditDlgIconPickerIconSwitchedCB, d
	);
	icon_picker_set_changed_cb(
	    w,
	    EDVMimeTypeEditDlgAnyChangedCB, d
	);
	gtk_widget_show(w);

	/* Medium Icon Picker */
	d->icon_medium_ip = w = icon_picker_new(
#if defined(PROG_LANGUAGE_SPANISH)
"El Medio"
#elif defined(PROG_LANGUAGE_FRENCH)
"Moyen"
#elif defined(PROG_LANGUAGE_GERMAN)
"Medium"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Il Mezzo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Middel"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Mdio"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Middel"
#else
"Medium"
#endif
	    " 32x32:",
	    ip_flags,
	    ICON_PICKER_ICON_POSITION_CENTER,
	    32, 32,				/* Icon size */
	    38, 38,				/* Display size */
	    80,					/* Label width */
	    icon_sel_dlg
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	icon_picker_set_switched_icon_cb(
	    w,
	    EDVMimeTypeEditDlgIconPickerIconSwitchedCB, d
	);
	icon_picker_set_changed_cb(
	    w,
	    EDVMimeTypeEditDlgAnyChangedCB, d
	);
	gtk_widget_show(w);

	/* Small Icon Picker */
	d->icon_small_ip = w = icon_picker_new(
#if defined(PROG_LANGUAGE_SPANISH)
"Pequeo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Petit"
#elif defined(PROG_LANGUAGE_GERMAN)
"Klein"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Piccolo"
#elif defined(PROG_LANGUAGE_DUTCH)
"Klein"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Pequeno"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Liten"
#else
"Small"
#endif
	    " 20x20:",
	    ip_flags,
	    ICON_PICKER_ICON_POSITION_CENTER,
	    20, 20,				/* Icon size */
	    26, 26,				/* Display size */
	    80,					/* Label width */
	    icon_sel_dlg
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	icon_picker_set_switched_icon_cb(
	    w,
	    EDVMimeTypeEditDlgIconPickerIconSwitchedCB, d
	);
	icon_picker_set_changed_cb(
	    w,
	    EDVMimeTypeEditDlgAnyChangedCB, d
	);
	gtk_widget_show(w);


	/* Horizontal GtkSeparator */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons GtkHBox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;

	/* OK button */
	d->ok_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_ok_20x20_xpm, "OK", NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgOKCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_o, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_o);

	/* Apply button */
	d->apply_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_select_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Aplique"
#elif defined(PROG_LANGUAGE_FRENCH)
"Appliquer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Verwenden"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Applicare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Toepas"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Aplique"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Gjeld"
#else
"Apply"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgApplyCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_a, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_a);

	/* Cancel button */
	d->cancel_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_cancel_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cancele"
#elif defined(PROG_LANGUAGE_FRENCH)
"Annuler"
#elif defined(PROG_LANGUAGE_GERMAN)
"Heben"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Annullare"
#elif defined(PROG_LANGUAGE_DUTCH)
"Annuleer"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Cancelamento"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Kanseller"
#else
"Cancel"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);

	/* Close button */
	d->close_btn = w = GUIButtonPixmapLabelH(
	    (guint8 **)icon_close_20x20_xpm,
#if defined(PROG_LANGUAGE_SPANISH)
"Cierre"
#elif defined(PROG_LANGUAGE_FRENCH)
"Quitter"
#elif defined(PROG_LANGUAGE_GERMAN)
"Nah"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Vicino"
#elif defined(PROG_LANGUAGE_DUTCH)
"Einde"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Prximo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Nr"
#else
"Close"
#endif
	    , NULL
	);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_widget_set_usize(
	    w,
	    GUI_BUTTON_HLABEL_WIDTH_DEF, GUI_BUTTON_HLABEL_HEIGHT_DEF
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(EDVMimeTypeEditDlgCancelCB), d
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_c, GDK_CONTROL_MASK, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	GUIButtonLabelUnderline(w, GDK_c);


	EDVMimeTypeEditDlgUpdateMenus(d);

	d->freeze_count--;

	return(d);
}

/*
 *	Resets the MIME Types Edit Window's has changes marker.
 */
void EDVMimeTypeEditDlgResetHasChanges(
	edv_mime_type_edit_dlg_struct *d,
	const gboolean has_changes
)
{
	if(d == NULL)
	    return;

	if(d->has_changes == has_changes)
	    return;

	d->has_changes = has_changes;
	EDVMimeTypeEditDlgUpdateMenus(d);
}

/*
 *	Updates the MIE Types Edit Window's widgets to reflect current
 *	values.
 */
void EDVMimeTypeEditDlgUpdateMenus(edv_mime_type_edit_dlg_struct *d)
{
	gboolean	sensitive,
			has_changes,
			read_only;
	gint nrows, nselected, sel_row, mt_num;
	const EDVMIMEType *m;
	gchar *s;
	GList *glist;
	GtkCList *clist;
	EDVCore *core;

	if(d == NULL)
	    return;

	d->freeze_count++;

	core = d->core;

	has_changes = d->has_changes;
	mt_num = d->mt_num;
	m = EDV_MIME_TYPE(g_list_nth_data(
	    core->mime_types_list,
	    (guint)mt_num
	));
	read_only = (m != NULL) ? m->read_only : FALSE;

	/* Title */
	s = g_strconcat(
	    EDV_MIME_TYPE_EDIT_DLG_TITLE,
	    (has_changes) ? " (*)" : "",
	    NULL
	);
	gtk_window_set_title(GTK_WINDOW(d->toplevel), s);
	g_free(s);

	/* Commands GtkCList */
	clist = GTK_CLIST(d->commands_clist);
	nrows = clist->rows;
	nselected = g_list_length(clist->selection);
	glist = clist->selection_end;
	sel_row = (glist != NULL) ? (gint)glist->data : -1;
	sensitive = (nselected <= 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->command_add_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_add_mi, sensitive);
	sensitive = (nselected == 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->command_edit_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_edit_mi, sensitive);
	sensitive = (nselected >= 1) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->command_remove_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_remove_mi, sensitive);
	sensitive = ((g_list_find(clist->selection, (gpointer)0) == NULL) &&
	    (nselected > 0)) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->command_shift_up_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_shift_up_mi, sensitive);
	sensitive = ((g_list_find(clist->selection, (gpointer)(nrows - 1)) == NULL) &&
	    (nselected > 0)) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->command_shift_down_btn, sensitive);
	gtk_widget_set_sensitive(d->commands_menu_shift_down_mi, sensitive);

	/* OK, Apply, Cancel, Close Buttons */
	sensitive = (has_changes && !read_only) ? TRUE : FALSE;
	gtk_widget_set_sensitive(d->ok_btn, sensitive);
	gtk_widget_set_sensitive(d->apply_btn, sensitive);
	sensitive = has_changes;
	gtk_widget_set_sensitive(d->cancel_btn, sensitive);
	sensitive = (has_changes && !read_only) ? FALSE : TRUE;
	gtk_widget_set_sensitive(d->close_btn, sensitive);
	if(has_changes)
	{
	    gtk_widget_show(d->ok_btn);
	    gtk_widget_show(d->apply_btn);
	    gtk_widget_show(d->cancel_btn);
	    gtk_widget_hide(d->close_btn);
	}
	else
	{
	    gtk_widget_show(d->ok_btn);
	    gtk_widget_show(d->apply_btn);
	    gtk_widget_hide(d->cancel_btn);
	    gtk_widget_show(d->close_btn);
	}

	d->freeze_count--;
}

/*
 *	Checks if the MIME Types Edit Window is mapped.
 */
gboolean EDVMimeTypeEditDlgIsMapped(
	edv_mime_type_edit_dlg_struct *d
)
{
	if(d == NULL)
	    return(FALSE);

	return(GTK_WIDGET_MAPPED(d->toplevel));
}

/*
 *	Maps the MIME Types Edit Window.
 */
void EDVMimeTypeEditDlgMap(
	edv_mime_type_edit_dlg_struct *d
)
{
	if(d == NULL)
	    return;

	gtk_widget_show_raise(d->toplevel);
	PUListBoxGrabFocus(d->class_pulistbox);
}

/*
 *	Unmaps the MIME Types Edit Window.
 */
void EDVMimeTypeEditDlgUnmap(
	edv_mime_type_edit_dlg_struct *d
)
{
	if(d == NULL)
	    return;

	gtk_widget_hide(d->toplevel);
}

/*
 *	Deletes the MIME Types Edit Window.
 */
void EDVMimeTypeEditDlgDelete(
	edv_mime_type_edit_dlg_struct *d
)
{
	GtkCList *clist;

	if(d == NULL)
	    return;

	EDVMimeTypeEditDlgUnmap(d);

	d->freeze_count++;

	clist = GTK_CLIST(d->commands_clist);
	gtk_clist_freeze(clist);
	gtk_clist_clear(clist);
	gtk_clist_thaw(clist);

	d->icon_sel_dlg = IconSelDlgUnref(d->icon_sel_dlg);

	gtk_widget_destroy(d->commands_menu);

	gtk_widget_destroy(d->toplevel);
	gtk_accel_group_unref(d->accelgrp);

	(void)edv_pixmap_unref(d->mime_type_icon);
	(void)edv_pixmap_unref(d->run_in_terminal_icon);

	g_free(d->browse_path_rtn);

	d->freeze_count--;

	g_free(d);
}
