/*
			      PostScript Library

	Units legend:

	Dots and Pixels are the same thing, 1 dot is 1 pixel.

	All relativity is from the upper-left corner to the upper-left
	corner unless otherwise noted.

	Paper sizes are in pixels when the DPI is 100 unless otherwise
	noted.
 */

#ifndef LIBPS_H
#define LIBPS_H

#include <stdio.h>
#include <sys/types.h>
#if defined(WIN32)
# include "../include/os.h"
#endif


/*
 *	Default DPI:
 */
#define PSDefaultDPIX		72
#define PSDefaultDPIY		72

/*
 *	Error Codes:
 */
#define PSSuccess		0
#define PSError			1
#define PSBadValue		2		/* Invalid value or invalid format */
#define PSErrorSystem		3		/* Out of memory or out of disk space */
#define PSAbort			4		/* User aborted */
#define PSErrorInternal		5


/*
 *	Progress Function Type:
 */
typedef int (*PSProgressFunction)(
        void *progress_data,
        const int i, const int n,
        const int width, const int height,
        const int bpl, const int bpp,
        const u_int8_t *image_data
);


/*
 *	PostScript Format:
 */
typedef struct _PSFormatStruct		PSFormatStruct;
struct _PSFormatStruct {

	/* These flags define which members are defined */
#define PSFlagDPIX		(1 << 0)
#define PSFlagDPIY		(1 << 1)
#define PSFlagPaperWidth	(1 << 2)
#define PSFlagPaperHeight	(1 << 3)
#define PSFlagOutputX		(1 << 4)
#define PSFlagOutputY		(1 << 5)
	unsigned int	flags;

	int		dpi_x,			/* Dots per inch */
			dpi_y;

	int		paper_width,		/* Paper size at dpi = 100 */
			paper_height;

	int		output_x,		/* Output offset from upper-left, in pixels */
			output_y;

};
#define PS_FORMAT(p)		((PSFormatStruct *)(p))


/*
 *	Returns a statically allocated string describing the last
 *	error that occured or NULL if no error occured.
 */
extern const char *PSLastError(void);


/*
 *	Reads a PostScript image from the specified stream.
 *
 *	The returned image data will be 4 bytes per pixel in RGBA
 *	format, the calling function must delete the returned image
 *	data.
 *
 *	Returns PSSuccess on success or PSError on error. Can also
 *	return PSBadValue if the stream does not contain valid
 *	PostScript data.
 */
extern int PSReadRGBA(
	FILE *fp,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,			/* 4 bytes RGBA (will be modified) */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	PSProgressFunction progress_cb, void *progress_data,
	int *user_aborted
);

/*
 *	Reads a PostScript image from the specified file.
 *
 *	The returned image data will be 4 bytes per pixel in RGBA
 *	format, the calling function must delete the returned image
 *	data.
 *
 *	Returns PSSuccess on success or PSError on error. Can also
 *	return PSBadValue if the file is not a PostScript image.
 */
extern int PSReadFileRGBA(
	const char *path,
	int *width_rtn, int *height_rtn,
	int *bpl_rtn,
	u_int8_t **rgba_rtn,
	u_int8_t *bg_color,	/* 4 bytes RGBA (will be modified) */
	int *x_rtn, int *y_rtn,
	int *base_width_rtn, int *base_height_rtn,
	char **creator_rtn, char **title_rtn,
	char **author_rtn, char **comments_rtn,
	u_int8_t def_alpha_value,
	PSProgressFunction progress_cb, void *progress_data,
	int *user_aborted
);


/*
 *      Writes a PostScript image to the specified stream.
 *
 *	The specified image data must be 4 bytes per pixel RGBA format
 *	and the PostScript image file will be written in RGB color.
 *
 *      Returns PSSuccess on success or PSError on error. Can also
 *	return PSBadValue if any of the specified values are invalid.
 */
extern int PSWriteFilePS3RGBA(
	FILE *fp,
	PSFormatStruct *fmt,
	int width, int height,	/* Size of image in pixels */
	int bpl,		/* Bytes per line */
	u_int8_t *rgba,
	const char *creator, const char *title,
	const char *author, const char *comments,
	PSProgressFunction progress_cb, void *progress_data,
	int *user_aborted
);

/*
 *	Writes a PostScript image to the specified stream.
 *
 *	The specified image data must be 3 bytes per pixel RGB format 
 *	and the PostScript image file will be written in RGB color.
 *
 *	Returns PSSuccess on success or PSError on error. Can also
 *	return PSBadValue if any of the specified values are invalid.
 */
extern int PSWriteFilePS3RGB(
	FILE *fp,
	PSFormatStruct *fmt,
	int width, int height,	/* Size of image in pixels */
	int bpl,		/* Bytes per line */
	u_int8_t *rgb,
	const char *creator, const char *title,
	const char *author, const char *comments,
	PSProgressFunction progress_cb, void *progress_data,
	int *user_aborted
);

/*
 *	Writes a PostScript image to the specified stream.
 *
 *      The specified image data must be 1 bytes per pixel Greyscale
 *	format and the PostScript image file will be written in RGB
 *	color.
 *
 *	Returns PSSuccess on success or PSError on error. Can also
 *	return PSBadValue if any of the specified values are invalid.
 */
extern int PSWriteFilePS3Grey(
	FILE *fp,
	PSFormatStruct *fmt,
	int width, int height,	/* Size of image in pixels */
	int bpl,		/* Bytes per line */
	u_int8_t *grey,
	const char *creator, const char *title,
	const char *author, const char *comments,
	PSProgressFunction progress_cb, void *progress_data,
	int *user_aborted
);


#endif	/* LIBPS_H */
