#include <stdlib.h>
#include <string.h>
#include <glib.h>

#include "edv_utils.h"
#include "edv_process.h"
#include "edv_path.h"
#include "edv_interps.h"
#include "edv_context_private.h"
#include "edv_window.h"
#include "edv_cfg_list.h"
#include "config.h"


static gchar *edv_window_evaluate_path(const gchar *path);
static void edv_window_realize_command(
	EDVContext *ctx,
	const gchar *cmd
);


/* About Dialog */
void edv_window_about_dialog_map(
	EDVContext *ctx,
	const gchar *page_name
);

/* VFS Browser */
void edv_window_vfs_browser_new(
        EDVContext *ctx,
        const gchar *path,
        const gchar *tree_origin
);

/* Image Browser */
void edv_window_image_browser_new(
	EDVContext *ctx,
	const gchar *path
);

/* Archiver */
void edv_window_archiver_new(
        EDVContext *ctx,
        const gchar *path,
        const gchar *password
);

/* Recycle Bin */
void edv_window_recycle_bin_map(EDVContext *ctx);

/* MIME Types List Window */
void edv_window_mime_types_list_map(
	EDVContext *ctx,
	const gchar *type
);

/* Devices List Window */
void edv_window_devices_list_map(
	EDVContext *ctx,
	const gchar *path
);

/* History List Window */
void edv_window_history_list_map(
        EDVContext *ctx,
        const gint event_index
);

/* Options Window */
void edv_window_options_map(
        EDVContext *ctx,
        const gchar *page_name
);

/* Customize Window */
void edv_window_customize_map(
        EDVContext *ctx,
        const gchar *page_name
);


/* Properties Dialog */
void edv_window_properties_dialog_new(
	EDVContext *ctx,
	const gchar *path,
	const gchar *page_name
);

/* Find Window */
void edv_window_find_map(
        EDVContext *ctx,
        const gchar *location,
        const gchar *value 
);

/* Run Dialog */
void edv_window_run_dialog_map(
        EDVContext *ctx,
        const gchar *command,
        const gchar *working_directory_path
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Checks if the given path is not an absolute path, if it is
 *	not then the current working directory will be prefixed to it.
 *
 *	In addition '~' prefix substitutions will be performed.
 */
static gchar *edv_window_evaluate_path(const gchar *path)
{
	gchar	*cwd = edv_getcwd(),
		*dpath = edv_path_evaluate(
			cwd,
			path
		);
	g_free(cwd);
	return(dpath);
}

/*
 *	Executes the command.
 *
 *	If Endeavour is running then this call will block until the
 *	executed process has finished, otherwise the process is
 *	executed async.
 *
 *	The cmd specifies the entire command that will be used to
 *	run Endeavour and instruct it to start up with the window
 *	that we want with our specified arguments.
 */
static void edv_window_realize_command(
	EDVContext *ctx,
	const gchar *cmd
)
{
/* TODO Add --no-splash or --safe-mode arguments */
	CfgList *cfg_list = ctx->cfg_list;
	gchar *shell_prog;
	const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
			*shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	);

	/* Check if Endeavour is already running */
	if(edv_interps_get_lock(cfg_list) > 0)
	{
		/* Run Endeavour and block, in which case the process
		 * that we run will send an InterPS command to the
		 * original process to open a new window with our
		 * specified arguments
		 */
		gint status;
		(void)edv_system_wait_shell(
			cmd,
			shell_prog,
			shell_args,
			&status
		);
	}
	else
	{
		/* Run Endeavour and instruct it to open a new window
		 * with our specified arguments
		 */
		gchar *shell_prog;
		const gchar	*shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL),
				*shell_args = edv_strarg(
			shell_cmd,
			&shell_prog,
			TRUE,			/* Parse escapes */
			TRUE			/* Parse quotes */
		);
		const gint pid = edv_system_shell(
			cmd,
			shell_prog,
			shell_args
		);
		if(pid > -1)
		{
			/* Since all window mapping functions must
			 * block until a window has been created, we
			 * need to wait for Endeavour to create the
			 * InterPS lock link before returning
			 */
			gulong elapsed_sec;
			GTimer *timer = g_timer_new();
			g_timer_start(timer);
			while(edv_interps_get_lock(cfg_list) <= 0)
			{
				elapsed_sec = (gulong)(g_timer_elapsed(timer, NULL) / 1000000.0f);

				/* Give up after 5 minutes */
				if(elapsed_sec > (5l * 60l))
					break;

				edv_usleep(8000l);
			}
			g_timer_destroy(timer);
		}
	}

	g_free(shell_prog);
}


/*
 *	Maps the About Dialog.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	If page_name is not NULL then it specifies the name of the
 *	page to switch to.
 */
void edv_window_about_dialog_map(
	EDVContext *ctx,
	const gchar *page_name
)
{
	const gchar *win_name_arg = edv_window_type_to_window_name(
		EDV_WINDOW_ABOUT_DIALOG
	);
	gchar *cmd;

	if(ctx == NULL)
		return;

	if(!STRISEMPTY(page_name))
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"%s\"",
			ctx->prog_path,
			win_name_arg,
			page_name
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\"",
			ctx->prog_path,
			win_name_arg
		);
	edv_window_realize_command(
		ctx,
		cmd
	);
	g_free(cmd);
}


/*
 *	Creates a new VFS Browser.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	If path is not NULL then it specifies the initial location..
 *
 *	If tree_origin is not NULL then it specifies the initial
 *	directory tree origin path.
 */
void edv_window_vfs_browser_new(
        EDVContext *ctx,
        const gchar *path,
        const gchar *tree_origin
)
{
	const gchar *win_name_arg = edv_window_type_to_window_name(
		EDV_WINDOW_VFS_BROWSER
	);
	gchar *cmd;

	if(ctx == NULL)
		return;

	if((path != NULL) && (tree_origin != NULL))
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"%s\" \"%s\"",
			ctx->prog_path,
			win_name_arg,
			path,
			tree_origin
		);
	else if(path != NULL)
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"%s\"",
			ctx->prog_path,
			win_name_arg,
			path
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\"",
			ctx->prog_path,
			win_name_arg
		);
	edv_window_realize_command(
		ctx,
		cmd
	);
	g_free(cmd);
}

/*
 *	Creates a new Image Browser.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	If path is not NULL then it specifies the initial location..
 */
void edv_window_image_browser_new(
	EDVContext *ctx,
	const gchar *path
)
{
	const gchar *win_name_arg = edv_window_type_to_window_name(
		EDV_WINDOW_IMAGE_BROWSER
	);
	gchar *cmd;

	if(ctx == NULL)
		return;

	if(!STRISEMPTY(path))
	{
		gchar *dpath = edv_window_evaluate_path(path);
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"%s\"",
			ctx->prog_path,
			win_name_arg,
			dpath
		);
		g_free(dpath);
	}
	else
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\"",
			ctx->prog_path,
			win_name_arg
		);
	edv_window_realize_command(
		ctx,
		cmd
	);
	g_free(cmd);
}

/*
 *      Creates a new Archiver.
 *
 *      The ctx specifies the Endeavour 2 context.
 *
 *      If path is not NULL then it specifies the initial location..
 *
 *      If password is not NULL then it specifies the password.
 */
void edv_window_archiver_new(
        EDVContext *ctx,
        const gchar *path,
        const gchar *password
)
{
	const gchar *win_name_arg = edv_window_type_to_window_name(
		EDV_WINDOW_ARCHIVER
	);
	gchar *cmd;

	if(ctx == NULL)
		return;

	if((path != NULL) && (password != NULL))
	{
		gchar *dpath = edv_window_evaluate_path(path);
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"%s\" \"%s\"",
			ctx->prog_path,
			win_name_arg,
			dpath,
			password
		);
		g_free(dpath);
	}
	else if(path != NULL)
	{
		gchar *dpath = edv_window_evaluate_path(path);
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"%s\"",
			ctx->prog_path,
			win_name_arg,
			dpath
		);
		g_free(dpath);
	}
	else
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\"",
			ctx->prog_path,
			win_name_arg
		);
	edv_window_realize_command(
		ctx,
		cmd
	);
	g_free(cmd);
}

/*
 *	Maps the Recycle Bin.
 *
 *	The ctx specifies the Endeavour 2 context.
 */
void edv_window_recycle_bin_map(EDVContext *ctx)
{
	const gchar *win_name_arg = edv_window_type_to_window_name(
		EDV_WINDOW_RECYCLE_BIN
	);
	gchar *cmd;

	if(ctx == NULL)
		return;

	cmd = g_strdup_printf(
		"\"%s\" \"--%s\"",
		ctx->prog_path,
		win_name_arg
	);
	edv_window_realize_command(
		ctx,
		cmd
	);
	g_free(cmd);
}


/*
 *	Maps the MIME Types List Window.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	If type is not NULL then it specifies the initial MIME Type
 *	to display.
 */
void edv_window_mime_types_list_map(
	EDVContext *ctx,
	const gchar *type
)
{
	const gchar *win_name_arg = edv_window_type_to_window_name(
		EDV_WINDOW_MIME_TYPES_LIST
	);
	gchar *cmd;

	if(ctx == NULL)
		return;

	if(!STRISEMPTY(type))
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"\" \"%s\"",
			ctx->prog_path,
			win_name_arg,
			type
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\"",
			ctx->prog_path,
			win_name_arg
		);
	edv_window_realize_command(
		ctx,
		cmd
	);
	g_free(cmd);
}

/*
 *	Maps the Devices List Window.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	If path is not NULL then it specifies the path to an object
 *	on the initial Device to display, including the device's
 *	mount path or device path itself.
 */
void edv_window_devices_list_map(
	EDVContext *ctx,
	const gchar *path
)
{
	const gchar *win_name_arg = edv_window_type_to_window_name(
		EDV_WINDOW_DEVICES_LIST
	);
	gchar *cmd;

	if(ctx == NULL)
		return;

	if(!STRISEMPTY(path))
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"%s\"",
			ctx->prog_path,
			win_name_arg,
			path
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\"",
			ctx->prog_path,
			win_name_arg
		);
	edv_window_realize_command(
		ctx,
		cmd
	);
	g_free(cmd);
}

/*
 *	Maps the History List Window.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	If event_index is not negative then it specifies the initial
 *	event to display.
 */
void edv_window_history_list_map(
        EDVContext *ctx,
        const gint event_index
)
{
	const gchar *win_name_arg = edv_window_type_to_window_name(
		EDV_WINDOW_HISTORY_LIST
	);
	gchar *cmd;

	if(ctx == NULL)
		return;

	if(event_index > -1)
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"\" \"%i\"",
			ctx->prog_path,
			win_name_arg,
			event_index
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\"",
			ctx->prog_path,
			win_name_arg
		);
	edv_window_realize_command(
		ctx,
		cmd
	);
	g_free(cmd);
}


/*
 *	Maps the Options Window.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	If page_name is not NULL then it specifies the name of the
 *	page to switch to.
 */
void edv_window_options_map(
        EDVContext *ctx,
        const gchar *page_name
)
{
	const gchar *win_name_arg = edv_window_type_to_window_name(
		EDV_WINDOW_OPTIONS
	);
	gchar *cmd;

	if(ctx == NULL)
		return;

	if(!STRISEMPTY(page_name))
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"\" \"%s\"",
			ctx->prog_path,
			win_name_arg,
			page_name
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\"",
			ctx->prog_path,
			win_name_arg
		);
	edv_window_realize_command(
		ctx,
		cmd
	);
	g_free(cmd);
}


/*
 *      Maps the Customize Window.
 *
 *      The ctx specifies the Endeavour 2 context.
 *
 *      If page_name is not NULL then it specifies the name of the
 *      page to switch to.
 */
void edv_window_customize_map(
        EDVContext *ctx,
        const gchar *page_name
)
{
	const gchar *win_name_arg = edv_window_type_to_window_name(
		EDV_WINDOW_CUSTOMIZE
	);
	gchar *cmd;

	if(ctx == NULL)
		return;

	if(!STRISEMPTY(page_name))
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"\" \"%s\"",
			ctx->prog_path,
			win_name_arg,
			page_name
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\"",
			ctx->prog_path,
			win_name_arg
		);
	edv_window_realize_command(
		ctx,
		cmd
	);
	g_free(cmd);
}


/*
 *	Creates a new Properties Dialog.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	The path specifies the initial object to display.
 *
 *	If page_name is not NULL then it specifies the name of the
 *	page to switch to.
 */
void edv_window_properties_dialog_new(
        EDVContext *ctx,
        const gchar *path,
        const gchar *page_name
)
{
	const gchar *win_name_arg = edv_window_type_to_window_name(
		EDV_WINDOW_PROPERTIES_DIALOG
	);
	gchar *cmd;

	if(ctx == NULL)
		return;

	if(!STRISEMPTY(path))
	{
		gchar *dpath = edv_window_evaluate_path(path);

		if(!STRISEMPTY(page_name))
			cmd = g_strdup_printf(
				"\"%s\" \"--%s\" \"%s\" \"%s\"",
				ctx->prog_path,
				win_name_arg,
				dpath,
				page_name
			);
		else
			cmd = g_strdup_printf(
				"\"%s\" \"--%s\" \"%s\"",
				ctx->prog_path,
				win_name_arg,
				dpath
			);

		g_free(dpath);
	}
	else
	{
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\"",
			ctx->prog_path,
			win_name_arg
		);
	}
	edv_window_realize_command(
		ctx,
		cmd
	);
	g_free(cmd);
}


/*
 *	Maps the Find Window.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	If location is not NULL then location specifies the initial
 *	location.
 *
 *	If value is not NULL then it specifies the initial search
 *	value.
 */
void edv_window_find_map(
        EDVContext *ctx,
        const gchar *location,
        const gchar *value 
)
{
	const gchar *win_name_arg = edv_window_type_to_window_name(
		EDV_WINDOW_FIND
	);
	gchar *cmd;

	if(ctx == NULL)
		return;

	if(!STRISEMPTY(value))
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"%s\" \"%s\"",
			ctx->prog_path,
			win_name_arg,
			STRISEMPTY(location) ? "" : location,
			value
		);
	else if(!STRISEMPTY(location))
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"%s\"",
			ctx->prog_path,
			win_name_arg,
			location
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\"",
			ctx->prog_path,
			win_name_arg
		);
	edv_window_realize_command(
		ctx,
		cmd
	);
	g_free(cmd);
}


/*
 *	Maps the Run Dialog.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	If command is not NULL then it specifies the initial command.
 *
 *	If working_directory_path is not NULL then it specifies the
 *	iniital working directory path.
 */
void edv_window_run_dialog_map(
        EDVContext *ctx,
        const gchar *command,
        const gchar *working_directory_path
)
{
	const gchar *win_name_arg = edv_window_type_to_window_name(
		EDV_WINDOW_RUN_DIALOG
	);
	gchar *cmd;

	if(ctx == NULL)
		return;

	if(!STRISEMPTY(working_directory_path))
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"%s\" \"%s\"",
			ctx->prog_path,
			win_name_arg,
			STRISEMPTY(command) ? "" : command,
			working_directory_path
		);
	else if(!STRISEMPTY(command))
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\" \"%s\"",
			ctx->prog_path,
			win_name_arg,
			command
		);
	else
		cmd = g_strdup_printf(
			"\"%s\" \"--%s\"",
			ctx->prog_path,
			win_name_arg
		);
	edv_window_realize_command(
		ctx,
		cmd
	);
	g_free(cmd);
}
