#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <glib.h>

#include "edv_types.h"
#include "edv_utils.h"
#include "edv_path.h"
#include "edv_link.h"
#include "edv_vfs_obj.h"
#include "edv_vfs_obj_stat.h"


static void edv_vfs_object_set_stat(
	EDVVFSObject *obj,
	struct stat *lstat_buf
);
static void edv_vfs_object_update_link_flags(EDVVFSObject *obj);

EDVVFSObject *edv_vfs_object_stat(const gchar *path);
EDVVFSObject *edv_vfs_object_lstat(const gchar *path);
EDVVFSObject *edv_vfs_object_fstat(const gint fd);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Sets the VFS object's statistics from struct stat.
 *
 *	The obj specifies the VFS object.
 *
 *	The lstat_buf specifies the struct stat.
 *
 *	Since the struct stat contains less information than the VFS
 *	object, not all of the VFS object's members will be set by
 *	this call.
 *
 *	The VFS object's name and path will not be set by this call
 *	the calling function should call edv_vfs_object_set_path*() as
 *	needed before this call.
 *
 *	The VFS object's link_hints will not be set by this call, the
 *	calling function should call edv_vfs_object_update_link_flags() as
 *	needed after this call.
 *
 *	The object's type, permissions, access_time, modify_time,
 *	change_time, owner_id, group_id, hard_links, size, device,
 *	inode, device_type, block_size, and blocks will be set.
 */
static void edv_vfs_object_set_stat(
	EDVVFSObject *obj,
	struct stat *lstat_buf
)
{
	mode_t m;

	if((obj == NULL) || (lstat_buf == NULL))
	    return;

	m = lstat_buf->st_mode;

	/* Type */
	obj->type = edv_stat_mode_to_object_type(m);

	/* Index */
	obj->device_index = (gulong)lstat_buf->st_dev;
	obj->index = (gulong)lstat_buf->st_ino;

	/* Size */
	obj->size = (gulong)lstat_buf->st_size;

	/* Assume link is valid when setting stats, calling function
	 * may update this afterwards
	 */
	obj->link_hint_flags = EDV_LINK_HINT_TARGET_EXISTS;

	/* Permissions */
	obj->permissions = edv_stat_mode_to_edv_permissions(m);

	/* Time Stamps */
	obj->access_time = (gulong)lstat_buf->st_atime;
	obj->modify_time = (gulong)lstat_buf->st_mtime;
	obj->change_time = (gulong)lstat_buf->st_ctime;

	/* Ownership */
	obj->owner_id = (gint)lstat_buf->st_uid;
	obj->group_id = (gint)lstat_buf->st_gid;

	/* Device Type */
	obj->device_type = (gint)lstat_buf->st_rdev;

	/* Block transfer size and blocks allocated */
	obj->block_size = (gulong)lstat_buf->st_blksize;
	obj->blocks = (gulong)lstat_buf->st_blocks;

	/* Number of hard links */
	obj->hard_links = (gint)lstat_buf->st_nlink;

#ifdef S_ISLNK
	/* If this is a link then set the link target */
	if(S_ISLNK(m))
	{
	    g_free(obj->link_target);
	    obj->link_target = edv_link_get_target(obj->path);
	}
#endif
}

/*
 *	Updates the VFS Object's link_hint_flags.
 */
static void edv_vfs_object_update_link_flags(EDVVFSObject *obj)
{
	struct stat stat_buf;
	gchar           *path,
			*parent;

	if(obj == NULL)
	    return;

	obj->link_hint_flags = 0;

	if(!EDV_VFS_OBJECT_IS_LINK(obj) ||
	   (obj->path == NULL) || (obj->link_target == NULL)
	)
	    return;

	parent = g_dirname(obj->path);
	if(parent == NULL)
	    return;

	/* Get the full path to the link's target */
	path = edv_path_evaluate(
	    parent,
	    obj->link_target
	);
	if(path == NULL)
	{
	    g_free(parent);
	    return;
	}

	/* Get the link target's destination statistics */
	if(!stat((const char *)path, &stat_buf))
	{
	    obj->link_hint_flags |= EDV_LINK_HINT_TARGET_EXISTS;
#ifdef S_ISDIR
	    if(S_ISDIR(stat_buf.st_mode))
	    {
		obj->link_hint_flags |= EDV_LINK_HINT_TARGET_DIRECTORY;
		if(edv_link_is_infinately_recursive(obj->path))
		    obj->link_hint_flags |= EDV_LINK_HINT_TARGET_GRAND_PARENT;
	    }
#endif
	}

	g_free(path);
	g_free(parent);
}


/*
 *	Gets the VFS object's destination statistics.
 *
 *	The path specifies the full path to the object.
 *
 *	Returns a new dynamically allocated EDVVFSObject with all of its
 *	statistics filled or NULL on error.
 */
EDVVFSObject *edv_vfs_object_stat(const gchar *path)
{
	struct stat stat_buf;
	EDVVFSObject *obj;

	if(STRISEMPTY(path))
	{
	    errno = EINVAL;
	    return(NULL);
	}

	if(stat((const char *)path, &stat_buf))
	    return(NULL);

	obj = edv_vfs_object_new();
	if(obj == NULL)
	    return(NULL);

	edv_vfs_object_set_path(obj, path);
	edv_vfs_object_set_stat(obj, &stat_buf);
	edv_vfs_object_update_link_flags(obj);

	return(obj);
}

/*
 *	Gets the VFS object's local statistics.
 *
 *	The path specifies the full path to the object.
 *
 *	Returns a new dynamically allocated EDVVFSObject with all of its
 *	statistics filled or NULL on error.
 */
EDVVFSObject *edv_vfs_object_lstat(const gchar *path)
{
	struct stat lstat_buf;
	EDVVFSObject *obj;

	if(STRISEMPTY(path))
	{
	    errno = EINVAL;
	    return(NULL);
	}

	if(lstat((const char *)path, &lstat_buf))
	    return(NULL);

	obj = edv_vfs_object_new();
	if(obj == NULL)
	    return(NULL);

	edv_vfs_object_set_path(obj, path);
	edv_vfs_object_set_stat(obj, &lstat_buf);
	edv_vfs_object_update_link_flags(obj);

	return(obj);
}

/*
 *	Gets the VFS object's statistics from the descriptor.
 *
 *	The fd specifies the descriptor. Since the descriptor does not
 *	contain the path, the VFS object's name and full_path will not
 *	be set.
 *
 *	Returns a new dynamically allocated EDVVFSObject with all of its
 *	statistics filled or NULL on error.
 */
EDVVFSObject *edv_vfs_object_fstat(const gint fd)
{
	struct stat stat_buf;
	EDVVFSObject *obj;

	if(fd < 0)
	{
	    errno = EINVAL;
	    return(NULL);
	}

	if(fstat((int)fd, &stat_buf))
	    return(NULL);

	obj = edv_vfs_object_new();
	if(obj == NULL)
	    return(NULL);

	edv_vfs_object_set_stat(obj, &stat_buf);
	edv_vfs_object_update_link_flags(obj);

	return(obj);
}
