/*
		      Endeavour Mark II - VFS Object
 */

#ifndef EDV_VFS_OBJ_H
#define EDV_VFS_OBJ_H

#include <glib.h>
#include "edv_types.h"

typedef struct _EDVVFSObject		EDVVFSObject;
#define EDV_VFS_OBJECT(p)		((EDVVFSObject *)(p))


/*
 *	VFS Object:
 */
struct _EDVVFSObject {

	/* Type */
	EDVObjectType	type;

	/* Index Location */
	gulong		device_index,		/* Index of the device that
						 * this object resides on */

			index;			/* Index on the device that
						 * this object resides at */

	/* Name & Location */
	gchar		*name,
			*path;			/* Full path with name */

	/* Size (in bytes) */
	gulong		size;

	/* Link (for type EDV_OBJECT_TYPE_LINK) */
	gchar		*link_target;
	EDVLinkHintFlags	link_hint_flags;

	/* Permissions */
	EDVPermissionFlags	permissions;

	/* Time Stamps (in seconds since EPOCH) */
	gulong		access_time,
			modify_time,
			change_time;

	/* Ownership */
	gint		owner_id,
			group_id;

	/* Device Type (for types EDV_OBJECT_TYPE_DEVICE_*) */
	gint		device_type;		/* Device major and minor
						 * numbers (used only if
						 * type is one of
						 * EDV_OBJECT_TYPE_DEVICE_*
						 *
						 * Use edv_device_numbers_parse()
						 * to parse this value */

	/* Misc */
	gulong		block_size;		/* IO size in bytes for
						 * reading/writing from/to
						 * this object */
	gulong		blocks;			/* Blocks allocated for this
						 * object */

	gint		hard_links;		/* Number of hard links to
						 * this object */

	/* Meta Data List */
	GList           *meta_data_list;	/* GList of EDVProperty *
						 * properties */

};

/*
 *	Check Type Macros:
 */
#define EDV_VFS_OBJECT_IS_FILE(p)	\
 (((p) != NULL) ? ((p)->type == EDV_OBJECT_TYPE_FILE) : FALSE)
#define EDV_VFS_OBJECT_IS_DIRECTORY(p)	\
 (((p) != NULL) ? ((p)->type == EDV_OBJECT_TYPE_DIRECTORY) : FALSE)
#define EDV_VFS_OBJECT_IS_LINK(p)	\
 (((p) != NULL) ? ((p)->type == EDV_OBJECT_TYPE_LINK) : FALSE)
#define EDV_VFS_OBJECT_IS_DEVICE_BLOCK(p)	\
 (((p) != NULL) ? ((p)->type == EDV_OBJECT_TYPE_DEVICE_BLOCK) : FALSE)
#define EDV_VFS_OBJECT_IS_DEVICE_CHARACTER(p)	\
 (((p) != NULL) ? ((p)->type == EDV_OBJECT_TYPE_DEVICE_CHARACTER) : FALSE)
#define EDV_VFS_OBJECT_IS_FIFO(p)	\
 (((p) != NULL) ? ((p)->type == EDV_OBJECT_TYPE_FIFO) : FALSE)
#define EDV_VFS_OBJECT_IS_SOCKET(p)	\
 (((p) != NULL) ? ((p)->type == EDV_OBJECT_TYPE_SOCKET) : FALSE)

/*
 *	Check Link Hints Macros:
 */
#define EDV_VFS_OBJECT_LINK_TARGET_EXISTS(p)	\
 (((p) != NULL) ? ((p)->link_hint_flags & EDV_LINK_HINT_TARGET_EXISTS) : FALSE)
#define EDV_VFS_OBJECT_IS_LINK_TARGET_DIRECTORY(p)	\
 (((p) != NULL) ? ((p)->link_hint_flags & EDV_LINK_HINT_TARGET_DIRECTORY) : FALSE)
#define EDV_VFS_OBJECT_IS_LINK_TARGET_GRAND_PARENT(p)	\
 (((p) != NULL) ? ((p)->link_hint_flags & EDV_LINK_HINT_TARGET_GRAND_PARENT) : FALSE)


/*
 *	Creates a new EDVVFSObject.
 *
 *	Returns a new dynamically allocated EDVVFSObject with all of its
 *	values zero'ed or NULL on error.
 */
extern EDVVFSObject *edv_vfs_object_new(void);

/*
 *	Creates a new EDVVFSObject of a specific type.
 *
 *	The type specifies the EDVVFSObject's type.
 *
 *	Returns a new dynamically allocated EDVVFSObject with its type
 *	value set or NULL on error.
 */
extern EDVVFSObject *edv_vfs_object_new_type(const EDVObjectType type);

/*
 *	Creates a new EDVVFSObject of type EDV_VFS_OBJECT_TYPE_ERROR.
 *
 *	The path specifies the full path to the error (which may or
 *	may not exist since this is an error).
 *
 *	The error_code specifies the errno error code.
 *
 *	The error_msg specifies the error message that will be set on
 *	the new EDVVFSObject's meta data properties list.
 *
 *	Returns a new dynamically allocated EDVVFSObject with its values
 *	set to describe the error or NULL on error.
 */
extern EDVVFSObject *edv_vfs_object_new_error(
	const gchar *path,
	const gint error_code,
	const gchar *error_msg
);

/*
 *	Coppies the EDVVFSObject.
 *
 *	The obj specifies the EDVVFSObject to copy.
 *
 *	Returns a new dynamically allocated copy of the EDVVFSObject or
 *	NULL on error.
 */
extern EDVVFSObject *edv_vfs_object_copy(EDVVFSObject *obj);

/*
 *	Sets the EDVVFSObject's name and full path from the specified
 *	path.
 *
 *	The obj specifies the EDVVFSObject to be set.
 *
 *	The path specifies the full path to derive the EDVVFSObject's
 *	name and full path from. If the path is NULL then the name
 *	and full path will be cleared.
 */
extern void edv_vfs_object_set_path(
	EDVVFSObject *obj,
	const gchar *path
);

/*
 *	Sets the EDVVFSObject's name and full path explicitly.
 *
 *	The obj specifies the EDVVFSObject to be set.
 *
 *	The name specifies the new name. If name is NULL then the VFS
 *	object's name will be cleared
 *
 *	The path specifies the new full path. If path is NULL then
 *	the EDVVFSObject's path will be cleared
 */
extern void edv_vfs_object_set_path2(
	EDVVFSObject *obj,
	const gchar *name,
	const gchar *path
);

/*
 *	Sets the EDVVFSObject's values from another EDVVFSObject.
 *
 *	The obj specifies the EDVVFSObject to be set.
 *
 *	The obj_values specifies the values to set obj with.
 */
extern void edv_vfs_object_set_object(
	EDVVFSObject *obj,			/* Target */
	EDVVFSObject *obj_values		/* Source */
);

/*
 *	Deletes and zeros all the values on the EDVVFSObject.
 *
 *	The obj specifies the EDVVFSObject who's values are to be cleared.
 */
extern void edv_vfs_object_clear(EDVVFSObject *obj);

/*
 *	Deletes the EDVVFSObject.
 *
 *	The obj specifies the EDVVFSObject to delete.
 */
extern void edv_vfs_object_delete(EDVVFSObject *obj);


#endif	/* EDV_VFS_OBJ_H */
