#include <errno.h>

#include <glib.h>

#include "edv_types.h"
#include "edv_utils.h"
#include "edv_path.h"
#include "edv_link.h"
#include "edv_property.h"
#include "edv_vfs_obj.h"


/* VFS Object */
EDVVFSObject *edv_vfs_object_new(void);
EDVVFSObject *edv_vfs_object_new_type(const EDVObjectType type);
EDVVFSObject *edv_vfs_object_new_error(
	const gchar *path,
	const gint error_code,
	const gchar *error_msg
);
EDVVFSObject *edv_vfs_object_copy(EDVVFSObject *obj);
void edv_vfs_object_set_path(
	EDVVFSObject *obj,
	const gchar *path
);
void edv_vfs_object_set_path2(
	EDVVFSObject *obj,
	const gchar *name,
	const gchar *path
);
void edv_vfs_object_set_object(
	EDVVFSObject *obj,			/* Target */
	EDVVFSObject *obj_values		/* Source */
);
void edv_vfs_object_clear(EDVVFSObject *obj);
void edv_vfs_object_delete(EDVVFSObject *obj);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new EDVVFSObject.
 *
 *	Returns a new dynamically allocated EDVVFSObject with all of its
 *	values zero'ed or NULL on error.
 */
EDVVFSObject *edv_vfs_object_new(void)
{
	return(EDV_VFS_OBJECT(g_malloc0(sizeof(EDVVFSObject))));
}

/*
 *	Creates a new EDVVFSObject of a specific type.
 *
 *	The type specifies the EDVVFSObject's type.
 *
 *	Returns a new dynamically allocated EDVVFSObject with its type
 *	value set or NULL on error.
 */
EDVVFSObject *edv_vfs_object_new_type(const EDVObjectType type)
{
	EDVVFSObject *obj = edv_vfs_object_new();
	if(obj == NULL)
	    return(NULL);

	obj->type = type;

	return(obj);
}

/*
 *	Creates a new EDVVFSObject of type EDV_VFS_OBJECT_TYPE_ERROR.
 *
 *	The path specifies the full path to the error (which may or
 *	may not exist since this is an error).
 *
 *	The error_code specifies the errno error code.
 *
 *	The error_msg specifies the error message that will be set on
 *	the new EDVVFSObject's meta data properties list.
 *
 * 	Returns a new dynamically allocated EDVVFSObject with its values
 *	set to describe the error or NULL on error.
 */
EDVVFSObject *edv_vfs_object_new_error(
	const gchar *path,
	const gint error_code,
	const gchar *error_msg
)
{
	EDVVFSObject *obj = edv_vfs_object_new_type(EDV_OBJECT_TYPE_ERROR);
	if(obj == NULL)
	    return(NULL);

	edv_vfs_object_set_path(obj, path);
	if(!STRISEMPTY(error_msg))
	    obj->meta_data_list = edv_properties_list_set_s(
		obj->meta_data_list,
		EDV_PROP_NAME_ERROR,
		error_msg,
		TRUE
	    );

	return(obj);
}

/*
 *	Coppies the EDVVFSObject.
 *
 *	The obj specifies the EDVVFSObject to copy.
 *
 *	Returns a new dynamically allocated copy of the EDVVFSObject or
 *	NULL on error.
 */
EDVVFSObject *edv_vfs_object_copy(EDVVFSObject *obj)
{
	EDVVFSObject *new_obj;

	if(obj == NULL)
	{
	    errno = EINVAL;
	    return(NULL);
	}

	new_obj = edv_vfs_object_new();
	if(new_obj == NULL)
	    return(NULL);

	new_obj->type = obj->type;

	new_obj->device_index = obj->device_index;
	new_obj->index = obj->index;

	new_obj->name = STRDUP(obj->name);
	new_obj->path = STRDUP(obj->path);

	new_obj->size = obj->size;

	new_obj->link_target = STRDUP(obj->link_target);
	new_obj->link_hint_flags = obj->link_hint_flags;

	new_obj->permissions = obj->permissions;

	new_obj->access_time = obj->access_time;
	new_obj->modify_time = obj->modify_time;
	new_obj->change_time = obj->change_time;

	new_obj->owner_id = obj->owner_id;
	new_obj->group_id = obj->group_id;

	new_obj->device_type = obj->device_type;

	new_obj->block_size = obj->block_size;
	new_obj->blocks = obj->blocks;

	new_obj->hard_links = obj->hard_links;

	new_obj->meta_data_list = edv_properties_list_copy(obj->meta_data_list);

	return(new_obj);
}

/*
 *	Sets the EDVVFSObject's name and full path from the specified
 *	path.
 *
 *	The obj specifies the EDVVFSObject to be set.
 *
 *	The path specifies the full path to derive the EDVVFSObject's
 *	name and full path from. If the path is NULL then the name
 *	and full path will be cleared.
 */
void edv_vfs_object_set_path(
	EDVVFSObject *obj,
	const gchar *path
)
{
	if(obj == NULL)
	    return;

	/* Reset the Object's name and full path first */
	g_free(obj->name);
	obj->name = NULL;
	g_free(obj->path);
	obj->path = NULL;

	/* Enough information to set the name and full path? */
	if(!STRISEMPTY(path))
	{
	    /* Set the full path, assuming it is an absolute path */
	    obj->path = g_strdup(path);
	    if(obj->path != NULL)
	    {
		const gchar *name;

		/* Remove any tailing deliminators */
		edv_path_simplify(obj->path);

		/* Get the name from the full path */
		name = g_basename(obj->path);
		if(STRISEMPTY(name))
		    obj->name = g_strdup(obj->path);
		else
		    obj->name = g_strdup(name);
	    }
	}
}

/*
 *	Sets the EDVVFSObject's name and full path explicitly.
 *
 *	The obj specifies the EDVVFSObject to be set.
 *
 *	The name specifies the new name. If name is NULL then the VFS
 *	object's name will be cleared
 *
 *	The path specifies the new full path. If path is NULL then
 *	the EDVVFSObject's path will be cleared
 */
void edv_vfs_object_set_path2(
	EDVVFSObject *obj,
	const gchar *name,
	const gchar *path
)
{
	if(obj == NULL)
	    return;

	g_free(obj->name);
	obj->name = STRDUP(name);

	g_free(obj->path);
	obj->path = STRDUP(path);
}

/*
 *	Sets the EDVVFSObject's values from another EDVVFSObject.
 *
 *	The obj specifies the EDVVFSObject to be set.
 *
 *	The obj_values specifies the values to set obj with.
 */
void edv_vfs_object_set_object(
	EDVVFSObject *obj,			/* Target */
	EDVVFSObject *obj_values		/* Source */
)
{
	EDVVFSObject		*tar_obj = obj,
				*src_obj = obj_values;

	if((tar_obj == NULL) || (src_obj == NULL))
	    return;

	tar_obj->type = src_obj->type;

	tar_obj->device_index = src_obj->device_index;
	tar_obj->index = src_obj->index;

	g_free(tar_obj->name);
	tar_obj->name = STRDUP(src_obj->name);
	g_free(tar_obj->path);
	tar_obj->path = STRDUP(src_obj->path);

	tar_obj->size = src_obj->size;

	g_free(tar_obj->link_target);
	tar_obj->link_target = STRDUP(src_obj->link_target);
	tar_obj->link_hint_flags = src_obj->link_hint_flags;

	tar_obj->permissions = src_obj->permissions;

	tar_obj->access_time = src_obj->access_time;
	tar_obj->modify_time = src_obj->modify_time;
	tar_obj->change_time = src_obj->change_time;

	tar_obj->owner_id = src_obj->owner_id;
	tar_obj->group_id = src_obj->group_id;

	tar_obj->device_type = src_obj->device_type;

	tar_obj->block_size = src_obj->block_size;
	tar_obj->blocks = src_obj->blocks;

	tar_obj->hard_links = src_obj->hard_links;

	tar_obj->meta_data_list = edv_properties_list_delete(tar_obj->meta_data_list);
	tar_obj->meta_data_list = edv_properties_list_copy(src_obj->meta_data_list);
}

/*
 *	Deletes and zeros all the values on the EDVVFSObject.
 *
 *	The obj specifies the EDVVFSObject who's values are to be cleared.
 */
void edv_vfs_object_clear(EDVVFSObject *obj)
{
	if(obj == NULL)
	    return;

	g_free(obj->name);
	g_free(obj->path);

	g_free(obj->link_target);

	obj->meta_data_list = edv_properties_list_delete(obj->meta_data_list);

	(void)memset(
	    obj,
	    0x00,
	    sizeof(EDVVFSObject)
	);
}

/*
 *	Deletes the EDVVFSObject.
 *
 *	The obj specifies the EDVVFSObject to delete.
 */
void edv_vfs_object_delete(EDVVFSObject *obj)
{
	if(obj == NULL)
	    return;

	g_free(obj->name);
	g_free(obj->path);

	g_free(obj->link_target);

	(void)edv_properties_list_delete(obj->meta_data_list);

	g_free(obj);
}
