/*
		    Endeavour Mark II - Utilities

	These functions do not depend on Endeavour 2 or GTK+.

	See also: edv_directory.h edv_link.h edv_path.h edv_process.h
	edv_utils_gtk.h
 */

#ifndef EDV_UTILS_H
#define EDV_UTILS_H

#include <glib.h>
#include "edv_types.h"


/*
 *	Concatonates two strings togeather.
 *
 *	The sa specifies the source string. The source string will
 *	be either deleted or reallocated by this function, and
 *	therefore, the calling function should not reference it
 *	after this call.
 *
 *	The sb specifies the string to append to sa.
 *
 *	Returns a dynamically allocated string describing sa
 *	concatonated to sb. If sa is NULL and sb is not NULL then
 *	a dynamically allocated string describing sb is returned.
 *	If sa and sb are both NULL then a dynamically allocated
 *	empty string is returned.
 */
extern gchar *edv_strcat(
	gchar *sa,
	const gchar *sb
);

/*
 *	Substitutes tokens in a string.
 *
 *	The s specifies the string containing the tokens.
 *
 *	The token specifies the string describing the token.
 *
 *	The value specifies the string to replace token with in s.
 *
 *	Returns a dynamically allocated string describing s with
 *	any substitutions made or NULL on error.
 *
 *	Example:
 *
 *	s		"my%SPACEstring%SPACE"
 *	token		"%SPACE"
 *	value		" "
 *
 *	Return:		"my string "
 */
extern gchar *edv_strsub(
	const gchar *s,
	const gchar *token,
	const gchar *value
);

/*
 *	Substitutes any occurance of "%HH" in the string with a
 *	single character described by HH in hexidecimal value.
 *
 *	The s specifies the string to substitute.
 *
 *	Returns s.
 *
 *	Example:
 *
 *	s		"my%20stri%6Eg"
 *
 *	Return:		"my string"
 */
extern gchar *edv_strsubh(gchar *s);


/*
 *	Gets the next argument in the command string.
 *
 *	The cmd specifies the command string.
 *
 *	If arg_rtn is not NULL then *arg_rtn will be set to a 
 *	dynamically allocated string describing the next argument 
 *	encountered in cmd or NULL if no argument is encountered. The 
 *	calling function must delete the returned string.
 *
 *	If parse_escapes is TRUE then any occurances of the escape
 *	character '\' will be parsed and any escape characters in
 *	the return value for the first argument will be removed.
 *
 *	If parse_quotes is TRUE then if the first argument starts
 *	with the quote character '"' then the return value for the
 *	first argument will contain all the characters up until the
 *	next quote character (excluding the quote characters
 *	themselves) or end of string.
 *
 *	Returns the pointer to the next argument in cmd or NULL if
 *	there are no more arguments in cmd.
 */
extern const gchar *edv_strarg(
	const gchar *cmd,
	gchar **arg_rtn,
	const gboolean parse_escapes,
	const gboolean parse_quotes
);

/*
 *	Gets a list of arguments from the command string.
 * 
 *	The cmd specifies the command string.
 *
 *	If parse_escapes is TRUE then any occurances of the escape
 *	character '\' will be parsed and any escape characters in
 *	the return value for the first argument will be removed.
 *
 *	If parse_quotes is TRUE then if the first argument starts
 *	with the quote character '"' then the return value for the
 *	first argument will contain all the characters up until the
 *	next quote character (excluding the quote characters
 *	themselves) or end of string.
 *
 *	Returns a gchar ** list of gchar * strings describing each
 *	argument where the last item in the gchar ** list is NULL.
 *	The calling function must delete the returned list and each
 *	string.
 */
extern gchar **edv_strexp_args(
	const gchar *cmd,
	const gboolean parse_escapes,
	const gboolean parse_quotes
);

/*
 *	Gets a GList of gchar * strings describing each argument from
 *	the command string.
 *
 *	The cmd specifies the command string.
 *
 *	If parse_escapes is TRUE then any occurances of the escape
 *	character '\' will be parsed and any escape characters in
 *	the return value for the first argument will be removed.
 *
 *	If parse_quotes is TRUE then if the first argument starts
 *	with the quote character '"' then the return value for the
 *	first argument will contain all the characters up until the
 *	next quote character (excluding the quote characters
 *	themselves) or end of string.
 *
 *	Returns a GList of gchar * strings describing each argument.
 *	The calling function must delete the returned list and each
 *	string.
 */
extern GList *edv_strexp_args_list(
	const gchar *cmd,
	const gboolean parse_escapes,
	const gboolean parse_quotes
);


/*
 *	Converts the conical window name to EDVWindowType type.
 *
 *	The win_name specifies the conical window name which can be
 *	one of:
 *
 *		"about_dialog"
 *		"vfs_browser"
 *		"image_browser"
 *		"archiver"
 *		"recycle_bin"
 *		"mime_types"
 *		"devices"
 *		"history"
 *		"options"
 *		"customize"
 *		"properties_dialog"
 *		"find"
 *		"object_operations_dialog"
 *		"run_dialog"
 *		"help"
 *
 *	Returns one of EDV_WINDOW_*.
 */
extern EDVWindowType edv_window_name_to_window_type(const gchar *win_name);

/*
 *	Converts EDVWindowType type to conical window name.
 *
 *	The win_type specifies the EDVWindowType.
 *
 *	Returns a statically allocated string describing the conical
 *	window name.
 */
extern const gchar *edv_window_type_to_window_name(const EDVWindowType win_type);


/*
 *	Renames or moves the object.
 *
 *	The old_path specifies the path to the object.
 *
 *	The new_path specifies the new name of the object or the
 *	location to move it to. If moving to a new location, then
 *	the new location may not be on a different physical device.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_rename(
	const gchar *old_path,
	const gchar *new_path
);


/*
 *	Converts the stat() mode_t type to EDVObjectType type.
 *
 *	The m specifies the stat() mode_t type value. Only the
 *	bit pertaining to the type in m is checked, all other bits
 *	are ignored.
 *
 *	Returns the EDVObjectType type value or
 *	EDV_OBJECT_TYPE_UNKNOWN on failed match.
 */
extern EDVObjectType edv_stat_mode_to_object_type(const guint m);

/*
 *	Converts the EDVObjectType type to stat() mode_t type.
 *
 *	The type specifies the EDVObjectType which must be one of
 *	EDV_OBJECT_TYPE_*.
 *
 *	Returns the mode_t type value or 0 on failed match.
 */
extern guint edv_object_type_to_stat_mode(const EDVObjectType type);

/*
 *	Converts the EDVObjectType type to a string name.
 *
 *	The type specifies the EDVObjectType which must be one of
 *	EDV_OBJECT_TYPE_*.
 *
 *	Returns a statically allocated string describing the type
 *	name.
 */
extern const gchar *edv_object_type_to_object_name(const EDVObjectType type);

/*
 *	Converts the EDVObjectType type to a string name in
 *	lowercase.
 *
 *	The type specifies the EDVObjectType which must be one of
 *	EDV_OBJECT_TYPE_*.
 *
 *	Returns a statically allocated string describing the type
 *	name in lowercase.
 */
extern const gchar *edv_object_type_to_object_name_lower(const EDVObjectType type);


/*
 *	Creates a string describing the value with ',' deliminators.
 *
 *	The i specifies the value.
 *
 *	Returns a statically allocated string describing the
 *	value with ',' deliminators or NULL on error.
 */
extern const gchar *edv_str_size_delim(const gulong i);
extern const gchar *edv_str_size_delim_char(
	const gulong i,
	const gchar delim_char
);

/*
 *	Formats a string describing the size with the specified
 *	format.
 *
 *	The size specifies the size in bytes.
 *
 *	The size_format specifies the format.
 *
 *	The block_size specifies the size of each block in bytes.
 *
 *	The delim_char specifies the character to use as the
 *	deliminator.
 *
 *	If allow_unit_conversion is TRUE then the returned string
 *	will describe the size in units other than bytes if size_format
 *	specifies a size format other than bytes. Otherwise FALSE
 *	forces the returned string to describe the size in bytes.
 *
 *	Returns a statically allocated string.
 */
extern const gchar *edv_str_size_format(
	const gulong size,
	const EDVSizeFormat size_format,
	const gulong block_size,
	const gchar delim_char,
	const gboolean allow_unit_conversion
);


/*
 *	Returns the current umask.
 */
extern guint edv_get_umask(void);

/*
 *	Sets the umask.
 */
extern void edv_set_umask(const guint m);


/*
 *	Converts the stat() mode_t permissions to EDVPermissionFlags
 *	permissions.
 *
 *	The m specifies the stat() mode_t permissions.
 *
 *	Returns the EDVPermissionFlags permissions.
 */
extern EDVPermissionFlags edv_stat_mode_to_edv_permissions(const guint m);

/*
 *	Converts the EDVPermissionFlags permissions to stat() mode_t
 *	permissions.
 *
 *	The permissions specifies the EDVPermissionFlags, which can
 *	be any of EDV_PERMISSION_*.
 *
 *	Returns the stat() mode_t permissions.
 */
extern guint edv_edv_permissions_to_stat_mode(const EDVPermissionFlags permissions);

/*
 *	Converts the EDVPermissionFlags permissions to a string
 *	description.
 *
 *	The permissions specifies the EDVPermissionFlags, which can
 *	be any of EDV_PERMISSION_*.
 *
 *	Returns a dynamically allocated string describing the
 *	permissions.
 */
extern gchar *edv_str_permissions(const EDVPermissionFlags permissions);

/*
 *	Gets the current umask as EDVPermissionFlags.
 */
extern EDVPermissionFlags edv_permissions_get_default(void);

/*
 *	Sets the umask with EDVPermissionFlags.
 */
extern void edv_permissions_set_default(const EDVPermissionFlags permissions);

/*
 *	Sets the permissions of the object.
 *
 *	The path specifies the path to the object.
 *
 *	The m specifies the stat() mode_t permissions.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_chmod(
	const gchar *path,
	const guint m
);

/*
 *	Sets the permissions of the object.
 *
 *	The fd specifies the object's descriptor.
 *
 *	The m specifies the stat() mode_t permissions.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_fchmod(
	const gint fd,
	const guint m
);

/*
 *	Sets the permissions of the object.
 *
 *	The path specifies the path to the object.
 *
 *	The permissions specifies the EDVPermissionFlags permissions.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_permissions_set(
	const gchar *path,
	const EDVPermissionFlags permissions
);

/*
 *	Sets the permissions of the object.
 *
 *	The fd specifies the object's descriptor.
 *
 *	The permissions specifies the EDVPermissionFlags permissions.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_permissions_set_fd(
	const gint fd,
	const EDVPermissionFlags permissions
);


/*
 *	Sets the ownership and group of the object.
 *
 *	The path specifies the path to the object.
 *
 *	The uid specifies the new owner.
 *
 *	The gid specifies the new group.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_chown(
	const gchar *path,
	const gint uid,
	const gint gid
);
extern gint edv_lchown(
	const gchar *path,
	const gint uid,
	const gint gid
);

/*
 *	Sets the ownership and group of the object.
 *
 *	The fd specifies the object's descriptor.
 *
 *	The uid specifies the new owner.
 *
 *	The gid specifies the new group.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_fchown(
	const gint fd,
	const gint uid,
	const gint gid
);


/*
 *	Set the access and modify times of the object.
 *
 *	The path specifies the path to the object.
 *
 *	The access_time specifies the access time in seconds since
 *	EPOCH. If access_time is (gulong)-1 then the access time will
 *	not be modified.
 *
 *	The modify_time specifies the modify time in seconds since
 *	EPOCH. If modify_time is (gulong)-1 then the access time will
 *	not be modified.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_utime(
	const gchar *path,
	const gulong access_time,
	const gulong modify_time
);


/*
 *	Gets the device major and minor numbers from the device value.
 *
 *	The rdev specifies the device value which should come from
 *	*stat()'s struct stat (dev_t)st_rdev value.
 *
 *	The major_rtn specifies the device major number return value.
 *
 *	The minor_rtn specifies the device minor number return value.
 */
extern void edv_device_numbers_parse(
	const gint rdev,
	gint *major_rtn, gint *minor_rtn
);

/*
 *	Combines the device major and minor numbers into a single
 *	device value.
 *
 *	The major specifies the device major number.
 *
 *	The minor specifies the device minor number.
 *
 *	Returns the device value which confirms to *stat()'s struct
 *	stat (dev_t)st_rdev value.
 */
extern gint edv_device_numbers_format(const gint major, const gint minor);


/*
 *	Gets the current working directory.
 *
 *	Returns a dynamically allocated string describing the current
 *	working directory.
 */
extern gchar *edv_getcwd(void);

/*
 *	Sets the current working directory.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_setcwd(const gchar *path);


/*
 *	Updates the access time of the object or creates a new file
 *	of 0 size.
 *
 *	The path specifies the object.
 *
 *	The t specifies the access time to set in seconds since
 *	EPOCH. If t is (gulong)-1 then the current time will be used.
 *
 *	If create_as_needed is TRUE and path refers to a non-existant
 *	object then a new file of 0 size will be created.
 *
 *	Returns 0 on success or non-zero on error. If the object does
 *	not exist and create_as_needed is FALSE then -7 is returned
 *	and errno is set to ENOENT.
 */
extern gint edv_touch(
	const gchar *path,
	const gulong t,
	const gboolean create_as_needed
);


/*
 *	Gets the system's temporary files directory.
 *
 *	If the system's temporary files directory environment
 *	variable was set then its value will be returned, otherwise
 *	a compile-time value of the system's temporary files
 *	directory will be returned.
 *
 *	Returns a dynamically allocated string describing the path
 *	to the temporary files directory.
 */
extern gchar *edv_tmp_directory(void);

/*
 *	Creates a new temporary file.
 *
 *	If tmp_dir_path is not NULL or not an empty string then it
 *	specifies the path to the directory to create the new
 *	temporary file in, otherwise the temporary files directory
 *	specified by the environment will be used.
 *
 *	Returns a dynamically allocated string describing the path
 *	to the new temporary file.
 */
extern gchar *edv_tmp_name(const gchar *tmp_dir_path);


/*
 *	Unlinks (removes locally) the object.
 *
 *	The path specifies the object to remove. The path must not be
 *	a directory.
 * 
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_unlink(const gchar *path);


/*
 *	Finds the full path to the program by looking at the PATH
 *	environment variable.
 *
 *	The name specifies the program object's name.
 *
 *	Returns a dynamically allocated string describing the completed
 *	full path to the program or NULL if there was no match. If
 *	name was already a full path then a copy of name is returned.
 */
extern gchar *edv_which(const gchar *name);

/*
 *	Commit buffer caches to disk.
 *
 *	Always returns 0.
 */
extern gint edv_sync(void);


/*
 *	Gets the current time in seconds since EPOCH.
 */
extern gulong edv_time(void);

/*
 *	Gets the time of day in milliseconds.
 */
extern gulong edv_time_ms(void);

/*
 *	Microsleep.
 *
 *	The us specifies the time to sleep in microseconds.
 */
extern void edv_usleep(const gulong us);


/*
 *	Gets the idle interval in milliseconds from the priority.
 */
extern gulong edv_get_interval_from_load_images_priority(
	const EDVListsLoadImagesPriority priority
);


/*
 *	Checks the descriptor status.
 *
 *	The fd specifies the descriptor.
 *
 *	The mode specifies a string describing the mode which can
 *	contain any of the following characters to describe the
 *	checking of the descriptor status:
 *
 *	r		Data is waiting to be read.
 *	w		Writing to the descriptor will not block.
 *
 *	The timeout_ms specifies the timeout in milliseconds. If
 *	timeout_ms is 0 then this call will not block and return
 *	immediately.
 *
 *	Returns TRUE if any of the specified mode condition were met
 *	or FALSE otherwise or on error.
 */
extern gboolean edv_poll(
	const gint fd,
	const gchar *mode,
	const gulong timeout_ms
);

/*
 *	Checks the descriptor's read status.
 *
 *	The fd specifies the descriptor.
 *
 *	Same as calling edv_poll(fd, "r", 0l).
 *
 *	Returns TRUE if data is waiting to be read or FALSE otherwise
 *	or on error.
 */
extern gboolean edv_poll_read(const gint fd);

/*
 *	Checks the descriptor's write status.
 *
 *	The fd specifies the descriptor.
 *
 *	Same as calling edv_poll(fd, "w", 0l).
 *
 *	Returns TRUE if writing to the descriptor will not block or
 *	FALSE otherwise or on error.
 */
extern gboolean edv_poll_write(const gint fd);


/*
 *	Opens a GList of gchar * strings from a file.
 *
 *	The path specifies the full path to the file to open from.
 *
 *	If max_lines is positive then no more than max_lines will be
 *	read from the file.
 *
 *	If strip_crs is TRUE then carrage return characters ('\r')
 *	will not be stored into the return strings.
 *
 *	Returns a GList of gchar * strings describing each line. The
 *	calling function must delete the returned list and each string.
 */
extern GList *edv_open_text_file_glist(
        const gchar *path,
        const gint max_lines,
        const gboolean strip_crs
);

/*
 *	Saves the GList of gchar * strings to a file.
 * 
 *	The path specifies the full path to the file to save to.
 *
 *	The lines_list specifies a GList of gchar * strings to
 *	write to the file.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_save_text_file_glist(
        const gchar *path,
        GList *lines_list
);


/*
 *	Gets the host name.
 *
 *	Returns a dynamically allocated string describing the host name
 *	or NULL on error.
 */
extern gchar *edv_get_host_name(void);

/*
 *	Sets the host name.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_set_host_name(const gchar *name);


#endif	/* EDV_UTILS_H */
