#include <stdio.h>
#include <stdlib.h>				/* For mkstemp() */
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <time.h>
#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>				/* For gettimeofday() */
#include <fcntl.h>				/* For open() */
#include <glib.h>
#include <unistd.h>

#include "../../include/string.h"

#include "edv_types.h"
#include "edv_utils.h"
#include "edv_path.h"
#include "edv_stream.h"
#include "edv_vfs_obj.h"
#include "edv_vfs_obj_stat.h"

#include "config.h"


/* Strings */
gchar *edv_strcat(
	gchar *sa,
	const gchar *sb
);

/* String Substitution */
gchar *edv_strsub(
	const gchar *s,
	const gchar *token,
	const gchar *value
);
gchar *edv_strsubh(gchar *s);

/* Command String Parsing */
const gchar *edv_strarg(
	const gchar *cmd,
	gchar **arg_rtn,
	const gboolean parse_escapes,
	const gboolean parse_quotes
);
gchar **edv_strexp_args(
	const gchar *cmd,
	const gboolean parse_escapes,
	const gboolean parse_quotes
);
GList *edv_strexp_args_list(
	const gchar *cmd,
	const gboolean parse_escapes,
	const gboolean parse_quotes
);

/* Window Type */
EDVWindowType edv_window_name_to_window_type(const gchar *win_name);
const gchar *edv_window_type_to_window_name(const EDVWindowType win_type);

/* Rename */
gint edv_rename(
	const gchar *old_path,
	const gchar *new_path
);

/* Type */
EDVObjectType edv_stat_mode_to_object_type(const guint m);
guint edv_object_type_to_stat_mode(const EDVObjectType type);

/* Type Name */
const gchar *edv_object_type_to_object_name(const EDVObjectType type);
const gchar *edv_object_type_to_object_name_lower(const EDVObjectType type);

/* Sizes */
const gchar *edv_str_size_delim(const gulong i);
const gchar *edv_str_size_delim_char(
	const gulong i,
	const gchar delim_char
);
const gchar *edv_str_size_format(
	const gulong size,
	const EDVSizeFormat size_format,
	const gulong block_size,
	const gchar delim_char,
	const gboolean allow_unit_conversion
);

/* UMask */
guint edv_get_umask(void);
void edv_set_umask(const guint m);

/* Permissions */
EDVPermissionFlags edv_stat_mode_to_edv_permissions(const guint m);
guint edv_edv_permissions_to_stat_mode(const EDVPermissionFlags permissions);
gchar *edv_str_permissions(const EDVPermissionFlags permissions);
EDVPermissionFlags edv_permissions_get_default(void);
void edv_permissions_set_default(const EDVPermissionFlags permissions);
gint edv_chmod(
	const gchar *path,
	const guint m
);
gint edv_fchmod(
	const gint fd,
	const guint m
);
gint edv_permissions_set(
	const gchar *path,
	const EDVPermissionFlags permissions
);
gint edv_permissions_set_fd(
	const gint fd,
	const EDVPermissionFlags permissions
);

/* Ownership */
gint edv_chown(
	const gchar *path,
	const gint uid,
	const gint gid
);
gint edv_lchown(
	const gchar *path,
	const gint uid,
	const gint gid
);
gint edv_fchown(
	const gint fd,
	const gint uid,
	const gint gid
);

/* Time */
gint edv_utime(
	const gchar *path,
	const gulong access_time,
	const gulong modify_time
);

/* Device Numbers */
void edv_device_numbers_parse(
	const gint rdev,
	gint *major_rtn, gint *minor_rtn
);
gint edv_device_numbers_format(const gint major, const gint minor);

/* Current Working Directory */
gchar *edv_getcwd(void);
gint edv_setcwd(const gchar *path);

/* Touch */
gint edv_touch(
	const gchar *path,
	const gulong t,
	const gboolean create_as_needed
);

/* Temporary Files */
gchar *edv_tmp_directory(void);
gchar *edv_tmp_name(const gchar *tmp_dir_path);

/* Unlink */
gint edv_unlink(const gchar *path);

/* Sync */
gint edv_sync(void);

/* Time */
gulong edv_time(void);
gulong edv_time_ms(void);

/* Microsleep */
void edv_usleep(const gulong us);

/* Idle Interval From Priority */
gulong edv_get_interval_from_load_images_priority(
	const EDVListsLoadImagesPriority priority
);

/* Polling */
gboolean edv_poll(
	const gint fd,
	const gchar *mode,
	const gulong timeout_ms
);
gboolean edv_poll_read(const gint fd);
gboolean edv_poll_write(const gint fd);

/* GList File IO */
GList *edv_open_text_file_glist(
	const gchar *path,
	const gint max_lines,
	const gboolean strip_crs
);
gint edv_save_text_file_glist(
	const gchar *path,
	GList *lines_list
);

/* Host Name */
gchar *edv_get_host_name(void);
gint edv_set_host_name(const gchar *name);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Concatonates two strings togeather.
 *
 *	The sa specifies the source string. The source string will
 *	be either deleted or reallocated by this function, and
 *	therefore, the calling function should not reference it
 *	after this call.
 *
 *	The sb specifies the string to append to sa.
 *
 *	Returns a dynamically allocated string describing sa
 *	concatonated to sb. If sa is NULL and sb is not NULL then
 *	a dynamically allocated string describing sb is returned.
 *	If sa and sb are both NULL then a dynamically allocated
 *	empty string is returned.
 */
gchar *edv_strcat(
	gchar *sa,
	const gchar *sb
)
{
	if(sa != NULL) {
		if(sb != NULL) {
			gchar *sc = g_strconcat(
				sa,
				sb,
				NULL
			);
			if(sc != NULL)
			{
				g_free(sa);
				sa = sc;
			}
		}
	} else {
		if(sb != NULL)
			sa = g_strdup(sb);
		else
			sa = g_strdup("");
	}
	return(sa);
}


/*
 *	Substitutes tokens in a string.
 *
 *	The s specifies the string containing the tokens.
 *
 *	The token specifies the string describing the token.
 *
 *	The value specifies the string to replace token with in s.
 *
 *	Returns a dynamically allocated string describing s with
 *	any substitutions made or NULL on error.
 *
 *	Example:
 *
 *	s		"my%SPACEstring%SPACE"
 *	token		"%SPACE"
 *	value		" "
 *
 *	Return:		"my string "
 */
gchar *edv_strsub(
	const gchar *s,
	const gchar *token,
	const gchar *value
)
{
	return((gchar *)strsub(
		(const char *)s,
		(const char *)token,
		(const char *)value
	));
}

/*
 *	Substitutes any occurance of "%HH" in the string with a
 *	single character described by HH in hexidecimal value.
 *
 *	The s specifies the string to substitute.
 *
 *	Returns s.
 *
 *	Example:
 *
 *	s		"my%20stri%6Eg"
 *
 *	Return:		"my string"
 */
gchar *edv_strsubh(gchar *s)
{
	gchar *s_orig = s;

	if(s_orig == NULL)
		return(s_orig);

	/* Iterate through the string and evaluate all tokens */
	while(*s != '\0')
	{
		if(*s == '%')
		{
			guint i;
			gchar	*s1,
				*s2,
				hex_str[3];

			/* Get the two characters after the token character
			 * and store them in hex_str
			 *
			 * Break if the end of the string
			 * is encountered
			 */
			s++;
			if(*s == '\0')
				break;

			hex_str[0] = *s;

			s++;
			if(*s == '\0')
				break;

			hex_str[1] = *s;
			hex_str[2] = '\0';

			/* Convert the hex value described in hex_str into
			 * its intended character value
			 */
			if(sscanf((const char *)hex_str, "%x", &i) < 1)
				i = 0;

			/* Seek back to the token start position */
			s -= 2;

			/* Replace the token with the intended character */
			*s = (gchar)i;

			/* Delete the two characters after the token start
			 * position
			 */
			for(s1 = s + 1,
			    s2 = s + 3;
			    *s2 != '\0';
			    s1++,
			    s2++
			)
				*s1 = *s2;

			*s1 = '\0';
		}
		s++;
	}

	return(s_orig);
}


/*
 *	Gets the next argument in the command string.
 *
 *	The cmd specifies the command string.
 * 
 *	If arg_rtn is not NULL then *arg_rtn will be set to a
 *	dynamically allocated string describing the next argument
 *	encountered in cmd or NULL if no argument is encountered. The
 *	calling function must delete the returned string.
 *
 *	If parse_escapes is TRUE then any occurances of the escape
 *	character '\' will be parsed and any escape characters in
 *	the return value for the first argument will be removed.
 *
 *	If parse_quotes is TRUE then if the first argument starts
 *	with the quote character '"' then the return value for the
 *	first argument will contain all the characters up until the
 *	next quote character (excluding the quote characters
 *	themselves) or end of string.
 *
 *	Returns the pointer to the next argument in cmd or NULL if
 *	there are no more arguments in cmd.
 */
const gchar *edv_strarg(
	const gchar *cmd,
	gchar **arg_rtn,
	const gboolean parse_escapes,
	const gboolean parse_quotes
)
{
	const gchar	escape_char = '\\',
					quote_char = '"';
	const gchar *arg, *arg_next;

	if(arg_rtn != NULL)
		*arg_rtn = NULL;

	if(cmd == NULL)
		return(NULL);

	arg = cmd;
	while(ISBLANK(*arg))
		arg++;

	/* Argument in quotes? */
	if((*arg == quote_char) && parse_quotes)
	{
		gint len;
		const gchar *arg_end;

		arg++;                      /* Seek past the quote */

		/* Seek to the end of the argument as the next quote character */
		arg_end = arg;
		while(*arg_end != '\0')
		{
			/* Escape this character? */
			if(*arg_end == escape_char)
			{
				arg_end++;
				if(*arg_end != '\0')
					arg_end++;
				continue;
			}

			/* End quote pair reached? */
			if(*arg_end == quote_char)
				break;

			arg_end++;
		}
		len = arg_end - arg;
		if(len > 0)
		{
			if(arg_rtn != NULL)
			{
				gchar *s = (gchar *)g_malloc(
					(len + 1) * sizeof(gchar)
				);
				if(s != NULL)
				{
					(void)memcpy(
						s,
						arg,
						len * sizeof(gchar)
					);
					s[len] = '\0';

					/* Remove escape characters? */
					if(parse_escapes)
					{
						gint i;
						for(i = 0; s[i] != '\0'; i++)
						{
							if(s[i] == escape_char)
							{
								s = strdelchr(s, i);
								if(s[i] == '\0')
									break;
							}
						}
					}
				}
				*arg_rtn = s;
			}
		}
		arg_next = arg_end;
		if(*arg_next == quote_char)
			arg_next++;

		while(ISBLANK(*arg_next))
			arg_next++;
	}
	else
	{
		gint len;
		const gchar *arg_end = arg;

		/* Seek to the end of the argument as the next blank character */
		while(*arg_end != '\0')
		{
			/* Escape this character? */
			if(*arg_end == escape_char)
			{
				arg_end++;
				if(*arg_end != '\0')
					arg_end++;
				continue;
			}

			/* End of this argument reached? */
			if(ISBLANK(*arg_end))
				break;

			arg_end++;
		}
		len = arg_end - arg;
		if(len > 0)
		{
			if(arg_rtn != NULL)
			{
				gchar *s = (gchar *)g_malloc(
					(len + 1) * sizeof(gchar)
				);
				if(s != NULL)
				{
					(void)memcpy(
						s,
						arg,
						len * sizeof(gchar)
					);
					s[len] = '\0';

					/* Remove escape characters? */
					if(parse_escapes)
					{
						gint i;
						for(i = 0; s[i] != '\0'; i++)
						{
							if(s[i] == escape_char)
							{
								s = strdelchr(s, i);
								if(s[i] == '\0')
									break;
							}
						}
					}
				}
				*arg_rtn = s;
			}
		}
		arg_next = arg_end;
		while(ISBLANK(*arg_next))
			arg_next++;
	}

	/* Is there a next argument? */
	if(*arg_next != '\0')
		return(arg_next);
	else
		return(NULL);
}

/*
 *	Gets a list of arguments from the command string.
 *
 *	The cmd specifies the command string.
 *
 *	If parse_escapes is TRUE then any occurances of the escape
 *	character '\' will be parsed and any escape characters in
 *	the return value for the first argument will be removed.
 *
 *	If parse_quotes is TRUE then if the first argument starts
 *	with the quote character '"' then the return value for the
 *	first argument will contain all the characters up until the
 *	next quote character (excluding the quote characters
 *	themselves) or end of string.
 *
 *	Returns a gchar ** list of gchar * strings describing each
 *	argument where the last item in the gchar ** list is NULL.
 *	The calling function must delete the returned list and each
 *	string.
 */
gchar **edv_strexp_args(
	const gchar *cmd,
	const gboolean parse_escapes,
	const gboolean parse_quotes
)
{
	gint nargs = 0;
	gchar		*arg,
					**args_list = NULL;
	const gchar *arg_next = cmd;
	while(arg_next != NULL)
	{
		arg_next = edv_strarg(
			arg_next,
			&arg,
			parse_escapes,
			parse_quotes
		);
		if(arg != NULL)
		{
			const gint i = nargs;
			nargs++;
			args_list = (gchar **)g_realloc(
				args_list,
				nargs * sizeof(gchar *)
			);
			if(args_list == NULL)
			{
				g_free(arg);
				return(NULL);
			}

			args_list[i] = arg;
		}
	}

	/* Add the last argument in the pointer array as a NULL pointer */
	if(args_list != NULL)
	{
		const gint i = nargs;
		nargs++;
		args_list = (gchar **)g_realloc(
			args_list,
			nargs * sizeof(gchar *)
		);
		if(args_list == NULL)
			return(NULL);

		args_list[i] = NULL;
	}

	return(args_list);
}

/*
 *	Gets a GList of gchar * strings describing each argument from
 *	the command string.
 *
 *	The cmd specifies the command string.
 *
 *	If parse_escapes is TRUE then any occurances of the escape
 *	character '\' will be parsed and any escape characters in
 *	the return value for the first argument will be removed.
 *
 *	If parse_quotes is TRUE then if the first argument starts
 *	with the quote character '"' then the return value for the
 *	first argument will contain all the characters up until the
 *	next quote character (excluding the quote characters
 *	themselves) or end of string.
 *
 *	Returns a GList of gchar * strings describing each argument.
 *	The calling function must delete the returned list and each
 *	string.
 */
GList *edv_strexp_args_list(
	const gchar *cmd,
	const gboolean parse_escapes,
	const gboolean parse_quotes
)
{
	gchar *arg;
	const gchar *arg_next = cmd;
	GList *args_list = NULL;
	while(arg_next != NULL)
	{
		arg_next = edv_strarg(
			arg_next,
			&arg,
			parse_escapes,
			parse_quotes
		);
		if(arg != NULL)
			args_list = g_list_append(
				args_list,
				arg
			);
	}

	return(args_list);
}


/*
 *	Converts the conical window name to EDVWindowType type.
 *
 *	The win_name specifies the conical window name which can be
 *	one of:
 *
 *		"about_dialog"
 *		"vfs_browser"
 *		"image_browser"
 *		"archiver"
 *		"recycle_bin"
 *		"mime_types"
 *		"devices"
 *		"history"
 *		"options"
 *		"customize"
 *		"properties_dialog"
 *		"find"
 *		"object_operations_dialog"
 *		"run_dialog"
 *		"help"
 *
 *	Returns one of EDV_WINDOW_*.
 */
EDVWindowType edv_window_name_to_window_type(const gchar *win_name)
{
	if(win_name == NULL)
		return(EDV_WINDOW_NONE);

	/* About Dialog */
	if(!g_strcasecmp(win_name, "about_dialog") ||
	   !g_strcasecmp(win_name, "about_dlg") ||
	   !g_strcasecmp(win_name, "aboutdlg") ||
	   !g_strcasecmp(win_name, "about")
	)
	{
		return(EDV_WINDOW_ABOUT_DIALOG);
	}
	/* VFS Browser */
	else if(!g_strcasecmp(win_name, "vfs_browser") ||
		!g_strcasecmp(win_name, "file_browser") ||
		!g_strcasecmp(win_name, "filebrowser") ||
		!g_strcasecmp(win_name, "browser")
	)
	{
		return(EDV_WINDOW_VFS_BROWSER);
	}
	/* Image Browser */
	else if(!g_strcasecmp(win_name, "image_browser") ||
		!g_strcasecmp(win_name, "imagebrowser") ||
		!g_strcasecmp(win_name, "imbr")
	)
	{
		return(EDV_WINDOW_IMAGE_BROWSER);
	}
	/* Archiver */
	else if(!g_strcasecmp(win_name, "archiver"))
	{
		return(EDV_WINDOW_ARCHIVER);
	}
	/* Recycle Bin */
	else if(!g_strcasecmp(win_name, "recycle_bin") ||
		!g_strcasecmp(win_name, "recyclebin") ||
		!g_strcasecmp(win_name, "rec_bin") ||
		!g_strcasecmp(win_name, "recbin")
	)
	{
		return(EDV_WINDOW_RECYCLE_BIN);
	}
	/* MIME Types Window */
	else if(!g_strcasecmp(win_name, "mime_types_list") ||
		!g_strcasecmp(win_name, "mime_types") ||
		!g_strcasecmp(win_name, "mime_types_window") ||
		!g_strcasecmp(win_name, "mimetypes_window") ||
		!g_strcasecmp(win_name, "mimetypes")
	)
	{
		return(EDV_WINDOW_MIME_TYPES_LIST);
	}
	/* Devices Window */
	else if(!g_strcasecmp(win_name, "devices_list") ||
		!g_strcasecmp(win_name, "devices") ||
		!g_strcasecmp(win_name, "devices_window")
	)
	{
		return(EDV_WINDOW_DEVICES_LIST);
	}
	/* History Window */
	else if(!g_strcasecmp(win_name, "history_list") ||
		!g_strcasecmp(win_name, "history") ||
		!g_strcasecmp(win_name, "history_window")
	)
	{
		return(EDV_WINDOW_HISTORY_LIST);
	}
	/* Options Window */
	else if(!g_strcasecmp(win_name, "options") ||
		!g_strcasecmp(win_name, "options_window")
	)
	{
		return(EDV_WINDOW_OPTIONS);
	}
	/* Customize Window */
	else if(!g_strcasecmp(win_name, "customize") ||
		!g_strcasecmp(win_name, "customize_window")
	)
	{
		return(EDV_WINDOW_CUSTOMIZE);
	}
	/* Properties Dialog */
	else if(!g_strcasecmp(win_name, "properties_dialog") ||
		!g_strcasecmp(win_name, "properties") ||
		!g_strcasecmp(win_name, "prop") ||
		!g_strcasecmp(win_name, "prop_win") ||
		!g_strcasecmp(win_name, "prop_dlg") ||
		!g_strcasecmp(win_name, "propdlg")
	)
	{
		return(EDV_WINDOW_PROPERTIES_DIALOG);
	}
	/* Find Window */
	else if(!g_strcasecmp(win_name, "find") ||
		!g_strcasecmp(win_name, "find_window") ||
		!g_strcasecmp(win_name, "find_win") ||
		!g_strcasecmp(win_name, "find_dialog") ||
		!g_strcasecmp(win_name, "find_dlg")
	)
	{
		return(EDV_WINDOW_FIND);
	}
	/* Object Operations Dialog */
	else if(!g_strcasecmp(win_name, "object_operations_dialog") ||
		!g_strcasecmp(win_name, "object_operations") ||
		!g_strcasecmp(win_name, "object_op_dlg") ||
		!g_strcasecmp(win_name, "object_op") ||
		!g_strcasecmp(win_name, "obj_op_dlg") ||
		!g_strcasecmp(win_name, "obj_op")
	)
	{
		return(EDV_WINDOW_OBJECT_OPERATIONS_DIALOG);
	}
	/* Run Dialog */
	else if(!g_strcasecmp(win_name, "run_dialog") ||
		!g_strcasecmp(win_name, "run") ||
		!g_strcasecmp(win_name, "run_dlg") ||
		!g_strcasecmp(win_name, "rundlg")
	)
	{
		return(EDV_WINDOW_RUN_DIALOG);
	}
	/* Help */
	else if(!g_strcasecmp(win_name, "help"))
	{
		return(EDV_WINDOW_HELP);
	}

	return(EDV_WINDOW_NONE);
}

/*
 *	Converts EDVWindowType type to conical window name.
 *
 *	The win_type specifies the EDVWindowType.
 *
 *	Returns a statically allocated string describing the conical
 *	window name.
 */
const gchar *edv_window_type_to_window_name(const EDVWindowType win_type)
{
	switch(win_type)
	{
	  case EDV_WINDOW_NONE:
		return("none");
		break;
	  case EDV_WINDOW_ABOUT_DIALOG:
		return("about_dialog");
		break;
	  case EDV_WINDOW_VFS_BROWSER:
		return("vfs_browser");
		break;
	  case EDV_WINDOW_IMAGE_BROWSER:
		return("image_browser");
		break;
	  case EDV_WINDOW_ARCHIVER:
		return("archiver");
		break;
	  case EDV_WINDOW_RECYCLE_BIN:
		return("recycle_bin");
		break;
	  case EDV_WINDOW_MIME_TYPES_LIST:
		return("mime_types_list");
		break;
	  case EDV_WINDOW_DEVICES_LIST:
		return("devices_list");
		break;
	  case EDV_WINDOW_HISTORY_LIST:
		return("history_list");
		break;
	  case EDV_WINDOW_OPTIONS:
		return("options");
		break;
	  case EDV_WINDOW_CUSTOMIZE:
		return("customize");
		break;
	  case EDV_WINDOW_PROPERTIES_DIALOG:
		return("properties_dialog");
		break;
	  case EDV_WINDOW_FIND:
		return("find");
		break;
	  case EDV_WINDOW_OBJECT_OPERATIONS_DIALOG:
		return("object_operations_dialog");
		break;
	  case EDV_WINDOW_RUN_DIALOG:
		return("run_dialog");
		break;
	  case EDV_WINDOW_HELP:
		return("help");
		break;
	}

	return("none");
}


/*
 *	Renames or moves the object.
 *
 *	The old_path specifies the path to the object.
 *
 *	The new_path specifies the new name of the object or the
 *	location to move it to. If moving to a new location, then
 *	the new location may not be on a different physical device.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_rename(
	const gchar *old_path,
	const gchar *new_path
)
{
	if(STRISEMPTY(old_path) || STRISEMPTY(new_path))
	{
		errno = EINVAL;
		return(-2);
	}

	return((gint)rename(
		(const char *)old_path,
		(const char *)new_path
	));
}


/*
 *	Converts the stat() mode_t type to EDVObjectType type.
 *
 *	The m specifies the stat() mode_t type value. Only the
 *	bit pertaining to the type in m is checked, all other bits
 *	are ignored.
 *
 *	Returns the EDVObjectType type value or
 *	EDV_OBJECT_TYPE_UNKNOWN on failed match.
 */
EDVObjectType edv_stat_mode_to_object_type(const guint m)
{
	const mode_t mm = (mode_t)m;
#ifdef S_ISREG
	if(S_ISREG(mm))
#else
# warning "S_ISREG was not #defined, there will be no way to check for a regular file"
	if(TRUE)
#endif
		return(EDV_OBJECT_TYPE_FILE);
#ifdef S_ISDIR
	else if(S_ISDIR(mm))
		return(EDV_OBJECT_TYPE_DIRECTORY);
#endif
#ifdef S_ISLNK
	else if(S_ISLNK(mm))
		return(EDV_OBJECT_TYPE_LINK);
#endif
#ifdef S_ISCHR
	else if(S_ISCHR(mm))
		return(EDV_OBJECT_TYPE_DEVICE_CHARACTER);
#endif
#ifdef S_ISBLK
	else if(S_ISBLK(mm))
		return(EDV_OBJECT_TYPE_DEVICE_BLOCK);
#endif
#ifdef S_ISFIFO
	else if(S_ISFIFO(mm))
		return(EDV_OBJECT_TYPE_FIFO);
#endif
#ifdef S_ISSOCK
	else if(S_ISSOCK(mm))
		return(EDV_OBJECT_TYPE_SOCKET);
#endif
	else
		return(EDV_OBJECT_TYPE_FILE);
}

/*
 *	Converts the EDVObjectType type to stat() mode_t type.
 *
 *	The type specifies the EDVObjectType which must be one of
 *	EDV_OBJECT_TYPE_*.
 *
 *	Returns the mode_t type value or 0 on failed match.
 */
guint edv_object_type_to_stat_mode(const EDVObjectType type)
{
	switch(type)
	{
	  case EDV_OBJECT_TYPE_UNKNOWN:
		break;
	  case EDV_OBJECT_TYPE_FILE:
#ifdef S_IFREG
		return(S_IFREG);
#else
# warning "S_IFREG was not #defined, there will be no way to describe a regular file"
#endif
		break;
	  case EDV_OBJECT_TYPE_DIRECTORY:
#ifdef S_IFDIR
		return(S_IFDIR);
#endif
		break;
	  case EDV_OBJECT_TYPE_LINK:
#ifdef S_IFLNK
		return(S_IFLNK);
#endif
		break;
	  case EDV_OBJECT_TYPE_FIFO:
#if defined(S_IFFIFO)
		return(S_IFFIFO);
#elif defined(S_IFIFO)
		return(S_IFIFO);
#endif
		break;
	  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
#ifdef S_IFBLK
		return(S_IFBLK);
#endif
		break;
	  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
#ifdef S_IFCHR
		return(S_IFCHR);
#endif
		break;
	  case EDV_OBJECT_TYPE_SOCKET:
#ifdef S_IFSOCK
		return(S_IFSOCK);
#endif
		break;
	  case EDV_OBJECT_TYPE_ERROR:
		break;
	}

	return(0);
}

/*
 *	Converts the EDVObjectType type to a string name.
 *
 *	The type specifies the EDVObjectType which must be one of
 *	EDV_OBJECT_TYPE_*.
 *
 *	Returns a statically allocated string describing the type
 *	name.
 */
const gchar *edv_object_type_to_object_name(const EDVObjectType type)
{
	switch(type)
	{
	  case EDV_OBJECT_TYPE_UNKNOWN:
		return("Unknown");
		break;
	  case EDV_OBJECT_TYPE_FILE:
		return("File");
		break;
	  case EDV_OBJECT_TYPE_DIRECTORY:
		return("Directory");
		break;
	  case EDV_OBJECT_TYPE_LINK:
		return("Link");
		break;
	  case EDV_OBJECT_TYPE_FIFO:
		return("FIFO Pipe");
		break;
	  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
		return("Block Device");
		break;
	  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
		return("Character Device");
		break;
	  case EDV_OBJECT_TYPE_SOCKET:
		return("Socket");
		break;
	  case EDV_OBJECT_TYPE_ERROR:
		return("Error");
		break;
	}

	return("Unknown");
}

/*
 *	Converts the EDVObjectType type to a string name in
 *	lowercase.
 *
 *	The type specifies the EDVObjectType which must be one of
 *	EDV_OBJECT_TYPE_*.
 *
 *	Returns a statically allocated string describing the type
 *	name in lowercase.
 */
const gchar *edv_object_type_to_object_name_lower(const EDVObjectType type)
{
	switch(type)
	{
	  case EDV_OBJECT_TYPE_UNKNOWN:
		return("unknown");
		break;
	  case EDV_OBJECT_TYPE_FILE:
		return("file");
		break;
	  case EDV_OBJECT_TYPE_DIRECTORY:
		return("directory");
		break;
	  case EDV_OBJECT_TYPE_LINK:
		return("link");
		break;
	  case EDV_OBJECT_TYPE_FIFO:
		return("FIFO pipe");
		break;
	  case EDV_OBJECT_TYPE_DEVICE_BLOCK:
		return("block device");
		break;
	  case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
		return("character device");
		break;
	  case EDV_OBJECT_TYPE_SOCKET:
		return("socket");
		break;
	  case EDV_OBJECT_TYPE_ERROR:
		return("error");
		break;
	}

	return("unknown");
}


/*
 *	Creates a string describing the value with ',' deliminators.
 *
 *	The i specifies the value.
 *
 *	Returns a statically allocated string describing the
 *	value with ',' deliminators or NULL on error.
 */
const gchar *edv_str_size_delim(const gulong i)
{
	return(edv_str_size_delim_char(i, ','));
}


const gchar *edv_str_size_delim_char(const gulong i, const gchar delim_char)
{
#define RTN_STR_LEN	80
	static gchar rtn_s[RTN_STR_LEN];

	/* 3 digits or less? (no commas needed) */
	if(i < 1000l)
	{
		g_snprintf(rtn_s, sizeof(rtn_s), "%ld", i);
	}
	else
	{
		gint delim_counter, src_s_len;
		gchar	src_s[sizeof(rtn_s)], *src_ptr,
					*rtn_s_ptr = rtn_s,
					*rtn_s_end = rtn_s_ptr + sizeof(rtn_s);

		/* Generate the source string describing the size without
		 * any deliminators
		 */
		g_snprintf(src_s, sizeof(src_s), "%ld", i);

		/* Calculate the length of the source string */
		src_s_len = STRLEN(src_s);

		/* Initialize the comma counter */
		delim_counter = src_s_len % 3;
		if(delim_counter <= 0)
			delim_counter = 3;

		/* Iterate through the source string converting/copying
		 * to the return string and adding deliminators
		 */
		src_ptr = src_s;
		while((*src_ptr != '\0') && (rtn_s_ptr < rtn_s_end))
		{
			/* At the position to put in a deliminator? */
			if(delim_counter <= 0)
			{
				*rtn_s_ptr++ = delim_char;
				delim_counter = 3;	/* Reset the deliminator counter */
			}

			if(rtn_s_ptr >= rtn_s_end)
				break;

			*rtn_s_ptr++ = *src_ptr++;
			delim_counter--;
		}

		/* Null terminate the return string */
		if(rtn_s_ptr < rtn_s_end)
			*rtn_s_ptr = '\0';
		else
			*(rtn_s_end - 1) = '\0';
	}

	return(rtn_s);
#undef RTN_STR_LEN
}

/*
 *	Formats a string describing the size with the specified
 *	format.
 *
 *	The size specifies the size in bytes.
 *
 *	The size_format specifies the format.
 *
 *	The block_size specifies the size of each block in bytes.
 *
 *	The delim_char specifies the character to use as the
 *	deliminator.
 *
 *	If allow_unit_conversion is TRUE then the returned string
 *	will describe the size in units other than bytes if size_format
 *	specifies a size format other than bytes. Otherwise FALSE
 *	forces the returned string to describe the size in bytes.
 *
 *	Returns a statically allocated string.
 */
const gchar *edv_str_size_format(
	const gulong size,
	const EDVSizeFormat size_format,
	const gulong block_size,
	const gchar delim_char,
	const gboolean allow_unit_conversion
)
{
#define RTN_STR_LEN	80
	static gchar rtn_s[RTN_STR_LEN];

	/* Human readable */
	if((size_format == EDV_SIZE_FORMAT_HUMAN_READABLE) &&
	   allow_unit_conversion
	)
	{
		gulong _block_size = block_size;
		if(_block_size <= 0l)
			_block_size = 1024l;

		/* Gigabytes */
		if((size / _block_size / _block_size / _block_size) >= 1l)
		{
			const gulong converted_size = size / _block_size /
				_block_size / _block_size;
			if(converted_size >= 10l)
				g_snprintf(
					rtn_s, sizeof(rtn_s),
					"%ldG",
					converted_size
				);
			else
				g_snprintf(
					rtn_s, sizeof(rtn_s),
					"%.1fG",
					((gfloat)size / (gfloat)_block_size /
						(gfloat)_block_size / (gfloat)_block_size)
				);
		}
		/* Megabytes */
		else if((size / _block_size / _block_size) >= 1l)
		{
			const gulong converted_size = size / _block_size /
				_block_size;
			if(converted_size >= 10l)
				g_snprintf(
					rtn_s, sizeof(rtn_s),
					"%ldM",
					converted_size
				);
			else
				g_snprintf(
					rtn_s, sizeof(rtn_s),
					"%.1fM",
					((gfloat)size / (gfloat)_block_size /
						(gfloat)_block_size)
				);
		}
		/* Kilobytes */
		else if((size / _block_size) >= 1l)
		{
			const gulong converted_size = size / _block_size;
			if(converted_size >= 10l)
				g_snprintf(
					rtn_s, sizeof(rtn_s),
					"%ldK",
					converted_size
				);
			else
				g_snprintf(
					rtn_s, sizeof(rtn_s),
					"%.1fK",
					((gfloat)size / (gfloat)_block_size)
				);
		}
		/* Bytes */
		else
		{
			g_snprintf(
				rtn_s, sizeof(rtn_s),
				"%ld",
				size
			);
		}
	}
	/* Blocks */
	else if((size_format == EDV_SIZE_FORMAT_BLOCKS) &&
			allow_unit_conversion
	)
	{
		gulong _block_size = block_size;
		if(_block_size <= 0l)
			_block_size = 1024l;

		if((size / _block_size) >= 10l)
			g_snprintf(
				rtn_s, sizeof(rtn_s),
				"%sK",
				edv_str_size_delim(size / _block_size)
			);
		else
			g_snprintf(
				rtn_s, sizeof(rtn_s),
				"%.1fK",
				((gfloat)size / (gfloat)_block_size)
			);
	}
	/* Exact Deliminated */
	else if(size_format == EDV_SIZE_FORMAT_DELIMINATED)
	{
		strcpy(
			(char *)rtn_s,
			(const char *)edv_str_size_delim_char(size, delim_char)
		);
	}
	/* Exact */
	else    /* EDV_SIZE_FORMAT_RAW */
	{
		g_snprintf(
			rtn_s, sizeof(rtn_s),
			"%ld",
			size
		);
	}

	return(rtn_s);
#undef RTN_STR_LEN
}


/*
 *	Returns the current umask.
 */
guint edv_get_umask(void)
{
	const guint m = (guint)umask(0);
	(void)umask((mode_t)m);
	return(m);
}

/*
 *	Sets the umask.
 */
void edv_set_umask(const guint m)
{
	(void)umask((mode_t)m);
}


/*
 *	Converts the stat() mode_t permissions to EDVPermissionFlags
 *	permissions.
 *
 *	The m specifies the stat() mode_t permissions.
 *
 *	Returns the EDVPermissionFlags permissions.
 */
EDVPermissionFlags edv_stat_mode_to_edv_permissions(const guint m)
{
	const mode_t mm = (mode_t)m;
	EDVPermissionFlags p = 0x00000000;

	if(mm & S_IXUSR)
		p |= EDV_PERMISSION_UX;
	if(mm & S_IRUSR)
		p |= EDV_PERMISSION_UR;
	if(mm & S_IWUSR)
		p |= EDV_PERMISSION_UW;

	if(mm & S_IXGRP)
		p |= EDV_PERMISSION_GX;
	if(mm & S_IRGRP)
		p |= EDV_PERMISSION_GR;
	if(mm & S_IWGRP)
		p |= EDV_PERMISSION_GW;

	if(mm & S_IXOTH)
		p |= EDV_PERMISSION_OX;
	if(mm & S_IROTH)
		p |= EDV_PERMISSION_OR;
	if(mm & S_IWOTH)
		p |= EDV_PERMISSION_OW;

	if(mm & S_ISUID)
		p |= EDV_PERMISSION_SETUID;
	if(mm & S_ISGID)
		p |= EDV_PERMISSION_SETGID;
	if(mm & S_ISVTX)
		p |= EDV_PERMISSION_STICKY;

	return(p);
}

/*
 *	Converts the EDVPermissionFlags permissions to stat() mode_t
 *	permissions.
 *
 *	The permissions specifies the EDVPermissionFlags, which can
 *	be any of EDV_PERMISSION_*.
 *
 *	Returns the stat() mode_t permissions.
 */
guint edv_edv_permissions_to_stat_mode(const EDVPermissionFlags permissions)
{
	mode_t mm = 0;

	if(permissions & EDV_PERMISSION_UX)
		mm |= S_IXUSR;
	if(permissions & EDV_PERMISSION_UR)
		mm |= S_IRUSR;
	if(permissions & EDV_PERMISSION_UW)
		mm |= S_IWUSR;

	if(permissions & EDV_PERMISSION_GX)
		mm |= S_IXGRP;
	if(permissions & EDV_PERMISSION_GR)
		mm |= S_IRGRP;
	if(permissions & EDV_PERMISSION_GW)
		mm |= S_IWGRP;

	if(permissions & EDV_PERMISSION_OX)
		mm |= S_IXOTH;
	if(permissions & EDV_PERMISSION_OR)
		mm |= S_IROTH;
	if(permissions & EDV_PERMISSION_OW)
		mm |= S_IWOTH;

	if(permissions & EDV_PERMISSION_SETUID)
		mm |= S_ISUID;
	if(permissions & EDV_PERMISSION_SETGID)
		mm |= S_ISGID;
	if(permissions & EDV_PERMISSION_STICKY)
		mm |= S_ISVTX;

	return((guint)mm);
}

/*
 *	Converts the EDVPermissionFlags permissions to a string
 *	description.
 *
 *	The permissions specifies the EDVPermissionFlags, which can
 *	be any of EDV_PERMISSION_*.
 *
 *	Returns a dynamically allocated string describing the
 *	permissions.
 */
gchar *edv_str_permissions(const EDVPermissionFlags permissions)
{
	return(g_strdup_printf(
		"%c%c%c%c%c%c%c%c%c",
		(permissions & EDV_PERMISSION_UR) ? 'r' : '-',
		(permissions & EDV_PERMISSION_UW) ? 'w' : '-',
		(permissions & EDV_PERMISSION_SETUID) ?
			'S' :
			((permissions & EDV_PERMISSION_UX) ? 'x' : '-'),
		(permissions & EDV_PERMISSION_GR) ? 'r' : '-',
		(permissions & EDV_PERMISSION_GW) ? 'w' : '-',
		(permissions & EDV_PERMISSION_SETGID) ?
			'G' :
			((permissions & EDV_PERMISSION_GX) ? 'x' : '-'),
		(permissions & EDV_PERMISSION_OR) ? 'r' : '-',
		(permissions & EDV_PERMISSION_OW) ? 'w' : '-',
		(permissions & EDV_PERMISSION_STICKY) ?
			'T' :
			((permissions & EDV_PERMISSION_OX) ? 'x' : '-')
	));
}

/*
 *	Gets the current umask as EDVPermissionFlags.
 */
EDVPermissionFlags edv_permissions_get_default(void)
{
	return(edv_stat_mode_to_edv_permissions(edv_get_umask()));
}

/*
 *	Sets the umask with EDVPermissionFlags.
 */
void edv_permissions_set_default(const EDVPermissionFlags permissions)
{
	edv_set_umask(edv_edv_permissions_to_stat_mode(permissions));
}

/*
 *	Sets the permissions of the object.
 *
 *	The path specifies the path to the object.
 *
 *	The m specifies the stat() mode_t permissions.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_chmod(
	const gchar *path,
	const guint m
)
{
	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

	return((gint)chmod(
		(const char *)path,
		(mode_t)m
	));
}

/*
 *	Sets the permissions of the object.
 *
 *	The fd specifies the object's descriptor.
 *
 *	The m specifies the stat() mode_t permissions.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_fchmod(
	const gint fd,
	const guint m
)
{
	if(fd < 0)
	{
		errno = EINVAL;
		return(-2);
	}

	return((gint)fchmod(
		(int)fd,
		m
	));
}

/*
 *	Sets the permissions of the object.
 *
 *	The path specifies the path to the object.
 *
 *	The permissions specifies the EDVPermissionFlags permissions.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_permissions_set(
	const gchar *path,
	const EDVPermissionFlags permissions
)
{
	return(edv_chmod(
		path,
		edv_edv_permissions_to_stat_mode(permissions)
	));
}

/*
 *	Sets the permissions of the object.
 *
 *	The fd specifies the object's descriptor.
 *
 *	The permissions specifies the EDVPermissionFlags permissions.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_permissions_set_fd(
	const gint fd,
	const EDVPermissionFlags permissions
)
{
	return(edv_fchmod(
		fd,
		edv_edv_permissions_to_stat_mode(permissions)
	));
}


/*
 *	Sets the ownership and group of the object.
 *
 *	The path specifies the path to the object.
 *
 *	The uid specifies the new owner.
 *
 *	The gid specifies the new group.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_chown(
	const gchar *path,
	const gint uid,
	const gint gid
)
{
	uid_t _uid;
	gid_t _gid;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

	_uid = (uid_t)uid;
	_gid = (gid_t)gid;

	/* Need to use the object's current time values? */
	if((uid < 0) ||
	   (gid < 0)
	)
	{
		/* Get the object's current time values */
		struct stat stat_buf;
		if(stat((const char *)path, &stat_buf))
			return(-1);

		/* Set the current owner and group value(s) */
		if(uid < 0)
			_uid = stat_buf.st_uid;
		if(gid < 0)
			_gid = stat_buf.st_gid;
	}

	return((gint)chown(
		(const char *)path,
		_uid,
		_gid
	));
}

gint edv_lchown(
	const gchar *path,
	const gint uid,
	const gint gid
)
{
	uid_t _uid;
	gid_t _gid;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

	_uid = (uid_t)uid;
	_gid = (gid_t)gid;

	/* Need to use the object's current time values? */
	if((uid < 0) ||
	   (gid < 0)
	)
	{
		/* Get the object's current time values */
		struct stat stat_buf;
		if(stat((const char *)path, &stat_buf))
			return(-1);

		/* Set the current owner and group value(s) */
		if(uid < 0)
			_uid = stat_buf.st_uid;
		if(gid < 0)
			_gid = stat_buf.st_gid;
	}

	return((gint)lchown(
		(const char *)path,
		_uid,
		_gid
	));
}

/*
 *	Sets the ownership and group of the object.
 *
 *	The fd specifies the object's descriptor.
 *
 *	The uid specifies the new owner.
 *
 *	The gid specifies the new group.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_fchown(
	const gint fd,
	const gint uid,
	const gint gid
)
{
	uid_t _uid;
	gid_t _gid;

	if(fd < 0)
	{
		errno = EINVAL;
		return(-2);
	}

	_uid = (uid_t)uid;
	_gid = (gid_t)gid;

	/* Need to use the object's current time values? */
	if((uid < 0) ||
	   (gid < 0)
	)
	{
		/* Get the object's current time values */
		struct stat stat_buf;
		if(fstat((int)fd, &stat_buf))
			return(-1);

		/* Set the current owner and group value(s) */
		if(uid < 0)
			_uid = stat_buf.st_uid;
		if(gid < 0)
			_gid = stat_buf.st_gid;
	}

	return((gint)fchown(
		(int)fd,
		_uid,
		_gid
	));
}


/*
 *	Set the access and modify times of the object.
 *
 *	The path specifies the path to the object.
 *
 *	The access_time specifies the access time in seconds since
 *	EPOCH. If access_time is (gulong)-1 then the access time will
 *	not be modified.
 *
 *	The modify_time specifies the modify time in seconds since
 *	EPOCH. If modify_time is (gulong)-1 then the access time will
 *	not be modified.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_utime(
	const gchar *path,
	const gulong access_time,
	const gulong modify_time
)
{
	struct utimbuf buf;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

	buf.actime = (time_t)access_time;
	buf.modtime = (time_t)modify_time;

	/* Need to use the object's current time values? */
	if((access_time == (gulong)-1) ||
	   (modify_time == (gulong)-1)
	)
	{
		/* Get the object's current time values */
		struct stat stat_buf;
		if(stat((const char *)path, &stat_buf))
			return(-1);

		/* Set the current time value(s) */
		if(access_time == (gulong)-1)
			buf.actime = stat_buf.st_atime;
		if(modify_time == (gulong)-1)
			buf.modtime = stat_buf.st_mtime;
	}

	return((gint)utime(
		(const char *)path,
		&buf
	));
}


/*
 *	Gets the device major and minor numbers from the device value.
 *
 *	The rdev specifies the device value which should come from
 *	*stat()'s struct stat (dev_t)st_rdev value.
 *
 *	The major_rtn specifies the device major number return value.
 *
 *	The minor_rtn specifies the device minor number return value.
 */
void edv_device_numbers_parse(const gint rdev, gint *major_rtn, gint *minor_rtn)
{
	if(major_rtn != NULL)
		*major_rtn = (gint)(((guint32)rdev >> 8) & 0x000000ff);
	if(minor_rtn != NULL)
		*minor_rtn = (gint)((guint32)rdev & 0x000000ff);
}

/*
 *	Combines the device major and minor numbers into a single
 *	device value.
 *
 *	The major specifies the device major number.
 *
 *	The minor specifies the device minor number.
 *
 *	Returns the device value which confirms to *stat()'s struct
 *	stat (dev_t)st_rdev value.
 */
gint edv_device_numbers_format(const gint major, const gint minor)
{
	return((gint)(
		(((guint32)major & 0x000000ff) << 8) |
		((guint32)minor & 0x000000ff)
	));
}


/*
 *	Gets the current working directory.
 *
 *	Returns a dynamically allocated string describing the current
 *	working directory.
 */
gchar *edv_getcwd(void)
{
	return(STRDUP(g_get_current_dir()));
}

/*
 *	Sets the current working directory.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_setcwd(const gchar *path)
{
	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

	return((gint)chdir((const gchar *)path));
}


/*
 *	Updates the access time of the object or creates a new file
 *	of 0 size.
 *
 *	The path specifies the object.
 *
 *	The t specifies the access time to set in seconds since
 *	EPOCH. If t is (gulong)-1 then the current time will be used.
 *
 *	If create_as_needed is TRUE and path refers to a non-existant
 *	object then a new file of 0 size will be created.
 *
 *	Returns 0 on success or non-zero on error. If the object does
 *	not exist and create_as_needed is FALSE then -7 is returned
 *	and errno is set to ENOENT.
 */
gint edv_touch(
	const gchar *path,
	const gulong t,
	const gboolean create_as_needed
)
{
	gint fd;
	guint m;
	gulong _t;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

	/* Use the current time? */
	if(t == (gulong)-1)
		_t = edv_time();		/* Use the current time */
	else
		_t = t;				/* Use the specified time */

	/* If the object exists then set its access time (do not change
	 * its modified time) and return
	 */
	if(edv_path_exists(path))
		return(edv_utime(
			path,
			_t,			/* Access time */
			(gulong)-1		/* Do not change modify time */
		));

	/* The object does not exist, do not create it? */
	if(!create_as_needed)
	{
		errno = ENOENT;
		return(-7);
	}

	/* Create a new file of 0 size */
	m = edv_get_umask();
	fd = (gint)open(
		(const char *)path,
		O_CREAT |			/* Create as needed */
		O_WRONLY |			/* Open in write mode */
		O_APPEND,			/* Append (do not truncate) */
		(mode_t)(~m) &
			(S_IRUSR | S_IWUSR |
			 S_IRGRP | S_IWGRP |
			 S_IROTH | S_IWOTH)
	);
	if(fd < 0)
		return(-1);

	(void)close((int)fd);

	/* Set the access time as the specified time? */
	if(t != (gulong)-1)
		return(edv_utime(
			path,
			t,			/* Access time */
			(gulong)-1		/* Do not change modify time */
		));
	else
		return(0);
}


/*
 *	Gets the system's temporary files directory.
 *
 *	If the system's temporary files directory environment
 *	variable was set then its value will be returned, otherwise
 *	a compile-time value of the system's temporary files
 *	directory will be returned.
 *
 *	Returns a dynamically allocated string describing the path
 *	to the temporary files directory.
 */
gchar *edv_tmp_directory(void)
{
	const gchar *tmp_dir_path = g_getenv(ENV_VAR_NAME_TMPDIR);
	if(STRISEMPTY(tmp_dir_path))
	{
#if defined(P_tmpdir)
		tmp_dir_path = P_tmpdir;
#elif defined(PATH_TMP)
		tmp_dir_path = PATH_TMP;
#elif defined(_WIN32)
		tmp_dir_path = "C:\\TEMP";
#else
#warning "Unable to find compile-time declaration of temporary files directory, defaulting to \"/tmp\""
		tmp_dir_path = "/tmp";
#endif
	}

	return(g_strdup(tmp_dir_path));
}

/*
 *	Creates a new temporary file.
 *
 *	If tmp_dir_path is not NULL or not an empty string then it
 *	specifies the path to the directory to create the new
 *	temporary file in, otherwise the temporary files directory
 *	specified by the environment will be used.
 *
 *	Returns a dynamically allocated string describing the path
 *	to the new temporary file.
 */
gchar *edv_tmp_name(const gchar *tmp_dir_path)
{
	gint		error_code,
					fd;
	gchar		*path,
					*_tmp_dir_path;

	/* If a temporary files directory path was specified then
	 * use it, otherwise use the temporary files directory path
	 * specified by the environment
	 */
	if(STRISEMPTY(tmp_dir_path))
		_tmp_dir_path = edv_tmp_directory();
	else
		_tmp_dir_path = g_strdup(tmp_dir_path);
	if(_tmp_dir_path == NULL)
		return(NULL);

	/* Format the template path to the new temporary file for
	 * use with mkstemp()
	 */
	path = edv_paths_join(
		_tmp_dir_path,
		"XXXXXX"
	);
	error_code = (gint)errno;

	/* Delete the temporary files directory path */
	g_free(_tmp_dir_path);

	if(path == NULL)
	{
		errno = (int)error_code;
		return(NULL);
	}

	/* Create the tempory file and modify the template path */
	fd = (gint)mkstemp((char *)path);
	error_code = (gint)errno;
	if(fd > -1)
	{
		/* The old mkstemp() in glibc 2.0.6 creates a file with
		 * the permissions set to 0666, which is a security risk,
		 * glibc 2.0.7 and later fixes this however we explicitly
		 * set the permissions to 0600 to ensure that the new
		 * temporary file has the 0600 permissions regardless of
		 * the behavior of mkstemp()
		 */
		(void)edv_fchmod(
			fd,
			S_IRUSR | S_IWUSR
		);
		(void)close((int)fd);
	}
	else
	{
		g_free(path);
		path = NULL;
	}

	errno = (int)error_code;

	return(path);
}


/*
 *	Unlinks (removes locally) the object.
 *
 *	The path specifies the object to remove. The path must not be
 *	a directory.
 * 
 *	Returns 0 on success or non-zero on error.
 */
gint edv_unlink(const gchar *path)
{
	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

	return((gint)unlink((const char *)path));
}


/*
 *	Finds the full path to the program by looking at the PATH
 *	environment variable.
 *
 *	The name specifies the program object's name.
 *
 *	Returns a dynamically allocated string describing the completed
 *	full path to the program or NULL if there was no match. If
 *	name was already a full path then a copy of name is returned.
 */
gchar *edv_which(const gchar *name)
{
	struct stat stat_buf;
	gint i;
	const gchar *path_list;
	gchar           *s,
					*matched_path,
					**pathv;

	if(STRISEMPTY(name))
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Specified name already has an absolute path to it? */
	if(g_path_is_absolute(name))
		return(g_strdup(name));

	/* Get the value of the path environment */
	path_list = g_getenv(ENV_VAR_NAME_PATH);
	if(path_list == NULL)
	{
		errno = ENOENT;
		return(NULL);
	}

	/* Break up the path environment into individual paths */
	pathv = g_strsplit(path_list, G_SEARCHPATH_SEPARATOR_S, -1);
	if(pathv == NULL)
	{
		errno = ENOENT;
		return(NULL);
	}

	/* Check each individual path location for the specified name */
	matched_path = NULL;
	for(i = 0; pathv[i] != NULL; i++);
	for(i--; i >= 0; i--)
	{
		s = g_strconcat(
			pathv[i],
			G_DIR_SEPARATOR_S,
			name,
			NULL
		);
		if(s == NULL)
			continue;

		/* Check if this object exists */
		if(!stat((const char *)s, &stat_buf))
		{
			const mode_t m = stat_buf.st_mode;

#ifdef S_ISREG
			/* Check if this object is a file */
			if(S_ISREG(m))
			{
				/* Check if this file is executable */
				if((m & S_IXUSR) ||
				   (m & S_IXGRP) ||
				   (m & S_IXOTH)
				)
				{
					matched_path = s;
					break;
				}
			}
#else
#warning "S_ISREG was not defined, there will be no way to check if an object's type is a file"
#endif
		}

		g_free(s);
	}

	g_strfreev(pathv);

	if(matched_path == NULL)
		errno = ENOENT;

	return(matched_path);
}


/*
 *	Commit buffer caches to disk.
 */
gint edv_sync(void)
{
	return(sync());
}


/*
 *	Get the current time in seconds since EPOCH.
 */
gulong edv_time(void)
{
	return((gulong)time(NULL));
}

/*
 *	Gets the time of day in milliseconds.
 */
extern gulong edv_time_ms(void)
{
#if defined(_WIN32)
	SYSTEMTIME t;
	GetSystemTime(&t);
	return(
		(gulong)(
			(((((t.wHour * 60.0) + t.wMinute) * 60) + t.wSecond) * 1000) +
			t.wMilliseconds
		)
	);
#else
	struct timeval tv[1];

	if(gettimeofday(tv, NULL) < 0)
		return(0l);

	return(
		(gulong)((tv->tv_sec % 86400) * 1000) +
		(gulong)(tv->tv_usec / 1000)
	);
#endif
}

/*
 *	Microsleep.
 *
 *	The us specifies the time to sleep in microseconds.
 */
void edv_usleep(const gulong us)
{
	usleep((unsigned long)us);
}


/*
 *	Gets the idle interval in milliseconds from the priority.
 */
gulong edv_get_interval_from_load_images_priority(
	const EDVListsLoadImagesPriority priority
)
{
	switch(priority)
	{
	  case EDV_LISTS_LOAD_IMAGES_PRIORITY_HIGHEST:
		return(11l);
		break;
	  case EDV_LISTS_LOAD_IMAGES_PRIORITY_HIGH:
		return(50l);
		break;
	  case EDV_LISTS_LOAD_IMAGES_PRIORITY_NORMAL:
		return(100l);
		break;
	  case EDV_LISTS_LOAD_IMAGES_PRIORITY_LOW:
		return(200l);
		break;
	  case EDV_LISTS_LOAD_IMAGES_PRIORITY_LOWEST:
		return(300l);
		break;
	}

	return(300l);
}


/*
 *	Checks the descriptor status.
 *
 *	The fd specifies the descriptor.
 *
 *	The mode specifies a string describing the mode which can
 *	contain any of the following characters to describe the
 *	checking of the descriptor status:
 *
 *	r		Data is waiting to be read.
 *	w		Writing to the descriptor will not block.
 *
 *	The timeout_ms specifies the timeout in milliseconds. If
 *	timeout_ms is 0 then this call will not block and return
 *	immediately.
 *
 *	Returns TRUE if any of the specified mode condition were met
 *	or FALSE otherwise or on error.
 */
gboolean edv_poll(
	const gint fd,
	const gchar *mode,
	const gulong timeout_ms
)
{
	fd_set		read_fd_set,
					write_fd_set,
					*read_fd_set_ptr,
					*write_fd_set_ptr;
	struct timeval tv;
	gchar c;
	const gchar *s;

	if((fd < 0) || (mode == NULL))
	{
		errno = EINVAL;
		return(FALSE);
	}

	/* Set up the read and write fd_set values based on the
	 * specified modes, the pointers to each fd_set will be set
	 * to the actual fd_set based on if their corresponding mode
	 * was specified
	 */
	read_fd_set_ptr = NULL;
	write_fd_set_ptr = NULL;
	s = mode;
	while(*s != '\0')
	{
		c = (gchar)tolower((char)*s);
		if(c == 'r')
		{
			read_fd_set_ptr = &read_fd_set;
			FD_ZERO(read_fd_set_ptr);
			FD_SET(
				(int)fd,
				read_fd_set_ptr
			);
		}
		else if(c == 'w')
		{
			write_fd_set_ptr = &write_fd_set;
			FD_ZERO(write_fd_set_ptr);
			FD_SET(
				(int)fd,
				write_fd_set_ptr
			);
		}
		s++;
	}

	/* Set up the timeout */
	if(timeout_ms == 0l)
	{
		tv.tv_sec = 0l;
		tv.tv_usec = 0l;
	}
	else if(timeout_ms < 1000l)
	{
		tv.tv_sec = 0l;
		tv.tv_usec = (time_t)(timeout_ms * 1000l);
	}
	else
	{
		tv.tv_sec = (time_t)(timeout_ms / 1000l);
		tv.tv_usec = (time_t)((timeout_ms % 1000l) * 1000l);
	}

	return((select(
		(int)fd + 1,			/* Highest descriptor index + 1 */
		read_fd_set_ptr,
		write_fd_set_ptr,
		NULL,				/* No exceptions */
		&tv				/* Timeout */
	) > 0) ? TRUE : FALSE);
}

/*
 *	Checks the descriptor's read status.
 *
 *	The fd specifies the descriptor.
 * 
 *	Same as calling edv_poll(fd, "r", 0l).
 *
 *	Returns TRUE if data is waiting to be read or FALSE otherwise
 *	or on error.
 */
gboolean edv_poll_read(const gint fd)
{
	fd_set read_fd_set;
	struct timeval tv;

	if(fd < 0)
	{
		errno = EINVAL;
		return(FALSE);
	}

	FD_ZERO(&read_fd_set);
	FD_SET(
		(int)fd,
		&read_fd_set
	);

	/* Zero the timeout so that select() will not block and return
	 * immediately
	 */
	tv.tv_sec = 0l;
	tv.tv_usec = 0l;

	return((select(
		(int)fd + 1,			/* Highest descriptor index + 1 */
		&read_fd_set,
		NULL,				/* No write descriptors */
		NULL,				/* No exceptions */
		&tv				/* Timeout */
	) > 0) ? TRUE : FALSE);
}

/*
 *	Checks the descriptor's write status.
 *
 *	The fd specifies the descriptor.
 *
 *	Same as calling edv_poll(fd, "w", 0l).
 *
 *	Returns TRUE if writing to the descriptor will not block or
 *	FALSE otherwise or on error.
 */
gboolean edv_poll_write(const gint fd)
{
	fd_set write_fd_set;
	struct timeval tv;

	if(fd < 0)
	{
		errno = EINVAL;
		return(FALSE);
	}

	FD_ZERO(&write_fd_set);
	FD_SET(
		(int)fd,
		&write_fd_set
	);

	/* Zero the timeout so that select() will not block and return
	 * immediately
	 */
	tv.tv_sec = 0l;
	tv.tv_usec = 0l;

	return((select(
		(int)fd + 1,			/* Highest descriptor index + 1 */
		NULL,				/* No read descriptors */
		&write_fd_set,
		NULL,				/* No exceptions */
		&tv				/* Timeout */
	) > 0) ? TRUE : FALSE);
}


/*
 *	Opens a GList of gchar * strings from a file.
 *
 *	The path specifies the full path to the file to open from.
 *
 *	If max_lines is positive then no more than max_lines will be
 *	read from the file.
 *
 *	If strip_crs is TRUE then carrage return characters ('\r')
 *	will not be stored into the return strings.
 *
 *	Returns a GList of gchar * strings describing each line. The
 *	calling function must delete the returned list and each string.
 */
GList *edv_open_text_file_glist(
	const gchar *path,
	const gint max_lines,
	const gboolean strip_crs
)
{
	FILE *fp;
	gint lines_read;
        gchar *line_buf;
	GList *lines_list;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Open the file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
		return(NULL);

	/* Read each line from the file */
	line_buf = NULL;
	lines_read = 0;
	lines_list = NULL;
        while(!feof(fp))
	{
		/* Read until a newline character, null character, or
		 * EOF is read
		 *
		 * edv_stream_read_strptrbrk() returns TRUE only if
		 * one of the end characters was read so we know we
		 * got a complete line
		 */
                if(edv_stream_read_strptrbrk(
                        fp,
                        &line_buf,
			"\n",
			FALSE,			/* Exclude end character */
                        TRUE			/* Block */
                ))
		{
			/* Remove carrage return characters? */
			if(strip_crs)
			{
				gchar *s = edv_strsub(
					line_buf,
					"\r",
					""
				);
				if(s != NULL)
				{
					g_free(line_buf);
					line_buf = s;
				}
			}
			/* Append this line to the lines list */
			lines_list = g_list_append(
				lines_list,
				line_buf
			);
                        line_buf = NULL;

			lines_read++;
			if(max_lines > 0)
			{
				if(lines_read >= max_lines)
					break;
			}
                }
        }

	/* Append the remaining line to the lines list */
	if(line_buf != NULL)
	{
		lines_list = g_list_append(
			lines_list,
			line_buf
		);
/*		line_buf = NULL; */
	}

	/* Close the file */
        (void)fclose(fp);

	return(lines_list);
}

/*
 *	Saves the GList of gchar * strings to a file.
 * 
 *	The path specifies the full path to the file to save to.
 *
 *	The lines_list specifies a GList of gchar * strings to
 *	write to the file.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_save_text_file_glist(
	const gchar *path,
	GList *lines_list
)
{
	size_t		units_to_write,
			units_written;
	FILE *fp;
	const gchar *s;
	GList *glist;
	EDVPermissionFlags permissions;
	EDVVFSObject *obj;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

	obj = edv_vfs_object_stat(path);
	if(obj != NULL)
	{
		permissions = obj->permissions;
		edv_vfs_object_delete(obj);
	}
	else
	{
		permissions = (~edv_permissions_get_default()) &
			(EDV_PERMISSION_UR | EDV_PERMISSION_UW |
			 EDV_PERMISSION_GR | EDV_PERMISSION_GW |
			 EDV_PERMISSION_OR | EDV_PERMISSION_OW);
	}

	/* Open the file for writing */
	fp = fopen((const char *)path, "wb");
	if(fp == NULL)
		return(-1);

	/* Write each line to the file */
	for(glist = lines_list; glist != NULL; glist = g_list_next(glist))
	{
		/* Get this line */
		s = (const gchar *)glist->data;
		if(s == NULL)
			continue;

		/* Write this line */
		units_to_write = (size_t)strlen(s);
		if(units_to_write > 0l)
		{
			units_written = fwrite(
				s,
				sizeof(gchar),
				units_to_write,
				fp
			);
			if(units_written != units_to_write)
				return(-1);
		}
		if(fputc('\n', fp) == EOF)
			return(-1);
	}

	/* Close the file */
	if(fclose(fp))
		return(-1);

	(void)edv_permissions_set(
		path,
		permissions
	);

	return(0);
}


/*
 *	Gets the host name.
 *
 *	Returns a dynamically allocated string describing the host name
 *	or NULL on error.
 */
gchar *edv_get_host_name(void)
{
	gchar name[64 + 1];

	if(gethostname((char *)name, sizeof(name)))
		return(NULL);

	name[sizeof(name) - 1] = '\0';

	return(g_strdup(name));
}

/*
 *	Sets the host name.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_set_host_name(const gchar *name)
{
	if(STRISEMPTY(name))
	{
		errno = EINVAL;
		return(-2);
	}

	return((gint)sethostname((const char *)name, strlen(name)));
}
