/*
		    Endeavour Mark II - Stream

	Checking, reading from, and writing to streams.


	Examples:


	Example of using edv_stream_read_str() to read an entire
	text file at once:

	// Open the text file
	FILE *fp = fopen("/etc/issue", "rb");
	// Read the entire file as a single string
	gchar *s = edv_stream_read_str(
		fp,
		TRUE				// Block
	);
	if(s != NULL)
	{
		// print the entire string to stdout
		g_print("%s", s);
		g_free(s);
	}
	(void)fclose(fp);


	Example of using edv_stream_read_line() to read a text file
	one line at a time:

	// Open the text file
	FILE *fp = fopen("/etc/issue", "rb");
	// Read and print each line using a line buffer, the line
	// buffer must be initialized to NULL
	gchar *line_buf = NULL;
	while(!feof(fp)) {
		// Read until a newline character, null character, or EOF
		// is read, if a newline character was read then
		// edv_stream_read_lineptr() returns TRUE
		if(edv_stream_read_lineptr(
			fp,
			&line_buf,		// Will be reallocated
						// by the call
			TRUE			// Block
		)) {
			g_print("%s", line_buf);
			// Reset the buffer for the next line
			g_free(line_buf);
			line_buf = NULL;
		}
	}
	(void)fclose(fp);


	Example uf using edv_stream_read_strptrbrk() to read output
	from a child process:

	// Exacute a process
	FILE *cstdout, *cstderr;
	gchar *cstdout_buf = NULL, *cstderr_buf = NULL;
	gint pid = edv_system_shell_streams(
		"/usr/sbin/traceroute freshmeat.net",
		NULL, NULL,			// Default shell & args
		NULL,				// No child stdin
		&cstdout,
		&cstderr
	);
	while(pid > 0) {
		if(!edv_pid_exists(pid))
			pid = 0;
		// Read from the child stdout,
		// edv_stream_read_strptrbrk() returns TRUE if
		// a newline or return character was read
		if(edv_stream_read_strptrbrk(
			cstdout,
			&cstdout_buf,
			"\n\r",
			FALSE,			// Do not store the \n or \r
						// to the cstdout_buf
			FALSE			// Nonblocking since this is
						// is a process' stream we
						// are reading from and we
						// want to do other things
						// while reading from it
		)) {
			// Got a compelete line, print it
			g_print("%s\n", cstdout_buf);
			// Reset the buffer for the next line
			g_free(cstdout_buf);
			cstdout_buf = NULL;
		}
		// Read from the child stderr the same way as stdout
		if(edv_stream_read_strptrbrk(
			cstderr,
			&cstderr_buf,
			"\n\r",
			FALSE,
			FALSE
		)) {
			g_printerr("%s\n", cstderr_buf);
			g_free(cstderr_buf);
			cstderr_buf = NULL;
		}
		// Do other things here
	}
	(void)fclose(cstdout);
	(void)fclose(cstderr);


	See also: edv_utils.c
 */

#ifndef EDV_STREAM_H
#define EDV_STREAM_H

#include <stdio.h>				/* For FILE * */
#include <glib.h>


/*
 *	Reads the characters from the stream as a string until a
 *	null character or EOF is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; one of the end characters, a null
 *	character, or EOF is read. Otherwise this function returns
 *	when any of the above conditions are met or when no more data
 *	is available to be read from the stream.
 *
 *	Returns a dynamically allocated string describing the
 *	characters read from the stream or NULL if no data was
 *	available or on error.
 */
extern gchar *edv_stream_read_str(
	FILE *stream,
	const gboolean block_until_end
);

/*
 *	Reads the characters from the stream to a string buffer
 *	until a null character or EOF is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	The s specifies the string buffer. The string buffer will
 *	be either deleted or reallocated by this function, and
 *	therefore, the calling function should not reference it
 *	after this call.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; one of the end characters, a null
 *	character, or EOF is read. Otherwise this function returns
 *	when any of the above conditions are met or when no more data
 *	is available to be read from the stream.
 *
 *	Returns a dynamically allocated string describing the
 *	characters read from the stream or NULL if no data was
 *	available or on error.
 */
extern gchar *edv_stream_read_strbuf(
	FILE *stream,
	gchar *s,
	const gboolean block_until_end
);


/*
 *      Reads the characters from the stream as a string until;
 *      one of the specified end characters, a null character or EOF
 *      is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	If end_characters is not NULL then it specifies a list of
 *	characters to stop reading at when one of them is read. This
 *	function always stops reading when a null character or EOF
 *	is read.
 *
 *	If include_end_character is TRUE then the end character that
 *	was read (if encountered) will be included to the return
 *	string. This has no affect is end_characters is NULL.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; one of the end characters, a null
 *	character, or EOF is read. Otherwise this function returns
 *	when any of the above conditions are met or when no more data
 *	is available to be read from the stream.
 *
 *	Returns a dynamically allocated string describing the
 *	characters read from the stream or NULL if no data was
 *	available or on error.
 */
extern gchar *edv_stream_read_strbrk(
	FILE *stream,
	const gchar *end_characters,
	const gboolean include_end_character,
	const gboolean block_until_end
);

/*
 *	Reads the characters from the stream to a string buffer
 *	until; one of the specified end characters, a null character
 *	or EOF is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	The s specifies the string buffer. The string buffer will
 *	be either deleted or reallocated by this function, and
 *	therefore, the calling function should not reference it
 *	after this call.
 *
 *	If end_characters is not NULL then it specifies a list of
 *	characters to stop reading at when one of them is read. This
 *	function always stops reading when a null character or EOF
 *	is read.
 *
 *	If include_end_character is TRUE then the end character that
 *	was read (if encountered) will be included to the return
 *	string. This has no affect is end_characters is NULL.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; one of the end characters, a null
 *	character, or EOF is read. Otherwise this function returns
 *	when any of the above conditions are met or when no more data
 *	is available to be read from the stream.
 *
 *	Returns a dynamically allocated string describing the
 *	characters read from the stream or NULL if no data was
 *	available or on error.
 */
extern gchar *edv_stream_read_strbufbrk(
	FILE *stream,
	gchar *s,
	const gchar *end_characters,
	const gboolean include_end_character,
	const gboolean block_until_end
);

/*
 *	Reads the characters from the stream to a string buffer
 *	until; one of the specified end characters, a null character
 *	or EOF is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	The s_ptr specifies the pointer to the string buffer. The
 *	value of *s_ptr must be initialized prior to this call and
 *	it may be deleted or reallocated by this call. The calling
 *	function must delete *s_ptr.
 *
 *	If end_characters is not NULL then it specifies a list of
 *	characters to stop reading at when one of them is read. This
 *	function always stops reading when a null character or EOF
 *	is read.
 *
 *	If include_end_character is TRUE then the end character that
 *	was read (if encountered) will be included to the return
 *	string. This has no affect is end_characters is NULL.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; one of the end characters, a null
 *	character, or EOF is read. Otherwise this function returns
 *	when any of the above conditions are met or when no more data
 *	is available to be read from the stream.
 *
 *	Returns TRUE if one of end_characters was read or FALSE on all
 *	other cases (a null character or EOF was read, no more data to
 *	be read, or error).
 */
extern gboolean edv_stream_read_strptrbrk(
        FILE *stream,
        gchar **s_ptr,
        const gchar *end_characters,
        const gboolean include_end_character,
        const gboolean block_until_end
);


/*
 *	Reads the characters from the stream as a string until; a
 *	newline character, null character, or EOF is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; a newline character, a null character,
 *	or EOF is read. Otherwise this function returns when any of
 *	the above conditions are met or when no more data is available
 *	to be read from the stream.
 *
 *	Returns a dynamically allocated string describing the
 *	characters read from the stream including the newline
 *	character or NULL if no data was available or on error.
 */
extern gchar *edv_stream_read_line(
	FILE *stream,
	const gboolean block_until_end
);

/*
 *	Reads the characters from the stream to a string buffer until;
 *	a newline character, null character, or EOF is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	The s specifies the string buffer. The string buffer will
 *	be either deleted or reallocated by this function, and
 *	therefore, the calling function should not reference it
 *	after this call.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; a newline character, a null character,
 *	or EOF is read. Otherwise this function returns when any of
 *	the above conditions are met or when no more data is available
 *	to be read from the stream.
 *
 *	Returns a dynamically allocated string describing the
 *	characters read from the stream including the newline
 *	character or NULL if no data was available or on error.
 */
extern gchar *edv_stream_read_linebuf(
	FILE *stream,
	gchar *s,
	const gboolean block_until_end
);

/*
 *	Reads the characters from the stream to a string buffer
 *	until; a newline, a null character, or EOF is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	The s_ptr specifies the pointer to the string buffer. The
 *	value of *s_ptr must be initialized prior to this call and
 *	it may be deleted or reallocated by this call. The calling
 *	function must delete *s_ptr.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; one of the end characters, a null
 *	character, or EOF is read. Otherwise this function returns
 *	when any of the above conditions are met or when no more data
 *	is available to be read from the stream.
 *
 *	Returns TRUE if a newline was read or FALSE on all other cases
 *	(a null character or EOF was read, no more data to be read, or
 *	error).
 */
extern gboolean edv_stream_read_lineptr(
        FILE *stream,
        gchar **s_ptr,
        const gboolean block_until_end
);


#endif	/* EDV_STREAM_H */
