#include <stdio.h>
#include <errno.h>
#include <glib.h>

#include "edv_utils.h"
#include "edv_stream.h"


/* Read String */
gchar *edv_stream_read_str(
	FILE *stream,
	const gboolean block_until_end
);
gchar *edv_stream_read_strbuf(
	FILE *stream,
	gchar *s,
	const gboolean block_until_end
);

/* Read String Break */
gchar *edv_stream_read_strbrk(
	FILE *stream,
	const gchar *end_characters,
	const gboolean include_end_character,
	const gboolean block_until_end
);
gchar *edv_stream_read_strbufbrk(
	FILE *stream,
	gchar *s,
	const gchar *end_characters,
	const gboolean include_end_character,
	const gboolean block_until_end
);
gboolean edv_stream_read_strptrbrk(
	FILE *stream,
	gchar **s_ptr,
	const gchar *end_characters,
	const gboolean include_end_character,
	const gboolean block_until_end
);

/* Read Line (String Ending With Newline) */
gchar *edv_stream_read_line(
	FILE *stream,
	const gboolean block_until_end
);
gchar *edv_stream_read_linebuf(
	FILE *stream,
	gchar *s,
	const gboolean block_until_end
);
gboolean edv_stream_read_lineptr(
	FILE *stream,
	gchar **s_ptr,
	const gboolean block_until_end
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Reads the characters from the stream as a string until a
 *	null character or EOF is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; one of the end characters, a null
 *	character, or EOF is read. Otherwise this function returns
 *	when any of the above conditions are met or when no more data
 *	is available to be read from the stream.
 *
 *	Returns a dynamically allocated string describing the
 *	characters read from the stream or NULL if no data was
 *	available or on error.
 */
gchar *edv_stream_read_str(
	FILE *stream,
	const gboolean block_until_end
)
{
	return(edv_stream_read_strbufbrk(
		stream,
		NULL,
		NULL,
		FALSE,
		block_until_end
	));
}

/*
 *	Reads the characters from the stream to a string buffer
 *	until a null character or EOF is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	The s specifies the string buffer. The string buffer will
 *	be either deleted or reallocated by this function, and
 *	therefore, the calling function should not reference it
 *	after this call.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; one of the end characters, a null
 *	character, or EOF is read. Otherwise this function returns
 *	when any of the above conditions are met or when no more data
 *	is available to be read from the stream.
 *
 *	Returns a dynamically allocated string describing the
 *	characters read from the stream or NULL if no data was
 *	available or on error.
 */
gchar *edv_stream_read_strbuf(
	FILE *stream,
	gchar *s,
	const gboolean block_until_end
)
{
	return(edv_stream_read_strbufbrk(
		stream,
		s,
		NULL,
		FALSE,
		block_until_end
	));
}


/*
 *	Reads the characters from the stream as a string until;
 *	one of the specified end characters, a null character or EOF
 *	is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	If end_characters is not NULL then it specifies a list of
 *	characters to stop reading at when one of them is read. This
 *	function always stops reading when a null character or EOF
 *	is read.
 *
 *	If include_end_character is TRUE then the end character that
 *	was read (if encountered) will be included to the return
 *	string. This has no affect is end_characters is NULL.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; one of the end characters, a null
 *	character, or EOF is read. Otherwise this function returns
 *	when any of the above conditions are met or when no more data
 *	is available to be read from the stream.
 *
 *	Returns a dynamically allocated string describing the
 *	characters read from the stream or NULL if no data was
 *	available or on error.
 */
gchar *edv_stream_read_strbrk(
	FILE *stream,
	const gchar *end_characters,
	const gboolean include_end_character,
	const gboolean block_until_end
)
{
	return(edv_stream_read_strbufbrk(
		stream,
		NULL,
		end_characters,
		include_end_character,
		block_until_end
	));
}

/*
 *	Reads the characters from the stream to a string buffer
 *	until; one of the specified end characters, a null character
 *	or EOF is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	The s specifies the string buffer. The string buffer will
 *	be either deleted or reallocated by this function, and
 *	therefore, the calling function should not reference it
 *	after this call.
 *
 *	If end_characters is not NULL then it specifies a list of
 *	characters to stop reading at when one of them is read. This
 *	function always stops reading when a null character or EOF
 *	is read.
 *
 *	If include_end_character is TRUE then the end character that
 *	was read (if encountered) will be included to the return
 *	string. This has no affect is end_characters is NULL.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; one of the end characters, a null
 *	character, or EOF is read. Otherwise this function returns
 *	when any of the above conditions are met or when no more data
 *	is available to be read from the stream.
 *
 *	Returns a dynamically allocated string describing the
 *	characters read from the stream or NULL if no data was
 *	available or on error.
 */
gchar *edv_stream_read_strbufbrk(
	FILE *stream,
	gchar *s,
	const gchar *end_characters,
	const gboolean include_end_character,
	const gboolean block_until_end
)
{
	gchar *ls = s;
	(void)edv_stream_read_strptrbrk(
		stream,
		&ls,
		end_characters,
		include_end_character,
		block_until_end
	);
	return(ls);
}

/*
 *	Reads the characters from the stream to a string buffer
 *	until; one of the specified end characters, a null character
 *	or EOF is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	The s_ptr specifies the pointer to the string buffer. The
 *	value of *s_ptr must be initialized prior to this call and
 *	it may be deleted or reallocated by this call. The calling
 *	function must delete *s_ptr.
 *
 *	If end_characters is not NULL then it specifies a list of
 *	characters to stop reading at when one of them is read. This
 *	function always stops reading when a null character or EOF
 *	is read.
 *
 *	If include_end_character is TRUE then the end character that
 *	was read (if encountered) will be included to the return
 *	string. This has no affect is end_characters is NULL.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; one of the end characters, a null
 *	character, or EOF is read. Otherwise this function returns
 *	when any of the above conditions are met or when no more data
 *	is available to be read from the stream.
 *
 *	Returns TRUE if one of end_characters was read or FALSE on all
 *	other cases (a null character or EOF was read, no more data to
 *	be read, or error).
 */
gboolean edv_stream_read_strptrbrk(
	FILE *stream,
	gchar **s_ptr,
	const gchar *end_characters,
	const gboolean include_end_character,
	const gboolean block_until_end
)
{
	gboolean status;

	if((stream == NULL) || (s_ptr == NULL))
	{
		errno = EINVAL;
		return(FALSE);
	}

	status = FALSE;

	/* Read according to the specified block method */
	if(block_until_end)
	{
		int c;
		gchar	buf[32],
			*buf_ptr,
			*buf_end;

		/* Keep reading until; an end character, null
		 * character, or EOF is read
		 */
		while(!feof(stream))
		{
			/* Reset the read buffer, set the end point to
			 * one less than the actual end so we can set
			 * the null byte after the read buffer loop
			 */
			buf_ptr = buf;
			buf_end = buf + sizeof(buf) - 1;

			/* Read buffer loop */
			while(buf_ptr < buf_end)
			{
				/* Get the next character from the stream */
				c = fgetc(stream);

				/* Always stop reading at the end of
				 * file or null character
				 */
				if(c == EOF)
				{
					*buf_ptr = '\0';
					/* Only append the read buffer
					 * if it was not empty on EOF
					 * so that an immediate EOF on
					 * this call causes this call
					 * to return NULL
					 */
					if(*buf != '\0')
						*s_ptr = edv_strcat(
							*s_ptr,
							buf
						);
					return(status);
				}
				if(c == '\0')
				{
					*buf_ptr = '\0';
					*s_ptr = edv_strcat(
						*s_ptr,
						buf
					);
					return(status);
				}

				/* Check if one of the specified end
				 * characters was read
				 */
				if(end_characters != NULL)
				{
					const gchar *ecp = end_characters;
					while(*ecp != '\0')
					{
						if((gchar)c == *ecp)
							break;
						ecp++;
					}
					/* End character read? */
					if(*ecp != '\0')
					{
						/* Mark that an end
						 * character was read
						 */
						status = TRUE;

						*buf_ptr = '\0';
						*s_ptr = edv_strcat(
							*s_ptr,
							buf
						);
						if(include_end_character)
						{
							buf[0] = (gchar)c;
							buf[1] = '\0';
							*s_ptr = edv_strcat(
								*s_ptr,
								buf
							);
						}
						return(status);
					}
				}

				/* Add this character to the read buffer */
				*buf_ptr = (gchar)c;
				buf_ptr++;
			}

			/* Null terminate the read buffer */
			*buf_ptr = '\0';

			/* Append the read buffer to the return string */
			*s_ptr = edv_strcat(
				*s_ptr,
				buf
			);

		}
	}
	else
	{
		int c;
		gchar	buf[32],
			*buf_ptr,
			*buf_end;
		const gint fd = (gint)fileno(stream);

		/* Keep reading only when there is data to be read */
		while(!feof(stream) && edv_poll_read(fd))
		{
			/* Reset the read buffer, set the end point to
			 * one less than the actual end so we can set
			 * the null byte after the read buffer loop
			 */
			buf_ptr = buf;
			buf_end = buf + sizeof(buf) - 1;

			/* Read buffer loop */
			while(buf_ptr < buf_end)
			{
				/* Get the next character from the stream */
				c = fgetc(stream);

				/* Always stop reading at the end of
				 * file or null character
				 */
				if(c == EOF)
				{
					*buf_ptr = '\0';
					/* Only append the read buffer
					 * if it was not empty on EOF
					 * so that an immediate EOF on
					 * this call causes this call
					 * to return NULL
					 */
					*s_ptr = edv_strcat(
						*s_ptr,
						buf
					);
					return(status);
				}
				if(c == '\0')
				{
					*buf_ptr = '\0';
					*s_ptr = edv_strcat(
						*s_ptr,
						buf
					);
					return(status);
				}

				/* Check if one of the specified end
				 * characters was read
				 */
				if(end_characters != NULL)
				{
					const gchar *ecp = end_characters;
					while(*ecp != '\0')
					{
						if((gchar)c == *ecp)
							break;
						ecp++;
					}
					/* End character read? */
					if(*ecp != '\0')
					{
						/* Mark that an end
						 * character was read
						 */
						status = TRUE;

						*buf_ptr = '\0';
						*s_ptr = edv_strcat(
							*s_ptr,
							buf
						);
						if(include_end_character)
						{
							buf[0] = (gchar)c;
							buf[1] = '\0';
							*s_ptr = edv_strcat(
								*s_ptr,
								buf
							);
						}
						return(status);
					}
				}

				/* Add this character to the read buffer */
				*buf_ptr = (gchar)c;
				buf_ptr++;

				/* No more data to be read? */
				if(!edv_poll_read(fd))
					break;
			}

			/* Null terminate the read buffer */
			*buf_ptr = '\0';

			/* Append the read buffer to the return string */
			*s_ptr = edv_strcat(
				*s_ptr,
				buf
			);
		}
	}

	return(status);
}


/*
 *	Reads the characters from the stream as a string until; a
 *	newline character, null character, or EOF is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; a newline character, a null character,
 *	or EOF is read. Otherwise this function returns when any of
 *	the above conditions are met or when no more data is available
 *	to be read from the stream.
 *
 *	Returns a dynamically allocated string describing the
 *	characters read from the stream including the newline
 *	character or NULL if no data was available or on error.
 */
gchar *edv_stream_read_line(
	FILE *stream,
	const gboolean block_until_end
)
{
	return(edv_stream_read_strbufbrk(
		stream,
		NULL,
		"\n",
		TRUE,
		block_until_end
	));
}

/*
 *	Reads the characters from the stream to a string buffer until;
 *	a newline character, null character, or EOF is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	The s specifies the string buffer. The string buffer will
 *	be either deleted or reallocated by this function, and
 *	therefore, the calling function should not reference it
 *	after this call.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; a newline character, a null character,
 *	or EOF is read. Otherwise this function returns when any of
 *	the above conditions are met or when no more data is available
 *	to be read from the stream.
 *
 *	Returns a dynamically allocated string describing the
 *	characters read from the stream including the newline
 *	character or NULL if no data was available or on error.
 */
gchar *edv_stream_read_linebuf(
	FILE *stream,
	gchar *s,
	const gboolean block_until_end
)
{
	return(edv_stream_read_strbufbrk(
		stream,
		s,
		"\n",
		TRUE,
		block_until_end
	));
}

/*
 *	Reads the characters from the stream to a string buffer
 *	until; a newline, a null character, or EOF is read.
 *
 *	The stream specifies the stream to read from.
 *
 *	The s_ptr specifies the pointer to the string buffer. The
 *	value of *s_ptr must be initialized prior to this call and
 *	it may be deleted or reallocated by this call. The calling
 *	function must delete *s_ptr.
 *
 *	If block_until_end is TRUE then this function will block
 *	and keep reading until; one of the end characters, a null
 *	character, or EOF is read. Otherwise this function returns
 *	when any of the above conditions are met or when no more data
 *	is available to be read from the stream.
 *
 *	Returns TRUE if a newline was read or FALSE on all other cases
 *	(a null character or EOF was read, no more data to be read, or
 *	error).
 */
gboolean edv_stream_read_lineptr(
	FILE *stream,
	gchar **s_ptr,
	const gboolean block_until_end
)
{
	return(edv_stream_read_strptrbrk(
		stream,
		s_ptr,
		"\n",
		TRUE,				/* Include end character */
		block_until_end
	));
}
