#include <glib.h>

#include "edv_types.h"
#include "edv_property.h"
#include "edv_recycled_obj.h"


EDVRecycledObject *edv_recycled_object_new(void);
EDVRecycledObject *edv_recycled_object_copy(EDVRecycledObject *obj);
void edv_recycled_object_set_object(
	EDVRecycledObject *obj,
	EDVRecycledObject *obj_values
);
void edv_recycled_object_clear(EDVRecycledObject *obj);
void edv_recycled_object_delete(EDVRecycledObject *obj);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new EDVRecycledObject.
 *
 *	Returns a new dynamically allocated EDVRecycledObject with all of its
 *	values zero'ed or NULL on error.
 */
EDVRecycledObject *edv_recycled_object_new(void)
{
	return(EDV_RECYCLED_OBJECT(g_malloc0(
	    sizeof(EDVRecycledObject)
	)));
}

/*
 *	Creates a new EDVRecycledObject of a specific type.
 *
 *	The type specifies the EDVRecycledObject's type.
 *
 *	Returns a new dynamically allocated EDVRecycledObject with its
 *	type value set or NULL on error.
 */
EDVRecycledObject *edv_recycled_object_new_type(const EDVObjectType type)
{
	EDVRecycledObject *obj = edv_recycled_object_new();
	if(obj == NULL)
	    return(NULL);

	obj->type = type;

	return(obj);
}

/*
 *	Coppies the EDVRecycledObject.
 *
 *	The obj specifies the EDVRecycledObject to copy.
 *
 *	Returns a new dynamically allocated copy of the
 *	EDVRecycledObject or NULL on error.
 */
EDVRecycledObject *edv_recycled_object_copy(EDVRecycledObject *obj)
{
	EDVRecycledObject *new_obj;

	if(obj == NULL)
	    return(NULL);

	new_obj = edv_recycled_object_new();
	if(new_obj == NULL)
	    return(NULL);

	new_obj->type = obj->type;

	new_obj->index = obj->index;

	new_obj->name = STRDUP(obj->name);
	new_obj->original_path = STRDUP(obj->original_path);

	new_obj->size = obj->size;
	new_obj->storage_size = obj->storage_size;

	new_obj->link_target = STRDUP(obj->link_target);

	new_obj->permissions = obj->permissions;

	new_obj->access_time = obj->access_time;
	new_obj->modify_time = obj->modify_time;
	new_obj->change_time = obj->change_time;
	new_obj->deleted_time = obj->deleted_time;

	new_obj->owner_id = obj->owner_id;
	new_obj->group_id = obj->group_id;

	new_obj->meta_data_list = edv_properties_list_copy(obj->meta_data_list);

	return(new_obj);
}

/*
 *	Sets the EDVRecycledObject's values from another EDVRecycledObject.
 *
 *	The obj specifies the EDVRecycledObject to be set.
 *
 *	The obj_values specifies the values to set obj with.
 */
void edv_recycled_object_set_object(
	EDVRecycledObject *obj,
	EDVRecycledObject *obj_values
)
{
	EDVRecycledObject	*tar_obj = obj,
				*src_obj = obj_values;

	if((tar_obj == NULL) || (src_obj == NULL))
	    return;

	tar_obj->type = src_obj->type;

	tar_obj->index = src_obj->index;

	g_free(tar_obj->name);
	tar_obj->name = STRDUP(src_obj->name);
	g_free(tar_obj->original_path);
	tar_obj->original_path = STRDUP(src_obj->original_path);

	tar_obj->size = src_obj->size;
	tar_obj->storage_size = src_obj->storage_size;

	g_free(tar_obj->link_target);
	tar_obj->link_target = STRDUP(src_obj->link_target);

	tar_obj->permissions = src_obj->permissions;

	tar_obj->access_time = src_obj->access_time;
	tar_obj->modify_time = src_obj->modify_time;
	tar_obj->change_time = src_obj->change_time;
	tar_obj->deleted_time = src_obj->deleted_time;

	tar_obj->owner_id = src_obj->owner_id;
	tar_obj->group_id = src_obj->group_id;

	tar_obj->meta_data_list = edv_properties_list_delete(tar_obj->meta_data_list);
	tar_obj->meta_data_list = edv_properties_list_copy(src_obj->meta_data_list);
}

/*
 *	Deletes and zeros all the values on the EDVRecycledObject.
 *
 *	The obj specifies the EDVRecycledObject who's values are to be
 *	cleared.
 */
void edv_recycled_object_clear(EDVRecycledObject *obj)
{
	if(obj == NULL)
	    return;

	g_free(obj->name);
	g_free(obj->original_path);
	g_free(obj->link_target);
	obj->meta_data_list = edv_properties_list_delete(obj->meta_data_list);

	(void)memset(
	    obj,
	    0x00,
	    sizeof(EDVRecycledObject)
	);
}

/*
 *	Deletes the EDVRecycledObject.
 *
 *	The obj specifies the EDVRecycledObject to delete.
 */
void edv_recycled_object_delete(EDVRecycledObject *obj)
{
	if(obj == NULL)
	    return;

	g_free(obj->name);
	g_free(obj->original_path);
	g_free(obj->link_target);
	(void)edv_properties_list_delete(obj->meta_data_list);

	g_free(obj);
}
