/*
	     Endeavour Mark II API - Recycle Bin Index File IO

	Low-level Recycle Bin access and manipulation.

	Includes low-level delete, purge, and recover of objects.

	Most functions take a gchar *index_file argument, the default
	value is obtained with:

	gchar *index_file = edv_get_s(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX)

	Example:

	GList *indices_list = edv_recycle_bin_index_list_indicies(
		edv_get_s(EDV_CFG_PARM_FILE_RECYCLE_BIN_INDEX)
	);

	See also: edv_cfg_list.h edv_get.h edv_recycle.h
	edv_recycled_obj.h edv_recycled_obj_stat.h
 */

#ifndef EDV_RECYCLE_BIN_INDEX_H
#define EDV_RECYCLE_BIN_INDEX_H

#include <glib.h>
#include "edv_vfs_obj.h"
#include "edv_recycled_obj.h"


typedef struct _EDVRecycleBinIndex	EDVRecycleBinIndex;
#define EDV_RECBIN_INDEX(p)		((EDVRecycleBinIndex *)(p))


/*
 *	Gets the last error message.
 */
extern const gchar *edv_recycle_bin_index_get_error(void);


/*
 *	Gets the full path to the recycle bin directory from the
 *	recycled objects index file.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file which will be used to derive the location of the
 *	recycle bin directory from by using its parent as the recycle
 *	bin directory.
 *
 *	Returns a dynamically allocated string describing the path to
 *	the recycle bin directory or NULL on error.
 */
extern gchar *edv_recycle_bin_index_get_recbin_directory_path(const gchar *index_path);

/*
 *	Gets the full path to the VFS object that contains the
 *	recycled object's data.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file.
 *
 *	The index specifies the recycled object.
 *
 *	Returns a dynamically allocated string describing the path to
 *	the recycled object or NULL on error.
 */
extern gchar *edv_recycle_bin_index_get_recycled_object_path(
	const gchar *index_path,
	const gulong index
);


/*
 *	Gets the total number of recycled objects in the recycle bin.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file.
 *
 *	Returns the total number of recycled objects in the recycle
 *	bin.
 */
extern gint edv_recycle_bin_index_get_total(const gchar *index_path);

/*
 *	Gets the list of all the recycled object indices.
 *
 *	Returns a GList of gulong recycled object indices or NULL on
 *	error.
 */
extern GList *edv_recycle_bin_index_list_indicies(const gchar *index_path);


/*
 *	Opens the recycle bin index file for reading.
 *
 *	edv_recycle_bin_index_next() should be called to get the very
 *	first and subsequent recycled objects.
 *
 *	The index_path specifies the recycle bin index file.
 *
 *	Returns the recycle bin index stream or NULL if there are no
 *	recycled objects or error. The returned stream must be closed
 *	by calling edv_recycle_bin_index_close().
 */
extern EDVRecycleBinIndex *edv_recycle_bin_index_open(const gchar *index_path);

/*
 *	Seeks to the next recycled object in the recycled objects index
 *	stream,
 *
 *	The rp specifies the recycled objects index stream.
 *
 *	Returns a statically allocated EDVRecycledObject describing the
 *	next recycled object or NULL on end of file or error. The
 *	returned EDVRecycledObject must not be modified or deleted.
 */
extern EDVRecycledObject *edv_recycle_bin_index_next(EDVRecycleBinIndex *rp);

/*
 *	Seeks to a recycled object in the recycled objects index
 *	stream.
 *
 *	The rp specifies the recycled objects index stream.
 *
 *	The index specifies the index of the recycled object to seek
 *	to.
 *
 *	Returns a statically allocated EDVRecycledObject describing the
 *	next recycled object or NULL on end of file or error. The
 *	returned EDVRecycledObject must not be modified or deleted.
 */
extern EDVRecycledObject *edv_recycle_bin_index_seek(
	EDVRecycleBinIndex *rp,
	const gulong index
);

/*
 *	Gets the statistics of the recycled objects index file.
 *
 *	The rp specifies the recycled objects index stream.
 *
 *	Returns a dynamically allocated EDVVFSObject describing the
 *	recycle bin index stream's statistics or NULL on error.
 */
extern EDVVFSObject *edv_recycle_bin_index_stat(EDVRecycleBinIndex *rp);

/*
 *	Gets the current position of the recycle bin index stream
 *	in units of bytes.
 *
 *	The rp specifies the recycled objects index stream.
 *
 *	Returns the current position in bytes.
 */
extern gulong edv_recycle_bin_index_tell(EDVRecycleBinIndex *rp);

/*
 *	Checks for the end of file indicator on the the recycle bin
 *	index stream.
 *
 *	The rp specifies the recycled objects index stream.
 *
 *	Returns TRUE if the end of file indicator is set.
 */
extern gboolean edv_recycle_bin_index_eof(EDVRecycleBinIndex *rp);

/*
 *	Closes the recycle bin index stream.
 *
 *	The rp specifies the recycled objects index stream.
 */
extern void edv_recycle_bin_index_close(EDVRecycleBinIndex *rp);


/*
 *	Checks if the recycle bin is locked and if the process that 
 *	locked it is currently running.
 *
 *	The index_path specifies the recycled objects index file.
 *
 *	Returns the pid of process that locked the recycle bin or 0 if
 *	the recycle bin is unlocked.
 */
extern gint edv_recycle_bin_index_get_lock(const gchar *index_path);

/*
 *	Checks if the recycle bin is unlocked and if it is then locks
 *	the recycle bin.
 *
 *	The index_path specifies the recycled objects index file.
 *
 *	The pid specifies the process ID requesting the lock.
 *
 *	Returns 0 if the lock succeeded, non-zero on error, or -6 if
 *	the recycle bin is already locked.
 */
extern gint edv_recycle_bin_index_lock(
	const gchar *index_path,
	const gint pid
);

/*
 *	Unlocks the recycle bin.
 *
 *	The index_path specifies the recycled objects index file.
 *
 *	The pid specifies the process ID requesting the unlock.
 *
 *	Returns 0 if the unlock succeeded or if the recycle bin was
 *	already unlocked or non-zero on error.
 */
extern gint edv_recycle_bin_index_unlock(
	const gchar *index_path,
	const gint pid
);


/*
 *	Adds an entry to the recycled objects index file.
 *
 *	This function does not actually recycle or recover any object.
 *
 *	The index_path specifies the recycled objects index file.
 *
 *	The obj specifies the recycled object values that will be
 *	used for the new entry. The obj will not be modified or
 *	deleted by this call.
 *
 *	Returns the new entry's index or 0 on error.
 */
extern gulong edv_recycle_bin_index_add(
	const gchar *index_path,
	EDVRecycledObject *obj
);

/*
 *	Sets an entry in the recycle bin index file.
 *
 *	This function does not actually recycle or recover any object.
 *
 *	The index_path specifies the recycle bin index file.
 *
 *	The index specifies the entry to set.
 *
 *	The obj specifies the recycled object values (except for
 *	the index) that will be used to set the entry. The recycled
 *	object will not be modified or deleted by this call.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_recycle_bin_index_set(
	const gchar *index_path,
	const gulong index,
	EDVRecycledObject *obj
);

/*
 *	Removes an entry from the recycle bin index file.
 *
 *	The index_path specifies the recycle bin index file.
 *
 *	The index specifies the entry to remove.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_recycle_bin_index_remove(
	const gchar *index_path,
	const gulong index
);


/*
 *	Low-level recycling of a VFS object to the Recycle Bin under
 *	a specific index. This function will not create a new index,
 *	the index should be created and obtained by a prior call to
 *	edv_recycle_bin_index_add().
 *
 *	The index_path specifies the recycled objects index file, the
 *	recycle bin index file will not be updated.
 *
 *	The index specifies the index to create the recycled object
 *	under.
 *
 *	The path specifies the VFS object to recycle, the actual
 *	VFS object on the device will be deleted by this call. If the
 *	object is a directory then it must be empty.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_recycle_bin_index_recycle(
	const gchar *index_path,
	const gulong index,
	const gchar *path,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
);

/*
 *	Low-level recovering of a recycled object to the VFS.
 *
 *	The index_path specifies the recycled objects index file, the
 *	recycle bin index file will not be updated. Upon success,
 *	the calling function should call edv_recycle_bin_index_remove()
 *	to remove the entry from the index file.
 *
 *	The index specifies the recycled object to recover. The actual
 *	recycled object will be deleted upon successful recover.
 *
 *	The path specifies the full path of the recovered object. This
 *	object must not already exist or else the recovery will fail.
 *	If path is NULL then the recycled object's original location
 *	and name will be used.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_recycle_bin_index_recover(
	const gchar *index_path,
	const gulong index,
	const gchar *path,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
);

/*
 *	Low-level purging of a recycled object from the Recycle Bin.
 *
 *	The index_path specifies the recycled objects index file, the
 *	recycle bin index file will not be updated. Upon success,
 *	the calling function should call edv_recycle_bin_index_remove()
 *	to remove the entry from the index file.
 *
 *	The index specifies the recycled object to purge. The actual
 *	recycled object will be deleted upon successful purge.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_recycle_bin_index_purge(
	const gchar *index_path,
	const gulong index,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
);

/*
 *	Low-level purging of all recycled objects from the Recycle Bin.
 *
 *	This is similar to edv_recycle_bin_index_purge() except that it
 *	purges of the all recycled objects in a single and more
 *	efficient pass.
 *
 *	The index_path specifies the recycled objects index file, the
 *	recycle bin index file will not be updated. Upon success,
 *	the calling function should remove the entire index file 
 *	(since all recycled objects have been purged).
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_recycle_bin_index_purge_all(
	const gchar *index_path,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
);


#endif	/* EDV_RECYCLE_BIN_INDEX_H */
