#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <fcntl.h>				/* mknod() */
#include <sys/types.h>				/* mknod() */
#include <sys/stat.h>				/* mknod() */
#include <glib.h>
#include <unistd.h>				/* mknod() */

#include "../../include/fio.h"

#include "edv_types.h"
#include "edv_utils.h"
#include "edv_path.h"
#include "edv_link.h"
#include "edv_directory.h"
#include "edv_process.h"
#include "edv_vfs_obj.h"
#include "edv_vfs_obj_stat.h"
#include "edv_recycled_obj.h"
#include "edv_recycle_bin_index.h"

#include "config.h"


/*
 *	Return values legend:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value.
 *	-3	Systems error, out of memory, or out of disk space.
 *	-4	User responded with "Cancel".
 *	-5	User responded with "No" or response was not available.
 *	-6	An operation is already in progress.
 */


/* Error Message */
const gchar *edv_recycle_bin_index_get_error(void);
static void edv_recycle_bin_index_set_error_message(const gchar *msg);

/* Get Paths */
gchar *edv_recycle_bin_index_get_recbin_directory_path(const gchar *index_path);
gchar *edv_recycle_bin_index_get_recycled_object_path(
	const gchar *index_path,
	const gulong index
);

/* Create Recycle Bin Directory */
static gint edv_recycle_bin_index_create_recycle_bin_directory(
	const gchar *index_path
);

/* Get Index List & Total */
gint edv_recycle_bin_index_get_total(const gchar *index_path);
GList *edv_recycle_bin_index_list_indicies(const gchar *index_path);

/* Lock */
static gchar *edv_recycle_bin_index_get_lock_path(const gchar *index_path);
gint edv_recycle_bin_index_get_lock(const gchar *index_path);
gint edv_recycle_bin_index_lock(
	const gchar *index_path,
	const gint pid
);
gint edv_recycle_bin_index_unlock(
	const gchar *index_path,
	const gint pid
);

/* Index File Open, Next, and Close */
static gint edv_recycle_bin_index_read_next_entry(
	FILE *fp,
	EDVRecycledObject *obj
);
EDVRecycleBinIndex *edv_recycle_bin_index_open(const gchar *index_path);
EDVRecycledObject *edv_recycle_bin_index_next(EDVRecycleBinIndex *rp);
EDVRecycledObject *edv_recycle_bin_index_seek(
	EDVRecycleBinIndex *rp,
	const gulong index
);
EDVVFSObject *edv_recycle_bin_index_stat(EDVRecycleBinIndex *rp);
gulong edv_recycle_bin_index_tell(EDVRecycleBinIndex *rp);
gboolean edv_recycle_bin_index_eof(EDVRecycleBinIndex *rp);
void edv_recycle_bin_index_close(EDVRecycleBinIndex *rp);

/* Index Entry Adding, Setting, and Removing */
static void edv_recycle_bin_index_write_object_iterate(
	FILE *fp,
	const gulong index,
	EDVRecycledObject *obj
);
gulong edv_recycle_bin_index_add(
	const gchar *index_path,
	EDVRecycledObject *obj
);
gint edv_recycle_bin_index_set(
	const gchar *index_path,
	const gulong index,
	EDVRecycledObject *obj
);
gint edv_recycle_bin_index_remove(
	const gchar *index_path,
	const gulong index
);

/* Delete, Recover, and Purge Actual Object */
gint edv_recycle_bin_index_recycle(
	const gchar *index_path,
	const gulong index,
	const gchar *path,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
);
gint edv_recycle_bin_index_recover(
	const gchar *index_path,
	const gulong index,
	const gchar *path,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
);
gint edv_recycle_bin_index_purge(
	const gchar *index_path,
	const gulong index,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
);
gint edv_recycle_bin_index_purge_all(
	const gchar *index_path,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
);


/*
 *	Error Message Pointer & Buffer:
 */
static const gchar	*edv_recycle_bin_index_last_error_ptr = NULL;
static gchar		edv_recycle_bin_index_last_error_buf[1024];


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Recycle Bin Index Flags:
 */
typedef enum {
	EDV_RECYCLE_BIN_INDEX_EOF	= (1 << 0),
	EDV_RECYCLE_BIN_INDEX_ERROR	= (1 << 1)
} EDVRecycleBinIndexFlags;


/*
 *	Recycle Bin Index Context:
 */
struct _EDVRecycleBinIndex {

	EDVRecycleBinIndexFlags	flags;
	gchar		*path;			/* Full path to the index file */
	FILE		*fp;			/* Index file stream opened
						 * for reading */

	gulong		index;			/* Current recycled object's
						 * index */

	EDVRecycledObject	*obj;		/* Buffer containing the
						 * statistics of the current
						 * recycled object for
						 * returning to caller */

	EDVVFSObject	*stats;			/* Index file statistics */

	gulong		current_position;	/* Current position of fp
						 * in bytes */

};


/*
 *	Gets the last error message.
 */
const gchar *edv_recycle_bin_index_get_error(void)
{
	return(edv_recycle_bin_index_last_error_ptr);
}

/*
 *	Sets or clears the error message.
 *
 *	The msg specifies the error message. If msg is NULL then the
 *	error message is cleared.
 */
static void edv_recycle_bin_index_set_error_message(const gchar *msg)
{
	if(msg != NULL)
	{
		const gint n = (gint)sizeof(edv_recycle_bin_index_last_error_buf);
		edv_recycle_bin_index_last_error_ptr = (const gchar *)strncpy(
			(char *)edv_recycle_bin_index_last_error_buf,
			(const char *)msg,
			(size_t)n
		);
		edv_recycle_bin_index_last_error_buf[n - 1] = '\0';
	}
	else
	{
		*edv_recycle_bin_index_last_error_buf = '\0';
		edv_recycle_bin_index_last_error_ptr = NULL;
	}
}


/*
 *	Gets the full path to the recycle bin directory from the
 *	recycled objects index file.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file which will be used to derive the location of the
 *	recycle bin directory from by using its parent as the recycle
 *	bin directory.
 *
 *	Returns a dynamically allocated string describing the path to
 *	the recycle bin directory or NULL on error.
 */
gchar *edv_recycle_bin_index_get_recbin_directory_path(const gchar *index_path)
{
	if(STRISEMPTY(index_path))
	{
		edv_recycle_bin_index_last_error_ptr =
"Recycled objects index file was not specified";
		errno = EINVAL;
		return(NULL);
	}

	return(g_dirname(index_path));
}

/*
 *	Gets the full path to the VFS object that contains the
 *	recycled object's data.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file.
 *
 *	The index specifies the recycled object.
 *
 *	Returns a dynamically allocated string describing the path to
 *	the recycled object or NULL on error.
 */
gchar *edv_recycle_bin_index_get_recycled_object_path(
	const gchar *index_path,
	const gulong index
)
{
	gchar		*path,
			*recbin_path;

	if(index == 0l)
	{
		edv_recycle_bin_index_last_error_ptr =
"Invalid recycled object index";
		errno = EINVAL;
		return(NULL);
	}

	/* Get the full path to the recycle bin from the index file path */
	recbin_path = edv_recycle_bin_index_get_recbin_directory_path(index_path);
	if(recbin_path == NULL)
		return(NULL);

	/* Format the full path to the VFS object containing the
	 * recycled object's data
	 */
	path = g_strdup_printf(
		"%s%c%ld",
		recbin_path,
		G_DIR_SEPARATOR,
		index
	);

	g_free(recbin_path);

	return(path);
}


/*
 *	Creates the recycle bin directory as needed.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file.
 *
 *	Returns 0 on success or if the recycle bin directory already
 *	exists or non-zero on error.
 */
static gint edv_recycle_bin_index_create_recycle_bin_directory(
	const gchar *index_path
)
{
	gchar *recbin_path;

	if(STRISEMPTY(index_path))
	{
		edv_recycle_bin_index_last_error_ptr =
"Recycled objects index file was not specified";
		errno = EINVAL;
		return(-2);
	}

	/* Get the full path to the recycle bin from the index file path */
	recbin_path = edv_recycle_bin_index_get_recbin_directory_path(index_path);
	if(recbin_path == NULL)
		return(-2);

	/* Create the recycle bin directory as needed */
	if(edv_directory_create(
		recbin_path,
		TRUE,				/* Create parents */
		NULL				/* No new paths list return */
	))
	{
		const gint error_code = (gint)errno;
		edv_recycle_bin_index_last_error_ptr =
"Unable to create the recycle bin directory";
		g_free(recbin_path);
		errno = (int)error_code;
		return(-1);
	}

	/* Set the permissions so that only the user may read, write
	 * and list it
	 */
	(void)edv_permissions_set(
		recbin_path,
		EDV_PERMISSION_UR | EDV_PERMISSION_UW | EDV_PERMISSION_UX
	);

	g_free(recbin_path);

	return(0);
}


/*
 *	Gets the total number of recycled objects in the recycle bin.
 *
 *	The index_path specifies the full path to the recycled objects
 *	index file.
 *
 *	Returns the total number of recycled objects in the recycle
 *	bin.
 */
gint edv_recycle_bin_index_get_total(const gchar *index_path)
{
	gulong total;
	gchar *recbin_path;
	EDVDirectory *dp;

	edv_recycle_bin_index_set_error_message(NULL);

	if(STRISEMPTY(index_path))
	{
		edv_recycle_bin_index_last_error_ptr =
"Recycled objects index file was not specified";
		errno = EINVAL;
		return(0);
	}

	/* Get the full path to the recycle bin from the index file path */
	recbin_path = edv_recycle_bin_index_get_recbin_directory_path(index_path);
	if(recbin_path == NULL)
		return(0);

	/* Get the total number of objects in the recycle bin directory
	 * by counting the number of objects in the recycle bin
	 */
	total = 0l;
	dp = edv_directory_open(
		recbin_path,
		FALSE,				/* Unsorted */
		FALSE				/* Exclude notations */
	);
	if(dp != NULL)
	{
		const gchar *name;

		/* Count all the objects in the recycle bin directory */
		for(name = edv_directory_next(dp);
		    name != NULL;
		    name = edv_directory_next(dp)
		)
		{
			/* Count only objects with numeric names,
			 * this should allow us to skip the recycled
			 * objects index file and the
			 * EDV_NAME_DIRECTORY_PROPERTIES_FILE file
			 */
			if(isdigit((int)*name))
				total++;
		}

		edv_directory_close(dp);
	}

	g_free(recbin_path);

	return((gint)total);
}

/*
 *	Gets the list of all the recycled object indices.
 *
 *	Returns a GList of gulong recycled object indices or NULL on
 *	error.
 */
GList *edv_recycle_bin_index_list_indicies(const gchar *index_path)
{
	FILE *fp;
	gint vi[1];
	GList *indices_list;

	edv_recycle_bin_index_set_error_message(NULL);

	if(STRISEMPTY(index_path))
	{
		edv_recycle_bin_index_last_error_ptr =
"Recycled objects index file was not specified";
		errno = EINVAL;
		return(NULL);
	}

	/* Open the recycled objects index file for reading */
	fp = fopen((const char *)index_path, "rb");
	if(fp == NULL)
	{
		const gint error_code = (gint)errno;
		switch(error_code)
		{
		    case ENOENT:
			/* If the index file does not exist then it
			 * implies there are no recycled objects and
			 * that this is not an error
			 */
			errno = 0;
			break;
		    default:
			edv_recycle_bin_index_last_error_ptr =
"Unable to open the recycled objects index file for writing";
			break;
		}
		errno = (int)error_code;
		return(NULL);
	}

	/* Begin reading the recycled objects index file */
	indices_list = NULL;
	while(!FSeekToParm(
		fp,
		"BeginRecycledObject",
		EDV_CFG_COMMENT_CHAR,
		EDV_CFG_DELIMINATOR_CHAR
	))
	{
		/* Get the index of this recycled object from the
		 * value of the start block and seek to the next
		 * parameter
		 */
		FGetValuesI(fp, vi, 1);

		/* Add this index to the indicies list */
		indices_list = g_list_append(
			indices_list,
			(gpointer)vi[0]		/* gulong */
		);

		/* Seek to the end of this recycled object block */
		if(FSeekToParm(
			fp,
			"EndRecycledObject",
			EDV_CFG_COMMENT_CHAR,
			EDV_CFG_DELIMINATOR_CHAR
		))
		{
			/* No corresponding end block, this block may
			 * be incomplete/corrupt
			 */
			break;
		}
		else
		{
			/* Skip the value and seek to the start of the
			 * next parameter
			 */
			FSeekNextLine(fp);
		}
	}

	/* Close the recycled objects index file */
	(void)fclose(fp);

	return(indices_list);
}


/*
 *	Reads the next recycled object entry in the recycle bin index
 *	stream.
 *
 *	The fp specifies the stream. It will be repositioned at the
 *	start of the line following the entry that was read by this
 *	call.
 *
 *	The obj specifies the buffer to store the values of the
 *	next entry to.
 *
 *	Returns 0 on success or non-zero on error or end of file.
 */
static gint edv_recycle_bin_index_read_next_entry(
	FILE *fp,
	EDVRecycledObject *obj
)
{
	gint status = -1;
	gchar *parm = NULL;

	/* Clear the existing values */
	edv_recycled_object_clear(obj);

	/* Begin reading the recycle bin index file and store the values
	 * on to the EDVRecycledObject
	 */
	while(!feof(fp))
	{
		/* Read the next parameter */
		parm = (gchar *)FSeekNextParm(
			fp,
			(char *)parm,
			EDV_CFG_COMMENT_CHAR,
			EDV_CFG_DELIMINATOR_CHAR
		);
		if(parm == NULL)
			break;

		/* Begin handling by parameter */

		/* Name */
		else if(!g_strcasecmp(parm, "Name"))
		{
			g_free(obj->name);
			obj->name = (gchar *)FGetString(fp);
		}
		/* OriginalPath */
		else if(!g_strcasecmp(parm, "OriginalPath"))
		{
			g_free(obj->original_path);
			obj->original_path = (gchar *)FGetString(fp);
		}
		/* DateDeleted */
		else if(!g_strcasecmp(parm, "DateDeleted"))
		{
			long vl[1];
			FGetValuesL(fp, vl, 1);
			obj->deleted_time = (gulong)vl[0];
		}
		/* Type */
		else if(!g_strcasecmp(parm, "Type"))
		{
			gint vi[1];
			FGetValuesI(fp, (int *)vi, 1);
			obj->type = (EDVObjectType)vi[0];
		}
		/* LinkedTo */
		else if(!g_strcasecmp(parm, "LinkedTo"))
		{
		    /* This link target is a simplified version and is
			 * not to be considered the actual (unmodified) link
			 * target value (which is obtained from the
			 * recycled file of the link)
			 */
			g_free(obj->link_target);
			obj->link_target = (gchar *)FGetString(fp);
		}
		/* Permissions */
		else if(!g_strcasecmp(parm, "Permissions"))
		{
			gint vi[1];
			FGetValuesI(fp, (int *)vi, 1);
			obj->permissions = (EDVPermissionFlags)vi[0];
		}
		/* AccessTime */
		else if(!g_strcasecmp(parm, "AccessTime"))
		{
			long vl[1];
			FGetValuesL(fp, vl, 1);
			obj->access_time = (gulong)vl[0];
		}
		/* ModifyTime */
		else if(!g_strcasecmp(parm, "ModifyTime"))
		{
			long vl[1];
			FGetValuesL(fp, vl, 1);
			obj->modify_time = (gulong)vl[0];
		}
		/* ChangeTime */
		else if(!g_strcasecmp(parm, "ChangeTime"))
		{
			long vl[1];
			FGetValuesL(fp, vl, 1);
			obj->change_time = (gulong)vl[0];
		}
		/* OwnerID */
		else if(!g_strcasecmp(parm, "OwnerID"))
		{
			gint vi[1];
			FGetValuesI(fp, (int *)vi, 1);
			obj->owner_id = vi[0];
		}
		/* GroupID */
		else if(!g_strcasecmp(parm, "GroupID"))
		{
			gint vi[1];
			FGetValuesI(fp, (int *)vi, 1);
			obj->group_id = vi[0];
		}
		/* Size */
		else if(!g_strcasecmp(parm, "Size"))
		{
			long vl[1];
			FGetValuesL(fp, vl, 1);
			obj->size = (gulong)vl[0];
		}

		/* EndRecycledObject */
		else if(!g_strcasecmp(parm, "EndRecycledObject"))
		{
			/* Skip this value and seek to the next line and
			 * then stop reading
			 */
			FSeekNextLine(fp);
			status = 0;
			break;
		}
		else
		{
			/* Unsupported parameter (ignore this) */
			FSeekNextLine(fp);
		}
	}

	/* Delete the parameter */
	g_free(parm);

	return(status);
}

/*
 *	Opens the recycle bin index file for reading.
 *
 *	edv_recycle_bin_index_next() should be called to get the very
 *	first and subsequent recycled objects.
 *
 *	The index_path specifies the recycle bin index file.
 *
 *	Returns the recycle bin index stream or NULL if there are no
 *	recycled objects or error. The returned stream must be closed
 *	by calling edv_recycle_bin_index_close().
 */
EDVRecycleBinIndex *edv_recycle_bin_index_open(const gchar *index_path)
{
	FILE *fp;
	EDVVFSObject *stats;
	EDVRecycleBinIndex *rp;

	edv_recycle_bin_index_set_error_message(NULL);

	if(STRISEMPTY(index_path))
	{
		edv_recycle_bin_index_last_error_ptr =
"Recycled objects index file was not specified";
		errno = EINVAL;
		return(NULL);
	}

	/* Open the recycled objects index file for reading */
	fp = fopen((const char *)index_path, "rb");
	if(fp == NULL)
	{
		const gint error_code = (gint)errno;
		switch(error_code)
		{
		  case ENOENT:
			return(NULL);
			break;
		  default:
			edv_recycle_bin_index_last_error_ptr =
"Unable to open the recycled objects index file for reading";
			return(NULL);
			break;
		}
	}

	/* Get the recycled objects index file's statistics */
	stats = edv_vfs_object_fstat((gint)fileno(fp));
	if(stats == NULL)
	{
		const gint error_code = (gint)errno;
		(void)fclose(fp);
		edv_recycle_bin_index_last_error_ptr =
"Unable to get the recycled objects index file's statistics";
		errno = (int)error_code;
		return(NULL);
	}

	/* Create a new recycle bin index context */
	rp = EDV_RECBIN_INDEX(g_malloc(sizeof(EDVRecycleBinIndex)));
	if(rp == NULL)
	{
		const gint error_code = (gint)errno;
		edv_vfs_object_delete(stats);
		(void)fclose(fp);
		edv_recycle_bin_index_last_error_ptr =
"Memory allocation error";
		errno = (int)error_code;
		return(NULL);
	}

	/* Initialize all the values */
	rp->path = g_strdup(index_path);
	rp->fp = fp;
/*	rp->index = 0l; */
	rp->obj = edv_recycled_object_new();
	rp->stats = stats;
/*	rp->current_position = 0l; */

	return(rp);
}

/*
 *	Seeks to the next recycled object in the recycled objects index
 *	stream,
 *
 *	The rp specifies the recycled objects index stream.
 *
 *	Returns a statically allocated EDVRecycledObject describing the
 *	next recycled object or NULL on end of file or error.
 *	The returned EDVRecycledObject must not be modified or deleted.
 */
EDVRecycledObject *edv_recycle_bin_index_next(EDVRecycleBinIndex *rp)
{
	FILE *fp;
	gchar *parm;
	EDVRecycledObject *obj;

	edv_recycle_bin_index_set_error_message(NULL);
	errno = 0;

	/* If the stream is NULL then consider it as the end of file */
	if(rp == NULL)
	{
		errno = ENODATA;
		return(NULL);
	}

	/* Get the values from the recycle bin index context */
	fp = rp->fp;
	if(fp == NULL)
	{
		edv_recycle_bin_index_last_error_ptr =
"Recycle bin index stream was not properly initialized";
		errno = EBADF;
		return(NULL);
	}

	if(rp->obj == NULL)
	{
		edv_recycle_bin_index_last_error_ptr =
"Memory allocation error";
		errno = ENOMEM;
		return(NULL);
	}

	/* Begin reading the recycle bin index file */
	parm = NULL;
	obj = NULL;
	while(!feof(fp))
	{
		/* Read the next parameter */
		parm = (gchar *)FSeekNextParm(
			fp,
			(char *)parm,
			EDV_CFG_COMMENT_CHAR,
			EDV_CFG_DELIMINATOR_CHAR
		);
		if(parm == NULL)
			break;

		/* Begin handling by the parameter */

		/* Start of a recycled object block? */
		if(!g_strcasecmp(parm, "BeginRecycledObject"))
		{
			long vl[1];
			gchar *recycled_object_path;

			FGetValuesL(fp, vl, 1);

			/* Update the current index */
			rp->index = (gulong)vl[0];

			/* Get the object buffer and mark that we have
			 * obtained the recycled object that we are
			 * looking for
			 */
			obj = rp->obj;

			/* Read all subsequent parameters from the current
			 * file position, store them on the recycled object,
			 * and reposition the stream after this entry
			 */
			(void)edv_recycle_bin_index_read_next_entry(
				fp,
				obj
			);

			/* Set the index and storage_size since they are
			 * not set by
			 * edv_recycle_bin_index_read_next_entry()
			 */
			obj->index = rp->index;
			recycled_object_path = edv_recycle_bin_index_get_recycled_object_path(
				rp->path,
				rp->index
			);
			if(recycled_object_path != NULL)
			{
				EDVVFSObject *obj_lvalues = edv_vfs_object_lstat(recycled_object_path);
				if(obj_lvalues != NULL)
				{
					obj->storage_size = obj_lvalues->size;
					edv_vfs_object_delete(obj_lvalues);
				}
			}

			/* Stop reading once this recycled object has
			 * been obtained
			 */
			break;
		}
		else
		{
			/* Other parameter, skip */
			FSeekNextLine(fp);
		}
	}

	/* Delete the parameter */
	g_free(parm);

	/* Update the current position */
	rp->current_position = (gulong)ftell(fp);

	return(obj);
}

/*
 *	Seeks to a recycled object in the recycled objects index 
 *	stream.
 *
 *	The rp specifies the recycled objects index stream.
 *
 *	The index specifies the index of the recycled object to seek
 *	to.
 *
 *	Returns a statically allocated EDVRecycledObject describing the
 *	next recycled object or NULL on end of file or error. The
 *	returned EDVRecycledObject must not be modified or deleted.
 */
EDVRecycledObject *edv_recycle_bin_index_seek(
	EDVRecycleBinIndex *rp,
	const gulong index
)
{
	FILE *fp;
	gchar *parm;
	gulong last_pos;
	EDVRecycledObject *obj;

	edv_recycle_bin_index_set_error_message(NULL);
	errno = 0;

	/* If the stream is NULL then consider it as the end of file */
	if(rp == NULL)
	{
		errno = ENODATA;
		return(NULL);
	}

	/* Get the values from the recycle bin index context */
	fp = rp->fp;
	if(fp == NULL)
	{
		edv_recycle_bin_index_last_error_ptr =
"Recycle bin index stream was not properly initialized";
		errno = EBADF;
		return(NULL);
	}

	if(rp->obj == NULL)
	{
		edv_recycle_bin_index_last_error_ptr =
"Memory allocation error";
		errno = ENOMEM;
		return(NULL);
	}

	/* Record the current position of the stream for resetting of
	 * the stream position in case we did not find the recycled
	 * object
	 */
	last_pos = (gulong)ftell(fp);

	/* Must rewind the stream to the beginning and clear any errors
	 * in order to have a through search
	 */
	rewind(fp);

	/* Reading the recycle bin index file to searching for the
	 * specified recycled object index
	 */
	parm = NULL;
	obj = NULL;
	while(!feof(fp))
	{
		/* Read the next parameter */
		parm = (gchar *)FSeekNextParm(
			fp,
			(char *)parm,
			EDV_CFG_COMMENT_CHAR,
			EDV_CFG_DELIMINATOR_CHAR
		);
		if(parm == NULL)
			break;

		/* Begin handling by the parameter */

		/* Start of a recycled object block? */
		if(!g_strcasecmp(parm, "BeginRecycledObject"))
		{
			long vl[1];
			gchar *recycled_object_path;

			FGetValuesL(fp, vl, 1);

			/* Not the specified recycled object? */
			if((gulong)vl[0] != index)
				continue;

			/* Update the current index */
			rp->index = (gulong)vl[0];

			/* Get the object buffer and mark that we have
			 * obtained the recycled object that we are
			 * looking for
			 */
			obj = rp->obj;

			/* Read all subsequent parameters from the current
			 * file position, store them on the recycled object,
			 * and reposition the stream after this entry
			 */
			(void)edv_recycle_bin_index_read_next_entry(
				fp,
				obj
			);

			/* Set the index and storage_size since they are
			 * not set by
			 * edv_recycle_bin_index_read_next_entry()
			 */
			obj->index = rp->index;
			recycled_object_path = edv_recycle_bin_index_get_recycled_object_path(
				rp->path,
				rp->index
			);
			if(recycled_object_path != NULL)
			{
				EDVVFSObject *obj_lvalues = edv_vfs_object_lstat(recycled_object_path);
				if(obj_lvalues != NULL)
				{
					obj->storage_size = obj_lvalues->size;
					edv_vfs_object_delete(obj_lvalues);
				}
			}

			/* Stop reading once we have seeked to the
			 * specified recycled object
			 */
			break;
		}
		else
		{
			/* Other parameter, skip */
			FSeekNextLine(fp);
		}
	}

	/* Delete the parameter */
	g_free(parm);

	/* Matched the recycled object? */
	if(obj != NULL)
	{
		/* Update the curent position */
		rp->current_position = (gulong)ftell(fp);
		return(obj);
	}
	else
	{
		/* Unable to find the recycled object that matches the
		 * specified index
		 *
		 * Seek back to the original position and return NULL
		 */
		if(fseek(
			fp,
			(long)last_pos,
			SEEK_SET
		) == 0)
			rp->current_position = last_pos;
		errno = ENOENT;
		return(NULL);
	}
}

/*
 *	Gets the statistics of the recycled objects index file.
 *
 *	The rp specifies the recycled objects index stream.
 *
 *	Returns a dynamically allocated EDVVFSObject describing the
 *	recycle bin index stream's statistics or NULL on error.
 */
EDVVFSObject *edv_recycle_bin_index_stat(EDVRecycleBinIndex *rp)
{
	if(rp == NULL)
	{
		errno = ENODATA;
		return(NULL);
	}

	return(edv_vfs_object_copy(rp->stats));
}

/*
 *	Gets the current position of the recycle bin index stream
 *	in units of bytes.
 *
 *	The rp specifies the recycled objects index stream.
 *
 *	Returns the current position in bytes.
 */
gulong edv_recycle_bin_index_tell(EDVRecycleBinIndex *rp)
{
	if(rp == NULL)
	{
		errno = EBADF;
		return(0l);
	}

	errno = 0;

	return(rp->current_position);
}

/*
 *	Checks for the end of file indicator on the the recycle bin
 *	index stream.
 *
 *	The rp specifies the recycled objects index stream.
 *
 *	Returns TRUE if the end of file indicator is set.
 */
gboolean edv_recycle_bin_index_eof(EDVRecycleBinIndex *rp)
{
	if(rp == NULL)
	{
		errno = ENODATA;
		return(FALSE);
	}

	if(rp->fp == NULL)
	{
		errno = EBADF;
		return(FALSE);
	}

	return((gboolean)feof(rp->fp));
}

/*
 *	Closes the recycle bin index stream.
 *
 *	The rp specifies the recycled objects index stream.
 */
void edv_recycle_bin_index_close(EDVRecycleBinIndex *rp)
{
	edv_recycle_bin_index_set_error_message(NULL);

	if(rp == NULL)
	{
		/* No recycle bin index context, this may indicate that
		 * there are no recycled objects (do not treat this as an
		 * error)
		 */
		return;
	}

	/* Delete the recycled object buffer */
	edv_recycled_object_delete(rp->obj);

	/* Delete the index file statistics */
	edv_vfs_object_delete(rp->stats);

	/* Close the recycle bin index file */
	if(rp->fp != NULL)
		(void)fclose(rp->fp);

	/* Delete the index file's path */
	g_free(rp->path);

	g_free(rp);
}


/*
 *	Gets the lock link path.
 */
static gchar *edv_recycle_bin_index_get_lock_path(const gchar *index_path)
{
	gchar		*lock_path,
			*parent_path,
			*recycle_bin_path = edv_recycle_bin_index_get_recbin_directory_path(index_path);
	if(recycle_bin_path == NULL)
		return(NULL);

	parent_path = g_dirname(recycle_bin_path);

	g_free(recycle_bin_path);

	if(parent_path == NULL)
		return(NULL);

	lock_path = edv_paths_join(
		parent_path,
		EDV_NAME_DEF_RECYCLE_BIN_LOCK_LINK
	);

	g_free(parent_path);

	return(lock_path);
}


/*
 *	Checks if the recycle bin is locked and if the process that
 *	locked it is currently running.
 *
 *	The index_path specifies the recycled objects index file.
 *
 *	Returns the pid of process that locked the recycle bin or 0 if
 *	the recycle bin is unlocked.
 */
gint edv_recycle_bin_index_get_lock(const gchar *index_path)
{
	gint		error_code,
			pid;
	gchar		*target,
			*lock_path = edv_recycle_bin_index_get_lock_path(index_path);
	if(lock_path == NULL)
		return(0);

	/* Get the lock link's value */
	target = edv_link_get_target(lock_path);

	error_code = (gint)errno;

	g_free(lock_path);

	/* Lock link does not exist or error reading it? */
	if(target == NULL)
	{
		errno = (int)error_code;
		return(0);
	}

	/* Get pid from the link's target and check if it is
	 * actually running
	 */
	pid = (gint)atoi((const char *)target);

	g_free(target);

	if(edv_pid_exists(pid))
		return(pid);
	else
		return(0);
}

/*
 *	Checks if the recycle bin is unlocked and if it is then locks 
 *	the recycle bin.
 *
 *	The index_path specifies the recycled objects index file.
 *
 *	The pid specifies the process ID requesting the lock.
 *
 *	Returns 0 if the lock succeeded, non-zero on error, or -6 if 
 *	the recycle bin is already locked. 
 */
gint edv_recycle_bin_index_lock(
	const gchar *index_path,
	const gint pid
)
{
	gchar		*target,
			*lock_path = edv_recycle_bin_index_get_lock_path(index_path);
	if(lock_path == NULL)
	    return(-2);

	/* Check if the recycle bin is already locked
	 *
	 * Get the lock link's value
	 */
	target = edv_link_get_target(lock_path);
	if(target != NULL)
	{
		/* Get pid from the link's target and check if it is
		 * actually running
	         */
		const gint pid = (gint)atoi((const char *)target);
	        if(edv_pid_exists(pid))
		{
			g_free(target);
			g_free(lock_path);
			errno = EBUSY;
			return(-6);
		}

		g_free(target);
	}

	/* Format the link's target with the specified pid */
	target = g_strdup_printf(
		"%i",
		pid
	);
	if(target == NULL)
	{
		const gint error_code = (gint)errno;
		g_free(lock_path);
		errno = (int)error_code;
		return(-3);
	}

	/* (Re)create the lock link */
	if(edv_unlink(lock_path))
	{
		const gint error_code = (gint)errno;
		if(error_code != ENOENT)
		{
			/* The object exists but we were unable to
			 * remove it
			 */
			g_free(target);
			g_free(lock_path);
			errno = (int)error_code;
			return(-1);
		}
	}
	if(edv_link_create(
		lock_path,			/* Path */
		target				/* Target */
	))
	{
		const gint error_code = (gint)errno;
		g_free(target);
		g_free(lock_path);
		errno = (int)error_code;
		return(-1);
	}

	g_free(target);
	g_free(lock_path);

	return(0);
}

/*
 *	Unlocks the recycle bin.
 *
 *	The index_path specifies the recycled objects index file.
 *
 *	The pid specifies the process ID requesting the unlock.
 *
 *	Returns 0 if the unlock succeeded or if the recycle bin was
 *	already unlocked or non-zero on error.
 */
gint edv_recycle_bin_index_unlock(
	const gchar *index_path,
	const gint pid
)
{
	gint		status,
			error_code;
	gchar *lock_path = edv_recycle_bin_index_get_lock_path(index_path);
	if(lock_path == NULL)
		return(-2);

	status = edv_unlink(lock_path);

	error_code = (gint)errno;

	g_free(lock_path);

	if(status != 0)
	{
		if(error_code == ENOENT)
			return(0);
	}

	errno = (gint)error_code;

	return(status);
}


/*
 *	Writes the recycled object to the index file.
 *
 *	The fp specifies the opened file pointer to the recycle bin
 *	index file at the position in the file where the recycled
 *	object's information should be written to.
 *
 *	The index specifies the recycled object's index which will
 *	be written to the index file.
 *
 *	The obj specifies the recycled object who's information
 *	(except for its index) will be written to the index file.
 *
 *	This function is used by edv_recycle_bin_index_add() and
 *	edv_recycle_bin_index_remove().
 */
static void edv_recycle_bin_index_write_object_iterate(
	FILE *fp,
	const gulong index,
	EDVRecycledObject *obj
)
{
	/* BeginRecycledObject */
	(void)fprintf(
		fp,
		"BeginRecycledObject = %ld\n",
		(unsigned long)index
	);
	/* Name */
	if(!STRISEMPTY(obj->name))
		(void)fprintf(
			fp,
			"\tName = %s\n",
			(const char *)obj->name
		);
	/* OriginalPath */
	if(!STRISEMPTY(obj->original_path))
		(void)fprintf(
			fp,
			"\tOriginalPath = %s\n",
			(const char *)obj->original_path
		);
	/* DateDeleted */
	(void)fprintf(
		fp,
		"\tDateDeleted = %ld\n",
		(unsigned long)obj->deleted_time
	);
	/* Type */
	(void)fprintf(
		fp,
		"\tType = %i\n",
		(int)obj->type
	);
	/* LinkedTo */
	if(!STRISEMPTY(obj->link_target))
	{
		/* Make a copy of the link target and remove any non
		 * printable characters from it
		 *
		 * This link target is a simplified version and is not
		 * to be considered the actual (unmodified) link target
		 * value (which is obtained from the recycled file of the
		 * link)
		 */
		gchar	*link_target = g_strdup(obj->link_target),
					*s;
		for(s = link_target; *s != '\0'; s++)
		{
			if(!isprint((char)*s))
				*s = ' ';
		}
		(void)fprintf(
			fp,
			"\tLinkedTo = %s\n",
			(const char *)link_target
		);
		g_free(link_target);
	}
	/* Permissions */
	(void)fprintf(
		fp,
		"\tPermissions = %i\n",
		(unsigned int)obj->permissions
	);
	/* AccessTime */
	(void)fprintf(
		fp,
		"\tAccessTime = %ld\n",
		(unsigned long)obj->access_time
	);
	/* ModifyTime */
	(void)fprintf(
		fp,
		"\tModifyTime = %ld\n",
		(unsigned long)obj->modify_time
	);
	/* ChangeTime */
	(void)fprintf(
		fp,
		"\tChangeTime = %ld\n",
		(unsigned long)obj->change_time
	);
	/* OwnerID */
	(void)fprintf(
		fp,
		"\tOwnerID = %i\n",
		(int)obj->owner_id
	);
	/* GroupIO */
	(void)fprintf(
		fp,
		"\tGroupID = %i\n",
		(int)obj->group_id
	);
	/* Size */
	(void)fprintf(
		fp,
		"\tSize = %ld\n",
		(unsigned long)obj->size
	);

	/* EndRecycledObject */
	(void)fprintf(
		fp,
		"EndRecycledObject\n"
	);
}

/*
 *	Adds an entry to the recycled objects index file.
 *
 *	This function does not actually recycle or recover any object.
 *
 *	The index_path specifies the recycled objects index file.
 *
 *	The obj specifies the recycled object values that will be
 *	used for the new entry. The obj will not be modified or
 *	deleted by this call.
 *
 *	Returns the new entry's index or 0 on error.
 */
gulong edv_recycle_bin_index_add(
	const gchar *index_path,
	EDVRecycledObject *obj
)
{
	FILE *fp;
	gulong		cur_index,
			new_index = 0l;
	GList		*glist,
			*indices_list;

	edv_recycle_bin_index_set_error_message(NULL);

	if(STRISEMPTY(index_path))
	{
		edv_recycle_bin_index_last_error_ptr =
"Recycled objects index file was not specified";
		errno = EINVAL;
		return(0l);
	}

	/* Get the list of indices in the recycled objects index file */
	indices_list = edv_recycle_bin_index_list_indicies(index_path);

	/* Iterate through the index list, looking for an index that
	 * is not in the list so that we can use it
	 *
	 * The new_index is initially set to 1, if an available index
	 * is found then new_index will be set to that index value
	 */
	for(cur_index = 1l; cur_index != 0l; cur_index++)
	{
		/* Iterate through index list, checking if cur_index
		 * matches any index in the list and breaking the loop
		 * prematurly if there is a match
		 */
		for(glist = indices_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			if(cur_index == (gulong)glist->data)
				break;
		}
		/* Was the cur_index not in the indices list? */
		if(glist == NULL)
		{
			new_index = cur_index;
			break;
		}
	}

	/* Delete the indices list */
	g_list_free(indices_list);

	/* No more indices available? */
	if(new_index == 0l)
	{
		edv_recycle_bin_index_last_error_ptr =
"Unable to generate a new index value for the recycled object";
		errno = ENOSPC;
		return(0l);
	}

	/* Create the recycle bin directory as needed */
	if(edv_recycle_bin_index_create_recycle_bin_directory(index_path))
		return(0l);

	/* Add a new entry to the recycled objects index file
	 *
	 * Open the recycle bin index file for write append
	 */
	fp = fopen((const char *)index_path, "ab");
	if(fp != NULL)
	{
		const gint fd = (gint)fileno(fp);

		/* Set the recycled objects index file's permissions so
		 * that only the owner can read and write to it
		 */
		if(edv_permissions_set_fd(
			fd,
			EDV_PERMISSION_UR | EDV_PERMISSION_UW
		))
		{
/* What if the recycle bin is not on a filesystem that supports
 * permissions? Should we just ignore this error?
 */
			const gint error_code = (gint)errno;
			edv_recycle_bin_index_last_error_ptr =
"Unable to set the recycled objects index file's permissions";
			(void)fclose(fp);
			errno = (int)error_code;
			return(0);
		}

		/* Write the recycled object to the recycled objects
		 * index file
		 */
		if(obj != NULL)
			edv_recycle_bin_index_write_object_iterate(
				fp,
				new_index,
				obj
			);

		/* Close the recycle bin index file */
		if(fclose(fp))
		{
			const gint error_code = (gint)errno;
			edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled objects index file";
			errno = (int)error_code;
		}
	}
	else
	{
		const gint error_code = (gint)errno;
		edv_recycle_bin_index_last_error_ptr =
"Unable to open the recycled objects index file for writing";
		errno = (int)error_code;
		return(0l);
	}

	return(new_index);
}

/*
 *	Sets an entry in the recycle bin index file.
 *
 *	This function does not actually recycle or recover any object.
 *
 *	The index_path specifies the recycle bin index file.
 *
 *	The index specifies the entry to set.
 *
 *	The obj specifies the recycled object values that will be
 *	used for the new entry. The obj will not be modified or
 *	deleted by this call.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_recycle_bin_index_set(
	const gchar *index_path,
	const gulong index,
	EDVRecycledObject *obj
)
{
	FILE *out_fp;
	gint		out_fd,
			entries_set;
	gchar		*parent_path,
			*in_path,
			*out_path;
	EDVRecycleBinIndex *rp;

	edv_recycle_bin_index_set_error_message(NULL);

	if(STRISEMPTY(index_path))
	{
		edv_recycle_bin_index_last_error_ptr =
"Recycled objects index file was not specified";
		errno = EINVAL;
		return(-2);
	}

	/* Create the recycle bin directory as needed */
	if(edv_recycle_bin_index_create_recycle_bin_directory(index_path))
		return(-1);

	/* Format the paths to the input and output index files */
	parent_path = g_dirname(index_path);
	if(parent_path == NULL)
	{
		edv_recycle_bin_index_last_error_ptr =
"Unable to generate the paths to the input and output index files";
		return(-1);
	}

	in_path = g_strdup(index_path);
	out_path = edv_tmp_name(parent_path);
	g_free(parent_path);
	if((in_path == NULL) || (out_path == NULL))
	{
		const gint error_code = (gint)errno;
		g_free(in_path);
		g_free(out_path);
		edv_recycle_bin_index_last_error_ptr =
"Unable to generate the paths to the input and output index files";
		errno = (int)error_code;
		return(-1);
	}

	/* Create/open the output file for writing */
	out_fp = fopen((const char *)out_path, "wb");
	if(out_fp == NULL)
	{
		const gint error_code = (gint)errno;
		g_free(in_path);
		g_free(out_path);
		edv_recycle_bin_index_last_error_ptr =
"Unable to open the output recycled objects index file for writing";
		errno = (int)error_code;
		return(-1);
	}

	out_fd = (gint)fileno(out_fp);

	/* Set the recycled objects index file's permissions so that
	 * only the owner can read and write to it
	 */
	if(edv_permissions_set_fd(
		out_fd,
		EDV_PERMISSION_UR | EDV_PERMISSION_UW
	))
	{
		const gint error_code = (gint)errno;
		g_free(in_path);
		(void)fclose(out_fp);
		(void)edv_unlink(out_path);
		g_free(out_path);
		edv_recycle_bin_index_last_error_ptr =
"Unable to set the permissions of the output index file";
		errno = (int)error_code;
		return(-1);
	}

	entries_set = 0;

	/* Open the input recycle bin index file */
	rp = edv_recycle_bin_index_open(in_path);
	if(rp != NULL)
	{
		/* Iterate through all all the indices in the input
		 * file and set all entries with indicies that match
		 * the specified index
		 */
		EDVRecycledObject *cur_obj;
		for(cur_obj = edv_recycle_bin_index_next(rp);
		    cur_obj != NULL;
		    cur_obj = edv_recycle_bin_index_next(rp)
		)
		{
			/* Indices do not match? */
			if(cur_obj->index != index)
			{
				/* Write the corresponding entry from
				 * the input file to the output file
				 */
				edv_recycle_bin_index_write_object_iterate(
					out_fp,
					cur_obj->index,
					cur_obj
				);
			}
			else
			{
				/* Indicies match
				 *
				 * If the object values were specified
				 * then write the specified object
				 * values to the output index file,
				 * otherwise do not write this object's
				 * values to the index file which will
				 * effectively remove it
				 */
				if(obj != NULL)
					edv_recycle_bin_index_write_object_iterate(
						out_fp,
						cur_obj->index,
						obj
					);
				entries_set++;
			}
		}

		/* Close the input recycle bin index file */
		edv_recycle_bin_index_close(rp);
	}

	/* Close the output recycle bin index file */
	if(fclose(out_fp))
	{
		const gint error_code = (gint)errno;
		g_free(in_path);
		edv_unlink(out_path);
		g_free(out_path);
		edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the output recycle bin index file";
		errno = (int)error_code;
		return(-1);
	}

	/* Remove the old input recycle bin index file (if any) */
	(void)edv_unlink(in_path);

	/* Rename the output recycle bin index file to the input
	 * (true) recycle bin index file
	 */
	if(edv_rename(
		out_path,
		in_path
	))
	{
		const gint error_code = (gint)errno;
		g_free(in_path);
		g_free(out_path);
		edv_recycle_bin_index_last_error_ptr =
"Unable to rename the output recycle bin index file to the input recycle bin index file";
		errno = (int)error_code;
		return(-1);
	}

	g_free(in_path);
	g_free(out_path);

	return((entries_set > 0) ? 0 : -1);
}

/*
 *	Removes an entry from the recycle bin index file.
 *
 *	The index_path specifies the recycle bin index file.
 *
 *	The index specifies the entry to remove.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_recycle_bin_index_remove(
	const gchar *index_path,
	const gulong index
)
{
	return(edv_recycle_bin_index_set(
		index_path,
		index,
		NULL
	));
}


/*
 *	Low-level recycling of a VFS object to the Recycle Bin under
 *	a specific index. This function will not create a new index,
 *	the index should be created and obtained by a prior call to
 *	edv_recycle_bin_index_add().
 *
 *	The index_path specifies the recycled objects index file, the
 *	recycle bin index file will not be updated.
 *
 *	The index specifies the index to create the recycled object
 *	under.
 *
 *	The path specifies the VFS object to recycle, the actual
 *	VFS object on the device will be deleted by this call. If the
 *	object is a directory then it must be empty.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_recycle_bin_index_recycle(
	const gchar *index_path,
	const gulong index,
	const gchar *path,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
)
{
	gint status;
	gchar		*recbin_path,
			*in_path,
			*out_path;
	EDVVFSObject *in_obj;

	edv_recycle_bin_index_set_error_message(NULL);

	if(STRISEMPTY(index_path))
	{
		edv_recycle_bin_index_last_error_ptr =
"Recycled objects index file was not specified";
		errno = EINVAL;
		return(-2);
	}

	if(index == 0l)
	{
		edv_recycle_bin_index_last_error_ptr =
"Invalid recycled object index";
		errno = EINVAL;
		return(-2);
	}

	if(STRISEMPTY(path))
	{
		edv_recycle_bin_index_last_error_ptr =
"Object to delete was not specified";
		errno = EINVAL;
		return(-2);
	}

	/* Create the recycle bin directory as needed */
	if(edv_recycle_bin_index_create_recycle_bin_directory(index_path))
		return(-1);

	in_path = NULL;
	out_path = NULL;
	in_obj = NULL;

#define CLEANUP_RETURN(_v_,_error_code_) {	\
 edv_vfs_object_delete(in_obj);			\
 g_free(in_path);				\
 g_free(out_path);				\
						\
 errno = (int)(_error_code_);			\
						\
 return(_v_);					\
}

	/* Format the full path to the object to delete */
	in_path = g_strdup(path);
	if(in_path == NULL)
	{
		const gint error_code = (gint)errno;
		edv_recycle_bin_index_last_error_ptr =
"Memory allocation error";
		CLEANUP_RETURN(-3, error_code);
	}

	/* Get the full path to the recycle bin from the index file path */
	recbin_path = edv_recycle_bin_index_get_recbin_directory_path(index_path);
	if(recbin_path == NULL)
	{
		const gint error_code = (gint)errno;
		edv_recycle_bin_index_last_error_ptr =
"Unable to obtain the recycle bin directory from the recycled objects index file";
		CLEANUP_RETURN(-2, error_code);
	}

	/* Format the full path to the recycled object, this works
	 * the same as edv_recycle_bin_index_get_recycled_object_path()
	 * but faster
	 */
	out_path = g_strdup_printf(
		"%s%c%ld",
		recbin_path,
		G_DIR_SEPARATOR,
		index
	);

	g_free(recbin_path);

	if(out_path == NULL)
	{
		edv_recycle_bin_index_last_error_ptr =
"Unable to format the path to the recycled object";
		CLEANUP_RETURN(-2, ENOMEM);
	}

	/* Get the statistics of the object to delete */
	in_obj = edv_vfs_object_lstat(in_path);
	if(in_obj == NULL)
	{
		const gint error_code = (gint)errno;
		edv_recycle_bin_index_last_error_ptr =
"Unable to get the statistics of the object to delete";
		CLEANUP_RETURN(-1, error_code);
	}

	status = 0;

	/* Copy in_path to out_path by in_path's type */

	/* Regular file? */
	if(EDV_VFS_OBJECT_IS_FILE(in_obj))
	{
		const gulong file_size = in_obj->size;
		gulong io_buf_len = in_obj->block_size;
		size_t coppied_size;
		FILE	*in_fp,
			*out_fp;
		gint out_fd;
		guint8 *io_buf;
		EDVVFSObject *out_obj;

		/* Report the initial progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 0l, file_size))
			{
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		/* Open the file to be deleted for reading */
		in_fp = fopen((const char *)in_path, "rb");
		if(in_fp == NULL)
		{
			const gint error_code = (gint)errno;
			edv_recycle_bin_index_set_error_message(g_strerror(error_code));
			CLEANUP_RETURN(-1, error_code);
		}

		/* Report the progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(
				progress_data,
				0l, file_size
			))
			{
				(void)fclose(in_fp);
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		/* Create/open the recycled file for writing */
		out_fp = fopen((const char *)out_path, "wb");
		if(out_fp == NULL)
		{
			const gint error_code = (gint)errno;
			(void)fclose(in_fp);
			edv_recycle_bin_index_last_error_ptr =
"Unable to open the recycled file for writing";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Report the progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 0l, file_size))
			{
				(void)fclose(in_fp);
				(void)fclose(out_fp);
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		out_fd = (gint)fileno(out_fp);

		/* Get the recycled file's statistics */
		out_obj = edv_vfs_object_fstat(out_fd);
		if(out_obj != NULL)
		{
			/* Need to use a smaller I/O buffer? */
			if(out_obj->block_size < io_buf_len)
				io_buf_len = out_obj->block_size;
			edv_vfs_object_delete(out_obj);
		}
		else
		{
			const gint error_code = (gint)errno;
			(void)fclose(in_fp);
			(void)fclose(out_fp);
			(void)edv_unlink(out_path);
			edv_recycle_bin_index_last_error_ptr =
"Unable to get the recycled file's statistics";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Set the recycled file's permissions so that only
		 * the owner can read and write to it
		 */
		if(edv_permissions_set_fd(
			out_fd,
			EDV_PERMISSION_UR | EDV_PERMISSION_UW
		))
		{
			const gint error_code = (gint)errno;
			(void)fclose(in_fp);
			(void)fclose(out_fp);
			(void)edv_unlink(out_path);
			edv_recycle_bin_index_last_error_ptr =
"Unable to set the recycled file's permissions";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Create the I/O buffer */
		if(io_buf_len > 1l)
		{
			/* Allocate the I/O buffer */
			io_buf = (guint8 *)g_malloc(io_buf_len * sizeof(guint8));
			if(io_buf == NULL)
			{
				const gint error_code = (gint)errno;
				(void)fclose(in_fp);
				(void)fclose(out_fp);
				(void)edv_unlink(out_path);
				edv_recycle_bin_index_last_error_ptr =
"Memory allocation error";
				CLEANUP_RETURN(-3, error_code);
			}
		}
		else
		{
			io_buf = NULL;			/* Copy one byte at a time */
		}

		coppied_size = 0l;

		/* Copy one block at a time? */
		if(io_buf != NULL)
		{
			size_t	units_read,
					units_written;

			while(!feof(in_fp))
			{
				/* Read the next block from the file to be deleted */
				units_read = fread(
					io_buf,
					sizeof(guint8),
					(size_t)io_buf_len,
					in_fp
				);
				/* Check if an error ocured while reading from the
				 * file to be deleted
				 */
				if(ferror(in_fp))
				{
					/* Read error */
					edv_recycle_bin_index_last_error_ptr =
"An error occured while reading from the file to be deleted";
					status = -1;

					/* Attempt to write any data that was read */
					if((units_read > 0l) && (units_read <= (size_t)io_buf_len))
					{
						const size_t units_written = fwrite(
							io_buf,
							sizeof(guint8),
							units_read,
							out_fp
						);
						if(units_written > 0l)
							coppied_size += units_written;
					}

					break;
				}
				/* No data read? */
				else if(units_read <= 0l)
				{
					/* Check for the EOF marker at
					 * the top of this while() loop
					 */
					continue;
				}

				/* More than the requested number of
				 * units read?
				 */
				if(units_read > (size_t)io_buf_len)
				{
					edv_recycle_bin_index_last_error_ptr =
"Too many bytes were read from the file to be deleted";
					status = -1;
					break;
				}

				/* Write this block to the recycled file */
				units_written = fwrite(
					io_buf,
					sizeof(guint8),
					units_read,
					out_fp
				);
				if(units_written != units_read)
				{
					if(units_written > 0l)
						coppied_size += units_written;
					edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled file";
					status = -1;
					break;
				}

				/* Update the coppied size */
				coppied_size += units_written;

				/* Report progress? */
				if(progress_cb != NULL)
				{
					if(progress_cb(
						progress_data,
						(gulong)coppied_size, file_size
					))
					{
						status = -4;
						break;
					}
				}
			}

			/* Delete the I/O buffer */
			g_free(io_buf);
		}
		else
		{
			/* Copy one byte at a time */
			gulong copy_cnt = 0l;

			/* Read the first character from the file to
			 * be deleted
			 */
			gint c = (gint)fgetc(in_fp);
			while((int)c != EOF)
			{
				/* Write this character to the recycled
				 * file
				 */
				if(fputc((int)c, out_fp) == EOF)
				{
					edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled file";
					status = -1;
					break;
				}

				/* Increment the copy counter */
				copy_cnt++;

				/* Update the coppied size */
				coppied_size++;

				/* Time to report the progress? */
				if(copy_cnt >= 10000l)
				{
					/* Report the progress */
					if(progress_cb != NULL)
					{
						if(progress_cb(
							progress_data,
							(gulong)coppied_size, file_size
						))
						{
							status = -4;
							break;
						}
					}
					copy_cnt = 0l;
				}

				/* Read the next character from the
				 * file to be deleted
				 */
				c = (gint)fgetc(in_fp);
			}
			/* Did an error occur while reading from the
			 * file to be deleted?
			 */
			if(ferror(in_fp))
			{
				/* Read error */
				edv_recycle_bin_index_last_error_ptr =
"An error occured while reading from the file to be deleted";
				status = -1;
			}
		}

		/* Close the file to be deleted */
		(void)fclose(in_fp);

		/* Close the recycled file */
		if(fclose(out_fp))
		{
			const gint error_code = (gint)errno;
			(void)edv_unlink(out_path);
			edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled file";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Did not copy all of the data? */
		if(coppied_size < (size_t)file_size)
		{
			/* No previous error encountered or user abort? */
			if(status == 0)
			{
				edv_recycle_bin_index_last_error_ptr =
"Unable to copy the entire file to be deleted";
				status = -1;
			}
		}

		/* Report the final progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(
				progress_data,
				file_size, file_size
			))
				status = -4;
		}

		/* If the user aborted then remove the recycled file */
		if(status == -4)
			(void)edv_unlink(out_path);

		/* If no error occured then remove the file to be
		 * deleted
		 */
		if(status == 0)
		{
			if(edv_unlink(in_path))
			{
				const gint error_code = (gint)errno;
				(void)edv_unlink(out_path);
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the file";
				CLEANUP_RETURN(-1, error_code);
			}
		}
	}
	/* Directory? */
	else if(EDV_VFS_OBJECT_IS_DIRECTORY(in_obj))
	{
		FILE *out_fp;

		/* Report the initial progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 0l, 1l))
			{
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		/* Open the recycled file for writing */
		out_fp = fopen((const char *)out_path, "wb");
		if(out_fp == NULL)
		{
			const gint error_code = (gint)errno;
			edv_recycle_bin_index_last_error_ptr =
"Unable to open the recycled file for writing";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Do not store any data in the output file for
		 * directories
		 */

		/* Close the recycled file */
		if(fclose(out_fp))
		{
			const gint error_code = (gint)errno;
			(void)edv_unlink(out_path);
			edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled file";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Report the final progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 1l, 1l))
				status = -4;
		}

		/* If the user aborted then remove the recycled file */
		if(status == -4)
			(void)edv_unlink(out_path);

		/* If no error occured then remove the directory */
		if(status == 0)
		{
			if(edv_directory_remove(
				in_path,
				FALSE,			/* Do not recurse */
				FALSE,			/* Do not force */
				NULL,			/* No paths list return */
				NULL, NULL
			))
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the directory";
				(void)edv_unlink(out_path);
				CLEANUP_RETURN(-1, error_code);
			}
		}
	}
	/* Link? */
	else if(EDV_VFS_OBJECT_IS_LINK(in_obj))
	{
		gchar *v;

		/* Report the initial progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 0l, 2l))
			{
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		/* Get the link's target value */
		v = edv_link_get_target(in_path);
		if(v == NULL)
		{
			const gint error_code = (gint)errno;
			edv_recycle_bin_index_last_error_ptr =
"Unable to get the link's target value";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Report the progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 1l, 2l))
				status = -4;
		}

		if(status == 0)
		{
			/* Open the recycled file for writing */
			FILE *out_fp = fopen((const char *)out_path, "wb");
			if(out_fp == NULL)
			{
				const gint error_code = (gint)errno;
				g_free(v);
				edv_recycle_bin_index_last_error_ptr =
"Unable to open the recycled file for writing";
				CLEANUP_RETURN(-1, error_code);
			}

			/* Write the link's target value */
			if(fputs((const char *)v, out_fp) == EOF)
			{
				const gint error_code = (gint)errno;
				(void)fclose(out_fp);
				(void)edv_unlink(out_path);
				g_free(v);
				edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled file";
				CLEANUP_RETURN(-1, error_code);
			}

			/* Close the recycled file */
			if(fclose(out_fp))
			{
				const gint error_code = (gint)errno;
				(void)edv_unlink(out_path);
				g_free(v);
				edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled file";
				CLEANUP_RETURN(-1, error_code);
			}
		}

		/* Delete the link's value */
		g_free(v);

		/* Report the final progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 2l, 2l))
				status = -4;
		}

		/* If the user aborted then remove the recycled file */
		if(status == -4)
			(void)edv_unlink(out_path);

		/* Remove the link */
		if(status == 0)
		{
			if(edv_unlink(in_path))
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the link";
				(void)edv_unlink(out_path);
				CLEANUP_RETURN(-1, error_code);
			}
		}
	}
	/* FIFO pipe? */
	else if(EDV_VFS_OBJECT_IS_FIFO(in_obj))
	{
		/* Report the initial progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 0l, 2l))
			{
				CLEANUP_RETURN(-4, EINTR);
			}
		}


		/* Report the progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 1l, 2l))
				status = -4;
		}

		if(status == 0)
		{
			/* Open the recycled file for writing */
			FILE *out_fp = fopen((const char *)out_path, "wb");
			if(out_fp == NULL)
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to open the recycled file for writing";
				CLEANUP_RETURN(-1, error_code);
			}

			/* Do not write any data to the output file for FIFO
			 * pipes
			 */

			/* Close the recycled file */
			if(fclose(out_fp))
			{
				const gint error_code = (gint)errno;
				(void)edv_unlink(out_path);
				edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled file";
				CLEANUP_RETURN(-1, error_code);
			}
		}

		/* Report the final progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 2l, 2l))
				status = -4;
		}

		/* If the user aborted then remove the recycled file */
		if(status == -4)
			(void)edv_unlink(out_path);

		/* If no error occured then remove the FIFO pipe */
		if(status == 0)
		{
			if(edv_unlink(in_path))
			{
				const gint error_code = (gint)errno;
				(void)edv_unlink(out_path);
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the FIFO pipe";
				CLEANUP_RETURN(-1, error_code);
			}
		}
	}
	/* Block device? */
	else if(EDV_VFS_OBJECT_IS_DEVICE_BLOCK(in_obj))
	{
		gint major, minor;

		/* Report the initial progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 0l, 2l))
			{
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		/* Get block device's major and minor numbers */
		edv_device_numbers_parse(
			(gint)in_obj->device_type,
			&major, &minor
		);

		/* Report the progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 1l, 2l))
				status = -4;
		}

		if(status == 0)
		{
			/* Open the recycled file for writing */
			FILE *out_fp = fopen((const char *)out_path, "wb");
			if(out_fp == NULL)
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to open the recycled file for writing";
				CLEANUP_RETURN(-1, error_code);
			}

			/* Write the major and minor numbers to recycled file */
			if(fwrite(&major, sizeof(major), 1l, out_fp) != 1l)
			{
				const gint error_code = (gint)errno;
				(void)fclose(out_fp);
				(void)edv_unlink(out_path);
				edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled file";
				CLEANUP_RETURN(-1, error_code);
			}
			if(fwrite(&minor, sizeof(minor), 1l, out_fp) != 1l)
			{
				const gint error_code = (gint)errno;
				(void)fclose(out_fp);
				(void)edv_unlink(out_path);
				edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled file";
				CLEANUP_RETURN(-1, error_code);
			}

			/* Close the recycled file */
			if(fclose(out_fp))
			{
				const gint error_code = (gint)errno;
				(void)edv_unlink(out_path);
				edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled file";
				CLEANUP_RETURN(-1, error_code);
			}
		}

		/* Report the final progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 2l, 2l))
				status = -4;
		}

		/* If the user aborted then remove the recycled file */
		if(status == -4)
			(void)edv_unlink(out_path);

		/* If no error occured then remove the block device */
		if(status == 0)
		{
			if(edv_unlink(in_path))
			{
				const gint error_code = (gint)errno;
				(void)edv_unlink(out_path);
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the block device";
				CLEANUP_RETURN(-1, error_code);
			}
		}
	}
	/* Character device? */
	else if(EDV_VFS_OBJECT_IS_DEVICE_CHARACTER(in_obj))
	{
		gint major, minor;

		/* Report the initial progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 0l, 2l))
			{
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		/* Get the character device's major and minor numbers */
		edv_device_numbers_parse(
			(gint)in_obj->device_type,
			&major, &minor
		);

		/* Report the progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 1l, 2l))
				status = -4;
		}

		if(status == 0)
		{
			/* Open the recycled file for writing */
			FILE *out_fp = fopen((const char *)out_path, "wb");
			if(out_fp == NULL)
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to open the recycled file for writing";
				CLEANUP_RETURN(-1, error_code);
			}

			/* Write the major and minor numbers to recycled file */
			if(fwrite(&major, sizeof(major), 1l, out_fp) < 1l)
			{
				const gint error_code = (gint)errno;
				(void)fclose(out_fp);
				(void)edv_unlink(out_path);
				edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled file";
				CLEANUP_RETURN(-1, error_code);
			}
			if(fwrite(&minor, sizeof(minor), 1l, out_fp) != 1l)
			{
				const gint error_code = (gint)errno;
				(void)fclose(out_fp);
				(void)edv_unlink(out_path);
				edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled file";
				CLEANUP_RETURN(-1, error_code);
			}

			/* Close the recycled file */
			if(fclose(out_fp))
			{
				const gint error_code = (gint)errno;
				(void)edv_unlink(out_path);
				edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled file";
				CLEANUP_RETURN(-1, error_code);
			}
		}

		/* Report the final progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 2l, 2l))
				status = -4;
		}

		/* If the user aborted then remove the recycled file */
		if(status == -4)
			(void)edv_unlink(out_path);

		/* If no error occured then remove the character device */
		if(status == 0)
		{
			if(edv_unlink(in_path))
			{
				const gint error_code = (gint)errno;
				(void)edv_unlink(out_path);
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the character device";
				CLEANUP_RETURN(-1, error_code);
			}
		}
	}
	/* Socket? */
	else if(EDV_VFS_OBJECT_IS_SOCKET(in_obj))
	{
		/* Report the initial progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 0l, 1l))
			{
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		if(status == 0)
		{
			/* Open the recycled file for writing */
			FILE *out_fp = fopen((const char *)out_path, "wb");
			if(out_fp == NULL)
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to open the recycled file for writing";
				CLEANUP_RETURN(-1, error_code);
			}

			/* Do not write any data to for sockets */

			/* Close the recycled file */
			if(fclose(out_fp))
			{
				const gint error_code = (gint)errno;
				(void)edv_unlink(out_path);
				edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recycled file";
				CLEANUP_RETURN(-1, error_code);
			}
		}

		/* Report the final progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 1l, 1l))
				status = -4;
		}

		/* If the user aborted then remove the recycled file */
		if(status == -4)
			(void)edv_unlink(out_path);

		/* Remove the socket */
		if(status == 0)
		{
			if(edv_unlink(in_path))
			{
				const gint error_code = (gint)errno;
				(void)edv_unlink(out_path);
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the socket";
				CLEANUP_RETURN(-1, error_code);
			}
		}
	}
	/* Unsupported type? */
	else
	{
		edv_recycle_bin_index_last_error_ptr =
"Unsupported object type";
		CLEANUP_RETURN(-2, EINVAL);
	}

	CLEANUP_RETURN(status, 0);
#undef CLEANUP_RETURN
}

/*
 *	Low-level recovering of a recycled object to the VFS.
 *
 *	The index_path specifies the recycled objects index file, the
 *	recycle bin index file will not be updated. Upon success,
 *	the calling function should call edv_recycle_bin_index_remove()
 *	to remove the entry from the index file.
 *
 *	The index specifies the recycled object to recover. The actual
 *	recycled object will be deleted upon successful recover.
 *
 *	The path specifies the full path of the recovered object. This
 *	object must not already exist or else the recovery will fail.
 *	If path is NULL then the recycled object's original location
 *	and name will be used.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_recycle_bin_index_recover(
	const gchar *index_path,
	const gulong index,
	const gchar *path,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
)
{
	gboolean	restore_permissions,
			restore_ownership,
			restore_time;
	gint status;
	gchar		*recbin_path,
			*in_path,
			*out_path;
	EDVVFSObject *in_obj;
	EDVRecycledObject *obj;
	EDVRecycleBinIndex *rp;

	edv_recycle_bin_index_set_error_message(NULL);

	if(STRISEMPTY(index_path))
	{
		edv_recycle_bin_index_last_error_ptr =
"Recycled objects index file was not specified";
		errno = EINVAL;
		return(-2);
	}

	if(index == 0l)
	{
		edv_recycle_bin_index_last_error_ptr =
"Invalid recycled object index";
		errno = EINVAL;
		return(-2);
	}

	in_path = NULL;
	out_path = NULL;
	obj = NULL;
	in_obj = NULL;

#define CLEANUP_RETURN(_v_,_error_code_) {	\
 g_free(in_path);				\
 g_free(out_path);				\
 edv_recycled_object_delete(obj);		\
 edv_vfs_object_delete(in_obj);			\
						\
 errno = (int)(_error_code_);			\
						\
 return(_v_);					\
}

	/* Get the recycled object's statistics */
	rp = edv_recycle_bin_index_open(index_path);
	obj = edv_recycle_bin_index_seek(
		rp,
		index
	);
	if(obj != NULL)
		obj = edv_recycled_object_copy(obj);
	edv_recycle_bin_index_close(rp);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
		edv_recycle_bin_index_last_error_ptr =
"Unable to get the statistics of the recycled object";
		CLEANUP_RETURN(-1, error_code);
	}

	/* Get the full path to the recycle bin from the index file path */
	recbin_path = edv_recycle_bin_index_get_recbin_directory_path(index_path);
	if(recbin_path == NULL)
	{
		const gint error_code = (gint)errno;
		edv_recycle_bin_index_last_error_ptr =
"Unable to obtain the recycle bin directory from the recycled objects index file";
		CLEANUP_RETURN(-2, error_code);
	}

	/* Format the full path to the recycled object */
	in_path = g_strdup_printf(
		"%s%c%ld",
		recbin_path,
		G_DIR_SEPARATOR,
		index
	);

	g_free(recbin_path);

	if(in_path == NULL)
	{
		edv_recycle_bin_index_last_error_ptr =
"Unable to format the path to the recycled object";
		CLEANUP_RETURN(-1, ENOMEM);
	}

	/* Format the full path to the recovered object */
	if(path != NULL)
	{
		/* Use the specified alternate location */
		out_path = g_strdup(path);
	}
	else
	{
		/* Use the original location */
		out_path = edv_paths_join(
			obj->original_path,
			obj->name
		);
	}
	if(out_path == NULL)
	{
		const gint error_code = (gint)errno;
		edv_recycle_bin_index_last_error_ptr =
"Unable to format the path to the recovered object";
		CLEANUP_RETURN(-1, error_code);
	}

	/* Make sure that the recycled object exists and that the
	 * recovered object does not exist
	 */
	in_obj = edv_vfs_object_lstat(in_path);
	if(in_obj == NULL)
	{
		const gint error_code = (gint)errno;
		edv_recycle_bin_index_last_error_ptr =
"Unable to get the recycled object's statistics";
		CLEANUP_RETURN(-1, error_code);
	}
	if(edv_path_lexists(out_path))
	{
		edv_recycle_bin_index_last_error_ptr =
"An object already exists at the recovery location";
 	    CLEANUP_RETURN(-1, EEXIST);
	}

	status = 0;
	restore_permissions = FALSE;
	restore_ownership = FALSE;
	restore_time = FALSE;

	/* Copy in_path to out_path by the object's type */

	/* Regular file? */
	if(obj->type == EDV_OBJECT_TYPE_FILE)
	{
		const gulong file_size = in_obj->size;
		gulong io_buf_len = in_obj->block_size;
		size_t coppied_size;
		FILE	*in_fp,
			*out_fp;
		gint out_fd;
		guint8 *io_buf;
		EDVVFSObject *tar_obj;

		/* Report the initial progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(progress_data, 0l, file_size))
			{
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		/* Open the recycled file for reading */
		in_fp = fopen((const char *)in_path, "rb");
		if(in_fp == NULL)
		{
			const gint error_code = (gint)errno;
			edv_recycle_bin_index_last_error_ptr =
"Unable to open the recycled file for reading";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Report the progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 0l, file_size))
			{
				(void)fclose(in_fp);
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		/* Create/open the recovered file for writing */
		out_fp = fopen((const char *)out_path, "wb");
		if(out_fp == NULL)
		{
			const gint error_code = (gint)errno;
			(void)fclose(in_fp);
			edv_recycle_bin_index_set_error_message(g_strerror(error_code));
			CLEANUP_RETURN(-1, error_code);
		}

		/* Report the progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(
				progress_data,
				0l, file_size
			))
			{
				(void)fclose(in_fp);
				(void)fclose(out_fp);
				(void)edv_unlink(out_path);
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		out_fd = (gint)fileno(out_fp);

		/* Get the recovered file's statistics */
		tar_obj = edv_vfs_object_fstat(out_fd);
		if(tar_obj != NULL)
		{
			/* Need to use a smaller I/O buffer? */
			if(tar_obj->block_size < io_buf_len)
				io_buf_len = tar_obj->block_size;
			edv_vfs_object_delete(tar_obj);
		}
		else
		{
			const gint error_code = (gint)errno;
			(void)fclose(in_fp);
			(void)fclose(out_fp);
			edv_recycle_bin_index_last_error_ptr =
"Unable to get the recovered file's statistics";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Copy one block at a time? */
		if(io_buf_len > 1l)
		{
			/* Allocate the I/O buffer */
			io_buf = (guint8 *)g_malloc(io_buf_len * sizeof(guint8));
			if(io_buf == NULL)
			{
				const gint error_code = (gint)errno;
				(void)fclose(in_fp);
				(void)fclose(out_fp);
				edv_recycle_bin_index_last_error_ptr =
"Memory allocation error";
				CLEANUP_RETURN(-3, error_code);
			}
		}
		else
		{
			io_buf = NULL;		/* Copy one byte at a time */
		}

		coppied_size = 0l;

		/* Copy one block at a time? */
		if(io_buf != NULL)
		{
			size_t	units_read,
					units_written;

			while(!feof(in_fp))
			{
				/* Read the next block from the
				 * recycled file
				 */
				units_read = fread(
					io_buf,
					sizeof(guint8),
					(size_t)io_buf_len,
					in_fp
				);
				/* Check if an error ocured while
				 * reading from the recycled file
				 */
				if(ferror(in_fp))
				{
					/* Read error */
					edv_recycle_bin_index_last_error_ptr =
"An error occured while reading from the recycled file";
					status = -1;

					/* Attempt to write any data
					 * that was read
					 */
					if((units_read > 0l) && (units_read <= (size_t)io_buf_len))
					{
						const size_t units_written = fwrite(
							io_buf,
							sizeof(guint8),
							units_read,
							out_fp
						);
						if(units_written > 0l)
							coppied_size += units_written;
					}

					break;
				}
				/* No data read? */
				else if(units_read <= 0l)
				{
					/* Check for the EOF marker at
					 * the top of this while() loop
					 */
					continue;
				}

				/* More than the requested number of
				 * units read?
				 */
				if(units_read > (size_t)io_buf_len)
				{
					edv_recycle_bin_index_last_error_ptr =
"Too many bytes were read from the recycled file";
					status = -1;
					break;
				}

				/* Write this block to the recovered
				 * file
				 */
				units_written = fwrite(
					io_buf,
					sizeof(guint8),
					units_read,
					out_fp
				);
				if(units_written != units_read)
				{
					/* Write error */
					if(units_written > 0l)
						coppied_size += units_written;
					edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recovered file";
					status = -1;
					break;
				}

				/* Update the coppied size */
				coppied_size += units_written;

				/* Report the progress */
				if(progress_cb != NULL)
				{
					if(progress_cb(
						progress_data,
						(gulong)coppied_size, file_size
					))
					{
						status = -4;
						break;
					}
				}
			}

			/* Delete the I/O buffer */
			g_free(io_buf);
		}
		else
		{
			/* Copy one byte at a time */
			gulong copy_cnt = 0l;

			/* Read the first character from the recycled
			 * file
			 */
			gint c = (gint)fgetc(in_fp);
			while((int)c != EOF)
			{
				/* Write this character to the recovered
				 * file
				 */
				if(fputc((int)c, out_fp) == EOF)
				{
					edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recovered file";
					status = -1;
					break;
				}

				/* Increment the copy counter */
				copy_cnt++;

				/* Update the coppied size */
				coppied_size++;

				/* Time to report progress? */
				if(copy_cnt >= 10000l)
				{
					/* Report the progress */
					if(progress_cb != NULL)
					{
						if(progress_cb(
							progress_data,
							(gulong)coppied_size, file_size
						))
						{
							status = -4;
							break;
						}
					}
					copy_cnt = 0l;
				}

				/* Read the next character from the
				 * recycled file
				 */
				c = (gint)fgetc(in_fp);
			}
			/* Did an error occur while reading from the recycled
			 * file?
			 */
			if(ferror(in_fp))
			{
				/* Read error */
				edv_recycle_bin_index_last_error_ptr =
"An error occured while reading from the recycled file";
				status = -1;
			}
		}

		/* Close the recycled file */
		(void)fclose(in_fp);

		/* Close the recovered file */
		if(fclose(out_fp))
		{
			edv_recycle_bin_index_last_error_ptr =
"An error occured while writing to the recovered file";
			status = -1;
		}

		/* Did not copy all of the data? */
		if(coppied_size < file_size)
		{
			/* No previous error encountered or user abort? */
			if(status == 0)
			{
				edv_recycle_bin_index_last_error_ptr =
"Unable to copy the entire file to be recovered";
				status = -1;
			}
		}

		/* Report the final progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(
				progress_data,
				file_size, file_size
			))
				status = -4;
		}

		/* If the user aborted then remove the recovered file */
		if(status == -4)
			(void)edv_unlink(out_path);

		/* If there was no error then remove the recycled file */
		if(status == 0)
		{
			if(edv_unlink(in_path))
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the recycled object";
				CLEANUP_RETURN(-1, error_code);
			}
		}

		restore_permissions = TRUE;
		restore_ownership = TRUE;
		restore_time = TRUE;
	}
	/* Directory? */
	else if(obj->type == EDV_OBJECT_TYPE_DIRECTORY)
	{
		/* Report the initial progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(progress_data, 0l, 1l))
			{
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		/* Create the recovered directory */
		if(edv_directory_create(
			out_path,
			FALSE,			/* Do not create parents */
			NULL			/* No new paths list return */
		))
		{
			const gint error_code = (gint)errno;
			edv_recycle_bin_index_last_error_ptr =
"Unable to create the recovered directory";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Report the final progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 1l, 1l))
				status = -4;
		}

		/* If the user aborted then remove the recovered file */
		if(status == -4)
			(void)edv_unlink(out_path);

		/* If there was no error then remove the recycled file */
		if(status == 0)
		{
			if(edv_unlink(in_path))
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the recycled object";
				CLEANUP_RETURN(-1, error_code);
			}
		}

		restore_permissions = TRUE;
		restore_ownership = TRUE;
		restore_time = TRUE;
	}
	/* Link? */
	else if(obj->type == EDV_OBJECT_TYPE_LINK)
	{
		const gulong file_size = in_obj->size;
		size_t units_read;
		FILE *in_fp;
		gchar *v;

		/* Report the initial progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(progress_data, 0l, file_size))
			{
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		/* Open the recycled file for reading */
		in_fp = fopen((const char *)in_path, "rb");
		if(in_fp == NULL)
		{
			const gint error_code = (gint)errno;
			edv_recycle_bin_index_last_error_ptr =
"Unable to open the recycled file for reading";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Allocate the link target value buffer */
		v = (gchar *)g_malloc((file_size + 1) * sizeof(gchar));
		if(v == NULL)
		{
			const gint error_code = (gint)errno;
			(void)fclose(in_fp);
			edv_recycle_bin_index_last_error_ptr =
"Memory allocation error";
			CLEANUP_RETURN(-3, error_code);
		}

		/* Read the input file as the link target value */
		units_read = fread(
			v,
			sizeof(gchar),
			(size_t)file_size,
			in_fp
		);
		if(units_read < (size_t)file_size)
		{
			const gint error_code = (gint)errno;
			(void)fclose(in_fp);
			g_free(v);
			edv_recycle_bin_index_last_error_ptr =
"An error occured while reading from the recycled file";
			CLEANUP_RETURN(-1, error_code);
		}

		v[file_size] = '\0';

		/* Close the recycled file */
		(void)fclose(in_fp);

		/* Report the progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(
				progress_data,
				file_size, file_size
			))
				status = -4;
		}

		/* Create the link containing the target value read
		 * from the input file
		 */
		if(edv_link_create(
			out_path,			/* Path */
			v				/* Target */
		))
		{
			const gint error_code = (gint)errno;
			g_free(v);
			edv_recycle_bin_index_last_error_ptr =
"Unable to create the recovered link";
			CLEANUP_RETURN(-1, error_code);
		}

		g_free(v);

		/* Report the final progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(
				progress_data,
				file_size, file_size
			))
				status = -4;
		}

		/* If the user aborted then remove the recovered file */
		if(status == -4)
			(void)edv_unlink(out_path);

		/* If there was no error then remove the recycled file */
		if(status == 0)
		{
			if(edv_unlink(in_path))
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the recycled object";
				CLEANUP_RETURN(-1, error_code);
			}
		}

		/* Do not restore permissions for links */
		restore_ownership = TRUE;
		/* Do not restore time for links */
	}
	/* FIFO pipe? */
	else if(obj->type == EDV_OBJECT_TYPE_FIFO)
	{
#if defined(S_IFFIFO) || defined(S_IFIFO)
#ifdef S_IFFIFO
		const mode_t m = S_IFFIFO | S_IRUSR | S_IWUSR;
#else
		const mode_t m = S_IFIFO | S_IRUSR | S_IWUSR;
#endif

		/* Report the initial progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(progress_data, 0l, 1l))
			{
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		/* Create the FIFO pipe */
		if(mknod(
			(const char *)out_path,
			m,
			0
		))
		{
			const gint error_code = (gint)errno;
			edv_recycle_bin_index_last_error_ptr =
"Unable to create the recovered FIFO pipe";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Report the final progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 1l, 1l))
				status = -4;
		}

		/* If there was no error then remove the recycled file */
		if(status == 0)
		{
			if(edv_unlink(in_path))
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the recycled object";
				CLEANUP_RETURN(-1, error_code);
			}
		}

		restore_permissions = TRUE;
		restore_ownership = TRUE;
		restore_time = TRUE;
#else
		edv_recycle_bin_index_last_error_ptr =
"Unsupported object type";
		CLEANUP_RETURN(-2, EINVAL);
#endif
	}
	/* Block device? */
	else if(obj->type == EDV_OBJECT_TYPE_DEVICE_BLOCK)
	{
#if defined(S_IFBLK)
		const mode_t m = S_IFBLK | S_IRUSR | S_IWUSR;
		FILE *in_fp;
		gint major, minor;

		/* Report the initial progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(progress_data, 0l, 2l))
			{
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		/* Open the recycled file for reading */
		in_fp = fopen((const char *)in_path, "rb");
		if(in_fp == NULL)
		{
			const gint error_code = (gint)errno;
			edv_recycle_bin_index_last_error_ptr =
"Unable to open the recycled file for reading";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Read the major number */
		if(fread(&major, sizeof(major), 1l, in_fp) < 1l)
		{
			const gint error_code = (gint)errno;
			(void)fclose(in_fp);
			edv_recycle_bin_index_last_error_ptr =
"An error occured while reading from the recycled file";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Read the minor number */
		if(fread(&minor, sizeof(minor), 1l, in_fp) < 1l)
		{
			const gint error_code = (gint)errno;
			(void)fclose(in_fp);
			edv_recycle_bin_index_last_error_ptr =
"An error occured while reading from the recycled file";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Close the recycled file */
		(void)fclose(in_fp);

		/* Report the progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 1l, 2l))
				status = -4;
		}

		if(status == 0)
		{
			/* Get the device number */
			const gint rdev = edv_device_numbers_format(major, minor);

			/* Create the block device */
			if(mknod(
				(const char *)out_path,
				m,
				(dev_t)rdev
			))
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to create the recovered block device";
				CLEANUP_RETURN(-1, error_code);
			}
		}

		/* Report the final progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 2l, 2l))
				status = -4;
		}

		/* If the user aborted then remove the recovered file */
		if(status == -4)
			(void)edv_unlink(out_path);

		/* If there was no error then remove the recycled file */
		if(status == 0)
		{
			if(edv_unlink(in_path))
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the recycled object";
				CLEANUP_RETURN(-1, error_code);
			}
		}

		restore_permissions = TRUE;
		restore_ownership = TRUE;
		restore_time = TRUE;
#else
		edv_recycle_bin_index_last_error_ptr =
"Unsupported object type";
		CLEANUP_RETURN(-2, EINVAL);
#endif
	}
	/* Character device? */
	else if(obj->type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	{
#if defined(S_IFCHR)
		const mode_t m = S_IFCHR | S_IRUSR | S_IWUSR;
		FILE *in_fp;
		gint major, minor;

		/* Report the initial progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(progress_data, 0l, 2l))
			{
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		/* Open the recycled file for reading */
		in_fp = fopen((const char *)in_path, "rb");
		if(in_fp == NULL)
		{
			const gint error_code = (gint)errno;
			edv_recycle_bin_index_last_error_ptr =
"Unable to open the recycled file for reading";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Read the major number */
		if(fread(&major, sizeof(major), 1, in_fp) < 1)
		{
			const gint error_code = (gint)errno;
			(void)fclose(in_fp);
			edv_recycle_bin_index_last_error_ptr =
"An error occured while reading from the recycled file";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Read the minor number */
		if(fread(&minor, sizeof(minor), 1, in_fp) < 1)
		{
			const gint error_code = (gint)errno;
			(void)fclose(in_fp);
			edv_recycle_bin_index_last_error_ptr =
"An error occured while reading from the recycled file";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Close the recycled file */
		(void)fclose(in_fp);

		/* Report the progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 1l, 2l))
				status = -4;
		}

		if(status == 0)
		{
			/* Get the device number */
			const gint rdev = edv_device_numbers_format(major, minor);

			/* Create the character device */
			if(mknod(
				(const char *)out_path,
				m,
				(dev_t)rdev
			))
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to create the recovered character device";
				CLEANUP_RETURN(-1, error_code);
			}
		}

		/* Report the final progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 2l, 2l))
				status = -4;
		}

		/* If the user aborted then remove the recovered file */
		if(status == -4)
			(void)edv_unlink(out_path);

		/* If there was no error then remove the recycled file */
		if(status == 0)
		{
			if(edv_unlink(in_path))
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the recycled object";
				CLEANUP_RETURN(-1, error_code);
			}
		}

		restore_permissions = TRUE;
		restore_ownership = TRUE;
		restore_time = TRUE;
#else
		edv_recycle_bin_index_last_error_ptr =
"Unsupported object type";
		CLEANUP_RETURN(-2, EINVAL);
#endif
	}
	/* Socket? */
	else if(obj->type == EDV_OBJECT_TYPE_SOCKET)
	{
#if defined(S_IFSOCK)
		const mode_t m = S_IFSOCK | S_IRUSR | S_IWUSR;

		/* Report the initial progress */
		if(progress_cb != NULL)
		{
			if(progress_cb(progress_data, 0l, 1l))
			{
				CLEANUP_RETURN(-4, EINTR);
			}
		}

		/* Create the socket */
		if(mknod(
			(const char *)out_path,
			m,
			0
		))
		{
			const gint error_code = (gint)errno;
			edv_recycle_bin_index_last_error_ptr =
"Unable to create the recovered socket";
			CLEANUP_RETURN(-1, error_code);
		}

		/* Report the final progress */
		if((progress_cb != NULL) && (status == 0))
		{
			if(progress_cb(progress_data, 1l, 1l))
				status = -4;
		}

		/* If the user aborted then remove the recovered file */
		if(status == -4)
			(void)edv_unlink(out_path);

		/* If there was no error then remove the recycled file */
		if(status == 0)
		{
			if(edv_unlink(in_path))
			{
				const gint error_code = (gint)errno;
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the recycled object";
				CLEANUP_RETURN(-1, error_code);
			}
		}

		restore_permissions = TRUE;
		restore_ownership = TRUE;
		restore_time = TRUE;
#else
		edv_recycle_bin_index_last_error_ptr =
"Unsupported object type";
		CLEANUP_RETURN(-2, EINVAL);
#endif
	}
	/* Unsupported type? */
	else
	{
		edv_recycle_bin_index_last_error_ptr =
"Unsupported object type";
		CLEANUP_RETURN(-2, EINVAL);
	}


	/* Begin restoring the properties of the recovered object */

	/* Restore the ownership? */
	if(restore_ownership)
	{
		(void)edv_lchown(
			out_path,
			obj->owner_id,
			obj->group_id
		);
	}

	/* Restore the permissions? */
	if(restore_permissions)
	{
		(void)edv_permissions_set(
			out_path,
			obj->permissions
		);
	}

	/* Restore the time stamps? */
	if(restore_time)
	{
		(void)edv_utime(
			out_path,
			obj->access_time,
			obj->modify_time
		);
	}

	CLEANUP_RETURN(status, 0);
#undef CLEANUP_RETURN
}

/*
 *	Low-level purging of a recycled object from the Recycle Bin.
 *
 *	The index_path specifies the recycled objects index file, the
 *	recycle bin index file will not be updated. Upon success,
 *	the calling function should call edv_recycle_bin_index_remove()
 *	to remove the entry from the index file.
 *
 *	The index specifies the recycled object to purge. The actual
 *	recycled object will be deleted upon successful purge.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_recycle_bin_index_purge(
	const gchar *index_path,
	const gulong index,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
)
{
	gint status;
	gchar		*recbin_path,
			*in_path;

	edv_recycle_bin_index_set_error_message(NULL);

	if(STRISEMPTY(index_path))
	{
		edv_recycle_bin_index_last_error_ptr =
"Recycled objects index file was not specified";
		errno = EINVAL;
		return(-2);
	}

	if(index == 0l)
	{
		edv_recycle_bin_index_last_error_ptr =
"Invalid recycled object index";
		errno = EINVAL;
		return(-2);
	}

	/* Create the recycle bin directory as needed */
	if(edv_recycle_bin_index_create_recycle_bin_directory(index_path))
		return(-1);

	/* Get the full path to the recycle bin from the index file path */
	recbin_path = edv_recycle_bin_index_get_recbin_directory_path(index_path);
	if(recbin_path == NULL)
	{
		const gint error_code = (gint)errno;
		edv_recycle_bin_index_last_error_ptr =
"Unable to obtain the recycle bin directory from the recycled objects index file";
		errno = (int)error_code;
		return(-2);
	}

	/* Format the full path to the recycled object, this works
	 * the same as edv_recycle_bin_index_get_recycled_object_path()
	 * but faster
	 */
	in_path = g_strdup_printf(
		"%s%c%ld",
		recbin_path,
		G_DIR_SEPARATOR,
		index
	);

	g_free(recbin_path);

	if(in_path == NULL)
	{
		edv_recycle_bin_index_last_error_ptr =
"Unable to format the path to the recycled object";
		errno = ENOMEM;
		return(-1);
	}

	/* Remove the recycled object */
	status = 0;

	/* Report initial progress? */
	if((progress_cb != NULL) && (status == 0))
	{
		if(progress_cb(progress_data, 0, 1))
			status = -4;
	}

	/* Remove the recycled object if the user did not abort and
	 * there were no errors
	 */
	if(status == 0)
		(void)edv_unlink(in_path);

	/* Report the final progress? */
	if((progress_cb != NULL) && (status == 0))
	{
		if(progress_cb(progress_data, 1, 1))
			status = -4;
	}

	g_free(in_path);

	return(status);
}

/*
 *	Low-level purging of all recycled objects from the Recycle Bin.
 *
 *	This is similar to edv_recycle_bin_index_purge() except that it
 *	purges of the all recycled objects in a single and more
 *	efficient pass.
 *
 *	The index_path specifies the recycled objects index file, the
 *	recycle bin index file will not be updated. Upon success,
 *	the calling function should remove the entire index file
 *	(since all recycled objects have been purged).
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_recycle_bin_index_purge_all(
	const gchar *index_path,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
)
{
	gint status;
	gchar *recbin_path;
	GList *names_list;

	edv_recycle_bin_index_set_error_message(NULL);

	if(STRISEMPTY(index_path))
	{
		edv_recycle_bin_index_last_error_ptr =
"Recycled objects index file was not specified";
		errno = EINVAL;
		return(-2);
	}

	/* Create the recycle bin directory as needed */
	if(edv_recycle_bin_index_create_recycle_bin_directory(index_path))
		return(-1);

	/* Get the full path to the recycle bin from the index file path */
	recbin_path = edv_recycle_bin_index_get_recbin_directory_path(index_path);
	if(recbin_path == NULL)
	{
		const gint error_code = (gint)errno;
		edv_recycle_bin_index_last_error_ptr =
"Unable to obtain the recycle bin directory from the recycled objects index file";
		errno = (int)error_code;
		return(-2);
	}

	status = 0;

	/* Get all contents of the recycled objects directory, which
	 * should include the recycled objects index file
	 */
	names_list = edv_directory_list(
		recbin_path,
		FALSE,				/* Unsorted */
		FALSE				/* Exclude notations */
	);
	if(names_list != NULL)
	{
		const gint nobjs = g_list_length(names_list);
		gint i;
		const gchar *name;
		gchar *full_path;
		GList *glist;

		/* Report initial progress? */
		if((progress_cb != NULL) && (status == 0))
		{
			/* Report progress and check for user abort */
			if(progress_cb(progress_data, 0, nobjs))
				status = -4;
		}

		/* Purge all the objects in the recycle bin directory
		 * (this would include the recycled objects index file
		 * too)
		 */
		for(glist = names_list, i = 0;
			glist != NULL;
			glist = g_list_next(glist), i++
		)
		{
			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			if(status != 0)
				break;

			/* Get this recycled object's full path */
			full_path = g_strconcat(
				recbin_path,
				G_DIR_SEPARATOR_S,
				name,
				NULL
			);
			if(full_path == NULL)
			{
				g_free(glist->data);
				continue;
			}

			/* Report progress? */
			if(progress_cb != NULL)
			{
				/* Report progress and check for user abort */
				if(progress_cb(progress_data, i, nobjs))
				{
					g_free(full_path);
					status = -4;
					break;
				}
			}

			/* Purge this recycled object */
			if(edv_unlink(full_path))
			{
				g_free(full_path);
				edv_recycle_bin_index_last_error_ptr =
"Unable to delete the recycled object";
				status = -1;
				break;
			}

			g_free(full_path);
			g_free(glist->data);
		}
		while(glist != NULL)
		{
			g_free(glist->data);
			glist = g_list_next(glist);
		}

		g_list_free(names_list);

		/* Report the final progress? */
		if((progress_cb != NULL) && (status == 0))
		{
			/* Report progress and check for user abort */
			if(progress_cb(progress_data, nobjs, nobjs))
				status = -4;
		}
	}

	g_free(recbin_path);

	return(status);
}
