/*
	Endeavour Mark II API - Object Recycle, Recover, and Purge

	High-level convience functions for recycle, recover, and
	purge.

	All of these functions will test and lock the Recycle Bin
	during their calls, so the calling function should not lock
	the Recycle Bin.

	See also: edv_recycle_bin_index.h edv_recycled_obj.h
	edv_recycled_obj_stat.h
 */

#ifndef EDV_RECYCLE_H
#define EDV_RECYCLE_H

#include <glib.h>
#include "edv_context.h"


/*
 *	Gets the last error message that occured when calling
 *	edv_recycle(), edv_recover(), or edv_purge().
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	Returns a statically allocated string describing the last
 *	error or NULL if there was no error.
 */
extern const gchar *edv_recycle_get_error(EDVContext *ctx);


/*
 *	Recycles a VFS object.
 *
 *	This function always recycles the object regardless of what
 *	the configuration value EDV_CFG_PARM_DELETE_METHOD is set to.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	The path specifies the VFS object to recycle. If the object is
 *	a directory then it must be empty.
 *
 *	If notify is TRUE then a VFS object removed signal will be
 *	queued for the specified object and a recycled object added
 *	signal will be queued for the recycled object.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	Returns the recycled object index or 0 on error. If the Recycle
 *	Bin is locked then this call will fail and return 0 with errno
 *	set to EBUSY.
 */
extern gulong edv_recycle(
	EDVContext *ctx,
	const gchar *path,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
);

/*
 *	Recovers a recycled object from the recycle bin.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	The index specifies the index of the recycled object to
 *	recover.
 *
 *	If alternate_recovery_path is not NULL then it specifies the
 *	full path to the directory that is to be used as the alternate
 *	recovery location for the recycled object, otherwise the
 *	recycled object's original location will be used as the
 *	recovery location.
 *
 *	If notify is TRUE then a recycled object removed signal will
 *	be queued for the specified recycled object and a VFS object
 *	added signal will be queued for the recovered object.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *      Returns 0 on success or non-zero on error. If the Recycle Bin
 *	is locked then this call will fail and return -6 with errno
 *	set to EBUSY.
 */
extern gint edv_recover(
	EDVContext *ctx,
	const gulong index,
	const gchar *alternate_recovery_path,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
);

/*
 *	Purges a recycled object from the recycle bin.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	The index specifies the recycled object to purge.
 *
 *	If notify is TRUE then a recycled object removed signal will
 *	be queued for the specified recycled object.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	Returns 0 on success or non-zero on error. If the Recycle Bin
 *	is locked then this call will fail and return -6 with errno
 *	set to EBUSY.
 */
extern gint edv_purge(
	EDVContext *ctx,
	const gulong index,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
);


#endif	/* EDV_RECYCLE_H */
