#include <errno.h>
#include <glib.h>

#include "edv_types.h"
#include "edv_utils.h"
#include "edv_path.h"
#include "edv_vfs_obj.h"
#include "edv_vfs_obj_stat.h"
#include "edv_recycled_obj.h"
#include "edv_recycled_obj_stat.h"
#include "edv_convert_obj.h"
#include "edv_recycle_bin_index.h"
#include "edv_notify.h"
#include "edv_recycle.h"
#include "edv_context_private.h"

#include "config.h"


const gchar *edv_recycle_get_error(EDVContext *ctx);
static void edv_recycle_set_error(
	EDVContext *ctx,
	const gchar *msg
);

gulong edv_recycle(
	EDVContext *ctx,
	const gchar *path,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
);

gint edv_recover(
	EDVContext *ctx,
	const gulong index,
	const gchar *alternate_recovery_path,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
);

gint edv_purge(
	EDVContext *ctx,
	const gulong index,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the last error message that occured when calling
 *	edv_recycle(), edv_recover(), or edv_purge().
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	Returns a statically allocated string describing the last
 *	error or NULL if there was no error.
 */
const gchar *edv_recycle_get_error(EDVContext *ctx)
{
	if(ctx == NULL)
		return(NULL);

	return(ctx->last_error_ptr);
}

/*
 *	Sets the last error message.
 */
static void edv_recycle_set_error(
	EDVContext *ctx,
	const gchar *msg
)
{
	if(msg == ctx->last_error_buf)
		return;

	g_free(ctx->last_error_buf);
	ctx->last_error_buf = STRDUP(msg);

	ctx->last_error_ptr = ctx->last_error_buf;
}


/*
 *	Recycles a VFS object.
 *
 *	This function always recycles the object and hence it does not
 *	check the EDV_CFG_PARM_DELETE_METHOD setting.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	The path specifies the VFS object to recycle. If the object is
 *	a directory then it must be empty.
 *
 *	If notify is TRUE then a VFS object removed signal will be
 *	queued for the specified object and a recycled object added
 *	signal will be queued for the recycled object.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	Returns the recycled object index or 0 on error. If the Recycle
 *	Bin is locked then this call will fail and return 0 with errno
 *	set to EBUSY.
 */
gulong edv_recycle(
	EDVContext *ctx,
	const gchar *path,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
)
{
	gint status;
	gchar		*cwd,
			*dpath,
			*parent_path;
	gulong		index,
			deleted_time;
	const gchar *index_path;
	EDVVFSObject *obj_lvalues;
	EDVRecycledObject *obj;


	if(ctx == NULL)
	{
		errno = EINVAL;
		return(0l);
	}

	edv_recycle_set_error(ctx, NULL);

	if(STRISEMPTY(path))
	{
		edv_recycle_set_error(
			ctx,
"VFS object to recycle was not specified"
		);
		errno = EINVAL;
		return(0l);
	}

	index_path = ctx->recycle_bin_index_path;
	if(STRISEMPTY(index_path))
	{
		edv_recycle_set_error(
			ctx,
"Recycle bin index file path was not defined on the context"
		);
		errno = EINVAL;
		return(0l);
	}

	/* Attempt to lock the recycle bin */
	status = edv_recycle_bin_index_lock(
		index_path,
		ctx->pid
	);
	if(status != 0)
	{
		const gint error_code = (gint)errno;
		switch(status)
		{
		    case -6:
			edv_recycle_set_error(
				ctx,
"Recycle Bin is currently locked and in use"
			);
			break;
		    default:
			edv_recycle_set_error(
				ctx,
"Unable to lock the recycle bin"
			);
			break;
		}
		errno = (int)error_code;
		return(0l);
	}

	cwd = edv_getcwd();
	dpath = edv_path_evaluate(cwd, path);
	deleted_time = edv_time();

	/* Get the local statistics of the VFS object to be recycled */
	obj_lvalues = edv_vfs_object_lstat(dpath);
	if(obj_lvalues == NULL)
	{
		const gint error_code = (gint)errno;
		edv_recycle_set_error(ctx, g_strerror(error_code));
		(void)edv_recycle_bin_index_unlock(
			index_path,
			ctx->pid
		);
		g_free(dpath);
		g_free(cwd);
		errno = (int)error_code;
		return(0l);
	}

	/* Get this object's location */
	parent_path = g_dirname(dpath);
	if(parent_path == NULL)
		parent_path = g_strdup("/");

	/* Create/convert a new recycled object based on the VFS
	 * object's values
	 */
	obj = edv_convert_vfs_object_to_recycled_object(obj_lvalues);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
		edv_recycle_set_error(ctx, g_strerror(error_code));
		(void)edv_recycle_bin_index_unlock(
			index_path,
			ctx->pid
		);
		edv_vfs_object_delete(obj_lvalues);
		g_free(parent_path);
		g_free(dpath);
		g_free(cwd);
		errno = (int)error_code;
		return(0);
	}

	/* Set the values on the recycled object that were not converted */
	obj->index = 0;
	obj->deleted_time = deleted_time;

	/* Add an entry to the recycled objects index file */
	index = edv_recycle_bin_index_add(
		index_path,
		obj
	);
	if(index != 0l)
	{
		/* Recycle this object */
		if(edv_recycle_bin_index_recycle(
			index_path,
			index,
			dpath,
			progress_cb, progress_data
		))
		{
			/* An error occured while recycling the object */
			gchar *s = STRDUP(edv_recycle_bin_index_get_error());
			edv_recycle_set_error(ctx, s);
			g_free(s);
			(void)edv_recycle_bin_index_remove(
				index_path,
				index
			);
			index = 0l;
		}
		/* Notify Endeavour about the recycled object being
		 * recycled?
		 */
		else if(notify)
		{
			edv_notify_queue_vfs_object_removed(
				ctx,
				dpath
			);
			edv_notify_queue_recycled_object_added(
				ctx,
				index
			);
		}
	}
	else
	{
		gchar *s = STRDUP(edv_recycle_bin_index_get_error());
		edv_recycle_set_error(ctx, s);
		g_free(s);
	}

	/* Unlock the recycle bin */
	(void)edv_recycle_bin_index_unlock(
		index_path,
		ctx->pid
	);

	edv_vfs_object_delete(obj_lvalues);
	edv_recycled_object_delete(obj);
	g_free(parent_path);
	g_free(dpath);
	g_free(cwd);

	return(index);
}

/*
 *	Recovers a recycled object from the recycle bin.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	The index specifies the index of the recycled object to
 *	recover.
 *
 *	If alternate_recovery_path is not NULL then it specifies the
 *	full path to the directory that is to be used as the alternate
 *	recovery location for the recycled object, otherwise the
 *	recycled object's original location will be used as the
 *	recovery location.
 *
 *	If notify is TRUE then a recycled object removed signal will
 *	be queued for the specified recycled object and a VFS object
 *	added signal will be queued for the recovered object.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	Returns 0 on success or non-zero on error. If the Recycle Bin
 *	is locked then this call will fail and return -6 with errno
 *	set to EBUSY.
 */
gint edv_recover(
	EDVContext *ctx,
	const gulong index,
	const gchar *alternate_recovery_path,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
)
{
	gint status;
	gchar *dpath;
	const gchar *index_path;
	EDVRecycledObject *obj;

	if(ctx == NULL)
	{
		errno = EINVAL;
		return(-2);
	}

	edv_recycle_set_error(ctx, NULL);

	index_path = ctx->recycle_bin_index_path;
	if(STRISEMPTY(index_path))
	{
		edv_recycle_set_error(
			ctx,
"Recycle bin index file path was not defined on the context"
		);
		errno = EINVAL;
		return(-2);
	}

	/* Attempt to lock the recycle bin */
	status = edv_recycle_bin_index_lock(
		index_path,
		ctx->pid
	);
	if(status != 0)
	{
		const gint error_code = (gint)errno;
		switch(status)
		{
		    case -6:
			edv_recycle_set_error(
				ctx,
"Recycle Bin is currently locked and in use"
			);
			break;
		    default:
			edv_recycle_set_error(
				ctx,
"Unable to lock the recycle bin"
			);
			break;
		}
		errno = (int)error_code;
		return(status);
	}

	/* Get the statistics of the recycled object to recover */
	obj = edv_recycled_object_stat(
		index_path,
		index
	);
	if(obj == NULL)
	{
		const gint error_code = (gint)errno;
		edv_recycle_set_error(ctx, g_strerror(error_code));
		(void)edv_recycle_bin_index_unlock(
			index_path,
			ctx->pid
		);
		errno = (int)error_code;
		return(-1);
	}

	/* Alternate recovery path specified? */
	if(alternate_recovery_path != NULL)
	{
		const gchar *original_name = obj->name;
		gchar	*cwd = edv_getcwd(),
			*parent_path = edv_path_evaluate(
				cwd,
				alternate_recovery_path
			);
		if((parent_path == NULL) || (original_name == NULL))
		{
			const gint error_code = (gint)errno;
			edv_recycle_set_error(ctx, g_strerror(error_code));
			(void)edv_recycle_bin_index_unlock(
				index_path,
				ctx->pid
			);
			edv_recycled_object_delete(obj);
			g_free(parent_path);
			g_free(cwd);
			errno = (int)error_code;
			return(-1);
		}
		dpath = g_strconcat(
			parent_path,
			G_DIR_SEPARATOR_S,
			original_name,
			NULL
		);
		g_free(cwd);
		g_free(parent_path);
	}
	else
	{
		/* Use the original location and name */
		dpath = NULL;
	}

	/* Recover the recycled object */
	status = edv_recycle_bin_index_recover(
		index_path,
		index,
		dpath,				/* Full path to the recovered
						 * object */
		progress_cb, progress_data
	);
	if(status != 0)
	{
		/* An error occured while recovering the object */
		gchar *s = STRDUP(edv_recycle_bin_index_get_error());
		edv_recycle_set_error(ctx, s);
		g_free(s);
	}
	else
	{
		/* Success, now remove the recycled object entry from the
		 * recycled objects index file
		 */
		(void)edv_recycle_bin_index_remove(
			index_path,
			index
		);

		/* Notify Endeavour about the recycled object being
		 * recovered?
		 */
		if(notify)
		{
			gchar *new_path;
			if(dpath != NULL)
				new_path = g_strdup(dpath);
			else
				new_path = edv_paths_join(obj->original_path, obj->name);
			edv_notify_queue_vfs_object_added(
				ctx,
				new_path
			);
			edv_notify_queue_recycled_object_removed(
				ctx,
				index
			);
			g_free(new_path);
		}
	}

	/* Unlock the recycle bin */
	(void)edv_recycle_bin_index_unlock(
		index_path,
		ctx->pid
	);

	edv_recycled_object_delete(obj);
	g_free(dpath);

	return(status);
}

/*
 *	Purges a recycled object from the recycle bin.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	The index specifies the recycled object to purge.
 *
 *	If notify is TRUE then a recycled object removed signal will
 *	be queued for the specified recycled object.
 *
 *	If progress_cb is not NULL then it will be called during the
 *	operation to report the progress. If it returns non-zero then
 *	the operation will be aborted.
 *
 *	The progress_data specifies the user data which will be passed
 *	to the progress_cb function.
 *
 *	Returns 0 on success or non-zero on error. If the Recycle Bin
 *	is locked then this call will fail and return -6 with errno
 *	set to EBUSY.
 */
gint edv_purge(
	EDVContext *ctx,
	const gulong index,
	const gboolean notify,
	gint (*progress_cb)(
		gpointer progress_data,
		const gulong i, const gulong n
	),
	gpointer progress_data
)
{
	gint status;
	const gchar *index_path;

	if(ctx == NULL)
	{
		errno = EINVAL;
		return(-2);
	}

	edv_recycle_set_error(ctx, NULL);

	index_path = ctx->recycle_bin_index_path;
	if(STRISEMPTY(index_path))
	{
		edv_recycle_set_error(
			ctx,
"Recycle bin index file path was not defined on the context"
		);
		errno = EINVAL;
		return(-2);
	}

	/* Attempt to lock the recycle bin */
	status = edv_recycle_bin_index_lock(
		index_path,
		ctx->pid
	);
	if(status != 0)
	{
		const gint error_code = (gint)errno;
		switch(status)
		{
		    case -6:
			edv_recycle_set_error(
				ctx,
"Recycle Bin is currently locked and in use"
			);
			break;
		    default:
			edv_recycle_set_error(
				ctx,
"Unable to lock the recycle bin"
			);
			break;
		}
		errno = (int)error_code;
		return(status);
	}

	/* Purge the recycled object */
	status = edv_recycle_bin_index_purge(
		index_path,
		index,
		progress_cb, progress_data
	);
	if(status != 0)
	{
		/* An error occured while purging the object */
		gchar *s = STRDUP(edv_recycle_bin_index_get_error());
		edv_recycle_set_error(ctx, s);
		g_free(s);
	}
	else
	{
		/* Success, now remove the recycled object entry from the
		 * recycled objects index file
		 */
		(void)edv_recycle_bin_index_remove(
			index_path,
			index
		);

		/* Notify Endeavour about the recycled object being
		 * purged?
		 */
		if(notify)
			edv_notify_queue_recycled_object_removed(
				ctx,
				index
			);
	}

	/* Unlock the recycle bin */
	(void)edv_recycle_bin_index_unlock(
		index_path,
		ctx->pid
	);

	return(status);
}
