#define _GNU_SOURCE				/* For FNM_CASEFOLD */
#include <stdio.h>
#include <string.h>
#include <fnmatch.h>
#include <errno.h>
#include <glib.h>

#include "../../include/fio.h"

#include "edv_utils.h"
#include "edv_property.h"
#include "edv_vfs_obj.h"
#include "edv_vfs_obj_stat.h"
#include "edv_property_fio.h"

#include "config.h"


/* Read Stream */
static gchar *edv_properties_list_stream_get_line(
	FILE *fp,
	const gboolean evaluate_escape_sequences,
	const gboolean concat_escaped_newlines
);
#define EDV_GET_STREAM_LINE_EVAL(_fp_)			\
edv_properties_list_stream_get_line(			\
	(_fp_), TRUE, FALSE				\
)
#define EDV_GET_STREAM_LINE_RAW(_fp_)			\
edv_properties_list_stream_get_line(			\
	(_fp_), FALSE, FALSE				\
)
#define EDV_GET_STREAM_LINE_FULL(_fp_)			\
edv_properties_list_stream_get_line(			\
	(_fp_), FALSE, TRUE				\
)

#define EDV_GET_STREAM_PARAMETER(_fp_,_last_line_)	\
(gchar *)FSeekNextParm(					\
 (_fp_),						\
 (char *)(_last_line_),					\
 EDV_CFG_COMMENT_CHAR,					\
 EDV_CFG_DELIMINATOR_CHAR				\
)

#define EDV_SEEK_NEXT_LINE(_fp_)			\
FSeekNextLine(_fp_)

/* Parsers */
static gchar *edv_properties_list_check_parse_section_name(const gchar *line);
static gchar *edv_properties_list_parse_parameter(const gchar *line);

/* Get */
gchar *edv_properties_list_stream_get_s(
	FILE *fp,
	const gchar *section,
	const gchar *name
);
gchar *edv_properties_list_file_get_s(
	const gchar *path,
	const gchar *section,
	const gchar *name
);

GList *edv_properties_list_stream_get_list(
	FILE *fp,
	const gchar *section,
	const gchar *name_filter
);
GList *edv_properties_list_file_get_list(
	const gchar *path,
	const gchar *section,
	const gchar *name_filter
);

/* Set */
static void edv_properties_list_stream_write_section(
	FILE *fp,
	const gchar *section
);
static void edv_properties_list_stream_write_property(
	FILE *fp,
	const gchar *name,
	const gchar *value
);
gint edv_properties_list_stream_set_s(
	FILE *in_fp,
	FILE *out_fp,
	const gchar *section,
	const gchar *name,
	const gchar *value
);
gint edv_properties_list_file_set_s(
	const gchar *path,
	const gchar *section,
	const gchar *name,
	const gchar *value
);

/* Remove */
gint edv_properties_list_stream_remove(
	FILE *in_fp,
	FILE *out_fp,
	const gchar *section,
	const gchar *name 
);
gint edv_properties_list_file_remove(
	const gchar *path,
	const gchar *section,
	const gchar *name 
);

gint edv_properties_list_stream_remove_section(
	FILE *in_fp,
	FILE *out_fp,
	const gchar *section
);
gint edv_properties_list_file_remove_section(
	const gchar *path,
	const gchar *section
);

/* Clean */
static gint edv_properties_list_count_section_properties(FILE *fp);
void edv_properties_list_stream_clean(
	FILE *in_fp,
	FILE *out_fp,
	const gboolean remove_empty_sections,
	const gboolean remove_comments,
	const gboolean remove_empty_lines
);
void edv_properties_list_file_clean(
	const gchar *path,
	const gboolean remove_empty_sections,
	const gboolean remove_comments,
	const gboolean remove_empty_lines
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))


/*
 *	Reads one entire line from the stream, excluding the ending
 *	newline character, and positions the stream at the start of
 *	the next line or end of file.
 *
 *	If evaluate_escape_sequences is TRUE then all escape
 *	characters will be evaluated and converted before storing
 *	to the return string.
 *
 *	If concat_escaped_newlines is TRUE and
 *	evaluate_escape_sequences is FALSE then any escaped newline
 *	characters will be included in the return string along with
 *	their preceeding escape character and the stream will be
 *	read until the first unescaped newline character is encountered
 *	or end of file.
 *
 *	If both evaluate_escape_sequences and concat_escaped_newlines
 *	are FALSE then the line will be read as-is from the stream
 *	without any escape characters being evaluated or converted.
 *
 *	Returns a dynamically allocated string describing the line
 *	that was read from the stream or NULL on end of file or error.
 */
static gchar *edv_properties_list_stream_get_line(
	FILE *fp,
	const gboolean evaluate_escape_sequences,
	const gboolean concat_escaped_newlines
)
{
	if(fp == NULL)
		return(NULL);

	if(evaluate_escape_sequences)
	{
		return((gchar *)FGetString(fp));
	}
	else if(concat_escaped_newlines)
	{
		gchar	*s,
					*line = (gchar *)FGetStringLiteral(fp);
		while(line != NULL)
		{
			/* Seek s to the end of the line and check if it ends
			 * with an escape character
			 */
			s = line;
			if(*s != '\0')
			{
				s++;
				while(*s != '\0')
					s++;
				s--;
				if(*s == '\\')
				{
					gchar	*s,
							*line_next = (gchar *)FGetStringLiteral(fp);
					if(line_next == NULL)
						return(line);

					s = g_strconcat(
						line,
						"\n",
						line_next,
						NULL
					);
					g_free(line_next);
					g_free(line);
					line = s;
					continue;
				}
			}
			break;
		}
		return(line);
	}
	else
	{
		return((gchar *)FGetStringLiteral(fp));
	}
}

/*
 *	Parses the section heading name from the line.
 *
 *	If the line does not contain a section heading then NULL is
 *	returned.
 */
static gchar *edv_properties_list_check_parse_section_name(const gchar *line)
{
	gint len;
	const gchar	*s = line,
					*s_end;
	gchar *section;

	if(s == NULL)
		return(NULL);

	while(ISBLANK(*s))
		s++;

	/* Is the first non-blank character not a section heading
	 * start deliminator character?
	 */
	if(*s != '[')
		return(NULL);

	/* Seek s past the section heading start deliminator */
	s++;

	/* Seek s_end to the end of the section heading */
	s_end = s;
	while((*s_end != '\0') && (*s_end != ']'))
		s_end++;

	len = s_end - s;
	section = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	if(section == NULL)
		return(NULL);

	if(len > 0)
		(void)memcpy(
			section,
			s,
			(size_t)(len * sizeof(gchar))
		);
	section[len] = '\0';

	return(g_strstrip(section));
}

/*
 *	Parses the paramaeter from the line.
 */
static gchar *edv_properties_list_parse_parameter(const gchar *line)
{
	gint len;
	const gchar	*s = line,
					*s_end;
	gchar *parm;

	if(s == NULL)
		return(NULL);

	while(ISBLANK(*s))
		s++;

	s_end = (const gchar *)strchr(
		(const char *)s,
		EDV_CFG_DELIMINATOR_CHAR
	);
	if(s_end == NULL)
	{
		/* No deliminator found between the parameter and value
		 * so just copy the entire line and return it as the
		 * parameter
		 */
		parm = g_strdup(s);
		if(parm == NULL)
			return(NULL);

		return(g_strstrip(parm));
	}

	len = s_end - s;
	parm = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	if(parm == NULL)
		return(NULL);

	if(len > 0)
		(void)memcpy(
			parm,
			s,
			(size_t)(len * sizeof(gchar))
		);
	parm[len] = '\0';

	return(g_strstrip(parm));
}


/*
 *	Gets the property value from a stream.
 *
 *	The fp specifies the stream which should be positioned at
 *	at the starting point of the search.
 *
 *	The section specifies the name of the section to get the
 *	property value from. If section is NULL then the property
 *	value will be obtained from its first occurance regardless
 *	of which section it is in.
 *
 *	The name specifies the property name.
 *
 *	Returns a dynamically allocated string describing the value or
 *	NULL on error.
 */
gchar *edv_properties_list_stream_get_s(
	FILE *fp,
	const gchar *section,
	const gchar *name
)
{
	if((fp == NULL) || STRISEMPTY(name))
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Get the property value from its first occurance regardless
	 * of which section it is in?
	 */
	if(STRISEMPTY(section))
	{
		gchar	*parm = NULL,
					*value = NULL;

		/* Look for the first occurance of the specified property
		 * and get its value
		 */
		while(!feof(fp))
		{
			/* Read the next parameter */
			parm = EDV_GET_STREAM_PARAMETER(fp, parm);
			if(parm == NULL)
				break;

			/* Does this parameter match the specified property name? */
			if(!g_strcasecmp(parm, name))
			{
				value = EDV_GET_STREAM_LINE_FULL(fp);
				break;
			}
			else
			{
				/* Ignore this value and seek to the start of the
				 * next line
				 */
				EDV_SEEK_NEXT_LINE(fp);
			}
		}
		g_free(parm);

		errno = (value != NULL) ? 0 : ENOENT;

		return(value);
	}
	else
	{
		/* Seek to the specified section first and then get the
		 * property's value
		 */
		gchar	*s,
					*line = NULL,
					*parm = NULL,
					*value = NULL;

		/* Seek to the specified section */
		while(!feof(fp))
		{
			/* Get this entire line and seek to the start of the
			 * next line
			 */
			g_free(line);
			line = EDV_GET_STREAM_LINE_FULL(fp);
			if(line == NULL)
				break;

			/* Does this line contain a section heading? */
			s = edv_properties_list_check_parse_section_name(line);
			if(s != NULL)
			{
				/* Now s points to this section name with spaces
				 * stripped
				 *
				 * Does this section name match the specified
				 * section name?
				 */
				if(!g_strcasecmp(s, section))
				{
					g_free(s);
					break;
				}
				g_free(s);
			}
		}
		g_free(line);
/*	    line = NULL; */

		/* At this point fp should be at the start of the first
		 * parameter of the specified section or at the end of
		 * the stream
		 *
		 * Look for the specified property and get its value
		 */
		while(!feof(fp))
		{
			/* Read the next parameter */
			parm = EDV_GET_STREAM_PARAMETER(fp, parm);
			if(parm == NULL)
				break;

			/* Next section reached?
			 *
			 * Note that it is safe to use
			 * edv_properties_list_check_parse_section_name() here
			 * because even though EDV_GET_STREAM_PARAMETER() will only
			 * return the characters before the space,
			 * edv_properties_list_check_parse_section_name() will
			 * only be used to check if it starts with a section
			 * heading deliminator even though it will not be
			 * able to get the entire section heading name
			 */
			s = edv_properties_list_check_parse_section_name(parm);
			if(s != NULL)
			{
				/* Stop reading, thus ignoring any subsequent
				 * parameters, when the next section has been
				 * reached, this is so that we do not incorrectly
				 * obtain the a value from a subsequent section
				 */
				g_free(s);
				break;
			}

			/* Does this parameter match the specified property name? */
			if(!g_strcasecmp(parm, name))
			{
				value = EDV_GET_STREAM_LINE_FULL(fp);
				break;
			}
			else
			{
				/* Ignore this value and seek to the start of the
				 * next line
				 */
				EDV_SEEK_NEXT_LINE(fp);
			}
		}
		g_free(parm);

		errno = (value != NULL) ? 0 : ENOENT;

		return(value);
	}
}

/*
 *	Gets the property value from a file.
 *
 *	The path specifies the properties list file.
 *
 *	The section specifies the name of the section to get the
 *	property value from. If section is NULL then the property
 *	value will be obtained from its first occurance regardless
 *	of which section it is in.
 *
 *	The name specifies the property name.
 *
 *	Returns a dynamically allocated string describing the value or
 *	NULL on error.
 */
gchar *edv_properties_list_file_get_s(
	const gchar *path,
	const gchar *section,
	const gchar *name
)
{
	FILE *fp;
	gchar *value;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(NULL);
	}

	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
		return(NULL);

	value = edv_properties_list_stream_get_s(
		fp,
		section,
		name
	);

	(void)fclose(fp);

	return(value);
}


/*
 *	Gets the property values from a stream.
 *
 *	The fp specifies the stream which should be positioned at
 *	at the starting point of the search.
 *
 *	The section specifies the name of the section to get the
 *	property values from. If section is NULL then the property
 *	values from all the sections will be obtained.
 *
 *	The name_filter specifies which property values to get. If
 *	name_filter is NULL then all the properties will be obtained.
 *
 *	Returns a GList of EDVProperty properties or NULL on error.
 */
GList *edv_properties_list_stream_get_list(
	FILE *fp,
	const gchar *section,
	const gchar *name_filter
)
{
	GList *properties_list;

	if(fp == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	properties_list = NULL;

	/* Get properties from all sections? */
	if(STRISEMPTY(section))
	{
		gchar	*s,
					*parm = NULL;
		while(!feof(fp))
		{
			parm = EDV_GET_STREAM_PARAMETER(fp, parm);
			if(parm == NULL)
				break;

			/* Is this a section heading?
			 *
			 * Note that it is safe to use
			 * edv_properties_list_check_parse_section_name() here
			 * because even though EDV_GET_STREAM_PARAMETER() will
			 * only return the characters before the space,
			 * edv_properties_list_check_parse_section_name() will
			 * only be used to check if it starts with a section
			 * heading deliminator even though it will not be
			 * able to get the entire section heading name
			 */
			s = edv_properties_list_check_parse_section_name(parm);
			if(s != NULL)
			{
				/* Do not add this to the properties list */
				g_free(s);
				EDV_SEEK_NEXT_LINE(fp);
				continue;
			}

			/* Check if this property's name is in the specified
			 * name filter or should we get all properties
			 */
			if(STRISEMPTY(name_filter) ?
				TRUE :
				(fnmatch(
					(const char *)name_filter,
					(const char *)parm,
					FNM_NOESCAPE |
#ifdef FNM_CASEFOLD
					FNM_CASEFOLD
#else
					0
#endif
				) == 0)
			)
			{
				/* Get this property */
				EDVProperty *prop = edv_property_new();
				if(prop != NULL)
				{
					prop->name = g_strdup(parm);
					prop->value = EDV_GET_STREAM_LINE_FULL(fp);
					properties_list = g_list_append(
						properties_list,
						prop
					);
				}
				else
				{
					EDV_SEEK_NEXT_LINE(fp);
				}
			}
			else
			{
				EDV_SEEK_NEXT_LINE(fp);
			}
		}
		g_free(parm);
	}
	else
	{
		/* Get properties from the specified section */
		gboolean in_section = FALSE;
		gchar	*s,
					*parm = NULL;
		while(!feof(fp))
		{
			/* Are we currently in the specified section? */
			if(in_section)
			{
				/* Read the next parameter */
				parm = EDV_GET_STREAM_PARAMETER(fp, parm);
				if(parm == NULL)
					break;

				/* Is this a new section heading?
				 *
				 * Note that it is safe to use
				 * edv_properties_list_check_parse_section_name() here
				 * because even though EDV_GET_STREAM_PARAMETER()
				 * will only return the characters before the
				 * space, edv_properties_list_check_parse_section_name()
				 * will only be used to check if it starts with a
				 * section heading deliminator even though it will
				 * not be able to get the entire section heading
				 * name
				 */
				s = edv_properties_list_check_parse_section_name(parm);
				if(s != NULL)
				{
					/* Mark that we are no longer in the specified
					 * section
					 */
					in_section = FALSE;
					g_free(s);
					EDV_SEEK_NEXT_LINE(fp);
					continue;
				}

				/* Check if this property's name is in the specified
				 * name filter or should we get all properties
				 */
				if(STRISEMPTY(name_filter) ?
					TRUE :
					(fnmatch(
						(const char *)name_filter,
						(const char *)parm,
						FNM_NOESCAPE |
#ifdef FNM_CASEFOLD
						FNM_CASEFOLD
#else
						0
#endif
					) == 0)
				)
				{
					/* Get this property */
					EDVProperty *prop = edv_property_new();
					if(prop != NULL)
					{
						prop->name = g_strdup(parm);
						prop->value = EDV_GET_STREAM_LINE_FULL(fp);
						properties_list = g_list_append(
							properties_list,
							prop
						);
					}
					else
					{
						EDV_SEEK_NEXT_LINE(fp);
					}
				}
				else
				{
					EDV_SEEK_NEXT_LINE(fp);
				}
		    }
			else
			{
				/* Not in the specified section, read the next line */
				g_free(parm);
				parm = EDV_GET_STREAM_LINE_FULL(fp);
				if(parm == NULL)
					break;

				/* Is this parameter the start of a new section? */
				s = edv_properties_list_check_parse_section_name(parm);
				if(s != NULL)
				{
					/* Is this section the specified section? */
					if(!g_strcasecmp(s, section))
						in_section = TRUE;
					else
						in_section = FALSE;
					g_free(s);
				}
			}
		}
		g_free(parm);
	}

	return(properties_list);
}


/*
 *	Gets the property values from a file.
 *
 *	The path specifies the properties list file.
 *
 *	The section specifies the name of the section to get the
 *	property values from. If section is NULL then the property
 *	values from all the sections will be obtained.
 *
 *	The name_filter specifies which property values to get. If
 *	name_filter is NULL then all the properties will be obtained.
 *
 *	Returns a GList of EDVProperty properties or NULL on error.
 */
GList *edv_properties_list_file_get_list(
	const gchar *path,
	const gchar *section,
	const gchar *name_filter
)
{
	FILE *fp;
	GList *properties_list;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(NULL);
	}

	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
		return(NULL);

	properties_list = edv_properties_list_stream_get_list(
		fp,
		section,
		name_filter
	);

	(void)fclose(fp);

	return(properties_list);
}

/*
 *	Writes the section to the stream.
 */
static void edv_properties_list_stream_write_section(
	FILE *fp,
	const gchar *section
)
{
	(void)fprintf(
		fp,
		"[%s]\n",
		section
	);
}

/*
 *	Writes the property to the stream.
 */
static void edv_properties_list_stream_write_property(
	FILE *fp,
	const gchar *name,
	const gchar *value
)
{
	(void)fprintf(
		fp,
		"%s = %s\n",
		name,
		value
	);
}

/*
 *	Sets the property value to a stream.
 *
 *	If in_fp is not NULL then it specifies the stream of the
 *	original properties list which must be opened for reading.
 *
 *	The out_fp specifies the stream to write the new properties
 *	list to which must be opened for writing or append.
 *
 *	The section specifies the name of the section to set the
 *	property under. If section is NULL then the property will
 *	be appened.
 *
 *	The name specifies the property name.
 *
 *	The value specifies the property value. If value is NULL then
 *	the property will be removed.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_properties_list_stream_set_s(
	FILE *in_fp,
	FILE *out_fp,
	const gchar *section,
	const gchar *name,
	const gchar *value
)
{
	if((out_fp == NULL) || STRISEMPTY(name))
	{
		errno = EINVAL;
		return(-2);
	}

	/* No source properties list? */
	if(in_fp == NULL)
	{
		/* Just write the specified section and property to the
		 * output stream
		 */

		/* Write the section if it was specified */
		if(!STRISEMPTY(section))
			edv_properties_list_stream_write_section(
				out_fp,
				section
			);

		/* Write the property if it was specified */
		if(!STRISEMPTY(value))
			edv_properties_list_stream_write_property(
				out_fp,
				name,
				value
			);
	}
	/* No section specified? */
	else if(STRISEMPTY(section))
	{
		/* Copy the input list to the output list while checking
		 * for a property that matches the specified property, if
		 * one is encountered in the input list then the specified
		 * property will be written (or omitted if the specified
		 * property value is empty) to the output list instead of
		 * the input list's property.
		 */
		gboolean	wrote_this_property = FALSE;
		gchar	*s,
					*line = NULL;
		while(!feof(in_fp))
		{
			/* Get the next line exactly as is from the specified
			 * stream because we will need to write it exactly as
			 * it was to the output stream
			 */
			g_free(line);
			line = EDV_GET_STREAM_LINE_FULL(in_fp);
			if(line == NULL)
				break;

#define WRITE_LINE(_fp_,_line_)	{	\
 (void)fprintf(				\
  (_fp_),				\
  "%s\n",				\
  (_line_)				\
 );					\
}

			/* Parse the parameter from this line */
			s = edv_properties_list_parse_parameter(line);
			if(s != NULL)
			{
				/* Does this parameter match the specified
				 * property's name?
				 */
				if(!g_strcasecmp(s, name))
				{
					/* Already wrote the specified property to the
					 * output list?
					 */
					if(wrote_this_property)
					{
						/* Omit this subsequent occurance of the
						 * property from the output list
						 */
					}
					else
					{
						/* Was the specified property's value
						 * not specified?
						 */
						if(STRISEMPTY(value))
						{
							/* Omit the property from the output
							 * list, which effectively removes it
							 */
						}
						else
						{
							/* Write the specified property to the
							 * output list, which effectively sets
							 * its value
							 */
							edv_properties_list_stream_write_property(
								out_fp,
								name,
								value
							);
						}

						/* Mark that the specified property has
						 * now been either written to or removed
						 * from the output list
						 */
						wrote_this_property = TRUE;
					}
				}
				else
				{
					/* This is not the specified property so just
					 * write it exactly as is to the output list
					 */
					WRITE_LINE(out_fp, line);
				}
				g_free(s);
			}
			else
			{
				/* Unable to parse this line so just write it
				 * exactly as is to the output list
				 */
				WRITE_LINE(out_fp, line);
			}
#undef WRITE_LINE
		}
		g_free(line);
/*	    line = NULL; */

		/* Did not write or remove the specified property to the
		 * output list?
		 */
		if(!wrote_this_property)
		{
			/* Write the specified property to the output list if
			 * its value was specified
			 */
			if(!STRISEMPTY(value))
			{
				edv_properties_list_stream_write_property(
					out_fp,
					name,
					value
				);

				/* Mark that the specified property has now been
				 * either written to or removed from the output
				 * list
				 */
				wrote_this_property = TRUE;
			}
		}
	}
	else
	{
		/* Copy the input list to the output list exactly */
		gboolean	in_section = FALSE,
					section_encountered = FALSE,
					wrote_this_property = FALSE;
		gchar	*s,
					*line = NULL;
		while(!feof(in_fp))
		{
			/* Get the next line exactly as is from the specified
			 * stream because we will need to write it exactly as
			 * it was to the output stream
			 */
			g_free(line);
			line = EDV_GET_STREAM_LINE_FULL(in_fp);
			if(line == NULL)
				break;

#define WRITE_LINE(_fp_,_line_)	{	\
 (void)fprintf(				\
  (_fp_),				\
  "%s\n",				\
  (_line_)				\
 );					\
}

			/* Does this line contain a section heading? */
			s = edv_properties_list_check_parse_section_name(line);
			if(s != NULL)
			{
				/* Does this section's name match the specified
				 * section's name?
				 */
				if(!g_strcasecmp(s, section))
				{
					/* We are now in the specified section, check
					 * if we have encountered the specified
					 * section before
					 */
					if(section_encountered)
					{
						/* Duplicate section matching the
						 * specified section was encountered,
						 * this is acceptable but not good
						 */
					}
					else
					{
						/* Mark that we have now encountered the
						 * specified section
						 */
						section_encountered = TRUE;
					}

					/* Mark that we are now in the specified section */
					in_section = TRUE;
				}
				else
				{
					/* Not the specified section
					 *
					 * Check if we were just in the specified
					 * section but did not write the specified
					 * property to the output list?
					 */
					if(in_section && !wrote_this_property)
					{
						/* Was the specified property's value
						 * not specified?
						 */
						if(STRISEMPTY(value))
						{
							/* Omit the property from the output
							 * list, which effectively removes it
							 */
						}
						else
						{
							/* Write the specified property to the
							 * output list, which effectively sets
							 * its value
							 */
							edv_properties_list_stream_write_property(
								out_fp,
								name,
								value
							);
						}

						/* Mark that the specified property has
						 * now been either written to or removed
						 * from the output list
						 */
						wrote_this_property = TRUE;
					}

					/* Mark that we are no longer in the specified
					 * section
					 */
					in_section = FALSE;
				}

				/* Write this section's heading to the output
				 * list, note that we need to do this after
				 * writing the specifed property to the output
				 * list so that the specified property does not
				 * go in to wrong section
				 */
				WRITE_LINE(out_fp, line);

				g_free(s);
			}
			else
			{
				/* This line does not contain a section heading
				 *
				 * Are we currently in the specified section?
				 */
				if(in_section)
				{
					/* Get the parameter from this line */
					s = edv_properties_list_parse_parameter(line);
					if(s != NULL)
					{
						/* Does this parameter match the specified
						 * property name?
						 */
						if(!g_strcasecmp(s, name))
						{
							/* Was the specified property's value
							 * not specified?
							 */
							if(STRISEMPTY(value))
							{
								/* Omit the property from the
								 * output list, which effectively
								 * removes it
								 */
							}
							else
							{
								/* Write the specified property to
								 * the output list, which
								 * effectively sets its value
								 */
								edv_properties_list_stream_write_property(
									out_fp,
									name,
									value
								);
							}

							/* Mark that the specified property
							 * has now been either written to or
							 * removed from the output list
							 */
							wrote_this_property = TRUE;
						}
						else
						{
							/* Not the specified property so just
							 * copy this line exactly
							 */
							WRITE_LINE(out_fp, line);
						}
						g_free(s);
					}
					else
					{
						/* Unable to parse the parameter so just
						 * copy this line exactly
						 */
						WRITE_LINE(out_fp, line);
					}
				}
				else
				{
					/* Not in the specified section so just copy
					 * this line exactly
					 */
					WRITE_LINE(out_fp, line);
				}
			}
#undef WRITE_LINE
		}
		g_free(line);
/*	    line = NULL; */

		/* Did not write or remove the specified property to the
		 * output list?
		 */
		if(!wrote_this_property)
		{
			/* Write the specified property to the output list its
			 * value was specified
			 */
			if(!STRISEMPTY(value))
			{
				/* If the specified section was never encountered
				 * then write the specified section heading to the
				 * output list before the specified property
				 */
				if(!section_encountered)
					edv_properties_list_stream_write_section(
						out_fp,
						section
					);

				/* Write the specified property to the output list */
				edv_properties_list_stream_write_property(
					out_fp,
					name,
					value
				);

				/* Mark that the specified property has now been
				 * either written to or removed from the output
				 * list
				 */
				wrote_this_property = TRUE;
			}
		}
	}

	return(0);
}

/*
 *	Sets the property value to a file.
 *
 *	The path specifies the properties list file. If the properties
 *	list file does not exist then a new one will be created.
 *
 *	The section specifies the name of the section to set the
 *	property under. If section is NULL then the property will
 *	be appened.
 *
 *	The name specifies the property name.
 *
 *	The value specifies the property value. If value is NULL then
 *	the property will be removed.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_properties_list_file_set_s(
	const gchar *path,
	const gchar *section,
	const gchar *name,
	const gchar *value
)
{
	FILE		*in_fp,
			*out_fp;
	gint status;
	const gchar *in_path = path;
	gchar		*parent,
			*out_path;
	EDVPermissionFlags permissions;

	if(STRISEMPTY(in_path))
	{
		errno = EINVAL;
		return(-2);
	}

	/* Get the parent directory of the specified path */
	if(g_path_is_absolute(in_path))
		parent = g_dirname(in_path);
	else
		parent = edv_getcwd();
	if(parent == NULL)
		return(-1);

	/* Create a temporary file name for the output file in the
	 * parent directory of the specified path
	 */
	out_path = edv_tmp_name(parent);

	/* Delete the parent directory path */
	g_free(parent);

	if(out_path == NULL)
		return(-1);

	/* Open the specified file for reading */
	in_fp = fopen((const char *)in_path, "rb");
	if(in_fp != NULL)
	{
		EDVVFSObject *obj = edv_vfs_object_fstat((gint)fileno(in_fp));
		if(obj != NULL)
		{
			permissions = obj->permissions;
			edv_vfs_object_delete(obj);
		}
		else
		{
			/* Get the default permissions */
			permissions = (~edv_permissions_get_default()) &
				(EDV_PERMISSION_UR | EDV_PERMISSION_UW |
				 EDV_PERMISSION_GR | EDV_PERMISSION_GW |
				 EDV_PERMISSION_OR | EDV_PERMISSION_OW);

		}
	}
	else
	{
		const gint error_code = (gint)errno;
		if(error_code == ENOENT)
		{
			/* It is okay if the specified file does not exist
			 *
			 * Get the default permissions
			 */
			permissions = (~edv_permissions_get_default()) &
				(EDV_PERMISSION_UR | EDV_PERMISSION_UW |
				 EDV_PERMISSION_GR | EDV_PERMISSION_GW |
				 EDV_PERMISSION_OR | EDV_PERMISSION_OW);
		}
		else
		{
			return(-1);
		}
	}

	/* Create the output file for writing */
	out_fp = fopen((const char *)out_path, "wb");
	if(out_fp == NULL)
	{
		const gint error_code = (gint)errno;
		if(in_fp != NULL)
			(void)fclose(in_fp);
		g_free(out_path);
		errno = (int)error_code;
		return(-1);
	}

	/* Set the property value */
	status = edv_properties_list_stream_set_s(
		in_fp,
		out_fp,
		section,
		name,
		value
	);
	if(status != 0)
	{
		const gint error_code = (gint)errno;
		if(in_fp != NULL)
			(void)fclose(in_fp);
		(void)fclose(out_fp);
		g_free(out_path);
		errno = (int)error_code;
		return(status);
	}

	/* Close the specified file */
	if(in_fp != NULL)
		(void)fclose(in_fp);

	/* Close the output file */
	if(fclose(out_fp))
	{
		/* An error occured while writing ot the output file */
		const gint error_code = (gint)errno;
		g_free(out_path);
		errno = (int)error_code;
		return(-1);
	}

	/* Remove the specified file */
	if(edv_unlink(in_path))
	{
		if(errno != ENOENT)
		{
			const gint error_code = (gint)errno;
			g_free(out_path);
			errno = (int)error_code;
			return(-1);
		}
	}

	/* Rename the output file to the name of the specified file */
	if(edv_rename(
		out_path,			/* Source */
		in_path				/* Target */
	))
	{
		const gint error_code = (gint)errno;
		g_free(out_path);
		errno = (int)error_code;
		return(-1);
	}

	/* Restore the permissions */
	(void)edv_permissions_set(
		in_path,
		permissions
	);

	g_free(out_path);

	return(0);
}


/*
 *	Removes the property value from a stream.
 *
 *	If in_fp is not NULL then it specifies the stream of the
 *	original properties list which must be opened for reading.
 *
 *	The out_fp specifies the stream to write the new properties
 *	list to which must be opened for writing or append.
 *
 *	The section specifies the name of the section to set the
 *	property under. If section is NULL then the property will
 *	be appened.
 *
 *	The name specifies the property name.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_properties_list_stream_remove(
	FILE *in_fp,
	FILE *out_fp,
	const gchar *section,
	const gchar *name
)
{
	return(edv_properties_list_stream_set_s(
		in_fp,
		out_fp,
		section,
		name,
		NULL				/* Remove */
	));
}

/*
 *	Removes the property value from a file.
 *
 *	The path specifies the properties list file. If the properties
 *	list file does not exist then a new one will be created.
 *
 *	The section specifies the name of the section to set the
 *	property under. If section is NULL then the property will
 *	be appened.
 *
 *	The name specifies the property name.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_properties_list_file_remove(
	const gchar *path,
	const gchar *section,
	const gchar *name 
)
{
	return(edv_properties_list_file_set_s(
		path,
		section,
		name,
		NULL				/* Remove */
	));
}


/*
 *	Removes the section and any properties in it from a stream.
 *
 *	If in_fp is not NULL then it specifies the stream of the
 *	original properties list which must be opened for reading.
 *
 *	The out_fp specifies the stream to write the new properties
 *	list to which must be opened for writing or append.
 *
 *	The section specifies the name of the section to remove.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_properties_list_stream_remove_section(
	FILE *in_fp,
	FILE *out_fp,
	const gchar *section
)
{
	if((out_fp == NULL) || STRISEMPTY(section))
	{
		errno = EINVAL;
		return(-2);
	}

	/* No source properties list? */
	if(in_fp == NULL)
	{
		/* Do nothing */
	}
	else
	{
		/* Copy the input list to the output list exactly */
		gboolean	in_section = FALSE,
					section_encountered = FALSE;
		gchar	*s,
					*line = NULL;
		while(!feof(in_fp))
		{
			/* Get the next line exactly as is from the specified
			 * stream because we will need to write it exactly as
			 * it was to the output stream
			 */
			g_free(line);
			line = EDV_GET_STREAM_LINE_FULL(in_fp);
			if(line == NULL)
				break;

#define WRITE_LINE(_fp_,_line_)	{	\
 (void)fprintf(				\
  (_fp_),				\
  "%s\n",				\
  (_line_)				\
 );					\
}

			/* Does this line contain a section heading? */
			s = edv_properties_list_check_parse_section_name(line);
			if(s != NULL)
			{
				/* Does this section's name match the specified
				 * section's name?
				 */
				if(!g_strcasecmp(s, section))
				{
					/* We are now in the specified section, check
					 * if we have encountered the specified
					 * section before
					 */
					if(section_encountered)
					{
						/* Duplicate section matching the
						 * specified section was encountered,
						 * this is acceptable but not good
						 */
					}
					else
					{
						/* Mark that we have now encountered the
						 * specified section
						 */
						section_encountered = TRUE;
					}

					/* Mark that we are now in the specified section */
					in_section = TRUE;
				}
				else
				{
					/* Not the specified section
					 *
					 * Check if we were in the specified section
					 */
					if(in_section)
					{
						/* Mark that we are no longer in the
						 * specified section
						 */
						in_section = FALSE;
					}

					/* Write this section's heading to the output */
					WRITE_LINE(out_fp, line);
				}

				g_free(s);
			}
			else
			{
				/* This line does not contain a section heading
				 *
				 * Are we currently in the specified section?
				 */
				if(in_section)
				{
					/* Do not write this line to the output
					 * stream so thus removing it
					 */
				}
				else
				{
					/* Not in the specified section so just copy
					 * this line exactly
					 */
					WRITE_LINE(out_fp, line);
				}
			}
#undef WRITE_LINE
		}
		g_free(line);
/*	    line = NULL; */
	}

	return(0);
}

/*
 *	Removes the section and any properties in it from a file.
 *
 *	The path specifies the properties list file.
 *
 *	The section specifies the name of the section to set the
 *	property under. If section is NULL then the property will
 *	be appened.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_properties_list_file_remove_section(
	const gchar *path,
	const gchar *section
)
{
	FILE		*in_fp,
			*out_fp;
	gint status;
	const gchar *in_path = path;
	gchar		*parent,
			*out_path;
	EDVPermissionFlags permissions;


	if(STRISEMPTY(in_path))
	{
		errno = EINVAL;
		return(-2);
	}

	/* Get the parent directory of the specified path */
	if(g_path_is_absolute(in_path))
		parent = g_dirname(in_path);
	else
		parent = edv_getcwd();
	if(parent == NULL)
		return(-1);

	/* Create a temporary file name for the output file in the
	 * parent directory of the specified path
	 */
	out_path = edv_tmp_name(parent);

	/* Delete the parent directory path */
	g_free(parent);

	if(out_path == NULL)
		return(-1);

	/* Open the specified file for reading */
	in_fp = fopen((const char *)in_path, "rb");
	if(in_fp != NULL)
	{
		EDVVFSObject *obj = edv_vfs_object_fstat((gint)fileno(in_fp));
		if(obj != NULL)
		{
			permissions = obj->permissions;
			edv_vfs_object_delete(obj);
		}
		else
		{
			/* Get the default permissions */
			permissions = (~edv_permissions_get_default()) &
				(EDV_PERMISSION_UR | EDV_PERMISSION_UW |
				 EDV_PERMISSION_GR | EDV_PERMISSION_GW |
				 EDV_PERMISSION_OR | EDV_PERMISSION_OW);

		}
	}
	else
	{
		const gint error_code = (gint)errno;
		if(error_code == ENOENT)
		{
			/* It is okay if the specified file does not exist
			 *
			 * Get the default permissions
			 */
			permissions = (~edv_permissions_get_default()) &
				(EDV_PERMISSION_UR | EDV_PERMISSION_UW |
				 EDV_PERMISSION_GR | EDV_PERMISSION_GW |
				 EDV_PERMISSION_OR | EDV_PERMISSION_OW);
		}
		else
		{
			return(-1);
		}
	}

	/* Create the output file for writing */
	out_fp = fopen((const char *)out_path, "wb");
	if(out_fp == NULL)
	{
		const gint error_code = (gint)errno;
		if(in_fp != NULL)
			(void)fclose(in_fp);
		g_free(out_path);
		errno = (int)error_code;
		return(-1);
	}

	/* Remove the section */
	status = edv_properties_list_stream_remove_section(
		in_fp,
		out_fp,
		section
	);
	if(status != 0)
	{
		const gint error_code = (gint)errno;
		if(in_fp != NULL)
			(void)fclose(in_fp);
		(void)fclose(out_fp);
		g_free(out_path);
		errno = (int)error_code;
		return(status);
	}

	/* Close the specified file */
	if(in_fp != NULL)
		(void)fclose(in_fp);

	/* Close the output file */
	if(fclose(out_fp))
	{
		/* An error occured while writing ot the output file */
		const gint error_code = (gint)errno;
		g_free(out_path);
		errno = (int)error_code;
		return(-1);
	}

	/* Remove the specified file */
	if(edv_unlink(in_path))
	{
		if(errno != ENOENT)
		{
			const gint error_code = (gint)errno;
			g_free(out_path);
			errno = (int)error_code;
			return(-1);
		}
	}

	/* Rename the output file to the name of the specified file */
	if(edv_rename(
		out_path,			/* Source */
		in_path				/* Target */
	))
	{
		const gint error_code = (gint)errno;
		g_free(out_path);
		errno = (int)error_code;
		return(-1);
	}

	/* Restore the permissions */
	(void)edv_permissions_set(
		in_path,
		permissions
	);

	g_free(out_path);

	return(0);
}


/*
 *	Counts the number of properties encountered at the
 *	current stream position until the next section is encountered
 *	or end of file.
 *
 *	The fp specifies the stream. The stream will be repositioned
 *	at the same position as it was at the start ofthis call.
 */
static gint edv_properties_list_count_section_properties(FILE *fp)
{
	const long last_pos = ftell(fp);
	gint nproperties = 0;
	const gchar *s;
	gchar *line = NULL;
	while(!feof(fp))
	{
		g_free(line);
		line = EDV_GET_STREAM_LINE_FULL(fp);
		if(line == NULL)
			break;

		s = line;
		while(ISBLANK(*s))
			s++;

		/* Start of next section? */
		if(*s == '[')
			break;

		/* Comment? */
		if(*s == EDV_CFG_COMMENT_CHAR)
			continue;

		/* Empty line? */
		if(*s == '\0')
			continue;

		nproperties++;
	}
	g_free(line);
	(void)fseek(fp, last_pos, SEEK_SET);
	return(nproperties);
}

/*
 *	Cleans the property list.
 *
 *	If in_fp is not NULL then it specifies the stream of the
 *	original properties list which must be opened for reading.
 *
 *	The out_fp specifies the stream to write the new properties
 *	list to which must be opened for writing or append.
 *
 *	If remove_empty_sections is TRUE then any sections with no
 *	properties will be removed.
 *
 *	If remove_comments is TRUE then any comments will be removed.
 *
 *	If remove_empty_lines is TRUE then any empty lines will be
 *	removed.
 */
void edv_properties_list_stream_clean(
	FILE *in_fp,
	FILE *out_fp,
	const gboolean remove_empty_sections,
	const gboolean remove_comments,
	const gboolean remove_empty_lines
)
{
	gchar	*s,
			*line;

	if((in_fp == NULL) || (out_fp == NULL))
		return;

	line = NULL;
	while(!feof(in_fp))
	{
		/* Get the next line exactly as is from the specified
		 * stream because we will need to write it exactly as
		 * it was to the output stream
		 */
		g_free(line);
		line = EDV_GET_STREAM_LINE_FULL(in_fp);
		if(line == NULL)
			break;

#define WRITE_LINE(_fp_,_line_)	{	\
 (void)fprintf(				\
  (_fp_),				\
  "%s\n",				\
  (_line_)				\
 );					\
}

		/* Does this line contain a section heading? */
		s = edv_properties_list_check_parse_section_name(line);
		if(s != NULL)
		{
			/* Remove empty sections? */
			if(remove_empty_sections)
			{
				/* Does this section contain at least one property? */
				if(edv_properties_list_count_section_properties(in_fp) > 0)
				{
					/* Write this section heading to the output
					 * list
					 */
					WRITE_LINE(out_fp, line);
				}
				else
				{
					/* Omit this section heading from the output
					 * list, thus removing it
					 */
				}
			}
			else
			{
				/* Write this section heading to the output list */
				WRITE_LINE(out_fp, line);
			}
			g_free(s);
		}
		else
		{
			const gchar *s = line;
			while(ISBLANK(*s))
				s++;

			/* Check if this line is a comment */
			if(*s == EDV_CFG_COMMENT_CHAR)
			{
				if(!remove_comments)
				{
					/* Write this comment to the output list */
					WRITE_LINE(out_fp, line);
				}
			}
			/* Empty line? */
			else if(*s == '\0')
			{
				if(!remove_empty_lines)
				{
					/* Write this empty line to the output list */
					WRITE_LINE(out_fp, line);
				}
			}
			else
			{
				/* All else write this line to the output list */
				WRITE_LINE(out_fp, line);
			}
		}
#undef WRITE_LINE
	}
	g_free(line);
}

/*
 *	Cleans the property list file.
 *
 *	The path specifies the properties list file.
 *
 *	If remove_empty_sections is TRUE then any sections with no
 *	properties will be removed.
 *
 *	If remove_comments is TRUE then any comments will be removed.
 *
 *	If remove_empty_lines is TRUE then any empty lines will be
 *	removed.
 */
void edv_properties_list_file_clean(
	const gchar *path,
	const gboolean remove_empty_sections,
	const gboolean remove_comments,
	const gboolean remove_empty_lines
)
{
	FILE		*in_fp,
			*out_fp;
	const gchar *in_path = path;
	gchar		*parent,
			*out_path;
	EDVPermissionFlags permissions;

	if(STRISEMPTY(in_path))
		return;

	/* Get the parent directory of the specified path */
	if(g_path_is_absolute(in_path))
		parent = g_dirname(in_path);
	else
		parent = edv_getcwd();
	if(parent == NULL)
		return;

	/* Create a temporary file name for the output file in the
	 * parent directory of the specified path
	 */
	out_path = edv_tmp_name(parent);

	/* Delete the parent directory path */
	g_free(parent);

	if(out_path == NULL)
		return;

	/* Open the specified file for reading */
	in_fp = fopen((const char *)in_path, "rb");
	if(in_fp != NULL)
	{
		EDVVFSObject *obj = edv_vfs_object_fstat((gint)fileno(in_fp));
		if(obj != NULL)
		{
			permissions = obj->permissions;
			edv_vfs_object_delete(obj);
		}
		else
		{
			/* Get the default permissions */
			permissions = (~edv_permissions_get_default()) &
				(EDV_PERMISSION_UR | EDV_PERMISSION_UW |
				 EDV_PERMISSION_GR | EDV_PERMISSION_GW |
				 EDV_PERMISSION_OR | EDV_PERMISSION_OW);

		}
	}
	else
	{
		/* Unable to open the input file so there is nothing
		 * to clean
		 */
		g_free(out_path);
		return;
	}

	/* Create the output file for writing */
	out_fp = fopen((const char *)out_path, "wb");
	if(out_fp == NULL)
	{
		(void)fclose(in_fp);
		g_free(out_path);
		return;
	}

	/* Clean the properties list */
	edv_properties_list_stream_clean(
		in_fp,
		out_fp,
		remove_empty_sections,
		remove_comments,
		remove_empty_lines
	);

	/* Close the specified file */
	(void)fclose(in_fp);

	/* Close the output file */
	if(fclose(out_fp))
	{
		/* An error occured while writing ot the output file */
		g_free(out_path);
		return;
	}

	/* Remove the specified file */
	if(edv_unlink(in_path))
	{
		if(errno != ENOENT)
		{
			g_free(out_path);
			return;
		}
	}

	/* Rename the output file to the name of the specified file */
	if(edv_rename(
		out_path,			/* Source */
		in_path				/* Target */
	))
	{
		g_free(out_path);
		return;
	}

	/* Restore the permissions */
	(void)edv_permissions_set(
		in_path,
		permissions
	);

	g_free(out_path);
}
