#include <errno.h>
#include <glib.h>

#include "edv_types.h"
#include "edv_path.h"
#include "edv_property.h"
#include "edv_property_fio.h"
#include "edv_property_directory.h"

#include "config.h"


/* Get/set directory properties from property file */
static GList *edv_property_directory_get_list(const gchar *path);
static gint edv_property_directory_set_value_s(
	const gchar *path,
	const gchar *name,
	const gchar *value
);

/* Get */
gchar *edv_property_directory_get_icon_path(
	const gchar *path,
	const EDVIconSize icon_size
);
gchar *edv_property_directory_get_icon_opened_path(
	const gchar *path,
	const EDVIconSize icon_size
);

/* Set */
gint edv_property_directory_set_icon_path(
	const gchar *path,
	const EDVIconSize icon_size,
	const gchar *icon_path
);
gint edv_property_directory_set_icon_opened_path(
	const gchar *path,
	const EDVIconSize icon_size,
	const gchar *icon_path
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the list of properties in the directory's section of
 *	the directory properties file.
 *
 *	The path specifies the directory that contains the
 *	properties file.
 *
 *	Returns a GList of EDVProperties.
 */
static GList *edv_property_directory_get_list(const gchar *path)
{
	gchar *dir_prop_path;
	GList *prop_list;

	if(path == NULL)
	{
	    errno = EINVAL;
	    return(NULL);
	}

	/* Get the path to the directory properties file
	 *
	 * The directory properties file is assumed to be located in
	 * the specified directory
	 */
	dir_prop_path = g_strconcat(
	    path,				/* Specified directory */
	    G_DIR_SEPARATOR_S,
	    EDV_NAME_DIRECTORY_PROPERTIES_FILE,	/* Directory properties file */
	    NULL
	);

	/* Get the list of properties from the directory properties file
	 * for this directory
	 */
	prop_list = edv_properties_list_file_get_list(
	    dir_prop_path,
	    ".",				/* This directory's section */
	    NULL				/* All properties in that
						 * section */
	);

	g_free(dir_prop_path);

	return(prop_list);
}

/*
 *	Gets the list of properties in the directory's section of
 *	the directory properties file.
 *
 *	The path specifies the directory that contains the
 *	properties file.
 *
 *	The name specifies the property's name.
 *
 *	The value specifies the property's value.
 */
static gint edv_property_directory_set_value_s(
	const gchar *path,
	const gchar *name,
	const gchar *value
)
{
	gint status;
	gchar *dir_prop_path;

	if(path == NULL)
	{
	    errno = EINVAL;
	    return(-2);
	}

	/* Get the path to the directory properties file
	 *
	 * The directory properties file is assumed to be located in
	 * the specified directory
	 */
	dir_prop_path = g_strconcat(
	    path,				/* Specified directory */
	    G_DIR_SEPARATOR_S,
	    EDV_NAME_DIRECTORY_PROPERTIES_FILE,	/* Directory properties file */
	    NULL
	);

	/* Set the specified value to the specified property in the
	 * specified directory's section in the directory properties
	 * file
	 */
	status = edv_properties_list_file_set_s(
	    dir_prop_path,
	    ".",				/* This directory's section */
	    name,
	    value
	);

	g_free(dir_prop_path);

	return(status);
}


/*
 *	Gets the icon file path from the directory properties file.
 *
 *	The path specifies the directory that contains the
 *	properties file.
 *
 *	The icon_size specifies the icon size.
 *
 *	Returns a dynamically allocated string describing the path to
 *	the icon file.
 */
gchar *edv_property_directory_get_icon_path(
	const gchar *path,
	const EDVIconSize icon_size
)
{
	gchar *value;
	GList		*glist,
			*prop_list;
	EDVProperty *p;

	prop_list = edv_property_directory_get_list(path);
	if(prop_list == NULL)
	    return(NULL);

	value = NULL;
	for(glist = prop_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    p = EDV_PROPERTY(glist->data);
	    if(p == NULL)
		continue;

	    if((value == NULL) && (p->name != NULL))
	    {
		switch(icon_size)
		{
		  case EDV_ICON_SIZE_16:
		    break;
		  case EDV_ICON_SIZE_20:
		    if(!g_strcasecmp(p->name, EDV_PROP_NAME_ICON_FILE_SMALL))
			value = STRDUP(p->value);
		    break;
		  case EDV_ICON_SIZE_32:
		    if(!g_strcasecmp(p->name, EDV_PROP_NAME_ICON_FILE_MEDIUM))
			value = STRDUP(p->value);
		    break;
		  case EDV_ICON_SIZE_48:
		    if(!g_strcasecmp(p->name, EDV_PROP_NAME_ICON_FILE_LARGE))
			value = STRDUP(p->value);
		    break;
		}
	    }

	    edv_property_delete(p);
	}

	g_list_free(prop_list);

	/* Check if the icon file path is not a full path */
	if(value != NULL)
	{
	    if(!g_path_is_absolute(value))
	    {
		/* Complete the icon file path */
		gchar *eval_path = edv_path_evaluate(
		    path,			/* Parent */
		    value			/* Path */
		);
		if(eval_path != NULL)
		{
		    g_free(value);
		    value = eval_path;
		}
	    }
	}

	return(value);
}

gchar *edv_property_directory_get_icon_opened_path(
	const gchar *path,
	const EDVIconSize icon_size
)
{
	gchar *value;
	GList		*glist,
			*prop_list;
	EDVProperty *p;

	prop_list = edv_property_directory_get_list(path);
	if(prop_list == NULL)
	    return(NULL);

	value = NULL;
	for(glist = prop_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    p = EDV_PROPERTY(glist->data);
	    if(p == NULL)
		continue;

	    if((value == NULL) && (p->name != NULL))
	    {
		switch(icon_size)
		{
		  case EDV_ICON_SIZE_16:
		    break;
		  case EDV_ICON_SIZE_20:
		    if(!g_strcasecmp(p->name, EDV_PROP_NAME_ICON_OPENED_FILE_SMALL))
			value = STRDUP(p->value);
		    break;
		  case EDV_ICON_SIZE_32:
		    if(!g_strcasecmp(p->name, EDV_PROP_NAME_ICON_OPENED_FILE_MEDIUM))
			value = STRDUP(p->value);
		    break;
		  case EDV_ICON_SIZE_48:
		    if(!g_strcasecmp(p->name, EDV_PROP_NAME_ICON_OPENED_FILE_LARGE))
			value = STRDUP(p->value);
		    break;
		}
	    }

	    edv_property_delete(p);
	}

	g_list_free(prop_list);

	/* Check if the icon file path is not a full path */
	if(value != NULL)
	{
	    if(!g_path_is_absolute(value))
	    {
		/* Complete the icon file path */
		gchar *eval_path = edv_path_evaluate(
		    path,			/* Parent */
		    value			/* Path */
		);
		if(eval_path != NULL)
		{
		    g_free(value);
		    value = eval_path;
		}
	    }
	}

	return(value);
}


/*
 *	Sets the icon file path to the directory properties file.
 *
 *	The path specifies the directory that contains the
 *	properties file.
 *
 *	The icon_size specifies the icon size.
 *
 *	The value specifies the property value.
 */
gint edv_property_directory_set_icon_path(
	const gchar *path,
	const EDVIconSize icon_size,
	const gchar *icon_path
)
{
	const gchar	*name = NULL,
			*child_path;
	switch(icon_size)
	{
	  case EDV_ICON_SIZE_16:
	    break;
	  case EDV_ICON_SIZE_20:
	    name = EDV_PROP_NAME_ICON_FILE_SMALL;
	    break;
	  case EDV_ICON_SIZE_32:
	    name = EDV_PROP_NAME_ICON_FILE_MEDIUM;
	    break;
	  case EDV_ICON_SIZE_48:
	    name = EDV_PROP_NAME_ICON_FILE_LARGE;
	    break;
	}
	if(name == NULL)
	{
	    errno = EINVAL;
	    return(-2);
	}

	/* If the specified icon path is a child of the directory
	 * then seek icon path to its child position
	 */
	child_path = edv_path_child(
	    path,
	    icon_path
	);
	if(child_path == NULL)
	    child_path = icon_path;

	return(edv_property_directory_set_value_s(
	    path,
	    name,
	    child_path
	));
}

gint edv_property_directory_set_icon_opened_path(
	const gchar *path,
	const EDVIconSize icon_size,
	const gchar *icon_path
)
{
	const gchar	*name = NULL,
			*child_path;
	switch(icon_size)
	{
	  case EDV_ICON_SIZE_16:
	    break;
	  case EDV_ICON_SIZE_20:
	    name = EDV_PROP_NAME_ICON_OPENED_FILE_SMALL;
	    break;
	  case EDV_ICON_SIZE_32:
	    name = EDV_PROP_NAME_ICON_OPENED_FILE_MEDIUM;
	    break;
	  case EDV_ICON_SIZE_48:
	    name = EDV_PROP_NAME_ICON_OPENED_FILE_LARGE;
	    break;
	}
	if(name == NULL)
	{
	    errno = EINVAL;
	    return(-2);
	}

	/* If the specified icon path is a child of the directory
	 * then seek icon path to its child position
	 */
	child_path = edv_path_child(
	    path,
	    icon_path
	);
	if(child_path == NULL)
	    child_path = icon_path;

	return(edv_property_directory_set_value_s(
	    path,
	    name,
	    child_path
	));
}
