#include <stdlib.h>
#include <errno.h>
#include <glib.h>

#include "edv_property.h"


/* Property */
EDVProperty *edv_property_new(void);
EDVProperty *edv_property_copy(EDVProperty *prop);
void edv_property_delete(EDVProperty *prop);

/* Properties List */
GList *edv_properties_list_set_s(
	GList *properties_list,
	const gchar *name, const gchar *value,
	const gboolean create_as_needed
);
GList *edv_properties_list_set_i(
	GList *properties_list,
	const gchar *name,
	const gint value,
	const gboolean create_as_needed
);
GList *edv_properties_list_set_ul(
	GList *properties_list,
	const gchar *name,
	const gulong value,
	const gboolean create_as_needed
);
GList *edv_properties_list_set_f(
	GList *properties_list,
	const gchar *name,
	const gfloat value,
	const gboolean create_as_needed
);
GList *edv_properties_list_update(
	GList *properties_list,
	GList *updated_properties_list,
	const gboolean create_as_needed,
	const gboolean remove_non_existant
);
GList *edv_properties_list_remove(
	GList *properties_list,
	const gchar *name
);
EDVProperty *edv_properties_list_get(
	GList *properties_list,
	const gchar *name
);
const gchar *edv_properties_list_get_s(
	GList *properties_list,
	const gchar *name
);
gint edv_properties_list_get_i(
	GList *properties_list,
	const gchar *name
);
gulong edv_properties_list_get_ul(
	GList *properties_list,
	const gchar *name
);
gfloat edv_properties_list_get_f(
	GList *properties_list,
	const gchar *name
);

GList *edv_properties_list_copy(GList *properties_list);
GList *edv_properties_list_delete(GList *properties_list);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new Property.
 */
EDVProperty *edv_property_new(void)
{
	return(EDV_PROPERTY(g_malloc0(sizeof(EDVProperty))));
}

/*
 *	Coppies the Property.
 */
EDVProperty *edv_property_copy(EDVProperty *prop)
{
	EDVProperty *new_prop;

	if(prop == NULL)
	{
		errno = ENOENT;
		return(NULL);
	}

	new_prop = edv_property_new();
	if(new_prop == NULL)
		return(NULL);

	new_prop->name = STRDUP(prop->name);
	new_prop->value = STRDUP(prop->value);

	return(new_prop);
}

/*
 *	Deletes the Property.
 */
void edv_property_delete(EDVProperty *prop)
{
	if(prop == NULL)
		return;

	g_free(prop->name);
	g_free(prop->value);
	g_free(prop);
}


/*
 *	Sets the property value in the properties list.
 *
 *	The properties_list specifies the properties list, a GList of
 *	EDVProperty * object properties.
 *
 *	The name specifies the name of the property to set.
 *
 *	The value specifies the property value to set. If value is NULL
 *	or an empty string then the property will be removed.
 *
 *	If create_as_needed is TRUE and the property does not yet exist
 *	then it will be created.
 *
 *	Returns the modified properties_list.
 */
GList *edv_properties_list_set_s(
	GList *properties_list,
	const gchar *name,
	const gchar *value,
	const gboolean create_as_needed
)
{
	EDVProperty *prop;

	if(STRISEMPTY(name))
		return(properties_list);

	/* Remove the property? */
	if(STRISEMPTY(value))
		return(edv_properties_list_remove(
			properties_list,
			name
		));

	/* Find the property that matches the specified name */
	prop = edv_properties_list_get(properties_list, name);
	if(prop != NULL)
	{
		/* Update the property name for case-insensitive corrections */
		g_free(prop->name);
		prop->name = g_strdup(name);
	}
	else
	{
		/* No such property exists, do not create it? */
		if(!create_as_needed)
			return(properties_list);

		/* Create a new property */
		prop = edv_property_new();
		if(prop == NULL)
			return(properties_list);

		/* Set the new property's name */
		prop->name = g_strdup(name);

		/* Append the new property to the proprties list */
		properties_list = g_list_append(
			properties_list,
			prop
		);
	}

	/* Set the value */
	g_free(prop->value);
	prop->value = g_strdup(value);

	return(properties_list);
}

GList *edv_properties_list_set_i(
	GList *properties_list,
	const gchar *name,
	const gint value,
	const gboolean create_as_needed
)
{
	gchar *s = g_strdup_printf("%i", value);
	properties_list = edv_properties_list_set_s(
		properties_list,
		name,
		s,
		create_as_needed
	);
	g_free(s);
	return(properties_list);
}

GList *edv_properties_list_set_ul(
	GList *properties_list,
	const gchar *name,
	const gulong value,
	const gboolean create_as_needed
)
{
	gchar *s = g_strdup_printf("%ld", value);
	properties_list = edv_properties_list_set_s(
		properties_list,
		name,
		s,
		create_as_needed
	);
	g_free(s);
	return(properties_list);
}

GList *edv_properties_list_set_f(
	GList *properties_list,
	const gchar *name,
	const gfloat value,
	const gboolean create_as_needed
)
{
	gchar *s = g_strdup_printf("%f", value);
	properties_list = edv_properties_list_set_s(
		properties_list,
		name,
		s,
		create_as_needed
	);
	g_free(s);
	return(properties_list);
}

/*
 *	Updates the properties list values.
 *
 *	The properties_list specifies the properties list, a GList of
 *	EDVProperty * object properties.
 *
 *	The updated_properties_list specifies the properties list, a
 *	GList of EDVProperty * object properties, who's values will
 *	be used to update the properties in properties_list with.
 *	The updated_properties_list will not be modified in any way.
 *
 *	If create_as_needed is TRUE then any new properties found
 *	in updated_properties_list will be created in properties_list.
 *
 *	If remove_non_existant is TRUE then any properties in
 *	updated_properties_list that are not in properties_list will
 *	be removed from properties_list.
 *
 *	Returns the modified properties_list.
 */
GList *edv_properties_list_update(
	GList *properties_list,
	GList *updated_properties_list,
	const gboolean create_as_needed,
	const gboolean remove_non_existant
)
{
	GList *glist;
	EDVProperty *prop;

	/* The two lists may not be the same */
	if(properties_list == updated_properties_list)
		return(properties_list);

	/* Remove any properties in properties_list that are not found
	 * in updated_properties_list?
	 */
	if(remove_non_existant)
	{
		GList *glist = properties_list;
		EDVProperty *prop;
		while(glist != NULL)
		{
			prop = EDV_PROPERTY(glist->data);

			/* Remove any NULL properties from properties_list */
			if(prop == NULL)
			{
				properties_list = g_list_remove(
					properties_list,
					NULL
				);
				glist = properties_list;
				continue;
			}

			/* If this property has no name then remove
			 * it from properties_list
			 */
			if(STRISEMPTY(prop->name))
			{
				edv_property_delete(prop);
				properties_list = g_list_remove(
					properties_list,
					prop
				);
				glist = properties_list;
				continue;
			}

			/* Does this property not exist in
			 * updated_properties_list?
			 */
			if(edv_properties_list_get(
				updated_properties_list,
				prop->name
			) == NULL)
			{
				/* Remove this property from properties_list */
				gchar *name = g_strdup(prop->name);
				properties_list = edv_properties_list_remove(
					properties_list,
					name
				);
				g_free(name);

				/* Start from the beginning of
				 * properties_list again
				 */
				glist = properties_list;
				continue;
			}

			glist = g_list_next(glist);
		}
	}

	/* Update the properties in properties_list with the properties
	 * in updated_properties_list
	 */
	for(glist = updated_properties_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		prop = EDV_PROPERTY(glist->data);
		if(prop == NULL)
			continue;

		if(STRISEMPTY(prop->name) || STRISEMPTY(prop->value))
			continue;

		properties_list = edv_properties_list_set_s(
			properties_list,
			prop->name,
			prop->value,
			create_as_needed
		);
	}

	return(properties_list);
}

/*
 *	Removes the property from the properties list.
 *
 *	The properties_list specifies the properties list, a GList of
 *	EDVProperty * object properties.
 *
 *	The name specifies the name of the property to remove.
 *
 *	Returns the modified properties_list.
 */
GList *edv_properties_list_remove(
	GList *properties_list,
	const gchar *name
)
{
	EDVProperty *prop;

	if((properties_list == NULL) || STRISEMPTY(name))
		return(properties_list);

	/* Find the property that matches the specified name */
	prop = edv_properties_list_get(
		properties_list,
		name
	);
	if(prop == NULL)
		return(properties_list);

	/* Delete this property */
	edv_property_delete(prop);

	/* Remove this property from the properties list */
	properties_list = g_list_remove(
		properties_list,
		prop
	);

	return(properties_list);
}

/*
 *	Gets the property from the properties list.
 *
 *	The properties_list specifies the properties list, a GList of
 *	EDVProperty * object properties.
 *
 *	The name specifies the name of the property to get.
 *
 *	Returns the property or NULL on error.
 */
EDVProperty *edv_properties_list_get(
	GList *properties_list,
	const gchar *name
)
{
	GList *glist;
	EDVProperty *prop;

	if(properties_list == NULL)
	{
		errno = ENOENT;
		return(NULL);
	}

	if(STRISEMPTY(name))
	{
		errno = EINVAL;
		return(NULL);
	}

	for(glist = properties_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		prop = EDV_PROPERTY(glist->data);
		if(prop == NULL)
			continue;

		if(prop->name == NULL)
			continue;

		if(!g_strcasecmp(prop->name, name))
			return(prop);
	}

	errno = ENOENT;
	return(NULL);
}

const gchar *edv_properties_list_get_s(
	GList *properties_list,
	const gchar *name
)
{
	EDVProperty *p = edv_properties_list_get(
		properties_list,
		name
	);
	if(p == NULL)
		return(NULL);

	return(p->value);
}

gint edv_properties_list_get_i(
	GList *properties_list,
	const gchar *name
)
{
	EDVProperty *p = edv_properties_list_get(
		properties_list,
		name
	);
	if(p == NULL)
		return(0);

	if(p->value == NULL)
		return(0);

	return((gint)atoi((const char *)p->value));
}

gulong edv_properties_list_get_ul(
	GList *properties_list,
	const gchar *name
)
{
	EDVProperty *p = edv_properties_list_get(
		properties_list,
		name
	);
	if(p == NULL)
		return(0l);

	if(p->value == NULL)
		return(0l);

	return((gulong)atol((const char *)p->value));
}

gfloat edv_properties_list_get_f(
	GList *properties_list,
	const gchar *name
)
{
	EDVProperty *p = edv_properties_list_get(
		properties_list,
		name
	);
	if(p == NULL)
		return(0.0f);

	if(p->value == NULL)
		return(0.0f);

	return(ATOF(p->value));
}


/*
 *	Coppies the properties list.
 *
 *	The properties_list specifies the properties list, a GList of
 *	EDVProperty * object properties.
 *
 *	Returns the coppied properties_list.
 */
GList *edv_properties_list_copy(GList *properties_list)
{
	GList		*glist,
			*coppied_properties_list;

	if(properties_list == NULL)
		return(NULL);

	coppied_properties_list = NULL;
	for(glist = properties_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
		coppied_properties_list = g_list_append(
			coppied_properties_list,
			edv_property_copy(EDV_PROPERTY(glist->data))
		);

	return(coppied_properties_list);
}

/*
 *	Deletes the properties list.
 *
 *	The properties_list specifies the properties list, a GList of
 *	EDVProperty * object properties.
 *
 *	This function always returns NULL.
 */
GList *edv_properties_list_delete(GList *properties_list)
{
	if(properties_list == NULL)
		return(NULL);

	g_list_foreach(
		properties_list,
		(GFunc)edv_property_delete,
		NULL
	);
	g_list_free(properties_list);

	return(NULL);
}
