/*
	           Endeavour Mark II API - Process

	Process testing, status, forking, and execution.

	See also: edv_utils.h
 */

#ifndef EDV_PROCESS_H
#define EDV_PROCESS_H

#include <stdio.h>				/* For FILE * */
#include <glib.h>


/*
 *	Gets the process ID of the current process.
 */
extern gint edv_pid_get_current(void);
#define edv_getpid		edv_pid_get_current

/*
 *	Gets the process ID of the parent of the current process.
 */
extern gint edv_pid_get_current_parent(void);
#define edv_getppid		edv_pid_get_current_parent

/*
 *	Checks if the process is running.
 *
 *	The pid specifies the process ID. The pid must be positive.
 *
 *	Returns TRUE if the process is running or FALSE if it is not or
 *	error.
 */
extern gboolean edv_pid_exists(const gint pid);

/*
 *	Gets the schedualing priority of the process.
 *
 *	The pid specifies the process ID. If pid is 0 then the
 *	schedualing of the current process is returned. Otherwise
 *	pid must be positive.
 *
 *	Returns the schedualing priority of the process or 0 on error.
 *	The returned value ranges are platform-dependent.
 */
extern gint edv_pid_get_priority(const gint pid);


/*
 *	Sets a SIGCHLD signal callback and creates a child process with
 *	the standard streams connected for communication between the
 *	parent and the child processes.
 *
 *	If signal_cb is not NULL then signal_cb will be set as the
 *	SIGCHLD signal callback that will be called when the child
 *	process has exited. If signal_cb is NULL then this function
 *	will set its own handler for the SIGCHLD signal.
 *
 *	For the parent's thread of execution, if any of stdin_rtn,
 *	stdout_rtn, or stderr_rtn are not NULL then stdin, stdout, or
 *	stderr will be redirected and the return values for the streams
 *	will be set to *stdin_rtn, *stdout_rtn, or *stderr_rtn
 *	respectively. The calling function must close the returned
 *	streams with fclose(). The stream value; *stdin_rtn will be
 *	opened for writing and line buffered, *stdout_rtn will be
 *	opened for reading and line buffered, and *stderr_rtn will be
 *	opened for reading and unbuffered.
 *
 *	For the child's thread of execution, if any of stdin_rtn,
 *	stdout_rtn, or stderr_rtn are not NULL then *stdin_rtn,
 *	*stdout_rtn, or *stderr_rtn will always be set to NULL.
 *
 *	Returns the child process ID for the parent's thread of
 *	execution, 0 for the child's thread of execution, or negative
 *	on error.
 */
extern gint edv_fork_wait_streams(
        void (*signal_cb)(gint),
        FILE **stdin_rtn,
        FILE **stdout_rtn,
        FILE **stderr_rtn
);

/*
 *	Sets a SIGCHLD signal callback and creates a child process.
 *
 *	If signal_cb is not NULL then signal_cb will be set as the
 *	SIGCHLD signal callback that will be called when the child
 *	process has exited. If signal_cb is NULL then this function
 *	will set its own handler for the SIGCHLD signal.
 *
 *	Returns the child process ID for the parent's thread of
 *	execution, 0 for the child's thread of execution, or negative
 *	on error.
 */
extern gint edv_fork_wait(void (*signal_cb)(gint));

/*
 *	Creates a child process.
 *
 *	The calling function must set a SIGCHLD callback to wait for
 *	the child process to exit.
 *
 *	Returns the child process ID for the parent's thread of
 *	execution, 0 for the child's thread of execution, or negative
 *	on error.
 */
extern gint edv_fork(void);

/*
 *	Executes the command in the background using a particular
 *	shell and redirects the standard streams.
 *
 *	The cmd specifies the command. The command must contain an
 *	absolute path to an executable and any subsequent arguments or
 *	shell tokens supported by the shell.
 *
 *	The shell_path specifies the full path to the shell. If
 *	shell_path is NULL then the shell specified by the environment
 *	variable SHELL will be used. If no environment variable is
 *	specified then "/bin/sh" will be used.
 *
 *	The shell_args specifies the shell's arguments. If shell_args
 *	is NULL then "-c" will be used.
 *
 *	If any of stdin_rtn, stdout_rtn, or stderr_rtn are not NULL
 *	then stdin, stdout, or stderr will be redirected and the
 *	return values for the streams will be set to *stdin_rtn,
 *	*stdout_rtn, or *stderr_rtn respectively. The calling function
 *	must close the returned streams with fclose(). The stream
 *	value; *stdin_rtn will be opened for writing and line buffered,
 *	*stdout_rtn will be opened for reading and line buffered, and
 *	*stderr_rtn will be opened for reading and unbuffered.
 *
 *	Returns the process ID or negative on error.
 */
extern gint edv_system_shell_streams(
	const gchar *cmd,
	const gchar *shell_path,
	const gchar *shell_args,
	FILE **stdin_rtn,
	FILE **stdout_rtn,
	FILE **stderr_rtn
);
extern gint edv_system_shell(
	const gchar *cmd,
	const gchar *shell_path,
	const gchar *shell_args
);
extern gint edv_system(const gchar *cmd);

/*
 *	Executes the command using a particular shell and blocks until
 *	it has exited.
 *
 *	The cmd specifies the command. The command must contain an
 *	absolute path to an executable and any subsequent arguments or
 *	shell tokens supported by the shell.
 *
 *	The shell_path specifies the full path to the shell. If
 *	shell_path is NULL then the shell specified by the environment
 *	variable SHELL will be used. If no environment variable is
 *	specified then "/bin/sh" will be used.
 *
 *	The shell_args specifies the shell's arguments. If shell_args
 *	is NULL then "-c" will be used.
 *
 *	If status_rtn is not NULL then the return value of the command
 *	will be set to *status_rtn.
 *
 *	Returns the process ID or negative on error.
 */
extern gint edv_system_wait_shell(
	const gchar *cmd,
	const gchar *shell_path,
	const gchar *shell_args,
	gint *status_rtn
);
extern gint edv_system_wait(
	const gchar *cmd,
	gint *status_rtn
);


#endif	/* EDV_PROCESS_H */
