#ifndef _USE_BSD
# define _USE_BSD				/* Needed for wait3() in
						 * sys/wait.h */
#endif
#include <stdio.h>				/* For FILE * */
#include <errno.h>
#include <limits.h>				/* For PATH_MAX and NAME_MAX */
#include <sys/types.h>
#include <signal.h>
#include <sched.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <glib.h>
#include <unistd.h>

#include "edv_process.h"

#include "config.h"


/*
 *	Environment Variables List:
 *
 *	For use with execve().
 */
extern char		**environ;


/* Get PID */
gint edv_pid_get_current(void);
gint edv_pid_get_current_parent(void);

/* Status */
gboolean edv_pid_exists(const gint pid);
gint edv_pid_get_priority(const gint pid);

/* Signals */
static void edv_system_wait_cb(int s);
static gpointer edv_signal(
	const gint signal_num,
	void (*handler)(gint)
);

/* Fork */
gint edv_fork_wait_streams(
	void (*signal_cb)(gint),
	FILE **stdin_rtn,
	FILE **stdout_rtn,
	FILE **stderr_rtn
);
gint edv_fork_wait(void (*signal_cb)(gint));
gint edv_fork(void);

/* System */
gint edv_system_shell_streams(
	const gchar *cmd,
	const gchar *shell_path,
	const gchar *shell_args,
	FILE **stdin_rtn,
	FILE **stdout_rtn,
	FILE **stderr_rtn
);
gint edv_system_shell(
	const gchar *cmd,
	const gchar *shell_path,
	const gchar *shell_args
);
gint edv_system(const gchar *cmd);
gint edv_system_wait_shell(
	const gchar *cmd,
	const gchar *shell_path,
	const gchar *shell_args,
	gint *status_rtn
); 
gint edv_system_wait(
	const gchar *cmd,
	gint *status_rtn
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define CLOSE(_fd_)	(((_fd_) > -1) ? (gint)close((int)(_fd_)) : -1)


/*
 *	Gets the process ID of the current process.
 */
gint edv_pid_get_current(void)
{
	return((gint)getpid());
}

/*
 *	Gets the process ID of the parent of the current process.
 */
gint edv_pid_get_current_parent(void)
{
	return((gint)getppid());
}


/*
 *	Checks if the process is running.
 *
 *	The pid specifies the process ID. The pid must be positive.
 *
 *	Returns TRUE if the process is running or FALSE if it is not or
 *	error.
 */
gboolean edv_pid_exists(const gint pid)
{
#if defined(__linux__)
	/* To improve the efficiency of this function on Linux just
	 * check if the process' directory exists in the PATH_PROC
	 * directory to see if it is running
	 */
	gint status;
	gchar *path = g_strdup_printf(
	    PATH_PROC G_DIR_SEPARATOR_S "%i",
	    pid
	);
	if(path == NULL)
	    return(FALSE);

	status = (gint)access((const char *)path, F_OK);

	g_free(path);

	return((status == 0) ? TRUE : FALSE);
#else
# ifdef _POSIX_PRIORITY_SCHEDULING
	struct sched_param sp;

	if(pid <= 0)
	{
	    errno = EINVAL;
	    return(FALSE);
	}

	/* Get the schedualing parameters for the process in order to
	 * check if the process is running
	 */
	if(sched_getparam(
	    (pid_t)pid,
	    &sp
	) == 0)
	{
	    /* Got the schedualing parameters which means the process
	     * is running
	     */
	    return(TRUE);
	}
	else
	{
	    const gint error_code = (gint)errno;
	    switch(error_code)
	    {
	      case ESRCH:			/* Process could not be found */
		return(FALSE);
		break;
	      default:				/* Assume process could not
						 * be found */
		return(FALSE);
		break;
	    }
	}
# else
#  warning "_POSIX_PRIORITY_SCHEDULING was not defined in unistd.h which suggests sched_getparam() may not be available so edv_pid_exists() will always return FALSE."
	return(FALSE);
# endif		/* _POSIX_PRIORITY_SCHEDULING */
#endif   
}

/*
 *	Gets the schedualing priority of the process.
 *
 *	The pid specifies the process ID. If pid is 0 then the
 *	schedualing of the current process is returned. Otherwise
 *	pid must be positive.
 *
 *	Returns the schedualing priority of the process or 0 on error.
 *	The returned value ranges are platform-dependent.
 */
gint edv_pid_get_priority(const gint pid)
{
#ifdef _POSIX_PRIORITY_SCHEDULING
	struct sched_param sp;

	if(pid < 0)
	{
	    errno = EINVAL;
	    return(0);
	}

	/* Get the schedualing parameters for the process */
	if(sched_getparam(
	    (pid_t)pid,
	    &sp
	) == 0)
	{
	    return((gint)sp.sched_priority);
	}
	else
	{
	    return(0);
	}
#else
# warning "_POSIX_PRIORITY_SCHEDULING was not defined in unistd.h which suggests sched_getparam() may not be available so EDVGetProcessStatus() will always return 0."
	return(0);
#endif	/* _POSIX_PRIORITY_SCHEDULING */
}


/*
 *	SIGCHLD signal callback.
 *
 *	Used by the non-blocking EDVSystem*() functions to wait for
 *	a child process to exit.
 */
static void edv_system_wait_cb(int s)
{
	gint status;
	while(wait3((int *)&status, WNOHANG, NULL) > 0);
}

/*
 *	Adds a signal using the BSD style signal(2).
 *
 *	Returns the old signal action handler of type
 *	void (*handler)(gint).
 */
static gpointer edv_signal(
	const gint signal_num,
	void (*handler)(gint)
)
{
	struct sigaction	act,
				old_act;

	act.sa_handler = handler;
	act.sa_flags = 0;

	if(sigaction(
	    (int)signal_num,
	    &act,
	    &old_act
	) == -1)
	    return((gpointer)SIG_ERR);
	else
	    return((gpointer)old_act.sa_handler);
}


/*
 *	Sets a SIGCHLD signal callback and creates a child process with
 *	the standard streams connected for communication between the
 *	parent and the child processes.
 *
 *	If signal_cb is not NULL then signal_cb will be set as the
 *	SIGCHLD signal callback that will be called when the child
 *	process has exited. If signal_cb is NULL then this function
 *	will set its own handler for the SIGCHLD signal.
 *
 *      For the parent's thread of execution, if any of stdin_rtn,
 *      stdout_rtn, or stderr_rtn are not NULL then stdin, stdout, or
 *      stderr will be redirected and the return values for the streams
 *      will be set to *stdin_rtn, *stdout_rtn, or *stderr_rtn
 *      respectively. The calling function must close the returned
 *      streams with fclose(). The stream value; *stdin_rtn will be
 *      opened for writing and line buffered, *stdout_rtn will be
 *	opened for reading and line buffered, and *stderr_rtn will be
 *	opened for reading and unbuffered.
 *
 *	For the child's thread of execution, if any of stdin_rtn,
 *	stdout_rtn, or stderr_rtn are not NULL then *stdin_rtn,
 *	*stdout_rtn, or *stderr_rtn will always be set to NULL.
 *
 *	Returns the child process ID for the parent's thread of
 *	execution, 0 for the child's thread of execution, or negative
 *	on error.
 */
gint edv_fork_wait_streams(
	void (*signal_cb)(gint),
	FILE **stdin_rtn,
	FILE **stdout_rtn,
	FILE **stderr_rtn
)
{
	gint		stdin_pair[2],
			stdout_pair[2],
			stderr_pair[2],
			pid;

	/* Create the streams as needed
	 *
	 * Standard input
	 */
	if(stdin_rtn != NULL)
	{
	    *stdin_rtn = NULL;
	    if(pipe((int *)stdin_pair))
	    {
		stdin_pair[0] = -1;
		stdin_pair[1] = -1;
	    }
	}
	else
	{
	    stdin_pair[0] = -1;
	    stdin_pair[1] = -1;
	}
	/* Standard output */
	if(stdout_rtn != NULL)
	{
	    *stdout_rtn = NULL;
	    if(pipe((int *)stdout_pair))
	    {
		stdout_pair[0] = -1;
		stdout_pair[1] = -1;
	    }
	}
	else
	{
	    stdout_pair[0] = -1;
	    stdout_pair[1] = -1;
	}
	/* Standard error */
	if(stderr_rtn != NULL)
	{
	    *stderr_rtn = NULL;
	    if(pipe((int *)stderr_pair))
	    {
		stderr_pair[0] = -1;
		stderr_pair[1] = -1;
	    }
	}
	else
	{
	    stderr_pair[0] = -1;
	    stderr_pair[1] = -1;
	}

#ifdef SIGCHLD
	/* Set the callback that will handle the SIGCHLD signal */
	edv_signal(
	    SIGCHLD,
	    (signal_cb != NULL) ? signal_cb : edv_system_wait_cb
	);
#else
# warning "SIGCHLD was not defined, edv_fork_wait_streams() will not be able to wait for child processes that have existed"
#endif

	/* Fork */
	pid = edv_fork();
	if(pid < 0)
	    return(pid);

	/* Are we the child? */
	if(pid == 0)
	{
	    /* Redirect standard input reads to our read descriptor by
	     * closing stdin and then make stdin a copy of our read
	     * descriptor
	     */
	    gint fd = stdin_pair[0];
	    if(fd > -1)
		(void)dup2(
		    (int)fd,
		    fileno(stdin)
		);

	    /* Redirect standard output writes to our write descriptor
	     * by closing stdout and then make stdout a copy of our
	     * write descriptor
	     */
	    fd = stdout_pair[1];
	    if(fd > -1)
		(void)dup2(
		    (int)fd,
		    fileno(stdout)
		);

	    /* Redirect standard error writes to our write descriptor
	     * by closing stderr and then make stderr a copy of our
	     * write descriptor
	     */
	    fd = stderr_pair[1];
	    if(fd > -1)
		(void)dup2(
		    (int)fd,
		    fileno(stderr)
		);

	    /* Close the original descriptors for a clean environment
	     * before returning
	     */
	    (void)CLOSE(stdin_pair[0]);
	    (void)CLOSE(stdin_pair[1]);
	    (void)CLOSE(stdout_pair[0]);
	    (void)CLOSE(stdout_pair[1]);
	    (void)CLOSE(stderr_pair[0]);
	    (void)CLOSE(stderr_pair[1]);
	}
	else
	{
	    /* Parent */
	    gint fd;

	    /* Set the return streams
	     *
	     * Standard input
	     */
	    fd = stdin_pair[1];
	    if((stdin_rtn != NULL) && (fd > -1))
	    {
		FILE *fp = fdopen((int)fd, "wb");
		if(fp != NULL)
		{
		    /* Set standard input as line buffered */
		    (void)setlinebuf(fp);
		    *stdin_rtn = fp;
		}
	    }

	    /* Close the read end of our stdin since it is not needed */
	    (void)CLOSE(stdin_pair[0]);

	    /* Standard output */
	    fd = stdout_pair[0];
	    if((stdout_rtn != NULL) && (fd > -1))
	    {
		FILE *fp = fdopen((int)fd, "rb");
		if(fp != NULL)
		{
		    /* Set standard output as line buffered */
		    (void)setlinebuf(fp);
		    *stdout_rtn = fp;
		}
	    }

	    /* Close the write end of our stdout since it is not needed
	     * and otherwise reading from it will never result in an EOF
	     */
	    (void)CLOSE(stdout_pair[1]);

	    /* Standard error */
	    fd = stderr_pair[0];
	    if((stderr_rtn != NULL) && (fd > -1))
	    {
		FILE *fp = fdopen((int)fd, "rb");
		if(fp != NULL)
		{
		    /* Set standard error as unbuffered */
		    (void)setvbuf(
			fp,
			NULL,
			_IONBF,
			0
		    );
		    *stderr_rtn = fp;
		}
	    }

	    /* Close the write end of our stderr since it is not needed
	     * and otherwise reading from it will never result in an EOF
	     */
	    (void)CLOSE(stderr_pair[1]);
	}

	return(pid);
}

/*
 *	Sets a SIGCHLD signal callback and creates a child process.
 *
 *	If signal_cb is not NULL then signal_cb will be set as the
 *	SIGCHLD signal callback that will be called when the child
 *	process has exited. If signal_cb is NULL then this function
 *	will set its own handler for the SIGCHLD signal.
 *
 *	Returns the child process ID for the parent's thread of
 *	execution, 0 for the child's thread of execution, or negative
 *	on error.
 */
gint edv_fork_wait(void (*signal_cb)(gint))
{
	return(edv_fork_wait_streams(
	    signal_cb,
	    NULL,
	    NULL,
	    NULL
	));
}

/*
 *	Creates a child process.
 *
 *	The calling function must set a SIGCHLD callback to wait for
 *	the child process to exit.
 *
 *	Returns the child process ID for the parent's thread of
 *	execution, 0 for the child's thread of execution, or negative
 *	on error.
 */
gint edv_fork(void)
{
	return((gint)fork());
}

/*
 *	Executes the command in the background using a particular
 *	shell and redirects the standard streams.
 *
 *	The cmd specifies the command. The command must contain an
 *	absolute path to an executable and any subsequent arguments or
 *	shell tokens supported by the shell.
 *
 *	The shell_path specifies the full path to the shell. If
 *	shell_path is NULL then the shell specified by the environment
 *	variable SHELL will be used. If no environment variable is
 *	specified then "/bin/sh" will be used.
 *
 *	The shell_args specifies the shell's arguments. If shell_args
 *	is NULL then "-c" will be used.
 *
 *	If any of stdin_rtn, stdout_rtn, or stderr_rtn are not NULL
 *	then stdin, stdout, or stderr will be redirected and the
 *	return values for the streams will be set to *stdin_rtn,
 *	*stdout_rtn, or *stderr_rtn respectively. The calling function
 *	must close the returned streams with fclose(). The stream
 *	value; *stdin_rtn will be opened for writing, *stdout_rtn
 *	will be opened for reading, and *stderr_rtn will be opened for
 *	reading.
 *
 *	Returns the process ID or negative on error.
 */     
gint edv_system_shell_streams(
	const gchar *cmd,
	const gchar *shell_path,
	const gchar *shell_args,
	FILE **stdin_rtn,
	FILE **stdout_rtn,
	FILE **stderr_rtn
)
{
	gint pid;

	if(stdin_rtn != NULL)
	    *stdin_rtn = NULL;
	if(stdout_rtn != NULL)
	    *stdout_rtn = NULL;
	if(stderr_rtn != NULL)
	    *stderr_rtn = NULL;

	/* Use the default shell? */
	if(STRISEMPTY(shell_path))
	    shell_path = g_getenv(ENV_VAR_NAME_SHELL);
	else if(!g_path_is_absolute(shell_path))
	    shell_path = g_getenv(ENV_VAR_NAME_SHELL);
	if(shell_path == NULL)
	    shell_path = EDV_PATH_PROG_DEF_SHELL;

	/* Use the default shell arguments? */
	if(STRISEMPTY(shell_args))
	    shell_args = EDV_ARGS_DEF_SHELL;

	/* No command specified? */
	if(STRISEMPTY(cmd))
	{
	    errno = EINVAL;
	    return(-2);
	}

	/* Sets the default SIGCHLD signal callback and creates a
	 * child process with the standard streams connected for
	 * communication between the parent and the child processes
	 */
	pid = edv_fork_wait_streams(
	    NULL,				/* Set the default SIGCHLD
						 * signal callback */
	    stdin_rtn,
	    stdout_rtn,
	    stderr_rtn
	);
	if(pid < 0)
	    return(pid);

	/* Are we the child? */
	if(pid == 0)
	{
	    /* Set up the shell name, shell arguments, and command */
	    gchar *argv[4];
	    argv[0] = (gchar *)g_basename(shell_path);
	    argv[1] = (gchar *)shell_args;
	    argv[2] = (gchar *)cmd;
	    argv[3] = NULL;

	    /* Execute the command */
	    (void)execve(
		(const char *)shell_path,	/* Full path to the shell */
		(char **)argv,			/* Shell name, shell arguments,
						 * and command */
		environ				/* Environment variables list */
	    );

	    /* This point is only reached if execve() fails */
	    exit(1);
	}

	return(pid);
}

gint edv_system_shell(
	const gchar *cmd,
	const gchar *shell_path,
	const gchar *shell_args
)
{
	return(edv_system_shell_streams(
	    cmd,
	    shell_path,
	    shell_args,
	    NULL, NULL, NULL			/* Do not create any standard
						 * streams */
	));
}

gint edv_system(const gchar *cmd)
{
	return(edv_system_shell(
	    cmd,
	    NULL,				/* Use the detault shell */
	    NULL				/* No shell arguments */
	));
}

/*
 *	Executes the command using a particular shell and blocks until
 *	it has exited.
 *
 *	The cmd specifies the command. The command must contain an
 *	absolute path to an executable and any subsequent arguments or
 *	shell tokens supported by the shell.
 *
 *	The shell_path specifies the full path to the shell. If
 *	shell_path is NULL then the shell specified by the environment
 *	variable SHELL will be used. If no environment variable is
 *	specified then "/bin/sh" will be used.
 *
 *	The shell_args specifies the shell's arguments. If shell_args
 *	is NULL then "-c" will be used.
 *
 *	If status_rtn is not NULL then the return value of the command
 *	will be set to *status_rtn.
 *
 *	Returns the process ID or negative on error.
 */
gint edv_system_wait_shell(
	const gchar *cmd,
	const gchar *shell_path,
	const gchar *shell_args,
	gint *status_rtn
)
{
	gint pid;

	if(status_rtn != NULL)
	    *status_rtn = 0;

	/* Use the default shell? */
	if(STRISEMPTY(shell_path))
	    shell_path = g_getenv(ENV_VAR_NAME_SHELL);
	else if(!g_path_is_absolute(shell_path))
	    shell_path = g_getenv(ENV_VAR_NAME_SHELL);
	if(shell_path == NULL)
	    shell_path = EDV_PATH_PROG_DEF_SHELL;

	/* Use the default shell arguments? */
	if(STRISEMPTY(shell_args))
	    shell_args = EDV_ARGS_DEF_SHELL;

	/* No command specified? */
	if(STRISEMPTY(cmd))
	{
	    errno = EINVAL;
	    return(-2);
	}

	/* Create a child process */
	pid = edv_fork();
	if(pid < 0)
	    return(pid);

	/* Are we the child? */
	if(pid == 0)
	{
	    /* Set up the shell name, shell arguments, and command */
	    gchar *argv[4];
	    argv[0] = (gchar *)g_basename(shell_path);
	    argv[1] = (gchar *)shell_args;
	    argv[2] = (gchar *)cmd;
	    argv[3] = NULL;

	    /* Execute the command */
	    (void)execve(
		(const char *)shell_path,	/* Full path to the shell */
		(char **)argv,			/* Shell name, shell arguments,
						 * and command */
		environ				/* Environment variables list */
	    );

	    /* This point is only reached if execve() fails */
	    exit(1);
	}
	else
	{
	    /* We are the parent, wait/block until the child has exited */
	    gint status;
	    const gint wait_status = (gint)waitpid(
		(int)pid,
		&status,
		0				/* No wait options */
	    );
	    if(wait_status == -1)
	    {
		const gint error_code = (gint)errno;

		/* If WNOHANG was not specified and an unblocked
		 * signal or SIGCHLD was caught, then ERESTARTSYS is
		 * returned by the system call, however the library
		 * interface is not allowed to return ERESTARTSYS, but
		 * will return EINTR
		 */
#if defined(ERESTARTSYS) && defined(EINTR)
		if((error_code == ERESTARTSYS) ||
		   (error_code == EINTR)
		)
#elif defined(EINTR)
		if(error_code == EINTR)
#else
		if(FALSE)
#endif
		{
		    /* An unblocked signal or SIGCHLD was caught */
		}
	    }
	   if(status_rtn != NULL)
		*status_rtn = status;
	}

	return(pid);
}

gint edv_system_wait(
	const gchar *cmd,
	gint *status_rtn
)
{
	return(edv_system_wait_shell(
	    cmd,
	    NULL,				/* Use the default shell */
	    NULL,				/* No shell arguments */
	    status_rtn
	));
}
