/*
                    Endeavour Mark II - Path Utilities

	These functions do not depend on Endeavour 2 or GTK+.

	See also: edv_directory.h edv_link.h edv_utils.h

 */

#ifndef EDV_PATH_H
#define EDV_PATH_H

#include <glib.h>


/*
 *	Counts the number of path compoents.
 *
 *	Empty compoents (double deliminators) and "." compoents will
 *	not be counted.
 *
 *	Parent relative compoents such as ".." will substract from
 *	the number of compoents, so 0 or negative can be returned.
 *
 *	The path specifies the path. If path is a full path then the
 *	toplevel will be counted as one compoent.
 *
 *	Returns the number of compoents, including 0 or negative
 *	since relative compoents such as ".." are counted.
 */
extern gint edv_path_depth(const gchar *path);


/*
 *	Removes any tailing deliminators in the path.
 *
 *	The path specifies the path string which may be modified.
 */
extern void edv_path_strip(gchar *path);

/*
 *	Simplifies the path.
 *
 *	Removes any tailing deliminators and reduces any occurances
 *	of "/.." or "/.".
 *
 *	The path specifies the path string which may be modified.
 */
extern void edv_path_simplify(gchar *path);


/*
 *	Checks if the path's destination exists.
 *
 *	The path specifies the path.
 *
 *	Returns TRUE if the path's destination exists or FALSE if it
 *	does not or an error occured.
 */
extern gboolean edv_path_exists(const gchar *path);

/*
 *	Checks if the path exists locally.
 *
 *	The path specifies the path.
 *
 *	Returns TRUE if the path exists locally or FALSE if it does
 *	not or an error occured.
 */
extern gboolean edv_path_lexists(const gchar *path);

/*
 *	Checks if the path is readable by the current permissions of
 *	the process.
 *
 *	Note that this function does not check the actual permissions
 *	set on the object, certain users such as the super user may
 *	still be able to read the object even when its permissions
 *	are not set so.
 *
 *	The path specifies the path.
 */
extern gboolean edv_path_is_readable(const gchar *path);

/*
 *	Checks if the path is writable by the current permissions of
 *	the process.
 *
 *	Note that this function does not check the actual permissions
 *	set on the object, certain users such as the super user may
 *	still be able to write the object even when its permissions
 *	are not set so.
 *
 *	The path specifies the path.
 */
extern gboolean edv_path_is_writable(const gchar *path);

/*
 *	Checks if the path is executable by the current permissions of
 *	the process.
 *
 *	Note that this function does not check the actual permissions
 *	set on the object, certain users such as the super user may
 *	still be able to execute the object even when its permissions
 *	are not set so.
 *
 *	The path specifies the path.
 */
extern gboolean edv_path_is_executable(const gchar *path);

/*
 *	Checks if the path is "hidden" (a name starting with a '.'
 *	character).
 *
 *	The path specifies the path.
 */
extern gboolean edv_path_is_hidden(const gchar *path);

/*
 *	Checks if the parent path is a parent or grand parent of the
 *	child path or is the same as the child path.
 *
 *	The parent_path specifies the parent path which must be a
 *	full path.
 *
 *	The path specifies the child path which must be a full path.
 *
 *	Returns TRUE if parent_path is a parent or grand parent of
 *	path or is the same as path or FALSE if otherwise.
 */
extern gboolean edv_path_is_parent(
        const gchar *parent_path,
        const gchar *path
);

/*
 *	Checks if the path exists and its destination is a directory.
 *
 *	The path specifies the path.
 */
extern gboolean edv_path_is_directory(const gchar *path);

/*
 *	Checks if the path exists locally and is locally a directory.
 *
 *	The path specifies the path.
 */
extern gboolean edv_path_is_ldirectory(const gchar *path);


/*
 *	Gets the extension from the name.
 *
 *	The name specifies the object's name without the path.
 *
 *	Returns a pointer within name starting at the '.' character
 *	describing the extension or NULL if no extension is found.
 */
extern const gchar *edv_name_get_extension(const gchar *name);

/*
 *	Gets the extension from the path.
 * 
 *	The path specifies the path.
 *
 *	Returns a pointer within path starting at the '.' character
 *	describing the extension or NULL if no extension is found.
 */
extern const gchar *edv_path_get_extension(const gchar *path);


/*
 *	Checks if the name matches the extension or a list of
 *	extensions.
 *
 *	The name specifies the object's name without the path.
 *
 *	The ext specifies either an extension or a space-separated
 *	list of extensions. The extension must start with the '.'
 *	character.
 *
 *	Wild cards such as '*' and '?' are accepted in the
 *	space-separated extensions list.
 *
 *	The following example will return TRUE.
 *
 *	name	= "myfile.doc"
 *	ext	= ".txt .doc *rc Makefile*"
 *
 *	Returns TRUE if at least one of the extensions matches the
 *	name's extension.
 */
extern gboolean edv_name_has_extension(
	const gchar *name,
	const gchar *ext
);

/*
 *	Joins two paths togeather.
 *
 *	The parent_path specifies the parent path which must be a full
 *	path.
 *
 *	The child_path specifies the child path. If child_path is NULL
 *	then a copy of parent_path is returned. If child_path is a full
 *	path then a copy of child_path is returned. Otherwise a copy of
 *	the parent_path prefixed to the child_path is returned.
 *
 *	Returns a dynamically allocated string describing the joined
 *	path or NULL on error.
 */
extern gchar *edv_paths_join(
	const gchar *parent_path,
	const gchar *child_path
);

/*
 *	Shortens a path if it exceeds a maximum number of characters
 *	for display purposes.
 *
 *	The path specifies the path.
 *
 *	The max specifies the maximum number of characters allowed.
 *
 *	Returns a dynamically allocated copy of the path that is no
 *	longer than max characters. A "..." will be prefixed to the
 *	returned string contents if it was longer than max
 *	characters.
 */
extern gchar *edv_path_shorten(
	const gchar *path,
	const gint max
);

/*
 *	Evaluates a path.
 *
 *	The parent_path specifies the string describing the parent
 *	path or current location. It is used only if path is not a
 *	full path. If parent_path is NULL or not a full path then the
 *	toplevel directory will be used as the parent path instead.
 *
 *	The path specifies the string describing the path to be
 *	evaluated.
 *
 *	The path will be evaulated as follows:
 *
 *	Checks if the path has a "file://" prefix and, if it does,
 *	removes it from the path.
 *
 *	Checks if the path is "." and, if it does, replaces the
 *	path with the parent path.
 *
 *	Checks if the path is ".." and, if it does, replaces the
 *	path with the parent of the parent path.      
 *
 *	Checks if the path has a "~" prefix and, if it does,
 *	substitutes it with the home directory.
 *
 *	Postfixed to the parent path if path is not a full path.
 *
 *	Simplified, all occurances of ".." evaluated.
 *
 *	Tailing directory deliminators will be removed.
 *
 *	Returns a dynamically allocated string describing the
 *	evaluated path or NULL on error.
 */
extern gchar *edv_path_evaluate(
	const gchar *parent_path,
	const gchar *path
);


/*
 *	Complete Path Status Codes:
 */
typedef enum {
	EDV_COMPLETE_PATH_SUCCESS,		/* Successfully completed a
						 * full path to an existing
						 * object */
	EDV_COMPLETE_PATH_NONE,			/* General error or path not
						 * completed */
	EDV_COMPLETE_PATH_AMBIGUOUS,		/* Completed a full path to
						 * an existing object but there
						 * are other objects a with
						 * similar name */
	EDV_COMPLETE_PATH_PARTIAL		/* Path partially completed */
} EDVCompletePathStatus;

/*
 *	Completes the path.
 *
 *	The path specifies the string describing the path to complete,
 *	this string may be modified by this function. The path must be
 *	a full path. In the case of an empty path, the path will be
 *	returned describing toplevel.
 *
 *	If status is not NULL then *status will be set to one of
 *	EDV_COMPLETE_PATH_*.
 *
 *	Returns the reallocated path on success or NULL on error.
 */
extern gchar *EDVCompletePath(
	gchar *path,
	EDVCompletePathStatus *status
);

/*
 *	Gets the pointer to the child path portion of the path.
 *
 *	The parent_path specifies the parent path which must be a
 *	full path.
 *
 *	The path specifies the child path which must be a full path.
 *
 *	Returns the pointer to the start of the child in path or
 *	NULL if path is not a child of parent_path or if path and
 *	parent_path are the same.
 */
extern const gchar *edv_path_child(
        const gchar *parent_path,
        const gchar *path
);

/*
 *	Generates a relative path from a starting path to a
 *	destination path.
 *
 *	The start_path specifies the starting path which must be a
 *	full path.
 *
 *	The dest_path specifies the destination path which must be a
 *	full path.
 *
 *	Returns a dynamically allocated string describing the
 *	relative path or NULL on error.
 */
extern gchar *edv_path_plot_relative(
	const gchar *start_path,
	const gchar *dest_path
);


#endif	/* EDV_PATH_H */
