#include <string.h>
#include <errno.h>
#include <fnmatch.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <glib.h>
#include <unistd.h>

#include "../../include/disk.h"

#include "edv_path.h"

#include "config.h"


static gchar *G_STRCAT(gchar *sa, const gchar *sb);

/* Depth */
gint edv_path_depth(const gchar *path);

/* Simplify */
void edv_path_strip(gchar *path);
void edv_path_simplify(gchar *path);

/* Checking */
gboolean edv_path_exists(const gchar *path);
gboolean edv_path_lexists(const gchar *path);
gboolean edv_path_is_readable(const gchar *path);
gboolean edv_path_is_writable(const gchar *path);
gboolean edv_path_is_executable(const gchar *path);
gboolean edv_path_is_hidden(const gchar *path);
gboolean edv_path_is_parent(
	const gchar *parent_path,
	const gchar *path
);
gboolean edv_path_is_directory(const gchar *path);
gboolean edv_path_is_ldirectory(const gchar *path);

/* Extensions */
const gchar *edv_name_get_extension(const gchar *name);
const gchar *edv_path_get_extension(const gchar *path);
static gboolean edv_name_has_extension_iterate(
	const gchar *name, const gint name_len,
	const gchar *ext, const gint ext_len
);
gboolean edv_name_has_extension(
	const gchar *name,
	const gchar *ext
);

/* Generating */
gchar *edv_paths_join(
	const gchar *parent_path,
	const gchar *child_path
);
gchar *edv_path_shorten(
	const gchar *path,
	const gint max
);
gchar *edv_path_evaluate(
	const gchar *parent,
	const gchar *path
);
gchar *EDVCompletePath(
	gchar *path,
	EDVCompletePathStatus *status
);
const gchar *edv_path_child(
	const gchar *parent_path,
	const gchar *path
);
gchar *edv_path_plot_relative(
	const gchar *start_path,
	const gchar *dest_path
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)

#define ISBLANK(c)	(((c) == ' ') || ((c) == '\t'))

#define G_STRPFX(_s_,_pfx_)	(			\
 (strncmp((_s_),(_pfx_),strlen(_pfx_))) ?		\
  FALSE : TRUE						\
)


static gchar *G_STRCAT(gchar *sa, const gchar *sb)
{
	if(sa != NULL) {
		if(sb != NULL) {
			gchar *sc = g_strconcat(
				sa,
				sb,
				NULL
			);
			if(sc != NULL)
			{
				g_free(sa);
				sa = sc;
			}
		}
	} else {
		if(sb != NULL)
			sa = g_strdup(sb);
		else
			sa = g_strdup("");
	}
	return(sa);
}


/*
 *	Counts the number of path compoents.
 *
 *	Empty compoents (double deliminators) and "." compoents will
 *	not be counted.
 *
 *	Parent relative compoents such as ".." will substract from
 *	the number of compoents, so 0 or negative can be returned.
 *
 *	The path specifies the path. If path is a full path then the
 *	toplevel will be counted as one compoent.
 *
 *	Returns the number of compoents, including 0 or negative
 *	since relative compoents such as ".." are counted.
 */
gint edv_path_depth(const gchar *path)
{
	gint ncompoents;
	const gchar *s;

	if(path == NULL)
	{
		errno = EINVAL;
		return(0);
	}

	/* Start counting compoents */
	ncompoents = 0;

	/* Count and seek past toplevel */
	s = path;
	if(*s == G_DIR_SEPARATOR)
	{
		/* Count toplevel */
		ncompoents++;

		/* Seek to the start of the second compoent or end of
		 * string
		 */
		do {
			s++;
		} while(*s == G_DIR_SEPARATOR);
	}

	/* We should now be either at the start of the second compoent
	 * or end of string
	 *
	 * Count subsequent compoents
	 */
	while(*s != '\0')
	{
		/* Potential special notation? */
		if(*s == '.')
		{
			/* Seek past the first character and then
			 * do further checks on subsequent characters
			 * to determine if this is a special notation
			 */
			s++;

			/* Parent ".."? */
			if(*s == '.')
			{
				s++;
				if((*s == '\0') || (*s == G_DIR_SEPARATOR))
				{
					/* Parent notation
					 *
					 * Uncount a compoent
					 */
					ncompoents--;

					/* Seek to the start of the
					 * next compoent or end of
					 * string
					 */
					if(*s == G_DIR_SEPARATOR)
					{
						do {
							s++;
						} while(*s == G_DIR_SEPARATOR);
					}

					continue;
				}
			}
			/* Current "."? */
			else if((*s == '\0') || (*s == G_DIR_SEPARATOR))
			{
				/* Do not count this compoent
				 *
				 * Seek to the start of the next
				 * compoent or end of string
				 */
				if(*s == G_DIR_SEPARATOR)
				{
					do {
						s++;
					} while(*s == G_DIR_SEPARATOR);
				}
				continue;
			}
		}

		/* Count this compoent */
		ncompoents++;

		/* Seek to the next deliminator or end of string */
		do {
			s++;
		} while((*s != '\0') && (*s != G_DIR_SEPARATOR));

		/* If we are at another deliminator then seek past it
		 * to the start of the next compoent or end of string
		 */
		if(*s == G_DIR_SEPARATOR)
		{
			do {
				s++;
			} while(*s == G_DIR_SEPARATOR);
		}
	}

	return(ncompoents);
}


/*
 *	Removes any tailing deliminators in the path.
 *
 *	The path specifies the path string which may be modified.
 */
void edv_path_strip(gchar *path)
{
	if(STRISEMPTY(path))
		return;

	/* Remove any tailing deliminator */
	StripPath((char *)path);
}

/*
 *	Simplifies the path.
 *
 *	Removes any tailing deliminators and reduces any occurances
 *	of "/.." or "/.".
 *
 *	The path specifies the path string which may be modified.
 */
void edv_path_simplify(gchar *path)
{
	gboolean is_absolute;

	if(STRISEMPTY(path))
		return;

	is_absolute = g_path_is_absolute(path);

	/* Remove any tailing deliminator */
	edv_path_strip(path);

	/* Reduce all occurances of "/.." in the path */
	SimplifyPath((char *)path);

	/* If the path was originally a full path and now its
	 * become an empty string then set path as toplevel
	 */
	if(is_absolute && (*path == '\0'))
		(void)strcpy(
			(char *)path,
			G_DIR_SEPARATOR_S
		);
}


/*
 *	Checks if the path's destination exists.
 *
 *	The path specifies the path.
 *
 *	Returns TRUE if the path's destination exists or FALSE if it
 *	does not or an error occured.
 */
gboolean edv_path_exists(const gchar *path)
{
	struct stat stat_buf;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(FALSE);
	}

	if(stat((const char *)path, &stat_buf))
		return(FALSE);
	else
		return(TRUE);
}

/*
 *	Checks if the path exists locally.
 *
 *	The path specifies the path.
 *
 *	Returns TRUE if the path exists locally or FALSE if it does
 *	not or an error occured.
 */
gboolean edv_path_lexists(const gchar *path)
{
	struct stat lstat_buf;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(FALSE);
	}

	if(lstat((const char *)path, &lstat_buf))
		return(FALSE);
	else
		return(TRUE);
}

/*
 *	Checks if the path is readable by the current permissions of
 *	the process.
 *
 *	Note that this function does not check the actual permissions
 *	set on the object, certain users such as the super user may
 *	still be able to read the object even when its permissions
 *	are not set so.
 *
 (	The path specifies the path.
 */
gboolean edv_path_is_readable(const gchar *path)
{
	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(FALSE);
	}

	if(access((const char *)path, R_OK) == 0)
		return(TRUE);
	else
		return(FALSE);
}

/*
 *	Checks if the path is writable by the current permissions of
 *	the process.
 *
 *	Note that this function does not check the actual permissions
 *	set on the object, certain users such as the super user may
 *	still be able to write the object even when its permissions
 *	are not set so.
 *
 *	The path specifies the path.
 */
gboolean edv_path_is_writable(const gchar *path)
{
	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(FALSE);
	}

	if(access((const char *)path, W_OK) == 0)
		return(TRUE);
	else
		return(FALSE);
}

/*
 *	Checks if the path is executable by the current permissions of
 *	the process.
 *
 *	Note that this function does not check the actual permissions
 *	set on the object, certain users such as the super user may
 *	still be able to execute the object even when its permissions
 *	are not set so.
 *
 *	The path specifies the path.
 */
gboolean edv_path_is_executable(const gchar *path)
{
	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(FALSE);
	}

	if(access((const char *)path, X_OK) == 0)
		return(TRUE);
	else
		return(FALSE);
}

/*
 *	Checks if the path is "hidden" (a name starting with a '.'
 *	character).
 *
 *	The path specifies the path.
 */
gboolean edv_path_is_hidden(const gchar *path)
{
	const gchar *name;

	if(path == NULL)
		return(FALSE);

	name = g_basename(path);
	if(name == NULL)
		name = path;

	if(name[0] != '.')
		return(FALSE);

	if(name[1] == '\0')
		return(FALSE);

	if(name[1] == '.')
	{
		if(name[2] == '\0')
			return(FALSE);
	}

	return(TRUE);
}

/*
 *	Checks if the parent path is a parent or grand parent of the
 *	child path or is the same as the child path.
 *
 *	The parent_path specifies the parent path which must be a
 *	full path.
 *
 *	The path specifies the child path which must be a full path.
 *
 *	Returns TRUE if parent_path is a parent or grand parent of
 *	path or is the same as path or FALSE if otherwise.
 */
gboolean edv_path_is_parent(
	const gchar *parent_path,
	const gchar *path
)
{
	gboolean status;
	gchar		*lparent_path,
					*lpath;

	if((parent_path == NULL) || (path == NULL))
	{
		errno = EINVAL;
		return(FALSE);
	}

	/* The parent path and child path must both be absolute paths */
	if(!g_path_is_absolute(parent_path) || !g_path_is_absolute(path))
	{
		errno = EINVAL;
		return(FALSE);
	}

	/* Is the child path the same pointer as the parent path? */
	if(path == parent_path)
		return(TRUE);

	lpath = g_strdup(path);
	lparent_path = g_strdup(parent_path);

	/* Remove any tailing deliminators (because we don't know
	 * how many extraneous deliminators may be at the end)
	 */
	edv_path_strip(lpath);
	edv_path_strip(lparent_path);

	/* Append a single tailing deliminator to the paths so that
	 * G_STRPFX() can properly distinquish and check path segments
	 * and avoid matching partial path names
	 */
	lpath = G_STRCAT(
		lpath,
		G_DIR_SEPARATOR_S
	);
	lparent_path = G_STRCAT(
		lparent_path,
		G_DIR_SEPARATOR_S
	);

	/* Check if the parent path is a prefix of the child path */
	status = G_STRPFX(lpath, lparent_path);

	g_free(lpath);
	g_free(lparent_path);

	return(status);
}

/*
 *	Checks if the path exists and its destination is a directory.
 *
 *	The path specifies the path.
 */
gboolean edv_path_is_directory(const gchar *path)
{
	struct stat stat_buf;

	if(path == NULL)
	{
		errno = EINVAL;
		return(FALSE);
	}

	if(stat((const char *)path, &stat_buf))
		return(FALSE);

#ifdef S_ISDIR
	if(S_ISDIR(stat_buf.st_mode))
#else
	if(FALSE)
#endif
	{
		errno = EISDIR;
		return(TRUE);
	}
	else
	{
		errno = ENOTDIR;
		return(FALSE);
	}
}

/*
 *	Checks if the path exists locally and is locally a directory.
 *
 *	The path specifies the path.
 */
gboolean edv_path_is_ldirectory(const gchar *path)
{
	struct stat lstat_buf;

	if(path == NULL)
	{
		errno = EINVAL;
		return(FALSE);
	}

	if(lstat((const char *)path, &lstat_buf))
		return(FALSE);

#ifdef S_ISDIR
	if(S_ISDIR(lstat_buf.st_mode))
#else
	if(FALSE)
#endif
	{
		errno = EISDIR;
		return(TRUE);
	}
	else
	{
		errno = ENOTDIR;
		return(FALSE);
	}
}


/*
 *	Gets the extension from the name.
 *
 *	The name specifies the object's name without the path.
 *
 *	Returns a pointer within name starting at the '.' character
 *	describing the extension or NULL if no extension is found.
 */
const gchar *edv_name_get_extension(const gchar *name)
{
	if(name == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	return((const gchar *)strchr((const char *)name, '.'));
}

/*
 *	Gets the extension from the path.
 *
 *	The path specifies the path.
 *
 *	Returns a pointer within path starting at the '.' character
 *	describing the extension or NULL if no extension is found.
 */
const gchar *edv_path_get_extension(const gchar *path)
{
	const gchar *name;

	if(path == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	name = g_basename(path);
	if(name == NULL)
		name = path;

	return(edv_name_get_extension(name));
}

/*
 *	Called by edv_name_has_extension() to check if ext matches the path.
 *
 *	The name specifies the name without the path and may not be
 *	NULL.
 *
 *	The ext expecifies one extension (no spaces) with wildcards
 *	allowed and may not be NULL.
 */
static gboolean edv_name_has_extension_iterate(
	const gchar *name, const gint name_len,
	const gchar *ext, const gint ext_len
)
{
	/* Extension starts with a '.' deliminator? */
	if(*ext == '.')
	{
		const gchar *name_ext = name + (name_len - ext_len);

		/* Check if ext is a suffix of name */
		if(name_len < ext_len)
			return(FALSE);

		return((g_strcasecmp(name_ext, ext)) ? FALSE : TRUE);
	}
	/* Not an extension, so use fnmatch() */
	else
	{
		return((fnmatch(ext, name, 0) == 0) ? TRUE : FALSE);
	}
}


/*
 *	Checks if the name matches the extension or a list of
 *	extensions.
 *
 *	The name specifies the object's name without the path.
 *
 *	The ext specifies either an extension or a space-separated
 *	list of extensions. The extension must start with the '.'
 *	character.
 *
 *	Wild cards such as '*' and '?' are accepted in the
 *	space-separated extensions list.
 *
 *	The following example will return TRUE.
 *
 *	name    = "myfile.doc"
 *	ext     = ".txt .doc *rc Makefile*"
 *
 *	Returns TRUE if at least one of the extensions matches the
 *	name's extension.
 */
gboolean edv_name_has_extension(
	const gchar *name,
	const gchar *ext
)
{
	gint		name_len,
			ext_len;
	const gchar	*ss,
			*ext_ptr;
	gchar		*st,
			cur_ext[NAME_MAX];

	if(STRISEMPTY(name) || STRISEMPTY(ext))
		return(FALSE);

	name_len = strlen(name);

	/* Set ext_ptr to start of extensions list string, seeking past
	 * any initial spaces
	 */
	ext_ptr = ext;
	while(ISBLANK(*ext_ptr))
		ext_ptr++;

	/* Iterate through each word in the extensions list */
	while(*ext_ptr != '\0')
	{
		/* Copy this word in the extensions list string to
		 * cur_ext and calculate this extension's length as
		 * ext_len
		 */
		for(ext_len = 0,
			st = cur_ext,
			ss = ext_ptr;
			(ext_len < (gint)(sizeof(cur_ext) - 1)) &&
			!ISBLANK(*ss) && (*ss != '\0');
		    ext_len++
		)
			*st++ = *ss++;
		*st = '\0';

		/* Check this extension word matches */
		if(edv_name_has_extension_iterate(name, name_len, cur_ext, ext_len))
			return(TRUE);

		/* At this point ss should be at the end of the word (at
		 * the first space) or at the end of the extensions list
		 * string
		 */
		ext_ptr = ss;
		while(ISBLANK(*ext_ptr))
			ext_ptr++;
	}

	return(FALSE);
}

/*
 *	Joins two paths togeather.
 *
 *	The parent_path specifies the parent path which must be a full
 *	path.
 *
 *	The child_path specifies the child path. If child_path is NULL
 *	then a copy of parent_path is returned. If child_path is a full
 *	path then a copy of child_path is returned. Otherwise a copy of
 *	the parent_path prefixed to the child_path is returned.
 *
 *	Returns a dynamically allocated string describing the joined
 *	path or NULL on error.
 */
gchar *edv_paths_join(
	const gchar *parent_path,
	const gchar *child_path
)
{
	const gchar *parent_end;

	if(parent_path == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	if(!g_path_is_absolute(parent_path))
	{
		errno = EINVAL;
		return(NULL);
	}

	if(child_path == NULL)
		return(g_strdup(parent_path));

	if(parent_path == child_path)
		return(g_strdup(parent_path));

	/* If the child path is a full path then return a copy of the
	 * child
	 */
	if(g_path_is_absolute(child_path))
		return(g_strdup(child_path));

	/* At this point both parent_path and child_path are not NULL,
	 * not the same, and child_path is a relative path
	 */

	/* Seek to the last character of parent_path */
	parent_end = parent_path;
	while(*parent_end != '\0')
		parent_end++;
	if(parent_end > parent_path)
		parent_end--;

	/* Is the parent possibly an empty string? */
	if(parent_end == parent_path)
		return(g_strconcat(
			G_DIR_SEPARATOR_S,
			child_path,
			NULL
		));
	/* Does parent_path have a tailing deliminator? */
	else if(*parent_end == G_DIR_SEPARATOR)
		return(g_strconcat(
			parent_path,
			child_path,
			NULL
		));
	else
		return(g_strconcat(
			parent_path,
			G_DIR_SEPARATOR_S,
			child_path,
			NULL
		));
}

/*
 *	Shortens a path if it exceeds a maximum number of characters 
 *	for display purposes.
 *
 *	The path specifies the path.
 *
 *	The max specifies the maximum number of characters allowed.
 *
 *	Returns a dynamically allocated copy of the path that is no
 *	longer than max characters. A "..." will be prefixed to the
 *	returned string contents if it was longer than max
 *	characters.
 */
gchar *edv_path_shorten(
	const gchar *path,
	const gint max
)
{
	gint len;

	if(path == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	len = STRLEN(path);
	if((len > max) && (max > 3))
	{
		/* Need to shorten string */
		const gint i = len - max + 3;

		return(g_strdup_printf(
			"...%s", &path[i]
		));
	}
	else
	{
		return(g_strdup(path));
	}
}


/*
 *	Evaluates a path.
 *
 *	The parent_path specifies the string describing the parent
 *	path or current location. It is used only if path is not a
 *	full path. If parent_path is NULL or not a full path then the
 *	toplevel directory will be used as the parent path instead.
 *
 *	The path specifies the string describing the path to be
 *	evaluated.
 *
 *	The path will be evaulated as follows:
 *
 *	Checks if the path has a "file://" prefix and, if it does,
 *	removes it from the path.
 *
 *	Checks if the path is "." and, if it does, replaces the
 *	path with the parent path.
 *
 *	Checks if the path is ".." and, if it does, replaces the
 *	path with the parent of the parent path.
 *
 *	Checks if the path has a "~" prefix and, if it does,
 *	substitutes it with the home directory.
 *
 *	Postfixed to the parent path if path is not a full path.
 *
 *	Simplified, all occurances of ".." evaluated.
 *
 *	Tailing directory deliminators will be removed.
 *
 *	Returns a dynamically allocated string describing the
 *	evaluated path or NULL on error.
 */
gchar *edv_path_evaluate(
	const gchar *parent_path,
	const gchar *path
)
{
	gchar *eval_path;

	if(path == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	if(parent_path == NULL)
#if defined(_WIN32)
		parent_path = "\\";
#else
		parent_path = "/";
#endif
	else if(!g_path_is_absolute(parent_path))
#if defined(_WIN32)
		parent_path = "\\";
#else
		parent_path = "/";
#endif

	/* Remove the "file://" prefix as needed */
	if(G_STRPFX(path, "file://"))
	{
		const gint pfx_len = STRLEN("file://");
		eval_path = g_strdup(path + pfx_len);
	}
	/* Current directory? */
	else if(!strcmp((const char *)path, "."))
	{
		eval_path = g_strdup(parent_path);
	}
	/* Parent directory? */
	else if(!strcmp((const char *)path, ".."))
	{
		eval_path = g_dirname(parent_path);
	}
	/* Home directory prefix? */
	else if(*path == '~')
	{
		/* Prefix the value from the HOME environment variable to
		 * the input path and generate the new path
		 */
		const gchar	*home = g_getenv(ENV_VAR_NAME_HOME),
				*sub_path = path + 1;

		/* If home directory environment variable value was not set
		 * then assume toplevel
		 */
		if(home == NULL)
#if defined(_WIN32)
			home = "\\";
#else
			home = "/";
#endif

		/* Seek sub_path, which is now after the prefix, past
		 * any deliminator characters
		 */
		while(*sub_path == G_DIR_SEPARATOR)
			sub_path++;

		/* Create the evaluated full path with the home directory
		 * prefixed
		 */
		eval_path = g_strconcat(
			home,
			G_DIR_SEPARATOR_S,
			sub_path,
			NULL
		);
	}
	else
	{
		/* Create the evaluated full path */
		eval_path = g_path_is_absolute(path) ?
			g_strdup(path) :
			edv_paths_join(parent_path, path);
	}
	if(eval_path == NULL)
	{
		errno = ENOMEM;
		return(NULL);
	}

	/* Begin simplifying the generated path */

	/* Simplify path, reducing occurances of "../" */
	SimplifyPath((char *)eval_path);

	/* Remove any tailing deliminators */
	edv_path_strip(eval_path);

	return(eval_path);
}

/*
 *	Completes the path.
 *
 *	The path specifies the string describing the path to complete,
 *	this string may be modified by this function. The path must
 *	be a full path. In the case of an empty path, the path will be
 *	returned describing toplevel.
 *
 *	If status is not NULL then *status will be set to one of
 *	EDV_COMPLETE_PATH_*.
 *
 *	Returns the reallocated path on success or NULL on error.
 */
gchar *EDVCompletePath(
	gchar *path,
	EDVCompletePathStatus *status
)
{
	int _status;

	path = (gchar *)CompletePath((char *)path, &_status);
	if(status != NULL)
	{
		switch(_status)
		{
		  case COMPLETE_PATH_SUCCESS:
			*status = EDV_COMPLETE_PATH_SUCCESS;
			break;
		  case COMPLETE_PATH_NONE:
			*status = EDV_COMPLETE_PATH_NONE;
			break;
		  case COMPLETE_PATH_AMBIGUOUS:
			*status = EDV_COMPLETE_PATH_AMBIGUOUS;
			break;
		  case COMPLETE_PATH_PARTIAL:
			*status = EDV_COMPLETE_PATH_PARTIAL;
			break;
		}
	}

	return(path);
}

/*
 *	Gets the pointer to the child path portion of the path.
 *
 *	The parent_path specifies the parent path which must be a
 *	full path.
 *
 *	The path specifies the child path which must be a full path.
 *
 *	Returns the pointer to the start of the child in path or
 *	NULL if path is not a child of parent_path or if path and
 *	parent_path are the same.
 */
const gchar *edv_path_child(
	const gchar *parent_path,
	const gchar *path
)
{
	/* Is the child path not a child of the parent path or is not
	 * the parent path?
	 */
	if(!edv_path_is_parent(parent_path, path))
		return(NULL);

	/* Seek past the parent portion */
	path += (gint)strlen((const char *)parent_path);

	/* Seek past any deliminators so that the returned child
	 * path pointer does not start with a deliminator
	 */
	while(*path == G_DIR_SEPARATOR)
		path++;

	/* Was the child path the same as the parent path? */
	if(*path == '\0')
		return(NULL);

	return(path);
}


/*
 *	Generates a relative path from a starting path to a
 *	destination path.
 *
 *	The start_path specifies the starting path. The start_path
 *	must be a full path.
 *
 *	The dest_path specifies the destination path. The dest_path
 *	must be a full path.
 *
 *	Returns a dynamically allocated string describing the
 *	relative path or NULL on error.
 */
gchar *edv_path_plot_relative(
	const gchar *start_path,
	const gchar *dest_path
)
{
	gint i, tar_deliminators = 0;
	gchar *parent_path, *rel_path;
	const gchar *src_s, *tar_s, *tar_s2;

	if(STRISEMPTY(start_path) || STRISEMPTY(dest_path))
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Get the path to the link's parent */
	parent_path = g_dirname(start_path);
	if(parent_path == NULL)
		parent_path = g_strdup(start_path);
	if(parent_path == NULL)
	{
		errno = ENOMEM;
		return(NULL);
	}

	rel_path = NULL;

	/* If the destination and the starting path parents are the
	 * same paths then just return "."
	 */
	if(!strcmp((const char *)dest_path, (const char *)parent_path))
	{
		rel_path = g_strdup(".");
		g_free(parent_path);
		return(rel_path);
	}

	/* Set the starting destination and parent path positions past
	 * their toplevel characters
	 */
	src_s = (gchar *)strchr((char *)dest_path, G_DIR_SEPARATOR);
	if(src_s != NULL)
		src_s++;
	else
		src_s = dest_path + 1;

	tar_s = (gchar *)strchr((char *)parent_path, G_DIR_SEPARATOR);
	if(tar_s != NULL)
		tar_s++;
	else
		tar_s = parent_path + 1;

	/* Seek src_s and tar_s to the first character of which they
	 * differ
	 */
	while((*src_s != '\0') && (*tar_s != '\0'))
	{
		if(*src_s != *tar_s)
			break;

		src_s++;
		tar_s++;
	}
	/* Deliminator at source position where difference was
	 * encountered? If so then we need to decrement deliminator
	 * count by one
	 *
	 * This will get added up to or past 0 further below
	 */
	if(*src_s == G_DIR_SEPARATOR)
		tar_deliminators--;

	/* Seek source position backwards to last deliminator, but keep
	 * it one position ahead of the last deliminator
	 */
	while(src_s > dest_path)
	{
		if(*src_s == G_DIR_SEPARATOR)
		{
			src_s++;
			break;
		}

		src_s--;
	}
	/* If source position seeked all the way back to the beginning
	 * then increment it one past the first character to skip the
	 * toplevel character and thus keep source position infront of
	 * the last deliminator
	 */
	if(src_s <= dest_path)
		src_s = dest_path + 1;


	/* Count deliminators in target path from where target position
	 * differed from the corresponding source position
	 */
	tar_deliminators++;
	tar_s2 = tar_s;
	while(*tar_s2 != '\0')
	{
		if(*tar_s2 == G_DIR_SEPARATOR)
			tar_deliminators++;

		tar_s2++;
	}

	/* Special check, if the target happens to be just toplevel
	 * then do not count any deliminators
	 */
	if(!strcmp((const char *)parent_path, "/"))
		tar_deliminators = 0;

	/* Begin generating new path */
	g_free(rel_path);
	rel_path = g_strdup("");
	for(i = 0; i < tar_deliminators; i++)
		rel_path = G_STRCAT(rel_path, "../");

	rel_path = G_STRCAT(rel_path, src_s);
	if(rel_path == NULL)
	{
		g_free(parent_path);
		errno = ENOMEM;
		return(rel_path);
	}

	/* If new path was generated as an empty string (perhaps if the
	 * source and target directories were the same), then just set
	 * the new path to "." to indicate current directory
	 */
	if(*rel_path == '\0')
	{
		rel_path = G_STRCAT(rel_path, ".");
		if(rel_path == NULL)
		{
			g_free(parent_path);
			errno = ENOMEM;
			return(rel_path);
		}
	}

	/* Remove any tailing deliminators */
	edv_path_strip(rel_path);

	g_free(parent_path);

	return(rel_path);
}
