#include <string.h>
#include <errno.h>
#include <glib.h>

#include "edv_types.h"
#include "edv_utils.h"
#include "edv_process.h"
#include "edv_path.h"
#include "edv_get.h"
#include "edv_vfs_obj.h"
#include "edv_vfs_obj_stat.h"
#include "edv_mime_type.h"
#include "edv_mime_type_get.h"
#include "edv_window.h"
#include "edv_open.h"
#include "edv_context_private.h"

#include "edv_cfg_list.h"
#include "config.h"


static gchar *edv_open_strip_extension(const gchar *s);
static gboolean edv_open_is_file_executable(const gchar *path);

static gchar *edv_open_format_command(
	EDVContext *ctx,
	const gchar *cmd,
	GList *paths_list
);
static EDVMIMEType *edv_open_match_mime_type(
	EDVContext *ctx,
	const gchar *type
);

static gint edv_open_iterate(
	EDVContext *ctx,
	const gchar *path,
	const gchar *command_name
);
gint edv_open(
	EDVContext *ctx,
	GList *paths_list,
	const gchar *command_name
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s) 	(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns a copy of the path or name without the extension.
 */
static gchar *edv_open_strip_extension(const gchar *s)
{
	gint len;
	gchar *path_wo_ext;
	const gchar	*name,
			*ext;

	if(s == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Get the name */
	name = strrchr(s, G_DIR_SEPARATOR);
	if(name != NULL)
		name++;
	else
		name = s;

	/* Get the extension */
	if(*name == '.')
		ext = strchr(name + 1, '.');
	else
		ext = strchr(name, '.');

	/* No extension? */
	if(ext == NULL)
		return(g_strdup(s));

	/* Get the path without the extension */
	len = (gint)(ext - s);
	path_wo_ext = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	if(len > 0)
		memcpy(path_wo_ext, s, len * sizeof(gchar));
	path_wo_ext[len] = '\0';

	/* Need to tack on the ending double quote character? */
	if(*path_wo_ext == '"')
	{
		gchar *s2 = g_strconcat(path_wo_ext, "\"", NULL);
		g_free(path_wo_ext);
		path_wo_ext = s2;
	}

	return(path_wo_ext);
}

/*
 *	Checks if the object specified by path exists, is a file, and
 *	is executable.
 */
static gboolean edv_open_is_file_executable(const gchar *path)
{
	EDVPermissionFlags permissions;
	EDVVFSObject *obj = edv_vfs_object_stat(path);
	if(obj == NULL)
		return(FALSE);

	/* Not a file? */
	if(!EDV_VFS_OBJECT_IS_FILE(obj))
	{
		edv_vfs_object_delete(obj);
		return(FALSE);
	}

	/* Is the file executable? */
	permissions = obj->permissions;
	if((permissions & EDV_PERMISSION_UX) ||
	   (permissions & EDV_PERMISSION_GX) ||
	   (permissions & EDV_PERMISSION_OX)
	)
	{
		edv_vfs_object_delete(obj);
		return(TRUE);
	}

	edv_vfs_object_delete(obj);

	return(FALSE);
}


/*
 *	Performs substitutions to the command string specified by cmd
 *	with the values specified in the context and the paths list.
 *
 *	Returns a dynamically allocated string describing the command  
 *	with all the substitutions made.
 */
static gchar *edv_open_format_command(
	EDVContext *ctx,
	const gchar *cmd,
	GList *paths_list
)
{
	const gchar	*path,
			*first_path,
			*last_path;
	gchar		*s,
			*cmd_rtn,
			*cwd_quoted,
			*display_quoted,
			*home_quoted,
			*names_quoted,
			*paths_quoted,
			*first_name_quoted, *first_name_wo_ext_quoted,
			*last_name_quoted, *last_name_wo_ext_quoted,
			*first_path_quoted, *first_path_wo_ext_quoted,
			*last_path_quoted, *last_path_wo_ext_quoted,
			*pid_quoted;
	GList *glist;

	if(cmd == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Begin creating substitution values */

	/* Current working directory */
	s = edv_getcwd();
	cwd_quoted = g_strdup_printf(
		"\"%s\"",
		(s != NULL) ? s : ""
	);
	g_free(s);

	/* Display address */
	display_quoted = g_strdup_printf(
		"\"%s\"",
		g_getenv(ENV_VAR_NAME_DISPLAY)
	);

	/* Home directory */
	home_quoted = g_strdup_printf(
		"\"%s\"",
		g_getenv(ENV_VAR_NAME_HOME)
	);

	/* All selected object names */
	names_quoted = g_strdup("");
	for(glist = paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		path = (gchar *)glist->data;
		if(path == NULL)
			continue;

		s = g_strconcat(
			names_quoted,
			"\"",
			g_basename(path),
			(glist->next != NULL) ? "\" " : "\"",
			NULL
		);
		g_free(names_quoted);
		names_quoted = s;
	}

	/* All selected object paths */
	paths_quoted = g_strdup("""");
	for(glist = paths_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		path = (gchar *)glist->data;
		if(path == NULL)
			continue;

		s = g_strconcat(
			paths_quoted,
			"\"",
			path,
			(glist->next != NULL) ? "\" " : "\"",
			NULL
		);
		g_free(paths_quoted);
		paths_quoted = s;
	}

	/* First selected object name and path */
	glist = g_list_first(paths_list);
	first_path = (glist != NULL) ? (gchar *)glist->data : NULL;
	if(first_path == NULL)
		first_path = "";
	first_name_quoted = g_strdup_printf("\"%s\"", g_basename(first_path));
	first_name_wo_ext_quoted = edv_open_strip_extension(first_name_quoted);
	first_path_quoted = g_strdup_printf("\"%s\"", first_path);
	first_path_wo_ext_quoted = edv_open_strip_extension(first_path_quoted);

	/* Last selected object name and path */
	glist = g_list_last(paths_list);
	last_path = (glist != NULL) ? (gchar *)glist->data : NULL;
	if(last_path == NULL)
		last_path = "";
	last_name_quoted = g_strdup_printf("\"%s\"", g_basename(last_path));
	last_name_wo_ext_quoted = edv_open_strip_extension(last_name_quoted);
	last_path_quoted = g_strdup_printf("\"%s\"", last_path);
	last_path_wo_ext_quoted = edv_open_strip_extension(last_path_quoted);

	/* Process ID */
	pid_quoted = g_strdup_printf("\"%i\"", edv_pid_get_current());


	/* Begin formatting the command, always use the quoted string
	 * as the value and substitute for the quoted token first then
	 * the unquoted token
	 */
	cmd_rtn = STRDUP(cmd);

#define CMDSUB(_token_,_val_)	{			\
 gchar *s = edv_strsub(cmd_rtn, (_token_), (_val_));	\
 g_free(cmd_rtn);					\
 cmd_rtn = s;						\
}

	/* "%cwd" - Current working directory */
	CMDSUB("\"%cwd\"", cwd_quoted);
	CMDSUB("%cwd", cwd_quoted);

	/* "%display" - Display address */
	CMDSUB("\"%display\"", display_quoted);
	CMDSUB("%display", display_quoted);

	/* "%home" - Home directory */
	CMDSUB("\"%home\"", home_quoted);
	CMDSUB("%home", home_quoted);


	/* "%names" - Names of the all selected objects (without paths) */
	CMDSUB("\"%names\"", names_quoted);
	CMDSUB("%names", names_quoted);

	/* "%paths" - Full paths of the all selected objects */
	CMDSUB("\"%paths\"", last_path_quoted);
	CMDSUB("%paths", last_path_quoted);


	/* "%first_name_wo_ext" - Name of the first selected object
	 * (without path or extension)
	 */
	CMDSUB("\"%first_name_wo_ext\"", first_name_wo_ext_quoted);
	CMDSUB("%first_name_wo_ext", first_name_wo_ext_quoted);

	/* "%first_name" - Name of the first selected object (without
	 * path)
	 */
	CMDSUB("\"%first_name\"", first_name_quoted);
	CMDSUB("%first_name", first_name_quoted);

	/* "%first_path_wo_ext" - Full path to the last selected
	 * object without the extension
	 */
	CMDSUB("\"%first_path_wo_ext\"", first_path_wo_ext_quoted);
	CMDSUB("%first_path_wo_ext", first_path_wo_ext_quoted);

	/* "%first_path" - Full path to the last selected object */
	CMDSUB("\"%first_path\"", first_path_quoted);
	CMDSUB("%first_path", first_path_quoted);


	/* "%name_wo_ext" - Name of the last selected object (without
	 * path or ext)
	 */
	CMDSUB("\"%name_wo_ext\"", last_name_wo_ext_quoted);
	CMDSUB("%name_wo_ext", last_name_wo_ext_quoted);

	/* "%name" - Name of last selected object (without path) */
	CMDSUB("\"%name\"", last_name_quoted);
	CMDSUB("%name", last_name_quoted);

	/* "%path_wo_ext" - Full path to the last selected object
	 * without the extension
	 */
	CMDSUB("\"%path_wo_ext\"", last_path_wo_ext_quoted);
	CMDSUB("%path_wo_ext", last_path_wo_ext_quoted);

	/* "%path" - Full path to last selected object */
	CMDSUB("\"%path\"", last_path_quoted);
	CMDSUB("%path", last_path_quoted);


	/* "%pid" - This program's process id */
	CMDSUB("\"%pid\"", pid_quoted);
	CMDSUB("%pid", pid_quoted);


	/* "%pe" - Full path to the last selected object without
	 * the extension
	 */
	CMDSUB("\"%pe\"", last_path_wo_ext_quoted);
	CMDSUB("%pe", last_path_wo_ext_quoted);

	/* "%p" - Full path to last selected object */
	CMDSUB("\"%p\"", last_path_quoted);
	CMDSUB("%p", last_path_quoted);

	/* "%s" - Full path to last selected object: */
	CMDSUB("\"%s\"", last_path_quoted);
	CMDSUB("%s", last_path_quoted);

#undef CMDSUB

	g_free(cwd_quoted);
	g_free(display_quoted);
	g_free(home_quoted);
	g_free(names_quoted);
	g_free(paths_quoted);
	g_free(first_name_quoted);
	g_free(first_name_wo_ext_quoted);
	g_free(last_name_quoted);
	g_free(last_name_wo_ext_quoted);
	g_free(first_path_quoted);
	g_free(first_path_wo_ext_quoted);
	g_free(last_path_quoted);
	g_free(last_path_wo_ext_quoted);
	g_free(pid_quoted);

	return(cmd_rtn);
}


/*
 *	Gets a MIME Type of class program who's type matches the
 *	specified type.
 */
static EDVMIMEType *edv_open_match_mime_type(
	EDVContext *ctx,
	const gchar *type
)
{
	GList *glist;
	EDVMIMEType *m;

	for(glist = ctx->mime_types_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
			continue;

		if((m->mt_class != EDV_MIME_TYPE_CLASS_PROGRAM) ||
		   STRISEMPTY(m->type)
		)
			continue;

		if(!strcmp((const char *)m->type, (const char *)type))
			return(m);
	}

	errno = ENOENT;

	return(NULL);
}


/*
 *	Opens the object specified by path.
 */
static gint edv_open_iterate(
	EDVContext *ctx,
	const gchar *path,
	const gchar *command_name
)
{
	gint status = 0;
	EDVMIMEType *m = edv_mime_types_list_match_vfs_object_path(
		ctx,
		path
	);
	if(m != NULL)
	{
		/* Found MIME Type associated with this object
		 *
		 * Use the appropriate handler specified by this MIME Type
		 * to open this object
		 */
		switch(m->handler)
		{
		  case EDV_MIME_TYPE_HANDLER_COMMAND:
			if(m->commands_list != NULL)
			{
				EDVMIMETypeCommand *cmd;

				/* If the command's name is specified then get the
				 * command who's name matches the specified command
				 * name
				 */
				if(!STRISEMPTY(command_name))
				{
					cmd = edv_mime_type_match_command_by_name(
						m,
						command_name
					);
				}
				else
				{
					GList *glist = m->commands_list;
					cmd = EDV_MIME_TYPE_COMMAND(glist->data);
				}

				/* Got the command? */
				if((cmd != NULL) ? !STRISEMPTY(cmd->command) : FALSE)
				{
					/* If the command is not an absolute path then
					 * it contains the type (the name) of another
					 * MIME Type in which we must get the MIME Type
					 * that it refers to
					 */
					if(*cmd->command != G_DIR_SEPARATOR)
					{
						const gchar *ref_type = cmd->command;

						cmd = NULL;
						m = edv_open_match_mime_type(
							ctx,
							ref_type
						);
						if(m != NULL)
						{
							GList *glist = m->commands_list;
							if(glist != NULL)
								cmd = EDV_MIME_TYPE_COMMAND(glist->data);
						}
					}

					/* Got the command? */
					if((cmd != NULL) ? !STRISEMPTY(cmd->command) : FALSE)
					{
						gint pid;
						gchar	*parent,
							*cmd_s,
							*shell_prog;
						const gchar	*shell_cmd = cmd->shell_command,
								*shell_args;
						GList *paths_list = NULL;

						/* Record the previous working dir */
						gchar *pwd = edv_getcwd();

						/* Set new working dir as the parent of
						 * the specified path
						 */
						parent = g_dirname(path);
						if(parent != NULL)
						{
							edv_setcwd(parent);
							g_free(parent);
						}

						/* If no shell command was specified by
						 * the MIME Type then use the shell
						 * command specified by the configuration
						 */
						if(STRISEMPTY(shell_cmd))
							shell_cmd = edv_get_s(ctx, EDV_CFG_PARM_PROG_SHELL);

						/* Get the shell program and arguments */
						shell_args = edv_strarg(
							shell_cmd,
							&shell_prog,
							TRUE,	/* Parse escapes */
							TRUE	/* Parse quotes */
						);

						/* Format the command */
						paths_list = g_list_append(
							paths_list,
							STRDUP(path)
						);
						cmd_s = edv_open_format_command(
							ctx,
							cmd->command,
							paths_list
						);
						if(paths_list != NULL)
						{
							g_list_foreach(paths_list, (GFunc)g_free, NULL);
							g_list_free(paths_list);
						}

						/* Execute the command */
						pid = edv_system_shell(
							cmd_s,
							shell_prog,
							shell_args
						);

						g_free(cmd_s);
						g_free(shell_prog);

						if(pid < 0)
							status = -1;

						/* Change back to the previous working dir */
						if(pwd != NULL)
						{
							edv_setcwd(pwd);
							g_free(pwd);
						}
					}
					else
					{
						status = -7;
					}
				}
				else
				{
					/* No command base */
					status = -7;
				}
			}
			else
			{
				/* No command */
				status = -7;
			}
			break;

		  case EDV_MIME_TYPE_HANDLER_EDV_ARCHIVER:
			edv_window_archiver_new(
				ctx,
				path,
				NULL		/* No password */
			);
			edv_context_flush(ctx);
			break;

		  case EDV_MIME_TYPE_HANDLER_EDV_IMAGE_BROWSER:
			edv_window_image_browser_new(ctx, path);
			edv_context_flush(ctx);
			break;

		  case EDV_MIME_TYPE_HANDLER_EDV_RECYCLE_BIN:
			edv_window_recycle_bin_map(ctx);
			edv_context_flush(ctx);
			break;
		}
	}
	else
	{
		/* No MIME Type associated with this object */
		const gchar *def_viewer_cmd = edv_get_s(
			ctx,
			EDV_CFG_PARM_PROG_DEF_VIEWER
		);

		/* If the object executable then run it */
		if(edv_open_is_file_executable(path))
		{
			gint pid;
			gchar	*parent,
				*shell_prog;
			const gchar	*shell_cmd = edv_get_s(ctx, EDV_CFG_PARM_PROG_SHELL),
					*shell_args = edv_strarg(
				shell_cmd,
				&shell_prog,
				TRUE,		/* Parse escapes */
				TRUE		/* Parse quotes */
			);

			/* Record the previous working dir */
			gchar *pwd = edv_getcwd();

			/* Set new working dir as the parent of the specified
			 * path
			 */
			parent = g_dirname(path);
			if(parent != NULL)
			{
				edv_setcwd(parent);
				g_free(parent);
			}

			/* Execute the executable object */
			pid = edv_system_shell(
				path,
				shell_prog,
				shell_args
			);
			if(pid < 0)
				status = -1;

			g_free(shell_prog);

			/* Change back to the previous working dir */
			if(pwd != NULL)
			{
				edv_setcwd(pwd);
				g_free(pwd);
			}
		}
		/* If the default viewer command is defined then open the
		 * object with the default viewer
		 */
		else if(!STRISEMPTY(def_viewer_cmd))
		{
			gint pid;
			gchar	*parent,
				*cmd_s,
				*shell_prog;
			const gchar	*shell_cmd = edv_get_s(ctx, EDV_CFG_PARM_PROG_SHELL),
					*shell_args = edv_strarg(
				shell_cmd,
				&shell_prog,
				TRUE,		/* Parse escapes */
				TRUE		/* Parse quotes */
			);
			GList *paths_list = NULL;

			/* Record the previous working dir */
			gchar *pwd = edv_getcwd();

			/* Set new working dir as the parent of the specified
			 * path
			 */
			parent = g_dirname(path);
			if(parent != NULL)
			{
				edv_setcwd(parent);
				g_free(parent);
			}

			/* Format the command */
			paths_list = g_list_append(
				paths_list,
				STRDUP(path)
			);
			cmd_s = edv_open_format_command(
				ctx,
				def_viewer_cmd,
				paths_list
			);
			if(paths_list != NULL)
			{
				g_list_foreach(paths_list, (GFunc)g_free, NULL);
				g_list_free(paths_list);
			}

			/* Execute the command */
			pid = edv_system_shell(
				cmd_s,
				shell_prog,
				shell_args
			);

			g_free(cmd_s);
			g_free(shell_prog);

			if(pid < 0)
				status = -1;

			/* Change back to the previous working dir */
			if(pwd != NULL)
			{
				edv_setcwd(pwd);
				g_free(pwd);
			}
		}
		/* All else report that there was no method to open the
		 * object
		 */
		else
		{
			status = -7;
		}
	}

	return(status);
}

/*
 *	Opens the VFS object(s) using the method defined by their
 *	associated MIME Types.
 *
 *	The paths_list specifies a GList of gchar * paths describing
 *	the VFS objects to open.
 *
 *	If command_name is not NULL then the command on the associated
 *	MIME Type who's name matches command_name will be used.
 *
 *	Typical values for command_name are; "default", "view",
 *	and "edit". However it is difficult to know what the user has
 *	set so it is recommended that you pass command_name as NULL
 *	when unsure.
 *
 *	Returns:
 *
 *	0	Success.
 *	-1	General error.
 *	-2	Invalid value or no such object.
 *	-3	System error or memory allocation error.
 *	-4	User aborted.
 *	-7      No associated MIME Type was found.
 */
gint edv_open(
	EDVContext *ctx,
	GList *paths_list,
	const gchar *command_name
)
{
	gint		status = 0,
			status2;
	GList *glist;

	if((ctx == NULL) || (paths_list == NULL))
	{
		errno = EINVAL;
		return(-2);
	}

	for(glist = paths_list; glist != NULL; glist = g_list_next(glist))
	{
		status2 = edv_open_iterate(
			ctx,
			(const gchar *)glist->data,
			command_name
		);
		if(status == 0)
			status2 = status;
	}

	return(status);
}
