/*
	 Endeavour Mark II API - Object Information Matching

	For obtaining the appropriate information (icons and type
	strings) to display the objects with.
 */

#ifndef EDV_OBJ_INFO_MATCH_H
#define EDV_OBJ_INFO_MATCH_H

#include <glib.h>
#include "edv_types.h"
#include "edv_vfs_obj.h"
#include "edv_context.h"


/*
 *	Match Relevency Flags:
 */
typedef enum {
	EDV_MATCH_OBJECT_INFO_NONE	= 0,
	EDV_MATCH_OBJECT_INFO_GENERAL	= (1 << 0),
						/* EDV_MIME_TYPE_CLASS_SYSTEM */
	EDV_MATCH_OBJECT_INFO_SPECIFIC	= (1 << 1),
						/* EDV_MIME_TYPE_CLASS_FORMAT */
	EDV_MATCH_OBJECT_INFO_EXACT	= (1 << 2),
						/* EDV_MIME_TYPE_CLASS_PROGRAM,
						 * EDV_MIME_TYPE_CLASS_UNIQUE,
						 * or device */
	EDV_MATCH_OBJECT_INFO_USER_SET	= (1 << 3)
} EDVMatchObjectRelevency;


/*
 *	Gets the path to the icon file that best matches the object
 *	information.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The type specifies the object's type, one of EDV_OBJECT_TYPE_*.
 *
 *	The path specifies a string describing the object's full path
 *	or name. If path is not a full path then specific object
 *	matching will not be made.
 *
 *	If link_valid is TRUE then it hints that, if type is
 *	EDV_OBJECT_TYPE_LINK, then it is a valid link.
 *
 *	The permissions specifies the object's EDVPermissionFlags
 *	permissions.
 *
 *	The icon_size specifies the requested icon size to be returned.
 *
 *	If icon_closed_path_rtn, icon_opened_path_rtn,
 *	icon_extended_path_rtn, and/or icon_hidden_path_rtn are not NULL
 *	then they specify the return values for their respective paths.
 *	The returned paths must be deleted by the calling function.
 *
 *	Returns any of EDV_MATCH_OBJECT_INFO_*.
 */
extern EDVMatchObjectRelevency edv_match_object_icon(
	EDVContext *ctx,
	const EDVObjectType type,
	const gchar *path,
	const gboolean link_valid,
	const EDVPermissionFlags permissions,
	const EDVIconSize icon_size,
	gchar **icon_closed_path_rtn,
	gchar **icon_opened_path_rtn,
	gchar **icon_extended_path_rtn,
	gchar **icon_hidden_path_rtn
);

/*
 *	Gets the path to the icon file that best matches the object
 *	struct stat information.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The obj specifies the VFS object.
 *
 *	The icon_size specifies the requested icon size to be returned.
 *
 *	If icon_closed_path_rtn, icon_opened_path_rtn,
 *	icon_extended_path_rtn, and/or icon_hidden_path_rtn are not NULL
 *	then they specify the return values for their respective paths.
 *	The returned paths must be deleted by the calling function.
 *
 *	Returns any of EDV_MATCH_OBJECT_INFO_*.
 */
extern EDVMatchObjectRelevency edv_match_vfs_object_icon(
	EDVContext *ctx,
	EDVVFSObject *obj,
	const EDVIconSize icon_size,
	gchar **icon_closed_path_rtn,
	gchar **icon_opened_path_rtn,
	gchar **icon_extended_path_rtn,
	gchar **icon_hidden_path_rtn
);


/*
 *	Gets the MIME Type type string that best matches the object
 *	information.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The type specifies the object's type.
 *
 *	The path specifies either the full path or just the name of
 *	the object.
 *
 *	The permissions specifies the object's permissions.
 *
 *	If type_string_rtn is not NULL then it specifies the return
 *	value for the type string. The returned type string must be
 *	deleted by the calling function.
 *
 *	Returns any of EDV_MATCH_OBJECT_INFO_*.
 */
extern EDVMatchObjectRelevency edv_match_object_type_string(
	EDVContext *ctx,
	const EDVObjectType type,
	const gchar *path,
	const EDVPermissionFlags permissions,
	gchar **type_string_rtn
);

/*
 *	Gets the MIME Type type string that best matches the object
 *	struct stat information.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The obj specifies the VFS object.
 *
 *	If type_string_rtn is not NULL then it specifies the return
 *	value for the type string. The returned type string must be
 *	deleted by the calling function.
 *
 *	Returns any of EDV_MATCH_OBJECT_INFO_*.
 */
extern EDVMatchObjectRelevency edv_match_vfs_object_type_string(
	EDVContext *ctx,
	EDVVFSObject *obj,
	gchar **type_string_rtn
);


#endif	/* EDV_OBJ_MATCH_H */
