#include <glib.h>

#include "edv_types.h"
#include "edv_utils.h"
#include "edv_path.h"
#include "edv_property.h"
#include "edv_property_directory.h"
#include "edv_vfs_obj.h"
#include "edv_device.h"
#include "edv_mime_type.h"
#include "edv_mime_type_get.h"
#include "edv_obj_info_match.h"
#include "edv_context_private.h"


EDVMatchObjectRelevency edv_match_object_icon(
	EDVContext *ctx,
	const EDVObjectType type,
	const gchar *path,
	const gboolean link_valid,
	const EDVPermissionFlags permissions,
	const EDVIconSize icon_size,
	gchar **icon_closed_path_rtn,
	gchar **icon_opened_path_rtn,
	gchar **icon_extended_path_rtn,
	gchar **icon_hidden_path_rtn
);
EDVMatchObjectRelevency edv_match_vfs_object_icon(
        EDVContext *ctx,
	EDVVFSObject *obj,
        const EDVIconSize icon_size,
        gchar **icon_closed_path_rtn,
        gchar **icon_opened_path_rtn,
        gchar **icon_extended_path_rtn,
        gchar **icon_hidden_path_rtn
);

EDVMatchObjectRelevency edv_match_object_type_string(
        EDVContext *ctx,
        const EDVObjectType type,
        const gchar *path,
        const EDVPermissionFlags permissions,
        gchar **type_string_rtn
);
EDVMatchObjectRelevency edv_match_vfs_object_type_string(
        EDVContext *ctx,
	EDVVFSObject *obj,
        gchar **type_string_rtn
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the path to the icon file that best matches the object
 *	information.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The type specifies the object's type, one of EDV_OBJECT_TYPE_*.
 *
 *	The path specifies a string describing the object's full path
 *	or name. If path is not a full path then specific object
 *	matching will not be made.
 *
 *	If link_valid is TRUE then it hints that, if type is
 *	EDV_OBJECT_TYPE_LINK, then it is a valid link.
 *
 *	The permissions specifies the object's EDVPermissionFlags
 *	permissions.
 *
 *	The icon_size specifies the requested icon size to be returned.
 *
 *	If icon_closed_path_rtn, icon_opened_path_rtn,
 *	icon_extended_path_rtn, and/or icon_hidden_path_rtn are not NULL
 *	then they specify the return values for their respective paths.
 *	The returned paths must be deleted by the calling function.
 *
 *	Returns any of EDV_MATCH_OBJECT_INFO_*.
 */
EDVMatchObjectRelevency edv_match_object_icon(
	EDVContext *ctx,
	const EDVObjectType type,
	const gchar *path,
	const gboolean link_valid,
	const EDVPermissionFlags permissions,
	const EDVIconSize icon_size,
	gchar **icon_closed_path_rtn,
	gchar **icon_opened_path_rtn,
	gchar **icon_extended_path_rtn,
	gchar **icon_hidden_path_rtn
)
{
	EDVMatchObjectRelevency match_status = EDV_MATCH_OBJECT_INFO_NONE;
	const gboolean is_file = (type == EDV_OBJECT_TYPE_FILE) ? TRUE : FALSE;
	const gchar *paths_list[EDV_MIME_TYPE_TOTAL_ICON_STATES];

	/* Reset the local icon list */
	(void)memset(
	    paths_list,
	    0x00,
	    sizeof(paths_list)
	);

	/* Reset the return values */
	if(icon_closed_path_rtn != NULL)
	    *icon_closed_path_rtn = NULL;
	if(icon_opened_path_rtn != NULL)
	    *icon_opened_path_rtn = NULL;
	if(icon_extended_path_rtn != NULL)
	    *icon_extended_path_rtn = NULL;
	if(icon_hidden_path_rtn != NULL)
	    *icon_hidden_path_rtn = NULL;

	if(ctx == NULL)
	    return(match_status);

/* Sets the icons from the MIME Type, realizing the MIME type as needed
 *
 * If one or more pixmaps were obtained then match_status will be set
 */
#define GET_MIMETYPE_PATHS(_m_)	{			\
 EDVMIMETypeIconState state;			\
 const gint nstates = EDV_MIME_TYPE_TOTAL_ICON_STATES;	\
							\
 /* Get icons by the requested size */			\
 switch(icon_size) {					\
  case EDV_ICON_SIZE_16:				\
   /* TODO, add support for size 16 (MINI) */		\
   break;						\
  case EDV_ICON_SIZE_20:				\
   for(state = 0; state < nstates; state++)		\
    paths_list[state] = (_m_)->small_icon_file[state];	\
   break;						\
  case EDV_ICON_SIZE_32:				\
   for(state = 0; state < nstates; state++)		\
    paths_list[state] = (_m_)->medium_icon_file[state];	\
   break;						\
  case EDV_ICON_SIZE_48:				\
   for(state = 0; state < nstates; state++)		\
    paths_list[state] = (_m_)->large_icon_file[state];	\
   break;						\
 }							\
							\
 /* If we got a non-NULL path then set the		\
  * EDV_MATCH_OBJECT_INFO_GENERAL on the match_status	\
  * so we know that we have matched at least something	\
  */							\
 for(state = 0; state < nstates; state++) {		\
  if(paths_list[state] != NULL) {			\
   match_status |= EDV_MATCH_OBJECT_INFO_GENERAL;	\
   break;						\
  }							\
 }							\
}

	/* If the object's type is a link then its icon should
	 * specifically display it as a link
	 */
	if(type == EDV_OBJECT_TYPE_LINK)
	{
	    const gchar *mime_type_str = EDV_MIME_TYPE_TYPE_INODE_LINK;
	    GList *glist;
	    EDVMIMEType *m;

	    /* Object is a link, now iterate through MIME Types list and
	     * find the MIME Type of class EDV_MIME_TYPE_CLASS_SYSTEM
	     * and use its icons
	     */
	    for(glist = ctx->mime_types_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
		    continue;

		/* Only handle if MIME Type class is a systems object */
		if((m->mt_class == EDV_MIME_TYPE_CLASS_SYSTEM) &&
		   !STRISEMPTY(m->type)
		)
		{
		    if(!strcmp((const char *)m->type, (const char *)mime_type_str))
		    {
			GET_MIMETYPE_PATHS(m);
			match_status |= EDV_MATCH_OBJECT_INFO_GENERAL;
			break;
		    }
		}
	    }
	}

	/* If the object's type is a directory then check if its
	 * directory properties specify a specific icon
	 */
	if(type == EDV_OBJECT_TYPE_DIRECTORY)
	{
	    /* The specified path must be a full path in order to get
	     * the directory properties
	     */
	    if((path != NULL) ? g_path_is_absolute(path) : FALSE)
	    {
		/* Closed */
		gchar *icon_path = edv_property_directory_get_icon_path(
		    path,
		    icon_size
		);
		if(icon_path != NULL)
		{
		    if(icon_closed_path_rtn != NULL)
			*icon_closed_path_rtn = g_strdup(icon_path);
		    match_status |= EDV_MATCH_OBJECT_INFO_SPECIFIC;
		    g_free(icon_path);
		}

		/* Opened */
		icon_path = edv_property_directory_get_icon_opened_path(
		    path,
		    icon_size
		);
		if(icon_path != NULL)
		{
		    if(icon_opened_path_rtn != NULL)
			*icon_opened_path_rtn = g_strdup(icon_path);
		    match_status |= EDV_MATCH_OBJECT_INFO_SPECIFIC;
		    g_free(icon_path);
		}
	    }

	    if(match_status != EDV_MATCH_OBJECT_INFO_NONE)
		return(match_status);
	}

	/* Check the devices list to see if this object matches
	 * one of the devices' mount path
	 */
	if((ctx->devices_list != NULL) &&
	   ((path != NULL) ? g_path_is_absolute(path) : FALSE) &&
	   (match_status == EDV_MATCH_OBJECT_INFO_NONE)
	)
	{
	    GList *glist;
	    EDVDevice *d;

	    /* Iterate through the list of devices */
	    for(glist = ctx->devices_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		d = EDV_DEVICE(glist->data);
		if(d == NULL)
		    continue;

		if(STRISEMPTY(d->mount_path))
		    continue;

		/* Specified path matches device's mount path? */
		if(!strcmp((const char *)d->mount_path, (const char *)path))
		{
		    EDVDeviceIconState device_icon_state;
		    EDVMIMETypeIconState state;

		    /* Get the appropriate device icon state
		     *
		     * Is this device mounted?
		     */
		    if(EDV_DEVICE_IS_MOUNTED(d))
		    {
			device_icon_state = EDV_DEVICE_ICON_STATE_STANDARD;
		    }
		    else
		    {
			device_icon_state = EDV_DEVICE_ICON_STATE_UNMOUNTED;
		    }

		    /* Get the device icon */
		    switch(icon_size)
		    {
		      case EDV_ICON_SIZE_16:
			break;
		      case EDV_ICON_SIZE_20:
			state = EDV_MIME_TYPE_ICON_STATE_STANDARD;
			paths_list[state] = d->small_icon_file[device_icon_state];
			if(paths_list[state] == NULL)
			    paths_list[state] = d->small_icon_file[
				EDV_DEVICE_ICON_STATE_STANDARD
			    ];
			state = EDV_MIME_TYPE_ICON_STATE_OPENED;
			paths_list[state] = d->small_icon_file[
			    EDV_DEVICE_ICON_STATE_SELECTED
			];
			if(paths_list[state] == NULL)
			    paths_list[state] = d->small_icon_file[
				EDV_DEVICE_ICON_STATE_STANDARD
			    ];
			break;
		      case EDV_ICON_SIZE_32:
			state = EDV_MIME_TYPE_ICON_STATE_STANDARD;
			paths_list[state] = d->medium_icon_file[device_icon_state];
			if(paths_list[state] == NULL)
			    paths_list[state] = d->medium_icon_file[
				EDV_DEVICE_ICON_STATE_STANDARD
			    ];
			state = EDV_MIME_TYPE_ICON_STATE_OPENED;
			paths_list[state] = d->medium_icon_file[
			    EDV_DEVICE_ICON_STATE_SELECTED
			];
			if(paths_list[state] == NULL)
			    paths_list[state] = d->medium_icon_file[
				EDV_DEVICE_ICON_STATE_STANDARD
			    ];
			break;
		      case EDV_ICON_SIZE_48:
			state = EDV_MIME_TYPE_ICON_STATE_STANDARD;
			paths_list[state] = d->large_icon_file[device_icon_state];
			if(paths_list[state] == NULL)
			    paths_list[state] = d->large_icon_file[
				EDV_DEVICE_ICON_STATE_STANDARD
			    ];
			state = EDV_MIME_TYPE_ICON_STATE_OPENED;
			paths_list[state] = d->large_icon_file[
			    EDV_DEVICE_ICON_STATE_SELECTED
			];
			if(paths_list[state] == NULL)
			    paths_list[state] = d->large_icon_file[
				EDV_DEVICE_ICON_STATE_STANDARD
			    ];
			break;
		    }
		    /* Got match? */
		    if(paths_list[EDV_MIME_TYPE_ICON_STATE_STANDARD] != NULL)
		    {
			match_status |= EDV_MATCH_OBJECT_INFO_EXACT;
			break;
		    }
		    break;
		}
	    }	/* Iterate through devices */

	    if(match_status != EDV_MATCH_OBJECT_INFO_NONE)
	    {
		/* Set the return values */
		if(icon_closed_path_rtn != NULL)
		    *icon_closed_path_rtn = STRDUP(paths_list[
			EDV_MIME_TYPE_ICON_STATE_STANDARD
		    ]);
		if(icon_opened_path_rtn != NULL)
		    *icon_opened_path_rtn = STRDUP(paths_list[
			EDV_MIME_TYPE_ICON_STATE_OPENED
		    ]);
		if(icon_extended_path_rtn != NULL)
		    *icon_extended_path_rtn = STRDUP(paths_list[
			EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
		    ]);
		if(icon_hidden_path_rtn != NULL)
		    *icon_hidden_path_rtn = STRDUP(paths_list[
			EDV_MIME_TYPE_ICON_STATE_HIDDEN
		    ]);

		return(match_status);
	    }
	}

	/* Check the MIME Types list to see if this object matches
	 * one of the MIME Types
	 */
	if((ctx->mime_types_list != NULL) &&
	   (match_status == EDV_MATCH_OBJECT_INFO_NONE)
	)
	{
	    const gchar	*name = (path != NULL) ? g_basename(path) : NULL,
			*value;
	    GList *glist;
	    EDVMIMEType *m;

	    /* Iterate through the MIME Types list */
	    for(glist = ctx->mime_types_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
		    continue;

		value = m->value;
		if(STRISEMPTY(value))
		    continue;

		/* Handle by MIME Type class */
		switch(m->mt_class)
		{
		  case EDV_MIME_TYPE_CLASS_SYSTEM:
		    break;
		  case EDV_MIME_TYPE_CLASS_FORMAT:
		    if((name != NULL) &&
		       (match_status == EDV_MATCH_OBJECT_INFO_NONE) &&
		       is_file
		    )
		    {
			if(edv_name_has_extension(name, value))
			{
			    GET_MIMETYPE_PATHS(m);
			    if(match_status != EDV_MATCH_OBJECT_INFO_NONE)
				match_status |= EDV_MATCH_OBJECT_INFO_SPECIFIC;
			}
		    }
		    break;
		  case EDV_MIME_TYPE_CLASS_PROGRAM:
		    if((path != NULL) ? g_path_is_absolute(path) : FALSE)
		    {
			if(!strcmp(value, path))
			{
			    GET_MIMETYPE_PATHS(m);
			    if(match_status != EDV_MATCH_OBJECT_INFO_NONE)
				match_status |= EDV_MATCH_OBJECT_INFO_EXACT;
			}
		    }
		    break;
		  case EDV_MIME_TYPE_CLASS_UNIQUE:
		    if((path != NULL) ? g_path_is_absolute(path) : FALSE)
		    {
			if(!strcmp(value, path))
			{
			    GET_MIMETYPE_PATHS(m);
			    if(match_status != EDV_MATCH_OBJECT_INFO_NONE)
				match_status |= EDV_MATCH_OBJECT_INFO_EXACT;
			}
		    }
		    break;
		}

		if(match_status != EDV_MATCH_OBJECT_INFO_NONE)
		    break;

	    }	/* Iterate through the MIME Types list */
	}



	/* If still did not get match, then use the basic system MIME
	 * types for the specified object
	 */
	if(match_status == EDV_MATCH_OBJECT_INFO_NONE)
	{
	    const gchar *mime_type_str = "";
	    GList *glist;
	    EDVMIMEType *m;

	    /* Get MIME Type type string used for matching of system
	     * object type
	     */
	    switch(type)
	    {
	      case EDV_OBJECT_TYPE_UNKNOWN:
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_UNKNOWN;
		break;

	      case EDV_OBJECT_TYPE_FILE:
		/* Check the file's permissions allow execution, in
		 * which case we use the file/executable MIME Type
		 * instead of file/regular
		 */
		if(permissions & (EDV_PERMISSION_UX |
		    EDV_PERMISSION_GX | EDV_PERMISSION_OX)
		)
		    mime_type_str = EDV_MIME_TYPE_TYPE_INODE_EXECUTABLE;
		else
		    mime_type_str = EDV_MIME_TYPE_TYPE_INODE_FILE;
		break;

	      case EDV_OBJECT_TYPE_DIRECTORY:
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_DIRECTORY;
		break;

	      case EDV_OBJECT_TYPE_LINK:
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_LINK;
		break;

	      case EDV_OBJECT_TYPE_DEVICE_BLOCK:
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_DEV_BLOCK;
		break;

	      case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_DEV_CHARACTER;
		break;

	      case EDV_OBJECT_TYPE_FIFO:
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_FIFO;
		break;

	      case EDV_OBJECT_TYPE_SOCKET:
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_SOCKET;
		break;

	      case EDV_OBJECT_TYPE_ERROR:
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_ERROR;
		break;
	    }

	    /* Iterate through the MIME Types list */
	    for(glist = ctx->mime_types_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
		    continue;

		/* Only handle if the MIME Type class is system */
		if((m->mt_class == EDV_MIME_TYPE_CLASS_SYSTEM) &&
		   !STRISEMPTY(m->type)
		)
		{
		    if(!strcmp((const char *)m->type, (const char *)mime_type_str))
		    {
			GET_MIMETYPE_PATHS(m);
			match_status |= EDV_MATCH_OBJECT_INFO_GENERAL;
			break;
		    }
		}
	    }

	    /* If this object does not match any of the system
	     * MIME Types then return with the unknown icons
	     */
	    if(match_status == EDV_MATCH_OBJECT_INFO_NONE)
	    {
		m = edv_mime_types_list_match_type(
		    ctx,
		    EDV_MIME_TYPE_TYPE_INODE_UNKNOWN
		);
		if(m != NULL)
		{
		    GET_MIMETYPE_PATHS(m);
		    match_status |= EDV_MATCH_OBJECT_INFO_GENERAL;
		}
	    }
	}

#undef GET_MIMETYPE_PATHS


	/* Set the return values */
	if(icon_closed_path_rtn != NULL)
	    *icon_closed_path_rtn = STRDUP(paths_list[
		EDV_MIME_TYPE_ICON_STATE_STANDARD
	    ]);
	if(icon_opened_path_rtn != NULL)
	    *icon_opened_path_rtn = STRDUP(paths_list[
		EDV_MIME_TYPE_ICON_STATE_OPENED
	    ]);
	if(icon_extended_path_rtn != NULL)
	    *icon_extended_path_rtn = STRDUP(paths_list[
		EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
	    ]);
	if(icon_hidden_path_rtn != NULL)
	    *icon_hidden_path_rtn = STRDUP(paths_list[
		EDV_MIME_TYPE_ICON_STATE_HIDDEN
	    ]);

	return(match_status);
}

/*
 *	Gets the path to the icon file that best matches the object
 *	struct stat information.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The obj specifies the VFS object.
 *
 *	The icon_size specifies the requested icon size to be returned.
 *
 *	If icon_closed_path_rtn, icon_opened_path_rtn,
 *	icon_extended_path_rtn, and/or icon_hidden_path_rtn are not NULL
 *	then they specify the return values for their respective paths.
 *	The returned paths must be deleted by the calling function.
 *
 *	Returns any of EDV_MATCH_OBJECT_INFO_*.
 */
EDVMatchObjectRelevency edv_match_vfs_object_icon(
        EDVContext *ctx,
	EDVVFSObject *obj,
        const EDVIconSize icon_size,
        gchar **icon_closed_path_rtn,
        gchar **icon_opened_path_rtn,
        gchar **icon_extended_path_rtn,
        gchar **icon_hidden_path_rtn
)
{
	if(obj == NULL)
	    return(EDV_MATCH_OBJECT_INFO_NONE);

	return(edv_match_object_icon(
	    ctx,
	    obj->type,
	    obj->path,
	    EDV_VFS_OBJECT_LINK_TARGET_EXISTS(obj),
	    obj->permissions,
	    icon_size,
	    icon_closed_path_rtn,
	    icon_opened_path_rtn,
	    icon_extended_path_rtn,
	    icon_hidden_path_rtn
	));
}


/*
 *	Gets the MIME Type type string that best matches the object
 *	information.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The type specifies the object's type.
 *
 *	The path specifies either the full path or just the name of
 *	the object.
 *
 *	The permissions specifies the object's permissions.
 *
 *	If type_string_rtn is not NULL then it specifies the return
 *	value for the type string. The returned type string must be
 *	deleted by the calling function.
 *
 *	Returns any of EDV_MATCH_OBJECT_INFO_*.
 */
EDVMatchObjectRelevency edv_match_object_type_string(
        EDVContext *ctx,
        const EDVObjectType type,
        const gchar *path,
        const EDVPermissionFlags permissions,
        gchar **type_string_rtn
)
{
	const gboolean is_file = (type == EDV_OBJECT_TYPE_FILE) ? TRUE : FALSE;
	EDVMatchObjectRelevency match_status = EDV_MATCH_OBJECT_INFO_NONE;
	const gchar *type_string = NULL;
	const gchar *name = (path != NULL) ? g_basename(path) : NULL;
	EDVMIMEType *m;

	if(type_string_rtn != NULL)
	    *type_string_rtn = NULL;

	if(ctx == NULL)
	    return(match_status);

	/* First check if the object is a link */
	if(type == EDV_OBJECT_TYPE_LINK)
	{
	    type_string = EDV_MIME_TYPE_TYPE_INODE_LINK;
	    match_status |= EDV_MATCH_OBJECT_INFO_GENERAL;
	}

	/* Check the MIME Types list for a MIME Type that matches
	 * this object object
	 */
	if((match_status == EDV_MATCH_OBJECT_INFO_NONE) &&
	   (ctx->mime_types_list != NULL)
	)
	{
	    const gchar *value;
	    GList *glist;

	    /* Iterate through the MIME Types list */
	    for(glist = ctx->mime_types_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
		    continue;

		value = m->value;
		if(STRISEMPTY(value))
		    continue;

		/* Handle by MIME Type class */
		switch(m->mt_class)
		{
		  case EDV_MIME_TYPE_CLASS_SYSTEM:
		    break;
		  case EDV_MIME_TYPE_CLASS_FORMAT:
		    if((name != NULL) &&
		       (match_status == EDV_MATCH_OBJECT_INFO_NONE) &&
		       is_file
		    )
		    {
			if(edv_name_has_extension(name, value))
			{
			    type_string = m->type;
			    if(type_string != NULL)
				match_status |= EDV_MATCH_OBJECT_INFO_GENERAL |
				    EDV_MATCH_OBJECT_INFO_SPECIFIC;
			}
		    }
		    break;
		  case EDV_MIME_TYPE_CLASS_PROGRAM:
		    if((path != NULL) ? g_path_is_absolute(path) : FALSE)
		    {
			if(!strcmp((const char *)value, (const char *)path))
			{
			    type_string = m->type;
			    if(type_string != NULL)
				match_status |= EDV_MATCH_OBJECT_INFO_GENERAL |
				    EDV_MATCH_OBJECT_INFO_SPECIFIC |
				    EDV_MATCH_OBJECT_INFO_EXACT;
			}
		    }
		    break;
		  case EDV_MIME_TYPE_CLASS_UNIQUE:
		    if((path != NULL) ? g_path_is_absolute(path) : FALSE)
		    {
			if(!strcmp((const char *)value, (const char *)path))
			{
			    type_string = m->type;
			    if(type_string != NULL)
				match_status |= EDV_MATCH_OBJECT_INFO_GENERAL |
				    EDV_MATCH_OBJECT_INFO_SPECIFIC |
				    EDV_MATCH_OBJECT_INFO_EXACT;
			}
		    }
		    break;
		}

		if(match_status != EDV_MATCH_OBJECT_INFO_NONE)
		    break;

	    }	/* Iterate through the MIME Types list */
	}

	/* If there was no match then set the MIME Type string to
	 * one based on the object's type
	 */
	if(match_status == EDV_MATCH_OBJECT_INFO_NONE)
	{
	    const gchar *s = "";

	    /* Get MIME Type type string used for matching of system
	     * object type
	     */
	    switch(type)
	    {
	      case EDV_OBJECT_TYPE_UNKNOWN:
		s = EDV_MIME_TYPE_TYPE_INODE_UNKNOWN;
		break;
	      case EDV_OBJECT_TYPE_FILE:
		/* Check the file's permissions allow execution, in which
		 * case we use the file/executable MIME Type instead of
		 * file/regular
		 */
		if(permissions & (EDV_PERMISSION_UX |
		    EDV_PERMISSION_GX | EDV_PERMISSION_OX)
		)
		    s = EDV_MIME_TYPE_TYPE_INODE_EXECUTABLE;
		else
		    s = EDV_MIME_TYPE_TYPE_INODE_FILE;
		break;
	      case EDV_OBJECT_TYPE_DIRECTORY:
		s = EDV_MIME_TYPE_TYPE_INODE_DIRECTORY;
		break;
	      case EDV_OBJECT_TYPE_LINK:
		s = EDV_MIME_TYPE_TYPE_INODE_LINK;
		break;
	      case EDV_OBJECT_TYPE_DEVICE_BLOCK:
		s = EDV_MIME_TYPE_TYPE_INODE_DEV_BLOCK;
		break;
	      case EDV_OBJECT_TYPE_DEVICE_CHARACTER:
		s = EDV_MIME_TYPE_TYPE_INODE_DEV_CHARACTER;
		break;
	      case EDV_OBJECT_TYPE_FIFO:
		s = EDV_MIME_TYPE_TYPE_INODE_FIFO;
		break;
	      case EDV_OBJECT_TYPE_SOCKET:
		s = EDV_MIME_TYPE_TYPE_INODE_SOCKET;
		break;
	      case EDV_OBJECT_TYPE_ERROR:
		s = EDV_MIME_TYPE_TYPE_INODE_ERROR;
		break;
	    }

	    type_string = s;

	    match_status |= EDV_MATCH_OBJECT_INFO_GENERAL;
	}

	/* Set the return value */
	if(type_string_rtn != NULL)
	    *type_string_rtn = STRDUP(type_string);

	return(match_status);
}

/*
 *	Gets the MIME Type type string that best matches the object
 *	struct stat information.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The obj specifies the VFS object.
 *
 *	If type_string_rtn is not NULL then it specifies the return
 *	value for the type string. The returned type string must be
 *	deleted by the calling function.
 *
 *	Returns any of EDV_MATCH_OBJECT_INFO_*.
 */
EDVMatchObjectRelevency edv_match_vfs_object_type_string(
        EDVContext *ctx,
	EDVVFSObject *obj,
        gchar **type_string_rtn
)
{
	if(obj == NULL)
	    return(EDV_MATCH_OBJECT_INFO_NONE);

	return(edv_match_object_type_string(
	    ctx,
	    obj->type,
	    obj->path,
	    obj->permissions,
	    type_string_rtn
	));
}
