#include <glib.h>
#include "edv_context_private.h"
#include "edv_notify.h"
#include "config.h"


void edv_notify_queue_vfs_object_added(
	EDVContext *ctx,
	const gchar *path
);
void edv_notify_queue_vfs_object_modified(
	EDVContext *ctx,
	const gchar *path,
	const gchar *new_path
);
void edv_notify_queue_vfs_object_removed(
	EDVContext *ctx,
	const gchar *path
);

void edv_notify_queue_vfs_object_mounted(
	EDVContext *ctx,
	const gchar *path
);
void edv_notify_queue_vfs_object_unmounted(
	EDVContext *ctx,
	const gchar *path
);

void edv_notify_queue_recycled_object_added(
	EDVContext *ctx,
	const gulong index
);
void edv_notify_queue_recycled_object_modified(
	EDVContext *ctx,
	const gulong index
);
void edv_notify_queue_recycled_object_removed(
	EDVContext *ctx,
	const gulong index
);

void edv_notify_queue_reconfigured(EDVContext *ctx);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Queues a VFS object added signal.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The path specifies the full path to the object that was
 *	added.
 */
void edv_notify_queue_vfs_object_added(
	EDVContext *ctx,
	const gchar *path
)
{
	gchar *cmd;
	if(ctx == NULL)
		return;
	cmd = g_strdup_printf(
		"vfs_object_added_notify \"%s\"",
		path
	);
	edv_context_queue_command(ctx, cmd);
	g_free(cmd);
}

/*
 *	Queues a VFS object modified signal.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The path specifies the full path to the object that was
 *	modified (it may not exist in cases of renaming or moving).
 *
 *	The new_path specifies the full path to the new path of the
 *	modified object. If new_path is NULL then only path will be
 *	sent (hinting that the name of the object was not modified).
 */
void edv_notify_queue_vfs_object_modified(
	EDVContext *ctx,
	const gchar *path,
	const gchar *new_path
)
{
	gchar *cmd;
	if(ctx == NULL)
		return;
	if(new_path != NULL)
		cmd = g_strdup_printf(
			"vfs_object_modified_notify \"%s\" \"%s\"",
			path, new_path
		);
	else
		cmd = g_strdup_printf(
			"vfs_object_modified_notify \"%s\"",
			path
		);
	edv_context_queue_command(ctx, cmd);
	g_free(cmd);
}

/*
 *	Queues a VFS object removed signal.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The path specifies the full path to the object that was
 *	removed.
 */
void edv_notify_queue_vfs_object_removed(
	EDVContext *ctx,
	const gchar *path
)
{
	gchar *cmd;
	if(ctx == NULL)
		return;
	cmd = g_strdup_printf(
		"vfs_object_removed_notify \"%s\"",
		path
	);
	edv_context_queue_command(ctx, cmd);
	g_free(cmd);
}


/*
 *	Queues a VFS object mounted signal.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The path specifies the full path to the mount path.
 */
void edv_notify_queue_vfs_object_mounted(
	EDVContext *ctx,
	const gchar *path
)
{
	gchar *cmd;
	if(ctx == NULL)
		return;
	cmd = g_strdup_printf(
		"vfs_object_mounted_notify \"%s\"",
		path
	);
	edv_context_queue_command(ctx, cmd);
	g_free(cmd);
}

/*
 *	Queues a VFS object unmounted signal.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The path specifies the full path to the mount path.
 */
void edv_notify_queue_vfs_object_unmounted(
	EDVContext *ctx,
	const gchar *path
)
{
	gchar *cmd;
	if(ctx == NULL)
		return;
	cmd = g_strdup_printf(
		"vfs_object_unmounted_notify \"%s\"",
		path
	);
	edv_context_queue_command(ctx, cmd);
	g_free(cmd);
}


/*
 *	Queues a recycled object added signal.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The index specifies the index of the recycled object.
 */
void edv_notify_queue_recycled_object_added(
	EDVContext *ctx,
	const gulong index
)
{
	gchar *cmd;
	if(ctx == NULL)
		return;
	cmd = g_strdup_printf(
		"recycled_object_added_notify %ld",
		index
	);
	edv_context_queue_command(ctx, cmd);
	g_free(cmd);
}

/*
 *	Queues a recycled object modified signal.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The index specifies the index of the recycled object.
 */
void edv_notify_queue_recycled_object_modified(
	EDVContext *ctx,
	const gulong index
)
{
	gchar *cmd;
	if(ctx == NULL)
		return;
	cmd = g_strdup_printf(
		"recycled_object_modified_notify %ld",
		index
	);
	edv_context_queue_command(ctx, cmd);
	g_free(cmd);
}

/*
 *	Queues a recycled object removed signal.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The index specifies the index of the recycled object.
 */
void edv_notify_queue_recycled_object_removed(
	EDVContext *ctx,
	const gulong index
)
{
	gchar *cmd;
	if(ctx == NULL)
		return;
	cmd = g_strdup_printf(
		"recycled_object_removed_notify %ld",
		index
	);
	edv_context_queue_command(ctx, cmd);
	g_free(cmd);
}


/*
 *	Queues a reconfigured signal.
 *
 *	The reconfigured signal notifies Endeavour when its
 *	configuration has changed.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
void edv_notify_queue_reconfigured(EDVContext *ctx)
{
	if(ctx == NULL)
		return;

	edv_context_queue_command(ctx, "reconfigured_notify");
}

