#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <glib.h>
#include "../../include/fio.h"
#include "edv_mime_type.h"
#include "edv_mime_types_list.h"
#include "config.h"


GList *edv_mime_types_list_file_open(
	GList *mime_types_list,
	const gchar *path
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Opens the MIME Types List from the MIME Types List file.
 *
 *	The mime_types_list specifies the MIME Types List, a GList of
 *	EDVMIMEType * MIME Types. The MIME Types opened from
 *	the MIME Types List file will be appended to the MIME Types
 *	List.
 *
 *	The path specifies the MIME Types List file.
 *
 *	Returns the MIME Types List, a GList of EDVMIMEType *
 *	MIME Types.
 */
GList *edv_mime_types_list_file_open(
	GList *mime_types_list,
	const gchar *path
)
{
	FILE *fp;
	struct stat stat_buf;
	gchar *parm;
	gulong file_size;
	EDVMIMEType *m = NULL;
	EDVMIMETypeCommand *cmd = NULL;

	if(STRISEMPTY(path))
	    return(mime_types_list);

	/* Open the MIME Types List file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
	    return(mime_types_list);

	/* Get the MIME Types List file's statistics */
	if(fstat(fileno(fp), &stat_buf))
	    file_size = 0l;
	else
	    file_size = (gulong)stat_buf.st_size;

	/* Begin reading MIME Types List file */
	parm = NULL;
	while(TRUE)
	{
	    /* Get the next parameter from the MIME Types List file */
	    parm = (gchar *)FSeekNextParm(
		fp,
		(char *)parm,
		EDV_CFG_COMMENT_CHAR,
		EDV_CFG_DELIMINATOR_CHAR
	    );
	    if(parm == NULL)
		break;

	    /* Begin handling by parameter */

	    /* BeginMIMEType */
	    if(!g_strcasecmp(parm, "BeginMIMEType"))
	    {
		gint value[1];
		EDVMIMETypeClass mt_class;

		/* Get the class */
		FGetValuesI(fp, value, 1);
		mt_class = (EDVMIMETypeClass)value[0];

		/* Append a new MIME Type to the MIME Types List */
		m = edv_mime_type_new_values(
		    mt_class,
		    NULL,
		    NULL,
		    NULL
		);
		if(m != NULL)
		    mime_types_list = g_list_append(
			mime_types_list,
			m
		    );
		else
		    break;
	    }

	    /* Type */
	    else if(!g_strcasecmp(parm, "Type"))
	    {
		gchar *type = FGetString(fp);
		if(m != NULL)
		{
		    g_free(m->type);
		    m->type = STRDUP(type);
		}
		g_free(type);
	    }
	    /* Value */
	    else if(!g_strcasecmp(parm, "Value"))
	    {
		gchar *value = FGetString(fp);
		if(m != NULL)
		{
		    g_free(m->value);
		    m->value = STRDUP(value);
		}
		g_free(value);
	    }
	    /* Description */
	    else if(!g_strcasecmp(parm, "Description"))
	    {
		gchar *description = FGetString(fp);
		if(m != NULL)
		{
		    g_free(m->description);
		    m->description = STRDUP(description);
		}
		g_free(description);
	    }

	    /* IconSmallStandard */
	    else if(!g_strcasecmp(parm, "IconSmallStandard"))
	    {
		gchar *path = FGetString(fp);
		gchar **storage = &m->small_icon_file[
		    EDV_MIME_TYPE_ICON_STATE_STANDARD
		];
		if(m != NULL)
		{
		    g_free(*storage);
		    *storage = STRDUP(path);
		}
		g_free(path);
	    }
	    /* IconSmallOpened */
	    else if(!g_strcasecmp(parm, "IconSmallOpened") ||
		    !g_strcasecmp(parm, "IconSmallSelected")
	    )
	    {
		gchar *path = FGetString(fp);
		gchar **storage = &m->small_icon_file[
		    EDV_MIME_TYPE_ICON_STATE_OPENED
		];
		if(m != NULL)
		{
		    g_free(*storage);
		    *storage = STRDUP(path);
		}
		g_free(path);
	    }
	    /* IconSmallInaccessible */
	    else if(!g_strcasecmp(parm, "IconSmallInaccessible") ||
		    !g_strcasecmp(parm, "IconSmallExtended")
	    )
	    {
		gchar *path = FGetString(fp);
		gchar **storage = &m->small_icon_file[
		    EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
		];
		if(m != NULL)
		{
		    g_free(*storage);
		    *storage = STRDUP(path);
		}
		g_free(path);
	    }
	    /* IconSmallHidden */
	    else if(!g_strcasecmp(parm, "IconSmallHidden"))
	    {
		gchar *path = FGetString(fp);
		gchar **storage = &m->small_icon_file[
		    EDV_MIME_TYPE_ICON_STATE_HIDDEN
		];
		if(m != NULL)
		{
		    g_free(*storage);
		    *storage = STRDUP(path);
		}
		g_free(path);
	    }

	    /* IconMediumStandard */
	    else if(!g_strcasecmp(parm, "IconMediumStandard"))
	    {
		gchar *path = FGetString(fp);
		gchar **storage = &m->medium_icon_file[
		    EDV_MIME_TYPE_ICON_STATE_STANDARD
		];
		if(m != NULL)
		{
		    g_free(*storage);
		    *storage = STRDUP(path);
		}
		g_free(path);
	    }
	    /* IconMediumOpened */
	    else if(!g_strcasecmp(parm, "IconMediumOpened") ||
		    !g_strcasecmp(parm, "IconMediumSelected")
	    )
	    {
		gchar *path = FGetString(fp);
		gchar **storage = &m->medium_icon_file[
		    EDV_MIME_TYPE_ICON_STATE_OPENED
		];
		if(m != NULL)
		{
		    g_free(*storage);
		    *storage = STRDUP(path);
		}
		g_free(path);
	    }
	    /* IconMediumInaccessible */
	    else if(!g_strcasecmp(parm, "IconMediumInaccessible") ||
		    !g_strcasecmp(parm, "IconMediumExtended")
	    )
	    {
		gchar *path = FGetString(fp);
		gchar **storage = &m->medium_icon_file[
		    EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
		];
		if(m != NULL)
		{
		    g_free(*storage);
		    *storage = STRDUP(path);
		}
		g_free(path);
	    }
	    /* IconMediumHidden */
	    else if(!g_strcasecmp(parm, "IconMediumHidden"))
	    {
		gchar *path = FGetString(fp);
		gchar **storage = &m->medium_icon_file[
		    EDV_MIME_TYPE_ICON_STATE_HIDDEN
		];
		if(m != NULL)
		{
		    g_free(*storage);
		    *storage = STRDUP(path);
		}
		g_free(path);
	    }

	    /* IconLargeStandard */
	    else if(!g_strcasecmp(parm, "IconLargeStandard"))
	    {
		gchar *path = FGetString(fp);
		gchar **storage = &m->large_icon_file[
		    EDV_MIME_TYPE_ICON_STATE_STANDARD
		];
		if(m != NULL)
		{
		    g_free(*storage);
		    *storage = STRDUP(path);
		}
		g_free(path);
	    }
	    /* IconLargeOpened */
	    else if(!g_strcasecmp(parm, "IconLargeOpened") ||
		    !g_strcasecmp(parm, "IconLargeSelected")
	    )
	    {
		gchar *path = FGetString(fp);
		gchar **storage = &m->large_icon_file[
		    EDV_MIME_TYPE_ICON_STATE_OPENED
		];
		if(m != NULL)
		{
		    g_free(*storage);
		    *storage = STRDUP(path);
		}
		g_free(path);
	    }
	    /* IconLargeInaccessible */
	    else if(!g_strcasecmp(parm, "IconLargeInaccessible") ||
		    !g_strcasecmp(parm, "IconLargeExtended")
	    )
	    {
		gchar *path = FGetString(fp);
		gchar **storage = &m->large_icon_file[
		    EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE
		];
		if(m != NULL)
		{
		    g_free(*storage);
		    *storage = STRDUP(path);
		}
		g_free(path);
	    }
	    /* IconLargeHidden */
	    else if(!g_strcasecmp(parm, "IconLargeHidden"))
	    {
		gchar *path = FGetString(fp);
		gchar **storage = &m->large_icon_file[
		    EDV_MIME_TYPE_ICON_STATE_HIDDEN
		];
		if(m != NULL)
		{
		    g_free(*storage);
		    *storage = STRDUP(path);
		}
		g_free(path);
	    }

	    /* Handler */
	    else if(!g_strcasecmp(parm, "Handler"))
	    {
		gchar *s = FGetString(fp);
		if(m != NULL)
		{
		    if(!g_strcasecmp(s, "Archiver"))
			m->handler = EDV_MIME_TYPE_HANDLER_EDV_ARCHIVER;
		    else if(!g_strcasecmp(s, "ImageBrowser"))
			m->handler = EDV_MIME_TYPE_HANDLER_EDV_IMAGE_BROWSER;
		    else if(!g_strcasecmp(s, "RecycleBin"))
			m->handler = EDV_MIME_TYPE_HANDLER_EDV_RECYCLE_BIN;
		    else
			m->handler = EDV_MIME_TYPE_HANDLER_COMMAND;
		}
		g_free(s);
	    }
	    /* BeginCommand */
	    else if(!g_strcasecmp(parm, "BeginCommand"))
	    {
		gchar *name = (gchar *)FGetString(fp);
		if(m != NULL)
		{
		    /* Create/set new context */
		    cmd = edv_mime_type_command_new();
		    if(cmd != NULL)
		    {
			m->commands_list = g_list_append(
			    m->commands_list,
			    cmd
			);
			if(!STRISEMPTY(name))
			    cmd->name = g_strdup(name);
		    }
		    else
		    {
			break;
		    }
		}
		g_free(name);
	    }
	    /* CommandCommand */
	    else if(!g_strcasecmp(parm, "CommandCommand"))
	    {
		gchar *command = (gchar *)FGetString(fp);
		if(cmd != NULL)
		{
		    g_free(cmd->command);
		    cmd->command = STRDUP(command);
		}
		g_free(command);
	    }
	    /* CommandShellCommand */
	    else if(!g_strcasecmp(parm, "CommandShellCommand"))
	    {
		gchar *shell_command = (gchar *)FGetString(fp);
		if(cmd != NULL)
		{
		    g_free(cmd->shell_command);
		    cmd->shell_command = STRDUP(shell_command);
		}
		g_free(shell_command);
	    }
	    /* CommandRunInTerminal */
	    else if(!g_strcasecmp(parm, "CommandRunInTerminal"))
	    {
		FSeekNextLine(fp);	/* This parameter has no value */
		if(cmd != NULL)
		{
		    cmd->flags |= EDV_MIME_TYPE_COMMAND_RUN_IN_TERMINAL;
		}
	    }
	    /* EndCommand */
	    else if(!g_strcasecmp(parm, "EndCommand"))
	    {
		FSeekNextLine(fp);	/* This parameter has no value */

		/* Reset the context */
		cmd = NULL;
	    }
	    /* Command (version 2.8.4 and older) */
	    else if(!g_strcasecmp(parm, "Command"))
	    {
		/* This parameter has been depreciated, we are handling
		 * it here for compatability purposes, see
		 * parameters BeginCommand and EndCommand
		 */
		gchar *command = (gchar *)FGetString(fp);
		if((m != NULL) && !STRISEMPTY(command))
		{
		    cmd = edv_mime_type_command_new();
		    if(cmd != NULL)
		    {
			/* Seek s to the name and command deliminator */
			gchar *s_cmd = (gchar *)strchr(
			    (char *)command,
			    EDV_CFG_DELIMINATOR_CHAR
			);
			if(s_cmd != NULL)
			{
			    gchar *s_end = s_cmd;
			    gint len = (gint)(s_end - command);

			    s_cmd++;

			    cmd->name = (gchar *)g_malloc((len + 1) * sizeof(gchar));
			    if(len > 0)
				memcpy(cmd->name, command, len * sizeof(gchar));
			    cmd->name[len] = '\0';
			    cmd->name = g_strstrip(cmd->name);

			    cmd->command = g_strdup(s_cmd);
			    cmd->command = g_strstrip(cmd->command);
			}
			else
			{
			    /* No command deliminator, implying there is
			     * only a command
			     *
			     * Create an arbitrary name for this command
			     */
			    cmd->name = g_strdup_printf(
				"Command #%i",
				g_list_length(m->commands_list) + 1
			    );

			    cmd->command = g_strdup(command);
			    cmd->command = g_strstrip(cmd->command);
			}

			m->commands_list = g_list_append(
			    m->commands_list,
			    cmd  
			);
		    }

		    /* Reset the context */
		    cmd = NULL;
		}
		g_free(command);
	    }

	    /* AccessTime */
	    else if(!g_strcasecmp(parm, "AccessTime"))
	    {
		glong value[1];
		FGetValuesL(fp, value, 1);
		if(m != NULL)
		    m->access_time = (gulong)value[0];
	    }
	    /* ModifyTime */
	    else if(!g_strcasecmp(parm, "ModifyTime"))
	    {
		glong value[1];
		FGetValuesL(fp, value, 1);
		if(m != NULL)
		    m->modify_time = (gulong)value[0];
	    }
	    /* ChangeTime */
	    else if(!g_strcasecmp(parm, "ChangeTime"))
	    {
		glong value[1];
		FGetValuesL(fp, value, 1);
		if(m != NULL)
		    m->change_time = (gulong)value[0];
	    }

	    /* EndMIMEType */
	    else if(!g_strcasecmp(parm, "EndMIMEType"))
	    {
		FSeekNextLine(fp);

		/* Reset the context */
		m = NULL;
	    }
	    /* Unsupported parameter */
	    else
	    {
		/* Ignore */
		FSeekNextLine(fp);
	    }
	}

	/* Delete the parameter */
	g_free(parm);

	/* Close the MIME Types List file */
	fclose(fp);

	return(mime_types_list);
}
