#include <glib.h>
#include "edv_utils.h"
#include "edv_path.h"
#include "edv_property.h"
#include "edv_vfs_obj.h"
#include "edv_vfs_obj_stat.h"
#include "edv_convert_obj.h"
#include "edv_mime_type.h"
#include "edv_context_private.h"
#include "edv_mime_type_get.h"


GList *edv_mime_types_list(EDVContext *ctx);

EDVMIMEType *edv_mime_types_list_match_vfs_object_path(
        EDVContext *ctx,
        const gchar *path
);
EDVMIMEType *edv_mime_types_list_match_properties_list(
	EDVContext *ctx,
        GList *properties_list
);
EDVMIMEType *edv_mime_types_list_match_type(
	EDVContext *ctx,
	const gchar *type
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the MIME Types list.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	Returns a GList of EDVMIMEType * MIME Types. The
 *	returned list must not be modified or deleted.
 */
GList *edv_mime_types_list(EDVContext *ctx)
{
	if(ctx == NULL)
	    return(NULL);

	return(ctx->mime_types_list);
}


/*
 *	Matches the MIME Type for a VFS object by path.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The path specifies the VFS object.
 *
 *	Returns the matched MIME Type or NULL on error, the returned
 *	MIME Type must not be modified or deleted.
 */
EDVMIMEType *edv_mime_types_list_match_vfs_object_path(
        EDVContext *ctx,
        const gchar *path
)
{
        GList *properties_list;
        EDVMIMEType *m;

        /* Get the object's local statistics */
        EDVVFSObject *obj = edv_vfs_object_lstat(path);
        if(obj == NULL)
            return(NULL);

        /* Convert the statistics to a properties list */
        properties_list = NULL;
        properties_list = edv_convert_vfs_object_to_properties_list(
            properties_list,
            obj
        );

        /* Delete the statistics */
        edv_vfs_object_delete(obj);

        /* Find a MIME Type associated with this object */
        m = edv_mime_types_list_match_properties_list(
            ctx,
            properties_list
        );

        properties_list = edv_properties_list_delete(properties_list);

	return(m);
}

/*
 *	Matches the MIME Type for an object described by a properties
 *	list.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The properties list specifies the object's properties, a GList
 *	of EDVProperty * properties.
 *
 *	Returns the matched MIME Type or NULL on error, the returned
 *	MIME Type must not be modified or deleted.
 */
EDVMIMEType *edv_mime_types_list_match_properties_list(
	EDVContext *ctx,
        GList *properties_list
)
{
	const gchar	*name = edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_NAME
	),
			*path = edv_properties_list_get_s(
	    properties_list,
	    EDV_PROP_NAME_PATH
	);
	const EDVObjectType type = (EDVObjectType)edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_TYPE
	);
	const gboolean is_file = (type == EDV_OBJECT_TYPE_FILE) ? TRUE : FALSE;
	GList *glist;
	EDVPermissionFlags permissions = (EDVPermissionFlags)edv_properties_list_get_i(
	    properties_list,
	    EDV_PROP_NAME_PERMISSIONS
	);
	EDVMIMEType *m;

	if(path == NULL)
	    path = name;

	if((ctx == NULL) || STRISEMPTY(path))
	    return(NULL);

	/* Is link? */
	if(type == EDV_OBJECT_TYPE_LINK)
	{
	    /* Get the EDV_MIME_TYPE_TYPE_INODE_LINK MIME Type */
	    for(glist = ctx->mime_types_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
		    continue;

		if(STRISEMPTY(m->type))
		    continue;

		if(m->mt_class == EDV_MIME_TYPE_CLASS_SYSTEM)
		{
		    if(!strcmp(
			(const char *)m->type,
			(const char *)EDV_MIME_TYPE_TYPE_INODE_LINK
		    ))
			return(m);
		}
	    }
	}

	/* Match for MIME Types of class; EDV_MIME_TYPE_CLASS_FORMAT,
	 * EDV_MIME_TYPE_CLASS_PROGRAM, or EDV_MIME_TYPE_CLASS_UNIQUE
	 */
	if(ctx->mime_types_list != NULL)
	{
	    const gchar *value;

	    for(glist = ctx->mime_types_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
		    continue;

		value = m->value;
		if(STRISEMPTY(value))
		    continue;

		/* Handle by the MIME Type's class */
		switch(m->mt_class)
		{
		  case EDV_MIME_TYPE_CLASS_SYSTEM:
		    break;
		  case EDV_MIME_TYPE_CLASS_FORMAT:
		    if(is_file && (name != NULL))
		    {
			if(edv_name_has_extension(name, value))
			    return(m);
		    }
		    break;
		  case EDV_MIME_TYPE_CLASS_PROGRAM:
		  case EDV_MIME_TYPE_CLASS_UNIQUE:
		    if(path != NULL)
		    {
			if(g_path_is_absolute(path))
			{
			    if(!strcmp(
				(const char *)value,
				(const char *)path
			    ))
				return(m);
			}
		    }
		    break;
		}
	    }
	}

	/* Match for MIME Types of class EDV_MIME_TYPE_CLASS_SYSTEM */
	if(ctx->mime_types_list != NULL)
	{
	    const gchar *mime_type_str;

	    /* Get MIME Type's type string based on the object's type */
	    if(is_file)
	    {
		/* Check the file's permissions that allow execution,
		 * in which case we use the
		 * EDV_MIME_TYPE_TYPE_INODE_EXECUTABLE
		 * instead EDV_MIME_TYPE_TYPE_INODE_FILE
		 */
		if((permissions & EDV_PERMISSION_UX) ||
		   (permissions & EDV_PERMISSION_GX) ||
		   (permissions & EDV_PERMISSION_OX)
		)
		    mime_type_str = EDV_MIME_TYPE_TYPE_INODE_EXECUTABLE;
		else
		    mime_type_str = EDV_MIME_TYPE_TYPE_INODE_FILE;
	    }
	    else if(type == EDV_OBJECT_TYPE_DIRECTORY)
	    {
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_DIRECTORY;
	    }
	    else if(type == EDV_OBJECT_TYPE_LINK)
	    {
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_LINK;
	    }
	    else if(type == EDV_OBJECT_TYPE_DEVICE_BLOCK)
	    {
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_DEV_BLOCK;
	    }
	    else if(type == EDV_OBJECT_TYPE_DEVICE_CHARACTER)
	    {
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_DEV_CHARACTER;
	    }
	    else if(type == EDV_OBJECT_TYPE_FIFO)
	    {
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_FIFO;
	    }
	    else if(type == EDV_OBJECT_TYPE_SOCKET)
	    {
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_SOCKET;
	    }
	    else
	    {
		/* All else assume file/regular */
		mime_type_str = EDV_MIME_TYPE_TYPE_INODE_UNKNOWN;
	    }

	    /* Check all MIME Types of class EDV_MIME_TYPE_CLASS_SYSTEM */
	    for(glist = ctx->mime_types_list;
		glist != NULL;
		glist = g_list_next(glist)
	    )
	    {
		m = EDV_MIME_TYPE(glist->data);
		if(m == NULL)
		    continue;

		if(STRISEMPTY(m->type))
		    continue;

		if(m->mt_class == EDV_MIME_TYPE_CLASS_SYSTEM)
		{
		    if(!strcmp(
			(const char *)m->type,
			(const char *)mime_type_str
		    ))
			return(m);
		}
	    }
	}

	return(NULL);
}

/*
 *	Matches the MIME Type by type string.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The type specifies the type string.
 *
 *	Returns the matched MIME Type or NULL on error, the returned
 *	MIME Type must not be modified or deleted.
 */
EDVMIMEType *edv_mime_types_list_match_type(
	EDVContext *ctx,
	const gchar *type
)
{
	GList *glist;
	EDVMIMEType *m;

	if((ctx == NULL) || STRISEMPTY(type))
	    return(NULL);

	for(glist = ctx->mime_types_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    m = EDV_MIME_TYPE(glist->data);
	    if(m == NULL)
		continue;

	    if(STRISEMPTY(m->type))
		continue;

	    if(!strcmp(
		(const char *)m->type,
		(const char *)type
	    ))
		return(m);
	}

	return(NULL);
}
