/*
                     Endeavour Mark II API - MIME Type
 */

#ifndef EDV_MIME_TYPE_H
#define EDV_MIME_TYPE_H

#include <glib.h>
#include "edv_types.h"


typedef struct _EDVMIMETypeCommand	EDVMIMETypeCommand;
#define EDV_MIME_TYPE_COMMAND(p)	((EDVMIMETypeCommand *)(p))

typedef struct _EDVMIMEType		EDVMIMEType;
#define EDV_MIME_TYPE(p)		((EDVMIMEType *)(p))


/*
 *	MIME Type Icon States:
 */
typedef enum{
	EDV_MIME_TYPE_ICON_STATE_STANDARD,	/* Directories = Closed
						 * Other = Standard */
	EDV_MIME_TYPE_ICON_STATE_OPENED,	/* Directories = Opened
						 * Other = Opened */
	EDV_MIME_TYPE_ICON_STATE_INACCESSIBLE,	/* Links = Broken
						 * Directories = No Access
						 * Other = Inaccessible */
	EDV_MIME_TYPE_ICON_STATE_HIDDEN		/* All = Hidden */
} EDVMIMETypeIconState;
#define EDV_MIME_TYPE_TOTAL_ICON_STATES	4

#define EDV_MIME_TYPE_ICON_STATE_NAMES	{	\
	"Standard",				\
	"Opened",				\
	"Inaccessible",				\
	"Hidden"				\
}


/*
 *	Inode MIME Types:
 *
 *	Used for EDV_MIME_TYPE_CLASS_SYSTEM MIME Types.
 */
#define EDV_MIME_TYPE_TYPE_INODE_UNKNOWN	"inode/unknown"
#define EDV_MIME_TYPE_TYPE_INODE_ERROR		"inode/error"
#define EDV_MIME_TYPE_TYPE_INODE_FILE		"inode/file"
#define EDV_MIME_TYPE_TYPE_INODE_EXECUTABLE	"inode/executable"
#define EDV_MIME_TYPE_TYPE_INODE_DIRECTORY	"inode/directory"
#define EDV_MIME_TYPE_TYPE_INODE_LINK		"inode/link"
#define EDV_MIME_TYPE_TYPE_INODE_DEV_BLOCK	"inode/dev-block"
#define EDV_MIME_TYPE_TYPE_INODE_DEV_CHARACTER	"inode/dev-character"
#define EDV_MIME_TYPE_TYPE_INODE_FIFO		"inode/fifo"
#define EDV_MIME_TYPE_TYPE_INODE_SOCKET		"inode/socket"


/*
 *	MIME Type Command Flags:
 */
typedef enum {
	EDV_MIME_TYPE_COMMAND_RUN_IN_TERMINAL	= (1 << 0)
} EDVMIMETypeCommandFlags;


/*
 *	MIME Type Command:
 */
struct _EDVMIMETypeCommand {
	gchar		*name,
			*command,
			*shell_command;
	EDVMIMETypeCommandFlags	flags;
};


/*
 *	MIME Type:
 */
struct _EDVMIMEType {

	EDVMIMETypeClass	mt_class;

	/* Value, its contents is determined by mt_class:
	 *
	 * mt_class			value is a
	 * --------                     ----------
	 * EDV_MIME_TYPE_CLASS_SYSTEM	*Ignored*
	 * EDV_MIME_TYPE_CLASS_FORMAT	Space separated list of extensions
	 * EDV_MIME_TYPE_CLASS_PROGRAM	Path to the program object
	 * EDV_MIME_TYPE_CLASS_UNIQUE	Path to the unique object
	 */
	gchar		*value;

	/* The MIME Type's type (aka the MIME Type's name) */
	gchar		*type;

	/* A one line verbose description */
	gchar		*description;

	/* Small 20x20 icons */
	gchar		*small_icon_file[EDV_MIME_TYPE_TOTAL_ICON_STATES];

	/* Medium 32x32 icons */
	gchar		*medium_icon_file[EDV_MIME_TYPE_TOTAL_ICON_STATES];

	/* Large 48x48 icons */
	gchar		*large_icon_file[EDV_MIME_TYPE_TOTAL_ICON_STATES];

	/* Handler, determines if a command should be used to open
	 * objects assicated with this MIME Type (in the case of member
	 * handler = EDV_MIME_TYPE_HANDLER_COMMAND) or to use one of
	 * Endeavour's windows to open the object
	 */
	EDVMIMETypeHandler	handler;

	/* Commands List
	 *
	 * Each item in the list points to a
	 * EDVMIMETypeCommand
	 *
	 * Command are only used if the handler is set to
	 * EDV_MIME_TYPE_HANDLER_COMMAND
	 *
	 * The first command is considered the default
	 *
	 * The intention of each command is related by the mt_class as
	 * follows:
	 *
	 * mt_class			value is a
	 * --------			----------
	 * EDV_MIME_TYPE_CLASS_SYSTEM	*Ignored*
	 * EDV_MIME_TYPE_CLASS_FORMAT	Program commands or program aliases
	 * EDV_MIME_TYPE_CLASS_PROGRAM	Program commands (no aliases)
	 * EDV_MIME_TYPE_CLASS_UNIQUE	Program commands or program aliases
	 */
	GList           *commands_list;

	/* Time stamps */
	gulong		access_time,	/* In systime seconds */
			modify_time,
			change_time;

};


/*
 *	Creates a new EDVMIMETypeCommand.
 *
 *	Returns a new dynamically allocated EDVMIMETypeCommand or NULL
 *	on error.
 */
extern EDVMIMETypeCommand *edv_mime_type_command_new(void);

/*
 *	Coppies the EDVMIMETypeCommand.
 *
 *	The cmd specifies the EDVMIMETypeCommand to copy.
 *
 *	Returns a new dynamically allocated copy of the specified
 *	EDVMIMETypeCommand or NULL on error.
 */
extern EDVMIMETypeCommand *edv_mime_type_command_copy(EDVMIMETypeCommand *cmd);

/*
 *	Deletes the EDVMIMETypeCommand.
 *
 *	The cmd specifies the EDVMIMETypeCommand to delete.
 */
extern void edv_mime_type_command_delete(EDVMIMETypeCommand *cmd);


/*
 *	Creates a new EDVMIMEType.
 *
 *	Returns a new dynamically allocated EDVMIMEType or NULL on error.
 */
extern EDVMIMEType *edv_mime_type_new(void);

/*
 *	Creates a new EDVMIMEType with values.
 *
 *	The mt_class specifies the EDVMIMEType's class, one of
 *	EDV_MIME_TYPE_CLASS_*.
 *
 *	If value is not NULL then it specifies the EDVMIMEType's value.
 *
 *	If type is not NULL then it specifies the EDVMIMEType's type.
 *
 *	If description is not NULL then it specifies the EDVMIMEType's
 *	description.
 *
 *	Returns a new dynamically allocated EDVMIMEType or NULL on error.
 */
extern EDVMIMEType *edv_mime_type_new_values(
	const EDVMIMETypeClass mt_class,
	const gchar *value,
	const gchar *type,
	const gchar *description
);

/*
 *	Coppies the EDVMIMEType.
 *
 *	The m specifies the EDVMIMEType to copy.
 *
 *	Returns a new dynamically allocated copy of the specified
 *	EDVMIMEType or NULL on error.
 */
extern EDVMIMEType *edv_mime_type_copy(EDVMIMEType *m);

/*
 *	Deletes the EDVMIMEType.
 *
 *	The m specifies the EDVMIMEType to delete.
 */
extern void edv_mime_type_delete(EDVMIMEType *m);


/*
 *	Gets the EDVMIMEType's icon path.
 *
 *	The m specifies the EDVMIMEType.
 *
 *	The size specifies the icon size.
 *
 *	The state specifies the icon state.
 *
 *	Returns the path to the icon file or NULL on error.
 */
extern const gchar *edv_mime_type_get_icon_path(
	EDVMIMEType *m,
	const EDVIconSize size,
	const EDVMIMETypeIconState state
);

/*
 *	Gets the EDVMIMEType's icon path.
 *
 *	The m specifies the EDVMIMEType.
 *
 *	The size specifies the icon size.
 *
 *	The state specifies the icon state.
 *
 *	If accept_smaller_size is TRUE then if the requested size is
 *	not available then a smaller size will be returned if one is
 *	available.
 *
 *	If accept_standard_state is TRUE then if the requested state
 *	is not available then the EDV_MIME_TYPE_ICON_STATE_STANDARD
 *	will be returned if it is available. If the
 *	EDV_MIME_TYPE_ICON_STATE_STANDARD is still not available then
 *	a smaller size will be returned if accept_smaller_size is TRUE.
 *
 *	Returns the path to the icon file or NULL on error.
 */
extern const gchar *edv_mime_type_get_best_icon_path(
        EDVMIMEType *m,
        const EDVIconSize size,
        const EDVMIMETypeIconState state,
        const gboolean accept_smaller_size,
        const gboolean accept_standard_state
);


/*
 *	Gets the EDVMIMEType's command value by the command's name.
 *
 *	The m specifies the EDVMIMEType.
 *
 *	The name specifies the command's name.
 *
 *	Returns the EDVMIMEType's command value or NULL on error.
 */
extern const gchar *edv_mime_type_match_command_value_by_name(
	EDVMIMEType *m,
	const gchar *name
);

/*
 *	Gets the EDVMIMEType's EDVMIMETypeCommand by the command's
 *	name.
 *
 *	The m specifies the EDVMIMEType.
 *
 *	The name specifies the command's name.
 *
 *	Returns the EDVMIMETypeCommand or NULL on error.
 */
extern EDVMIMETypeCommand *edv_mime_type_match_command_by_name(
	EDVMIMEType *m,
	const gchar *name
);


#endif	/* EDV_MIME_TYPE_H */
