#include <errno.h>
#include <glib.h>
#include "edv_types.h"
#include "edv_mime_type.h"


/* EDVMIMETypeCommand */
EDVMIMETypeCommand *edv_mime_type_command_new(void);
EDVMIMETypeCommand *edv_mime_type_command_copy(EDVMIMETypeCommand *cmd);
void edv_mime_type_command_delete(EDVMIMETypeCommand *cmd);

/* EDVMIMEType */
EDVMIMEType *edv_mime_type_new(void);
EDVMIMEType *edv_mime_type_new_values(
	const EDVMIMETypeClass mt_class,
	const gchar *value,
	const gchar *type,
	const gchar *description
);
EDVMIMEType *edv_mime_type_copy(EDVMIMEType *m);
void edv_mime_type_delete(EDVMIMEType *m);

/* Get Icon Path */
const gchar *edv_mime_type_get_icon_path(
	EDVMIMEType *m,
	const EDVIconSize size,
	const EDVMIMETypeIconState state 
);
const gchar *edv_mime_type_get_best_icon_path(
	EDVMIMEType *m,
	const EDVIconSize size,
	const EDVMIMETypeIconState state,
	const gboolean accept_smaller_size,
	const gboolean accept_standard_state
);

/* Get Command */
const gchar *edv_mime_type_match_command_value_by_name(
	EDVMIMEType *m,
	const gchar *name
);
EDVMIMETypeCommand *edv_mime_type_match_command_by_name(
	EDVMIMEType *m,
	const gchar *name
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new EDVMIMETypeCommand.
 *
 *	Returns a new dynamically allocated EDVMIMETypeCommand or NULL
 *	on error.
 */
EDVMIMETypeCommand *edv_mime_type_command_new(void)
{
	return(EDV_MIME_TYPE_COMMAND(g_malloc0(sizeof(EDVMIMETypeCommand))));
}

/*
 *	Coppies the EDVMIMETypeCommand.
 *
 *	The cmd specifies the EDVMIMETypeCommand to copy.
 *
 *	Returns a new dynamically allocated copy of the specified
 *	EDVMIMETypeCommand or NULL on error.
 */
EDVMIMETypeCommand *edv_mime_type_command_copy(EDVMIMETypeCommand *cmd)
{
	EDVMIMETypeCommand	*tar,
				*src = cmd;
	if(src == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	tar = edv_mime_type_command_new();
	if(tar == NULL)
		return(NULL);

	tar->name = STRDUP(src->name);
	tar->command = STRDUP(src->command);
	tar->shell_command = STRDUP(src->shell_command);
	tar->flags = src->flags;

	return(tar);
}

/*
 *	Deletes the EDVMIMETypeCommand. 
 *
 *	The cmd specifies the EDVMIMETypeCommand to delete.
 */
void edv_mime_type_command_delete(EDVMIMETypeCommand *cmd)
{
	if(cmd == NULL)
		return;

	g_free(cmd->name);
	g_free(cmd->command);
	g_free(cmd->shell_command);
	g_free(cmd);
}


/*
 *	Creates a new EDVMIMEType.
 *
 *	Returns a new dynamically allocated EDVMIMEType or NULL on error.
 */
EDVMIMEType *edv_mime_type_new(void)
{
	return(EDV_MIME_TYPE(g_malloc0(sizeof(EDVMIMEType))));
}

/*
 *	Creates a new EDVMIMEType with values.
 *
 *	The mt_class specifies the EDVMIMEType's class, one of
 *	EDV_MIME_TYPE_CLASS_*.
 *
 *	If value is not NULL then it specifies the EDVMIMEType's value.
 *
 *	If type is not NULL then it specifies the EDVMIMEType's type.
 *
 *	If description is not NULL then it specifies the EDVMIMEType's
 *	description.
 *
 *	Returns a new dynamically allocated EDVMIMEType or NULL on error.
 */
EDVMIMEType *edv_mime_type_new_values(
	const EDVMIMETypeClass mt_class,
	const gchar *value,
	const gchar *type,
	const gchar *description
)
{
	EDVMIMEType *m = edv_mime_type_new();
	if(m == NULL)
		return(m);

	m->mt_class = mt_class;
	m->value = STRDUP(value);
	m->type = STRDUP(type);
	m->description = STRDUP(description);

	return(m);
}

/*
 *	Coppies the EDVMIMEType.
 *
 *	The m specifies the EDVMIMEType to copy.
 *
 *	Returns a new dynamically allocated copy of the specified
 *	EDVMIMEType or NULL on error.
 */
EDVMIMEType *edv_mime_type_copy(EDVMIMEType *m)
{
	const gint nicon_states = EDV_MIME_TYPE_TOTAL_ICON_STATES;
	gint i;
	EDVMIMEType *tar;
	const EDVMIMEType *src = m;

	if(src == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	tar = edv_mime_type_new();
	if(tar == NULL)
		return(NULL);

	tar->mt_class = src->mt_class;
	tar->value = STRDUP(src->value);
	tar->type = STRDUP(src->type);
	tar->description = STRDUP(src->description);

	for(i = 0; i < nicon_states; i++)
	{
		tar->small_icon_file[i] = STRDUP(src->small_icon_file[i]);
		tar->medium_icon_file[i] = STRDUP(src->medium_icon_file[i]);
		tar->large_icon_file[i] = STRDUP(src->large_icon_file[i]);
	}

	if(src->commands_list != NULL)
	{
		GList *glist;
		EDVMIMETypeCommand	*src_cmd,
					*tar_cmd;

		for(glist = src->commands_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			src_cmd = EDV_MIME_TYPE_COMMAND(glist->data);
			if(src_cmd == NULL)
				continue;

			tar_cmd = edv_mime_type_command_new();
			if(tar_cmd == NULL)
				break;

			tar_cmd->name = STRDUP(src_cmd->name);
			tar_cmd->command = STRDUP(src_cmd->command);

			tar->commands_list = g_list_append(
				tar->commands_list,
				tar_cmd
			);
		}
	}

	return(tar);
}

/*
 *	Deletes the EDVMIMEType.
 *
 *	The m specifies the EDVMIMEType to delete.
 */
void edv_mime_type_delete(EDVMIMEType *m)
{
	const gint nicon_states = EDV_MIME_TYPE_TOTAL_ICON_STATES;
	gint i;

	if(m == NULL)
		return;

	/* Delete the icon file paths */
	for(i = 0; i < nicon_states; i++)
	{
		g_free(m->small_icon_file[i]);
		g_free(m->medium_icon_file[i]);
		g_free(m->large_icon_file[i]);
	}

	/* Delete the commands */
	if(m->commands_list != NULL)
	{
		g_list_foreach(
			m->commands_list, (GFunc)edv_mime_type_command_delete, NULL
		);
		g_list_free(m->commands_list);
	}                                 

	/* Delete the value, type, and description */
	g_free(m->value);
	g_free(m->type);
	g_free(m->description);

	g_free(m);
}


/*
 *	Gets the EDVMIMEType's icon path.
 *
 *	The m specifies the EDVMIMEType.
 *
 *	The size specifies the icon size.
 *
 *	The state specifies the icon state.
 *
 *	Returns the path to the icon file or NULL on error.
 */
const gchar *edv_mime_type_get_icon_path(
	EDVMIMEType *m,
	const EDVIconSize size,
	const EDVMIMETypeIconState state 
)
{
	if((m == NULL) ||
	   (state < 0) || (state >= EDV_MIME_TYPE_TOTAL_ICON_STATES)
	)
	{
		errno = EINVAL;
		return(NULL);
	}

	switch(size)
	{
	  case EDV_ICON_SIZE_16:
		return(NULL);			/* Not supported */
		break;
	  case EDV_ICON_SIZE_20:
		return(m->small_icon_file[state]);
		break;
	  case EDV_ICON_SIZE_32:
		return(m->medium_icon_file[state]);
		break;
	  case EDV_ICON_SIZE_48:
		return(m->large_icon_file[state]);
		break;
	}

	errno = ENOENT;

	return(NULL);
}

/*
 *	Gets the EDVMIMEType's icon path.
 *
 *	The m specifies the EDVMIMEType.
 *
 *	The size specifies the icon size.
 *
 *	The state specifies the icon state.
 *
 *	If accept_smaller_size is TRUE then if the requested size is
 *	not available then a smaller size will be returned if one is
 *	available.
 *
 *	If accept_standard_state is TRUE then if the requested state
 *	is not available then the EDV_MIME_TYPE_ICON_STATE_STANDARD
 *	will be returned if it is available. If the
 *	EDV_MIME_TYPE_ICON_STATE_STANDARD is still not available then
 *	a smaller size will be returned if accept_smaller_size is TRUE.
 *
 *	Returns the path to the icon file or NULL on error.
 */
const gchar *edv_mime_type_get_best_icon_path(
	EDVMIMEType *m,
	const EDVIconSize size,
	const EDVMIMETypeIconState state,
	const gboolean accept_smaller_size,
	const gboolean accept_standard_state
)
{
	const gchar *path;

	if(m == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	path = edv_mime_type_get_icon_path(
		m,
		size,
		state
	);
	if(path != NULL)
		return(path);

	if(accept_standard_state && (state != EDV_MIME_TYPE_ICON_STATE_STANDARD))
	{
		path = edv_mime_type_get_icon_path(
			m,
			size,
			EDV_MIME_TYPE_ICON_STATE_STANDARD
		);
		if(path != NULL)
			return(path);
	}

	if(accept_smaller_size)
	{
		gint smaller_size;

		while(TRUE)
		{
			smaller_size = 0;
			switch(size)
			{
			  case EDV_ICON_SIZE_16:
				break;
			  case EDV_ICON_SIZE_20:
				smaller_size = EDV_ICON_SIZE_16;
				break;
			  case EDV_ICON_SIZE_32:
				smaller_size = EDV_ICON_SIZE_20;
				break;
			  case EDV_ICON_SIZE_48:
				smaller_size = EDV_ICON_SIZE_32;
				break;
			}
			if(smaller_size == 0)
				break;

			path = edv_mime_type_get_icon_path(
				m,
				smaller_size,
				state
			);
			if(path != NULL)
				return(path);

			if(accept_standard_state)
			{
				path = edv_mime_type_get_icon_path(
					m,
					smaller_size,
					EDV_MIME_TYPE_ICON_STATE_STANDARD
				);
				if(path != NULL)
					return(path);
			}
		}

		errno = ENOENT;

		return(NULL);
	}
	else
	{
		errno = ENOENT;

		return(NULL);
	}
}


/*
 *	Gets the EDVMIMEType's command value by the command's name.
 *
 *	The m specifies the EDVMIMEType.
 *
 *	The name specifies the command's name.
 *
 *	Returns the EDVMIMEType's command value or NULL on error.
 */
const gchar *edv_mime_type_match_command_value_by_name(
	EDVMIMEType *m,
	const gchar *name
)
{
	GList *glist;
	EDVMIMETypeCommand *cmd;

	if((m == NULL) || STRISEMPTY(name))
	{
		errno = EINVAL;
		return(NULL);
	}

	for(glist = m->commands_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		cmd = EDV_MIME_TYPE_COMMAND(glist->data);
		if(cmd == NULL)
			continue;

		if(cmd->name == NULL)
			continue;

		if(!g_strcasecmp(cmd->name, name))
			return(cmd->command);
	}

	errno = ENOENT;

	return(NULL);
}

/*
 *	Gets the EDVMIMEType's EDVMIMETypeCommand by the command's
 *	name.
 *
 *	The m specifies the EDVMIMEType.
 *
 *	The name specifies the command's name.
 *
 *	Returns the EDVMIMETypeCommand or NULL on error.
 */
EDVMIMETypeCommand *edv_mime_type_match_command_by_name(
	EDVMIMEType *m,
	const gchar *name
)
{
	GList *glist;
	EDVMIMETypeCommand *cmd;

	if((m == NULL) || STRISEMPTY(name))
	{
		errno = EINVAL;
		return(NULL);
	}

	for(glist = m->commands_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		cmd = EDV_MIME_TYPE_COMMAND(glist->data);
		if(cmd == NULL)
			continue;

		if(cmd->name == NULL)
			continue;

		if(!g_strcasecmp(cmd->name, name))
			return(cmd);
	}

	errno = ENOENT;

	return(NULL);
}
