#include <glib.h>
#include "edv_utils.h"
#include "edv_process.h"
#include "edv_context.h"
#include "edv_get.h"
#include "edv_message.h"
#include "edv_cfg_list.h"
#include "config.h"


static void edv_message_execute_command(
	EDVContext *ctx,
	const gchar *command
);
static gchar *edv_message_get_message_dialog_path(EDVContext *ctx);
static void edv_message_nexus(
	EDVContext *ctx,
	const gchar *type_name,
	const gchar *title,
	const gchar *message,
	const gchar *details
);

void edv_message(
	EDVContext *ctx,
	const gchar *message
);
void edv_message_info(
	EDVContext *ctx,
	const gchar *title,
	const gchar *message,
	const gchar *details
);
void edv_message_warning(
	EDVContext *ctx,
	const gchar *title,
	const gchar *message,
	const gchar *details
);
void edv_message_error(
	EDVContext *ctx,
	const gchar *title,
	const gchar *message,
	const gchar *details
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Executes the command using the shell specified by the
 *	configuration.
 */
static void edv_message_execute_command(
	EDVContext *ctx,
	const gchar *command
)
{
	gchar *shell_prog;
	const gchar	*shell_cmd = edv_get_s(ctx, EDV_CFG_PARM_PROG_SHELL),
			*shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	);
	(void)edv_system_shell(
		command,
		shell_prog,
		shell_args
	);
	g_free(shell_prog);
}

/*
 *	Gets the full path to the message dialog program.
 */
static gchar *edv_message_get_message_dialog_path(EDVContext *ctx)
{
	return(g_strconcat(
		EDV_PATH_DEF_GLOBAL_LIB_DIR,
		G_DIR_SEPARATOR_S,
		EDV_NAME_BIN_SUBDIR,
		G_DIR_SEPARATOR_S,
		EDV_NAME_MESSAGE_DIALOG,
		NULL
	));
}

/*
 *	Display the message in a message dialog using a specific type.
 */
static void edv_message_nexus(
	EDVContext *ctx,
	const gchar *type_name,
	const gchar *title,
	const gchar *message,
	const gchar *details
)
{
	gchar		*prog_path,
			*dtitle,
			*dmessage,
			*cmd;

	if((ctx == NULL) || (message == NULL))
		return;

	/* Get the full path to the message dialog program */
	prog_path = edv_message_get_message_dialog_path(ctx);
	if(prog_path == NULL)
		return;

	/* If the title was specified then copy and escape all quotes
	 * in the title
	 */
	if(title != NULL)
		dtitle = edv_strsub(
			title,
			"\"",
			"\\\""
		);
	else
		dtitle = NULL;

	/* Copy and escape all quotes in the message */
	dmessage = edv_strsub(
		message,
		"\"",
		"\\\""
	);

	/* Generate the message dialog command */
	if(dtitle != NULL)
		cmd = g_strdup_printf(
"\"%s\" --quiet --type \"%s\" --center \"root\" --title \"%s\" \"%s\"",
			prog_path,
			type_name,
			dtitle,
			dmessage
		);
	else
		cmd = g_strdup_printf(
"\"%s\" --quiet --type \"%s\" --center \"root\" \"%s\"",
			prog_path,
			type_name,
			dmessage
		);

	/* Execute the message dialog command */
	edv_message_execute_command(
		ctx,
		cmd
	);

	g_free(cmd);
	g_free(prog_path);
	g_free(dtitle);
	g_free(dmessage);
}


void edv_message(
	EDVContext *ctx,
	const gchar *message
)
{
	edv_message_info(
		ctx,
		NULL,
		message,
		NULL
	);
}

void edv_message_info(
	EDVContext *ctx,
	const gchar *title,
	const gchar *message,
	const gchar *details
)
{
	edv_message_nexus(
		ctx,
		"info",
		title,
		message,
		details
	);
}

void edv_message_warning(
	EDVContext *ctx,
	const gchar *title,
	const gchar *message,
	const gchar *details
)
{
	edv_message_nexus(
		ctx,
		"warning",
		title,
		message,
		details
	);
}

void edv_message_error(
	EDVContext *ctx,
	const gchar *title,
	const gchar *message,
	const gchar *details
)
{
	edv_message_nexus(
		ctx,
		"error",
		title,
		message,
		details
	);
}
