#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <glib.h>
#include <unistd.h>

#include "edv_utils.h"
#include "edv_path.h"
#include "edv_link.h"
#include "edv_vfs_obj.h"
#include "edv_vfs_obj_stat.h"
#include "config.h"


/* Create */
gint edv_link_create(
	const gchar *path,
	const gchar *target
);

/* Get */
gchar *edv_link_get_target(const gchar *path);
gchar *edv_link_get_target_full(const gchar *path);

/* Check */
gboolean edv_link_is_infinately_recursive(const gchar *path);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)

#define STRPFX(s,p)	((((s) != NULL) && ((p) != NULL)) ? \
 !strncmp((s),(p),strlen(p)) : FALSE)


/*
 *	Creates a new link or sets the target value of an existing link.
 *
 *	The path specifies the path to the new link. If the object
 *	exists and is a link then its target value will be updated.
 *	If the object exists but is not a link then -1 will be returned
 *	and errno will be set to EEXIST.
 *
 *	The target specifies the link's target value, which may be      
 *	NULL for no value.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_link_create(
	const gchar *path,
	const gchar *target
)
{
	EDVVFSObject *obj;

	if(STRISEMPTY(path))
	{
	    errno = EINVAL;
	    return(-2);
	}

	if(target == NULL)
	    target = "";

	/* Remove the existing link if it already exists and is a link */
	obj = edv_vfs_object_lstat(path);
	if(obj != NULL)
	{
	    /* Object already exists, is it a link? */
	    if(EDV_VFS_OBJECT_IS_LINK(obj))
	    {
		/* Remove the existing link */
		if(edv_unlink(path))
		{
		    const gint error_code = (gint)errno;
		    edv_vfs_object_delete(obj);
		    errno = (int)error_code;
		    return(-1);
		}
	    }
	    else
	    {
		/* Exists but is not a link */
		edv_vfs_object_delete(obj);
		errno = EEXIST;
                return(-1);
            }

	    edv_vfs_object_delete(obj);
	}

	return((gint)symlink(
	    (const char *)target,
	    (const char *)path
	));
}


/*
 *	Gets the link's target value.
 *
 *	The path specifies the path to the link.
 *
 *	Returns a dynamically allocated string describing the link's
 *	target value or NULL on error.
 */
gchar *edv_link_get_target(const gchar *path)
{
	struct stat lstat_buf;
	gint len;
	gchar *target;

	if(STRISEMPTY(path))
	{
	    errno = EINVAL;
	    return(NULL);
	}

	if(lstat((const char *)path, &lstat_buf))
	    return(NULL);

#ifdef S_ISLNK
	if(!S_ISLNK(lstat_buf.st_mode))
#else
	if(TRUE)
#endif
	{
	    errno = EINVAL;
	    return(NULL);
	}

	len = (gint)lstat_buf.st_size;
	target = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	if(target == NULL)
	    return(NULL);

	if(len > 0)
	{
	    const gint bytes_read = (gint)readlink(
		(const char *)path,		/* Link */
		target,				/* Value return */
		(size_t)len			/* Value allocation */
	    );
	    if(bytes_read != len)
	    {
		const gint error_code = (gint)errno;
		g_free(target);
		errno = (int)error_code;
		return(NULL);
	    }
	}

	target[len] = '\0';

	return(target);
}

/*
 *	Gets the link's target value as a full path.
 *
 *	The path specifies the path to the link. The path must be a
 *	full path since its parent path will be used to derive the
 *	target value describing the full path.
 *
 *	Returns a dynamically allocated string describing the link's
 *	target value as a full path or NULL on error.
 */
gchar *edv_link_get_target_full(const gchar *path)
{
	gchar	*parent,
		*full_path,
		*v = edv_link_get_target(path);
	if(v == NULL)
	    return(NULL);

	if(g_path_is_absolute(v))
	    return(v);

	parent = g_dirname(path);
	if(parent == NULL)
	    return(v); 

	full_path = g_strconcat(
	    parent,
	    G_DIR_SEPARATOR_S,
	    v,
	    NULL
	);

	g_free(parent);
	g_free(v);

	edv_path_simplify(full_path);

	return(full_path);
}

/*
 *	Checks if the link has a target value that may possibly be
 *	infinately recursive.
 *
 *	The path specifies the path to the link.
 *
 *	Returns TRUE if the link's target value may be infinatly
 *	recursive.
 */
gboolean edv_link_is_infinately_recursive(const gchar *path)
{
	gboolean status = FALSE;
	gint nrecursions;
	gchar		*s,
			*parent_path,
			*cur_path,
			*target;

	if(STRISEMPTY(path))
	{
	    errno = EINVAL;
	    return(status);
	}

	/* Is the link's target not a directory? */
	if(!edv_path_is_directory(path))
	{
	    errno = ENOTDIR;
	    return(status);
	}

	/* Record the parent path of the starting link */
	parent_path = g_dirname(path);
	if(parent_path == NULL)
	    return(status);

	/* Iterate through each link to reach the target */
	cur_path = g_strdup(path);
	nrecursions = 0;
	while(cur_path != NULL)
	{
	    /* Get the link's target value
	     *
	     * If we get an error then it means that we have reached
	     * the end and cur_path is the final target
	     */
	    target = edv_link_get_target(cur_path);
	    if(target == NULL)
		break;

	    nrecursions++;

	    /* If the target refers to the current location then it
	     * may be infinately recursive
	     */
	    if(!strcmp((const char *)target, "."))
	    {
		g_free(target);
		g_free(cur_path);
		cur_path = NULL;
		status = TRUE;
		break;
	    }

	    /* Get this link's parent */
	    s = g_dirname(cur_path);
	    if(s == NULL)
	    {
		g_free(target);
		break;
	    }

	    /* Set the link's target as the current path */
	    g_free(cur_path);
	    cur_path = edv_path_evaluate(
		s,				/* Parent */
		target				/* Child */
	    );

	    g_free(s);
	    g_free(target);

	    /* If we are on the second or subsequent target then
	     * check if it refers back to the specified path
	     */
	    if(nrecursions >= 2)
	    {
		if(cur_path != NULL)
		{
		    if(!strcmp((const char *)cur_path, (const char *)path))
		    {
			g_free(cur_path);
			cur_path = NULL;	/* Stop recursing */
			status = TRUE;
		    }
		}
	    }
	}

	if(cur_path != NULL)
	{
	    /* Is the target the parent or a grand parent of the
	     * starting link?
	     */
	    if(edv_path_is_parent(
		cur_path,			/* Parent */
		parent_path			/* Child */
	    ))
		status = TRUE;
	}

	g_free(cur_path);
	g_free(parent_path);

	if(status)
	    errno = ELOOP;
	else
	    errno = 0;

	return(status);
}
