#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <glib.h>

#include "cfg.h"

#include "edv_utils.h"
#include "edv_path.h"
#include "edv_directory.h"
#include "edv_link.h"
#include "edv_process.h"
#include "edv_interps.h"

#include "edv_cfg_list.h"
#include "config.h"


static gchar *edv_interps_get_lock_path(const CfgList *cfg_list);
static gchar *edv_interps_get_path(const CfgList *cfg_list);

/* InterPS Lock Link */
gint edv_interps_get_lock(const CfgList *cfg_list);
gint edv_interps_make_lock(
	const CfgList *cfg_list,
	const gint pid,
	const gboolean force
);
void edv_interps_remove_lock(const CfgList *cfg_list);

/* InterPS Command */
void edv_interps_send_command_notify(
	const CfgList *cfg_list,
	const gint pid
);
gboolean edv_interps_command_pending(const CfgList *cfg_list);
void edv_interps_send_commands(
	const CfgList *cfg_list,
	const gint pid,
	const gchar **cmds_list
);
gchar **edv_interps_get_commands(const CfgList *cfg_list);
void edv_interps_remove_commands(const CfgList *cfg_list);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the path to the InterPS lock link path.
 *
 *	If the environment variable ENV_VAR_NAME_EDVINTERPSLOCK is
 *	set then this value will be returned as the InterPS lock link
 *	path, otherwise the InterPS lock link path will be derived
 *	from the configuration.
 */
static gchar *edv_interps_get_lock_path(const CfgList *cfg_list)
{
	const gchar *env_lock_link = g_getenv(ENV_VAR_NAME_EDVINTERPSLOCK);
	if(env_lock_link != NULL)
	{
		return(g_strdup(env_lock_link));
	}
	else
	{
		const gchar *local_data_dir = EDV_GET_S(EDV_CFG_PARM_DIR_LOCAL);
		return(g_strconcat(
			(local_data_dir != NULL) ? local_data_dir : "",
			G_DIR_SEPARATOR_S,
			EDV_NAME_INTERPS_LOCK_LINK,
			NULL
		));
	}
}

/*
 *	Returns a dynamically allocated string describing the full
 *	path to the InterPS directory.
 */
static gchar *edv_interps_get_path(const CfgList *cfg_list)
{
	const gchar *env_cmd_file = g_getenv(ENV_VAR_NAME_EDVINTERPS);
	if(env_cmd_file != NULL)
	{
		return(g_strdup(env_cmd_file));
	}
	else
	{   
		const gchar *local_data_dir = EDV_GET_S(EDV_CFG_PARM_DIR_LOCAL);
		return(g_strconcat(
			(local_data_dir != NULL) ? local_data_dir : "",
			G_DIR_SEPARATOR_S,
			EDV_NAME_INTERPS_DIR,
			NULL
		));
	}
}


/*
 *	Gets the pid of the currently running process of Endeavour.
 *
 *	The lock link will be checked and the actual process will be
 *	checked to see if it is actually running.
 *
 *	Returns the pid of the currently running process of Endeavour
 *	or 0 if it is not.
 */
gint edv_interps_get_lock(const CfgList *cfg_list)
{
	gint p;
	gchar		*v,
					*lock_link = edv_interps_get_lock_path(cfg_list);
	if(lock_link == NULL)
		return(0);

	/* Get the lock link's value */
	v = edv_link_get_target(lock_link);

	g_free(lock_link);

	/* Lock link does not exist or error reading it? */
	if(v == NULL)
		return(0);

	/* Get pid from the link's target and check if it is
	 * actually running
	 */
	p = (gint)atoi((const char *)v);

	g_free(v);

	if(edv_pid_exists(p))
		return(p);
	else
		return(0);
}

/*
 *	Creates a new InterPS lock link that refers to the specified
 *	pid.
 *
 *	The link will be tested to see if it already exists. If it does
 *	not exist or force is TRUE then a new lock will be created and
 *	refer to the specified pid.
 *
 *	Returns:
 *
 *	0	Success or link already exists.
 *	-1	General error or unable to (re)create link.
 *	-2	Invalid value.
 *	-3	Systems error or memory allocation error.
 */
gint edv_interps_make_lock(
	const CfgList *cfg_list,
	const gint pid,
	const gboolean force
)
{
	gchar		*target,
					*lock_link = edv_interps_get_lock_path(cfg_list);
	if(lock_link == NULL)
		return(-2);

	/* Lock link already exists and not forcing? */
	if(edv_path_exists(lock_link) && !force)
	{
		g_free(lock_link);
		return(0);
	}

	/* Format the link's target, which is the pid of the
	 * specified process p
	 */
	target = g_strdup_printf(
		"%i",
		pid
	);
	if(target == NULL)
	{
		const gint error_code = (gint)errno;
		g_free(lock_link);
		errno = (int)error_code;
		return(0);
	}

	/* (Re)create the lock link */
	if(edv_unlink(lock_link))
	{
		const gint error_code = (gint)errno;
		if(error_code != ENOENT)
		{
			/* The object exists but we were unable to remove it */
			g_free(target);
			g_free(lock_link);
			errno = (int)error_code;
			return(-1);
		}
	}
	if(edv_link_create(
		lock_link,				/* Path */
		target				/* Target */
	))
	{
		const gint error_code = (gint)errno;
		g_free(target);
		g_free(lock_link);
		errno = (int)error_code;
		return(-1);
	}

	g_free(target);
	g_free(lock_link);

	return(0);
}

/*
 *	Removes the InterPS lock link.
 */
void edv_interps_remove_lock(const CfgList *cfg_list)
{
	gchar *lock_link = edv_interps_get_lock_path(cfg_list);
	if(lock_link == NULL)
		return;

	(void)edv_unlink(lock_link);
	g_free(lock_link);
}


/*
 *	Sends a SIGUSR1 signal.
 *
 *	The pid specifies the process ID.
 */
void edv_interps_send_command_notify(
	const CfgList *cfg_list,
	const gint pid
)
{
	if(pid <= 0)
		return;

#ifdef SIGUSR1
	(void)kill(
		(int)pid,
		SIGUSR1
	);
#else
#warning SIGUSR1 is not defined, InterPS command support will not be fully compiled and functional
#endif
}

/*
 *	Checks if the InterPS direcotry exists and if there is at least
 *	one command queued.
 */
gboolean edv_interps_command_pending(const CfgList *cfg_list)
{
	EDVDirectory *dp;
	gchar *interps_path = edv_interps_get_path(cfg_list);
	if(interps_path == NULL)
		return(FALSE);

	/* Check if the InterPS directory exists */
	dp = edv_directory_open(
		interps_path,
		FALSE,				/* Not sorted */
		FALSE				/* Exclude notations */
	);
	if(dp == NULL)
	{
		g_free(interps_path);
		return(FALSE);
	}

	g_free(interps_path);

	/* Check if there is at least one command queued */
	if(edv_directory_next(dp) != NULL)
	{
		edv_directory_close(dp);
		return(TRUE);
	}
	else
	{
		edv_directory_close(dp);
		return(FALSE);
	}
}

/*
 *	Sends the InterPS commands list to the process.
 *
 *	The p specifies the process ID.
 *
 *	The cmds_list specifies the command list. The last pointer in
 *	the array must be NULL to mark the end of the list.
 */
void edv_interps_send_commands(
	const CfgList *cfg_list,
	const gint pid,
	const gchar **cmds_list
)
{
	FILE *fp;
	gint		i,
					len,
					fd;
	const gchar *s;
	gchar		*interps_path,
					*cmd_path;

	if((pid <= 0) || (cmds_list == NULL))
		return;

	interps_path = edv_interps_get_path(cfg_list);
	if(interps_path == NULL)
		return;

	/* Create the InterPS directory as needed */
	if(edv_directory_create(
		interps_path,
		TRUE,				/* Create parents */
		NULL				/* No new paths list return */
	))
	{
		g_free(interps_path);
		return;
	}

	/* Write each command into a file in the InterPS directory */
	for(i = 0; cmds_list[i] != NULL; i++)
	{
		s = cmds_list[i];
		len = (gint)strlen(s);
		if(len <= 0)
			continue;

		/* Create a new command file */
		cmd_path = edv_tmp_name(interps_path);
		if(cmd_path == NULL)
			break;

		/* Open the new command file for writing */
		fp = fopen((const char *)cmd_path, "wb");
		if(fp == NULL)
		{
			(void)edv_unlink(cmd_path);
			g_free(cmd_path);
			break;
		}

		fd = (gint)fileno(fp);

		/* Since InterPS is asyncronous we need to mark that this
		 * new command file is being written so that it does not
		 * get read by another process while we are writing it
		 */
		if(edv_permissions_set_fd(
			fd,
			EDV_PERMISSION_UW
		))
		{
			(void)fclose(fp);
			(void)edv_unlink(cmd_path);
			g_free(cmd_path);
			break;
		}

		/* Write this command into a command file */
		if(fwrite(
			s,
			sizeof(gchar),
			(size_t)len,
			fp
		) != (size_t)len)
		{
			/* A write error has occured, do not allow this command
			 * file to be processed since partial or corrupt
			 * commands may be a safety or security risk
			 */
			(void)fclose(fp);
			(void)edv_unlink(cmd_path);
			g_free(cmd_path);
			continue;
		}

		/* Close the command file */
		if(fclose(fp))
		{
			/* A write error has occured, do not allow this command
			 * file to be processed since partial or corrupt
			 * commands may be a safety or security risk
			 */
			(void)edv_unlink(cmd_path);
			g_free(cmd_path);
			continue;
		}

		/* Mark that the new command file has been written by
		 * setting it readable so that its contents can get read
		 * and its command be processed
		 */
		(void)edv_permissions_set(
			cmd_path,
			EDV_PERMISSION_UR | EDV_PERMISSION_UW
		);		

		g_free(cmd_path);
	}

	g_free(interps_path);

	/* Notify the running Endeavour process that InterPS commands
	 * have been queued
	 */
	edv_interps_send_command_notify(
		cfg_list,
		pid
	);
}

/*
 *	Gets all the InterPS commands from the InterPS command files
 *	and removes each InterPS command file that was read.
 *
 *	Returns a dynamically allocated list of strings describing
 *	the InterPS commands list, the last pointer in the list will be
 *	NULL to mark the end of the list.
 */
gchar **edv_interps_get_commands(const CfgList *cfg_list)
{
	FILE *fp;
	struct stat stat_buf;
	gint		i,
					len,
					ncmds;
	const gchar *name;
	gchar		*s,
					*cmd_path,
					**cmds_list;
	EDVDirectory *dp;
	gchar *interps_path = edv_interps_get_path(cfg_list);
	if(interps_path == NULL)
		return(NULL);

	/* Open the InterPS directory */
	dp = edv_directory_open(
		interps_path,
		FALSE,				/* Not sorted */
		FALSE				/* Exclude notations */
	);
	if(dp == NULL)
	{
		g_free(interps_path);
		return(NULL);
	}

	/* Get the commands from each command file */
	ncmds = 0;
	cmds_list = NULL;
	for(name = edv_directory_next(dp);
		name != NULL;
		name = edv_directory_next(dp)
	)
	{
		cmd_path = g_strconcat(
			interps_path,
			G_DIR_SEPARATOR_S,
			name,
			NULL
		);
		if(cmd_path == NULL)
			continue;

		/* Open this command file for reading */
		fp = fopen((const char *)cmd_path, "rb");
		if(fp == NULL)
		{
			/* Do not remove the command file when we are not able
			 * to open it for reading since it may be set
			 * write-only in the case of when a new command file
			 * is in the process of being written
			 */
			g_free(cmd_path);
			continue;
		}

		/* Get the command file's statistics */
		if(fstat(fileno(fp), &stat_buf))
		{
			/* Do not remove the command file when we are not able
			 * to get its statistics since it may be set write-only
			 * in the case of when a new command file is in the
			 * process of being written
			 */
			(void)fclose(fp);
			g_free(cmd_path);
			continue;
		}

		/* Not set readable by the user? */
		if(!(stat_buf.st_mode & S_IRUSR))
		{
			/* Do not remove the command file when it is not set
			 * readable by the user in the case of when a new
			 * command file is in the process of being written
			 */
			(void)fclose(fp);
			g_free(cmd_path);
			continue;
		}

		/* Empty file? */
		len = (gint)(stat_buf.st_size / sizeof(gchar));
		if(len <= 0)
		{
			/* Do not remove the command file when it is empty
			 * since we may have opened it between the time it
			 * was created and being written
			 */
			(void)fclose(fp);
			g_free(cmd_path);
			continue;
		}

		/* Allocate the string to hold the command */
		s = (gchar *)g_malloc((len + 1) * sizeof(gchar));
		if(s == NULL)
		{
			(void)fclose(fp);
			(void)edv_unlink(cmd_path);
			g_free(cmd_path);
			break;
		}

		/* Read the entire command file as a single command */
		if(fread(
			s,
			sizeof(gchar),
			(size_t)len,
			fp
		) != (size_t)len)
		{
			g_free(s);
			(void)fclose(fp);
			edv_unlink(cmd_path);
			g_free(cmd_path);
			continue;
		}

		s[len] = '\0';

		/* Append this command to the comands list */
		i = ncmds;
		ncmds = i + 1;
		cmds_list = (gchar **)g_realloc(
			cmds_list,
			ncmds * sizeof(gchar *)
		);
		if(cmds_list != NULL)
		{
			cmds_list[i] = s;
		}
		else
		{
			g_free(s);
			ncmds = 0;
		}

		/* Close this command file and remove it */
		(void)fclose(fp);
		(void)edv_unlink(cmd_path);
		g_free(cmd_path);
	}

	/* Close the InterPS directory */
	edv_directory_close(dp);
	g_free(interps_path);

	/* Append a NULL pointer to the comands list to mark its end */
	i = ncmds;
	ncmds = i + 1;
	cmds_list = (gchar **)g_realloc(
		cmds_list,
		ncmds * sizeof(gchar *)
	);
	if(cmds_list != NULL)
	{
		cmds_list[i] = NULL;
	}
	else
	{
		ncmds = 0;
	}

	return(cmds_list);
}

/*
 *	Removes all the InterPS command files and the InterPS directory.
 */
void edv_interps_remove_commands(const CfgList *cfg_list)
{
	const gchar *name;
	gchar *cmd_path;
	EDVDirectory *dp;
	gchar *interps_path = edv_interps_get_path(cfg_list);
	if(interps_path == NULL)
		return;

	/* Open the InterPS directory */
	dp = edv_directory_open(
		interps_path,
		FALSE,				/* Not sorted */
		FALSE				/* Exclude notations */
	);
	if(dp == NULL)
	{
		g_free(interps_path);
		return;
	}

	/* Remove each command file */
	for(name = edv_directory_next(dp);
		name != NULL;
		name = edv_directory_next(dp)
	)
	{
		cmd_path = g_strconcat(
			interps_path,
			G_DIR_SEPARATOR_S,
			name,
			NULL
		);
		if(cmd_path == NULL)
			continue;

		/* Remove this command file */
		(void)edv_unlink(cmd_path);
		g_free(cmd_path);
	}

	/* Close and remove the InterPS directory */
	edv_directory_close(dp);
	(void)edv_directory_remove(
		interps_path,
		FALSE,				/* Do not recurse */
		FALSE,				/* Do not force */
		NULL,				/* No paths list return */
		NULL, NULL
	);
	g_free(interps_path);
}
