#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <glib.h>
#include "edv_context_private.h"
#include "edv_id.h"
#include "edv_id_get.h"


/* UID */
GList *edv_uids_list(EDVContext *ctx);
EDVUID *edv_uids_list_match_uid(
	EDVContext *ctx,
	const gint user_id      
);
EDVUID *edv_uids_list_match_gid(
	EDVContext *ctx,
	const gint group_id
);
gint edv_uid_name_to_uid(
	EDVContext *ctx,
	const gchar *user_name  
);
gchar *edv_uid_uid_to_name(
	EDVContext *ctx,
	const gint user_id
);
gchar *edv_uid_gid_to_name(
	EDVContext *ctx,   
	const gint group_id
);

/* GID */
GList *edv_gids_list(EDVContext *ctx);
gint edv_gid_name_to_gid(
	EDVContext *ctx,
	const gchar *group_name
);
gchar *edv_gid_gid_to_name(     
	EDVContext *ctx,
	const gint group_id
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the UIDs list.
 *
 *	Returns a GList of EDVUID * which must not be modified
 *	or deleted.
 */
GList *edv_uids_list(EDVContext *ctx)
{
	if(ctx == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	return(ctx->uids_list);
}

/*
 *	Gets the UID by user ID.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The user_id specifies the user ID.
 *
 *	Returns the UID which must not be modified or deleted or NULL
 *	on error.
 */
EDVUID *edv_uids_list_match_uid(
	EDVContext *ctx,
	const gint user_id
)
{
	GList *glist;
	EDVUID *uid;

	if(ctx == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	for(glist = ctx->uids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		uid = EDV_UID(glist->data);
		if(uid == NULL)
			continue;

		if(uid->user_id == user_id)
			return(uid);
	}

	errno = ENOENT;

	return(NULL);
}

/*
 *	Gets the UID by group ID.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The group_id specifies the group ID.
 *
 *	Returns the UID which must not be modified or deleted or NULL
 *	on error.
 */
extern EDVUID *edv_uids_list_match_gid(
	EDVContext *ctx,
	const gint group_id
)
{
	GList *glist;
	EDVUID *uid;

	if(ctx == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	for(glist = ctx->uids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		uid = EDV_UID(glist->data);
		if(uid == NULL)
			continue;

		if(uid->group_id == group_id)
			return(uid);
	}

	errno = ENOENT;

	return(NULL);
}

/*
 *	Gets the user ID from the user name.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The user_name specifies the string describing the user name.
 *
 *	Returns the user ID or negative on error.
 */
gint edv_uid_name_to_uid(
	EDVContext *ctx,
	const gchar *user_name
)
{
	if((ctx == NULL) || STRISEMPTY(user_name))
	{
		errno = EINVAL;
		return(-2);
	}

	/* Match by number? */
	if(isdigit(*user_name))
	{
		return((gint)atoi((const char *)user_name));
	}
	else
	{
		GList *glist;
		EDVUID *uid;

		for(glist = ctx->uids_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			uid = EDV_UID(glist->data);
			if(uid == NULL)
				continue;

			if(uid->name == NULL)
				continue;

			if(!strcmp((const char *)uid->name, (const char *)user_name))
				return(uid->user_id);
		}
	}

	errno = ENOENT;

	return(-1);
}

/*
 *	Gets the user name or a string describing the user ID from the
 *	user ID.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The user_id specifies the user ID.
 *
 *	Returns a dynamically allocated string describing the user name
 *	or user ID or NULL on error. The calling function must delete
 *	the returned string.
 */
gchar *edv_uid_uid_to_name(
	EDVContext *ctx,
	const gint user_id
)
{
	EDVUID *uid = edv_uids_list_match_uid(ctx, user_id);
	if((uid != NULL) ? (uid->name != NULL) : FALSE)
	{
		return(g_strdup(uid->name));
	}
	else if(user_id < 0)
	{
		return(g_strdup(""));
	}
	else
	{
		return(g_strdup_printf(
			"%i",
			user_id
		));
	}
}

/*
 *	Gets the user name or a string describing the user ID from the
 *	group ID.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The group_id specifies the group ID.
 *
 *	Returns a dynamically allocated string describing the user name
 *	or user ID or NULL on error. The calling function must delete
 *	the returned string.
 */
gchar *edv_uid_gid_to_name(
	EDVContext *ctx,
	const gint group_id
)
{
	EDVUID *uid = edv_uids_list_match_gid(ctx, group_id);
	if((uid != NULL) ? (uid->name != NULL) : FALSE)
	{
		return(g_strdup(uid->name));
	}
	else if(group_id < 0)
	{
		return(g_strdup(""));
	}
	else
	{
		return(g_strdup_printf(
			"%i",
			group_id
		));
	}
}


/*
 *	Gets the GIDs list.
 *
 *	Returns a GList of EDVGID * which must not be modified
 *	or deleted.
 */
GList *edv_gids_list(EDVContext *ctx)
{
	if(ctx == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	return(ctx->gids_list);
}

/*
 *	Gets the GID by group ID.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The group_id specifies the group ID.
 *
 *	Returns the GID which must not be modified or deleted or NULL
 *	on error.
 */
EDVGID *edv_gids_list_match_gid(
	EDVContext *ctx,
	const gint group_id
)
{
	GList *glist;
	EDVGID *gid;

	if(ctx == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	for(glist = ctx->gids_list;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
		gid = EDV_GID(glist->data);
		if(gid == NULL)
			continue;

		if(gid->group_id == group_id)
			return(gid);
	}

	errno = ENOENT;

	return(NULL);
}

/*
 *	Gets the group ID from the group name.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The group_name specifies the string describing the group name.
 *
 *	Returns the user ID or negative on error.
 */
gint edv_gid_name_to_gid(
	EDVContext *ctx,
	const gchar *group_name
)
{
	if((ctx == NULL) || STRISEMPTY(group_name))
	{
		errno = EINVAL;
		return(-2);
	}

	/* Match by number? */
	if(isdigit(*group_name))
	{
		return((gint)atoi((const char *)group_name));
	}
	else
	{
		GList *glist;
		EDVGID *gid;

		for(glist = ctx->gids_list;
			glist != NULL;
			glist = g_list_next(glist)
		)
		{
			gid = EDV_GID(glist->data);
			if(gid == NULL)
				continue;

			if(gid->name == NULL)
				continue;

			if(!strcmp((const char *)gid->name, (const char *)group_name))
				return(gid->group_id);
		}
	}

	errno = ENOENT;

	return(-1);
}

/*
 *	Gets the group name or a string describing the group ID from
 *	the group ID.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The group_id specifies the group ID.
 *
 *	Returns a dynamically allocated string describing the group
 *	name or group ID or NULL on error or NULL on error. The
 *	calling function must delete the returned string.
 */
gchar *edv_gid_gid_to_name(
	EDVContext *ctx,
	const gint group_id
)
{
	EDVGID *gid = edv_gids_list_match_gid(ctx, group_id);
	if((gid != NULL) ? (gid->name != NULL) : FALSE)
	{
		return(g_strdup(gid->name));
	}
	else if(group_id < 0)
	{
		return(g_strdup(""));
	}
	else
	{
		return(g_strdup_printf(
			"%i",
			group_id
		));
	}
}
