#include <pwd.h>
#include <grp.h>
#include <glib.h>

#include "edv_id.h"
#include "edv_id_fio.h"


GList *edv_uids_list_open_from_system(
	GList *uids_list,
	const gint insert_index
);
GList *edv_gids_list_open_from_system(
	GList *gids_list,
	const gint insert_index
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets a list of UIDs from the system.
 *
 *	The uids_list specifies the UIDs list, a GList of EDVUID *
 *	uids.
 *
 *	If insert_index is non-negative then the opened UIDs will
 *	be inserted at the position in the uids_list specified by
 *	insert_index, otherwise the opened UIDs will be appended to
 *	the uisd_list.
 *
 *	Returns a GList of EDVUID * or NULL on error. The
 *	calling function must delete the returned list and each
 *	uid.
 */
GList *edv_uids_list_open_from_system(
	GList *uids_list,
	const gint insert_index
)
{
	struct passwd *pwent;
	gint last_insert_index = insert_index;
	EDVUID *uid;

	/* Open the passwords file and get the first entry */
	pwent = getpwent();

	/* Iterate through each entry */
	while(pwent != NULL)
	{
		/* Create a new UID */
		uid = edv_uid_new();
		if(uid == NULL)
			break;

		/* Get this UID */
		uid->name = STRDUP(pwent->pw_name);
		uid->password = STRDUP(pwent->pw_passwd);
		uid->user_id = (gint)pwent->pw_uid;
		uid->group_id = (gint)pwent->pw_gid;
		uid->verbose_name = STRDUP(pwent->pw_gecos);
		uid->home_directory = STRDUP(pwent->pw_dir);
		uid->shell_program = STRDUP(pwent->pw_shell);

		if(insert_index < 0)
		{
			uids_list = g_list_append(
				uids_list,
				uid
			);
		}
		else
		{
			uids_list = g_list_insert(
				uids_list,
				uid,
				last_insert_index
			);
			last_insert_index++;
		}

		/* Get the next entry */
		pwent = getpwent();
	}

	/* Close the passwords file */
	endpwent();

	return(uids_list);
}

/*
 *	Opens the GIDs from the system.
 *
 *	The gids_list specifies the GIDs list, a GList of EDVGID *
 *	gids.
 *
 *	If insert_index is non-negative then the opened GIDs will
 *	be inserted at the position in the gids_list specified by
 *	insert_index, otherwise the opened GIDs will be appended to
 *	the gids_list.
 *
 *	Returns a GList of EDVGID * or NULL on error. The
 *	calling function must delete the returned list and each
 *	gid.
 */
GList *edv_gids_list_open_from_system(
	GList *gids_list,
	const gint insert_index
)
{
	struct group *grent;
	gint last_insert_index = insert_index;
	EDVGID *gid;

	/* Open the groups file and get the first entry */
	grent = getgrent();

	/* Iterate through each entry */
	while(grent != NULL)
	{
		/* Create a new GID */
		gid = edv_gid_new();
		if(gid == NULL)
			break;

		/* Get this GID */
		gid->name = STRDUP(grent->gr_name);
		gid->password = STRDUP(grent->gr_passwd);
		gid->group_id = (gint)grent->gr_gid;

		/* Any group members to add? */
		if(grent->gr_mem != NULL)
		{
			gint i;
			for(i = 0; grent->gr_mem[i] != NULL; i++)
				gid->group_members_list = g_list_append(
					gid->group_members_list,
					STRDUP(grent->gr_mem[i])
				);
		}

		if(insert_index < 0)
		{
			gids_list = g_list_append(
				gids_list,
				gid
			);
		}
		else
		{
			gids_list = g_list_insert(
				gids_list,
				gid,
				last_insert_index
			);
			last_insert_index++;
		}

		/* Get the next entry */
		grent = getgrent();
	}

	/* Close the groups file */
	endgrent();

	return(gids_list);
}
