/*
	     Endeavour Mark II API - Configuration Fetching

	Gets configuration values and lists configuration parameters.
 */

#ifndef EDV_GET_H
#define EDV_GET_H

#include <glib.h>
#include "edv_context.h"


/*
 *	Value Types:
 */
typedef enum {
	EDV_GET_TYPE_NONE,
	EDV_GET_TYPE_INT8,
	EDV_GET_TYPE_UINT8,
	EDV_GET_TYPE_INT16,
	EDV_GET_TYPE_UINT16,
	EDV_GET_TYPE_INT32,
	EDV_GET_TYPE_UINT32,
	EDV_GET_TYPE_INT64,
	EDV_GET_TYPE_UINT64,
	EDV_GET_TYPE_FLOAT,
	EDV_GET_TYPE_DOUBLE,
	EDV_GET_TYPE_STRING,
	EDV_GET_TYPE_INT_LIST,
	EDV_GET_TYPE_STRING_LIST,
	EDV_GET_TYPE_COLOR,
	EDV_GET_TYPE_ACCELKEY_LIST,
	EDV_GET_TYPE_STYLE,
	EDV_GET_TYPE_MENU
} EDVGetType;


/*
 *	Gets the type of the configuration parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 *
 *	Can return EDV_GET_TYPE_NONE on error.
 */
extern EDVGetType edv_get_type(
	EDVContext *ctx,
	const gchar *parm
);

/*
 *	Gets the complete list of Endeavour 2 configuration parameters.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The total specifies the return value for the number of
 *	configuration parameters.
 *
 *	Returns a statically allocated GList of gchar * strings
 *	describing the configuration parameter names. The returned
 *	list must not be modified or deleted.
 */
extern GList *edv_get_cfg_list_parameters(EDVContext *ctx);

/*
 *	Gets the boolean value of the configuration parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 */
extern gboolean edv_get_b(
	EDVContext *ctx,
	const gchar *parm
);

/*
 *	Gets the integer value of the configuration parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 */
extern gint edv_get_i(
	EDVContext *ctx,
	const gchar *parm
);

/*
 *	Gets the long integer value of the configuration parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 */
extern glong edv_get_l(
	EDVContext *ctx,
	const gchar *parm
);

/*
 *	Gets the unsigned long integer value of the configuration
 *	parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 */
extern gulong edv_get_ul(
	EDVContext *ctx,
	const gchar *parm
);

/*
 *	Gets the float value of the configuration parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 */
extern gfloat edv_get_f(
	EDVContext *ctx,
	const gchar *parm
);

/*
 *	Gets the double value of the configuration parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 */
extern gdouble edv_get_d(
	EDVContext *ctx,
	const gchar *parm
);

/*
 *	Gets the string value of the configuration parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 *
 *	The returned pointer must not be modified or deleted. Can
 *	return NULL if the parameter does not exist.
 */
extern const gchar *edv_get_s(
	EDVContext *ctx,
	const gchar *parm
);

/*
 *	Gets the version of Endeavour 2 from the configuration.
 *
 *	Returns TRUE on success or FALSE on error.
 */
extern gboolean edv_get_version(
	EDVContext *ctx,
	gint *major,
	gint *minor,
	gint *release
);


#endif	/* EDV_GET_H */
