#include <errno.h>
#include <glib.h>
#include "../cfg.h"
#include "edv_context_private.h"
#include "edv_get.h"
#include "edv_cfg_list.h"
#include "config.h"


GList *edv_get_cfg_list_parameters(EDVContext *ctx);

EDVGetType edv_get_type(
	EDVContext *ctx,
	const gchar *parm
);
gboolean edv_get_b(
	EDVContext *ctx,
	const gchar *parm
);
gint edv_get_i(
	EDVContext *ctx,
	const gchar *parm
);
glong edv_get_l(
	EDVContext *ctx,
	const gchar *parm
);
gulong edv_get_ul(
	EDVContext *ctx,
	const gchar *parm
);
gfloat edv_get_f(
	EDVContext *ctx,
	const gchar *parm
);
gdouble edv_get_d(
	EDVContext *ctx,
	const gchar *parm
);
const gchar *edv_get_s(
	EDVContext *ctx,
	const gchar *parm
);

gboolean edv_get_version(
	EDVContext *ctx,
	gint *major, gint *minor, gint *release
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Gets the configuration parameters list.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The total specifies the return value for the number of
 *	configuration parameters.
 *
 *	Returns a dynamically allocated list of configuration
 *	parameters. The calling function must delete the returned
 *	pointer array and each string.
 */
GList *edv_get_cfg_list_parameters(EDVContext *ctx)
{
	if(ctx == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	/* If the configuration parameters list on the context was
	 * not allocated then it means this is the first time this
	 * function was called and that we should allocate it
	 * before returning it
	 */
	if(ctx->cfg_list_parameters == NULL)
	{
		CfgItem *cfg_item = ctx->cfg_list;
		if(cfg_item != NULL)
		{
			while(cfg_item->parameter != NULL)
			{
				ctx->cfg_list_parameters = g_list_append(
					ctx->cfg_list_parameters,
					g_strdup(cfg_item->parameter)
				);
				cfg_item++;
			}
		}
		else
		{
			errno = ENOENT;
		}
	}

	return(ctx->cfg_list_parameters);
}

/*
 *	Gets the type of the configuration parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 *
 *	Can return EDV_GET_TYPE_NONE on error.
 */
EDVGetType edv_get_type(
	EDVContext *ctx,
	const gchar *parm
)
{
	gint i;
	CfgItem *cfg_item;

	if((ctx == NULL) || STRISEMPTY(parm))
		return(EDV_GET_TYPE_NONE);

	i = CFGItemListMatchParameter(ctx->cfg_list, parm);
	if(i < 0)
		return(EDV_GET_TYPE_NONE);

	cfg_item = &ctx->cfg_list[i];
	switch(cfg_item->type)
	{
	  case CFG_ITEM_TYPE_NONE:
		return(EDV_GET_TYPE_NONE);
		break;
	  case CFG_ITEM_TYPE_INT8:
		return(EDV_GET_TYPE_INT8);
		break;
	  case CFG_ITEM_TYPE_UINT8:
		return(EDV_GET_TYPE_UINT8);
		break;
	  case CFG_ITEM_TYPE_INT16:
		return(EDV_GET_TYPE_INT16);
		break;
	  case CFG_ITEM_TYPE_UINT16:
		return(EDV_GET_TYPE_UINT16);
		break;
	  case CFG_ITEM_TYPE_INT32:
		return(EDV_GET_TYPE_INT32);
		break;
	  case CFG_ITEM_TYPE_UINT32:
		return(EDV_GET_TYPE_UINT32);
		break;
	  case CFG_ITEM_TYPE_INT64:
		return(EDV_GET_TYPE_INT64);
		break;
	  case CFG_ITEM_TYPE_UINT64:
		return(EDV_GET_TYPE_UINT64);
		break;
	  case CFG_ITEM_TYPE_FLOAT:
		return(EDV_GET_TYPE_FLOAT);
		break;
	  case CFG_ITEM_TYPE_DOUBLE:
		return(EDV_GET_TYPE_DOUBLE);
		break;
	  case CFG_ITEM_TYPE_STRING:
		return(EDV_GET_TYPE_STRING);
		break;
	  case CFG_ITEM_TYPE_INT_LIST:
		return(EDV_GET_TYPE_INT_LIST);
		break;
	  case CFG_ITEM_TYPE_STRING_LIST:
		return(EDV_GET_TYPE_STRING_LIST);
		break;
	  case CFG_ITEM_TYPE_COLOR:
		return(EDV_GET_TYPE_COLOR);
		break;
	  case CFG_ITEM_TYPE_ACCELKEY_LIST:
		return(EDV_GET_TYPE_ACCELKEY_LIST);
		break;
	  case CFG_ITEM_TYPE_STYLE:
		return(EDV_GET_TYPE_STYLE);
		break;
	  case CFG_ITEM_TYPE_MENU:
		return(EDV_GET_TYPE_MENU);
		break;
	}

	return(EDV_GET_TYPE_NONE);
}

/*
 *	Gets the boolean value of the configuration parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 */
gboolean edv_get_b(
	EDVContext *ctx,
	const gchar *parm
)
{
	if((ctx == NULL) || STRISEMPTY(parm))
		return(FALSE);

	return((gboolean)CFGItemListGetValueI(
		ctx->cfg_list,
		parm
	));
}

/*
 *	Gets the integer value of the configuration parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 */
gint edv_get_i(
	EDVContext *ctx,
	const gchar *parm
)
{
	if((ctx == NULL) || STRISEMPTY(parm))
		return(0);

	return(CFGItemListGetValueI(
		ctx->cfg_list,
		parm
	));
}

/*
 *	Gets the long integer value of the configuration parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 */
glong edv_get_l(
	EDVContext *ctx,
	const gchar *parm
)
{
	if((ctx == NULL) || STRISEMPTY(parm))
		return(0);

	return(CFGItemListGetValueL(
		ctx->cfg_list,
		parm
	));
}

/*
 *	Gets the unsigned long integer value of the configuration
 *	parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 */
gulong edv_get_ul(
	EDVContext *ctx,
	const gchar *parm
)
{
	if((ctx == NULL) || STRISEMPTY(parm))
		return(0);

	return(CFGItemListGetValueUL(
		ctx->cfg_list,
		parm
	));
}

/*
 *	Gets the float value of the configuration parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 */
gfloat edv_get_f(
	EDVContext *ctx,
	const gchar *parm
)
{
	if((ctx == NULL) || STRISEMPTY(parm))
		return(0.0f);

	return(CFGItemListGetValueF(
		ctx->cfg_list,
		parm
	));
}

/*
 *	Gets the double value of the configuration parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 */
gdouble edv_get_d(
	EDVContext *ctx,
	const gchar *parm
)
{
	if((ctx == NULL) || STRISEMPTY(parm))
		return(0.0);

	return(CFGItemListGetValueD(
		ctx->cfg_list,
		parm
	));
}

/*
 *	Gets the string value of the configuration parameter.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The parm specifies the configuration parameter.
 *
 *	The returned pointer must not be modified or deleted. Can
 *	return NULL if the parameter does not exist.
 */
const gchar *edv_get_s(
	EDVContext *ctx,
	const gchar *parm
)
{
	if((ctx == NULL) || STRISEMPTY(parm))
		return(NULL);

	return(CFGItemListGetValueS(
		ctx->cfg_list,
		parm
	));
}


/*
 *	Gets the version of Endeavour 2 from the configuration.
 *
 *	Returns TRUE on success or FALSE on error.
 */
gboolean edv_get_version(
	EDVContext *ctx,
	gint *major, gint *minor, gint *release
)
{
	const CfgList *cfg_list;

	if(major != NULL)
		*major = 0;
	if(minor != NULL)
		*minor = 0;
	if(release != NULL)
		*release = 0;

	if(ctx == NULL)
		return(FALSE);

	cfg_list = ctx->cfg_list;

	if(major != NULL)
		*major = CFGItemListGetValueI(
			cfg_list,
			EDV_CFG_PARM_VERSION_MAJOR
		);
	if(minor != NULL)
		*minor = CFGItemListGetValueI(
			cfg_list,
			EDV_CFG_PARM_VERSION_MINOR
		);
	if(release != NULL)
		*release = CFGItemListGetValueI(
			cfg_list,
			EDV_CFG_PARM_VERSION_RELEASE
		);

	return(TRUE);
}
