/*
                Endeavour Mark II API - Filesystem Type
 */

#ifndef EDV_FS_TYPE_H
#define EDV_FS_TYPE_H

#include <glib.h>

typedef struct _EDVFSType		EDVFSType;
#define EDV_FS_TYPE(p)			((EDVFSType *)(p))


/*
 *	Flags:
 */
typedef enum {
	EDV_FS_TYPE_NO_DEVICE		= (1 << 0)
} EDVFSTypeFlags;


/*
 *	Filesystem Type Codes & Names:
 */
typedef guint32				EDVFSTypeCode;

#define EDV_FS_TYPE_EMPTY		0x0000

#define EDV_FS_TYPE_MSDOS		0x4D44
#define EDV_FS_TYPE_MSDOS_NAME		"msdos"

#define EDV_FS_TYPE_VFAT		0x4D45	/* This is really msdos, 0x4D44 */
#define EDV_FS_TYPE_VFAT_NAME		"vfat"

#define EDV_FS_TYPE_ISO9660		0x9660
#define EDV_FS_TYPE_ISO9660_NAME	"iso9660"

#define EDV_FS_TYPE_HPFS		0xF995E849
#define EDV_FS_TYPE_HPFS_NAME		"hpfs"

#define EDV_FS_TYPE_EXT			0x137D
#define EDV_FS_TYPE_EXT_NAME		"ext"

#define EDV_FS_TYPE_EXT2		0xEF53
#define EDV_FS_TYPE_EXT2_NAME		"ext2"
#define EDV_FS_TYPE_EXT2_OLD		0xEF51
#define EDV_FS_TYPE_EXT2_OLD_NAME	"ext2"

#define EDV_FS_TYPE_EXT3		0xEF53	/* Same as EDV_FS_TYPE_EXT2 */
#define EDV_FS_TYPE_EXT3_NAME		"ext3"

#define EDV_FS_TYPE_MINIX		0x138F            
#define EDV_FS_TYPE_MINIX_NAME		"minix"           
#define EDV_FS_TYPE_MINIX_ORIG		0x137F
#define EDV_FS_TYPE_MINIX_ORIG_NAME	"minix"

#define EDV_FS_TYPE_MINIX2		0x2468
#define EDV_FS_TYPE_MINIX2_NAME		"minix2"
#define EDV_FS_TYPE_MINIX2_EXT		0x138F
#define EDV_FS_TYPE_MINIX2_EXT_NAME	"minix2"

#define EDV_FS_TYPE_NCP			0x564c
#define EDV_FS_TYPE_NCP_NAME		"ncp"

#define EDV_FS_TYPE_NFS			0x6969
#define EDV_FS_TYPE_NFS_NAME		"nfs"

#define EDV_FS_TYPE_SMB			0x517B
#define EDV_FS_TYPE_SMB_NAME		"smb"

#define EDV_FS_TYPE_XENIX		0x012FF7B4
#define EDV_FS_TYPE_XENIX_NAME		"xenix"

#define EDV_FS_TYPE_SYSV4		0x012FF7B5
#define EDV_FS_TYPE_SYSV4_NAME		"sysv4"

#define EDV_FS_TYPE_SYSV2		0x012FF7B6
#define EDV_FS_TYPE_SYSV2_NAME		"sysv2"

#define EDV_FS_TYPE_COH			0x012FF7B7
#define EDV_FS_TYPE_COH_NAME		"coh"

#define EDV_FS_TYPE_UFS			0x00011954
#define EDV_FS_TYPE_UFS_NAME		"ufs"

#define EDV_FS_TYPE_XIA			0x012FD16D
#define EDV_FS_TYPE_XIA_NAME		"xia"


#define EDV_FS_TYPE_SWAP                0x0050  /* Universal swap */
#define EDV_FS_TYPE_SWAP_NAME           "swap"

#define EDV_FS_TYPE_PROC		0x9fA0
#define EDV_FS_TYPE_PROC_NAME		"proc"


/*
 *	Filesystems List:
 *
 *	List of filesystem type conical names and their codes.
 *
 *	Each entry must be <name>, <code>, <flags> so that they align
 *	with the members of EDVFSType.
 */
#define EDV_FS_TYPE_LIST	{				\
{ EDV_FS_TYPE_MSDOS_NAME,	EDV_FS_TYPE_MSDOS,	0 },	\
{ EDV_FS_TYPE_VFAT_NAME,	EDV_FS_TYPE_VFAT,	0 },	\
								\
{ EDV_FS_TYPE_ISO9660_NAME,	EDV_FS_TYPE_ISO9660,	0 },	\
								\
{ EDV_FS_TYPE_HPFS_NAME,	EDV_FS_TYPE_HPFS,	0 },	\
								\
{ EDV_FS_TYPE_EXT_NAME,		EDV_FS_TYPE_EXT,	0 },	\
{ EDV_FS_TYPE_EXT2_NAME,	EDV_FS_TYPE_EXT2,	0 },	\
{ EDV_FS_TYPE_EXT3_NAME,	EDV_FS_TYPE_EXT3,	0 },	\
								\
{ EDV_FS_TYPE_MINIX_NAME,	EDV_FS_TYPE_MINIX,	0 },	\
{ EDV_FS_TYPE_MINIX2_NAME,	EDV_FS_TYPE_MINIX2,	0 },	\
								\
{ EDV_FS_TYPE_NCP_NAME,		EDV_FS_TYPE_NCP,	0 },	\
{ EDV_FS_TYPE_NFS_NAME,		EDV_FS_TYPE_NFS,	0 },	\
{ EDV_FS_TYPE_SMB_NAME,		EDV_FS_TYPE_SMB,	0 },	\
								\
{ EDV_FS_TYPE_XENIX_NAME,	EDV_FS_TYPE_XENIX,	0 },	\
{ EDV_FS_TYPE_SYSV4_NAME,	EDV_FS_TYPE_SYSV4,	0 },	\
{ EDV_FS_TYPE_SYSV2_NAME,	EDV_FS_TYPE_SYSV2,	0 },	\
								\
{ EDV_FS_TYPE_COH_NAME,		EDV_FS_TYPE_COH,	0 },	\
{ EDV_FS_TYPE_UFS_NAME,		EDV_FS_TYPE_UFS,	0 },	\
{ EDV_FS_TYPE_XIA_NAME,		EDV_FS_TYPE_XIA,	0 },	\
								\
{ EDV_FS_TYPE_SWAP_NAME,	EDV_FS_TYPE_SWAP,	0 },	\
{ EDV_FS_TYPE_PROC_NAME,	EDV_FS_TYPE_PROC,	0 },	\
								\
{ NULL,				0,			0 }	\
}
/*
 *	Filesystem Type:
 */
struct _EDVFSType {
	gchar		*name;			/* Filesystem's conical name */
	EDVFSTypeCode	code;			/* One of EDV_FS_TYPE_* */
	EDVFSTypeFlags	flags;
};


/*
 *	Gets the EDVFSTypeCode from a filesystem conical name.
 *
 *	Returns one of EDV_FS_TYPE_*.
 */
extern EDVFSTypeCode edv_fs_type_get_code_from_name(const gchar *name);

/*
 *	Gets the filesystem's conical name a EDVFSTypeCode.
 *
 *	Returns a statically allocated string describing the
 *	filesystem's conical name.
 */
extern const gchar *edv_fs_type_get_name_from_code(const EDVFSTypeCode code);


/*
 *	Creates a new EDVFSType.
 *
 *	Returns a new dynamically allocated EDVFSType with all of its
 *	values zero'ed or NULL on error.
 */
extern EDVFSType *edv_fs_type_new(void);

/*
 *	Copys the EDVFSType.
 *
 *	The fs_type specifies the EDVFSType to be coppied.
 *
 *	Returns a new dynamically allocated copy of the EDVFSType or
 *	NULL on error.
 */
extern EDVFSType *edv_fs_type_copy(EDVFSType *fs_type);

/*
 *	Deletes and zeros all the values on the EDVFSType.
 *
 *	The fs_type specifies the EDVFSType who's values are to be cleared.
 */
extern void edv_fs_type_clears(EDVFSType *fs_type);

/*
 *	Deletes the EDVFSType.
 *
 *	The fs_type specifies the EDVFSType to be deleted.
 */
extern void edv_fs_type_delete(EDVFSType *fs_type);


/*
 *	Gets the filesystem types list from the system.
 *
 *	Returns a dynamically allocated GList of EDVFSType * filesystem
 *	types. The returned GList and each EDVFSType * filesystem type
 *	must be deleted by calling edv_fs_types_list_delete().
 */
extern GList *edv_fs_types_list_get_from_system(void);

/*
 *	Deletes the list of filesystem types.
 *
 *	The list specifies a GList of EDVFSType * filesystem types.
 */
extern void edv_fs_types_list_delete(GList *fs_types_list);




#endif	/* EDV_FS_TYPE_H */
