#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>

#include "edv_utils.h"
#include "edv_stream.h"
#include "edv_fs_type.h"
#include "config.h"


/* Filesystem Name & Type Conversion */
EDVFSTypeCode edv_fs_type_get_code_from_name(const gchar *name);
const gchar *edv_fs_type_get_name_from_code(const EDVFSTypeCode code);

/* Filesystem Type */
EDVFSType *edv_fs_type_new(void);
EDVFSType *edv_fs_type_copy(EDVFSType *fs_type);
void edv_fs_type_clears(EDVFSType *fs_type);
void edv_fs_type_delete(EDVFSType *fs_type);

/* Filesystem Types List */
GList *edv_fs_types_list_get_from_system(void);
void edv_fs_types_list_delete(GList *fs_types_list);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Static string buffer for holding name returns.
 */
static gchar		edv_fs_type_name_buf[80];


/*
 *	Gets the EDVFSTypeCode from a filesystem conical name.
 *
 *	Returns one of EDV_FS_TYPE_*.
 */
EDVFSTypeCode edv_fs_type_get_code_from_name(const gchar *name)
{
	gint i;
	const EDVFSType	*fs_type,
			list[] = EDV_FS_TYPE_LIST;

	if(STRISEMPTY(name))
		return(EDV_FS_TYPE_EMPTY);

	for(i = 0; list[i].code != EDV_FS_TYPE_EMPTY; i++)
	{
		fs_type = &list[i];
		if(fs_type->name == NULL)
			continue;

		if(g_strcasecmp(fs_type->name, name))
			continue;

		return(fs_type->code);
	}

	return(EDV_FS_TYPE_EMPTY);
}

/*
 *	Gets the filesystem's conical name a EDVFSTypeCode.
 *
 *	Returns a statically allocated string describing the
 *	filesystem's conical name.
 */
const gchar *edv_fs_type_get_name_from_code(const EDVFSTypeCode code)
{
	gint i;
	const EDVFSType	*fs_type,
			list[] = EDV_FS_TYPE_LIST;

	for(i = 0; list[i].code != EDV_FS_TYPE_EMPTY; i++)
	{
		fs_type = &list[i];
		if(fs_type->name == NULL)
			continue;

		if(fs_type->code != code)
			continue;

		strncpy(
			(char *)edv_fs_type_name_buf,
			(const char *)fs_type->name,
			sizeof(edv_fs_type_name_buf)
		);
		edv_fs_type_name_buf[sizeof(edv_fs_type_name_buf) - 1] = '\0';

		return(edv_fs_type_name_buf);
	}

	*edv_fs_type_name_buf = '\0';

	return(edv_fs_type_name_buf);
}


/*
 *	Creates a new EDVFSType.
 *
 *	Returns a new dynamically allocated EDVFSType with all of its
 *	values zero'ed or NULL on error.
 */
EDVFSType *edv_fs_type_new(void)
{
	return(EDV_FS_TYPE(g_malloc0(sizeof(EDVFSType))));
}

/*
 *	Copys the EDVFSType.
 *
 *	The fs_type specifies the EDVFSType to be coppied.
 *
 *	Returns a new dynamically allocated copy of the EDVFSType or
 *	NULL on error.
 */
EDVFSType *edv_fs_type_copy(EDVFSType *fs_type)
{
	EDVFSType	*src = fs_type,
					*tar;
	if(src == NULL)
		return(NULL);

	tar = edv_fs_type_new();
	if(tar == NULL)
		return(NULL);

	tar->name = STRDUP(src->name);
	tar->code = src->code;
	tar->flags = src->flags;

	return(tar);
}

/*
 *	Deletes and zeros all the values on the EDVFSType.
 *
 *	The fs_type specifies the EDVFSType who's values are to be cleared.
 */
void edv_fs_type_clears(EDVFSType *fs_type)
{
	if(fs_type == NULL)
		return;

	g_free(fs_type->name);
	(void)memset(
		fs_type,
		0x00,
		sizeof(EDVFSType)
	);
}

/*
 *	Deletes the EDVFSType.
 *
 *	The fs_type specifies the EDVFSType to be deleted.
 */
void edv_fs_type_delete(EDVFSType *fs_type)
{
	if(fs_type == NULL)
		return;

	g_free(fs_type->name);
	g_free(fs_type);
}


/*
 *	Gets the filesystem types list from the system.
 *
 *	Returns a dynamically allocated GList of EDVFSType * filesystem
 *	types. The returned GList and each EDVFSType * filesystem type
 *	must be deleted by calling edv_fs_types_list_delete().
 */
GList *edv_fs_types_list_get_from_system(void)
{
	FILE *fp;
	GList *fs_types_list = NULL;

	/* Open the filesystems list file */
	fp = fopen(PATH_PROC_FILESYSTEMS, "rb");
	if(fp != NULL)
	{
		/* Read the filesystems list file */
		gboolean	swap_is_listed = FALSE,
				no_device;
		const gchar *s;
		gchar	*arg,
			*line;
		EDVFSType *fs_type;

		for(line = NULL;
		    edv_stream_read_strptrbrk(
			fp,
			&line,
			"\n\r",
			FALSE,			/* Exclude end character */
			TRUE			/* Block */
		    );
		    g_free(line), line = NULL
		)
		{
			s = line;
			s = edv_strarg(
				s,
				&arg,
				TRUE,		/* Parse escapes */
				TRUE		/* Parse quotes */
			);
			if(arg == NULL)
				continue;

			no_device = FALSE;

			if(!g_strcasecmp(arg, "nodev"))
			{
				no_device = TRUE;
				g_free(arg);
				s = edv_strarg(
					s,
					&arg,
					TRUE,	/* Parse escapes */
					TRUE	/* Parse quotes */
				);
				if(arg == NULL)
					continue;
			}

			/* Add this entry to the filesystem types list */
			fs_type = edv_fs_type_new();
			if(fs_type != NULL)
			{
				fs_type->name = g_strdup(arg);
				fs_type->code = edv_fs_type_get_code_from_name(arg);
				if(no_device)
					fs_type->flags |= EDV_FS_TYPE_NO_DEVICE;
				fs_types_list = g_list_append(
					fs_types_list,
					fs_type
				);
			}

			/* Mark if the "swap" device was listed */
			if(!g_strcasecmp(arg, "swap"))
				swap_is_listed = TRUE;

			g_free(arg);
		}
		g_free(line);

		/* Close the filesystems list file */
		(void)fclose(fp);

		/* Need to add "swap" to the list? */
		if(!swap_is_listed)
		{
			const gchar *name = "swap";
			EDVFSType *fs_type = edv_fs_type_new();
			if(fs_type != NULL)
			{
				fs_type->name = g_strdup(name);
				fs_type->code = edv_fs_type_get_code_from_name(name);
				fs_types_list = g_list_append(
					fs_types_list,
					fs_type
				);
			}
		}
	}

	return(fs_types_list);
}

/*
 *	Deletes the list of filesystem types.
 *
 *	The list specifies a GList of EDVFSType * filesystem types.
 */
void edv_fs_types_list_delete(GList *fs_types_list)
{
	if(fs_types_list == NULL)
		return;

	g_list_foreach(
		fs_types_list,
		(GFunc)edv_fs_type_delete,
		NULL
	);
	g_list_free(fs_types_list);
}

