/*
		Endeavour Mark II API - Low-Level Directory

	Low-level directory creating, removing, and listing.

	See also: edv_link.h edv_path.h edv_utils.h
 */

#ifndef EDV_DIRECTORY_H
#define EDV_DIRECTORY_H

#include <glib.h>


/*
 *	Directory Stream:
 */
typedef struct _EDVDirectory		EDVDirectory;
#define EDV_DIRECTORY			((EDVDirectory *)(p))


/*
 *	Creates a new directory.
 *
 *	The path specifies the full path to the new directory to 
 *	create.
 *
 *	If create_parents is TRUE then any parent directories will
 *	be created as needed.
 *
 *	If new_paths_list_rtn is not NULL then a list of gchar * full
 *	paths describing the new directories that been created will be
 *	returned. The calling function must delete the returned list
 *	and each string.
 *
 *	Returns 0 on success or if the directory already exists or
 *	non-zero on error.
 */
extern gint edv_directory_create(
	const gchar *path,
	const gboolean create_parents,
	GList **new_paths_list_rtn
);


/*
 *	Removes the directory.
 *
 *	The path specifies the full path to the directory to remove. 
 *	The path must refer to a directory, unless force is TRUE, in 
 *	which case the object will be removed regardless of its type.
 *
 *	If recursive is TRUE then the directory and any contents
 *	within it will be removed. If path refers to a link whos
 *	destination is a directory then only the link will be
 *	removed.
 *
 *	If force is TRUE then the object specified by path will be
 *	removed even if it is not a directory.
 *
 *	If removed_paths_list_rtn is not NULL then a list of gchar *
 *	full paths describing the objects that were removed will be
 *	returned. The calling function must delete the returned list
 *	and each string.
 *
 *	The progress_cb and progress_data specifies the progress
 *	callback. If progress_cb() returns FALSE then the operation
 *	will be aborted.
 *
 *	Returns 0 on success or non-zero on error.
 */
extern gint edv_directory_remove(
	const gchar *path,
	const gboolean recursive,
	const gboolean force,
	GList **removed_paths_list_rtn,
	gint (*progress_cb)(
		const gchar *,			/* Path to the object being removed */
		gpointer			/* Data */
	),
	gpointer progress_data
);


/*
 *	Opens the directory.
 *
 *	The path specifies the directory.
 *
 *	If sorted is TRUE then the directory entries will be sorted.
 *
 *	If include_notations is TRUE then directory entry notations
 *	as ".." and "." will be included.
 *
 *	Returns the directory stream or NULL on error.
 */
extern EDVDirectory *edv_directory_open(
	const gchar *path,
	const gboolean sorted,
	const gboolean include_notations
);

/*
 *	Gets the next entry in the directory.
 *
 *	The dp specifies the directory stream.
 *
 *	Returns a statically allocated string describing the name of
 *	the next entry or NULL on error or if there are no more
 *	entries.
 */
extern const gchar *edv_directory_next(EDVDirectory *dp);

/*
 *	Gets the current position in the directory stream.
 *
 *	The dp specifies the directory stream.
 *
 *	Returns the current position in the directory stream.
 */
extern gulong edv_directory_tell(EDVDirectory *dp);

/*
 *	Seeks to the start of the next entry in the directory stream
 *	that edv_directory_next() will return.
 *
 *	The dp specifies the directory stream.
 *
 *	The i specifies the new position in units of entries in the
 *	directory stream.
 *
 *	Returns the new position in the directory stream. If the actual
 *	number of entries in the directory stream is less than i then a
 *	short position, will be returned and the next call to
 *	edv_directory_next() will return NULL.
 */
extern gulong edv_directory_seek(
        EDVDirectory *dp,
        gulong i
);

/*
 *	Rewinds the directory stream back to the beginning.
 *
 *	The dp specifies the directory stream.
 */
extern void edv_directory_rewind(EDVDirectory *dp);

/*
 *	Closes the directory stream.
 *
 *	The dp specifies the directory stream.
 */
extern void edv_directory_close(EDVDirectory *dp);


/*
 *	Gets the list of directory entry names.
 *
 *	The path specifies the path to the directory to list.
 *
 *	If sort is TRUE then the directory entries will be sorted.
 *
 *	If include_notations is TRUE then directory entry notations
 *	as ".." and "." will be included.
 *
 *	Returns a dynamically allocated GList of gchar * strings
 *	describing the name of each entry in the directory. The returned
 *	GList and each gchar * string must be deleted by the calling
 *	function.
 */
extern GList *edv_directory_list(
	const gchar *path,
	const gboolean sort,
	const gboolean include_notations
);

/*
 *	Deletes the list of directory names.
 *
 *	The names_list specifies a GList of gchar * strings to be
 *	deleted.
 */
extern void edv_directory_list_delete(GList *names_list);


#endif	/* EDV_DIRECTORY_H */
