#include <string.h>
#include <errno.h>
#include <fcntl.h>				/* mkdir() */
#include <dirent.h>
#include <sys/types.h>				/* mkdir(), stat*() */
#include <sys/stat.h>				/* mkdir(), stat*() */
#include <glib.h>
#include <unistd.h>				/* mkdir() */

#include "../../include/disk.h"

#include "edv_utils.h"
#include "edv_path.h"
#include "edv_directory.h"
#include "config.h"


/*
 *	Directory Flags:
 */
typedef enum {
	EDV_DIRECTORY_SORT	= (1 << 0),
	EDV_DIRECTORY_NOTATIONS = (1 << 1)	/* Include "." and ".." */
} EDVDirectoryFlags;


/* Create */
gint edv_directory_create(
	const gchar *path,
	const gboolean create_parents,
	GList **new_paths_list_rtn
);

/* Remove */
static gint edv_directory_remove_iterate(
	const gchar *path,
	const gboolean force,
	GList **removed_paths_list_rtn,
	gint (*progress_cb)(
			const gchar *,
			gpointer
	),
	gpointer progress_data
);
gint edv_directory_remove(
	const gchar *path,
	const gboolean recursive,
	const gboolean force,
	GList **removed_paths_list_rtn,
	gint (*progress_cb)(
		const gchar *,			/* Path to the object being removed */
		gpointer			/* progress_data */
	),
	gpointer progress_data
);

/* Iterated Listing */
static gint edv_directory_sort_cb(gconstpointer a, gconstpointer b);
EDVDirectory *edv_directory_open(
	const gchar *path,
	const gboolean sort,
	const gboolean include_notations
);
const gchar *edv_directory_next(EDVDirectory *dp);
gulong edv_directory_tell(EDVDirectory *dp);
gulong edv_directory_seek(
	EDVDirectory *dp,
	gulong i
);
void edv_directory_rewind(EDVDirectory *dp);
void edv_directory_close(EDVDirectory *dp);

/* Listing */
GList *edv_directory_list(
	const gchar *path,
	const gboolean sort,
	const gboolean include_notations
);
void edv_directory_list_delete(GList *names_list);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? (gint)strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Directory Stream:
 */
struct _EDVDirectory {

	/* Pointer to the directory stream, only used when not sorted
	 * (!(flags & EDV_DIRECTORY_SORT))
	 */
	DIR		*dp;

	EDVDirectoryFlags	flags;

	/* Names list, only used when sorted (flags & EDV_DIRECTORY_SORT) */
	GList		*names_list,
			*names_list_current;
	gulong		names_list_position;

};


/*
 *	Creates a new directory.
 *
 *	The path specifies the full path to the new directory to
 *	create.
 *
 *	If create_parents is TRUE then any parent directories will
 *	be created as needed.
 *
 *	If new_paths_list_rtn is not NULL then a list of gchar * full
 *	paths describing the new directories that been created will be
 *	returned. The calling function must delete the returned list
 *	and each string.
 *
 *	Returns 0 on success or if the directory already exists or
 *	non-zero on error.
 */
gint edv_directory_create(
	const gchar *path,
	const gboolean create_parents,
	GList **new_paths_list_rtn
)
{
	struct stat stat_buf;
	const guint m = edv_get_umask();
	gchar *dpath;

	if(new_paths_list_rtn != NULL)
		*new_paths_list_rtn = NULL;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

	if(!g_path_is_absolute(path))
	{
		errno = EINVAL;
		return(-2);
	}

	dpath = g_strdup(path);
	if(dpath == NULL)
		return(-3);

	/* Create the parent directories as needed? */
	if(create_parents)
	{
		/* Check if each directory compoent exists and create them
		 * as needed
		 */
		gchar *s = dpath + 1;
		while(*s != '\0')
		{
			/* Deliminator reached? */
			if(*s == G_DIR_SEPARATOR)
			{
				/* Set this deliminator to null and check if this
				 * compoent does not exist
				 */
				*s = '\0';
				if(stat((const char *)dpath, &stat_buf))
				{
					const gint error_code = (gint)errno;
					if(error_code == ENOENT)
					{
						/* This compoent does not
						 * exist, so create it
						 */
#if defined(_WIN32)
						if(mkdir((const char *)dpath))
#else
						if(mkdir(
							(const char *)dpath,
							(mode_t)(~m) &
								(S_IRUSR | S_IWUSR | S_IXUSR |
								 S_IRGRP | S_IWGRP | S_IXGRP |
								 S_IROTH | S_IWOTH | S_IXOTH)
						))
#endif	/* !_WIN32 */
						{
							/* Unable to create the new directory */
							const gint error_code = (gint)errno;
							g_free(dpath);
							errno = (int)error_code;
							return(-1);
						}
						else
						{
							if(new_paths_list_rtn != NULL)
								*new_paths_list_rtn = g_list_append(
									*new_paths_list_rtn,
									g_strdup(dpath)
								);
						}
					}
					else
					{
						/* Unable to get this
						 * compoent's statistics
						 */
						g_free(dpath);
						errno = (int)error_code;
						return(-1);
					}
				}
				else
				{
#ifdef S_ISDIR
					/* The compoent already
					 * exists, check if it is not
					 * a directory
					 */
					if(!S_ISDIR(stat_buf.st_mode))
					{
						g_free(dpath);
						errno = ENOTDIR;
						return(-2);
					}
#else
#warning "S_ISDIR was not defined, there will be no way to check if an object is a directory"
#endif	/* S_ISDIR */
				}

				/* Restore the deliminator */
				*s = G_DIR_SEPARATOR;
			}
			s++;
		}
	}

	/* Check if the specified path does not exist and create it
	 * as needed
	 */
	if(stat((const char *)dpath, &stat_buf))
	{
		const gint error_code = (gint)errno;
		if(error_code == ENOENT)
		{
			/* The specified path not exist, create it */
#if defined(_WIN32)
			if(mkdir((const char *)dpath))
#else
			if(mkdir(
				(const char *)dpath,
				(mode_t)(~m) &
					(S_IRUSR | S_IWUSR | S_IXUSR |
					 S_IRGRP | S_IWGRP | S_IXGRP |
					 S_IROTH | S_IWOTH | S_IXOTH)
			))
#endif
			{
				/* Unable to create the new directory */
				const gint error_code = (gint)errno;
				g_free(dpath);
				errno = (int)error_code;
				return(-1);
			}
			else
			{
				if(new_paths_list_rtn != NULL)
					*new_paths_list_rtn = g_list_append(
						*new_paths_list_rtn,
						g_strdup(dpath)
					);
			}
		}
		else
		{
			/* Unable to get this compoent's statistics */
			g_free(dpath);
			errno = (int)error_code;
			return(-1);
		}
	}
	else
	{
#ifdef S_ISDIR
		/* The compoent already exists, check if it is not a
		 * directory
		 */
		if(!S_ISDIR(stat_buf.st_mode))
		{
			g_free(dpath);
			errno = ENOTDIR;
			return(-2);
		}
#endif	/* S_ISDIR */
	}

	g_free(dpath);

	return(0);
}


/*
 *	Removes the directory and any contents within the directory.
 */
static gint edv_directory_remove_iterate(
	const gchar *path,
	const gboolean force,
	GList **removed_paths_list_rtn,
	gint (*progress_cb)(
			const gchar *,
			gpointer
	),
	gpointer progress_data
)
{
	gint		status = 0,
			error_code = 0;
	GList *names_list = edv_directory_list(
		path,
		FALSE,				/* Unsorted */
		FALSE				/* Exclude notations */
	);
	if(names_list != NULL)
	{
		/* Delete each object in this directory */
		const gchar *name;
		gchar *child_path;
		GList *glist;

		for(glist = names_list;
		    glist != NULL;
		    glist = g_list_next(glist)
		)
		{
			name = (const gchar *)glist->data;
			if(name == NULL)
				continue;

			child_path = g_strconcat(
				path,
				G_DIR_SEPARATOR_S,
				name,
				NULL
			);
			if(child_path != NULL)
			{
				/* Report the progress */
				if(progress_cb != NULL)
				{
					if(!progress_cb(
						child_path,
						progress_data
					))
					{
						g_free(child_path);
						error_code = EINTR;
						status = -4;
						break;
					}
				}

				/* Is this child object locally a directory? */
				if(edv_path_is_ldirectory(child_path))
				{
					/* Remove this child directory and any contents
					 * within it
					 */
					status = edv_directory_remove_iterate(
						child_path,
						force,
						removed_paths_list_rtn,
						progress_cb, progress_data
					);
					if(status != 0)
						error_code = (gint)errno;
				}
				else
				{
					/* Remove this child object */
					status = edv_unlink(child_path);
					if(status != 0)
					{
						error_code = (gint)errno;
					}
					else
					{
						if(removed_paths_list_rtn != NULL)
							*removed_paths_list_rtn = g_list_append(
								*removed_paths_list_rtn,
								g_strdup(child_path)
							);
					}
				}
				g_free(child_path);
			}

			/* Unable to remove a child object? */
			if(status != 0)
				break;

			g_free(glist->data);
		}
		while(glist != NULL)
		{
			g_free(glist->data);
			glist = g_list_next(glist);
		}
		g_list_free(names_list);
	}

	/* Any error occured while removing one of this directory's
	 * contents?
	 */
	if(status != 0)
	{
		errno = (int)error_code;
		return(status);
	}

	/* Report the progress */
	if(progress_cb != NULL)
	{
		if(!progress_cb(
			path,
			progress_data
		))
		{
			errno = EINTR;
			return(-4);
		}
	}

	/* Remove this directory */
	status = (gint)rmdir((const char *)path);
	if(status != 0)
	{
		/* Unable to remove the directory */
		const gint error_code = (gint)errno;
#ifdef ENOTDIR
		if(error_code == ENOTDIR)
		{
			/* The specified path does not refer to a directory,
			 * check if we should remove only the object
			 */
			if(force)
			{
				/* Remove only this object */
				status = edv_unlink(path);
				if(status == 0)
				{
					if(removed_paths_list_rtn != NULL)
						*removed_paths_list_rtn = g_list_append(
							*removed_paths_list_rtn,
							g_strdup(path)
						);
				}
			}
		}
#else
#warning "ENOTDIR was not defined, there will be no way to check if an object is not a directory"
#endif	/* ENOTDIR */
	}
	else
	{
		/* Directory removed successfully */
		if(removed_paths_list_rtn != NULL)
			*removed_paths_list_rtn = g_list_append(
				*removed_paths_list_rtn,
				g_strdup(path)
			);
	}

	return(status);
}

/*
 *	Removes the directory.
 *
 *	The path specifies the full path to the directory to remove.
 *	The path must refer to a directory, unless force is TRUE, in
 *	which case the object will be removed regardless of its type.
 *
 *	If recursive is TRUE then the directory and any contents
 *	within it will be removed. If path refers to a link whos
 *	destination is a directory then only the link will be
 *	removed.
 *
 *	If force is TRUE then the object specified by path will be
 *	removed even if it is not a directory.
 *
 *	If removed_paths_list_rtn is not NULL then a list of gchar *
 *	full paths describing the objects that were removed will be
 *	returned. The calling function must delete the returned list
 *	and each string.
 *
 *	The progress_cb and progress_data specifies the progress
 *	callback. If progress_cb() returns FALSE then the operation
 *	will be aborted.
 *
 *	Returns 0 on success or non-zero on error.
 */
gint edv_directory_remove(
	const gchar *path,
	const gboolean recursive,
	const gboolean force,
	GList **removed_paths_list_rtn,
	gint (*progress_cb)(
		const gchar *,			/* Path to the object being removed */
		gpointer			/* Data */
	),
	gpointer progress_data
)
{
	gint status;

	if(removed_paths_list_rtn != NULL)
		*removed_paths_list_rtn = NULL;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(-2);
	}

	if(!g_path_is_absolute(path))
	{
		errno = EINVAL;
		return(-2);
	}

	/* Remove the directory and any contents within it? */
	if(recursive)
	{
		/* Remove the directory and any contents within it */
		status = edv_directory_remove_iterate(
			path,
			force,
			removed_paths_list_rtn,
			progress_cb, progress_data
		);
	}
	else
	{
		/* Report the progress */
		if(progress_cb != NULL)
		{
			if(!progress_cb(
				path,
				progress_data
			))
			{
				errno = EINTR;
				return(-4);
			}
		}

		/* Remove only the directory, do not recurse */
		status = (gint)rmdir((const char *)path);
		if(status != 0)
		{
			/* Unable to remove the directory */
			const gint error_code = (gint)errno;
			if(error_code == ENOTDIR)
			{
				/* The specified path does not refer
				 * to a directory, remove the object
				 * even if it is not a directory?
				 */
				if(force)
				{
					/* Remove the object */
					status = edv_unlink(path);
					if(status == 0)
					{
						if(removed_paths_list_rtn != NULL)
							*removed_paths_list_rtn = g_list_append(
								*removed_paths_list_rtn,
								g_strdup(path)
							);
					}
				}
			}
		}
		else
		{
			/* Directory removed successfully */
			if(removed_paths_list_rtn != NULL)
				*removed_paths_list_rtn = g_list_append(
					*removed_paths_list_rtn,
					g_strdup(path)
				);
		}
	}

	return(status);
}


/*
 *	g_list_sort() directory entry sort callback.
 */
static gint edv_directory_sort_cb(gconstpointer a, gconstpointer b)
{
	return((gint)strcmp(
		(const char *)a,
		(const char *)b
	));
}

/*
 *	Opens the directory.
 *
 *	The path specifies the path to the directory to open.
 *
 *	If sort is TRUE then the directory entries will be sorted.
 *
 *	If include_notations is TRUE then directory entry notations
 *	as ".." and "." will be included.
 *
 *	Returns the directory stream or NULL on error.
 */
EDVDirectory *edv_directory_open(
	const gchar *path,
	const gboolean sort,
	const gboolean include_notations
)
{
	DIR *dp;
	EDVDirectory *d;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Open the directory */
	dp = opendir((const char *)path);
	if(dp == NULL)
		return(NULL);

	/* Create a new directory stream */
	d = (EDVDirectory *)g_malloc0(sizeof(EDVDirectory));
	if(d == NULL)
	{
		(void)closedir(dp);
		errno = ENOMEM;
		return(NULL);
	}

	d->dp = dp;
	if(sort)
		d->flags |= EDV_DIRECTORY_SORT;
	if(include_notations)
		d->flags |= EDV_DIRECTORY_NOTATIONS;
/*
	d->names_list = NULL;
	d->names_list_current = NULL;
	d->names_list_position = 0l;
 */

	/* If the entries are to be sorted then we need to load all
	 * the entries first and report each one from the loaded
	 * list when edv_directory_next() is called
	 */
	if(sort)
	{
		struct dirent *dent;
		const gchar *name;
		for(dent = readdir(dp); dent != NULL; dent = readdir(dp))
		{
			name = (const gchar *)dent->d_name;

			/* Skip notations? */
			if(!include_notations)
			{
				if(!strcmp((const char *)name, ".") ||
				   !strcmp((const char *)name, "..")
				)
					continue;
			}

			d->names_list = g_list_append(
				d->names_list,
				g_strdup(name)
			);
		}
		if(d->names_list != NULL)
		{
			/* Sort */
			d->names_list = g_list_sort(
				d->names_list,
				edv_directory_sort_cb
			);

			/* Set the starting position */
			d->names_list_current = d->names_list;
/*			d->names_list_position = 0l; */
		}
	}

	return(d);
}

/*
 *	Gets the next entry in the directory.
 *
 *	The dp specifies the directory stream.
 *
 *	Returns a statically allocated string describing the name of
 *	the next entry or NULL on error or if there are no more
 *	entries.
 */
const gchar *edv_directory_next(EDVDirectory *dp)
{
	if(dp == NULL)
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Is the directory stream is sorted? */
	if(dp->flags & EDV_DIRECTORY_SORT)
	{
		/* Return and seek to the next name in the preloaded
		 * names list
		 *
		 * Get the current position in the names list
		 */
		if(dp->names_list_current != NULL)
		{
			/* Get the name of this entry */
			GList *glist = dp->names_list_current;
			const gchar *name = (const gchar *)glist->data;

			/* Seek the current position pointer to the
			 * next entry in the list
			 */
			dp->names_list_current = g_list_next(glist);
			dp->names_list_position++;

			return(name);
		}
		else
		{
			/* End of list reached */
			errno = 0;
			return(NULL);
		}
	}
	else
	{
		/* Get the next entry from readdir() */
		const gchar *name;
		struct dirent *dent = readdir(dp->dp);
		while(dent != NULL)
		{
			name = (const gchar *)dent->d_name;
			if(!(dp->flags & EDV_DIRECTORY_NOTATIONS))
			{
				if(!strcmp((const char *)name, ".") ||
				   !strcmp((const char *)name, "..")
				)
				{
					dent = readdir(dp->dp);
					continue;
				}
			}
			return(name);

			/* Never reached */
/*			dent = readdir(dp->dp); */
		}

		/* End of list reached */
		errno = 0;
		return(NULL);
	}
}

/*
 *	Gets the current position in the directory stream.
 *
 *	The dp specifies the directory stream.
 *
 *	Returns the current position in the directory stream.
 */
gulong edv_directory_tell(EDVDirectory *dp)
{
	if(dp == NULL)
	{
		errno = EINVAL;
		return(0l);
	}

	/* Is the directory stream is sorted? */
	if(dp->flags & EDV_DIRECTORY_SORT)
	{
		errno = 0;
		return(dp->names_list_position);
	}
	else
	{
		const off_t i = telldir(dp->dp);
		return((i >= 0l) ? (gulong)i : 0l);
	}
}

/*
 *	Seeks to the start of the next entry in the directory stream
 *	that edv_directory_next() will return.
 *
 *	The dp specifies the directory stream.
 *
 *	The i specifies the new position in units of entries in the
 *	directory stream.
 *
 *	Returns the new position in the directory stream. If the actual
 *	number of entries in the directory stream is less than i then a
 *	short position, will be returned and the next call to
 *	edv_directory_next() will return NULL.
 */
gulong edv_directory_seek(
	EDVDirectory *dp,
	gulong i
)
{
	if(dp == NULL)
		return(0l);

	/* Is the directory stream is sorted? */
	if(dp->flags & EDV_DIRECTORY_SORT)
	{
		dp->names_list_current = dp->names_list;
		for(dp->names_list_position = 0l;
		    dp->names_list_position < i;
		    dp->names_list_position++
		)
		{
			if(dp->names_list_current == NULL)
				break;

			dp->names_list_current = g_list_next(dp->names_list_current);
		}
		return(dp->names_list_position);
	}
	else
	{
		/* Seek using seekdir() */
		off_t i2;
		seekdir(
			dp->dp,
			(off_t)i
		);
		/* Get and return the new position */
		i2 = telldir(dp->dp);
		return((i2 >= 0l) ? (gulong)i2 : 0l);
	}

	/* Never reached */
	return(0l);
}

/*
 *	Rewinds the directory stream back to the beginning.
 *
 *	The dp specifies the directory stream.
 */
void edv_directory_rewind(EDVDirectory *dp)
{
	if(dp == NULL)
		return;

	/* Is the directory stream is sorted? */
	if(dp->flags & EDV_DIRECTORY_SORT)
	{
		/* Seek to the first name in the names list */
		dp->names_list_current = dp->names_list;
		dp->names_list_position = 0l;
	}
	else
	{
		/* Rewind using rewinddir() */
		rewinddir(dp->dp);
	}
}

/*
 *	Closes the directory stream.
 *
 *	The dp specifies the directory stream.
 */
void edv_directory_close(EDVDirectory *dp)
{
	if(dp == NULL)
		return;

	if(dp->names_list != NULL)
	{
		g_list_foreach(
			dp->names_list,
			(GFunc)g_free,
			NULL
		);
		g_list_free(dp->names_list);
	}

	if(dp->dp != NULL)
		(void)closedir(dp->dp);

	g_free(dp);
}

/*
 *	Gets a list of directory entry names.
 *
 *	The path specifies the path to the directory to list.
 *
 *	If sort is TRUE then the directory entries will be sorted.
 *
 *	If include_notations is TRUE then directory entry notations
 *	as ".." and "." will be included.
 *
 *	Returns a dynamically allocated GList of gchar * strings
 *	describing the name of each entry in the directory. The returned
 *	GList and each gchar * string must be deleted by calling
 *	edv_directory_list_delete().
 */
GList *edv_directory_list(
	const gchar *path,
	const gboolean sort,
	const gboolean include_notations
)
{
	DIR *dp;
	struct dirent *dent;
	const gchar *name;
	GList *names_list;

	if(STRISEMPTY(path))
	{
		errno = EINVAL;
		return(NULL);
	}

	/* Open the directory */
	dp = opendir((const char *)path);
	if(dp == NULL)
		return(NULL);

	/* Get the directory entries list */
	names_list = NULL;
	for(dent = readdir(dp); dent != NULL; dent = readdir(dp))
	{
		name = (const gchar *)dent->d_name;

		/* Skip notations? */
		if(!include_notations)
		{
			if(!strcmp((const char *)name, ".") ||
			   !strcmp((const char *)name, "..")
			)
				continue;
		}

		names_list = g_list_append(
			names_list,
			g_strdup(name)
		);
	}
	/* Sort? */
	if(sort && (names_list != NULL))
		names_list = g_list_sort(
			names_list,
			edv_directory_sort_cb
		);

	/* Close the directory */
	(void)closedir(dp);

	return(names_list);
}

/*
 *	Deletes the list of directory names.
 *
 *	The names_list specifies a GList of gchar * strings to be
 *	deleted.
 */
void edv_directory_list_delete(GList *names_list)
{
	if(names_list == NULL)
		return;

	g_list_foreach(names_list, (GFunc)g_free, NULL);
	g_list_free(names_list);
}
