#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include "../../include/fio.h"
#include "edv_fs_type.h"
#include "edv_device.h"
#include "edv_devices_list.h"
#include "config.h"


/* Open Devices List */
GList *edv_devices_list_file_open(
	GList *devices_list,
	const gchar *path
);
/* Save Devices List */
void edv_devices_list_file_save(
	GList *devices_list,
	const gchar *path
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Opens the Devices List from the Devices List file.
 *
 *	The devices_list specifies the Devices List, a GList of
 *	EDVDevice * Devices. The Devices opened from the
 *	Devices List file will be appended to the Devices List.
 *
 *	The path specifies the Devices List file.
 *
 *	Returns the Devices List, a GList of EDVDevice *
 *	Devices.
 */
GList *edv_devices_list_file_open(
	GList *devices_list,
	const gchar *path
)
{
	FILE *fp;
	gchar *parm;
	EDVDevice *d = NULL;

	if(STRISEMPTY(path))
		return(devices_list);

	/* Open the Devices List file for reading */
	fp = fopen((const char *)path, "rb");
	if(fp == NULL)
		return(devices_list);

	/* Begin reading the Devices List file */
	parm = NULL;
	while(TRUE)
	{
		/* Read the next parameter from the Devices List file */
		parm = (gchar *)FSeekNextParm(
			fp,
			(char *)parm,
			EDV_CFG_COMMENT_CHAR,
			EDV_CFG_DELIMINATOR_CHAR
		);
		if(parm == NULL)
			break;

		/* Begin handling by parameter */

		/* BeginDevice */
		if(!g_strcasecmp(parm, "BeginDevice"))
		{
			gchar *device_path = FGetString(fp);

			/* Check if an existing Device already exists in
			 * the list that matches the device_path
			 */
			if(devices_list != NULL)
			{
				GList *glist;
				EDVDevice *d2;
				for(glist = devices_list;
				    glist != NULL;
				    glist = g_list_next(glist)
				)
				{
					d2 = EDV_DEVICE(glist->data);
					if(d2 == NULL)
						continue;
					if(d2->device_path == NULL)
						continue;
					if(!strcmp(d2->device_path, device_path))
					{
						d = d2;
						break;
					}
				}
			}
			if(d == NULL)
			{
				/* No existing Device found that matches the
				 * device_path so create a new Device
				 */
				d = edv_device_new();
				if(d != NULL)
				{
					d->device_path = STRDUP(device_path);

					/* Append the new Device to the Devices List */
					devices_list = g_list_append(
						devices_list,
						d
					);
				}
			}

			g_free(device_path);
		}
		/* FileSystemTypeName */
		else if(!g_strcasecmp(parm, "FileSystemTypeName"))
		{
			gchar *fs_type_name = FGetString(fp);
			if(d != NULL)
			{
				d->fs_type_name = STRDUP(fs_type_name);
				d->fs_type_code = edv_fs_type_get_code_from_name(d->fs_type_name);
			}
			g_free(fs_type_name);
		}
		/* NoUnmount */
		else if(!g_strcasecmp(parm, "NoUnmount"))
		{
			gint v[1];
			FGetValuesI(fp, v, 1);
			if(d != NULL)
			{
				if(v[0])
					d->flags |= EDV_DEVICE_NO_UNMOUNT;
				else
					d->flags &= ~EDV_DEVICE_NO_UNMOUNT;
			}
		}
		/* ReadOnly */
		else if(!g_strcasecmp(parm, "ReadOnly"))
		{
			gint v[1];
			FGetValuesI(fp, v, 1);
			if(d != NULL)
			{
				if(v[0])
					d->flags |= EDV_DEVICE_READ_ONLY;
				else
					d->flags &= ~EDV_DEVICE_READ_ONLY;
			}
		}
		/* Unlisted */
		else if(!g_strcasecmp(parm, "Unlisted"))
		{
			gint v[1];
			FGetValuesI(fp, v, 1);
			if(d != NULL)
			{
				if(v[0])
					d->flags |= EDV_DEVICE_UNLISTED;
				else
					d->flags &= ~EDV_DEVICE_UNLISTED;
			}
		}
		/* NoScan */
		else if(!g_strcasecmp(parm, "NoScan"))
		{
			gint v[1];
			FGetValuesI(fp, v, 1);
			if(d != NULL)
			{
				if(v[0])
					d->flags |= EDV_DEVICE_NO_SCAN;
				else
					d->flags &= ~EDV_DEVICE_NO_SCAN;
			}
		}
		/* Name */
		else if(!g_strcasecmp(parm, "Name"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->name);
				d->name = STRDUP(v);
			}
			g_free(v);
		}
		/* MountPath */
		else if(!g_strcasecmp(parm, "MountPath"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->mount_path);
				d->mount_path = STRDUP(v);
			}
			g_free(v);
		}
		/* CommandMount */
		else if(!g_strcasecmp(parm, "CommandMount"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->command_mount);
				d->command_mount = STRDUP(v);
			}
			g_free(v);
		}
		/* CommandUnmount */
		else if(!g_strcasecmp(parm, "CommandUnmount"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->command_unmount);
				d->command_unmount = STRDUP(v);
			}
			g_free(v);
		}
		/* CommandEject */
		else if(!g_strcasecmp(parm, "CommandEject"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->command_eject);
				d->command_eject = STRDUP(v);
			}
			g_free(v);
		}
		/* CommandCheck */
		else if(!g_strcasecmp(parm, "CommandCheck"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->command_check);
				d->command_check = STRDUP(v);
			}
			g_free(v);
		}
		/* CommandTools */
		else if(!g_strcasecmp(parm, "CommandTools"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->command_tools);
				d->command_tools = STRDUP(v);
			}
			g_free(v);
		}
		/* CommandFormat */
		else if(!g_strcasecmp(parm, "CommandFormat"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				g_free(d->command_format);
				d->command_format = STRDUP(v);
			}
			g_free(v);
		}

		/* IconSmallStandard */
		else if(!g_strcasecmp(parm, "IconSmallStandard"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->small_icon_file[
					EDV_DEVICE_ICON_STATE_STANDARD
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}
		/* IconSmallSelected */
		else if(!g_strcasecmp(parm, "IconSmallSelected"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->small_icon_file[
					EDV_DEVICE_ICON_STATE_SELECTED
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}
		/* IconSmallUnmounted */
		else if(!g_strcasecmp(parm, "IconSmallUnmounted"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->small_icon_file[
					EDV_DEVICE_ICON_STATE_UNMOUNTED
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}

		/* IconMediumStandard */
		else if(!g_strcasecmp(parm, "IconMediumStandard"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->medium_icon_file[
					EDV_DEVICE_ICON_STATE_STANDARD
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}
		/* IconMediumSelected */
		else if(!g_strcasecmp(parm, "IconMediumSelected"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->medium_icon_file[
				    EDV_DEVICE_ICON_STATE_SELECTED
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}
		/* IconMediumUnmounted */
		else if(!g_strcasecmp(parm, "IconMediumUnmounted"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->medium_icon_file[
					EDV_DEVICE_ICON_STATE_UNMOUNTED
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}

		/* IconLargeStandard */
		else if(!g_strcasecmp(parm, "IconLargeStandard"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->large_icon_file[
				    EDV_DEVICE_ICON_STATE_STANDARD
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}
		/* IconLargeSelected */
		else if(!g_strcasecmp(parm, "IconLargeSelected"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->large_icon_file[
					EDV_DEVICE_ICON_STATE_SELECTED
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}
		/* IconLargeUnmounted */
		else if(!g_strcasecmp(parm, "IconLargeUnmounted"))
		{
			gchar *v = FGetString(fp);
			if(d != NULL)
			{
				gchar **t = &d->large_icon_file[
					EDV_DEVICE_ICON_STATE_UNMOUNTED
				];
				g_free(*t);
				*t = STRDUP(v);
			}
			g_free(v);
		}

		/* LastMountTime */
		else if(!g_strcasecmp(parm, "LastMountTime"))
		{
			glong v[1];
			FGetValuesL(fp, v, 1);
			if(d != NULL)
				d->last_mount_time = (gulong)v[0];
		}
		/* LastCheckTime */
		else if(!g_strcasecmp(parm, "LastCheckTime"))
		{
			glong v[1];
			FGetValuesL(fp, v, 1);
			if(d != NULL)
				d->last_check_time = (gulong)v[0];
		}

		/* End of a device block? */
		else if(!g_strcasecmp(parm, "EndDevice"))
		{
			FSeekNextLine(fp);

			/* Reset the context */
			d = NULL;
		}
		/* All else unsupported parameter or wrong context */
		else
		{
			FSeekNextLine(fp);
		}
	}

	/* Delete the parameter */
	g_free(parm);

	/* Close the Devices List file */
	fclose(fp);

	return(devices_list);
}

/*
 *	Saves the Devices List to the Devices List file.
 *
 *	The devices_list specifies the Devices List, a GList of
 *	EDVDevice * Devices.
 *
 *	The path specifies the Devices List file.
 */
void edv_devices_list_file_save(
	GList *devices_list,
	const gchar *path
)
{
	FILE *fp;
	const gchar *lpath;
	gchar *parent_path;
	GList *glist;
	EDVDevice *d;

	if((devices_list == NULL) || STRISEMPTY(path))
		return;

	/* Get parent directory and create it as needed */
	parent_path = g_dirname(path);
	if(parent_path != NULL)
	{
#if defined(S_IRUSR) && defined(S_IWUSR) && defined(S_IXUSR)
		rmkdir(parent_path, S_IRUSR | S_IWUSR | S_IXUSR);
#endif
		g_free(parent_path);
	}

	/* Open the Devices List file for writing */
	fp = fopen((const char *)path, "wb");
	if(fp == NULL)
		return;

	/* Save each Device in the Devices List */
	for(glist = devices_list;
		glist != NULL;
		glist = g_list_next(glist)
	)
	{
		d = EDV_DEVICE(glist->data);
		if(d == NULL)
			continue;

		/* BeginDevice */
		fprintf(
			fp,
			"BeginDevice = %s\n",
			d->device_path
		);

		/* FileSystemTypeName */
		fprintf(
			fp,
			"\tFileSystemTypeName = %s\n",
			d->fs_type_name
		);
		/* NoUnmount */
		fprintf(
			fp,
			"\tNoUnmount = %i\n",
			EDV_DEVICE_IS_NO_UNMOUNT(d)
		);
		/* ReadOnly */
		fprintf(
			fp,
			"\tReadOnly = %i\n",
			EDV_DEVICE_IS_READ_ONLY(d)
		);
		/* Unlisted */
		fprintf(
			fp,
			"\tUnlisted = %i\n",
			EDV_DEVICE_IS_UNLISTED(d)
		);
		/* Name */
		if(!STRISEMPTY(d->name))
			fprintf(
				fp,
				"\tName = %s\n",
				d->name
			);
		/* MountPath */
		if(!STRISEMPTY(d->mount_path))
			fprintf(
				fp,
				"\tMountPath = %s\n",
				d->mount_path
			);

		/* CommandMount */
		if(!STRISEMPTY(d->command_mount))
			fprintf(
				fp,
				"\tCommandMount = %s\n",
				d->command_mount
			);
		/* CommandUnmount */
		if(!STRISEMPTY(d->command_unmount))
			fprintf(
				fp,
				"\tCommandUnmount = %s\n",
				d->command_unmount
			);
		/* CommandEject */
		if(!STRISEMPTY(d->command_eject))
			fprintf(
				fp,
				"\tCommandEject = %s\n",
				d->command_eject
			);
		/* CommandCheck */
		if(!STRISEMPTY(d->command_check))
			fprintf(
				fp,
				"\tCommandCheck = %s\n",
				d->command_check
			);
		/* CommandTools */
		if(!STRISEMPTY(d->command_tools))
			fprintf(
				fp,
				"\tCommandTools = %s\n",
				d->command_tools
			);
		/* CommandFormat */
		if(!STRISEMPTY(d->command_format))
			fprintf(
				fp,
				"\tCommandFormat = %s\n",
				d->command_format
			);

		/* IconSmallStandard */
		lpath = d->small_icon_file[EDV_DEVICE_ICON_STATE_STANDARD];
		if(!STRISEMPTY(lpath))
			fprintf(
				fp,
				"\tIconSmallStandard = %s\n",
				lpath
			);
		/* IconSmallSelected */
		lpath = d->small_icon_file[EDV_DEVICE_ICON_STATE_SELECTED];
		if(!STRISEMPTY(lpath))
			fprintf(
				fp,
				"\tIconSmallSelected = %s\n",
				lpath
			);
		/* IconSmallUnmounted */
		lpath = d->small_icon_file[EDV_DEVICE_ICON_STATE_UNMOUNTED];
		if(!STRISEMPTY(lpath))
			fprintf(
				fp,
				"\tIconSmallUnmounted = %s\n",
				lpath
			);

		/* IconMediumStandard */
		lpath = d->medium_icon_file[EDV_DEVICE_ICON_STATE_STANDARD];
		if(!STRISEMPTY(lpath))
			fprintf(
				fp,
				"\tIconMediumStandard = %s\n",
				lpath
			);
		/* IconMediumSelected */
		lpath = d->medium_icon_file[EDV_DEVICE_ICON_STATE_SELECTED];
		if(!STRISEMPTY(lpath))
			fprintf(
				fp,
				"\tIconMediumSelected = %s\n",
				lpath
			);
		/* IconMediumUnmounted */
		lpath = d->medium_icon_file[EDV_DEVICE_ICON_STATE_UNMOUNTED];
		if(!STRISEMPTY(lpath))
			fprintf(
				fp,
				"\tIconMediumUnmounted = %s\n",
				lpath
			);

		/* IconLargeStandard */
		lpath = d->large_icon_file[EDV_DEVICE_ICON_STATE_STANDARD];
		if(!STRISEMPTY(lpath))
			fprintf(
				fp,
				"\tIconLargeStandard = %s\n",
				lpath
			);
		/* IconLargeSelected */
		lpath = d->large_icon_file[EDV_DEVICE_ICON_STATE_SELECTED];
		if(!STRISEMPTY(lpath))
			fprintf(
				fp,
				"\tIconLargeSelected = %s\n",
				lpath
			);
		/* IconLargeUnmounted */
		lpath = d->large_icon_file[EDV_DEVICE_ICON_STATE_UNMOUNTED];
		if(!STRISEMPTY(lpath))
			fprintf(
				fp,
				"\tIconLargeUnmounted = %s\n",
				lpath
			);

		/* LastMountTime */
		if(d->last_mount_time > 0l)
			fprintf(
				fp,
				"\tLastMountTime = %ld\n",
				d->last_mount_time
			);
		/* LastCheckTime */
		if(d->last_check_time > 0l)
			fprintf(
				fp,
				"\tLastCheckTime = %ld\n",
				d->last_check_time
			);

		/* EndDevice */
		fprintf(
			fp,
			"EndDevice\n"
		);
	}

	/* Close the Devices List file */
	(void)fclose(fp);
}
