#include <glib.h>
#include "cfg.h"
#include "edv_utils.h"
#include "edv_stream.h"
#include "edv_process.h"
#include "edv_fs_type.h"
#include "edv_device.h"
#include "edv_device_mount.h"
#include "edv_notify.h"
#include "edv_context_private.h"
#include "edv_cfg_list.h"
#include "config.h"


const gchar *edv_device_mount_get_error(EDVContext *ctx);
static void edv_device_mount_set_error(
	EDVContext *ctx,
	const gchar *msg
);

gint edv_device_mount(
	EDVContext *ctx,
	EDVDevice *d,
	const gboolean notify
);
gint edv_device_unmount(
	EDVContext *ctx,
	EDVDevice *d,
	const gboolean notify
);
gint edv_device_eject(
	EDVContext *ctx,
	EDVDevice *d,
	const gboolean notify
);


#define ATOI(s)		(((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)		(((s) != NULL) ? atol(s) : 0)
#define ATOF(s)		(((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)	(((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)	(((a) > (b)) ? (a) : (b))
#define MIN(a,b)	(((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)	(MIN(MAX((a),(l)),(h)))
#define STRLEN(s)	(((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)	(((s) != NULL) ? (*(s) == '\0') : TRUE)

#define FCLOSE(p)	(((p) != NULL) ? (gint)fclose(p) : -1)


/*
 *	Gets the last error message.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	Returns a statically allocated string describing the last
 *	error, the returned string must not be modified or deleted.
 */
const gchar *edv_device_mount_get_error(EDVContext *ctx)
{
	if(ctx == NULL)
		return(NULL);

	return(ctx->last_error_ptr);
}

/*
 *	Sets the last error message.
 */
static void edv_device_mount_set_error(
	EDVContext *ctx,
	const gchar *msg
)
{
        if(msg == ctx->last_error_buf)
                return;

	g_free(ctx->last_error_buf);
	ctx->last_error_buf = STRDUP(msg);

	ctx->last_error_ptr = ctx->last_error_buf;
}


/*
 *	Mounts the device.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	The d specifies the device.
 *
 *	If notify is set to TRUE then a mount notify event will be
 *	queued if mounting was successful.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint edv_device_mount(
	EDVContext *ctx,
	EDVDevice *d,
	const gboolean notify
)
{
	FILE		*cstdout,
			*cstderr;
	gint		pid,
			status;
	const gchar *fs_type_name;
	const gchar	*shell_cmd,
			*shell_args;
	gchar		*shell_prog,
			*cmd = NULL,
			*mount_path = NULL,
			*device_path = NULL;
	CfgList *cfg_list;

	if(ctx == NULL)
		return(-2);

	edv_device_mount_set_error(ctx, NULL);

	cfg_list = ctx->cfg_list;

	if(d == NULL)
	{
		edv_device_mount_set_error(
			ctx,
"No device specified"
		);
		return(-2);
	}

#define CLEANUP_RETURN(_v_)	{	\
 g_free(cmd);				\
 g_free(mount_path);			\
 g_free(device_path);			\
					\
 return(_v_);				\
}

	/* Already mounted? */
	if(EDV_DEVICE_IS_MOUNTED(d))
	{
		edv_device_mount_set_error(
			ctx,
"Device is already mounted"
		);
		return(-2);
	}

	/* Get copy of mount path */
	mount_path = STRDUP(d->mount_path);
	if(mount_path == NULL)
	{
		edv_device_mount_set_error(
			ctx,
"Mount path not specified by the device"
		);
		CLEANUP_RETURN(-2);
	}

	/* Get copy of device path */
	device_path = STRDUP(d->device_path);
	if(device_path == NULL)
	{
		edv_device_mount_set_error(
			ctx,
"Device path not specified by the device"
		);
		CLEANUP_RETURN(-2);
	}

	/* Get the filesystem type name */
	fs_type_name = d->fs_type_name;
	if(fs_type_name == NULL)
	{
		edv_device_mount_set_error(
			ctx,
"Unable to get the filesystem type"
		);
		CLEANUP_RETURN(-2);
	}

	/* Get the shell program and arguments */
	shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL);
	shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	);

	/* Format the mount command */
	if(STRISEMPTY(d->command_mount))
		cmd = g_strdup_printf(
			"\"%s\" \"%s\"%s",
			EDV_PATH_PROG_DEF_MOUNT,
			mount_path,
			EDV_DEVICE_IS_READ_ONLY(d) ? " -r" : ""
		);
	else
		cmd = g_strdup(d->command_mount);

	/* Execute the mount command */
	pid = edv_system_shell_streams(
		cmd,
		shell_prog,
		shell_args,
		NULL,
		&cstdout,
		&cstderr
	);

	g_free(shell_prog);

	if(pid > -1)
	{
		/* Monitor the mount process and wait for it to finish */
		gchar	*cstdout_buf = NULL,
			*cstderr_buf = NULL,
			*last_error = NULL;
		do {
			if(!edv_pid_exists(pid))
				pid = 0;
			cstdout_buf = edv_stream_read_strbuf(
				cstdout,
				cstdout_buf,
				FALSE		/* Nonblocking */
			);
			if(edv_stream_read_strptrbrk(
				cstderr,
				&cstderr_buf,
				"\n\r",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				g_free(last_error);
				last_error = g_strdup(cstderr_buf);
				g_free(cstderr_buf);
				cstderr_buf = NULL;
			}
			edv_usleep(8000l);
		} while(pid != 0);

		g_free(cstdout_buf);
		g_free(cstderr_buf);

		/* Update the device's mount state and statistics */
		edv_device_update_mount_state(d);
		edv_device_update_stats(d);

		/* Mounted the device successfully? */
		if(EDV_DEVICE_IS_MOUNTED(d))
		{
			/* Update the last mount time */
			d->last_mount_time = edv_time();

			if(notify)
				edv_notify_queue_vfs_object_mounted(
					ctx,
					mount_path
				);

			status = 0;
		}
		else
		{
			status = -1;
		}

		/* Record the last error from the mount execution */
		if(last_error != NULL)
		{
			edv_device_mount_set_error(
				ctx,
				last_error
			);
			g_free(last_error);
		}
	}
	else
	{
		edv_device_mount_set_error(
			ctx,
"Execution of mount command failed"
		);
		status = -1;
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Unmounts the device.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	The d specifies the device.
 *
 *	If notify is set to TRUE then a unmount notify event will be
 *	queued if unmounting was successful.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint edv_device_unmount(
	EDVContext *ctx,
	EDVDevice *d,
	const gboolean notify
)
{
	FILE		*cstdout,
			*cstderr;
	gint		pid,
			status;
	const gchar	*shell_cmd,
			*shell_args;
	gchar		*shell_prog,
			*cmd = NULL,
			*mount_path = NULL;
	CfgList *cfg_list;

	if(ctx == NULL)
		return(-2);

	edv_device_mount_set_error(ctx, NULL);

	cfg_list = ctx->cfg_list;

	if(d == NULL)
	{
		edv_device_mount_set_error(
			ctx,
"No device specified"
		);
		return(-2);
	}

	/* Device not allowed to be unmounted? */
	if(EDV_DEVICE_IS_NO_UNMOUNT(d))
	{
		edv_device_mount_set_error(
			ctx,
"Device does not permit unmounting"
		);
		return(-2);
	}

	/* Not mounted? */
	if(!EDV_DEVICE_IS_MOUNTED(d))
	{
		edv_device_mount_set_error(
			ctx,
"Device is not mounted"
		);
		return(-2);
	}

#define CLEANUP_RETURN(_v_)	{	\
 g_free(cmd);				\
 g_free(mount_path);			\
					\
 return(_v_);				\
}

	/* Get copy of mount path */
	mount_path = STRDUP(d->mount_path);
	if(mount_path == NULL)
	{
		edv_device_mount_set_error(
			ctx,
"Mount path not specified by the device"
		);
		CLEANUP_RETURN(-2);
	}

	/* Get the shell program and arguments */
	shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL);
	shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	);

	/* Format the unmount command */
	if(STRISEMPTY(d->command_unmount))
		cmd = g_strdup_printf(
			"\"%s\" \"%s\"",
			EDV_PATH_PROG_DEF_UMOUNT,
			mount_path
		);
	else
		cmd = g_strdup(d->command_unmount);

	/* Execute the unmount command */
	pid = edv_system_shell_streams(
		cmd,
		shell_prog,
		shell_args,
		NULL,
		&cstdout,
		&cstderr
	);

	g_free(shell_prog);

	if(pid > -1)
	{
		/* Monitor the unmount process and wait for it to finish */
		gchar	*cstdout_buf = NULL,
			*cstderr_buf = NULL,
			*last_error = NULL;
		do {
			if(!edv_pid_exists(pid))
				pid = 0;
			cstdout_buf = edv_stream_read_strbuf(
				cstdout,
				cstdout_buf,
				FALSE		/* Nonblocking */
			);
			if(edv_stream_read_strptrbrk(
				cstderr,
				&cstderr_buf,
				"\n\r",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				g_free(last_error);
				last_error = g_strdup(cstderr_buf);
				g_free(cstderr_buf);
				cstderr_buf = NULL;
			}
			edv_usleep(8000l);
		} while(pid != 0);

		g_free(cstdout_buf);
		g_free(cstderr_buf);

		/* Update the device's mount state and statistics */
		edv_device_update_mount_state(d);
		edv_device_update_stats(d);

		/* Unmounted the device successfully? */
		if(!EDV_DEVICE_IS_MOUNTED(d))
		{
			/* Update the last mount time */
			d->last_mount_time = edv_time();

			if(notify)
				edv_notify_queue_vfs_object_unmounted(ctx, mount_path);

			status = 0;
		}
		else
		{
			status = -1;
		}

		/* Record the last error from the mount execution */
		if(last_error != NULL)
		{
			edv_device_mount_set_error(
				ctx,
				last_error
			);
			g_free(last_error);
		}
	}
	else
	{
		edv_device_mount_set_error(
			ctx,
"Execution of unmount command failed"
		);
		status = -1;
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}

/*
 *	Ejects the media from the device, unmounting the device first
 *	if needed.
 *
 *	The ctx specifies the Endeavour 2 context.
 *
 *	The d specifies the device.
 *
 *	If notify is set to TRUE then a unmount notify event will be
 *	queued if the device needed to be unmounted and unmounting
 *	was successful.
 *
 *	Returns 0 on success and non-zero on error.
 */
gint edv_device_eject(
	EDVContext *ctx,
	EDVDevice *d,
	const gboolean notify
)
{
	FILE		*cstdout,
			*cstderr;
	gint		pid,
			status;
	const gchar	*shell_cmd,
			*shell_args;
	gchar		*shell_prog,
			*cmd = NULL,
			*device_path = NULL;
	CfgList *cfg_list;

	if(ctx == NULL)
		return(-2);

	edv_device_mount_set_error(ctx, NULL);

	cfg_list = ctx->cfg_list;

	if(d == NULL)
	{
		edv_device_mount_set_error(
			ctx,
"No device specified"
		);
		return(-2);
	}

	/* Media not allowed to be ejected from device? */
	if(EDV_DEVICE_IS_NO_UNMOUNT(d))
	{
		edv_device_mount_set_error(
			ctx,
"Device does not permit ejecting"
		);
		return(-2);
	}

	/* Not unmounted yet? */
	if(EDV_DEVICE_IS_MOUNTED(d))
	{
		if(edv_device_unmount(
			ctx,
			d,
			TRUE
		))
			return(-1);
	}

#define CLEANUP_RETURN(_v_)	{	\
 g_free(cmd);				\
 g_free(device_path);			\
					\
 return(_v_);				\
}

	/* Copy the device path */
	device_path = STRDUP(d->device_path);
	if(device_path == NULL)
	{
		edv_device_mount_set_error(
			ctx,
"Device path not specified by the device"
		);
		CLEANUP_RETURN(-2);
	}

	/* Get the shell program and arguments */
	shell_cmd = EDV_GET_S(EDV_CFG_PARM_PROG_SHELL);
	shell_args = edv_strarg(
		shell_cmd,
		&shell_prog,
		TRUE,				/* Parse escapes */
		TRUE				/* Parse quotes */
	);

	/* Format the eject command */
	if(STRISEMPTY(d->command_eject))
		cmd = g_strdup_printf(
			"\"%s\" \"%s\"",
			EDV_PATH_PROG_DEF_EJECT,
			device_path
		);
	else
		cmd = g_strdup(d->command_eject);

	/* Execute the eject command */
	pid = edv_system_shell_streams(
		cmd,
		shell_prog,
		shell_args,
		NULL,
		&cstdout,
		&cstderr
	);

	g_free(shell_prog);

	if(pid > -1)
	{
		/* Monitor the eject process and wait for it to finish */
		gchar	*cstdout_buf = NULL,
			*cstderr_buf = NULL,
			*last_error = NULL;
		do {
			if(!edv_pid_exists(pid))
				pid = 0;
			cstdout_buf = edv_stream_read_strbuf(
				cstdout,
				cstdout_buf,
				FALSE		/* Nonblocking */
			);
			if(edv_stream_read_strptrbrk(
				cstderr,
				&cstderr_buf,
				"\n\r",
				FALSE,		/* Exclude end character */
				FALSE		/* Nonblocking */
			))
			{
				g_free(last_error);
				last_error = g_strdup(cstderr_buf);
				g_free(cstderr_buf);
				cstderr_buf = NULL;
			}
			edv_usleep(8000l);
		} while(pid != 0);

		g_free(cstdout_buf);
		g_free(cstderr_buf);

		/* Always assume success for eject */
		status = 0;

		/* Record the last error from the mount execution */
		if(last_error != NULL)
		{
			edv_device_mount_set_error(
				ctx,
				last_error
			);
			g_free(last_error);
		}
	}
	else
	{
		edv_device_mount_set_error(
			ctx,
"Execution of eject command failed"
		);
		status = -1;
	}

	/* Close the output streams */
	(void)FCLOSE(cstdout);
	(void)FCLOSE(cstderr);

	CLEANUP_RETURN(status);
#undef CLEANUP_RETURN
}
