/*
		Endeavour Mark II API - Devices Fetching
 */

#ifndef EDV_DEVICE_GET_H
#define EDV_DEVICE_GET_H

#include <glib.h>
#include "edv_device.h"
#include "edv_context.h"


/*
 *	Gets the Devices list.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	Returns a GList of EDVDevice * devices, the returned list
 *	must not be modified or deleted.
 */
extern GList *edv_devices_list(EDVContext *ctx);


/*
 *	Matches the EDVDevice by mount path.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The mount_path specifies the mount path.
 *
 *	Returns the EDVDevice or NULL on error.
 */
extern EDVDevice *edv_devices_list_match_mount_path(
	EDVContext *ctx,
	const gchar *mount_path
);

/*
 *	Matches the EDVDevice by device path.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The device_path specifies the device path.
 *
 *	Returns the EDVDevice or NULL on error.
 */
extern EDVDevice *edv_devices_list_match_device_path(
	EDVContext *ctx,
	const gchar *device_path
);

/*
 *	Matches the EDVDevice by a VFS object that resides on it.
 *
 *	The ctx specifies the Endeavour 2 Context.
 *
 *	The path specifies the full path to any object on the device 
 *	including the device's mount point itself but not the device's
 *	device path.
 *
 *	Returns the EDVDevice or NULL on error.
 */
extern EDVDevice *edv_devices_list_match_path(
	EDVContext *ctx,
	const gchar *path
);


/*
 *	Updates all the EDVDevices' mount states with the current
 *	mount states obtained from the system.
 *
 *	Each Device's flags member's EDV_DEVICE_MOUNTED flag will
 *	be either set or unset.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
extern void edv_devices_list_update_mount_states(EDVContext *ctx);

/*
 *	Updates all the EDVDevices' statistics with the current
 *	statistics obtained from the system.
 *
 *	Each Device's blocks_total, blocks_available, blocks_free,
 *	block_size, indicies_total, indicies_free, fs_id, and
 *	name_length_max members will be updated by this call.
 *
 *	edv_devices_list_update_mount_states() should always be called
 *	prior to this call to ensure that the mount states are up
 *	to date.
 *
 *	The ctx specifies the Endeavour 2 Context.
 */
extern void edv_devices_list_update_stats(EDVContext *ctx);


#endif	/* EDV_DEVICE_GET_H */
